/*
  bdf2cgrom.c  2021.5.15 version
    convert .bdf font file to CGROM format for PC-6001
    by AKIKAWA, Hisashi
    This software is redistributable under the LGPLv2.1 or any later version.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define NCHR 256
#define SIZEX_DEFAULT 8
#define SIZEY_DEFAULT 12
#define BDFSIZE 64

enum {JIS, UNI};

int getbdfpar(FILE *fp, int ascii, int use_ascent, int *pixel_x, int *pixel_y, int *over_baseline);
void convert(FILE *fp, unsigned char bmp60[16], int size_x_out, int size_y_out, int pixel_x, int pixel_y, int over_baseline, int code_is_ambiguous);
int isnodata(unsigned char *bmp60);
void calcchr(unsigned char bmp60[][16], int pc6001a, int size_y_out);
int isambiguous(int code, int charset);

int main(int argc, char *argv[])
{
  int i, j;
  int opt;
  int ascii = 0, use_ascent = 0;
  int pc6001a = 0;
  int altsize;
  int code, chr;
  int code_is_ambiguous;
  int pixel_x = 16, pixel_y = 16, over_baseline = 16;
  int size_x_out = SIZEX_DEFAULT, size_y_out = SIZEY_DEFAULT;
  int charset;
  char line[100];
  char outfile[FILENAME_MAX + 1];
  unsigned char bmp60[NCHR][16] = {0};
  long fpos;
  FILE *fp;

  const int jistbl[] = {
    0x2121, 0x376e, 0x3250, 0x3f65, 0x4c5a, 0x3662, 0x455a, 0x467c, /* 00 */
    0x472f, 0x315f, 0x3b7e, 0x4a2c, 0x4943, 0x4934, 0x4069, 0x4b7c,
    0x2650, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 10 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x215f, 0x4267, 0x4366, 0x3e2e,
    0x2121, 0x212a, 0x2230, 0x2174, 0x2170, 0x2173, 0x2175, 0x2147, /* 20 */
    0x214a, 0x214b, 0x2176, 0x215c, 0x2124, 0x2231, 0x2125, 0x213f,
    0x2330, 0x2331, 0x2332, 0x2333, 0x2334, 0x2335, 0x2336, 0x2337, /* 30 */
    0x2338, 0x2339, 0x2127, 0x2128, 0x2163, 0x2161, 0x2164, 0x2129,
    0x2177, 0x2341, 0x2342, 0x2343, 0x2344, 0x2345, 0x2346, 0x2347, /* 40 */
    0x2348, 0x2349, 0x234a, 0x234b, 0x234c, 0x234d, 0x234e, 0x234f,
    0x2350, 0x2351, 0x2352, 0x2353, 0x2354, 0x2355, 0x2356, 0x2357, /* 50 */
    0x2358, 0x2359, 0x235a, 0x214e, 0x216f, 0x214f, 0x2130, 0x2132,
    0x2121, 0x2361, 0x2362, 0x2363, 0x2364, 0x2365, 0x2366, 0x2367, /* 60 */
    0x2368, 0x2369, 0x236a, 0x236b, 0x236c, 0x236d, 0x236e, 0x236f,
    0x2370, 0x2371, 0x2372, 0x2373, 0x2374, 0x2375, 0x2376, 0x2377, /* 70 */
    0x2378, 0x2379, 0x237a, 0x2150, 0x2925, 0x2151, 0x2232, 0x2121,
    0x263a, 0x263e, 0x2640, 0x263c, 0x217b, 0x217c, 0x2472, 0x2421, /* 80 */
    0x2423, 0x2425, 0x2427, 0x2429, 0x2463, 0x2465, 0x2467, 0x2443,
    0x2121, 0x2422, 0x2424, 0x2426, 0x2428, 0x242a, 0x242b, 0x242d, /* 90 */
    0x242f, 0x2431, 0x2433, 0x2435, 0x2437, 0x2439, 0x243b, 0x243d,
    0x2121, 0x2123, 0x2156, 0x2157, 0x2122, 0x2126, 0x2572, 0x2521, /* a0 */
    0x2523, 0x2525, 0x2527, 0x2529, 0x2563, 0x2565, 0x2567, 0x2543,
    0x213c, 0x2522, 0x2524, 0x2526, 0x2528, 0x252a, 0x252b, 0x252d, /* b0 */
    0x252f, 0x2531, 0x2533, 0x2535, 0x2537, 0x2539, 0x253b, 0x253d,
    0x253f, 0x2541, 0x2544, 0x2546, 0x2548, 0x254a, 0x254b, 0x254c, /* c0 */
    0x254d, 0x254e, 0x254f, 0x2552, 0x2555, 0x2558, 0x255b, 0x255e,
    0x255f, 0x2560, 0x2561, 0x2562, 0x2564, 0x2566, 0x2568, 0x2569, /* d0 */
    0x256a, 0x256b, 0x256c, 0x256d, 0x256f, 0x2573, 0x212b, 0x212c,
    0x243f, 0x2441, 0x2444, 0x2446, 0x2448, 0x244a, 0x244b, 0x244c, /* e0 */
    0x244d, 0x244e, 0x244f, 0x2452, 0x2455, 0x2458, 0x245b, 0x245e,
    0x245f, 0x2460, 0x2461, 0x2462, 0x2464, 0x2466, 0x2468, 0x2469, /* f0 */
    0x246a, 0x246b, 0x246c, 0x246d, 0x246f, 0x2473, 0x2121, 0x2121,
  };

  const int unitbl[] = {
    0x3000, 0x6708, 0x706b, 0x6c34, 0x6728, 0x91d1, 0x571f, 0x65e5, /* 00 */
    0x5e74, 0x5186, 0x6642, 0x5206, 0x79d2, 0x767e, 0x5343, 0x4e07,
    0x03c0, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 10 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x00d7, 0x5927, 0x4e2d, 0x5c0f,
    0x3000, 0xff01, 0xff02, 0xff03, 0xff04, 0xff05, 0xff06, 0x2019, /* 20 */
    0xff08, 0xff09, 0xff0a, 0xff0b, 0xff0c, 0xff0d, 0xff0e, 0xff0f,
    0xff10, 0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17, /* 30 */
    0xff18, 0xff19, 0xff1a, 0xff1b, 0xff1c, 0xff1d, 0xff1e, 0xff1f,
    0xff20, 0xff21, 0xff22, 0xff23, 0xff24, 0xff25, 0xff26, 0xff27, /* 40 */
    0xff28, 0xff29, 0xff2a, 0xff2b, 0xff2c, 0xff2d, 0xff2e, 0xff2f,
    0xff30, 0xff31, 0xff32, 0xff33, 0xff34, 0xff35, 0xff36, 0xff37, /* 50 */
    0xff38, 0xff39, 0xff3a, 0xff3b, 0xffe5, 0xff3d, 0xff3e, 0xff3f,
    0x3000, 0xff41, 0xff42, 0xff43, 0xff44, 0xff45, 0xff46, 0xff47, /* 60 */
    0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c, 0xff4d, 0xff4e, 0xff4f,
    0xff50, 0xff51, 0xff52, 0xff53, 0xff54, 0xff55, 0xff56, 0xff57, /* 70 */
    0xff58, 0xff59, 0xff5a, 0xff5b, 0xffe4, 0xff5d, 0xff5e, 0x3000,
    0x2660, 0x2665, 0x2663, 0x2666, 0x25cb, 0x25cf, 0x3092, 0x3041, /* 80 */
    0x3043, 0x3045, 0x3047, 0x3049, 0x3083, 0x3085, 0x3087, 0x3063,
    0x3000, 0x3042, 0x3044, 0x3046, 0x3048, 0x304a, 0x304b, 0x304d, /* 90 */
    0x304f, 0x3051, 0x3053, 0x3055, 0x3057, 0x3059, 0x305b, 0x305d,
    0x3000, 0x3002, 0x300c, 0x300d, 0x3001, 0x30fb, 0x30f2, 0x30a1, /* a0 */
    0x30a3, 0x30a5, 0x30a7, 0x30a9, 0x30e3, 0x30e5, 0x30e7, 0x30c3,
    0x30fc, 0x30a2, 0x30a4, 0x30a6, 0x30a8, 0x30aa, 0x30ab, 0x30ad, /* b0 */
    0x30af, 0x30b1, 0x30b3, 0x30b5, 0x30b7, 0x30b9, 0x30bb, 0x30bd,
    0x30bf, 0x30c1, 0x30c4, 0x30c6, 0x30c8, 0x30ca, 0x30cb, 0x30cc, /* c0 */
    0x30cd, 0x30ce, 0x30cf, 0x30d2, 0x30d5, 0x30d8, 0x30db, 0x30de,
    0x30df, 0x30e0, 0x30e1, 0x30e2, 0x30e4, 0x30e6, 0x30e8, 0x30e9, /* d0 */
    0x30ea, 0x30eb, 0x30ec, 0x30ed, 0x30ef, 0x30f3, 0x309b, 0x309c,
    0x305f, 0x3061, 0x3064, 0x3066, 0x3068, 0x306a, 0x306b, 0x306c, /* e0 */
    0x306d, 0x306e, 0x306f, 0x3072, 0x3075, 0x3078, 0x307b, 0x307e,
    0x307f, 0x3080, 0x3081, 0x3082, 0x3084, 0x3086, 0x3088, 0x3089, /* f0 */
    0x308a, 0x308b, 0x308c, 0x308d, 0x308f, 0x3093, 0x3000, 0x3000,
  };

  const int jisalttbl[][2] = {
    /* suits of playing cards */
    {0x80, 0x2225}, {0x81, 0x2227}, {0x82, 0x217a}, {0x83, 0x2221},
    /* double quotation mark */
    {0x22, 0x2149},
    /* apostrophe */
    {0x27, 0x2222f},
    /* hyphen-minus */
    {0x2d, 0x215d},
    /* square bracket */
    {0x5b, 0x214c}, {0x5d, 0x214d},
    /* tilde */
    {0x7e, 0x2141},
    /* broken bar */
    {0x7c, 0x2143},
  };

  const int unialttbl[][2] = {
    /* suits of playing cards */
    {0x80, 0x25b2}, {0x81, 0x25bc}, {0x82, 0x2605}, {0x83, 0x25c6},
    /* double quotation mark */
    {0x22, 0x201d},
    /* apostrophe */
    {0x27, 0xff07},
    /* hyphen-minus */
    {0x2d, 0x2212},
    /* square bracket */
    {0x5b, 0x3014}, {0x5d, 0x3015},
    /* tilde */
    {0x7e, 0x301c},
    /* broken bar */
    {0x7d, 0xff5c},
  };

  const int jistbl_6001a[] = {
    0x2121, 0x217c, 0x317a, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 00 */
    0x0000, 0x0000, 0x0000, 0x222c, 0x222d, 0x2224, 0x2226, 0x213f,
    0x2630, 0x217b, 0x317a, 0x2d74, 0x215f, 0x2131, 0x213d, 0x2132, /* 10 */
    0x464c, 0x2222, 0x2223, 0x0000, 0x0000, 0x0000, 0x0000, 0x2140,
    0x2121, 0x212a, 0x2230, 0x2174, 0x2170, 0x2173, 0x2175, 0x2147, /* 20 */
    0x214a, 0x214b, 0x2176, 0x215c, 0x2124, 0x2231, 0x2125, 0x213f,
    0x2330, 0x2331, 0x2332, 0x2333, 0x2334, 0x2335, 0x2336, 0x2337, /* 30 */
    0x2338, 0x2339, 0x2127, 0x2128, 0x2163, 0x2161, 0x2164, 0x2129,
    0x2177, 0x2341, 0x2342, 0x2343, 0x2344, 0x2345, 0x2346, 0x2347, /* 40 */
    0x2348, 0x2349, 0x234a, 0x234b, 0x234c, 0x234d, 0x234e, 0x234f,
    0x2350, 0x2351, 0x2352, 0x2353, 0x2354, 0x2355, 0x2356, 0x2357, /* 50 */
    0x2358, 0x2359, 0x235a, 0x214e, 0x216f, 0x214f, 0x2130, 0x2132,
    0x2121, 0x2361, 0x2362, 0x2363, 0x2364, 0x2365, 0x2366, 0x2367, /* 60 */
    0x2368, 0x2369, 0x236a, 0x236b, 0x236c, 0x236d, 0x236e, 0x236f,
    0x2370, 0x2371, 0x2372, 0x2373, 0x2374, 0x2375, 0x2376, 0x2377, /* 70 */
    0x2378, 0x2379, 0x237a, 0x2150, 0x2925, 0x2151, 0x2232, 0x2121,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 80 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2836,
    0x2835, 0x2833, 0x2834, 0x2832, 0x0000, 0x282c, 0x282d, 0x0000, /* 90 */
    0x282e, 0x282f, 0x2831, 0x2830, 0x0000, 0x0000, 0x0000, 0x0000,
    0x2121, 0x2166, 0x2125, 0x2176, 0x2165, 0x213f, 0x292e, 0x222c, /* a0 */
    0x2934, 0x222d, 0x222b, 0x222a, 0x215c, 0x214a, 0x214b, 0x2933,
    0x2167, 0x2333, 0x2643, 0x2334, 0x2335, 0x2336, 0x2645, 0x2651, /* b0 */
    0x2652, 0x2657, 0x2638, 0x2623, 0x264f, 0x2644, 0x264a, 0x2632,
    0x2641, 0x264d, 0x2d79, 0x2642, 0x264e, 0x2647, 0x2648, 0x2331, /* c0 */
    0x215e, 0x2654, 0x2650, 0x262b, 0x2332, 0x2251, 0x2231, 0x2653,
    0x2171, 0x2658, 0x226e, 0x2265, 0x2337, 0x2338, 0x2339, 0x2649, /* d0 */
    0x2655, 0x2162, 0x2656, 0x216b, 0x2330, 0x2646, 0x264b, 0x264c,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* e0 */
    0x236a, 0x236e, 0x236c, 0x2640, 0x217c, 0x217b, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* f0 */
    0x2121, 0x2121, 0x2121, 0x2121, 0x2121, 0x2121, 0x2121, 0x2121,
  };
  const int unitbl_6001a[] = {
    0x3000, 0x25cf, 0x51f9, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 00 */
    0x0000, 0x0000, 0x0000, 0x2191, 0x2193, 0x25b3, 0x25bd, 0xff0f,
    0x03a0, 0x25cb, 0x51f9, 0x2211, 0x00d7, 0xffe3, 0x2015, 0xff3f, /* 10 */
    0x51f8, 0x25a1, 0x25a0, 0x23b1, 0x23b0, 0x23a7, 0x23ad, 0xff3c,
    0x3000, 0xff01, 0xff02, 0xff03, 0xff04, 0xff05, 0xff06, 0x2019, /* 20 */
    0xff08, 0xff09, 0xff0a, 0xff0b, 0xff0c, 0xff0d, 0xff0e, 0xff0f,
    0xff10, 0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17, /* 30 */
    0xff18, 0xff19, 0xff1a, 0xff1b, 0xff1c, 0xff1d, 0xff1e, 0xff1f,
    0xff20, 0xff21, 0xff22, 0xff23, 0xff24, 0xff25, 0xff26, 0xff27, /* 40 */
    0xff28, 0xff29, 0xff2a, 0xff2b, 0xff2c, 0xff2d, 0xff2e, 0xff2f,
    0xff30, 0xff31, 0xff32, 0xff33, 0xff34, 0xff35, 0xff36, 0xff37, /* 50 */
    0xff38, 0xff39, 0xff3a, 0xff3b, 0xffe5, 0xff3d, 0xff3e, 0xff3f,
    0x3000, 0xff41, 0xff42, 0xff43, 0xff44, 0xff45, 0xff46, 0xff47, /* 60 */
    0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c, 0xff4d, 0xff4e, 0xff4f,
    0xff50, 0xff51, 0xff52, 0xff53, 0xff54, 0xff55, 0xff56, 0xff57, /* 70 */
    0xff58, 0xff59, 0xff5a, 0xff5b, 0xffe4, 0xff5d, 0xff5e, 0x3000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 80 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 90 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x3000, 0x2265, 0xff0e, 0xff0a, 0x2264, 0xff0f, 0x00b7, 0x2191, /* a0 */
    0x00bd, 0x2193, 0x2190, 0x2192, 0xff0b, 0xff08, 0xff09, 0x00bc,
    0x221e, 0xff13, 0x03b3, 0xff14, 0xff15, 0xff16, 0x03b5, 0x03c1, /* b0 */
    0x03c3, 0x03c8, 0x03a9, 0x0393, 0x03bf, 0x03b4, 0x03ba, 0x03a3,
    0x03b1, 0x03bd, 0x22bf, 0x03b2, 0x03be, 0x03b7, 0x03b8, 0xff11, /* c0 */
    0x00b1, 0x03c5, 0x03c0, 0x039b, 0xff12, 0x2295, 0xff0d, 0x03c4,
    0xffe0, 0x03c9, 0x2248, 0x221a, 0xff17, 0xff18, 0xff19, 0x03b9, /* d0 */
    0x03c6, 0x2260, 0x03c7, 0x00b0, 0xff10, 0x03b6, 0x03bb, 0x03bc,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* e0 */
    0x2660, 0x2665, 0x2666, 0x2663, 0x25cf, 0x25cb, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* f0 */
    0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000, 0x3000,
  };
  const int jisalttbl_6001a[][2] = {
    /* sigma */
    {0x13, 0x2632},
    /* double quotation mark */
    {0x22, 0x2149},
    /* apostrophe */
    {0x27, 0x2222f},
    /* hyphen-minus */
    {0x2d, 0x215d},
    /* square bracket */
    {0x5b, 0x214c},
    {0x5d, 0x214d},
    /* tilde */
    {0x7e, 0x2141},
    /* broken bar */
    {0x7c, 0x2143},
    /* inequality sign */
    {0xa1, 0x0000},
    {0xa4, 0x0000},
    /* dot */
    {0xa6, 0x2126},
    /* degree sign */
    {0xdb, 0x212c},
    /* suits of playing cards */
    {0xe8, 0x2225},
    {0xe9, 0x2227},
    {0xea, 0x2221},
    {0xeb, 0x217a},
  };
  const int unialttbl_6001a[][2] = {
    /* sigma */
    {0x13, 0x03a3},
    /* double quotation mark */
    {0x22, 0x201d},
    /* apostrophe */
    {0x27, 0xff07},
    /* hyphen-minus */
    {0x2d, 0x2212},
    /* square bracket */
    {0x5b, 0x3014},
    {0x5d, 0x3015},
    /* tilde */
    {0x7e, 0x301c},
    /* broken bar */
    {0x7c, 0xff5c},
    /* inequality sign */
    {0xa1, 0x2267},
    {0xa4, 0x2266},
    /* dot */
    {0xa6, 0x30fb},
    /* degree sign */
    {0xdb, 0x309c},
    /* suits of playing cards */
    {0xe8, 0x25b2},
    {0xe9, 0x25bc},
    {0xea, 0x25c6},
    {0xeb, 0x2605},
  };
  const int *tbl[2][2] = {
    {jistbl, unitbl},
    {jistbl_6001a, unitbl_6001a}
  };
  const int (*alttbl[][2])[2]= {
    {jisalttbl, unialttbl},
    {jisalttbl_6001a, unialttbl_6001a}
  };

  /* options */
  for (opt = 1; argc > opt && argv[opt][0] == '-'; opt++) {
    if (atoi(argv[opt]) == -8) {
      size_y_out = 8;
    } else if (atoi(argv[opt]) == -10) {
      size_y_out = 10;
    } else if (atoi(argv[opt]) == -12) {
      size_y_out = 12;
    } else if (strcmp(argv[opt], "-ascii") == 0) {
      ascii = 1;
    } else if (strcmp(argv[opt], "-pc6001a") == 0) {
      pc6001a = 1;
    } else if (strcmp(argv[opt], "-ascent") == 0) {
      use_ascent = 1;
    } else {
      printf("output height must be 8, 10, or 12\n");
      exit(1);
    }
  }
  if (pc6001a && size_y_out != SIZEY_DEFAULT) {
    printf("-pc6001a option needs height=12\n");
    exit(1);
  }

  if (argc < opt + 2) {
    printf("usage: bdf2cgrom [-8] [-10] [-12] [-ascii] [-pc6001a] [-ascent] fontfile.bdf outfile\n");
    exit(1);
  }

  if (strcmp(argv[opt], "-") == 0) {
    fp = stdin;
  } else {
    fp = fopen(argv[opt], "rb");
    if (fp == NULL) {
      printf("cannot open %s\n", argv[opt]);
      exit(1);
    }
  }

  /* convert */
  charset = getbdfpar(fp, ascii, use_ascent, &pixel_x, &pixel_y, &over_baseline);
  if (pc6001a) {
    altsize = sizeof(jisalttbl_6001a) / sizeof(jisalttbl_6001a[0][0]) / 2;
  } else {
    altsize = sizeof(jisalttbl) / sizeof(jisalttbl[0][0]) / 2;
  }

  while (fgets(line, sizeof(line), fp)) {
    if (sscanf(line, "ENCODING %d", &code) != 1 || code == 0) continue;

    fpos = ftell(fp);
    for (i = 0; i < NCHR; i++) {
      if ((ascii && code == i) || (!ascii && code == tbl[pc6001a][charset][i])) {
	code_is_ambiguous = isambiguous(code, charset);
	convert(fp, bmp60[i], size_x_out, size_y_out, pixel_x, pixel_y, over_baseline, code_is_ambiguous);
	fseek(fp, fpos, SEEK_SET);
      }
    }
    for (i = 0; i < altsize; i++) {
      if (code == alttbl[pc6001a][charset][i][1]) {
	chr = alttbl[pc6001a][charset][i][0];
	if (isnodata(bmp60[chr])) {
	  code_is_ambiguous = isambiguous(code, charset);
	  convert(fp, bmp60[chr], size_x_out, size_y_out, pixel_x , pixel_y, over_baseline, code_is_ambiguous);
	  fseek(fp, fpos, SEEK_SET);
	}
      }
    }
  }
  fclose(fp);

  /* calculate */
  calcchr(bmp60, pc6001a, size_y_out);

  /* no data */
  for (i = 0; i < NCHR; i++) {
    if (isnodata(bmp60[i]) && tbl[pc6001a][JIS][i] != 0x2121) {
      for (j = 0; j < 11; j++) {
	bmp60[i][j] = (1 << 8) - 2;
      }
      bmp60[i][5] = 0;
    }
  }

  /* output */
  strncpy(outfile, argv[opt + 1], sizeof(outfile));
  if (strcmp(outfile, "-") == 0) {
    fp = stdout;
  } else {
    fp = fopen(outfile, "wb");
    if (fp == NULL) {
      printf("cannot open %s\n", argv[opt + 1]);
      exit(1);
    }
  }
  for (i = 0; i < NCHR; i++) {
    for (j = 0; j < 16; j++) {
      fputc(bmp60[i][j], fp);
    }
  }
  fclose(fp);

  return 0;
}


int getbdfpar(FILE *fp, int ascii, int use_ascent, int *pixel_x, int *pixel_y, int *over_baseline)
{
  int code = 0;
  int xmin = 999, xmax = -1;
  int ymin = 999, ymax = -1;
  int ascent, descent;
  int size_x, size_y;
  int offset_x = 0, offset_y = 0;
  int result = JIS;
  char line[100] = "";
  char charset[100] = "";

  ascent = -1;
  descent = -1;

  while (fgets(line, sizeof(line), fp)) {
    if (sscanf(line, "FONTBOUNDINGBOX %d %d %d %d", pixel_x, pixel_y, &offset_x, &offset_y) == 4) {
      *over_baseline = *pixel_y + offset_y;
    }
    sscanf(line, "FONT_ASCENT %d", &ascent);
    sscanf(line, "FONT_DESCENT %d", &descent);
    if (sscanf(line, "CHARSET_REGISTRY %s", charset) == 1) {
      if (strncasecmp(charset, "\"JISX0208", 9) == 0 ||
	  strncasecmp(charset, "\"JISX0213", 9) == 0) {
	result = JIS;
      } else if (strcasecmp(charset, "\"ISO10646\"") == 0) {
	result = UNI;
      } else if (!ascii) {
	printf("cannot convert for charset %s\n", charset);
	exit(1);
      }
    }
    sscanf(line, "ENCODING %d", &code);
    if (sscanf(line, "BBX %d %d %d %d", &size_x, &size_y, &offset_x, &offset_y) == 4) {
      if (result != UNI ||
	  (0x0000 <= code && code <= 0x03ff) ||
	  (0x2000 <= code && code <= 0x2e39) ||  /* 2e3a = 2-em dash */
	  (0x2e3c <= code && code <= 0x9fff) ||  /* 2e3b = 3-em dash */
	  (0xf900 <= code && code <= 0xfaff) ||
	  (0xff00 <= code && code <= 0xffff)
	  ) {
        if (offset_x < xmin) {
	  xmin = offset_x;
	}
	if (offset_x + size_x > xmax) {
	  xmax = offset_x + size_x;
	}
	if (offset_y < ymin) {
	  ymin = offset_y;
	}
	if (offset_y + size_y > ymax) {
	  ymax = offset_y + size_y;
	}
      }
    }
  }

  *pixel_x = xmax - xmin;
  *pixel_y = ymax - ymin;
  if (use_ascent) {
    *pixel_y = ascent + descent;
    *over_baseline = ascent;
  }
  if (*pixel_x < 0 || *pixel_y < 0 || *over_baseline < 0 || strlen(charset) == 0) {
    printf("illegal bdf\n");
    exit(1);
  }
  rewind(fp);

  return result;
}


void convert(FILE *fp, unsigned char bmp60[16], int size_x_out, int size_y_out, int pixel_x, int pixel_y, int over_baseline, int code_is_ambiguous)
{
  int i, j;
  int x, y;
  int size_x, size_y;
  int offset_x, offset_y;
  unsigned long long data;
  double ratio_x, ratio_y;
  char line[100];
  unsigned char bmpbdf[BDFSIZE][BDFSIZE];

  memset(bmpbdf, 0, sizeof(bmpbdf));
  do {
    if (fgets(line, sizeof(line), fp) == NULL) break;
  } while (sscanf(line, "BBX %d %d %d %d", &size_x, &size_y, &offset_x, &offset_y) != 4);

  if (code_is_ambiguous && size_x + offset_x <= pixel_x / 2) {
    pixel_x /= 2;
  }

  if (pixel_x > size_x_out) {
    ratio_x = (double)pixel_x / size_x_out;
  } else {
    ratio_x = 1;
    offset_x += (size_x_out - pixel_x) / 2;
  }
  if (pixel_y > size_y_out) {
    ratio_y = (double)pixel_y / size_y_out;
  } else {
    ratio_y = 1;
    offset_y -= (size_y_out - pixel_y) / 2;
  }

  if (offset_x < 0) {
    offset_x = 0;
  }

  do {
    if (fgets(line, sizeof(line), fp) == NULL) break;
  } while (strncmp(line, "BITMAP", 6) != 0);

  for (i = 0; i < size_y; i++) {
    if (fgets(line, sizeof(line), fp) == NULL) break;
    while (sscanf(line, "%llx", &data) != 1) {
      ;
    }
    for (j = 0; j < size_x; j++) {
      x = j + offset_x;
      y = i + over_baseline - size_y - offset_y;
      if (0 <= x && x <= BDFSIZE && 0 <= y && y <= BDFSIZE) {
	bmpbdf[x][y] = (data >> ((BDFSIZE - size_x) % 8 + size_x - j - 1)) & 1;
      }
    }
  }

  if (size_y > 0) {
    for (i = 0; i < size_y_out; i++) {
      bmp60[i] = 0;
      for (j = 0; j < size_x_out; j++) {
	data = 0;
	for (x = j * ratio_x; x < (int)((j + 1) * ratio_x + .01) && x < BDFSIZE; x++) {
	  for (y = i * ratio_y; y < (int)((i + 1) * ratio_y + .01) && y < BDFSIZE; y++) {
	    data += bmpbdf[x][y];
	  }
	}
	if ((double)data / (x - (int)(j * ratio_x)) * (y - (int)(i * ratio_y)) > 0) {
	  bmp60[i] |= 1 << (size_x_out - j - 1);
	}
      }
    }
  }
}


int isnodata(unsigned char *bmp60)
{
  int i;
  unsigned char data = 0;
  for (i = 0; i < 16; i++) {
    data |= bmp60[i];
  }
  return !data;
}


void calcchr(unsigned char bmp60[][16], int pc6001a, int size_y_out)
{
  int i, j;
  int chr;
  const unsigned char dots2 = 0x03;
  const unsigned char dots3 = 0x07;
  const unsigned char dots4 = 0x0f;
  const unsigned char dots8 = 0xff;
  const int lower[] = {11, 10, 9, 8, 6, 4, 2, 0};
  const int line_shape[][4] = {
    /* left, right, up, down */
    {1, 1, 1, 0}, {1, 1, 0, 1}, {1, 0, 1, 1}, /* 11-13 */
    {0, 1, 1, 1}, {1, 1, 1, 1}, {0, 0, 1, 1}, {1, 1, 0, 0}, /* 14-17 */
    {0, 1, 0, 1}, {1, 0, 0, 1}, {0, 1, 1, 0}, {1, 0, 1, 0}, /* 18-1b */
  };
  const int boxdraw_6001a[] =
    {0x90, 0x91, 0x92, 0x93, 0x8f, 0x96, 0x95, 0x98, 0x99, 0x9a, 0x9b};

  /* box drawings */
  for (i = 0; i < sizeof(line_shape) / sizeof(line_shape[0][0]) / 4; i++) {
    chr = pc6001a ? boxdraw_6001a[i] : 0x11 + i;
    for (j = 0; j < 12; j++) {
      bmp60[chr][j] = 0;
    }
    bmp60[chr][size_y_out / 2 - 1] = bmp60[chr][size_y_out / 2] =
      (line_shape[i][0] ? (dots4 << 4) : 0) |
      (line_shape[i][1] ? dots4 : 0);
    for (j = 0; j < size_y_out / 2 + 1; j++) {
      bmp60[chr][j] |= line_shape[i][2] ? (dots2 << 3) : 0;
    }
    for (j = size_y_out / 2 - 1; j < size_y_out; j++) {
      bmp60[chr][j] |= line_shape[i][3] ? (dots2 << 3) : 0;
    }
  }

  if (!pc6001a) return;

  /* circle */
  bmp60[0x04][0] = bmp60[0x04][3] = dots2 << 5;
  bmp60[0x04][1] = bmp60[0x04][2] = 0x90;
  
  for (i = 0; i < 4; i++) {
    bmp60[0x03][i] = bmp60[0x04][i] >> 2;
    bmp60[0x0a][i] = bmp60[0x04][i] >> 4;
    bmp60[0x05][i + 4] = bmp60[0x04][i];
    bmp60[0x09][i + 4] = bmp60[0x0a][i];
    bmp60[0x06][i + 8] = bmp60[0x04][i];
    bmp60[0x07][i + 8] = bmp60[0x03][i];
    bmp60[0x08][i + 8] = bmp60[0x0a][i];
  }
  
  /* block */
  for (i = 0; i < 8; i++) {
    for (j = lower[i]; j < 12; j++) {
      chr = 0x80 + i;
      bmp60[chr][j] = dots8;
    }
  }
  for (i = 0; i < 7; i++) {
    for (j = 0; j < 12; j++) {
      chr = 0x88 + i;
      bmp60[chr][j] = 0x100 - (0x80 >> i);
    }
  }

  /* box drawings */
  bmp60[0x94][0] = bmp60[0x94][1] = dots8;
  for (i = 0; i < 12; i++) {
    bmp60[0x97][i] = dots2 << 6;
  }

  /* 1/4 arc */
  for (i = 0; i < 4; i++) {
    chr = 0x9c + i;
    for (j = 0; j < 4; j++) {
      bmp60[chr][j] = (i & 0x02) ? (dots2 << 3) : 0;
    }
    for (j = 8; j < 12; j++) {
      bmp60[chr][j] = (i & 0x02) ? 0 : (dots2 << 3);
    }
    bmp60[chr][(i & 0x02) ? 4 : 7] = (i & 0x01) ? (dots3 << 3) : (dots3 << 2);
    bmp60[chr][(i & 0x02) ? 5 : 6] = (i & 0x01) ? (dots4 << 4) : (dots4 << 0);
    bmp60[chr][(i & 0x02) ? 6 : 5] = (i & 0x01) ? (dots3 << 5) : (dots3 << 0);
  }

  /* double horizontal */
  for (i = 0; i < 4; i++) {
    chr = 0xe0 + i;
    for (j = 0; j < 12; j++) {
      bmp60[chr][j] = (i == 0) ? 0 : (dots2 << 3);
    }
    bmp60[chr][2] |= (i == 1) ? 0 : (dots4 << 4);
    bmp60[chr][2] |= (i == 3) ? 0 : (dots4 << 0);
    bmp60[chr][3] = bmp60[chr][2];
    bmp60[chr][8] = bmp60[chr][2];
    bmp60[chr][9] = bmp60[chr][2];
  }

  /* triangle */
  for (j = 0; j < 12; j++) {
    bmp60[0xe4][j] = (0x02 << (j * 8/12)) - 0x01;
    bmp60[0xe5][j] = 0x100 - (0x80 >> (j * 8/12));
    bmp60[0xe6][j] = (0x100 >> (j * 8/12)) - 0x01;
    bmp60[0xe7][j] = 0x100 - (0x01 << (j * 8/12));
  }

  /* box drawings diagonal */
  for (i = 0; i < 3; i++) {
    chr = 0xee + i;
    for (j = 0; j < 12; j++) {
      if (i == 0 || i == 2) {
	bmp60[chr][j] = 0x01 << (j * 2/3);
      }
      if (i == 1 || i == 2) {
	bmp60[chr][j] |= 0x80 >> (j * 2/3);
      }
    }
  }

  /* dotted line */
  bmp60[0xf1][5] = bmp60[0xf1][6] = 0xaa;
  bmp60[0xf7][5] = bmp60[0xf7][6] = 0xaa;
  bmp60[0xf3][5] = bmp60[0xf3][6] = dots8;
  bmp60[0xf4][5] = bmp60[0xf4][6] = dots8;
  bmp60[0xf5][5] = bmp60[0xf5][6] = 0xa0;
  bmp60[0xf6][5] = bmp60[0xf6][6] = 0x0a;

  for (i = 0; i < 6; i++) {
    bmp60[0xf5][i] |= dots2 << 3;
    bmp60[0xf6][i] |= dots2 << 3;
    if (i % 2 == 0) {
      bmp60[0xf2][i] |= dots2 << 3;
      bmp60[0xf3][i] |= dots2 << 3;
      bmp60[0xf7][i] |= dots2 << 3;
    }
  }
  for (i = 6; i < 12; i++) {
    bmp60[0xf5][i] |= dots2 << 3;
    bmp60[0xf6][i] |= dots2 << 3;
    if (i % 2 == 0) {
      bmp60[0xf2][i] |= dots2 << 3;
      bmp60[0xf4][i] |= dots2 << 3;
      bmp60[0xf7][i] |= dots2 << 3;
    }
  }
}


int isambiguous(int code, int charset)
{
  int i;
  const int eawtbl[] = {
    0x00b0, 0x00b1, 0x00b7, 0x00bc, 0x00bd, 0x00d7,
    0x0393, 0x039b, 0x03a0, 0x03a3, 0x03a9, 0x03b1, 0x03b2, 0x03b3,
    0x03b4, 0x03b5, 0x03b6, 0x03b7, 0x03b8, 0x03b9, 0x03ba, 0x03bb,
    0x03bc, 0x03bd, 0x03be, 0x03bf, 0x03c0, 0x03c1, 0x03c3, 0x03c4,
    0x03c5, 0x03c6, 0x03c7, 0x03c8, 0x03c9,
    0x2015, 0x2019, 0x201d, 0x2190, 0x2191, 0x2192, 0x2193, 0x2211,
    0x2212, 0x221a, 0x221e, 0x2248, 0x2260, 0x2264, 0x2265, 0x2266,
    0x2267, 0x2295, 0x22bf, 0x23a7, 0x23ad, 0x23b0, 0x23b1, 0x25a0,
    0x25a1, 0x25b2, 0x25b3, 0x25bc, 0x25bd, 0x25c6, 0x25cb, 0x25cf,
    0x2660, 0x2605, 0x2663, 0x2665, 0x2666,
  };

  if (charset == JIS) return 0;

  for (i = 0; i < sizeof(eawtbl) / sizeof(eawtbl[0]); i++) {
    if (code == eawtbl[i]) {
      return 1;
    }
  }
  return 0;
}
