/*
  bdf2cgrom66.c  2021.4.25 version
    convert .bdf font file to CGROM60.66/CGROM66.66 format for PC-6601
    by AKIKAWA, Hisashi
    This software is redistributable under the LGPLv2.1 or any later version.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define BDFSIZE 64

enum {JIS, UNI};

int getbdfpar(FILE *fp, int ascii, int use_ascent, int *pixel_x, int *pixel_y, int *over_baseline);
void convert(FILE *fp, unsigned char bmp60[16], int size_x_out, int size_y_out, int pixel_x, int pixel_y, int over_baseline, int code_is_ambiguous, double par1, double par2, double par3, double par4);
int isnodata(unsigned char *bmp60);
void calcline(unsigned char bmp60[][16], int size_y_out);
int isambiguous(int code, int charset);

int main(int argc, char *argv[])
{
  int i, j, k;
  int opt;
  int ascii = 0, use_ascent = 0;
  int code, chr;
  int code_is_ambiguous;
  int pixel_x = 16, pixel_y = 16, over_baseline = 16;
  int charset;
  char line[100];
  char outfile[FILENAME_MAX + 1];
  int mixdone1[0x100] = {0}, mixdone2[0x100] = {0};
  unsigned char bmp60[0x200][0x10] = {0};
  unsigned char bmpmix[0x100][0x10] = {0};
  unsigned char bmptmp[0x10] = {0};
  long fpos;
  FILE *fp1, *fp2, *fpout;

  const int jistbl[] = {
    0x2121, 0x376e, 0x3250, 0x3f65, 0x4c5a, 0x3662, 0x455a, 0x467c, /* 00 */
    0x472f, 0x315f, 0x3b7e, 0x4a2c, 0x4943, 0x4934, 0x4069, 0x4b7c,
    0x2650, 0x2835, 0x2833, 0x2834, 0x2832, 0x2836, 0x282d, 0x282c, /* 10 */
    0x282e, 0x282f, 0x2831, 0x2830, 0x215f, 0x4267, 0x4366, 0x3e2e,
    0x2121, 0x212a, 0x2230, 0x2174, 0x2170, 0x2173, 0x2175, 0x2147, /* 20 */
    0x214a, 0x214b, 0x2176, 0x215c, 0x2124, 0x2231, 0x2125, 0x213f,
    0x2330, 0x2331, 0x2332, 0x2333, 0x2334, 0x2335, 0x2336, 0x2337, /* 30 */
    0x2338, 0x2339, 0x2127, 0x2128, 0x2163, 0x2161, 0x2164, 0x2129,
    0x2177, 0x2341, 0x2342, 0x2343, 0x2344, 0x2345, 0x2346, 0x2347, /* 40 */
    0x2348, 0x2349, 0x234a, 0x234b, 0x234c, 0x234d, 0x234e, 0x234f,
    0x2350, 0x2351, 0x2352, 0x2353, 0x2354, 0x2355, 0x2356, 0x2357, /* 50 */
    0x2358, 0x2359, 0x235a, 0x214e, 0x216f, 0x214f, 0x2130, 0x2132,
    0x2121, 0x2361, 0x2362, 0x2363, 0x2364, 0x2365, 0x2366, 0x2367, /* 60 */
    0x2368, 0x2369, 0x236a, 0x236b, 0x236c, 0x236d, 0x236e, 0x236f,
    0x2370, 0x2371, 0x2372, 0x2373, 0x2374, 0x2375, 0x2376, 0x2377, /* 70 */
    0x2378, 0x2379, 0x237a, 0x2150, 0x2925, 0x2151, 0x2232, 0x2121,
    0x263a, 0x263e, 0x2640, 0x263c, 0x217b, 0x217c, 0x2472, 0x2421, /* 80 */
    0x2423, 0x2425, 0x2427, 0x2429, 0x2463, 0x2465, 0x2467, 0x2443,
    0x2121, 0x2422, 0x2424, 0x2426, 0x2428, 0x242a, 0x242b, 0x242d, /* 90 */
    0x242f, 0x2431, 0x2433, 0x2435, 0x2437, 0x2439, 0x243b, 0x243d,
    0x2121, 0x2123, 0x2156, 0x2157, 0x2122, 0x2126, 0x2572, 0x2521, /* a0 */
    0x2523, 0x2525, 0x2527, 0x2529, 0x2563, 0x2565, 0x2567, 0x2543,
    0x213c, 0x2522, 0x2524, 0x2526, 0x2528, 0x252a, 0x252b, 0x252d, /* b0 */
    0x252f, 0x2531, 0x2533, 0x2535, 0x2537, 0x2539, 0x253b, 0x253d,
    0x253f, 0x2541, 0x2544, 0x2546, 0x2548, 0x254a, 0x254b, 0x254c, /* c0 */
    0x254d, 0x254e, 0x254f, 0x2552, 0x2555, 0x2558, 0x255b, 0x255e,
    0x255f, 0x2560, 0x2561, 0x2562, 0x2564, 0x2566, 0x2568, 0x2569, /* d0 */
    0x256a, 0x256b, 0x256c, 0x256d, 0x256f, 0x2573, 0x212b, 0x212c,
    0x243f, 0x2441, 0x2444, 0x2446, 0x2448, 0x244a, 0x244b, 0x244c, /* e0 */
    0x244d, 0x244e, 0x244f, 0x2452, 0x2455, 0x2458, 0x245b, 0x245e,
    0x245f, 0x2460, 0x2461, 0x2462, 0x2464, 0x2466, 0x2468, 0x2469, /* f0 */
    0x246a, 0x246b, 0x246c, 0x246d, 0x246f, 0x2473, 0x2121, 0x2121,

    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 00 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 10 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 20 */
    0x222a, 0x222b, 0x222c, 0x222d, 0x2325, 0x2328, 0x2327, 0x2326,
    0x2330, 0x2331, 0x2332, 0x2333, 0x2334, 0x2335, 0x2336, 0x2337, /* 30 */
    0x2338, 0x2339, 0x0000, 0x0000, 0x0000, 0x2167, 0x2267, 0x2262,
    0x2958, 0x2964, 0x2970, 0x2960, 0x295e, 0x295f, 0x2b2a, 0x295a, /* 40 */
    0x2971, 0x296c, 0x2622, 0x2d79, 0x262b, 0x2632, 0x2635, 0x2638,
    0x2641, 0x2642, 0x2643, 0x2644, 0x2645, 0x2648, 0x264b, 0x264c, /* 50 */
    0x264d, 0x2651, 0x2652, 0x2654, 0x2655, 0x2658, 0x0000, 0x225f,
    0x2361, 0x2b37, 0x2b30, 0x295c, 0x2b43, 0x2b38, 0x0000, 0x0000, /* 60 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x2359, 0x2349,
    0x2a79, 0x2a69, 0x2a68, 0x2966, 0x2a76, 0x2a7c, 0x295d, 0x0000, /* 70 */
    0x0000, 0x2b34, 0x2b3c, 0x2b35, 0x2959, 0x0000, 0x0000, 0x0000,
    0x215f, 0x2160, 0x215e, 0x235b, 0x2166, 0x2162, 0x2165, 0x2163, /* 80 */
    0x2153, 0x225c, 0x213f, 0x2265, 0x2241, 0x2240, 0x2168, 0x2268,
    0x225e, 0x0000, 0x2224, 0x2225, 0x217a, 0x0000, 0x0000, 0x0000, /* 90 */
    0x0000, 0x0000, 0x2667, 0x0000, 0x2228, 0x0000, 0x2178, 0x0000,
    0x2169, 0x216a, 0x317a, 0x464c, 0x233a, 0x217d, 0x0000, 0x233b, /* a0 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x2251, 0x2252, 0x2253, 0x0000,
    0x0000, 0x5244, 0x0000, 0x0000, 0x0000, 0x0000, 0x217e, 0x4a38, /* b0 */
    0x0000, 0x0000, 0x217b, 0x2722, 0x0000, 0x0000, 0x0000, 0x0000,
    0x217c, 0x217c, 0x2223, 0x2142, 0x2154, 0x215f, 0x2131, 0x2152, /* c0 */
    0x2131, 0x2140, 0x2132, 0x2132, 0x2161, 0x217c, 0x2142, 0x2142,
    0x217c, 0x217c, 0x2223, 0x2142, 0x215f, 0x2154, 0x2154, 0x213f, /* d0 */
    0x2131, 0x2132, 0x213f, 0x2140, 0x2161, 0x2142, 0x217c, 0x2142,
    0x217c, 0x217c, 0x2223, 0x2142, 0x215f, 0x2155, 0x2155, 0x2140, /* e0 */
    0x2131, 0x2132, 0x2140, 0x213f, 0x2161, 0x2142, 0x2142, 0x2142,
    0x217c, 0x217c, 0x2223, 0x224b, 0x2155, 0x215f, 0x2131, 0x2153, /* f0 */
    0x2131, 0x213f, 0x2132, 0x2132, 0x2161, 0x217c, 0x217c, 0x2223,
  };

  const int unitbl[] = {
    0x3000, 0x6708, 0x706b, 0x6c34, 0x6728, 0x91d1, 0x571f, 0x65e5, /* 00 */
    0x5e74, 0x5186, 0x6642, 0x5206, 0x79d2, 0x767e, 0x5343, 0x4e07,
    0x03c0, 0x253b, 0x2533, 0x252b, 0x2523, 0x254b, 0x2503, 0x2501, /* 10 */
    0x250f, 0x2513, 0x2517, 0x251b, 0x00d7, 0x5927, 0x4e2d, 0x5c0f,
    0x3000, 0xff01, 0xff02, 0xff03, 0xff04, 0xff05, 0xff06, 0x2019, /* 20 */
    0xff08, 0xff09, 0xff0a, 0xff0b, 0xff0c, 0xff0d, 0xff0e, 0xff0f,
    0xff10, 0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17, /* 30 */
    0xff18, 0xff19, 0xff1a, 0xff1b, 0xff1c, 0xff1d, 0xff1e, 0xff1f,
    0xff20, 0xff21, 0xff22, 0xff23, 0xff24, 0xff25, 0xff26, 0xff27, /* 40 */
    0xff28, 0xff29, 0xff2a, 0xff2b, 0xff2c, 0xff2d, 0xff2e, 0xff2f,
    0xff30, 0xff31, 0xff32, 0xff33, 0xff34, 0xff35, 0xff36, 0xff37, /* 50 */
    0xff38, 0xff39, 0xff3a, 0xff3b, 0xffe5, 0xff3d, 0xff3e, 0xff3f,
    0x3000, 0xff41, 0xff42, 0xff43, 0xff44, 0xff45, 0xff46, 0xff47, /* 60 */
    0xff48, 0xff49, 0xff4a, 0xff4b, 0xff4c, 0xff4d, 0xff4e, 0xff4f,
    0xff50, 0xff51, 0xff52, 0xff53, 0xff54, 0xff55, 0xff56, 0xff57, /* 70 */
    0xff58, 0xff59, 0xff5a, 0xff5b, 0xffe4, 0xff5d, 0xff5e, 0x3000,
    0x2660, 0x2665, 0x2663, 0x2666, 0x25cb, 0x25cf, 0x3092, 0x3041, /* 80 */
    0x3043, 0x3045, 0x3047, 0x3049, 0x3083, 0x3085, 0x3087, 0x3063,
    0x3000, 0x3042, 0x3044, 0x3046, 0x3048, 0x304a, 0x304b, 0x304d, /* 90 */
    0x304f, 0x3051, 0x3053, 0x3055, 0x3057, 0x3059, 0x305b, 0x305d,
    0x3000, 0x3002, 0x300c, 0x300d, 0x3001, 0x30fb, 0x30f2, 0x30a1, /* a0 */
    0x30a3, 0x30a5, 0x30a7, 0x30a9, 0x30e3, 0x30e5, 0x30e7, 0x30c3,
    0x30fc, 0x30a2, 0x30a4, 0x30a6, 0x30a8, 0x30aa, 0x30ab, 0x30ad, /* b0 */
    0x30af, 0x30b1, 0x30b3, 0x30b5, 0x30b7, 0x30b9, 0x30bb, 0x30bd,
    0x30bf, 0x30c1, 0x30c4, 0x30c6, 0x30c8, 0x30ca, 0x30cb, 0x30cc, /* c0 */
    0x30cd, 0x30ce, 0x30cf, 0x30d2, 0x30d5, 0x30d8, 0x30db, 0x30de,
    0x30df, 0x30e0, 0x30e1, 0x30e2, 0x30e4, 0x30e6, 0x30e8, 0x30e9, /* d0 */
    0x30ea, 0x30eb, 0x30ec, 0x30ed, 0x30ef, 0x30f3, 0x309b, 0x309c,
    0x305f, 0x3061, 0x3064, 0x3066, 0x3068, 0x306a, 0x306b, 0x306c, /* e0 */
    0x306d, 0x306e, 0x306f, 0x3072, 0x3075, 0x3078, 0x307b, 0x307e,
    0x307f, 0x3080, 0x3081, 0x3082, 0x3084, 0x3086, 0x3088, 0x3089, /* f0 */
    0x308a, 0x308b, 0x308c, 0x308d, 0x308f, 0x3093, 0x3000, 0x3000,

    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 00 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, /* 10 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x27f2, 0x27f3, /* 20 */
    0x2192, 0x2190, 0x2191, 0x2193, 0x2197, 0x2199, 0x2196, 0x2198,
    0xff10, 0xff11, 0xff12, 0xff13, 0xff14, 0xff15, 0xff16, 0xff17, /* 30 */
    0xff18, 0xff19, 0x2491, 0x2492, 0x2493, 0x221e, 0x221d, 0x2252,
    0x00e2, 0x00ee, 0x00fb, 0x00ea, 0x00e8, 0x00e9, 0x0153, 0x00e4, /* 40 */
    0x00fc, 0x00f6, 0x0392, 0x22bf, 0x039b, 0x03a3, 0x03a6, 0x03a9,
    0x03b1, 0x03b2, 0x03b3, 0x03b4, 0x03b5, 0x03b8, 0x03bb, 0x03bc, /* 50 */
    0x03bd, 0x03c1, 0x03c3, 0x03c5, 0x03c6, 0x03c9, 0x03d5, 0x2202,
    0x0251, 0x028c, 0x0259, 0x00e6, 0x025a, 0x0254, 0x0000, 0x0000, /* 60 */
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x028f, 0x026a,
    0x014b, 0x0292, 0x0283, 0x00f0, 0x0272, 0x0281, 0x00e7, 0x0000, /* 70 */
    0x0000, 0x026f, 0x0265, 0x028a, 0x00e3, 0x0000, 0x0000, 0x0000,
    0x00d7, 0x00f7, 0x00b1, 0x2213, 0x2265, 0x2260, 0x2264, 0xff1c, /* 80 */
    0x3009, 0x2220, 0xff0f, 0x221a, 0x2229, 0x222a, 0x2234, 0x2235,
    0xfe35, 0xfe36, 0x25b3, 0x25b2, 0x2605, 0x0000, 0x25aa, 0x25a3, /* 90 */
    0x0000, 0x0000, 0x260e, 0x2733, 0x203b, 0x1d10b, 0x00a7, 0x1d10c,
    0x2642, 0x2640, 0x51f9, 0x51f8, 0x2299, 0x25ce, 0x0000, 0x0000, /* a0 */
    0x29b6, 0x0000, 0x0000, 0x229d, 0x2295, 0x2296, 0x2297, 0x2298,
    0x26e9, 0x534d, 0x26eb, 0x26e8, 0x3036, 0x26fb, 0x25c7, 0x6587, /* b0 */
    0x2b59, 0x0000, 0x25cb, 0x0411, 0x26e3, 0x2693, 0x2693, 0x1f6e7,
    0x25cf, 0x25cf, 0x25a0, 0x2016, 0x300a, 0x00d7, 0x203e, 0x3008, /* c0 */
    0x203e, 0xff3c, 0xff3f, 0xff3f, 0xff1d, 0x25cf, 0x2016, 0x2016,
    0x25cf, 0x25cf, 0x25a0, 0x2016, 0x00d7, 0x300a, 0x300a, 0xff0f, /* d0 */
    0x203e, 0xff3f, 0xff0f, 0xff3c, 0xff1d, 0x2016, 0x25cf, 0x2016,
    0x25cf, 0x25cf, 0x25a0, 0x2016, 0x00d7, 0x300b, 0x300b, 0xff3c, /* e0 */
    0x203e, 0xff3f, 0xff3c, 0xff0f, 0xff1d, 0x2016, 0x2016, 0x2016,
    0x25cf, 0x25cf, 0x25a0, 0x2228, 0x300b, 0x00d7, 0x203e, 0x3009, /* f0 */
    0x203e, 0xff0f, 0xff3f, 0xff3f, 0xff1d, 0x25cf, 0x25cf, 0x25a0,
  };

  const int jisalttbl[][2] = {
    /* suits of playing cards */
    {0x80, 0x2225}, {0x81, 0x2227}, {0x82, 0x217a}, {0x83, 0x2221},
    /* double quotation mark */
    {0x22, 0x2149},
    /* apostrophe */
    {0x27, 0x2222f},
    /* hyphen-minus */
    {0x2d, 0x215d},
    /* square bracket */
    {0x5b, 0x214c}, {0x5d, 0x214d},
    /* tilde */
    {0x7e, 0x2141},
    /* broken bar */
    {0x7c, 0x2143},
    /* semi-graphic */
    {0x126, 0x0000}, {0x127, 0x0000}, {0x14b, 0x2624}, {0x15e, 0x2655},
    {0x160, 0x0000}, {0x16e, 0x2359}, {0x16f, 0x2349}, {0x172, 0x2269},
    {0x184, 0x0000}, {0x186, 0x0000}, {0x190, 0x0000}, {0x1bf, 0x3b4e},
  };

  const int unialttbl[][2] = {
    /* suits of playing cards */
    {0x80, 0x25b2}, {0x81, 0x25bc}, {0x82, 0x2605}, {0x83, 0x25c6},
    /* double quotation mark */
    {0x22, 0x201d},
    /* apostrophe */
    {0x27, 0xff07},
    /* hyphen-minus */
    {0x2d, 0x2212},
    /* square bracket */
    {0x5b, 0x3014}, {0x5d, 0x3015},
    /* tilde */
    {0x7e, 0x301c},
    /* broken bar */
    {0x7d, 0xff5c},
    /* semi-graphic */
    {0x126, 0x21ba}, {0x127, 0x21bb}, {0x14b, 0x0394}, {0x15e, 0x03c6},
    {0x160, 0xff41}, {0x16e, 0xff39}, {0x16f, 0xff29}, {0x172, 0x222b},
    {0x184, 0x2267}, {0x186, 0x2266}, {0x190, 0x2312}, {0x1bf, 0x58eb},
  };

  const int jismixtbl[][3] = {
    /* left-right */
    {0x66, 0x2361, 0x2127}, {0x67, 0x2369, 0x2127},
    {0x68, 0x2375, 0x2127}, {0x69, 0x2b30, 0x2127},
    {0x6a, 0x2645, 0x2127}, {0x6b, 0x2b43, 0x2127},
    {0x6c, 0x236f, 0x2127}, {0x6d, 0x2379, 0x2127},
    {0x78, 0x2655, 0x2127}, {0x77, 0x2372, 0x212c},
    {0x3a, 0x2331, 0x2330}, {0x3b, 0x2331, 0x2331},
    {0x3c, 0x2331, 0x2332}, {0x46, 0x236f, 0x2365},
    {0x63, 0x2361, 0x2365},
    /* down-up */
    {0x7c, 0x2361, 0x2141}, {0x7d, 0x2645, 0x2141},
    {0x7e, 0x2b2a, 0x2141}, {0x7f, 0x2b38, 0x2141},
    {0x40, 0x2361, 0x2130}, {0x41, 0x2369, 0x2130},
    {0x42, 0x2375, 0x2130}, {0x43, 0x2365, 0x2130},
    {0x44, 0x2365, 0x212e}, {0x45, 0x2365, 0x212d},
    {0x47, 0x2361, 0x212f}, {0x48, 0x2375, 0x212f},
    {0x49, 0x236f, 0x212f}, {0x83, 0x215c, 0x215d},
    /* square */
    {0x97, 0x2222, 0x2223}, {0x98, 0x2222, 0x2222},
    {0x99, 0x2222, 0x215f}, {0xb9, 0x2222, 0x2348},
    /* circle */
    {0xa4, 0x217b, 0x2126}, {0xa6, 0x217b, 0x2222},
    {0xa7, 0x217b, 0x2223}, {0xa8, 0x217b, 0x2143},
    {0xa9, 0x217b, 0x2535}, {0xaa, 0x217b, 0x215c},
    {0xab, 0x217b, 0x215d}, {0xac, 0x217b, 0x215c},
    {0xad, 0x217b, 0x215d}, {0xae, 0x217b, 0x215f},
    {0xaf, 0x217b, 0x213f}, {0xb4, 0x217b, 0x2229},
    {0xb8, 0x217b, 0x215f},
    /* other */
    {0x61, 0x224a, 0x224a}, {0x95, 0x217c, 0x217c},
    {0x96, 0x2223, 0x2223}, {0x9b, 0x215c, 0x215f},
  };

  const int unimixtbl[][3] = {
    /* left-right */
    {0x66, 0xff41, 0xff1a}, {0x67, 0xff49, 0xff1a},
    {0x68, 0xff55, 0xff1a}, {0x69, 0x0259, 0xff1a},
    {0x6a, 0x03b5, 0xff1a}, {0x6b, 0x0259, 0xff1a},
    {0x6c, 0xff4f, 0xff1a}, {0x6d, 0xff59, 0xff1a},
    {0x78, 0x03d5, 0xff1a}, {0x77, 0xff52, 0x2019},
    {0x3a, 0xff11, 0xff10}, {0x3b, 0xff11, 0xff11},
    {0x3c, 0xff11, 0xff12}, {0x46, 0xff4f, 0xff45},
    {0x63, 0xff41, 0xff45},
    /* down-up */
    {0x7c, 0xff41, 0x301c}, {0x7d, 0x03b5, 0x301c},
    {0x7e, 0x0153, 0x301c}, {0x7f, 0x0254, 0x301c},
    {0x40, 0xff41, 0xff3e}, {0x41, 0xff49, 0xff3e},
    {0x42, 0xff55, 0xff3e}, {0x43, 0xff45, 0xff3e},
    {0x44, 0xff45, 0xff40}, {0x45, 0xff45, 0xff07},
    {0x47, 0xff41, 0x00a8}, {0x48, 0xff55, 0x00a8},
    {0x49, 0xff4f, 0x00a8}, {0x83, 0xff0b, 0xff0d},
    /* square */
    {0x97, 0x25a1, 0x25a0}, {0x98, 0x25a1, 0x25a1},
    {0x99, 0x25a1, 0x00d7}, {0xb9, 0x25a1, 0xff28},
    /* circle */
    {0xa4, 0x25cb, 0x30fb}, {0xa6, 0x25cb, 0x25a1},
    {0xa7, 0x25cb, 0x25a0}, {0xa8, 0x25cb, 0xff5c},
    {0xa9, 0x25cb, 0x30b5}, {0xaa, 0x25cb, 0xff0b},
    {0xab, 0x25cb, 0xff0d}, {0xac, 0x25cb, 0xff0b},
    {0xad, 0x25cb, 0xff0d}, {0xae, 0x25cb, 0x00d7},
    {0xaf, 0x25cb, 0xff0f}, {0xb4, 0x25cb, 0x3012},
    {0xb8, 0x25cb, 0x00d7},
    /* other */
    {0x61, 0x2227, 0x2227},
    {0x95, 0x25cf, 0x25cf}, {0x96, 0x25a0, 0x25a0},
    {0x9b, 0xff0b, 0x00d7},
  };
  const double mixpar[][9] = {
    /* left-right */
    {0x66, 1.5, 0, 1, 0,  3, 2, 1, 0}, {0x67, 1.5, 0, 1, 0,  3, 2, 1, 0},
    {0x68, 1.5, 0, 1, 0,  3, 2, 1, 0}, {0x69, 1.5, 0, 1, 0,  3, 2, 1, 0},
    {0x6a, 1.5, 0, 1, 0,  3, 2, 1, 0}, {0x6b, 1.5, 0, 1, 0,  3, 2, 1, 0},
    {0x6c, 1.5, 0, 1, 0,  3, 2, 1, 0}, {0x6d, 1.5, 0, 1, 0,  3, 2, 1, 0},
    {0x78, 1.5, 0, 1, 0,  3, 2, 1, 0}, {0x77, 1.5, 0, 1, 0,  3, 2, 1, 0},
    {0x3a, 2, 0, 1, 0,  2, 1, 1, 0}, {0x3b, 2, 0, 1, 0,  2, 1, 1, 0},
    {0x3c, 2, 0, 1, 0,  2, 1, 1, 0}, {0x46, 2, 0, 1, 0,  2, 0.75, 1, 0},
    {0x63, 2, 0, 1, 0,  2, 0.75, 1, 0},
    /* down-up */
    {0x7c, 1, 0, 1, 0,  1, 0, 2, 0}, {0x7d, 1, 0, 1, 0,  1, 0, 2, 0},
    {0x7e, 1, 0, 1, 0,  1, 0, 2, 0}, {0x7f, 1, 0, 1, 0,  1, 0, 2, 0},
    {0x40, 1, 0, 1, 0,  1, 0, 2, 0}, {0x41, 1, 0, 1, 0,  1, 0, 2, 0},
    {0x42, 1, 0, 1, 0,  1, 0, 2, 0}, {0x43, 1, 0, 1, 0,  1, 0, 2, 0},
    {0x44, 1, 0, 1, 0,  1, 0, 2, 0}, {0x45, 1, 0, 1, 0,  1, 0, 2, 0},
    {0x47, 1, 0, 1, 0,  1, 0, 2, 0}, {0x48, 1, 0, 1, 0,  1, 0, 2, 0},
    {0x49, 1, 0, 1, 0,  1, 0, 2, 0},
    {0x83, 1, 0, 1.5, -0.4,  1, 0, 2, 0},
    /* square */
    {0x97, 1, 0, 1, 0,  2, 0.5, 2, -0.5}, {0x98, 1, 0, 1, 0,  2, 0.5, 2, -0.5},
    {0x99, 1, 0, 1, 0,  1, 0, 1, 0}, {0xb9, 1, 0, 1, 0,  2, 0.5, 2, -0.5},
    /* circle */
    {0xa4, 1, 0, 1, 0,  3, 1, 3, -1}, {0xa6, 1, 0, 1, 0,  3, 1, 3, -1},
    {0xa7, 1, 0, 1, 0,  3, 1, 3, -1}, {0xa8, 1, 0, 1, 0,  4, 1.5, 1.5, -0.25},
    {0xa9, 1, 0, 1, 0,  1.6, 0.3, 2, -0.5}, {0xaa, 1, 0, 1, 0,  3, 1, 3, -1},
    {0xab, 1, 0, 1, 0,  3, 1, 3, -1},
    {0xac, 1, 0, 1, 0,  1.5, 0.25, 1.5, -0.25},
    {0xad, 1, 0, 1, 0,  1.5, 0.25, 1.5, -0.25},
    {0xae, 1, 0, 1, 0,  2, 0.5, 1.5, -0.25},
    {0xaf, 1, 0, 1, 0,  3, 1, 3, -1},
    {0xb4, 1, 0, 1, 0,  2, 0.5, 1.5, -0.25},
    {0xb8, 1, 0, 1, 0,  2, 0.5, 1.5, -0.25},
    /* other */
    {0x61, 1.33, 0.167, 1.33, -0.167,  1, 0, 1, 0},
    {0x95, 2, 0.5, 2, -0.5,  2, 0.5, 2, -0.5},
    {0x96, 2, 0.5, 2, -0.5,  2, 0.5, 2, -0.5},
    {0x9b, 1, 0, 1, 0,  1, 0, 1, 0},
  };

  const int *tbl[] = {jistbl, unitbl};
  const int (*(alttbl[]))[2] = {jisalttbl, unialttbl};
  const int (*(mixtbl[]))[3] = {jismixtbl, unimixtbl};

  /* options */
  for (opt = 1; argc > opt && argv[opt][0] == '-'; opt++) {
    if (strcmp(argv[opt], "-ascii") == 0) {
      ascii = 1;
    } else if (strcmp(argv[opt], "-ascent") == 0) {
      use_ascent = 1;
    } else {
      printf("usage: bdf2cgrom66 [-ascii] [-ascent] fontfile1.bdf [fontfile2.bdf]\n");
      exit(1);
    }
  }

  if (argc < opt + 1) {
      printf("usage: bdf2cgrom66 [-ascii] [-ascent] fontfile1.bdf [fontfile2.bdf]\n");
    exit(1);
  }

  if (strcmp(argv[opt], "-") == 0) {
    fp1 = stdin;
  } else {
    fp1 = fopen(argv[opt], "rb");
    if (fp1 == NULL) {
      printf("cannot open %s\n", argv[opt]);
      exit(1);
    }
  }

  if (argc < opt + 2) {
    fp2 = fp1;
  } else if (strcmp(argv[opt + 1], "-") == 0) {
    fp2 = stdin;
  } else {
    fp2 = fopen(argv[opt + 1], "rb");
    if (fp2 == NULL) {
      printf("cannot open %s\n", argv[opt + 1]);
      exit(1);
    }
  }

  /* CGROM60.66 */
  charset = getbdfpar(fp1, ascii, use_ascent, &pixel_x, &pixel_y, &over_baseline);
  while (fgets(line, sizeof(line), fp1)) {
    if (sscanf(line, "ENCODING %d", &code) != 1 || code == 0) continue;
    fpos = ftell(fp1);
    code_is_ambiguous = isambiguous(code, charset);
    for (i = 0; i < 0x100; i++) {
      if ((ascii && code == i) || (!ascii && code == tbl[charset][i])) {
	convert(fp1, bmp60[i], 8, 12, pixel_x, pixel_y, over_baseline, code_is_ambiguous, 1, 0, 1, 0);
	fseek(fp1, fpos, SEEK_SET);
      }
    }

    /* alternative */
    for (i = 0; i < sizeof(jisalttbl) / sizeof(jisalttbl[0][0]) / 2; i++) {
      if (alttbl[charset][i][0] < 0x100 && code == alttbl[charset][i][1]) {
	chr = alttbl[charset][i][0];
	if (isnodata(bmp60[chr])) {
	  convert(fp1, bmp60[chr], 8, 12, pixel_x, pixel_y, over_baseline, code_is_ambiguous, 1, 0, 1, 0);
	  fseek(fp1, fpos, SEEK_SET);
	}
      }
    }
  }
  if (fp1 == fp2) {
    rewind(fp2);
  } else {
    fclose(fp1);
  }

  /* box drawings */
  calcline(bmp60, 12);

  /* 2*3 block */
  for (i = 0; i < 0x40; i++) {
    for (j = 0; j < 12; j++) {
      bmp60[0x100 + i][j] =
	((i & (0x20 >> (j / 4 * 2))) ? 0xf0 : 0) |
	((i & (0x10 >> (j / 4 * 2))) ? 0x0f : 0);
    }
  }

  /* output */
  strncpy(outfile, "CGROM60.66", sizeof(outfile));
  fpout = fopen(outfile, "wb");
  if (fpout == NULL) {
    printf("cannot open %s\n", outfile);
    exit(1);
  }
  for (i = 0; i < 0x200; i++) {
    for (j = 0; j < 0x10; j++) {
      fputc(bmp60[i][j], fpout);
    }
  }
  fclose(fpout);


  /* CGROM66.66 */
  memset(bmp60, 0, sizeof(bmp60));
  charset = getbdfpar(fp2, ascii, use_ascent, &pixel_x, &pixel_y, &over_baseline);
  while (fgets(line, sizeof(line), fp2)) {
    if (sscanf(line, "ENCODING %d", &code) != 1 || code == 0) continue;
    fpos = ftell(fp2);
    code_is_ambiguous = isambiguous(code, charset);
    for (i = 0; i < 0x200; i++) {
      if ((ascii && code == i) || (!ascii && code == tbl[charset][i])) {
	convert(fp2, bmp60[i], 8, 10, pixel_x, pixel_y, over_baseline, code_is_ambiguous, 1, 0, 1, 0);
	fseek(fp2, fpos, SEEK_SET);
      }
    }

    /* alternative */
    for (i = 0; i < sizeof(jisalttbl) / sizeof(jisalttbl[0][0]) / 2; i++) {
      if (code == alttbl[charset][i][1]) {
	chr = alttbl[charset][i][0];
	if (isnodata(bmp60[chr])) {
	  convert(fp2, bmp60[chr], 8, 10, pixel_x, pixel_y, over_baseline, code_is_ambiguous, 1, 0, 1, 0);
	  fseek(fp2, fpos, SEEK_SET);
	}
      }
    }

    /* mixed character */
    for (i = 0; i < sizeof(jismixtbl) / sizeof(jismixtbl[0][0]) / 3; i++) {
      if ((!mixdone1[i] && code == mixtbl[charset][i][1]) ||
          (!mixdone2[i] && code == mixtbl[charset][i][2])) {
	chr = mixtbl[charset][i][0];
	for (j = 0; j < sizeof(mixpar) / sizeof(mixpar[0][0]) / 9; j++) {
	  if (mixpar[j][0] == chr) {
	    break;
	  }
	}
	if (j == sizeof(mixpar) / sizeof(mixpar[0][0]) / 9) continue;
	if (code == mixtbl[charset][i][1]) {
	  convert(fp2, bmptmp, 8, 10, pixel_x, pixel_y, over_baseline, code_is_ambiguous,
		  mixpar[j][1], mixpar[j][2], mixpar[j][3], mixpar[j][4]);
	  fseek(fp2, fpos, SEEK_SET);
	  mixdone1[i] = 1;
	  for (k = 0; k < 10; k++) {
	    bmpmix[i][k] |= bmptmp[k];
	  }
	}
	if (code == mixtbl[charset][i][2]) {
	  convert(fp2, bmptmp, 8, 10, pixel_x, pixel_y, over_baseline, code_is_ambiguous,
		  mixpar[j][5], mixpar[j][6], mixpar[j][7], mixpar[j][8]);
	  fseek(fp2, fpos, SEEK_SET);
	  mixdone2[i] = 1;
	  for (j = 0; j < 10; j++) {
	    bmpmix[i][j] |= bmptmp[j];
	  }
	}
      }
    }
  }
  fclose(fp2);

  /* box drawings */
  calcline(bmp60, 10);

  /* 2*2 block */
  for (i = 0; i < 0x10; i++) {
    for (j = 0; j < 10; j++) {
      bmp60[0x100 + i][j] =
	((i & (0x08 >> (j / 5 * 2))) ? 0xf0 : 0) |
	((i & (0x04 >> (j / 5 * 2))) ? 0x0f : 0);
    }
  }

  /* 1/10 block */
  for (i = 0x10; i < 0x19; i++) {
    for (j = 0; j < 10; j++) {
      bmp60[0x100 + i][j] = (j >= 0x19 - i) ? 0xff : 0;
    }
  }

  /* 1/8 block */
  for (i = 0x19; i < 0x20; i++) {
    for (j = 0; j < 10; j++) {
      bmp60[0x100 + i][j] = (0xfe << (i - 0x19)) & 0xff;
    }
  }

  /* triangle */
  for (j = 0; j < 10; j++) {
    bmp60[0x120][j] = (0x02 << (j * 8 / 10)) - 0x01;
    bmp60[0x121][j] = 0x100 - (0x80 >> (j * 8 / 10));
    bmp60[0x122][j] = (0x100 >> (j * 8 / 10)) - 0x01;
    bmp60[0x123][j] = 0x100 - (0x01 << (j * 8 / 10));
  }

  /* mesh */
  for (j = 0; j < 10; j++) {
    bmp60[0x124][j] = 0xcc >> (j & 2);
    bmp60[0x125][j] = 0xaa >> (j & 1);
  }

  /* mixed character */
  for (i = 0; i < sizeof(jismixtbl) / sizeof(jismixtbl[0][0]) / 3; i++) {
    chr = 0x100 + mixtbl[charset][i][0];
    if (isnodata(bmp60[chr]) && mixdone1[i] && mixdone2[i]) {
      memcpy(bmp60[chr], bmpmix[i], 10);
    }
  }

  /* no data */
  for (i = 0x101; i < 0x200; i++) {
    if (isnodata(bmp60[i])) {
      for (j = 0; j < 10 - 1; j++) {
	bmp60[i][j] = 0xfe;
      }
      bmp60[i][10 / 2 - 1] = 0;
    }
  }

  /* output */
  strncpy(outfile, "CGROM66.66", sizeof(outfile));
  fpout = fopen(outfile, "wb");
  if (fpout == NULL) {
    printf("cannot open %s\n", outfile);
    exit(1);
  }
  for (i = 0; i < 0x200; i++) {
    for (j = 0; j < 0x10; j++) {
      fputc(bmp60[i][j], fpout);
    }
  }
  fclose(fpout);

  return 0;
}


int getbdfpar(FILE *fp, int ascii, int use_ascent, int *pixel_x, int *pixel_y, int *over_baseline)
{
  int code = 0;
  int xmin = 999, xmax = -1;
  int ymin = 999, ymax = -1;
  int ascent, descent;
  int size_x, size_y;
  int offset_x = 0, offset_y = 0;
  int result = JIS;
  char line[100] = "";
  char charset[100] = "";

  ascent = -1;
  descent = -1;

  while (fgets(line, sizeof(line), fp)) {
    if (sscanf(line, "FONTBOUNDINGBOX %d %d %d %d", pixel_x, pixel_y, &offset_x, &offset_y) == 4) {
      *over_baseline = *pixel_y + offset_y;
    }
    sscanf(line, "FONT_ASCENT %d", &ascent);
    sscanf(line, "FONT_DESCENT %d", &descent);
    if (sscanf(line, "CHARSET_REGISTRY %s", charset) == 1) {
      if (strncasecmp(charset, "\"JISX0208", 9) == 0 ||
	  strncasecmp(charset, "\"JISX0213", 9) == 0) {
	result = JIS;
      } else if (strcasecmp(charset, "\"ISO10646\"") == 0) {
	result = UNI;
      } else if (!ascii) {
	printf("cannot convert for charset %s\n", charset);
	exit(1);
      }
    }
    sscanf(line, "ENCODING %d", &code);
    if (sscanf(line, "BBX %d %d %d %d", &size_x, &size_y, &offset_x, &offset_y) == 4) {
      if (result != UNI ||
	  (0x0000 <= code && code <= 0x03ff) ||
	  (0x2000 <= code && code <= 0x2e39) ||  /* 2e3a = 2-em dash */
	  (0x2e3c <= code && code <= 0x9fff) ||  /* 2e3b = 3-em dash */
	  (0xf900 <= code && code <= 0xfaff) ||
	  (0xff00 <= code && code <= 0xffff)
	  ) {
	if (offset_x < xmin) {
	  xmin = offset_x;
	}
	if (offset_x + size_x > xmax) {
	  xmax = offset_x + size_x;
	}
	if (offset_y < ymin) {
	  ymin = offset_y;
	}
	if (offset_y + size_y > ymax) {
	  ymax = offset_y + size_y;
	}
      }
    }
  }

  *pixel_x = xmax - xmin;
  *pixel_y = ymax - ymin;
  if (use_ascent) {
    *pixel_y = ascent + descent;
    *over_baseline = ascent;
  }
  if (*pixel_x < 0 || *pixel_y < 0 || *over_baseline < 0 || strlen(charset) == 0) {
    printf("illegal bdf\n");
    exit(1);
  }
  rewind(fp);

  return result;
}


void convert(FILE *fp, unsigned char bmp60[16], int size_x_out, int size_y_out, int pixel_x, int pixel_y, int over_baseline, int code_is_ambiguous, double par1, double par2, double par3, double par4)
{
  int i, j;
  int x, y;
  int size_x, size_y;
  int offset_x, offset_y;
  unsigned long long data;
  double ratio_x, ratio_y;
  char line[100];
  unsigned char bmpbdf[BDFSIZE][BDFSIZE];

  memset(bmpbdf, 0, sizeof(bmpbdf));
  do {
    if (fgets(line, sizeof(line), fp) == NULL) break;
  } while (sscanf(line, "BBX %d %d %d %d", &size_x, &size_y, &offset_x, &offset_y) != 4);

  if (code_is_ambiguous && size_x + offset_x <= pixel_x / 2) {
    pixel_x /= 2;
  }

  if (pixel_x > size_x_out) {
    ratio_x = (double)pixel_x / size_x_out;
  } else {
    ratio_x = 1;
    offset_x += (size_x_out - pixel_x) / 2;
  }
  if (pixel_y > size_y_out) {
    ratio_y = (double)pixel_y / size_y_out;
  } else {
    ratio_y = 1;
    offset_y -= (size_y_out - pixel_y) / 2;
  }

  if (offset_x < 0) {
    offset_x = 0;
  }

  ratio_x *= par1;
  offset_x += pixel_x * par2;
  ratio_y *= par3;
  offset_y += pixel_y * par4;

  do {
    if (fgets(line, sizeof(line), fp) == NULL) break;
  } while (strncmp(line, "BITMAP", 6) != 0);

  for (i = 0; i < size_y; i++) {
    if (fgets(line, sizeof(line), fp) == NULL) break;
    while (sscanf(line, "%llx", &data) != 1) {
      ;
    }
    for (j = 0; j < size_x; j++) {
      x = j + offset_x;
      y = i + over_baseline - size_y - offset_y;
      if (0 <= x && x <= BDFSIZE && 0 <= y && y <= BDFSIZE) {
	bmpbdf[x][y] = (data >> ((BDFSIZE - size_x) % 8 + size_x - j - 1)) & 1;
      }
    }
  }

  if (size_y > 0) {
    for (i = 0; i < size_y_out; i++) {
      bmp60[i] = 0;
      for (j = 0; j < size_x_out; j++) {
	data = 0;
	for (x = j * ratio_x; x < (int)((j + 1) * ratio_x + .01) && x < BDFSIZE; x++) {
	  for (y = i * ratio_y; y < (int)((i + 1) * ratio_y + .01) && y < BDFSIZE; y++) {
	    data += bmpbdf[x][y];
	  }
	}
	if ((double)data / (x - (int)(j * ratio_x)) * (y - (int)(i * ratio_y)) > 0) {
	  bmp60[i] |= 1 << (size_x_out - j - 1);
	}
      }
    }
  }
}


int isnodata(unsigned char *bmp60)
{
  int i;
  for (i = 0; i < 16; i++) {
    if (bmp60[i]) return 0;
  }
  return 1;
}


int isambiguous(int code, int charset)
{
  int i;
  const int eawtbl[] = {
    0x00a7, 0x00b1, 0x00d7, 0x00e2, 0x00e3, 0x00e4, 0x00e6, 0x00e7,
    0x00e8, 0x00e9, 0x00ea, 0x00ee, 0x00f0, 0x00f6, 0x00f7, 0x00fb,
    0x00fc,
    0x014b, 0x0153, 0x0251, 0x0254, 0x0259, 0x025a, 0x0265, 0x026a,
    0x026f, 0x0272, 0x0281, 0x0283, 0x028a, 0x028c, 0x028f, 0x0292,
    0x0392, 0x0394, 0x039b, 0x03a3, 0x03a6, 0x03a9, 0x03b1, 0x03b2,
    0x03b3, 0x03b4, 0x03b5, 0x03b8, 0x03bb, 0x03bc, 0x03bd, 0x03c0,
    0x03c1, 0x03c3, 0x03c5, 0x03c6, 0x03c9, 0x03d5, 0x0411,
    0x2016, 0x2019, 0x201d, 0x203b, 0x203e, 0x2190, 0x2191, 0x2192,
    0x2193, 0x2196, 0x2197, 0x2198, 0x2199, 0x21ba, 0x21bb, 0x2202,
    0x2212, 0x2213, 0x221a, 0x221d, 0x221e, 0x2220, 0x2227, 0x2228,
    0x2229, 0x222a, 0x222b, 0x2234, 0x2235, 0x2252, 0x2260, 0x2264,
    0x2265, 0x2266, 0x2267, 0x2295, 0x2296, 0x2297, 0x2298, 0x2299,
    0x229d, 0x22bf, 0x2312, 0x2491, 0x2492, 0x2493, 0x2501, 0x2503,
    0x250f, 0x2513, 0x2517, 0x251b, 0x2523, 0x252b, 0x2533, 0x253b,
    0x254b, 0x25a0, 0x25a1, 0x25a3, 0x25aa, 0x25b2, 0x25b3, 0x25bc,
    0x25c6, 0x25c7, 0x25cb, 0x25ce, 0x25cf, 0x2605, 0x260e, 0x2640,
    0x2642, 0x2660, 0x2663, 0x2665, 0x2666, 0x26e3, 0x26e8, 0x26e9,
    0x26eb, 0x26fb, 0x2733, 0x27f2, 0x27f3, 0x29b6, 0x2b59,
    0x1d10b, 0x1d10c, 0x1f6e7,
  };

  if (charset == JIS) return 0;

  for (i = 0; i < sizeof(eawtbl) / sizeof(eawtbl[0]); i++) {
    if (code == eawtbl[i]) {
      return 1;
    }
  }
  return 0;
}


void calcline(unsigned char bmp60[][16], int size_y_out)
{
  int i, j;
  int chr;
  const unsigned char dots2 = 0x03;
  const unsigned char dots4 = 0x0f;
  const int line_shape[][4] = {
    /* left, right, up, down */
    {1, 1, 1, 0}, {1, 1, 0, 1}, {1, 0, 1, 1}, /* 11-13 */
    {0, 1, 1, 1}, {1, 1, 1, 1}, {0, 0, 1, 1}, {1, 1, 0, 0}, /* 14-17 */
    {0, 1, 0, 1}, {1, 0, 0, 1}, {0, 1, 1, 0}, {1, 0, 1, 0}, /* 18-1b */
  };

  /* box drawings */
  for (i = 0; i < sizeof(line_shape) / sizeof(line_shape[0][0]) / 4; i++) {
    chr = 0x11 + i;
    for (j = 0; j < 16; j++) {
      bmp60[chr][j] = 0;
    }
    bmp60[chr][size_y_out / 2 - 1] = bmp60[chr][size_y_out / 2] =
      (line_shape[i][0] ? (dots4 << 4) : 0) |
      (line_shape[i][1] ? dots4 : 0);
    for (j = 0; j < size_y_out / 2 + 1; j++) {
      bmp60[chr][j] |= line_shape[i][2] ? (dots2 << 3) : 0;
    }
    for (j = size_y_out / 2 - 1; j < size_y_out; j++) {
      bmp60[chr][j] |= line_shape[i][3] ? (dots2 << 3) : 0;
    }
  }
}
