/* hlite.c, generic syntax highlighting, Ait Emacs, Kevin Bloom, BSD 3-Clause, 2023-2024 */

#include "header.h"
#include "util.h"

int state = ID_DEFAULT;
int next_state = ID_DEFAULT;
int skip_count = 0;
int exclude_state = ID_DEFAULT;
int exclude_count = 0;

char_t get_at(buffer_t *bp, point_t pt)
{
  return (*ptr(bp, pt));
}

void set_parse_state(buffer_t *bp, point_t pt, window_t *wp, int loop)
{
  register point_t po;

  state = ID_DEFAULT;
  next_state = ID_DEFAULT;
  skip_count = 0;

  if(bp->b_keywords != NULL &&
     bp->b_keywords->mlc != NULL && bp->b_keywords->emlc != NULL &&
     loop) {
    for (po =0; po < pt; po++)
        parse_text(bp, po);
    wp->w_hilite = state;
  }
}

void write_parse_state(window_t *wp)
{
  state = wp->w_hilite;
  next_state = wp->w_hilite;
  skip_count = 0;
}

int parse_text(buffer_t *bp, point_t pt)
{
//  if(bp->b_keywords == NULL)
//    return state;

  if (skip_count-- > 0) {
    if(exclude_count != 0)
      exclude_count--;
    if(exclude_state != ID_DEFAULT &&
       exclude_count == 0) {
       state = exclude_state;
       exclude_state = ID_DEFAULT;
    }
    return state;
  }

  char_t c_now = get_at(bp, pt);
  char_t c_prev = get_at(bp, pt-1);
  char_t next = c_now;
  int valid = TRUE, k = 0;
  state = next_state;

  if (state == ID_DEFAULT &&
      bp->b_keywords != NULL &&
      bp->b_keywords->mlc != NULL) {
    next = c_now;
    for(int i = 0; bp->b_keywords->mlc[i] != '\0'; i++) {
      next = get_at(bp, pt + i);
      if(next != bp->b_keywords->mlc[i]) {
        valid = FALSE;
        break;
      }
    }
    if(valid) {
      skip_count = 1;
      return (next_state = state = ID_BLOCK_COMMENT);
    }
    valid = TRUE;
  }

  if (state == ID_BLOCK_COMMENT &&
      bp->b_keywords != NULL &&
      bp->b_keywords->emlc != NULL) {
    next = c_now;
    for(int i = 0; bp->b_keywords->emlc[i] != '\0'; i++) {
      next = get_at(bp, pt + i);
      if(next != bp->b_keywords->emlc[i]) {
        valid = FALSE;
        break;
      }
    }
    if(valid) {
      skip_count = strlen(bp->b_keywords->emlc) - 1;
      next_state = ID_DEFAULT;
      return ID_BLOCK_COMMENT;
    }
    valid = TRUE;
  }

  if (state == ID_DEFAULT &&
      bp->b_keywords != NULL &&
      bp->b_keywords->slc[0] != '\0') {
    next = c_now;
    for(int i = 0; bp->b_keywords->slc[i] != '\0'; i++) {
      next = get_at(bp, pt + i);
      if(next != bp->b_keywords->slc[i]) {
        valid = FALSE;
        break;
      }
    }
    if(valid) {
      skip_count = 1;
      return (next_state = state = ID_LINE_COMMENT);
    }
    valid = TRUE;
  }

  if (state == ID_LINE_COMMENT && c_now == '\n')
    return (next_state = ID_DEFAULT);

  if (state == ID_DEFAULT && c_now == '"') {
    int enable = FALSE;
    char_t z = get_at(bp, pt+1);
    point_t end = pos(bp, bp->b_ebuf);
    for(point_t i = pt+1; z != '\n' && i <= end; i++, z = get_at(bp, i)) {
      if(z == '"') {
        enable = TRUE;
        break;
      }
      if(z == '\\' && get_at(bp, i+1) == '\n') {
        enable = TRUE;
        break;
      }
    }
    if(enable)
      return (next_state = ID_DOUBLE_STRING);
  }

  if (state == ID_DEFAULT &&
      bp->b_keywords != NULL &&
      bp->b_keywords->bqas &&
      c_now == '`')
    return (next_state = ID_BACK_STRING);

  if (state == ID_DEFAULT &&
      bp->b_keywords != NULL &&
      bp->b_keywords->sqas &&
      c_now == '\'') {
      int enable = FALSE;
    char_t z = get_at(bp, pt+1);
    point_t end = pos(bp, bp->b_ebuf);
    for(point_t i = pt+1; z != '\n' && i <= end; i++, z = get_at(bp, i)) {
      if(z == '\'') {
        enable = TRUE;
        break;
      }
    }
    if(enable)
      return (next_state = ID_SINGLE_STRING);
  }

  if (state == ID_DOUBLE_STRING && c_now == '\\') {
    skip_count = 1;
    return (next_state = ID_DOUBLE_STRING);
  }

  if (state == ID_DOUBLE_STRING && c_now == '"') {
    next_state = ID_DEFAULT;
    return ID_DOUBLE_STRING;
  }


  if (state == ID_SINGLE_STRING && c_now == '\\') {
    skip_count = 1;
    return (next_state = ID_SINGLE_STRING);
  }

  if (state == ID_DEFAULT &&
      bp->b_keywords != NULL &&
      bp->b_keywords->bqas &&
      c_now == '`')
    return (next_state = ID_BACK_STRING);

  if (state == ID_BACK_STRING && c_now == '\\') {
    skip_count = 1;
    return (next_state = ID_BACK_STRING);
  }


  if (state == ID_SINGLE_STRING && c_now == '\'') {
    next_state = ID_DEFAULT;
    return ID_SINGLE_STRING;
  }

  if (state == ID_BACK_STRING && c_now == '`') {
    next_state = ID_DEFAULT;
    return ID_BACK_STRING;
  }

  if(bp->b_keywords != NULL &&
     bp->b_keywords->keywords != NULL &&
     state == ID_DEFAULT &&
     (pt == 0 ||
       (is_symbol(c_prev) &&
         (c_prev != '-' && c_prev != '_'))
       || isspace(c_prev))) {
     for(int i = 0; bp->b_keywords->keywords[i].word != NULL; i++) {
       int l = 0, t = 0;
       exclude_count = 0;
       exclude_state = ID_DEFAULT;
       for(k = 0; bp->b_keywords->keywords[i].word[l] != '\0'; k++, l++) {
         c_now = get_at(bp, pt+k);
         /* at the end */
         if(bp->b_keywords->keywords[i].word[l] == '') {
          if(bp->b_keywords->keywords[i].word[l+1] == '\0') {
            for(; !isspace(c_now); k++) {
              c_now = get_at(bp, pt+k);
            }
            k--;
            break;
          } else {
            l++;
            for(; !isspace(c_now) && (bp->b_keywords->keywords[i].word[l] < 32 ||
              !is_symboli(c_now, bp->b_keywords->keywords[i].word[l])); k++) {
              if(bp->b_keywords->keywords[i].word[l] == c_now) {
                t = 1;
                break;
              }
              if(bp->b_keywords->keywords[i].word[l] == '' &&
                 bp->b_keywords->keywords[i].word[l+1] == c_now) {
                t = 2;
                break;
              }
              if(pt+k == pos(bp, bp->b_ebuf))
                break;
              c_now = get_at(bp, pt+k);
            }
            if(t == 0) {
              k = 0;
              break;
            }
            if(t == 1) {
              k--;
              continue;
            }
            if(t == 2) {
              l++;
              k--;
              break;
            }
          }
         }
         if(bp->b_keywords->keywords[i].word[l] == '') {
           if(bp->b_keywords->keywords[i].word[l+1] == c_now
              || bp->b_keywords->keywords[i].word[l+1] == '\0') {
             exclude_state = bp->b_keywords->keywords[i].color;
             exclude_count++;
             k--;
             continue;
           } else {
             k = 0;
             exclude_count = 0;
             break;
           }
         }
         if(bp->b_keywords->keywords[i].word[l] != c_now) {
           k = 0;
           break;
         }
       }
      c_now = get_at(bp, pt+k);
      if(k > 0 && (isspace(c_now) ||
         (is_symbol(c_now) && (c_now != '-' && (
           bp->b_keywords->keywords[i].word[l] == '_' || c_now != '_')))) &&
           (bp->b_keywords->keywords[i].word[l] == '\0' ||
            bp->b_keywords->keywords[i].word[l+1] == '\0')) {
         skip_count = k-1;
         next_state = ID_DEFAULT;
         if(exclude_state != ID_DEFAULT) {
           return (state = ID_DEFAULT);
         }
         return (state = bp->b_keywords->keywords[i].color);
      }
    }
  }

  if (state != ID_DEFAULT)
    return (next_state = state);

//  if (state == ID_DEFAULT && c_now >= '0' && c_now <= '9') {
//    next_state = ID_DEFAULT;
//    return (state = ID_DIGITS);
//  }

//  if (state == ID_DEFAULT && 1 == is_symbol(c_now)) {
//    next_state = ID_DEFAULT;
//    return (state = ID_SYMBOL);
//  }

  return (next_state = state);
}
