#include "slider.h"

#include "parser.h"
#include "script.h"

#define  PARSER_SLIDER		(1 << 8)

#define  NSLIDERS_ALLOC		20

#define  SLIDER_MINIMUM		1
#define  SLIDER_VALUE		2
#define  SLIDER_MAXIMUM		3
#define  SLIDER_LABEL		4

#define  SLIDER_ARGS		5

typedef struct
{
    int command;
    Parser_store *store[SLIDER_ARGS];
}   Slider;

static int nsliders = 0;
static int nsliders_alloc = 0;
static Slider *sliders;

static void free_slider_memory()
{
    if (nsliders_alloc > 0)
    {
	FREE(sliders, Slider);
	nsliders = nsliders_alloc = 0;
    }
}

static Status alloc_slider_memory(String error_msg)
{
    int n;

    if (nsliders_alloc == 0)
    {
	sprintf(error_msg, "allocating slider memory");

	n = NSLIDERS_ALLOC;
	MALLOC(sliders, Slider, n);
	nsliders_alloc = n;
    }
    else
    {
	sprintf(error_msg, "reallocating slider memory");

	n = nsliders_alloc + NSLIDERS_ALLOC;
	REALLOC(sliders, Slider, n);
	nsliders_alloc = n;
    }

    return  OK;
}

static void find_slider(int command, Generic_ptr data)
{
    int i;

    for (i = 0; i < nsliders; i++)
	if (((sliders[i].store[OUTPUT_X])->data == data) &&
						(sliders[i].command == -1))
	    sliders[i].command = command;
}

void check_for_slider(int command, int nstore, Parser_store **store)
{
    int i;

    for (i = 1; i < nstore; i++)
	if (store[i] && (store[i]->data_type & PARSER_SLIDER))
	    find_slider(command, store[i]->data);
}

static Status check_slider_data(Parser_store **store, String error_msg)
{
    float min, value, max;

    value = *((float *) (store[SLIDER_VALUE]->data));
    min = *((float *) (store[SLIDER_MINIMUM]->data));
    max = *((float *) (store[SLIDER_MAXIMUM]->data));

    if ((value < min) || (value > max))
    {
	sprintf(error_msg,
		"cannot have min = %5.2f, value = %5.2f, max = %5.2f",
							min, value, max);
	return  ERROR;
    }

    return  OK;
}

int change_slider(int slider, float value)
{
    float *p;
    Slider *s;

    s = sliders + slider;
    p = (float *) (s->store[OUTPUT_X]->data);
    *p = value;
    store_type_float(s->store[OUTPUT_X]);

    return  (s->command);
}

static Status do_slider(Bool first_run, int nstore, Parser_store **store,
							String error_msg)
{
    float x, *p;

    if (!first_run)
	return  OK;

    store_int_to_float(store[SLIDER_VALUE]);
    x = *((float *) (store[SLIDER_VALUE]->data));

    store_type_float(store[OUTPUT_X]);
    CHECK_STATUS(check_parser_alloc(store[OUTPUT_X], 1, error_msg));
    p = (float *) (store[OUTPUT_X]->data);

    CHECK_STATUS(check_slider_data(store, error_msg));

    *p = x;

    return  OK;
}

Status init_slider(int nstore, Parser_store **store, String error_msg)
{
    if (nsliders == nsliders_alloc)
	CHECK_STATUS(alloc_slider_memory(error_msg));

    if (setup_command(nstore, store, "slider", do_slider, error_msg) == ERROR)
	return  ERROR;

    store[OUTPUT_X]->data_type = PARSER_FRS | PARSER_SLIDER;

    sliders[nsliders].command = -1;
    COPY_VECTOR(sliders[nsliders].store, store, nstore);

    nsliders++;

    return  OK;
}

int number_sliders()
{
    return  nsliders;
}

String slider_label(int slider)
{
    return  (String) (sliders[slider].store[SLIDER_LABEL]->data);
}

void slider_numbers(int slider, Slider_info *info)
{
    info->minimum = *((float *) (sliders[slider].store[SLIDER_MINIMUM]->data));
    info->maximum = *((float *) (sliders[slider].store[SLIDER_MAXIMUM]->data));
    info->value = *((float *) (sliders[slider].store[SLIDER_VALUE]->data));
}
