#include "hpgl.h"

#include "date.h"
#include "output.h"
#include "style.h"

#define  HPGL_CM_SCALE  400 
#define  HPGL_PTS_SCALE  (HPGL_CM_SCALE / PTS_PER_CM) 

/*
#define  LABEL_TERMINATOR  3
*/
#define  LABEL_TERMINATOR  '^'

#define  N_PENS  8

/* approximate ratio of desired font size in x vs y direction */
#define  X_Y_FONT_RATIO		0.7

/* next two used to guess bounding box size for text */
/* fine-tuned to fit Microsoft Word (need I say more) */
#define  X_TEXT_SCALE		(1.0*X_Y_FONT_RATIO)
#define  Y_TEXT_SCALE		1.3

#define  DEFAULT_FONT_SIZE		(12 / PTS_PER_CM) /* 12 points in cm */

#define  X_BORDER_OFFSET_LEFT		500.0 /* HPGL units */
#define  X_BORDER_OFFSET_CENTER		900.0 /* HPGL units */
#define  X_BORDER_OFFSET_RIGHT		1300.0 /* HPGL units */
#define  Y_BORDER_OFFSET_TOP		600.0 /* HPGL units */
#define  Y_BORDER_OFFSET_BOTTOM		500.0 /* HPGL units */

/*
    hpgl_margins determined from hard clip limits defined by HP:
    Interfacing and Programming Manual, HP7550A Graphics Plotter
    See section on Hard-Clip Limits (in Graphics Limits) (p. 2-9).
    The plotter cannot plot outside the hard-clip limits.  It also
    seems that the hard-clip limit defines the origin of paper.

    As an example, the A4 paper is 210 x 297 mm, i.e. 8400 x 11880
    plotter units (40 per mm = 400 per cm).  The hard-clip size is
    7600 x 10870, hence the margin is 800 x 1010 in total, i.e.
    the paper is effectively this much smaller in size.  The
    conversion factor from inches to cms is assumed to be 2.54.

    The "other" figures are guesses, and should be safe.
    It is not known how "standard" are the numbers used below.
*/
    
static Paper_size hpgl_margins[] =
{
    { 800,	1010 },		/* A4 */
    { 1010,	830 },		/* A3 */
    { 796,	1006 },		/* A */
    { 1006,	822 },		/* B */
    { 1000,	1000 }		/* Other */
};

static FILE *file;
static int paper_width, paper_height;
static int plot_width, plot_height;
static int lower_x, lower_y;
static int upper_x, upper_y;
static int current_x, current_y;
static float ax, bx, ay, by;
static int color_type;
static Bool dashed_lines;

static float x_border_offset_left = X_BORDER_OFFSET_LEFT;
static float x_border_offset_center = X_BORDER_OFFSET_CENTER;
static float x_border_offset_right = X_BORDER_OFFSET_RIGHT;
static float y_border_offset_top = Y_BORDER_OFFSET_TOP;
static float y_border_offset_bottom = Y_BORDER_OFFSET_BOTTOM;

static float font_size = DEFAULT_FONT_SIZE;

static void set_paper_size(int size)
{
    float *s = get_paper_size(size);

    paper_width = HPGL_PTS_SCALE*s[0] - hpgl_margins[size][0];
    paper_height = HPGL_PTS_SCALE*s[1] - hpgl_margins[size][1];
}

static void set_orientation(int direction, int size)
{
    if (direction == LANDSCAPE)
	SWAP(paper_width, paper_height, float);
}

static void initialize_hpgl(int direction)
{
    /* initialize, pen 1 in holder */
    fprintf(file, "IN; SP 1;\n");

    /* define label terminator */
    fprintf(file, "DT%c;\n", LABEL_TERMINATOR);

    /* default HPGL paper direction seems to be LANDSCAPE */
    if (direction == PORTRAIT)
	fprintf(file, "RO 90;\n");  /* rotate 90 degrees */

    /* set P1 and P2 to default for current paper size */
    fprintf(file, "IP;\n");
}

static void set_plot_size(float *size)
{
    plot_width = HPGL_PTS_SCALE * size[0];
    plot_height = HPGL_PTS_SCALE * size[1];

    plot_width = MIN(plot_width, paper_width);
    plot_height = MIN(plot_height, paper_height);

    lower_x = (paper_width - plot_width) / 2;
    lower_y = (paper_height - plot_height) / 2;

    upper_x = lower_x + plot_width;
    upper_y = lower_y + plot_height;
}

static void do_border(Output_border *border)
{
    float x, y;

    if (border->use)
    {
	if (border->justification == JUSTIFICATION_RIGHT)
	{
	    x = paper_width - x_border_offset_right;
	}
	else if (border->justification == JUSTIFICATION_CENTER)
	{
	    x = paper_width / 2 - x_border_offset_center;
	}
	else /* (border->justification == JUSTIFICATION_LEFT) */
	{
	    x = x_border_offset_left;
	}

	if (border->edge == EDGE_BOTTOM)
	{
	    y = y_border_offset_bottom;
	}
	else /* (border->edge == EDGE_TOP) */
	{
	    y = paper_height - y_border_offset_top;
	}

	set_hpgl_font(border->font_name, border->font_size);

	fprintf(file, "PU %1.0f, %1.0f; PD;\n", x, y);

	fprintf(file, "SP %d;\n", 1);  /* pen 1 */

	fprintf(file, "LB%s%c\n", border->text, LABEL_TERMINATOR);
    }
}

static void do_outline_box()
{
    /* draw outline box */
    fprintf(file, "PU %d %d; PD; EA %d %d;\n",
					lower_x, lower_y, upper_x, upper_y);

    /* set clipping limits to this box */
/*
    fprintf(file, "IW %d %d %d %d;\n", lower_x, lower_y, upper_x, upper_y);
*/

    current_x = lower_x;  current_y = lower_y;
}

void start_hpgl(Generic_ptr data)
{
    Hpgl_data *hpgl_data = (Hpgl_data *) data;
    Output_choices *choices = hpgl_data->choices;

    file = hpgl_data->file;

    set_paper_size(choices->paper_size);
    set_orientation(choices->paper_dirn, choices->paper_size);

    initialize_hpgl(choices->paper_dirn);

    do_border(choices->title);
    do_border(choices->time_date);
    do_border(choices->file_name);

    color_type = choices->output_color;

    if (color_type == BLACK_WHITE)
	dashed_lines = TRUE;
    else
	dashed_lines = FALSE;

    set_plot_size(choices->plot_size);

    if (choices->outline_box)
	do_outline_box();
}

void end_hpgl()
{
    /* pen up, no pen in holder, initialize, page feed */
    fprintf(file, "PU; SP 0; IN; PG;\n");
}

void new_hpgl_range(float x0, float y0, float x1, float y1, Bool clip)
{
    ax = plot_width / (x1 - x0);  bx = - ax * x0 + lower_x;
    ay = plot_height / (y1 - y0);  by = - ay * y0 + lower_y;

    if (clip)
	fprintf(file, "IW %d %d %d %d;\n", lower_x, lower_y, upper_x, upper_y);
    else
	fprintf(file, "IW;\n");
}

#define  CONVERT_X(x)  (ax*(x) + bx)
#define  CONVERT_Y(y)  (ay*(y) + by)

void hpgl_line(float x0, float y0, float x1, float y1)
{
    int w0, w1, h0, h1;

    w0 = CONVERT_X(x0);
    w1 = CONVERT_X(x1);
    h0 = CONVERT_Y(y0);
    h1 = CONVERT_Y(y1);

    if ((current_x == w0) && (current_y == h0))  /* pen in correct place */
    	fprintf(file, "PD %d %d;\n", w1, h1);
    else
    	fprintf(file, "PU %d %d; PD %d %d;\n", w0, h0, w1, h1);

    current_x = w1;  current_y = h1;
}

/*
static void hpgl_left_text(String text, float x, float y)
{
    int w, h;

    w = CONVERT_X(x);
    h = CONVERT_Y(y);

    fprintf(file, "PU %d %d; PD; LB%s%c\n", w, h, text, LABEL_TERMINATOR);

    current_x = -1;  current_y = -1;
}
*/

void hpgl_text(String text, float x, float y, float a, float b)
{
/*    hpgl_left_text(text, x, y); */ /* does HPGL allow better? */
    int w, h;

    w = CONVERT_X(x);
    h = CONVERT_Y(y);

    w -= HPGL_CM_SCALE * X_TEXT_SCALE * a * font_size * strlen(text);
    h -= HPGL_CM_SCALE * Y_TEXT_SCALE * b * font_size;

    fprintf(file, "PU %d %d; PD; LB%s%c\n", w, h, text, LABEL_TERMINATOR);

    current_x = -1;  current_y = -1;
}

void set_hpgl_color(int color)
{
    /* select pen in valid range if color_type == COLOR_GRAY */
    if (color_type == COLOR_GRAY)
	fprintf(file, "SP %d;\n", 1 + (color - 1) % N_PENS);
}

void set_hpgl_font(String name, int size)
{
    /* character size in cm's */
    font_size = size / PTS_PER_CM;
    fprintf(file, "SI %5.2f %5.2f;\n", X_Y_FONT_RATIO*font_size, font_size);
}

void set_hpgl_line_style(int line_style)
{
    if (dashed_lines)
    {
	if (line_style == NORMAL_LINE_STYLE)
	    fprintf(file, "LT;\n");
	else /* line_style == DASHED_LINE_STYLE */
	    fprintf(file, "LT 2 1;\n");
    }
}
