#include "region_popup.h"

#include "region.hlp"

#include "callback.h"
#include "data.h"
#include "display_popup.h"
#include "fold_popup.h"
#include "info_popup.h"
#include "param.h"
#include "peaks_popup.h"
#include "position.h"
#include "ref.h"
#include "ref_popup.h"
#include "region.h"
#include "region_func.h"
#include "utility.h"
#include "window.h"

#define  VISIBLE_ITEMS  5

static Widget region_popup = (Widget) NULL;
static Widget region_form;

static Widget xregion_field;
static Widget yregion_field;

static Textfield_item region_items[] =
{
    { "x region: ", x_region, NO_CALLBACK, NO_DATA, &xregion_field },
    { "y region: ", y_region, NO_CALLBACK, NO_DATA, &yregion_field }
};

static int nregions = ARRAY_SIZE(region_items);

static Widget reference_boxes[NREF_TYPES];
static Radiobox_item reference_items[NREF_TYPES];

static int ref_type = REF_POINTS;
static int nref_types = ARRAY_SIZE(reference_boxes);

static Widget position_list;

static Widget name_field;

static Textfield_item saved_items[] =
{
    { "position name: ", position_name, NO_CALLBACK, NO_DATA, &name_field }
};

static int nsaved = ARRAY_SIZE(saved_items);

#define  NSTATS_BOXES  2

static Widget stats_boxes[NSTATS_BOXES];
static Radiobox_item stats_type_items[] =
{
    { "no", NO_CALLBACK, NO_DATA, stats_boxes + NO_STATS_FILE },
    { "yes", NO_CALLBACK, NO_DATA, stats_boxes + YES_STATS_FILE }
};

static int stats_type = NO_STATS_FILE;
static int nstats_types = ARRAY_SIZE(stats_boxes);

static Widget stats_field;

static Textfield_item stats_file_items[] =
{
    { "statistics file: ", statistics_file, NO_CALLBACK, NO_DATA, &stats_field }
};

static int nstats = ARRAY_SIZE(stats_file_items);

void set_region(int type)
{
    ref_type = type;

    if (region_popup)
    {
	set_radio_state(reference_boxes, nref_types, ref_type);
	set_textfields(region_items, nregions);
    }

    do_display_drawing();
}

void set_new_region(float *lower, float *upper)
{
    set_region_func(ref_type, lower, upper);

    set_textfields(region_items, nregions);
}

void update_region_params()
{
    if (region_popup)
    {
	get_textfields(region_items, nregions);
	get_textfields(saved_items, nsaved);
	get_textfields(stats_file_items, nstats);
    	ref_type = get_radio_state(reference_boxes, nref_types);
    	stats_type = get_radio_state(stats_boxes, nstats_types);

	sprintf(region_ref, "%d", ref_type);
	sprintf(stats_use_file, "%d", stats_type);
    }
    else
    {
	if (*region_ref)
	    ref_type = atoi(region_ref);

	if (*stats_use_file)
	    stats_type = atoi(stats_use_file);
    }
}

Status region_apply(int *p_ref_type, float *lower, float *upper,
			int *fold_type, int *flip_type, String error_msg)
{
    update_region_params();
    update_fold_params();

    CHECK_STATUS(region_apply_func(p_ref_type, lower, upper, fold_type, flip_type, error_msg));

    set_center_width(lower, upper);

    return  OK;
}

Status move_region(int *original, int *translation, String error_msg)
{
    CHECK_STATUS(move_region_func(original, translation, error_msg));

    set_region(ref_type);

    return  OK;
}

Status change_region(int *original, int *begin, int *end, int type,
							String error_msg)
{
    CHECK_STATUS(change_region_func(original, begin, end, type, error_msg));

    set_region(ref_type);

    return  OK;
}

static Status select_command(int position, String error_msg)
{
    CHECK_STATUS(region_select_func(position, error_msg));

    return  OK;
}

static void select_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    int position = list_position(cbs);
    Line error_msg;

    if (select_command(position, error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

Status region_select_command(String value, Generic_ptr data, String error_msg)
{
    int position;

    if (!position_name_exists(value, &position))
    {
	sprintf(error_msg, "saved position '%s' unknown", value);
	return  ERROR;
    }

    CHECK_STATUS(select_command(position, error_msg));

    return  OK;
}

static Status insert_command(String error_msg)
{
    CHECK_STATUS(region_insert_func(force_question, error_msg));

    if (position_list)
	insert_in_list(position_list, position_name, 0);

    return  OK;
}

static void insert_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    update_region_params();

    if (insert_command(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

Status region_insert_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(insert_command(error_msg));

    return  OK;
}

static Status go_to_command(String error_msg)
{
    int type;

    CHECK_STATUS(region_go_to_func(&type, error_msg));

    set_region(type);

    return  OK;
}

static void go_to_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    (void) go_to_command(error_msg); /* do not care about 'ERROR' */
}

Status region_go_to_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(go_to_command(error_msg));

    return  OK;
}

static Status delete_command(String error_msg)
{
    int position;

    CHECK_STATUS(region_delete_func(&position, error_msg));

    if (position_list)
	delete_from_list(position_list, position+1);

    return  OK;
}

static void delete_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    (void) delete_command(error_msg); /* do not care about 'ERROR' */
}

Status region_delete_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(delete_command(error_msg));

    return  OK;
}

static void apply_command()
{
    do_display_drawing();
}

static void apply_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    apply_command();
}

Status region_apply_command(String value, Generic_ptr data, String error_msg)
{
    apply_command();

    return  OK;
}

static Status stats_command(String error_msg)
{
    static Print_funcs print_funcs = { start_info, print_info, end_info };

    if (stats_type == YES_STATS_FILE)
	CHECK_STATUS(region_stats_open_func(statistics_file, error_msg));
 
    CHECK_STATUS(region_stats_func(&print_funcs, rectangle_displayed, error_msg));

    if (stats_type == YES_STATS_FILE)
	region_stats_close_func();

    return  OK;
}

static void stats_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    update_region_params();

    if (stats_command(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

Status region_stats_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(stats_command(error_msg));

    return  OK;
}

static Status whole_command(String error_msg)
{
    CHECK_STATUS(region_whole_func(error_msg));
	
    set_region(ref_type);

    return  OK;
}

static void whole_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    update_region_params();

    if (whole_command(error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

static void peaks_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    peaks_popup_callback(get_topshell(), data, cbs);
}

Status region_whole_command(String value, Generic_ptr data, String error_msg)
{
    CHECK_STATUS(whole_command(error_msg));

    return  OK;
}

static void fold_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    fold_popup_callback(get_topshell(), data, cbs);
}

static String position_string_func(int n, Generic_ptr data)
{
    return  get_position_name(n);
}

static Widget create_position_list_form(Widget parent)
{
    int npositions;
    Scrolled_list_info position_list_info;
    Widget label, buttons, texts, form;
    static Button_item button_items[] =
    {
	{ "insert",	insert_callback,	NULL },
	{ "go to",	go_to_callback,		NULL },
	{ "delete",	delete_callback,	NULL }
    };
    static int nbuttons = ARRAY_SIZE(button_items);

    form = create_form(parent);
    CHECK_WIDGET(form);

    label = create_label(form, "saved positions:");
    CHECK_WIDGET_DESTROY(label, form);
    attachments(label, FORM_ATTACH, NO_ATTACH, FORM_ATTACH, NO_ATTACH);

    texts = create_textfields(form, saved_items, nsaved);
    CHECK_WIDGET_DESTROY(texts, form);
    attachments(texts, label, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    buttons = create_vertical_buttons(form, button_items, nbuttons);
    CHECK_WIDGET_DESTROY(buttons, form);
    attachments(buttons, texts, NO_ATTACH, FORM_ATTACH, NO_ATTACH);
    offsets(buttons, 2*UNIT_OFFSET, NO_OFFSET, 2*UNIT_OFFSET, NO_OFFSET);

    npositions = get_number_positions();

    position_list_info.selection_policy = SINGLE_SELECT;
    position_list_info.nitems = npositions;
    position_list_info.func = position_string_func;
    position_list_info.nvisible_items = VISIBLE_ITEMS;
    position_list_info.callback = select_callback;

    position_list = create_scrolled_list(form, &position_list_info);
    CHECK_WIDGET_DESTROY(position_list, form);
    attachments(WIDGET_PARENT(position_list),
				texts, FORM_ATTACH, buttons, FORM_ATTACH);
    offsets(WIDGET_PARENT(position_list),
			UNIT_OFFSET, UNIT_OFFSET, 2*UNIT_OFFSET, UNIT_OFFSET);

    manage_widget(form);

    return  form;
}

static void create_region_popup(Widget parent)
{
    Widget buttons, buttons2, radio, texts, label, form, separator;
/*
    static Button_item whole_items[] =
    {
    	{ " whole ",	whole_callback,		NO_DATA }
    };
    static int nwhole = ARRAY_SIZE(whole_items);
*/
    static Button_item button_items[] =
    {
    	{ "stats",	stats_callback,		NO_DATA },
    	{ "whole",	whole_callback,		NO_DATA },
    	{ "peaks",	peaks_callback,		NO_DATA },
    	{ "folding",	fold_callback,		NO_DATA }
    };
    static int nbuttons = ARRAY_SIZE(button_items);
    Apply_dismiss_help_info adh_info;
    char apply_label[] = "apply";

    update_region_params();

    region_popup = create_popup(parent, "Region of Plane");
    CHECK_WIDGET_WARNING(region_popup);

    region_form = create_form(region_popup);
    CHECK_WIDGET_DESTROY_WARNING(region_form, region_popup);

    texts = create_textfields(region_form, region_items, nregions);
    CHECK_WIDGET_DESTROY_WARNING(texts, region_popup);
    attachments(texts, FORM_ATTACH, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    set_reference_items((Box_item *) reference_items, reference_boxes);
    radio = create_radiobox(region_form, reference_items, nref_types,
						ref_type, HORIZONTAL);
    CHECK_WIDGET_DESTROY_WARNING(radio, region_popup);
    attachments(radio, texts, NO_ATTACH, NO_ATTACH, FORM_ATTACH);

/*
    buttons = create_horizontal_buttons(region_form, whole_items, nwhole);
    CHECK_WIDGET_DESTROY_WARNING(buttons, region_popup);
    attachments(buttons, radio, NO_ATTACH, NO_ATTACH, FORM_ATTACH);
    offsets(buttons, UNIT_OFFSET, UNIT_OFFSET, UNIT_OFFSET, 2*UNIT_OFFSET);
*/

    separator = create_separator(region_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, region_popup);
    attachments(separator, radio, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);
/*
    attachments(separator, buttons, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);
*/
    offsets(separator, UNIT_OFFSET, NO_OFFSET, NO_OFFSET, NO_OFFSET);

    form = create_position_list_form(region_form);
    CHECK_WIDGET_DESTROY_WARNING(form, region_popup);
    attachments(form, separator, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    separator = create_separator(region_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, region_popup);
    attachments(separator, form, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    label = create_label(region_form, "use statistics file? ");
    CHECK_WIDGET_DESTROY_WARNING(label, region_form);
    attachments(label, separator, NO_ATTACH, FORM_ATTACH, NO_ATTACH);

    radio = create_radiobox(region_form, stats_type_items, nstats_types,
						stats_type, HORIZONTAL);
    CHECK_WIDGET_DESTROY_WARNING(radio, region_popup);
    attachments(radio, separator, NO_ATTACH, label, NO_ATTACH);

    texts = create_textfields(region_form, stats_file_items, nstats);
    CHECK_WIDGET_DESTROY_WARNING(texts, region_popup);
    attachments(texts, radio, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    separator = create_separator(region_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, region_popup);
    attachments(separator, texts, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    buttons = create_horizontal_buttons(region_form, button_items, nbuttons);
    CHECK_WIDGET_DESTROY_WARNING(buttons, region_popup);
    attachments(buttons, separator, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);
    offsets(buttons, UNIT_OFFSET, UNIT_OFFSET, NO_OFFSET, NO_OFFSET);

    adh_info.apply_label = apply_label;
    adh_info.apply_callback = apply_callback;
    adh_info.dismiss_form = region_form;
    adh_info.help_message = region_help;

    buttons2 = create_apply_dismiss_help(region_form, &adh_info);
    CHECK_WIDGET_DESTROY_WARNING(buttons2, region_popup);
    attachments(buttons2, buttons, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);
    offsets(buttons2, UNIT_OFFSET, UNIT_OFFSET, NO_OFFSET, NO_OFFSET);

    manage_widget(region_form);
}

void region_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!region_popup)
        create_region_popup(parent);

    if (region_popup)
        popup(region_form);
}

Status region_popup_command(String value, Generic_ptr data, String error_msg)
{
    region_popup_callback(get_topshell(), (Callback_ptr) NULL,
							(Callback_ptr) NULL);

    return  OK;
}
