#include "lp_extend.h"

#include "command.h"
#include "fblp.h"

static int ncodes = 0;
static int npoints[MAX_NCODES];
static int length[MAX_NCODES];
static int npred[MAX_NCODES];
static float cutoff[MAX_NCODES];
static float *data_store[MAX_NCODES];
static float *u[MAX_NCODES];
static float *v[MAX_NCODES];
static float *w[MAX_NCODES];
static float *t1[MAX_NCODES];
static float *t2[MAX_NCODES];
static float *t3[MAX_NCODES];
static float *t4[MAX_NCODES];
static float *t5[MAX_NCODES];

static void do_lp_extend(int code, float *data)
{
    int i, npts = npoints[code], nprd = npred[code];
    float *x;

    if (nprd < 0)
    {
	x = data_store[code];

	for (i = 0; i < npts; i++)
	{
	    x[2*i] = data[2*(npts-i-1)];
	    x[2*i+1] = data[2*(npts-i-1)+1];
	}
    }
    else
    {
	x = data;
    }

    forward_backward_lp(x, npts, length[code], ABS(nprd),
	u[code], v[code], w[code], t1[code], t2[code],
	t3[code], t4[code], t5[code], cutoff[code]);

    if (nprd < 0)
    {
	npts = npts + ABS(nprd);

	for (i = 0; i < npts; i++)
	{
	    data[2*i] = x[2*(npts-i-1)];
	    data[2*i+1] = x[2*(npts-i-1)+1];
	}
    }
}

static Status check_allocation(int code)
{
    int npts, lngth, nprd, ntot, m, n;

    npts = npoints[code];
    lngth = length[code];
    nprd = npred[code];
    ntot = npts + ABS(nprd);
    m = npts - lngth - 1;
    n = lngth;

    if (nprd < 0)
	MALLOC(data_store[code], float, 2*ntot);

    MALLOC(u[code], float, 2*m*n);
    MALLOC(v[code], float, 2*n*n);
    MALLOC(w[code], float, 2*n);

    MALLOC(t1[code], float, 2*(n+1));
    MALLOC(t2[code], float, 2*MAX(m,n+1));
    MALLOC(t3[code], float, 2*(n+1));
    MALLOC(t4[code], float, 2*n);
    MALLOC(t5[code], float, 2*n);

    return  OK;
}

Status init_lp_extend(Generic_ptr *param, String error_msg)
{
    int type, npts, nprd, lngth, n;
    float cut;
    Line msg;

    nprd = *((int *) param[0]);
	/* number of points to predict in complex points */

    lngth = *((int *) param[1]);
	/* length to use for SVD -- typically 0.25-0.33 npts */

    cut = *((float *) param[2]);
	/* cutoff */

    if (nprd == 0)
        RETURN_ERROR_MSG("'lp_extend': number of predicted points must be non-zero");

    sprintf(msg, "lp_extend %d %d %3.2e", nprd, lngth, cut);
    if (setup_command(&type, &npts, ncodes, msg,
					do_lp_extend, error_msg) == ERROR)
        return  ERROR;

    if (type == REAL_DATA)
	RETURN_ERROR_MSG("'lp_extend': must have complex data");

    npts /= 2;

    n = (npts - 1) / 2;
    if ((lngth < 1) || (lngth > n))
    {
        sprintf(error_msg,
		"'lp_extend': length (%d) must be >= 1 and <= %d",
		lngth, n);
	return  ERROR;
    }

    npoints[ncodes] = npts;
    npred[ncodes] = nprd;
    length[ncodes] = lngth;
    cutoff[ncodes] = cut;

    if (check_allocation(ncodes) == ERROR)
        RETURN_ERROR_MSG("'lp_extend': allocating memory");

    n = 2 * (npts + ABS(nprd));
    CHECK_STATUS(end_command(type, n, "lp_extend", error_msg));

    ncodes++;

    return  OK;
}
