#include "svd.h"

/* Idea for code comes from Numerical Recipes, pp 52 - 64. */
/* This in turn comes from "Singular Value Decomposition and */
/* Least Squares Solutions", by G.H. Golub and C. Reinsch, in */
/* Linear Algebra, ed. J.H. Wilkinson and C. Reinsch, pp 134 - 148. */
/* The major difference between these two codings is the introduction */
/* in the former of 'scale' in order to help reduce numerical error. */
/* There is also a re-organization of the bidiagonal diagonalization. */

/* svd calculation decomposes matrix a = u w v_transpose */
/* a[m][n], u[m][n], w[n][n], v[n][n], w is diagonal */

/* It is assumed that m >= n (this is not checked for). */
/* On input u stores the matrix, a, for which the svd is done. */
/* This is overwritten on output by the svd matrix u. */
/* Only the diagonal values of the matrix w are stored. */
/* t[n] is used for temporary storage. */

#define  MAX_NITER  30

#define HYPOTENOUS(a, b) \
	((ABS(a) > ABS(b)) ? \
		(d = b/a, ABS(a) * sqrt(1+d*d)) : \
		((b != 0) ? (d = a/b, ABS(b) * sqrt(1+d*d)) : 0))
    
void svd(float **u, float **v, float *w, float *t, int m, int n,
							Bool *converged)
{
    int i, iter, j, jj, k, l, nm;
    Bool flag;
    float c, d, f, h, s, x, y, z;
    float anorm = 0, g = 0, scale = 0;
 
/*  Householder's reduction to bidiagonal form  */

    for (i = 0; i < n; i++)
    {
	l = i + 1;
	t[i] = scale * g;
	g = s = scale = 0;

	for (k = i; k < m; k++)
	    scale += ABS(u[k][i]);

	if (scale != 0)
	{
	    for (k = i; k < m; k++)
	    {
		u[k][i] /= scale;
		s += u[k][i] * u[k][i];
	    }

	    f = u[i][i];
	    g = - SIGN(f) * sqrt(s);
	    h = f*g - s;
	    u[i][i] = f - g;

	    for (j = l; j < n; j++)
	    {
		s = 0;
		for (k = i; k < m; k++)
		    s += u[k][i] * u[k][j];

		f = s / h;
		for (k = i; k < m; k++)
		    u[k][j] += f * u[k][i];
	    }

	    for (k = i; k < m; k++)
		u[k][i] *= scale;
	}

	w[i] = scale * g;
	g = s = scale = 0;

	for (k = l; k < n; k++)
	    scale += ABS(u[i][k]);

	if (scale != 0)
	{
	    for (k = l; k < n; k++)
	    {
		u[i][k] /= scale;
		s += u[i][k] * u[i][k];
	    }

	    f = u[i][l];
	    g = - SIGN(f) * sqrt(s);
	    h = f*g - s;
	    u[i][l] = f - g;

	    for (k = l; k < n; k++)
		t[k] = u[i][k] / h;

	    for (j = l; j < m; j++)
	    {
		s = 0;
		for (k = l; k < n; k++)
		    s += u[j][k] * u[i][k];

		for (k = l; k < n; k++)
		    u[j][k] += s * t[k];
	    }

	    for (k = l; k < n; k++)
		u[i][k] *= scale;
	}

	d = ABS(w[i]) + ABS(t[i]);
	anorm = MAX(anorm, d);
    }

/*  Accumulation of right-hand transformations  */

    g = 0;
    for (i = n-1; i >= 0; i--)
    {
	l = i + 1;

	if (g != 0)
	{
	    for (j = l; j < n; j++)
		v[j][i] = (u[i][j] / u[i][l]) / g;

	    for (j = l; j < n; j++)
	    {
		s = 0;
		for (k = l; k < n; k++)
		    s += u[i][k] * v[k][j];

		for (k = l; k < n; k++)
		    v[k][j] += s * v[k][i];
	    }
	}

	for (j = l; j < n; j++)
	    v[i][j] = v[j][i] = 0;

	v[i][i] = 1;
	g = t[i];
    }

/*  Accumulation of left-hand transformations  */

    for (i = n-1; i >= 0; i--)
    {
	l = i + 1;
	g = w[i];

	for (j = l; j < n; j++)
	    u[i][j] = 0;

	if (g != 0)
	{
	    g = 1 / g;
	    for (j = l; j < n; j++)
	    {
		s = 0;
		for (k = l; k < m; k++)
		    s += u[k][i] * u[k][j];

		f = (s / u[i][i]) * g;
		for (k = i; k < m; k++)
		    u[k][j] += f * u[k][i];
	    }

	    for (j = i; j < m; j++)
		u[j][i] *= g;
	}
	else
	{
	    for (j = i; j < m; j++)
		u[j][i] = 0;
	}

	u[i][i] += 1;
    }

/*  Diagonalization of the bidiagonal form  */

    for (k = n-1; k >= 0; k--)
    {
	for (iter = 0; iter < MAX_NITER; iter++)
	{
	    flag = TRUE;
	    for (l = k; l >= 0; l--)
	    {
		nm = l - 1;
		if ((ABS(t[l]) + anorm) == anorm)
		{
		    flag = FALSE;
		    break;
		}

		if ((ABS(w[nm])+anorm) == anorm)
		    break;
	    }

	    if (flag)
	    {
		s = 1;
		for (i = l; i <= k; i++)
		{
		    f = s * t[i];
		    if ((ABS(f)+anorm) != anorm)
		    {
			g = w[i];
			h = HYPOTENOUS(f, g);
			w[i] = h;
			h = 1 / h;
			c = g * h;
			s = - f * h;

			for (j = 0; j < m; j++)
			{
			    y = u[j][nm];
			    z = u[j][i];
			    u[j][nm] = y*c + z*s;
			    u[j][i] = z*c - y*s;
			}
		    }	
		}
	    }

	    z = w[k];
	    if (l == k)
	    {
		if (z < 0)
		{
		    w[k] = -z;
		    for (j = 0; j < n; j++)
			v[j][k] = -v[j][k];
		}

		break;
	    }
	    else if (iter == (MAX_NITER-1))
	    {
		*converged = FALSE;
		return;
	    }

/*  Shift from bottom 2 x 2 minor  */

	    x = w[l];
	    nm = k - 1;
	    y = w[nm];
	    g = t[nm];
	    h = t[k];
	    f = ((y-z)*(y+z) + (g-h)*(g+h)) / (2*h*y);
	    g = HYPOTENOUS(f,1);
	    f = ((x-z)*(x+z) + h * ((y/(f+(SIGN(f)*g))) - h)) / x;

/*  Next QR transformation  */

	    c = s = 1;

	    for (j = l; j <= nm; j++)
	    {
		i = j + 1;
		g = t[i];
		y = w[i];
		h = s * g;
		g = c * g;
		z = HYPOTENOUS(f, h);
		t[j] = z;
		c = f / z;
		s = h / z;
		f = x*c + g*s;
		g = g*c - x*s;
		h = y * s;
		y = y * c;

		for (jj = 0; jj < n; jj++)
		{
		    x = v[jj][j];
		    z = v[jj][i];
		    v[jj][j] = x*c + z*s;
		    v[jj][i] = z*c - x*s;
		}

		z = HYPOTENOUS(f, h);
		w[j] = z;

		if (z != 0)
		{
		    z = 1 / z;
		    c = f * z;
		    s = h * z;
		}

		f = c*g + s*y;
		x = c*y - s*g;

		for (jj = 0; jj < m; jj++)
		{
		    y = u[jj][j];
		    z = u[jj][i];
		    u[jj][j] = y*c + z*s;
		    u[jj][i] = z*c - y*s;
		}
	    }

	    t[l] = 0;
	    t[k] = f;
	    w[k] = x;
	}
    }

    *converged = TRUE;
}

/* Given an svd decomposition, svd_fit calculates the least */
/* squares solution to the linear equation a x = d (d[m]). */
/* cutoff (>0 and <1) determines which svd values are signficant. */
/* t[n] is used for temporary storage. */

void svd_fit(float **u, float **v, float *w, float *x, float *d,
				float *t, float cutoff, int m, int n)
{
    int i, j, k;
    float s, wmin, wmax;

    wmax = 0;

    for (j = 0; j < n; j++)
	wmax = MAX(wmax, w[j]);

    wmin = cutoff * wmax;

    for (j = 0; j < n; j++)
	if (w[j] < wmin)
	    w[j] = 0;

    for (j = 0; j < n; j++)
    {
	s = 0;
	if (w[j] != 0)
	{
	    for (i = 0; i < m; i++)
		s += u[i][j] * d[i];

	    s /= w[j];
	}

	t[j] = s;
    }

    for (j = 0; j < n; j++)
    {
	s = 0;
	for (k = 0; k < n; k++)
	    s += v[j][k] * t[k];

	x[j] = s;
    }
}
