#include "slides.h"

#include "list.h"
#include "output.h"
#include "parse.h"
#include "ps.h"
#include "ps_func.h"
#include "ps_strings.h"
#include "utility.h"

#define  OUTPUT_FONT	"Times-Roman"

#define  FMT	"%6.2f "

static int ninputs;
static Line *input_file;
static Line output_file;
static FILE **file_in;
static FILE *file_out;

static String slides_file;

static Bool inputs_found = FALSE;
static Bool output_found = FALSE;

static Line title_text;
static Line title_font = OUTPUT_FONT;
static int title_size = 18;
static int title_edge = EDGE_TOP;
static int title_justification = JUSTIFICATION_LEFT;
static Bool title_use = FALSE;

static Line date_text;
static Line date_font = OUTPUT_FONT;
static int date_size = 12;
static int date_edge = EDGE_BOTTOM;
static int date_justification = JUSTIFICATION_RIGHT;
static Bool date_use = FALSE;

static Line file_text;
static Line file_font = OUTPUT_FONT;
static int file_size = 12;
static int file_edge = EDGE_BOTTOM;
static int file_justification = JUSTIFICATION_LEFT;
static Bool file_use = FALSE;

static Line *text_text;
static Line *text_font;
static int *text_size;
static int *text_edge;
static int *text_justification;
static float *text_x_offset;
static float *text_y_offset;
static Bool *text_use;

static Bool *slide_found;
static float *input_x;
static float *input_y;
static float *input_w;
static float *input_h;
static float *output_x;
static float *output_y;
static float *output_w;
static float *output_h;

static int output_gl = PS_OUTPUT;
static int paper_dirn = LANDSCAPE;
static int size_units = INCH_UNITS;
static int paper_size = A4_PAPER;
static Line other_size;
static Bool keep_border = FALSE;

static Output_border title_border;
static Output_border date_border;
static Output_border file_border;
static Output_border text_border;

static float *x_min;
static float *y_min;
static float *x_max;
static float *y_max;

static float *left_box;
static float *right_box;
static float *top_box;
static float *bottom_box;
static float *x_translate;
static float *y_translate;
static float *x_scale;
static float *y_scale;

static int slide = -1;

static float plot_size[PLOT_DIM];

#define  FOUND_TWICE(string) \
	 {   sprintf(error_msg, "'%s' found twice", \
						string);  return  ERROR;   }

#define  NOT_FOUND(string) \
	 {   sprintf(error_msg, "no '%s' found", \
						string);  return  ERROR;   }

#define  CHECK_INPUT(string) \
	 {   if (!inputs_found) \
	    {  sprintf(error_msg, "'inputs' must appear before '%s'", \
					string);  return  ERROR;   }   }

#define  CHECK_SLIDE(string) \
	 {    if (slide < 0) \
	    {  sprintf(error_msg, "'slide' must appear before '%s'", \
					string);  return  ERROR;   }   }

#define  CHECK_INPUT_SLIDE(string) \
	 {   CHECK_INPUT(string); CHECK_SLIDE(string);   }

static Status check_float_positive(String msg, float value, String error_msg)
{
    if (value <= 0)
    {
	sprintf(error_msg, "%s = %f, must be > 0", msg, value);
	return  ERROR;
    }

    return  OK;
}

static Status check_float_set(String msg, float value, String error_msg)
{
    if (value < 0)
    {
	sprintf(error_msg, "%s must be set", msg);
	return  ERROR;
    }

    return  OK;
}

static Status check_int_positive(String msg, int value, String error_msg)
{
    if (value < 0)
    {
	sprintf(error_msg, "%s = %d, must be >= 0", msg, value);
	return  ERROR;
    }

    return  OK;
}

static Status allocate_memory(int n, String error_msg)
{
    sprintf(error_msg, "allocating memory for input files");

    MALLOC(input_file, Line, n);
    MALLOC(file_in, FILE *, n);
    MALLOC(text_text, Line, n);
    MALLOC(x_min, float, n);
    MALLOC(y_min, float, n);
    MALLOC(x_max, float, n);
    MALLOC(y_max, float, n);
    MALLOC(left_box, float, n);
    MALLOC(right_box, float, n);
    MALLOC(top_box, float, n);
    MALLOC(bottom_box, float, n);
    MALLOC(x_translate, float, n);
    MALLOC(y_translate, float, n);
    MALLOC(x_scale, float, n);
    MALLOC(y_scale, float, n);
    MALLOC(text_font, Line, n);
    MALLOC(text_size, int, n);
    MALLOC(text_edge, int, n);
    MALLOC(text_justification, int, n);
    MALLOC(text_x_offset, float, n);
    MALLOC(text_y_offset, float, n);
    MALLOC(text_use, Bool, n);
    MALLOC(slide_found, Bool, n);
    MALLOC(input_x, float, n);
    MALLOC(input_y, float, n);
    MALLOC(input_w, float, n);
    MALLOC(input_h, float, n);
    MALLOC(output_x, float, n);
    MALLOC(output_y, float, n);
    MALLOC(output_w, float, n);
    MALLOC(output_h, float, n);

    return  OK;
}

static Bool have_no_text(String text)
{
    int i, n = strlen(text);

    for (i = 0; i < n; i++)
    {
	if (!isspace(text[i]))
	    return  FALSE;
    }

    return  TRUE;
}

static Status text_parse(String line, String text, String error_msg)
{
    char *ptr;

    if (have_no_text(line))
    {
	text[0] = 0;
	return  OK;
    }

    sprintf(error_msg, "text: ");
    error_msg += strlen(error_msg);

    if (!(ptr = strrchr(line, '"')))
	RETURN_ERROR_MSG("missing right \"");

    *ptr = 0;

    if (!(ptr = strchr(line, '"')))
	RETURN_ERROR_MSG("missing left \"");

    strcpy(text, ptr+1);

    return  OK;
}

static Status name_parse(int *value, String name, String msg,
					int n, String *names, String error_msg)
{
    int i;
    char *ptr;
    Line line;

    for (i = 0; i < n; i++)
    {
	strcpy(line, names[i]);

	if (ptr = strchr(line, ' '))  /* needed for paper_size_names */
	    *ptr = 0;

	if (equal_strings(name, line))
	    break;
    }

    if (i == n)
    {
	sprintf(error_msg, "unknown '%s' type '%s'", msg, name);
	return  ERROR;
    }

    *value = i;

    return  OK;
}

static Status edge_parse(int *edge, String msg, String name, String error_msg)
{
    CHECK_STATUS(name_parse(edge, name, msg, NEDGES, edge_names, error_msg));

    return  OK;
}

static Status justify_parse(int *justification, String msg, String name,
							String error_msg)
{
    CHECK_STATUS(name_parse(justification, name, msg, NJUSTIFICATIONS, justification_names, error_msg));

    return  OK;
}

static Status inputs_parse(Generic_ptr *var, String error_msg)
{
    int i;

    if (inputs_found)
	FOUND_TWICE("inputs");

    ninputs = *((int *) var[0]);

    inputs_found = TRUE;

    if (ninputs <= 0)
    {
	sprintf(error_msg, "#inputs (= %d) must be > 0", ninputs);
	return  ERROR;
    }

    CHECK_STATUS(allocate_memory(ninputs, error_msg));

    for (i = 0; i < ninputs; i++)
	slide_found[i] = FALSE;

    return  OK;
}

static Status output_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (output_found)
	FOUND_TWICE("output");

    strcpy(output_file, name);

    output_found = TRUE;

    return  OK;
}

static Status title_parse(Generic_ptr *var, String error_msg)
{
    sprintf(error_msg, "title: ");
    error_msg += strlen(error_msg);

    CHECK_STATUS(text_parse((String) (var[0]), title_text, error_msg));

    title_use = TRUE;

    return  OK;
}

static Status title_font_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    strcpy(title_font, name);

    return  OK;
}

static Status title_size_parse(Generic_ptr *var, String error_msg)
{
    title_size = *((int *) var[0]);

    CHECK_STATUS(check_int_positive("title_size", title_size, error_msg));

    return  OK;
}

static Status title_edge_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    CHECK_STATUS(edge_parse(&title_edge, "title_edge", name, error_msg));

    return  OK;
}

static Status title_justify_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    CHECK_STATUS(justify_parse(&title_justification, "title_justify", name, error_msg));

    return  OK;
}

static Status date_parse(Generic_ptr *var, String error_msg)
{
    date_use = TRUE;

    return  OK;
}

static Status date_font_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    strcpy(date_font, name);

    return  OK;
}

static Status date_size_parse(Generic_ptr *var, String error_msg)
{
    date_size = *((int *) var[0]);

    CHECK_STATUS(check_int_positive("date_size", date_size, error_msg));

    return  OK;
}

static Status date_edge_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    CHECK_STATUS(edge_parse(&date_edge, "date_edge", name, error_msg));

    return  OK;
}

static Status date_justify_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    CHECK_STATUS(justify_parse(&date_justification, "date_justify", name, error_msg));

    return  OK;
}

static Status slide_parse(Generic_ptr *var, String error_msg)
{
    Line msg;

    CHECK_INPUT("slide");

    slide = *((int *) var[0]) - 1;

    if ((slide < 0) || (slide >= ninputs))
    {
	sprintf(error_msg, "slide (= %d) must be in range (1, %d)",
						slide+1, ninputs);
	return  ERROR;
    }

    if (slide_found[slide])
    {
	sprintf(msg, "slide %d", slide + 1);
	FOUND_TWICE(msg);
    }

    slide_found[slide] = TRUE;
    input_file[slide][0] = 0;

    strcpy(text_font[slide], OUTPUT_FONT);
    text_size[slide] = 12;
    text_edge[slide] = EDGE_BOTTOM;
    text_justification[slide] = JUSTIFICATION_CENTER;
    text_x_offset[slide] = 0;
    text_y_offset[slide] = 0;

    input_x[slide] = -1;
    input_y[slide] = -1;
    input_w[slide] = -1;
    input_h[slide] = -1;
    output_x[slide] = -1;
    output_y[slide] = -1;
    output_w[slide] = -1;
    output_h[slide] = -1;

    text_use[slide] = FALSE;

    return  OK;
}

static Status file_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    CHECK_INPUT_SLIDE("file");

    strcpy(input_file[slide], name);

    return  OK;
}

static Status text_text_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("text");

    sprintf(error_msg, "text: ");
    error_msg += strlen(error_msg);

    CHECK_STATUS(text_parse((String) (var[0]), text_text[slide], error_msg));

    text_use[slide] = TRUE;

    return  OK;
}

static Status text_font_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    CHECK_INPUT_SLIDE("text_font");

    strcpy(text_font[slide], name);

    return  OK;
}

static Status text_size_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("text_size");

    text_size[slide] = *((int *) var[0]);

    CHECK_STATUS(check_int_positive("text_size", text_size[slide], error_msg));

    return  OK;
}

static Status text_edge_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    CHECK_INPUT_SLIDE("text_edge");

    CHECK_STATUS(edge_parse(text_edge+slide, "text_edge", name, error_msg));

    return  OK;
}

static Status text_justify_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    CHECK_INPUT_SLIDE("text_justify");

    CHECK_STATUS(justify_parse(text_justification+slide, "text_justify", name, error_msg));

    return  OK;
}

static Status text_x_offset_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("text_x_offset");

    text_x_offset[slide] = *((float *) var[0]);

    return  OK;
}

static Status text_y_offset_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("text_y_offset");

    text_y_offset[slide] = *((float *) var[0]);

    return  OK;
}

static Status input_x_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("input_x");

    input_x[slide] = *((float *) var[0]);

    return  OK;
}

static Status input_y_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("input_y");

    input_y[slide] = *((float *) var[0]);

    return  OK;
}

static Status input_w_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("input_w");

    input_w[slide] = *((float *) var[0]);

    CHECK_STATUS(check_float_positive("input_w", input_w[slide], error_msg));

    return  OK;
}

static Status input_h_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("input_h");

    input_h[slide] = *((float *) var[0]);

    CHECK_STATUS(check_float_positive("input_h", input_h[slide], error_msg));

    return  OK;
}

static Status output_x_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("output_x");

    output_x[slide] = *((float *) var[0]);

    return  OK;
}

static Status output_y_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("output_y");

    output_y[slide] = *((float *) var[0]);

    return  OK;
}

static Status output_w_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("output_w");

    output_w[slide] = *((float *) var[0]);

    CHECK_STATUS(check_float_positive("output_w", output_w[slide], error_msg));

    return  OK;
}

static Status output_h_parse(Generic_ptr *var, String error_msg)
{
    CHECK_INPUT_SLIDE("output_h");

    output_h[slide] = *((float *) var[0]);

    CHECK_STATUS(check_float_positive("output_h", output_h[slide], error_msg));

    return  OK;
}

static Status paper_size_parse(Generic_ptr *var, String error_msg)
{
    float x, y;
    String line = (String) var[0];
    Line name;

    if (sscanf(line, "%f %f", &x, &y) == 2)
    {
	paper_size = OTHER_SIZE;
	sprintf(other_size, "%2.1f %2.1f", x, y);
    }
    else if (sscanf(line, "%s", name) != 1)
    {
	RETURN_ERROR_MSG("missing paper_size value");
    }
    else
    {
	CHECK_STATUS(name_parse(&paper_size, name, "paper_size", NPAPER_SIZES-1, paper_size_names, error_msg));
    }

    return  OK;
}

static Status portrait_parse(Generic_ptr *var, String error_msg)
{
    paper_dirn = PORTRAIT;

    return  OK;
}

static Status centimeters_parse(Generic_ptr *var, String error_msg)
{
    size_units = CM_UNITS;

    return  OK;
}

static Status eps_parse(Generic_ptr *var, String error_msg)
{
    output_gl = EPS_OUTPUT;

    return  OK;
}

static Status border_parse(Generic_ptr *var, String error_msg)
{
    keep_border = TRUE;

    return  OK;
}

static int parse_int[] = { PARSE_INT };
static int parse_float[] = { PARSE_FLOAT };
static int parse_string[] = { PARSE_STRING };
static int parse_rest[] = { PARSE_REST };

static Parse_line slides_table[] =
{
    { "inputs",		1,	parse_int,		inputs_parse },
    { "output",		1,	parse_string,		output_parse },
    { "title",		1,	parse_rest,		title_parse },
    { "title_font",	1,	parse_string,		title_font_parse },
    { "title_size",	1,	parse_int,		title_size_parse },
    { "title_edge",	1,	parse_string,		title_edge_parse },
    { "title_justify",	1,	parse_string,		title_justify_parse },
    { "date",		0,	(int *) NULL,		date_parse },
    { "date_font",	1,	parse_string,		date_font_parse },
    { "date_size",	1,	parse_int,		date_size_parse },
    { "date_edge",	1,	parse_string,		date_edge_parse },
    { "date_justify",	1,	parse_string,		date_justify_parse },
    { "slide",		1,	parse_int,		slide_parse },
    { "file",		1,	parse_string,		file_parse },
    { "text",		1,	parse_rest,		text_text_parse },
    { "text_font",	1,	parse_string,		text_font_parse },
    { "text_size",	1,	parse_int,		text_size_parse },
    { "text_edge",	1,	parse_string,		text_edge_parse },
    { "text_justify",	1,	parse_string,		text_justify_parse },
    { "text_x_offset",	1,	parse_float,		text_x_offset_parse },
    { "text_y_offset",	1,	parse_float,		text_y_offset_parse },
    { "input_x",	1,	parse_float,		input_x_parse },
    { "input_y",	1,	parse_float,		input_y_parse },
    { "input_w",	1,	parse_float,		input_w_parse },
    { "input_h",	1,	parse_float,		input_h_parse },
    { "output_x",	1,	parse_float,		output_x_parse },
    { "output_y",	1,	parse_float,		output_y_parse },
    { "output_w",	1,	parse_float,		output_w_parse },
    { "output_h",	1,	parse_float,		output_h_parse },
    { "paper_size",	1,	parse_rest,		paper_size_parse },
    { "portrait",	0,	(int *) NULL,		portrait_parse },
    { "centimeters",	0,	(int *) NULL,		centimeters_parse },
    { "EPS",		0,	(int *) NULL,		eps_parse },
    { "keep_border",	0,	(int *) NULL,		border_parse },
    { (String) NULL,	0,	(int *) NULL,		no_parse_func }
};

static Status read_slides_file(String error_msg)
{
    int i;
    char *msg;
    Line msg2;

    sprintf(error_msg, "parsing '%s': ", slides_file);
    error_msg += strlen(error_msg);

    CHECK_STATUS(parse_file(slides_file, slides_table, TRUE, error_msg));

    if (!inputs_found)
	NOT_FOUND("inputs");

    if (!output_found)
	NOT_FOUND("output");

    msg = error_msg;

    for (i = 0; i < ninputs; i++)
    {
	error_msg = msg;

	sprintf(msg2, "slide %d", i + 1);

	if (!slide_found[i])
	    NOT_FOUND(msg2);
	
	sprintf(error_msg, "slide %d: ", i + 1);
	error_msg += strlen(error_msg);

	if (!*input_file[i])
	    NOT_FOUND("file");

	CHECK_STATUS(check_float_set("output_x", output_x[i], error_msg));
	CHECK_STATUS(check_float_set("output_y", output_y[i], error_msg));
	CHECK_STATUS(check_float_set("output_w", output_w[i], error_msg));
	CHECK_STATUS(check_float_set("output_h", output_h[i], error_msg));
    }

    return  OK;
}

static Status init_output(String error_msg)
{
    int i;
    char *msg;
    static Line fonts;
    static Output_choices choices = { &title_border, &date_border, &file_border,
		plot_size, other_size, A4_PAPER, LANDSCAPE, COLOR_GRAY, FALSE };
    static Output_setup output_setup = { SIZE_MARGIN, INCH_UNITS,
                                                        PS_OUTPUT, &choices };
    static Ps_data ps_data = { (FILE *) NULL, PS_OUTPUT, &choices, fonts };

    /* set up plot using margin of size 0 */
    plot_size[0] = 0;
    plot_size[1] = 0;

    output_setup.size_mode = SIZE_MARGIN;

    choices.paper_size = paper_size;
    choices.paper_dirn = paper_dirn;

    output_setup.size_units = size_units;
    output_setup.output_gl = output_gl;

    ps_data.file = file_out;
    ps_data.type = output_gl;

    CHECK_STATUS(setup_output(&output_setup, error_msg));

    CHECK_STATUS(init_ps_font(error_msg));

    CHECK_STATUS(insert_ps_font(title_font, error_msg));
    CHECK_STATUS(insert_ps_font(date_font, error_msg));

    for (i = 0; i < ninputs; i++)
    {
	sprintf(error_msg, "header for file #%d: ", i+1);
	msg = error_msg + strlen(error_msg);

	CHECK_STATUS(insert_ps_font(text_font[i], msg));
	CHECK_STATUS(read_ps_header(file_in[i], x_min+i, y_min+i, x_max+i, y_max+i, msg));
    }

    end_ps_font(fonts);

    start_ps((Generic_ptr) &ps_data);

    return  OK;
}

static void translate_scale_clip(int n)
{
    float tx, sx, cx0, cx1, ty, sy, cy0, cy1;

    tx = x_translate[n];
    sx = x_scale[n];
    cx0 = left_box[n];
    cx1 = right_box[n];

    ty = y_translate[n];
    sy = y_scale[n];
    cy0 = bottom_box[n];
    cy1 = top_box[n];

    fprintf(file_out, FMT FMT "PS_translate\n", tx, ty);
    fprintf(file_out, FMT FMT "PS_scale\n", sx, sy);
    fprintf(file_out, FMT FMT FMT FMT "PS_rectangle_clip\n",
							cx0, cy0, cx1, cy1);
}

static void translate_and_text(int n)
{
    float tx, ty;

    setup_border(&text_border, text_text[n], text_font[n], text_size[n],
				text_justification[n], text_edge[n], text_use[n]);

    tx = output_x[n];
    ty = output_y[n];

    fprintf(file_out, FMT FMT "PS_translate\n", tx, ty);

    text_border.text = text_text[n];
    do_ps_border(&text_border, output_w[n], output_h[n],
			text_x_offset[n], text_y_offset[n]);
}

static Status setup_sizes(String error_msg)
{
    int i;

    for (i = 0; i < ninputs; i++)
    {
	text_x_offset[i] = convert_to_printer_points(text_x_offset[i],
							size_units);
	text_y_offset[i] = convert_to_printer_points(text_y_offset[i],
							size_units);
	if (input_x[i] < 0)
	    input_x[i] = x_min[i];
	else
	    input_x[i] = convert_to_printer_points(input_x[i], size_units);

	if (input_y[i] < 0)
	    input_y[i] = y_min[i];
	else
	    input_y[i] = convert_to_printer_points(input_y[i], size_units);

	if (input_w[i] < 0)
	    input_w[i] = x_max[i] - x_min[i];
	else
	    input_w[i] = convert_to_printer_points(input_w[i], size_units);

	if (input_h[i] < 0)
	    input_h[i] = y_max[i] - y_min[i];
	else
	    input_h[i] = convert_to_printer_points(input_h[i], size_units);

	output_x[i] = convert_to_printer_points(output_x[i], size_units);
	output_y[i] = convert_to_printer_points(output_y[i], size_units);
	output_w[i] = convert_to_printer_points(output_w[i], size_units);
	output_h[i] = convert_to_printer_points(output_h[i], size_units);

	left_box[i] = output_x[i];
	right_box[i] = output_x[i] + output_w[i];
	bottom_box[i] = output_y[i];
	top_box[i] = output_y[i] + output_h[i];

	x_scale[i] = output_w[i] / input_w[i];
	x_translate[i] = output_x[i] - x_scale[i]*input_x[i];

	y_scale[i] = output_h[i] / input_h[i];
	y_translate[i] = output_y[i] - y_scale[i]*input_y[i];
    }

    return  OK;
}

static Status skip_top_of_file(int n, String error_msg)
{
/*  this is dangerous code but about best that can do given old PS files  */

    Line line;

    if (!fgets(line, LINE_SIZE, file_in[n]))  /* %%Page: 1 1 */
	RETURN_ERROR_MSG("reading 'Page' line in body");
/*
    if ((x_max[n]-x_min[n]) > (y_max[n]-y_min[n])) */  /* assume this means landscape */
/*    {
	if (!fgets(line, LINE_SIZE, file_in[n])) */ /* PS_rotate */
/*	    RETURN_ERROR_MSG("reading 'PS_rotate' line in body");

	if (!strstr(line, "PS_rotate"))
	    RETURN_ERROR_MSG("parsing 'PS_rotate' line in body");

	if (!fgets(line, LINE_SIZE, file_in[n])) */ /* PS_translate */
/*	    RETURN_ERROR_MSG("reading 'PS_translate' line in body");

	if (!strstr(line, "PS_translate"))
	    RETURN_ERROR_MSG("parsing 'PS_translate' line in body");
    }
*/
    return  OK;
}

static Status process_files(String error_msg)
{
    int i;
    char *msg;

    CHECK_STATUS(init_output(error_msg));

    CHECK_STATUS(setup_sizes(error_msg));

    for (i = 0; i < ninputs; i++)
    {
	sprintf(error_msg, "copying input file for slide #%d: ", i+1);
	msg = error_msg + strlen(error_msg);

	fprintf(file_out, "%% Slide #%d\n", i+1);

	fprintf(file_out, "PS_graphics_save\n");
	translate_scale_clip(i);
	CHECK_STATUS(skip_top_of_file(i, msg));
	CHECK_STATUS(copy_ps_body(file_in[i], file_out, keep_border, msg));
	fprintf(file_out, "PS_graphics_restore\n");

	fprintf(file_out, "PS_graphics_save\n");
	translate_and_text(i);
	fprintf(file_out, "PS_graphics_restore\n");
    }

    end_ps();

    return  OK;
}

void main(int argc, char **argv)
{
    int i;
    Line error_msg;

    printf(product);

    if (help_request(argc, argv, help_table))
	exit (0);

    if (argc != 2)
    {
        sprintf(error_msg, "correct usage: %s <slides file>", argv[0]);
        ERROR_AND_EXIT(error_msg);
    }

    slides_file = argv[1];

    if (read_slides_file(error_msg) == ERROR)
        ERROR_AND_EXIT(error_msg);

    for (i = 0; i < ninputs; i++)
    {
	if (OPEN_FOR_READING(file_in[i], input_file[i]))
	{
	    sprintf(error_msg, "opening \"%s\" for reading", input_file[i]);
	    ERROR_AND_EXIT(error_msg);
	}
    }

    if (OPEN_FOR_WRITING(file_out, output_file))
    {
	sprintf(error_msg, "opening \"%s\" for writing", output_file);
	ERROR_AND_EXIT(error_msg);
    }

    setup_border(&title_border, title_text, title_font, title_size,
				title_justification, title_edge, title_use);

    setup_border(&date_border, date_text, date_font, date_size,
				date_justification, date_edge, date_use);

    setup_border(&file_border, file_text, file_font, file_size,
				file_justification, file_edge, file_use);

    if (process_files(error_msg) == ERROR)
	ERROR_AND_EXIT(error_msg);
}
