#include "script.h"

#include "atom.h"
#include "par.h"
#include "parse.h"
#include "ref.h"
#include "utility.h"

#define  MAX_NINTENSITY_DISTS		10

static int ndim;

static char *input_shift_file = NULL;
static char *input_crosspeak_file = NULL;
static char *output_crosspeak_file = NULL;
static char *output_match_file = NULL;
static char *output_xplor_file = NULL;
static char *output_nilges_file = NULL;
static char *output_null_file = NULL;

static Bool input_found;

static int ncolumns;
static Columns columns[NCOLUMNS];

static int nintensity_dists;
static Intensity_dist intensity_dists[MAX_NINTENSITY_DISTS];

static float low[MAX_NDIM];
static float high[MAX_NDIM];
static char atom_type[MAX_NDIM];

static Bool split_output;

static int parse_string[] = { PARSE_STRING };
static int parse_int_free[] = { PARSE_INT | PARSE_FREE };
static int parse_int3[] = { PARSE_INT, PARSE_INT, PARSE_INT };
static int parse_float2[] = { PARSE_FLOAT, PARSE_FLOAT };
static int parse_float3[] = { PARSE_FLOAT, PARSE_FLOAT, PARSE_FLOAT };
static int parse_float4[] = { PARSE_FLOAT, PARSE_FLOAT, PARSE_FLOAT, PARSE_FLOAT };
static int parse_int_float[] = { PARSE_INT, PARSE_FLOAT };
static int parse_int_float2[] = { PARSE_INT, PARSE_FLOAT, PARSE_FLOAT };

static Status input_par_parse(Generic_ptr *var, String error_msg)
{
    int i;
    float p;
    String msg, input_par_file = (String) (*var);
    Par_info par_info;

    if (input_found)
	RETURN_ERROR_MSG("'input_par' found twice");

    input_found = TRUE;

    sprintf(error_msg, "par file: ");
    error_msg += strlen(error_msg);

    CHECK_STATUS(read_par_file(input_par_file, &par_info, error_msg));

    ndim = par_info.ndim;

    for (i = 0; i < ndim; i++)
    {
	p = par_info.npoints[i] + HALF;

	convert_from_point(REF_PPM, par_info.npoints[i], par_info.ref+i, &p);

	low[i] = p;
	high[i] = p + par_info.ref[i].sw / par_info.ref[i].sf;

	sprintf(error_msg, "dim %d: ", i+1);
	msg = error_msg + strlen(error_msg);

	CHECK_STATUS(find_atom_type(par_info.ref[i].nuc, &(atom_type[i]), msg));
    }

    return  OK;
}

static Status input_shift_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (input_shift_file)
	RETURN_ERROR_MSG("'input_shift' found twice");

    sprintf(error_msg, "allocating memory");
    STRING_MALLOC_COPY(input_shift_file, name);

    return  OK;
}

static Status input_crosspeak_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (input_crosspeak_file)
	RETURN_ERROR_MSG("'input_crosspeak' found twice");

    sprintf(error_msg, "allocating memory");
    STRING_MALLOC_COPY(input_crosspeak_file, name);

    return  OK;
}

static Status output_crosspeak_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (output_crosspeak_file)
	RETURN_ERROR_MSG("'output_crosspeak' found twice");

    sprintf(error_msg, "allocating memory");
    STRING_MALLOC_COPY(output_crosspeak_file, name);

    return  OK;
}

static Status output_match_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (output_match_file)
	RETURN_ERROR_MSG("'output_match' found twice");

    sprintf(error_msg, "allocating memory");
    STRING_MALLOC_COPY(output_match_file, name);

    return  OK;
}

static Status output_xplor_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (output_xplor_file)
	RETURN_ERROR_MSG("'output_xplor' found twice");

    sprintf(error_msg, "allocating memory");
    STRING_MALLOC_COPY(output_xplor_file, name);

    return  OK;
}

static Status output_nilges_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (output_nilges_file)
	RETURN_ERROR_MSG("'output_nilges' found twice");

    sprintf(error_msg, "allocating memory");
    STRING_MALLOC_COPY(output_nilges_file, name);

    return  OK;
}

static Status output_null_parse(Generic_ptr *var, String error_msg)
{
    String name = (String) (*var);

    if (output_null_file)
	RETURN_ERROR_MSG("'output_null' found twice");

    sprintf(error_msg, "allocating memory");
    STRING_MALLOC_COPY(output_null_file, name);

    return  OK;
}

static Status columns_parse(Generic_ptr *var, String error_msg)
{
    int i, n, a, c, dim;
    Bool shift_only = FALSE;

    if (!input_found)
	RETURN_ERROR_MSG("'columns' found before 'input_par'");

    if (ncolumns >= NCOLUMNS)
    {
	sprintf(error_msg, "'columns' must appear %d times", NCOLUMNS);
	return  ERROR;
    }

    n = *((int *) var[0]);

    if ((n < 1) || (n > MAX_COLUMNS))
    {
	sprintf(error_msg, "'columns' must have between 1 and %d values",
								MAX_COLUMNS);
	return  ERROR;
    }

    columns[ncolumns].ncolumns = n;
    columns[ncolumns].nresidue_ranges = 0;

    for (i = 0; i < n; i++)
    {
	a = *((int *) var[i+1]);
	c = ABS(a);

	if ((c < 1) || (c > ndim))
	{
	    sprintf(error_msg,
		"column absolute values must be between 1 and %d", ndim);
	    return  ERROR;
	}

	if ((a < 0) && (i == 0))
	    RETURN_ERROR_MSG("first column value must be > 0");

	dim = ndim - c; /* Ansig has dims backwards */

	columns[ncolumns].data[i].column = c - 1;
	columns[ncolumns].data[i].low = low[dim];
	columns[ncolumns].data[i].high = high[dim];
	columns[ncolumns].data[i].atom_type = atom_type[dim];
	columns[ncolumns].data[i].exclusion = FALSE;

	if (a < 0)
	    shift_only = TRUE;

	if (i == 0)
	{
	    if (atom_type[dim] != HYDROGEN)
	    {
		sprintf(error_msg,
		    "first column value (%d) atom type is '%c', must be '%c'",
						a, atom_type[dim], HYDROGEN);
		return  ERROR;
	    }
	}
	else /* (i == 1) */
	{
	    if (atom_type[dim] == HYDROGEN)
	    {
		sprintf(error_msg,
		    "second column value (%d) atom type must not be '%c'",
								a, HYDROGEN);
		return  ERROR;
	    }
	}
    }

/*  not very elegant, but should work  */
    if (shift_only)
	columns[ncolumns].ncolumns = n - 1;
 
    columns[ncolumns].nshift_columns = n;

    ncolumns++;

    return  OK;
}

static Status intensity_dist_parse(Generic_ptr *var, String error_msg)
{
    float intensity, distance;

    if (!input_found)
	RETURN_ERROR_MSG("'intensity_dist' found before 'input_par'");

    if (nintensity_dists >= MAX_NINTENSITY_DISTS)
    {
	sprintf(error_msg, "'intensity_dist' must appear no more than %d times",
							MAX_NINTENSITY_DISTS);
	return  ERROR;
    }

    intensity = *((float *) var[0]);
    distance = *((float *) var[1]);

    if (intensity < 0)
	RETURN_ERROR_MSG("intensity must be >= 0");

    if (distance < 0)
	RETURN_ERROR_MSG("distance must be >= 0");

    if (nintensity_dists > 0)
    {
	if (intensity >= intensity_dists[nintensity_dists-1].intensity)
	    RETURN_ERROR_MSG("intensities must be in decreasing magnitude");

	if (distance <= intensity_dists[nintensity_dists-1].distance)
	    RETURN_ERROR_MSG("distances must be in increasing magnitude");
    }

    intensity_dists[nintensity_dists].intensity = intensity;
    intensity_dists[nintensity_dists].distance = distance;
    intensity_dists[nintensity_dists].distance_minus = distance;
    intensity_dists[nintensity_dists].distance_plus = 0;

    nintensity_dists++;

    return  OK;
}

static Status intensity_dist2_parse(Generic_ptr *var, String error_msg)
{
    float intensity, distance, distance_minus;

    if (!input_found)
	RETURN_ERROR_MSG("'intensity_dist2' found before 'input_par'");

    if (nintensity_dists >= MAX_NINTENSITY_DISTS)
    {
	sprintf(error_msg, "'intensity_dist*' must appear no more than %d times",
							MAX_NINTENSITY_DISTS);
	return  ERROR;
    }

    intensity = *((float *) var[0]);
    distance = *((float *) var[1]);
    distance_minus = *((float *) var[2]);

    if (intensity < 0)
	RETURN_ERROR_MSG("intensity must be >= 0");

    if (distance < 0)
	RETURN_ERROR_MSG("distance must be >= 0");

    if (distance_minus < 0)
	RETURN_ERROR_MSG("distance_minus must be >= 0");

    if (nintensity_dists > 0)
    {
	if (intensity >= intensity_dists[nintensity_dists-1].intensity)
	    RETURN_ERROR_MSG("intensities must be in decreasing magnitude");

	if (distance <= intensity_dists[nintensity_dists-1].distance)
	    RETURN_ERROR_MSG("distances must be in increasing magnitude");
    }

    intensity_dists[nintensity_dists].intensity = intensity;
    intensity_dists[nintensity_dists].distance = distance;
    intensity_dists[nintensity_dists].distance_minus = distance_minus;
    intensity_dists[nintensity_dists].distance_plus = 0;

    nintensity_dists++;

    return  OK;
}

static Status intensity_dist3_parse(Generic_ptr *var, String error_msg)
{
    float intensity, distance, distance_minus, distance_plus;

    if (!input_found)
	RETURN_ERROR_MSG("'intensity_dist3' found before 'input_par'");

    if (nintensity_dists >= MAX_NINTENSITY_DISTS)
    {
	sprintf(error_msg, "'intensity_dist*' must appear no more than %d times",
							MAX_NINTENSITY_DISTS);
	return  ERROR;
    }

    intensity = *((float *) var[0]);
    distance = *((float *) var[1]);
    distance_minus = *((float *) var[2]);
    distance_plus = *((float *) var[3]);

    if (intensity < 0)
	RETURN_ERROR_MSG("intensity must be >= 0");

    if (distance < 0)
	RETURN_ERROR_MSG("distance must be >= 0");

    if (distance_minus < 0)
	RETURN_ERROR_MSG("distance_minus must be >= 0");

    if (nintensity_dists > 0)
    {
	if (intensity >= intensity_dists[nintensity_dists-1].intensity)
	    RETURN_ERROR_MSG("intensities must be in decreasing magnitude");

	if (distance <= intensity_dists[nintensity_dists-1].distance)
	    RETURN_ERROR_MSG("distances must be in increasing magnitude");
    }

    intensity_dists[nintensity_dists].intensity = intensity;
    intensity_dists[nintensity_dists].distance = distance;
    intensity_dists[nintensity_dists].distance_minus = distance_minus;
    intensity_dists[nintensity_dists].distance_plus = distance_plus;

    nintensity_dists++;

    return  OK;
}

static Column_data *find_column_data(int c)
{
    int i, j;

    c--;

    for (i = 0; i < ncolumns; i++)
    {
	for (j = 0; j < columns[i].ncolumns; j++)
	{
	    if (columns[i].data[j].column == c)
		return  (columns[i].data + j);
	}
    }

    return  (Column_data *) NULL;
}

static Status exclude_parse(Generic_ptr *var, String error_msg)
{
    int i, j, c;
    float shift, delta;
    Column_data *data;

    if (ncolumns != NCOLUMNS)
	RETURN_ERROR_MSG("'exclude' should be after both 'columns'");

    c = *((int *) var[0]);
    shift = *((float *) var[1]);
    delta = *((float *) var[2]);

    if (!(data = find_column_data(c)))
    {
	sprintf(error_msg, "'exclude': column %d not being used", c);
	return  ERROR;
    }

    if (data->exclusion)
    {
	sprintf(error_msg, "'exclude %d' appears twice", c);
	return  ERROR;
    }

    c--;
    for (i = 0; i < ncolumns; i++)
    {
	for (j = 0; j < columns[i].ncolumns; j++)
	{
	    if (columns[i].data[j].column == c)
	    {
		data = columns[i].data + j;

		data->exclusion = TRUE;
		data->shift = shift;
		data->delta = delta;
	    }
	}
    }

    return  OK;
}

static Status residues_parse(Generic_ptr *var, String error_msg)
{
    int c, n;

    if (ncolumns != NCOLUMNS)
	RETURN_ERROR_MSG("'residues' should be after both 'columns'");

    c = *((int *) var[0]);

    if ((c < 1) || (c > NCOLUMNS))
    {
	sprintf(error_msg,
	    "'residues' columns value must be between 1 and %d", NCOLUMNS);
	return  ERROR;
    }

    c--;

    n = columns[c].nresidue_ranges;

    if (n >= MAX_RANGES)
    {
	sprintf(error_msg, "'residues %d' cannot appear more than %d times",
							c+1, MAX_RANGES);
	return  ERROR;
    }

    columns[c].residue1[n] = *((int *) var[1]);
    columns[c].residue2[n] = *((int *) var[2]);
    columns[c].nresidue_ranges = n + 1;

    return  OK;
}

static Status spectral_width_parse(Generic_ptr *var, String error_msg)
{
    int i, j, c;
    float sw;
    Column_data *data;

    if (ncolumns != NCOLUMNS)
	RETURN_ERROR_MSG("'spectral_width' should be after both 'columns'");

    c = *((int *) var[0]);
    sw = *((float *) var[1]);

    if (!(data = find_column_data(c)))
    {
	sprintf(error_msg, "'spectral_width': column %d not being used", c);
	return  ERROR;
    }

    c--;
    for (i = 0; i < ncolumns; i++)
    {
	for (j = 0; j < columns[i].ncolumns; j++)
	{
	    if (columns[i].data[j].column == c)
	    {
		data = columns[i].data + j;

		data->high = data->low + sw;
	    }
	}
    }

    return  OK;
}

static Status split_output_parse(Generic_ptr *var, String error_msg)
{
    if (split_output)
	RETURN_ERROR_MSG("'split_output' appears twice");

    split_output = TRUE;

    return  OK;
}

static Status check_validity(String error_msg)
{
    if (!input_found)
	RETURN_ERROR_MSG("'input_par' not found");

    if (!input_shift_file)
	RETURN_ERROR_MSG("'input_shift' not found");

    if (!input_crosspeak_file)
	RETURN_ERROR_MSG("'input_crosspeak' not found");

    if (!output_match_file && !output_xplor_file && !output_nilges_file)
	RETURN_ERROR_MSG("no output command found");

    if (ncolumns != NCOLUMNS)
    {
	sprintf(error_msg, "'columns' must appear %d times", NCOLUMNS);
	return  ERROR;
    }

    if (nintensity_dists == 0)
	printf("Warning: found no 'intensity_dist', using distance = ABS(intensity)\n");
/*
	RETURN_ERROR_MSG("'intensity_dist' must appear at least once");
*/

    return  OK;
}

static Parse_line connect_table[] =
{
    { "input_par",		1,	parse_string,	input_par_parse },
    { "input_shift",		1,	parse_string,	input_shift_parse },
    { "input_crosspeak",	1,	parse_string,	input_crosspeak_parse },
    { "output_crosspeak",	1,	parse_string,	output_crosspeak_parse },
    { "output_match",		1,	parse_string,	output_match_parse },
    { "output_xplor",		1,	parse_string,	output_xplor_parse },
    { "output_nilges",		1,	parse_string,	output_nilges_parse },
    { "output_null",		1,	parse_string,	output_null_parse },
    { "columns",		1,	parse_int_free,	columns_parse },
    { "intensity_dist",		2,	parse_float2,	intensity_dist_parse },
    { "intensity_dist2",	3,	parse_float3,	intensity_dist2_parse },
    { "intensity_dist3",	4,	parse_float4,	intensity_dist3_parse },
    { "exclude",		3,	parse_int_float2, exclude_parse },
    { "residues",		3,	parse_int3,	residues_parse },
    { "spectral_width",		2,	parse_int_float,  spectral_width_parse },
    { "split_output",		0,	(int *) NULL,	split_output_parse },
    { (String) NULL,		0,	(int *) NULL,	no_parse_func }
};

Status read_script_file(String file, Connect_info *info,
				Connect_files *files, String error_msg)
{
    FREE(input_shift_file, char);
    FREE(input_crosspeak_file, char);
    FREE(output_crosspeak_file, char);
    FREE(output_match_file, char);
    FREE(output_xplor_file, char);
    FREE(output_nilges_file, char);
    FREE(output_null_file, char);

    input_found = FALSE;
    ncolumns = 0;
    nintensity_dists = 0;
    split_output = FALSE;

    sprintf(error_msg, "\"%s\": ", file);
    error_msg += strlen(error_msg);

    CHECK_STATUS(parse_file(file, connect_table, TRUE, error_msg));

    CHECK_STATUS(check_validity(error_msg));

    info->ndim = ndim;
    info->columns = columns;
    info->nintensity_dists = nintensity_dists;
    info->intensity_dists = intensity_dists;
    info->split_output = split_output;

    files->input_shift_file = input_shift_file;
    files->input_crosspeak_file = input_crosspeak_file;
    files->output_crosspeak_file = output_crosspeak_file;
    files->output_match_file = output_match_file;
    files->output_xplor_file = output_xplor_file;
    files->output_nilges_file = output_nilges_file;
    files->output_null_file = output_null_file;

    return  OK;
}
