#include "region_popup.h"

#include "region.hlp"

#include "callback.h"
#include "display_popup.h"
#include "info_popup.h"
#include "param.h"
#include "ref.h"
#include "ref_popup.h"
#include "region.h"
#include "window.h"

static Widget region_popup = (Widget) NULL;
static Widget region_form;

static Widget x_fit_boxes[LIMITS_DIM];
static Radiobox_item x_fit_items[] =
{
    { "fit data", NO_CALLBACK, NO_DATA, x_fit_boxes + FIT_DATA },
    { "fit region", NO_CALLBACK, NO_DATA, x_fit_boxes + FIT_REGION }
};

static int x_fit_type = FIT_DATA;

static Widget y_fit_boxes[LIMITS_DIM];
static Radiobox_item y_fit_items[] =
{
    { "fit data", NO_CALLBACK, NO_DATA, y_fit_boxes + FIT_DATA },
    { "fit region", NO_CALLBACK, NO_DATA, y_fit_boxes + FIT_REGION }
};

static int y_fit_type = FIT_DATA;

static int nfit_types = LIMITS_DIM;

static Widget x_region_field;
static Widget y_region_field;

static Textfield_item region_items[] =
{
    { "x region: ", x_region, NO_CALLBACK, NO_DATA, &x_region_field },
    { "y region: ", y_region, NO_CALLBACK, NO_DATA, &y_region_field }
};

static int nregions = ARRAY_SIZE(region_items);

static Region_info region_info =
    { FIT_DATA, FIT_DATA, (float *) NULL, (float *) NULL, x_region, y_region };

static Widget reference_boxes[NREF_TYPES];
static Radiobox_item reference_items[NREF_TYPES];

static int ref_type = REF_POINTS;
static int nref_types = ARRAY_SIZE(reference_boxes);

static void set_region(float *x, float *y)
{
    sprintf(x_region, "%3.2f %3.2f", x[0], x[1]);
    sprintf(y_region, "%3.2f %3.2f", y[0], y[1]);

    sprintf(x_fit, "%d", x_fit_type);
    sprintf(y_fit, "%d", y_fit_type);

    if (region_popup)
    {
	set_textfields(region_items, nregions);
	set_radio_state(x_fit_boxes, nfit_types, x_fit_type);
	set_radio_state(y_fit_boxes, nfit_types, y_fit_type);
	set_radio_state(reference_boxes, nref_types, ref_type);
    }
}

Status get_region(float *x, float *y, String error_msg)
{
    region_info.x_fit = x_fit_type;
    region_info.y_fit = y_fit_type;
    region_info.x = x;
    region_info.y = y;

    CHECK_STATUS(find_region(ref_type, &region_info, error_msg));

    return  OK;
}

void update_region_params()
{
    if (region_popup)
    {
	get_textfields(region_items, nregions);
    	x_fit_type = get_radio_state(x_fit_boxes, nfit_types);
    	y_fit_type = get_radio_state(y_fit_boxes, nfit_types);
	ref_type = get_radio_state(reference_boxes, nref_types);

	sprintf(x_fit, "%d", x_fit_type);
	sprintf(y_fit, "%d", y_fit_type);
	sprintf(region_ref, "%d", ref_type);
    }
    else
    {
	if (*x_fit)
	    x_fit_type = atoi(x_fit);

	if (*y_fit)
	    y_fit_type = atoi(y_fit);

	if (*region_ref)
	    ref_type = atoi(region_ref);
    }
}

Status region_apply(int *p_ref_type, float *x, float *y, String error_msg)
{
    update_region_params();

    *p_ref_type = ref_type;

    CHECK_STATUS(get_region(x, y, error_msg));

    set_region(x, y);

    return  OK;
}

static Status init_limits(float *x, float *y)
{
    Line error_msg;

    if (get_region(x, y, error_msg) == ERROR)
	ERROR_AND_RETURN_ERROR(error_msg);

/*  careful, this might cause problems if have (X, Y) plots  */
    check_orientation(ref_type, 1, x, x+1);

    return  OK;
}

static void calc_limits(float *x, int size, int begin, int end, int type)
{
    float c;

    if (type == EXPAND_REGION)
    {
	c = (x[1] - x[0]) / ((float) (size-1));
	x[1] = c * end + x[0];
	x[0] += c * begin;
    }
    else  /* type == CONTRACT_REGION */
    {
	c = (x[1] - x[0]) / ((float) (end - begin));
	x[1] = c * (size - 1 - begin) + x[0];
	x[0] -= c * begin;
    }
}

static void end_limits(float *x, float *y)
{
/*  careful, this might cause problems if have (X, Y) plots  */
    check_orientation(ref_type, 1, x, x+1);

    set_region(x, y);
}

void change_region(int *size, int *begin, int *end, Bool *done, int type)
{
    float x[LIMITS_DIM], y[LIMITS_DIM];

    if (done[0])
	x_fit_type = FIT_REGION;

    if (done[1])
	y_fit_type = FIT_REGION;

    if (init_limits(x, y) == ERROR)
	return;

    if (done[0])
	calc_limits(x, size[0], begin[0], end[0], type);

    if (done[1])
	calc_limits(y, size[1], begin[1], end[1], type);

    end_limits(x, y);
}

void find_point_stats(int *size, int *point)
{
    float c, x[LIMITS_DIM], y[LIMITS_DIM], desired_point[DISPLAY_DIM];
    static Print_funcs print_funcs = { start_info, print_info, end_info };

    if (init_limits(x, y) == ERROR)
	return;
 
    c = point[0] * (x[1] - x[0]) / ((float) (size[0]-1));
    desired_point[0] = x[0] + c;

    c = point[1] * (y[1] - y[0]) / ((float) (size[1]-1));
    desired_point[1] = y[0] + c;

    print_point_stats(ref_type, desired_point, &print_funcs);
}

static void find_region_stats()
{
    int size, begin, end;
    float c, x[DISPLAY_DIM], y[DISPLAY_DIM];
    static Print_funcs print_funcs = { start_info, print_info, end_info };

    if (init_limits(x, y) == ERROR)
	return;
 
    display_limits(&size, &begin, &end);
 
    c = (x[1] - x[0]) / ((float) (size-1));
    x[1] = c*end + x[0];
    x[0] += c*begin;

/*  careful, this might cause problems if have (X, Y) plots  */
    check_orientation(ref_type, 1, x, x+1);

    print_region_stats(ref_type, x, &print_funcs);
}

static void apply_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    Line error_msg;

    if (do_display_drawing(0, error_msg) == ERROR)
	ERROR_AND_RETURN(error_msg);
}

static void stats_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    update_region_params();
    find_region_stats();
}

static void x_fit_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    set_radio_state(x_fit_boxes, LIMITS_DIM, FIT_REGION);
}

static void y_fit_callback(Widget widget, Callback_ptr data, Callback_ptr cbs)
{
    set_radio_state(y_fit_boxes, LIMITS_DIM, FIT_REGION);
}

static Widget create_fit_form(Widget parent, Radiobox_item *items,
						int type, String message)
{
    Widget form, radio, label;

    form = create_form(parent);
    CHECK_WIDGET(form);

    label = create_label(form, message);
    CHECK_WIDGET_DESTROY(label, form);
    attachments(label, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH, NO_ATTACH);

    radio = create_radiobox(form, items, nfit_types, type, HORIZONTAL);
    CHECK_WIDGET_DESTROY(radio, form);
    attachments(radio, FORM_ATTACH, FORM_ATTACH, label, FORM_ATTACH);

    manage_widget(form);

    return  form;
}

static void create_region_popup(Widget parent)
{
    Widget buttons, x_form, y_form, texts, separator, radio;
    static Button_item button_items[] =
    {
    	{ "apply",	apply_callback,		NO_DATA },
    	{ "stats",	stats_callback,		NO_DATA },
#define  DISMISS_BUTTON  2
    	{ "dismiss",	dismiss_callback,	NO_DATA },
    	{ "help",	help_callback,		(Callback_ptr) region_help }
    };
    static int nbuttons = ARRAY_SIZE(button_items);

    update_region_params();

    region_popup = create_popup(parent, "Region for Display");
    CHECK_WIDGET_WARNING(region_popup);

    region_form = create_form(region_popup);
    CHECK_WIDGET_DESTROY_WARNING(region_form, region_popup);

    x_form = create_fit_form(region_form, x_fit_items, x_fit_type, "x fit: ");
    CHECK_WIDGET_DESTROY_WARNING(x_form, region_popup);
    attachments(x_form, FORM_ATTACH, NO_ATTACH, FORM_ATTACH, NO_ATTACH);

    y_form = create_fit_form(region_form, y_fit_items, y_fit_type, "y fit: ");
    CHECK_WIDGET_DESTROY_WARNING(y_form, region_popup);
    attachments(y_form, x_form, NO_ATTACH, FORM_ATTACH, NO_ATTACH);

    separator = create_separator(region_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, region_popup);
    attachments(separator, y_form, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    region_items[0].callback = x_fit_callback;
    region_items[1].callback = y_fit_callback;
    texts = create_textfields(region_form, region_items, nregions);
    CHECK_WIDGET_DESTROY_WARNING(texts, region_popup);
    attachments(texts, separator, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    set_reference_items((Box_item *) reference_items, reference_boxes);
    radio = create_radiobox(region_form, reference_items, nref_types,
						ref_type, HORIZONTAL);
    CHECK_WIDGET_DESTROY_WARNING(radio, region_popup);
    attachments(radio, texts, NO_ATTACH, NO_ATTACH, FORM_ATTACH);

    separator = create_separator(region_form);
    CHECK_WIDGET_DESTROY_WARNING(separator, region_popup);
    attachments(separator, radio, NO_ATTACH, FORM_ATTACH, FORM_ATTACH);

    button_items[DISMISS_BUTTON].data = (Callback_ptr) region_form;
    buttons = create_horizontal_buttons(region_form, button_items, nbuttons);
    CHECK_WIDGET_DESTROY_WARNING(buttons, region_popup);
    attachments(buttons, separator, FORM_ATTACH, FORM_ATTACH, FORM_ATTACH);
    offsets(buttons, UNIT_OFFSET, UNIT_OFFSET, NO_OFFSET, NO_OFFSET);

    manage_widget(region_form);
}

void region_popup_callback(Widget parent, Callback_ptr data, Callback_ptr cbs)
{
    if (!region_popup)
        create_region_popup(parent);

    if (region_popup)
	popup(region_form);
}
