#include "arrange.h"

#include "command.h"
#include "utility.h"

static int ncodes = 0;
static int npoints[MAX_NCODES];
static int value[MAX_NCODES];
static int gcf[MAX_NCODES];
static float real_value[MAX_NCODES];
static float imag_value[MAX_NCODES];
static float *data_copy[MAX_NCODES];

#define  NEXT_VALUE(m)  ((m + n - cycle) % n)

static void do_cycle(int code, float *data)
{
    int cycle, g, n, i, j, k;
    float d;

    cycle = value[code];
    n = npoints[code];
    g = gcf[code];

    for (i = 0; i < g; i++)
    {
	d = data[i];

	for (j = i, k = NEXT_VALUE(j); k != i; j = k, k = NEXT_VALUE(k))
	    data[j] = data[k];

	data[j] = d;
    }
}

static void do_lower(int code, float *data)
{
    int low, n;

    low = value[code];
    n = npoints[code];

    COPY_VECTOR(data, data+low, n);
}

static void do_reverse(int code, float *data)
{
    int step, n, i, j, k;

    step = value[code];
    n = npoints[code];

    for (i = 0; i < n/2; i += step)
    {
	j = n - step - i;

	for (k = 0; k < step; k++)
	    SWAP(data[i+k], data[j+k], float);
    }
}

static void do_shift(int code, float *data)
{
    int shift, n;

    shift = value[code];
    n = npoints[code];

    COPY_VECTOR_FROM_TOP(data+shift, data, n);
    ZERO_VECTOR(data, shift);
}

static void do_zerofill(int code, float *data)
{
    int n1, n2;

    n1 = npoints[code];
    n2 = value[code];

    data += n1;
    ZERO_VECTOR(data, n2);
}

static void do_scale(int code, float *data)
{
    int i, n;
    float v;

    n = npoints[code];
    v = real_value[code];
    data += value[code];

    for (i = 0; i < n; i++)
	data[i] *= v;
}

static void do_scale2(int code, float *data)
{
    int i, n;
    float vr, vi, d;

    n = npoints[code];
    vr = real_value[code];
    vi = imag_value[code];
    data += value[code];

    for (i = 0; i < n; i += 2)
    {
	d = data[i];
	data[i] = vr*d - vi*data[i+1];
	data[i+1] = vi*d + vr*data[i+1];
    }
}

static void do_set(int code, float *data)
{
    int i, n;
    float v;

    n = npoints[code];
    v = real_value[code];
    data += value[code];

    for (i = 0; i < n; i++)
	data[i] = v;
}

static void do_set2(int code, float *data)
{
    int i, n;
    float vr, vi;

    n = npoints[code];
    vr = real_value[code];
    vi = imag_value[code];
    data += value[code];

    for (i = 0; i < n; i += 2)
    {
	data[i] = vr;
	data[i+1] = vi;
    }
}

static void do_mask_ppmm(int code, float *data)
{
    int i, n = npoints[code];

    for (i = 0; i < n; i += 4)
    {
	data[i+2] = - data[i+2];
	data[i+3] = - data[i+3];
    }
}

static void do_mirror(int code, float *data)
{
    int i, m, n = npoints[code], v = value[code];

    m = n - v;

/*  first copy old data to upper part of new data  */

    for (i = m+n-1; i >= m; i--)
	data[i] = data[i-m];

/*  then mirror conjugate old data to lower part of new data  */

    for (i = 0; i < m; i += 2)
    {
	data[m-i-2] = data[i+n];
	data[m-i-1] = - data[i+n+1];
    }
}

static void do_riri2rrii(int code, float *data)
{
    int i, n = npoints[code];
    float *d = data_copy[code];

    for (i = 0; i < n; i++)
    {
	d[i] = data[2*i];
	d[i+n] = data[2*i+1];
    }

    n *= 2;
    COPY_VECTOR(data, d, n);
}

static void do_rrii2riri(int code, float *data)
{
    int i, n = npoints[code];
    float *d = data_copy[code];

    for (i = 0; i < n; i++)
    {
	d[2*i] = data[i];
	d[2*i+1] = data[i+n];
    }

    n *= 2;
    COPY_VECTOR(data, d, n);
}

Status init_cycle(Generic_ptr *param, String error_msg)
{
    int type, npts, cycle;
    Line msg;

    cycle = *((int *) param[0]);

    if (cycle == 0)
	return  OK;

    sprintf(msg, "cycle %d", cycle);
    if (setup_command(&type, &npts, ncodes, msg, do_cycle, error_msg)
								== ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	cycle = value[ncodes] = 2*cycle;
    else
	cycle = value[ncodes] = cycle;

    if (ABS(cycle) >= npts)
	RETURN_ERROR_MSG("'cycle': |value| must be < (complex/real) points");

    npoints[ncodes] = npts;
    gcf[ncodes] = greatest_common_factor(cycle, npts);

    CHECK_STATUS(end_command(type, npts, "cycle", error_msg));

    ncodes++;

    return  OK;
}

Status init_lower(Generic_ptr *param, String error_msg)
{
    int type, npts, low;
    Line msg;

    low = *((int *) param[0]);

    if (low < 1)
	RETURN_ERROR_MSG("'lower': value must be > 0");

    if (low == 1)
	return  OK;

    sprintf(msg, "lower %d", low);
    if (setup_command(&type, &npts, ncodes, msg, do_lower, error_msg)
								== ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	low = value[ncodes] = 2*(low - 1);
    else
	low = value[ncodes] = low - 1;

    if (low >= npts)
	RETURN_ERROR_MSG("'lower': value must be <= (complex/real) points");

    npts = npoints[ncodes] = npts - low;

    CHECK_STATUS(end_command(type, npts, "lower", error_msg));

    ncodes++;

    return  OK;
}

Status init_range(Generic_ptr *param, String error_msg)
{
    int type, npts, low, up;
    Line msg;

    low = *((int *) param[0]);
    up = *((int *) param[1]);

    if (low < 1)
	RETURN_ERROR_MSG("'range': lower value must be > 0");

    if (low > up)
	RETURN_ERROR_MSG("'range': upper value must be >= lower value");

    sprintf(msg, "range %d %d", low, up);
    if (setup_command(&type, &npts, ncodes, msg, do_lower, error_msg)
								== ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	low = value[ncodes] = 2*(low - 1);
    else
	low = value[ncodes] = low - 1;

    if (type == COMPLEX_DATA)
	up *= 2;

    if (up > npts)
      RETURN_ERROR_MSG("'range': upper value must be <= (complex/real) points");

    npts = npoints[ncodes] = up - low;

    CHECK_STATUS(end_command(type, npts, "range", error_msg));

    ncodes++;

    return  OK;
}

Status init_reverse(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "reverse", do_reverse, error_msg)
								== ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	value[ncodes] = 2;
    else
	value[ncodes] = 1;

    npoints[ncodes] = npts;

    CHECK_STATUS(end_command(type, npts, "reverse", error_msg));

    ncodes++;

    return  OK;
}

Status init_shift(Generic_ptr *param, String error_msg)
{
    int type, npts, shift;
    Line msg;

    shift = *((int *) param[0]);

    if (shift < 0)
	RETURN_ERROR_MSG("'shift': value must be >= 0");

    if (shift == 0)
	return  OK;

    sprintf(msg, "shift %d", shift);
    if (setup_command(&type, &npts, ncodes, msg, do_shift, error_msg)
								== ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	shift = value[ncodes] = 2*shift;
    else
	shift = value[ncodes] = shift;

    npoints[ncodes] = npts;
    npts += shift;

    CHECK_STATUS(end_command(type, npts, "shift", error_msg));

    ncodes++;

    return  OK;
}

Status init_upper(Generic_ptr *param, String error_msg)
{
    int type, npts, up;
    Line msg;

    up = *((int *) param[0]);

    if (up < 1)
	RETURN_ERROR_MSG("'upper': value must be > 0");

    sprintf(msg, "upper %d", up);
    if (setup_command(&type, &npts, ncodes, msg,
					do_nothing, error_msg) == ERROR)
        return  ERROR;

    if (type == COMPLEX_DATA)
	up *= 2;

    if (up > npts)
	RETURN_ERROR_MSG("'upper': value must be <= (complex/real) #points");

    npts = up;

    CHECK_STATUS(end_command(type, npts, "upper", error_msg));

    return  OK;
}

Status init_zerofill(Generic_ptr *param, String error_msg)
{
    int n = *((int *) param[0]);
    int type, npts;
    Line msg;

    if (n < 0)
	RETURN_ERROR_MSG("cannot zerofill negative amounts");
 
    if (n > 4)  /* protection against misuse */
	RETURN_ERROR_MSG("cannot zerofill so much");

    if (n == 0)
	return  OK;

    sprintf(msg, "zerofill %d", n);
    if (setup_command(&type, &npts, ncodes, msg,
				do_zerofill, error_msg) == ERROR)
	return  ERROR;

    npoints[ncodes] = npts;
    npts *= (1 << n);
    value[ncodes] = npts - npoints[ncodes];

    CHECK_STATUS(end_command(type, npts, "zerofill", error_msg));

    ncodes++;

    return  OK;
}

static Status range_check(int f, int l, int n, String msg, String error_msg)
{
    if (f < 1)
    {
	sprintf(error_msg, "'%s': first point = %d, must be >= 1", msg, f);
	return  ERROR;
    }

    if (l < f)
    {
	sprintf(error_msg,
	    "'%s': first point = %d, must be <= last point = %d", msg, f, l);
	return  ERROR;
    }

    if (l > n)
    {
	sprintf(error_msg,
	    "'%s': last point = %d, must be <= #points = %d", msg, l, n);
	return  ERROR;
    }

    return  OK;
}

Status init_scale(Generic_ptr *param, String error_msg)
{
    int f = *((int *) param[0]);
    int l = *((int *) param[1]);
    float v = *((float *) param[2]);
    int type, npts;
    Line msg;

    sprintf(msg, "scale %d %d %3.2e", f, l, v);
    if (setup_command(&type, &npts, ncodes, msg,
				do_scale, error_msg) == ERROR)
	return  ERROR;

    if (type == COMPLEX_DATA)
	RETURN_ERROR_MSG("'scale': data must be real");

    CHECK_STATUS(range_check(f, l, npts, "scale", error_msg));

    npoints[ncodes] = l - f + 1;
    value[ncodes] = f - 1;
    real_value[ncodes] = v;

    CHECK_STATUS(end_command(type, npts, "scale", error_msg));

    ncodes++;

    return  OK;
}

Status init_scale2(Generic_ptr *param, String error_msg)
{
    int f = *((int *) param[0]);
    int l = *((int *) param[1]);
    float vr = *((float *) param[2]);
    float vi = *((float *) param[3]);
    int type, npts;
    Line msg;

    sprintf(msg, "scale2 %d %d %3.2e %3.2e", f, l, vr, vi);
    if (setup_command(&type, &npts, ncodes, msg,
				do_scale2, error_msg) == ERROR)
	return  ERROR;

    if (type == REAL_DATA)
	RETURN_ERROR_MSG("'scale2': data must be complex");

    CHECK_STATUS(range_check(f, l, npts/2, "scale2", error_msg));

    npoints[ncodes] = 2*(l - f + 1);
    value[ncodes] = 2*(f - 1);
    real_value[ncodes] = vr;
    imag_value[ncodes] = vi;

    CHECK_STATUS(end_command(type, npts, "scale2", error_msg));

    ncodes++;

    return  OK;
}

Status init_set(Generic_ptr *param, String error_msg)
{
    int f = *((int *) param[0]);
    int l = *((int *) param[1]);
    float v = *((float *) param[2]);
    int type, npts;
    Line msg;

    sprintf(msg, "set %d %d %3.2e", f, l, v);
    if (setup_command(&type, &npts, ncodes, msg,
				do_set, error_msg) == ERROR)
	return  ERROR;

    if (type == COMPLEX_DATA)
	RETURN_ERROR_MSG("'set': data must be real");

    CHECK_STATUS(range_check(f, l, npts, "set", error_msg));

    npoints[ncodes] = l - f + 1;
    value[ncodes] = f - 1;
    real_value[ncodes] = v;

    CHECK_STATUS(end_command(type, npts, "set", error_msg));

    ncodes++;

    return  OK;
}

Status init_set2(Generic_ptr *param, String error_msg)
{
    int f = *((int *) param[0]);
    int l = *((int *) param[1]);
    float vr = *((float *) param[2]);
    float vi = *((float *) param[3]);
    int type, npts;
    Line msg;

    sprintf(msg, "set2 %d %d %3.2e %3.2e", f, l, vr, vi);
    if (setup_command(&type, &npts, ncodes, msg,
				do_set2, error_msg) == ERROR)
	return  ERROR;

    if (type == REAL_DATA)
	RETURN_ERROR_MSG("'set2': data must be complex");

    CHECK_STATUS(range_check(f, l, npts/2, "set2", error_msg));

    npoints[ncodes] = 2*(l - f + 1);
    value[ncodes] = 2*(f - 1);
    real_value[ncodes] = vr;
    imag_value[ncodes] = vi;

    CHECK_STATUS(end_command(type, npts, "set2", error_msg));

    ncodes++;

    return  OK;
}

Status init_mask_ppmm(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "mask_ppmm",
				do_mask_ppmm, error_msg) == ERROR)
	return  ERROR;

    if (type == COMPLEX_DATA)
	RETURN_ERROR_MSG("'mask_ppmm': data must be real");

    npoints[ncodes] = npts;

    CHECK_STATUS(end_command(type, npts, "mask_ppmm", error_msg));

    ncodes++;

    return  OK;
}

Status init_mirror_zero(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "mirror_zero",
					do_mirror, error_msg) == ERROR)
        return  ERROR;

    if (type == REAL_DATA)
	RETURN_ERROR_MSG("'mirror_zero': data must be complex");

    value[ncodes] = 2;
    npoints[ncodes] = npts;
    npts = 2*npts - 2;

    CHECK_STATUS(end_command(type, npts, "mirror_zero", error_msg));

    ncodes++;

    return  OK;
}

Status init_mirror_half(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "mirror_half",
					do_mirror, error_msg) == ERROR)
        return  ERROR;

    if (type == REAL_DATA)
	RETURN_ERROR_MSG("'mirror_half': data must be complex");

    value[ncodes] = 0;
    npoints[ncodes] = npts;
    npts = 2*npts;

    CHECK_STATUS(end_command(type, npts, "mirror_half", error_msg));

    ncodes++;

    return  OK;
}

Status init_riri2rrii(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "riri2rrii",
					do_riri2rrii, error_msg) == ERROR)
        return  ERROR;

    if (type == REAL_DATA)
	RETURN_ERROR_MSG("'riri2rrii': data must be complex");

    npoints[ncodes] = npts / 2;

    sprintf(error_msg, "allocating data copy memory");
    MALLOC(data_copy[ncodes], float, npts);

    CHECK_STATUS(end_command(type, npts, "riri2rrii", error_msg));

    ncodes++;

    return  OK;
}

Status init_rrii2riri(Generic_ptr *param, String error_msg)
{
    int type, npts;

    if (setup_command(&type, &npts, ncodes, "rrii2riri",
					do_rrii2riri, error_msg) == ERROR)
        return  ERROR;

    if (type == REAL_DATA)
	RETURN_ERROR_MSG("'rrii2riri': data must be complex");

    npoints[ncodes] = npts / 2;

    sprintf(error_msg, "allocating data copy memory");
    MALLOC(data_copy[ncodes], float, npts);

    CHECK_STATUS(end_command(type, npts, "rrii2riri", error_msg));

    ncodes++;

    return  OK;
}
