#include "weight.h"

#include "command.h"
#include "script.h" /* for get_reference() */

#define  SMALL_NUMBER  (1.0e-6)

static double exp_x; /* for below only */
#define  MAX_EXP  30.0
#define  MIN_EXP  -MAX_EXP
#define  EXP(x) \
    ((exp_x = x) > MAX_EXP ? LARGE_FLOAT : (exp_x < MIN_EXP) ? 0 : exp(exp_x))

static int ncodes = 0;
static int npoints[MAX_NCODES];
static float *multiplier[MAX_NCODES];
static int n;
static int step;

static void do_weight(int code, float *data)
{
    int i;
    float *m = multiplier[code];

    for (i = 0; i < npoints[code]; i++)
	data[i] *= m[i];
}

static Status check_allocation(int code)
{
    MALLOC(multiplier[code], float, npoints[code]);

    return  OK;
}

static void setup_step(int type, int code)
{
    if (type == COMPLEX_DATA)
    {
	n = npoints[code] / 2;
	step = 2;
    }
    else
    {
	n = npoints[code];
	step = 1;
    }
}

static void setup_decay(int type, int code, float d)
{
    int i, j;
    float c, mult, *m;

    setup_step(type, code);

/*
    d = SIGN(d) * MAX(ABS(d), SMALL_NUMBER);
    d = log(2.0) / d;
*/
    if (n > 1)
	d = log((double) d);
    else
	d = 1.0;
    
    m = multiplier[code];

    for (i = 0; i < n; i++)
    {
	c = ((float) i) / ((float) (n-1));
/*
	mult = exp(-c * d);
*/
	mult = EXP(c * d);

	for (j = 0; j < step; j++)
	    m[i*step+j] = mult;
    }
}

/*  decay_sw suggested by Rasmus Fogh  */
static void setup_decay_sw(int type, int code, float lb, float sw)
{
    int i, j;
    float r, mult, *m;

    setup_step(type, code);
    
    r = - PI * lb * step / (2 * sw);

    m = multiplier[code];

    for (i = 0; i < n; i++)
    {
	mult = EXP(r * i);

	for (j = 0; j < step; j++)
	    m[i*step+j] = mult;
    }
}

static void setup_gaussian(int type, int code, float a, float b)
{
    int i, j;
    float c, mult, *m;

    setup_step(type, code);

/*
    a = MAX(a, SMALL_NUMBER/2);
    b = (b-a) * (b-a);
    b = MAX(b, SMALL_NUMBER);
    b = 2*log(2.0) * a / (b * b);
*/
    b = log((double) b) / ((a-1)*(a-1));

    m = multiplier[code];

    for (i = 0; i < n; i++)
    {
	c = ((float) i) / ((float) (n-1));
/*
	mult = exp(-b * (-c + c*c/(2*a) + a/2));
*/
	mult = EXP(b * (c - a) * (c - a));

	for (j = 0; j < step; j++)
	    m[i*step+j] = mult;
    }
}

/*  gaussian_sw suggested by Rasmus Fogh  */
static void setup_gaussian_sw(int type, int code, float lb, float s, float sw)
{
    int i, j;
    float a, b, c, mult, *m;

    setup_step(type, code);

    a = - LN2 / (s * s);
    b = PI * lb * step / (2 * sw);
    c = - b * b * s * s / (4 * LN2);

    m = multiplier[code];

    for (i = 0; i < n; i++)
    {
	mult = EXP(a + b*i + c*i*i);

	for (j = 0; j < step; j++)
	    m[i*step+j] = mult;
    }
}

static void setup_sinebell(Bool squared, int type, int code, float angle)
{
    int i, j;
    float del, mult, *m;

    setup_step(type, code);

    angle *= RADIAN_SCALE;
    del = (PI - angle) / ((float) n);
    m = multiplier[code];

    for (i = 0; i < n; i++)
    {
	mult = sin(i*del + angle);
	if (squared)  mult *= mult;

	for (j = 0; j < step; j++)
	    m[i*step+j] = mult;
    }
}

/*  inv_cosine suggested by David Agard  */
static void setup_inv_cosine(int type, int code, float freq, float sw)
{
    int i, j;
    float c, mult, *m;
    double del;

    setup_step(type, code);

    del = HALF * PI * freq / sw;
    m = multiplier[code];

    for (i = 0; i < n; i++)
    {
	c = cos(i * del);
	mult = c / (c*c + SMALL_NUMBER);

	for (j = 0; j < step; j++)
	    m[i*step+j] = mult;
    }
}

static Status setup_weight_file(int type, int code, String file,
							String error_msg)
{
    int i, j;
    float v, *m;
    FILE *fp;

    setup_step(type, code);

    if (OPEN_FOR_READING(fp, file))
    {
	sprintf(error_msg, "opening '%s' for reading", file);
	return  ERROR;
    }

    sprintf(error_msg, "reading '%s': ", file);
    error_msg += strlen(error_msg);

    m = multiplier[code];

    for (i = 0; (i < n) && (fscanf(fp, "%f", &v) == 1); i++)
    {
	for (j = 0; j < step; j++)
	    m[i*step+j] = v;
    }

    fclose(fp);

    if (i < n)
    {
	sprintf(error_msg, "only found %d values, expecting %d", i, n);
	return  ERROR;
    }

    return  OK;
}

Status init_decay(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float d;
    Line msg;

    d = *((float *) param[0]);

    if (d < SMALL_NUMBER)
	RETURN_ERROR_MSG("'decay': end value must be > 0");

    sprintf(msg, "decay %3.2f", d);
    if (setup_command(&type, &npts, ncodes, msg,
					do_weight, error_msg) == ERROR)
        return  ERROR;

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'decay': allocating memory");

    setup_decay(type, ncodes, d);

    CHECK_STATUS(end_command(type, npts, "decay", error_msg));

    ncodes++;

    return  OK;
}

Status init_decay_sw(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float lb, sw;
    Line msg;

    lb = *((float *) param[0]);
    sw = *((float *) param[1]);

    if (sw < SMALL_NUMBER)
	sw = get_reference()->sw;

    if (sw < SMALL_NUMBER)
	RETURN_ERROR_MSG("'decay_sw': SW must be > 0");

    sprintf(msg, "decay_sw %2.1f %2.1f", lb, sw);
    if (setup_command(&type, &npts, ncodes, msg,
					do_weight, error_msg) == ERROR)
        return  ERROR;

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'decay_sw': allocating memory");

    setup_decay_sw(type, ncodes, lb, sw);

    CHECK_STATUS(end_command(type, npts, "decay_sw", error_msg));

    ncodes++;

    return  OK;
}

Status init_gaussian(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float a, b;
    Line msg;

    a = *((float *) param[0]);
    b = *((float *) param[1]);

    if (a < SMALL_NUMBER)
	RETURN_ERROR_MSG("'gaussian': one fraction must be > 0");

    if (a > (1-SMALL_NUMBER))
	RETURN_ERROR_MSG("'gaussian': one fraction must be < 1");

    if (b < SMALL_NUMBER)
	RETURN_ERROR_MSG("'gaussian': end value must be > 0");

    sprintf(msg, "gaussian %3.2f %3.2f", a, b);
    if (setup_command(&type, &npts, ncodes, msg,
					do_weight, error_msg) == ERROR)
        return  ERROR;

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'gaussian': allocating memory");

    setup_gaussian(type, ncodes, a, b);

    CHECK_STATUS(end_command(type, npts, "gaussian", error_msg));

    ncodes++;

    return  OK;
}

Status init_gaussian_sw(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float lb, s, sw;
    Line msg;

    lb = *((float *) param[0]);
    s = *((float *) param[1]);
    sw = *((float *) param[2]);

    if (sw < SMALL_NUMBER)
	sw = get_reference()->sw;

    if (sw < SMALL_NUMBER)
	RETURN_ERROR_MSG("'gaussian_sw': SW must be > 0");

    if (s < SMALL_NUMBER)
	RETURN_ERROR_MSG("'gaussian_sw': sharpening factor must be > 0");

    sprintf(msg, "gaussian_sw %2.1f %3.2f %2.1f", lb, s, sw);
    if (setup_command(&type, &npts, ncodes, msg,
					do_weight, error_msg) == ERROR)
        return  ERROR;

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'gaussian_sw': allocating memory");

    setup_gaussian_sw(type, ncodes, lb, s, sw);

    CHECK_STATUS(end_command(type, npts, "gaussian_sw", error_msg));

    ncodes++;

    return  OK;
}

Status init_sinebell(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float angle;
    Line msg;

    angle = *((float *) param[0]);

    sprintf(msg, "sinebell %3.2f", angle);
    if (setup_command(&type, &npts, ncodes, msg,
					do_weight, error_msg) == ERROR)
        return  ERROR;

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'sinebell': allocating memory");

    setup_sinebell(FALSE, type, ncodes, angle);

    CHECK_STATUS(end_command(type, npts, "sinebell", error_msg));

    ncodes++;

    return  OK;
}

Status init_sinebell2(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float angle;
    Line msg;

    angle = *((float *) param[0]);

    sprintf(msg, "sinebell2 %3.2f", angle);
    if (setup_command(&type, &npts, ncodes, msg,
					do_weight, error_msg) == ERROR)
        return  ERROR;

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'sinebell2': allocating memory");

    setup_sinebell(TRUE, type, ncodes, angle);

    CHECK_STATUS(end_command(type, npts, "sinebell2", error_msg));

    ncodes++;

    return  OK;
}

Status init_inv_cosine(Generic_ptr *param, String error_msg)
{
    int type, npts;
    float freq, sw;
    Line msg;

    freq = *((float *) param[0]);
    sw = *((float *) param[1]);

    if (freq < SMALL_NUMBER)
	RETURN_ERROR_MSG("'inv_cosine': frequency must be > 0");

    if (sw < SMALL_NUMBER)
	RETURN_ERROR_MSG("'inv_cosine': spectral width must be > 0");

    sprintf(msg, "inv_cosine %3.2f %3.2f", freq, sw);
    if (setup_command(&type, &npts, ncodes, msg,
					do_weight, error_msg) == ERROR)
        return  ERROR;

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'inv_cosine': allocating memory");

    setup_inv_cosine(type, ncodes, freq, sw);

    CHECK_STATUS(end_command(type, npts, "inv_cosine", error_msg));

    ncodes++;

    return  OK;
}

Status init_weight_file(Generic_ptr *param, String error_msg)
{
    int type, npts;
    Line msg;
    String file = ((char *) param[0]);

    sprintf(msg, "weight_file %s", file);
    if (setup_command(&type, &npts, ncodes, msg,
					do_weight, error_msg) == ERROR)
        return  ERROR;

    npoints[ncodes] = npts;

    if (check_allocation(ncodes) == ERROR)
	RETURN_ERROR_MSG("'weight_file': allocating memory");

    CHECK_STATUS(setup_weight_file(type, ncodes, file, error_msg));

    CHECK_STATUS(end_command(type, npts, "weight_file", error_msg));

    ncodes++;

    return  OK;
}
