/*
 * Copyright (c) 2001 Tommy Bohlin <tommy@gatespace.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/* util.c
 */

#include <irda.h>

/**********************************************************************
 * Presentation
 **********************************************************************/

void showBytes(const void* buf0, int len)
{
  u_char* buf=(u_char*)buf0;
  int i;

  for(i=0;i<len;i++) {
    int mod=i&15;

    if(mod!=0) birda_log(" ");
    birda_log("%02x",buf[i]);
    if(mod==15) birda_log("\n");
  }
  if((i&15)!=0) birda_log("\n");
}

void showChars(const void* buf0, int len)
{
  u_char* buf=(u_char*)buf0;
  int i;

  for(i=0;i<len;i++) {
    int c=buf[i];
    birda_log("%c", c>=32 && c<=126 ? c : '.');
  }
}

/**********************************************************************
 * Endian safe multibyte read/write
 **********************************************************************/

void putBELong(void* buf0, int val)
{
  u_char* buf=(u_char*)buf0;

  buf[0]=val>>24;
  buf[1]=val>>16;
  buf[2]=val>>8;
  buf[3]=val;
}

void putBEShort(void* buf0, int val)
{
  u_char* buf=(u_char*)buf0;

  buf[0]=val>>8;
  buf[1]=val;
}

int getBELong(const void* buf0) {
  u_char* buf=(u_char*)buf0;

  return (buf[0]<<24)|(buf[1]<<16)|(buf[2]<<8)|buf[3];
}

int getBEShort(const void* buf0) {
  u_char* buf=(u_char*)buf0;

  return (buf[0]<<8)|buf[1];
}

int getBEVariable(const void* buf0, int len)
{
  u_char* buf=(u_char*)buf0;
  int value=0;

  while(len-->0) value=(value<<8)|*buf++;
  return value;
}

int getLEVariable(const void* buf0, int len)
{
  u_char* buf=(u_char*)buf0;
  int value=0,shift=0;

  while(len-->0) {
    value|=(*buf++)<<shift;
    shift+=8;
  }
  return value;
}

/**********************************************************************
 * Parameter strings
 **********************************************************************/

int getLEParameter(int key, int deflt, const void* buf0, int len)
{
  u_char* buf=(u_char*)buf0;
  int i,olen;

  for(i=0;i<len-1;i+=2+olen) {
    olen=buf[i+1];
    if(buf[i]==key && olen>=1 && olen<=4 && i+2+olen<=len)
      return getLEVariable(buf+i+2,olen);
  }
  return deflt;
}

int getBEParameter(int key, int deflt, const void* buf0, int len)
{
  u_char* buf=(u_char*)buf0;
  int i,olen;

  for(i=0;i<len-1;i+=2+olen) {
    olen=buf[i+1];
    if(buf[i]==key && olen>=1 && olen<=4 && i+2+olen<=len)
      return getBEVariable(buf+i+2,olen);
  }
  return deflt;
}

/**********************************************************************
 * Bit operations
 **********************************************************************/

int highestBit(int mask) {
  int i;
  for(i=31;i>=0 && (mask&(1<<i))==0;i--);
  return i;
}

int lowestBit(int mask) {
  int i;
  for(i=0;i<32 && (mask&(1<<i))==0;i++);
  return i;
}
