#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include "inbuf.h"
#include "outbuf.h"
#include "strerr.h"
#include "open.h"
#include "hier.h"

static num copy(outbuf *bout,inbuf *bin)
{
  num n;
  char *x;

  for (;;) {
    n = inbuf_feed(bin);
    if (n < 0) return -2;
    if (!n) return 0;
    x = inbuf_PEEK(bin);
    if (outbuf_put(bout,x,n) == -1) return -3;
    inbuf_SEEK(bin,n);
  }
}

#define FATAL "instcopy: fatal: "

static num fdsourcedir = -1;

void hier_h(const char *home,num mode)
{
  if (mkdir(home,0700) == -1)
    if (errno != EEXIST)
      strerr_die4sys(111,FATAL,"unable to mkdir ",home,": ");
  if (chmod(home,mode) == -1)
    strerr_die4sys(111,FATAL,"unable to chmod ",home,": ");
}

void hier_d(const char *home,const char *subdir,num mode)
{
  if (chdir(home) == -1)
    strerr_die4sys(111,FATAL,"unable to switch to ",home,": ");
  if (mkdir(subdir,0700) == -1)
    if (errno != EEXIST)
      strerr_die6sys(111,FATAL,"unable to mkdir ",home,"/",subdir,": ");
  if (chmod(subdir,mode) == -1)
    strerr_die6sys(111,FATAL,"unable to chmod ",home,"/",subdir,": ");
}

static char inbuf_space[INBUF_SIZE];
static char outbuf_space[OUTBUF_SIZE];
static inbuf ssin;
static outbuf ssout;

void hier_c(const char *home,const char *subdir,const char *file,num mode)
{
  num fdin;
  num fdout;

  if (fchdir(fdsourcedir) == -1)
    strerr_die2sys(111,FATAL,"unable to switch back to source directory: ");

  fdin = open_read(file);
  if (fdin == -1)
    strerr_die4sys(111,FATAL,"unable to read ",file,": ");
  inbuf_init(&ssin,inbuf_unixread,fdin,inbuf_space,sizeof inbuf_space);

  if (chdir(home) == -1)
    strerr_die4sys(111,FATAL,"unable to switch to ",home,": ");
  if (chdir(subdir) == -1)
    strerr_die6sys(111,FATAL,"unable to switch to ",home,"/",subdir,": ");

  fdout = open_trunc(file);
  if (fdout == -1)
    strerr_die6sys(111,FATAL,"unable to write .../",subdir,"/",file,": ");
  outbuf_init(&ssout,outbuf_unixwrite,fdout,outbuf_space,sizeof outbuf_space);

  switch(copy(&ssout,&ssin)) {
    case -2:
      strerr_die4sys(111,FATAL,"unable to read ",file,": ");
    case -3:
      strerr_die6sys(111,FATAL,"unable to write .../",subdir,"/",file,": ");
  }

  close(fdin);
  if (outbuf_flush(&ssout) == -1)
    strerr_die6sys(111,FATAL,"unable to write .../",subdir,"/",file,": ");
  if (fsync(fdout) == -1)
    strerr_die6sys(111,FATAL,"unable to write .../",subdir,"/",file,": ");
  if (close(fdout) == -1) /* NFS silliness */
    strerr_die6sys(111,FATAL,"unable to write .../",subdir,"/",file,": ");

  if (chmod(file,mode) == -1)
    strerr_die6sys(111,FATAL,"unable to chmod .../",subdir,"/",file,": ");
}

int main(int argc,char **argv)
{
  fdsourcedir = open_read(".");
  if (fdsourcedir == -1)
    strerr_die2sys(111,FATAL,"unable to open current directory: ");

  umask(077);
  hier();
  _exit(0);
  return 0;
}
