/*
 control.c
 940901
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "config.h"

#ifdef USE_SIGNALS
#include <signal.h>
#include <setjmp.h>
#endif USE_SIGNALS

#include "commands.h"
#define NEED_TABLE
#include "commands.h"

#ifdef USE_UNISTD
#include <unistd.h>
#endif USE_UNISTD

#ifdef USE_READLINE
#include <dirent.h>
#include <readline/readline.h>
#include <readline/history.h>
#endif USE_READLINE

#include <sys/times.h>

#include "chipmunk.h"

#define MAXARGS 28

struct macro {
  struct macro *next;
  struct macline *mac;
  char *name;
} *firstmac = NULL;

struct macline {
  struct macline *next;
  struct macline *prev;
  char *line;
  int miscstuff;
} null_beyond = {NULL, NULL, NULL, 0},
  init = {&null_beyond, NULL, "init", 0};

struct maclev {
  int line_nr;
  int cargc;
  char *cargv[MAXARGS];
  struct macline *toexecute;
} stack[MAXDEPTH];

int maclevel = 0;

#define Line_nr    (stack[maclevel].line_nr)
#define Cargc      (stack[maclevel].cargc) 
#define Cargv      (stack[maclevel].cargv) 
#define Toexecute  (stack[maclevel].toexecute) 

#define Command_is(x) \
  !strncmp (Toexecute->line + strspn (Toexecute->line, PREC), x, strlen (x))


#define PREC " \t\r\n"


FILE *logfile = NULL;
int BASE    = 0;
int IOBASE  = 0;
int VERBOSE = 1;
int TRACE   = 0;
#ifdef USE_CHECK
int CHECK   = 0;
#endif USE_CHECK

#ifdef USE_CBREAK
                                                                  /* getkey */
int getkey (void) {
  char t;

  system ("stty cbreak");
  read (0, &t, 1);
  system ("stty -cbreak");
  return (t);
}
#endif USE_CBREAK


struct var {
struct var *next;
char *name;
char *val;
};

char statstring[]="0x00000000";
struct var statusvar = { NULL, "STATUS", statstring};
struct var *firstvar = &statusvar;

FILE *fin;
int batch;


#ifdef USE_BOGUS_USLEEP
                                                                  /* usleep */
int usleep (int t) {
  for (t *= PROCESSOR_SPEED; t; t--);
  return (0);
}

#endif USE_BOGUS_USLEEP
                                                                /* my_sscan */
int my_sscan (char *line, char **argv) {
  int argc, t;
  char *p, quote;

  t = 0;
  p = line;
  argc = 0;
  for ( ; *p; p++) {
    switch (*p) {
    case '\"': case '\'': case '`':
      quote = *p;
      for (p++; *p && *p != quote; p++)
        argv[argc][t++] = *p;
      if (!*p) {
        char msg [80];
        sprintf (msg, "unmatched %c", quote);
        do_abort (msg);
        return (-1);
      }
      break;
    case ' ': case '\r': case '\t': case '\n':
      if (t) {
        argv[argc++][t] = '\0';
        t = 0;
      } /* else eat space */
      break;
    case '(': case ')':
      if (t)
        argv[argc++][t] = '\0';
      t = 0;
      argv[argc  ][0] = *p;
      argv[argc++][1] = '\0';
      break;
    case '\\':
      p++;
      if (! *p) {
        p--;
        break;
      }
    default:
      argv[argc][t++] = *p;
      break;
    }
    if (argc > MAXARGS) {
      do_abort ("too many arguments");
      return (-1);
    }
  }
  if (t)
    argv[argc++][t] = '\0';
  if (argc > MAXARGS) {
    do_abort ("too many arguments");
    return (-1);
  }
  return (argc);
}


                                                             /*save_string  */
char *save_string (char *s) {
  char *t;

  t = MY_MALLOC (strlen (s) + 1);
  strcpy (t, s);
  return (t);
}


                                                                 /* myatoi  */
unsigned int scanint (char *str, unsigned int *t) {
  char *fmt = "%x";

  if ((str[0] == '0') && (str[1] == 'x')) {            str += 2;}
  if ((str[0] == '#')                   ) {fmt = "%d"; str += 1;}
  return (sscanf (str, fmt, t));
}


unsigned int myatoi (char *str) {
  unsigned int t;

  if (scanint (str, &t) != 1)
    return (0);
  return (t);
}



                                                                /* findvar  */
struct var *findvar (char *name) {
  struct var *vp;

  for (vp = firstvar; vp != NULL; vp = vp->next)
    if (strcmp (vp->name, name) == 0)
      return (vp);
  return (vp);
}


                                                              /* getintvar  */
int getintvar (char *name) {
  struct var *vp;

  if ((vp = findvar (name)) == NULL)
    return (0);
  return (myatoi (vp->val));
}

                                                              /* tstintvar  */
int tstenv (int argc, char **argv) {
  struct var *vp;
  int i, nf = 0, length = 100;

  for (i = 1; (i < argc) & !nf; i++) {
    if (isdigit (argv[i][0])) {
      if (maclevel > 0) {
        if (stack[maclevel-1].cargc <= (argv[i][0] - '0'))
          nf = 1;
      }else
        nf = 1;
    }else {
      if (argv[i][strlen (argv[i]) - 1] == '*')
        length = strlen (argv[i]) - 1 ;
      for (vp = firstvar; vp != NULL; vp = vp->next)
        if (strncmp (vp->name, argv[i], length) == 0)
          break;
      if (vp == NULL)
        nf = 1;
    }
  }
  if ((VERBOSE > 0) && !maclevel)
    printf ("%s environment.\n", nf ? "Not in" : "In");
  return (!nf);
}
                                                                   /* quit  */
int quit (int argc, char **argv) {
  if (logfile) fclose (logfile);
  if (VERBOSE > 0)
    printf ("bye bye.\n");
  if (argc > 1)
    exit (myatoi (argv[1]));
  else
    exit (0);
  /* Compilers that complain about no return here are broken. */
}

                                                                   /* help  */
int help (int argc, char **argv) {
  int i, j, n = 0, length = 100;

  if (argc<2)
    for (i = 0; i < NBUILTINS; i++) {
      printf ("%-10s-- %s\n", builtins[i].name, builtins[i].help);
      n++;
    }
  else 
    for (j = 1; j < argc; j++) {
      if (argv[j][strlen (argv[j]) - 1] == '*')
        length = strlen (argv[j]) - 1 ;
      for (i = 0; i < NBUILTINS; i++)
      if (strncmp (argv[j], builtins[i].name, length) == 0) {
        printf ("%-10s-- %s\n", builtins[i].name, builtins[i].help);
        printf ("             > %s %s\n", builtins[i].name, builtins[i].usage);
        n++;
      }
    }
  if (!n)
    printf ("Not found, type 'help' for a list of all commands.\n");
  return (n);
}


extern int major, minor, subminor;
                                                                    /* ver */
int ver (int argc, char **argv) {
  printf ("This is Chipmunk version %d.%d.%d .\n", major, minor, subminor);
  return (major);/* included for compatability */
}


int pmt (int argc, char **argv) {
  int a, n, l, i, e;
  register int s;
  register volatile short *p, val0, val1, *pe, t;

  val0 = 0;
  val1 = ~val0;
  MINARGC (4);
  a = BASE + myatoi (argv[1]);
  l = myatoi (argv[2]);
  s = myatoi (argv[3]);
  n = myatoi (argv[4]);
  e = 0;
  for (i = 0; i < n; i++) {
    if (e) {printf ("%04x %d\n", t, e); e = 0;}
    (void) *(volatile short *)(BASE + 0x10000);
    p = (short *) a;
    pe = p + s * l;
    for ( ; p < pe; p += s) {
      t = *p;
      if (t != val1) {
        (void) *(volatile short *)(BASE + 0x10000);
        e++;
      }
      *p = val0;
    }
    val0 = ~val0;
    val1 = ~val0;
  }
  return (0);
}


                                                                  /* macdef */
int macdef (int argc, char **argv) {
  struct macro *mp;
  struct macline *cmlp = NULL, *prev = NULL;
  char buf[1000], *t;
#ifdef USE_READLINE
  static char *rls;

/*   initialize_readline (); */
  rls = NULL;
#endif USE_READLINE

  MINARGC (2);
  for (mp = firstmac; mp != NULL; mp = mp ->next)
    if (strcmp (argv[1], mp->name) == 0) break;
  if (mp) macdel (argc, argv);
  mp = MY_MALLOC (sizeof (struct macro));
  mp->mac = NULL;
  mp->name = MY_MALLOC (strlen (argv[1]) + 1);
  strcpy (mp->name, argv[1]);
  do {
    if (fin == stdin) {
#ifdef USE_READLINE

      if (rls) free (rls);
      rls =  readline ("macdef> ");
      t = NULL;
      if (rls && (*rls)) {
        add_history (rls);
        strcpy (buf, rls);
        t = rls;
      }
#else
      printf ("macdef> ");
      t = fgets (buf, 1000, fin);
#endif USE_READLINE
    }
    else
      t = fgets (buf, 1000, fin);
    if (t == NULL) {
      buf[0] = '\n';
      buf[1] = '\0';
    }
    if (cmlp == NULL)
      cmlp = mp->mac = MY_MALLOC (sizeof (struct macline));
    else
      cmlp = cmlp->next = MY_MALLOC (sizeof (struct macline));
    cmlp->line = save_string (buf);
    cmlp->prev = prev;
    cmlp->next = NULL;
    if ((t == NULL) || ((fin == stdin) && ((strcmp (buf, "\n") == 0))))
      break;
    prev = cmlp;
  } while (1);
  mp->next = firstmac;
  firstmac = mp;
  return (0);
}

                                                                  /* macdel */
int macdel (int argc, char **argv) {
  struct macro *mp, *pp;
  struct macline *ml, *nml;
  int i;
  char *p;

  MINARGC (2);

  for (i = 1; i < argc; i++) {
    pp = NULL;
    for (mp = firstmac; mp != NULL; pp = mp, mp = mp ->next)
      if (strcmp (argv[i], mp->name) == 0) break;
    if (mp == NULL) {
      p = malloc (80);
      sprintf (p, "couldn't find macro `%s'", argv[i]);
      do_abort (p);
      free (p);
      return (-1);
    }
    if (pp)
      pp->next = mp->next;
    else
      firstmac = mp->next;
    
    for (ml = mp->mac; ml != NULL; ml = nml) {
      nml = ml->next;
      free (ml->line);
      free (ml);
    }
    free (mp->name);
    free (mp);
  }
  return (0);
}

                                                                  /* macro */
int macro (int argc, char **argv) {
  struct macro *mp;
  struct macline *cmlp = NULL, *prev = NULL;

  if (maclevel == 0) {
    do_abort ("use macdef instead on command line");
    return (1);
  }
  MINARGC (2);
  for (mp = firstmac; mp != NULL; mp = mp ->next)
    if (strcmp (argv[1], mp->name) == 0) break;
  if (mp)
    macdel (argc, argv);
  mp = MY_MALLOC (sizeof (struct macro));
  mp->mac = NULL;
  mp->name = MY_MALLOC (strlen (argv[1]) + 1);
  strcpy (mp->name, argv[1]);
  if (VERBOSE > 2)
    printf ("Defining macro %s.\n", mp->name);
  do {
    if (cmlp == NULL)
      cmlp = mp->mac = MY_MALLOC (sizeof (struct macline));
    else
      cmlp = cmlp->next = MY_MALLOC (sizeof (struct macline));

    cmlp->prev = prev;
    cmlp->next = NULL;
    if (Command_is ("endmac") ) {
      cmlp->line = save_string ("\n");
      break;
    }
    else
      cmlp->line = save_string (Toexecute->line);
    if ((Toexecute->next == NULL))
      break;
    prev = cmlp;
    Toexecute = Toexecute->next;
  } while (1);
  mp->next = firstmac;
  firstmac = mp;
  return (0);
}

                                                             /* do_endwhile */
int do_endmac (int argc, char **argv) {
  if (maclevel == 0) {
    do_abort ("not implemented from commandline");
    return (1);
  }
  return (0);
}


                                                             /* recurse_mac */
int recurse_mac (struct macro *mp) {
  struct macline *ml;
  int nl = 0;

  if (mp == NULL) return (0);
  if (mp->next != NULL)
    recurse_mac (mp->next);
  for (nl = 0, ml = mp->mac; ml != NULL; ml = ml->next, nl++);
  printf (" %-15s %3d lines\n", mp->name, nl);
  return (0);
}


                                                                  /* shomac */
int shomac (int argc, char **argv) {
  struct macro *mp;
  struct macline *ml;
  int length = 100, i;

  if (argc>1) {
    for (i = 1; i < argc; i++) {
      if (argv[i][strlen (argv[i])-1] == '*')
        length = strlen (argv[i]) - 1 ;
      for (mp = firstmac; mp != NULL; mp = mp->next) {
        if (strncmp (argv[i], mp->name, length) == 0) {
          printf ("\nMacro '%s':\n", mp->name);
          for (ml = mp->mac; ml != NULL; ml = ml->next)
            printf ("  %s", ml->line);
        }
      }
    }
  }
  else{
    if (firstmac) printf ("Macro's:\n");
    else printf ("No macro's defined.\n");
    recurse_mac (firstmac);
  }

  return (0);
}

                                                                  /* unset */
int unset (int argc, char **argv) {
  struct var *vp, *pp;
  int i = 1;
  char *p;

  MINARGC (2);

  for (i = 1; i < argc; i++) {
    if (strcmp ("STATUS", argv[i]) == 0) {
      do_abort ("can't unset variable STATUS");
      return (-1);
    }
    pp = NULL;
    for (vp = firstvar; vp != NULL; pp = vp, vp = vp->next)
      if (strcmp (argv[i], vp->name) == 0) break;
    if (vp == NULL) {
      p = malloc (80);
      sprintf (p, "couldn't find variable `%s'", argv[i]);
      do_abort (p);
      free (p);
      return (-1);
    }
    if (pp)
      pp->next = vp->next;
    else
      firstvar = vp->next;
    
    free (vp->val);
    free (vp->name);
    free (vp);
  }
  return (0);
}


                                                                     /* set */
int set (int argc, char **argv) {
  struct var *vp;
  int i, j, totlen, set, concat = 0;
  unsigned int value = 0;
  char val[20]="";

  MINARGC (2);

  if (strcmp ("STATUS", argv[1]) == 0) {
    do_abort ("can't set variable STATUS");
    return (1);
  }

  vp = findvar (argv[1]);
  if (vp == NULL) {
    vp = MY_MALLOC (sizeof (struct var));
    vp->name = MY_MALLOC (strlen (argv[1]) + 1);
    strcpy (vp->name, argv[1]);
    vp->next = firstvar;
    firstvar = vp;
  }
  else {
    value = myatoi (vp->val);
    free (vp->val);
  }
  set = 0;
  if (!set && (argc > 2)) { /* look for: set var op */
    set = 1;
    if      (strcmp (argv[2], "++" ) == 0) value ++;
    else if (strcmp (argv[2], "--" ) == 0) value --;
    else if (strcmp (argv[2], "~~" ) == 0) value = ~ value;
    else set --;
    if (set) { 
      MAXARGC(4);
      sprintf (val, "%x", value);
      vp->val = save_string (val);
    } 
  }
  if (!set && (argc > 3)) { /* look for: set var op val */
    set = 1;
    if      (strcmp (argv[2], "+=" ) == 0) value += myatoi(argv[3]);
    else if (strcmp (argv[2], "-=" ) == 0) value -= myatoi(argv[3]);
    else if (strcmp (argv[2], "*=" ) == 0) value *= myatoi(argv[3]);
    else if (strcmp (argv[2], "%=" ) == 0) value %= myatoi(argv[3]);
    else if (strcmp (argv[2], "/=" ) == 0) value /= myatoi(argv[3]);
    else if (strcmp (argv[2], "|=" ) == 0) value |= myatoi(argv[3]);
    else if (strcmp (argv[2], "&=" ) == 0) value &= myatoi(argv[3]);
    else if (strcmp (argv[2], "^=" ) == 0) value ^= myatoi(argv[3]);
    else if (strcmp (argv[2], ">>=") == 0) value >>= myatoi(argv[3]);
    else if (strcmp (argv[2], "<<=") == 0) value <<= myatoi(argv[3]);
    else if (strcmp (argv[2], "||=") == 0) value = value || myatoi(argv[3]);
    else if (strcmp (argv[2], "&&=") == 0) value = value && myatoi(argv[3]);
    else set --;
    if (set) { 
      MAXARGC(5);
      sprintf (val, "%x", value);
      vp->val = save_string (val);
    }
  }
  if (!set) {
    set = 1;
    if (strcmp (argv[2], "="  ) == 0) {
      j = 3;
      concat = 1;
    }
    else 
      j = 2;
    for (i = j, totlen = 0; i < argc; i++)
      totlen += strlen (argv[i]) + 1;
    vp->val = MY_MALLOC (totlen + 1);
    strcpy (vp->val, argv[j++]);
    for (totlen = 0; j < argc; j++) {
      if (!concat)
        strcat (vp->val, " ");
      strcat (vp->val, argv[j]);
    }
  }
  /* Maintain internal copies of handy variables: */
  BASE    = getintvar ("BASE");
  IOBASE  = getintvar ("IOBASE");
  VERBOSE = getintvar ("VERBOSE");
  TRACE   = getintvar ("TRACE");
#ifdef USE_CHECK
  CHECK   = getintvar ("CHECK");
#endif USE_CHECK
  return (0);
}


                                                            /* internal_set */
int internal_set (char *varname, char *value) {
  struct var *vp;

  vp = findvar (varname);
  if (vp == NULL) {
    vp = MY_MALLOC (sizeof (struct var));
    vp->name = MY_MALLOC (strlen (varname) + 1);
    strcpy (vp->name, varname);
    vp->next = firstvar;
    firstvar = vp;
  }
  else
    free (vp->val);
  vp->val = MY_MALLOC (strlen (value) + 1);
  strcpy (vp->val, value);
  return (0);
}

                                                        /* internal_set_int */
int internal_set_int (char *varname, int value) {
  char strval[80];
  sprintf (strval, "%x", value);
  internal_set (varname, strval);
  return (0);
}
                                                                 /* copyenv */
int copyenv (int argc, char **argv) {
  char *cargv[3];

  if ((cargv[2] = getenv (argv[1]))) {
    cargv[0] = "set";
    cargv[1] = argv[1];
    set (3, cargv);
    return (0);
  }
  else 
    return (1);
}

                                                                    /* echo */
int echo (int argc, char **argv) {
  int i, nonl, n = 0;

  nonl = (strcmp (argv[1], "-n") == 0);
  for (i = 1 + nonl; i < argc - 1; i++)
    if (*argv[i] != '\0') {
      if (VERBOSE >= 0)
        printf ("%s ", argv[i]);
      n +=  strlen (argv[i]) + 1;
    }
  if (argc-1) {
    if (VERBOSE >= 0)
      printf ("%s", argv[argc-1]);
    n += strlen (argv[argc-1]);
  }
  if (VERBOSE >= 0)
    if (nonl) fflush (stdout);
    else printf ("\n");
  return (n);
}

                                                                 /* do_putc */
int do_putc (int argc, char **argv) {
  int i;

  if (VERBOSE >= 0) {
    for (i = 1; i < argc; i++)
      putchar (myatoi (argv[i]));
    fflush (stdout);
  }
  return (argc - 1);
}

                                                             /* recurse_env */
void recurse_env (struct var *vp, char *name, int length) {
  if (vp->next != NULL)
    recurse_env (vp->next, name, length);

  if (strncmp (name, vp->name, length) == 0) {
    printf ("%10s",  vp->name);
    if (*vp->val == '\0') printf ("\n");
    else printf (" = %s\n", vp->val);
  }
}


                                                                     /* env */
int env (int argc, char **argv) {
  int j, length = 100;

  if (argc<2)
    recurse_env (firstvar, "*", 0);
  else
    for (j = 1; j < argc; j++) {
      if (argv[j][strlen (argv[j]) - 1] == '*')
        length = strlen (argv[j]) - 1 ;
      recurse_env (firstvar, argv[j], length);
    }

  return (0);
}


                                                                /* autoload */
int autoload (int argc, char **argv) {
  char buf[80],line[80];
  FILE *f, *tf;
  char *cargv[2];

  sprintf (buf, "%s.mac", argv[0]);
  f = fopen (buf, "r");
  if (f == NULL) {
    sprintf (buf, "%s", argv[0]);
    f = fopen (buf,"r");
    if (f == NULL) return (0);
    fgets (line,80,f) ;
    if (strncmp (line,"#!",2) != 0) return (0);
  }
  if ((VERBOSE > 0) && !maclevel) printf ("Autoloading %s.\n", buf);
  tf = fin;
  fin = f;
#ifdef USE_ALLOCA
  cargv[1] = alloca (strlen (argv[0]) + 1);
#else
  cargv[1] = MY_ALLOCA (strlen (argv[0]) + 1);
#endif USE_ALLOCA
  strcpy (cargv[1], argv[0]);
  macdef (2, cargv);
  fclose (f);
  fin = tf;
#ifndef USE_ALLOCA
  free (cargv[1]);
#endif USE_ALLOCA
  return (1);
}

                                                                  /* argpos */
int argpos (int argc, char **argv, char *s) {
/* returns number of last argument which is s */
  int i;
  for (i = 0; i < argc; i++)
    if (strcmp (argv[i], s) == 0) return (i);
  return (-1);
}

                                                                 /* argrpos */
int argrpos (int argc, char **argv, char *s) {
/* returns number of last argument which is s */
  int i;
  for (i = argc - 1; i >= 0; i--)
    if (strcmp (argv[i], s) == 0) return (i);
  return (-1);
}

                                                                 /* evalexp */
int evalexp (int argc, char **argv) {
  unsigned int a, b, result;
  int end, i, j;
  char *op;

  /* unary operators */
  end = argpos (argc, argv, ")");
  if (end == -1)
    end = argc;
  for (i = end - 2; i > 0; i--) {
    op =         argv[i  ];
    a  = myatoi (argv[i+1]);
    if      (strcmp (op, "!" ) == 0) result =!a;
    else if (strcmp (op, "~" ) == 0) result =~a;
    else if (strcmp (op, "%d") == 0) {sprintf (argv[i+1], "%d", a);
                                     result = myatoi (argv[i+1]);}
    else if (strcmp (op, "%c") == 0) result = (int)*argv[i+1];
#ifdef USE_CHECK
    else if ( (*op == '@') && CHECK) result = 0;
#endif USE_CHECK
    else if (strcmp (op, "@" ) == 0) result =*(unsigned int   *)(BASE+a);
    else if (strcmp (op, "@s") == 0) result =*(unsigned short *)(BASE+a);
    else if (strcmp (op, "@b") == 0) result =*(unsigned char  *)(BASE+a);
    else continue;
    sprintf (argv[i], "%x", result);
    argc -= 1;
    end  -= 1;
    for (j = i + 1; j < argc; j++)
      strcpy (argv[j], argv[j+1]);
    i++;
  }
  /* binary operators */
  for (i = 1; i < end - 2; i++) {
    a  = myatoi (argv[i]);
    op =         argv[i+1];
    b  = myatoi (argv[i+2]);
    if      (strcmp (op, "+" ) == 0) result = a +  b;
    else if (strcmp (op, "-" ) == 0) result = a -  b;
    else if (strcmp (op, "/" ) == 0) result = a /  b;
    else if (strcmp (op, "%" ) == 0) result = a %  b;
    else if (strcmp (op, "*" ) == 0) result = a *  b;
    else if (strcmp (op, "&" ) == 0) result = a &  b;
    else if (strcmp (op, "|" ) == 0) result = a |  b;
    else if (strcmp (op, "^" ) == 0) result = a ^  b;
    else if (strcmp (op, ">>") == 0) result = a >> b;
    else if (strcmp (op, "<<") == 0) result = a << b;
    else if (strcmp (op, "&&") == 0) result = a && b;
    else if (strcmp (op, "||") == 0) result = a || b;
    else if (strcmp (op, "==") == 0) result = a == b;
    else if (strcmp (op, "=s=")== 0) result = !strcmp (argv[i], argv[i+2]);
    else if (strcmp (op, "!=") == 0) result = a != b;
    else if (strcmp (op, "<" ) == 0) result = a <  b;
    else if (strcmp (op, ">" ) == 0) result = a >  b;
    else if (strcmp (op, "<=") == 0) result = a <= b;
    else if (strcmp (op, ">=") == 0) result = a >= b;
    else continue;
    sprintf (argv[i], "%x", result);
    argc -= 2;
    end  -= 2;
    for (j = i + 1; j < argc; j++)
      strcpy (argv[j], argv[j+2]);
    i--;
  }

  return (argc);
}

                                                        /* evalexp_grouping */
int evalexp_grouping (int argc, char **argv) {
  int br_open, br_close, i, argrem;

  while ((br_open = argrpos (argc, argv, "(")) != -1) {
    br_close = argpos (argc - br_open, argv + br_open, ")");
    if (br_close == -1) {
#if 0
      do_abort ("Too many ('s.\n");
      return (-1);
#else
      break;
#endif
    }
    argrem = (argc - br_open) - evalexp ((argc - br_open), argv + br_open);
    br_close += br_open - argrem;
    argc -= argrem;
    argc -= 1;
    for (i = br_close; i < argc; i++)
      strcpy (argv[i], argv[i+1]);
    argc -= 1;
    for (i = br_open; i < argc; i++)
     strcpy (argv[i], argv[i+1]);
  }

  br_close = argpos (argc, argv, ")");

#if 0
  if (br_close != -1) {
    do_abort ("Too many )'s.\n");
    return (-1);
  }
#endif

  argc = evalexp (argc, argv); 
  return (argc);
}




                                                                 /* skipfwd */
int skipfwd (char *incer, char *string, char *decer) {
  int level = 1;

  while ( level && Toexecute && Toexecute->line) {
    if (Command_is (string) && (level == 1))  level--;
    else
      if (Command_is (decer)) level--;
      else
        if (Command_is (incer))  level++;
    if (TRACE > 1)
      printf ("%-8.8s %d %-3d  %s", stack[maclevel-1].cargv[0], maclevel,
             Line_nr, Toexecute->line);
    Toexecute = Toexecute->next;
    Line_nr++;
  }
  return (Toexecute != NULL);
}


                                                                 /* skipbwd */
int skipbwd (char *incer, char *string, char *decer) {
  int level = 1;

  if (!Toexecute) {
     do_abort ("Sorry Not implemented yet.\n");
     return (0);
  }
  Toexecute = Toexecute->prev;
  Line_nr--;
  while (Toexecute && level) {
    Toexecute = Toexecute->prev;
    Line_nr--;
    if (!Toexecute) break;
    if (Command_is (string) && (level == 1))  level--;
    else
      if (Command_is (decer)) level--;
      else
        if (Command_is (incer)) level++;
    if ((TRACE > 1) && level)
      printf ("%-8.8s %d %-3d  %s", stack[maclevel-1].cargv[0], maclevel,
             Line_nr, Toexecute->line);
  }
  return (Toexecute != NULL);
}


                                                                  /* do_if  */
int do_if (int argc, char **argv) {
  unsigned int t;
  if (maclevel == 0) {
    do_abort ("not implemented from commandline");
    return (1);
  }
  if (!argc || (*argv[1] == '\0') || ((scanint (argv[1], &t) == 1) && (!t)))
    if (!skipfwd ("if", "else", "endif"))
      do_abort ("if without else or endif");
  return (getintvar ("STATUS"));
}

                                                                   /* do_if */
int do_while (int argc, char **argv) {
  if (maclevel == 0) {
    do_abort ("not implemented from commandline");
    return (1);
  }
  if (!myatoi (argv[1]))
    if (!skipfwd ("while", "endwhile", "endwhile"))
      do_abort ("while without endwhile");
  return (getintvar ("STATUS"));
}


                                                                /* do_else  */
int do_else (int argc, char **argv) {
  if (maclevel == 0) {
    do_abort ("not implemented from commandline");
    return (1);
  }
  if (!skipfwd ("if", "else", "endif"))
    do_abort ("if...else without endif");
  return (getintvar ("STATUS"));
}


                                                                /* do_endif */
int do_endif (int argc, char **argv) {
  if (maclevel == 0) {
    do_abort ("not implemented from commandline");
    return (1);
  }
  return (getintvar ("STATUS"));
}


                                                             /* do_endwhile */
int do_endwhile (int argc, char **argv) {
  unsigned int line;
  if (maclevel == 0) {
    do_abort ("not implemented from commandline");
    return (1);
  }
  line = Line_nr - 1;
  if (!skipbwd ("endwhile", "while", "while"))
    do_abort ("endwhile without while");
  return (getintvar ("STATUS"));
}


                                                                   /* shift */
int shift (int argc, char **argv) {
  int i;

  if (maclevel <= 0) {
    do_abort ("not implemented from commandline");
    return (-1);
  }

  for (i = 2; i < stack[maclevel-1].cargc; i++)
    strcpy (stack[maclevel-1].cargv[i - 1], stack[maclevel-1].cargv[i]);

  if (i == 2)
    return (1);

  stack[maclevel-1].cargc--;

  return (0);
}

                                                              /* substvars  */

int substvars (int argc, char **argv) {
  int i;
  char *substr;
  struct var *vp;
  int na, j;
  char *cp, *dp;

  for (i = 0; i < argc; i++) {
    if (argv[i][0] == '$') {
      if (isdigit (argv[i][1])) {
        if (maclevel <= 0) {
          do_abort ("can't do macro arg substitution on commandline");
          return (-1);
        }
        if (stack[maclevel-1].cargc > (argv[i][1] - '0'))
          substr = stack[maclevel-1].cargv[argv[i][1] - '0'];
        else{
          substr = "\0";
        }
      }
      else {
        j = 1;
        if (argv[i][1] == '$')
          j = 2;
        vp = findvar (argv[i]+j);
        if (vp == NULL) {
          substr = malloc (80);
          sprintf (substr,"undefined variable `%s'", argv[i]+j);
          do_abort (substr);
          free (substr);
          return (-1);
        }
        substr = vp->val;
      }
      if (argv[i][1] == '$') {
        for (cp = substr, na = 0; *cp != '\0'; cp++)
          if (*cp == ' ') na++;
        argc += na;
        if (argc > MAXARGS) {
          do_abort ("too many arguments");
          return (-1);
        }
        if (na) 
          for (j = argc; j > i; j--)
            strcpy (argv[j], argv[j - na]);
        for (j = i, cp = substr, dp = argv[j]; *cp != '\0'; cp++) {
          if (*cp != ' ') {
            *dp++ = *cp;
          }
          else{
            *dp = '\0';
            dp = argv[++j];
          }
        }
        *dp++ = '\0';
      }
      else
        strcpy (argv[i], substr);
    }
    else /* dec to hex conversion by sybstvar */
      if (argv[i][0] == '#')
        sprintf (argv[i], "%x", atoi (argv[i] + 1));
        /* warning, ^^this works because argv[1] is static allocated */
  }
  return (argc);
}


                                                                   /* syst  */
int syst (int argc, char **argv) {
  char *cmd;
  int return_val = 0;

  if (argc>1) {
    char buf[1024];
    int i;

    buf [0] = '\0';
    for (i = 1; i < argc; i++) {
      if (i != 1) strcat (buf, " ");
      strcat (buf, argv[i]);
    }
    return_val = system (buf);
  }
  else {
    cmd = getenv ("SHELL");
    if (cmd) 
      return_val = system (cmd);
    else {
      do_abort ("SHELL environment variable not set");
      return (-1);
    }
  }

  return (return_val);
}


                                                                   /* uslp  */
int uslp (int argc, char **argv) {
  int n;

  MINARGC (2);
  n = myatoi (argv[1]);
  usleep (n);
  return (0);
}

                                                               /* do_abort  */
int do_abort (char *msg) {
  if (maclevel > 0) {
    if (strncmp (msg, "normal", 6) == 0)
      maclevel = 0;
    else {
      fprintf (stderr, "At %s line %d: %s.\n", 
               stack[maclevel-1].cargv[0], Line_nr - 1, msg?msg:"abort");
      while (maclevel > 0) {
        fprintf (stderr, "Aborting macro %s at line %d.\n",
                 stack[maclevel-1].cargv[0], Line_nr - 1);
        maclevel--;
      }
    }
  }
  else
    fprintf (stderr, "ERR: %s.\n", msg?msg:"unknown error");
 
  return 0;
}

                                                                 /* mabort  */
int mabort (int argc, char **argv) {
  if (myatoi (argv[1]) == 0)
    return (do_abort ("normal exit"));
  else
    return (do_abort ("mabort encountered"));
}


                                                               /* doreturn  */
int doreturn (int argc, char **argv) {
  int retval = 0;

  if (argc) retval = myatoi (argv[1]);
  Toexecute = &null_beyond;
  if (maclevel == 0) exit (retval);
  return (retval);
}


                                                                   /* ask   */
int ask (int argc, char **argv) {
  int keycode;

  if (argc<2) printf ("Hit any key to continue ");
  else echo (argc, argv);
  fflush (stdout);
  keycode = tolower (getkey ());/* chnged to tolower. Rob */
  printf ("\n");
  return (keycode);
}


int timer (int argc, char **argv) {
  static struct tms last, cur;

  if (strcmp (argv[1], "reset") == 0)
      times (&last);
  else
      times (&cur);
  return ((cur.tms_utime + cur.tms_stime) - (last.tms_utime + last.tms_stime));
}
    
                                                                 /* savmac  */
int savmac (int argc, char **argv) {
  struct macro *mp;
  struct macline *ml;
  FILE *f;
  char buf[80], msg[80];


  MINARGC (2);
  for (mp = firstmac; mp && strcmp (argv[1], mp->name); mp = mp->next);
  if (mp) {
    sprintf (buf, "%s.mac", mp->name);
    printf ("Saving macro %s.\n", mp->name);
    if ((f = fopen (buf, "w")) == NULL) {
      sprintf (msg, "cannot open file `%s': %s", buf, strerror(errno));
      do_abort (msg);
      return (1);
    }
    for (ml = mp->mac; ml != NULL; ml = ml->next)
      fprintf (f, "%s", ml->line);
    fclose (f);
    return (0);
  }
  else{
    do_abort ("macro does not exist");
    return (1);
  }
}



#ifdef USE_READLINE

                                                    /*  initialize_readline */
void initialize_readline (void) {
  char *command_generator ();
  char **completion ();
  char *dummy_filename_completion_function ();

  /*  rl_basic_word_break_characters = " \t\n\"\\'`@><="; */

  rl_readline_name = "chipmunk";
  rl_completion_entry_function=(Function *)dummy_filename_completion_function;
  rl_attempted_completion_function = (Function *)completion;
}


                                   /*  *dummy_filename_completion_function  */
char *dummy_filename_completion_function (int state, char *text) {
  return (NULL);
}


                                                            /*  completion  */
char **completion (char *text, int start, int end) {
  char **matches;
  char * command_name_generator (char *text, int state);
  char *variable_name_generator (char *text, int state);
  char *   macro_name_generator (char *text, int state);
  char *    file_name_generator (char *text, int state);
  char cmd[100];

  matches = (char **) NULL;

  if (start == 0)
    matches = completion_matches (text, command_name_generator);

  if ((start > 0) && (rl_line_buffer[start-1] == '$'))
    matches = completion_matches (text, variable_name_generator);

  sscanf (rl_line_buffer, "%s", cmd);
  if ((strcmp (cmd, "set"    ) == 0) ||
      (strcmp (cmd, "unset"  ) == 0) ||
      (strcmp (cmd, "env"    ) == 0) ||
      (strcmp (cmd, "tstenv" ) == 0))
    matches = completion_matches (text, variable_name_generator);

  if ((strcmp (cmd, "help"   ) == 0) ||
      (strcmp (cmd, "?"      ) == 0))
    matches = completion_matches (text, command_name_generator);

  if ((strcmp (cmd, "shomac" ) == 0) ||
      (strcmp (cmd, "macdel" ) == 0) ||
      (strcmp (cmd, "savmac" ) == 0))
    matches = completion_matches (text, macro_name_generator);

  if ((strncmp(cmd, "save", 4) == 0) ||
      (strcmp (cmd, "syst"   ) == 0) ||
      (strncmp(cmd, "load", 4) == 0))
    matches = completion_matches (text, file_name_generator);

  return (matches);
}


                                               /*   command_name_generator  */
char * command_name_generator (char *text, int state) {
  static int cmd_index, len;
  static struct macro *mmp;
  static DIR *directory;
  char *nam;
  struct dirent *entry = (struct dirent *)NULL;
  char *tt;
  int dnamlen;



  if (!state) {
    cmd_index = 0;
    len = strlen (text);
    mmp = firstmac;
    directory = opendir (".");
  }
  while ((cmd_index < NBUILTINS) && (nam = builtins[cmd_index].name)) {
    cmd_index++;
    if (strncmp (nam , text, len) == 0)
      return (save_string (nam));
  }
  while ((mmp != NULL) && (nam = mmp->name)) {
    mmp = mmp->next;
    if (strncmp (nam, text, len) == 0) {
      return (save_string (nam));
    }
  }
  while (directory && (entry = readdir (directory))) {
    dnamlen = strlen (entry->d_name);
    if ((strncmp (entry->d_name, text, len) == 0) &&
        (strcmp (entry->d_name + (dnamlen - 4), ".mac") == 0)) {
      tt = MY_MALLOC (dnamlen + 1);
      strcpy (tt, entry->d_name);
      tt[dnamlen-4] = '\0';
      return (tt);
    }
  }
  if (directory)
    closedir (directory);
  return (char *) NULL;
}




                                               /*   file_name_generator  */
char * file_name_generator (char *text, int state) {
  static int len;
  static DIR *directory;
  struct dirent *entry = (struct dirent *)NULL;

  if (!state) {
    len = strlen (text);
    directory = opendir (".");
  }
  while (directory && (entry = readdir (directory))) {
    if (strncmp (entry->d_name, text, len) == 0) {
      return (save_string(entry->d_name));
    }
  }
  if (directory)
    closedir (directory);
  return (char *) NULL;
}


                                               /*  variable_name_generator  */
char *variable_name_generator (char *text, int state) {
  static struct var *vp;
  static int len;
  char *nam;

  if (!state) {
    len = strlen (text);
    vp = firstvar;
  }
  while ((vp != NULL) && (nam = vp->name)) {
    vp = vp->next;
    if (strncmp (nam, text, len) == 0)
      return (save_string (nam));
  }
  return (char *) NULL;
}

                                                  /*  macro_name_generator  */
char *macro_name_generator (char *text, int state) {
  static struct macro *mp;
  static int len;
  char *nam;

  if (!state) {
    len = strlen (text);
    mp = firstmac;
  }
  while ((mp != NULL) && (nam = mp->name)) {
    mp = mp->next;
    if (strncmp (nam, text, len) == 0)
      return (save_string (nam));
  }
  return (char *) NULL;
}

#endif USE_READLINE

#ifdef USE_SIGNALS

jmp_buf jmp_env;

void intsig (void) {
  printf ("INT signal recieved.\n");
#ifdef SYSV_SIGNALS
  signal (SIGINT,  SIGNAL_FUNCTION_CAST intsig);
#endif SYSV_SIGNALS
  longjmp (jmp_env, 3);
}

void bussig (void) {
  printf ("Buserror signal recieved.\n");
#ifdef SYSV_SIGNALS
  signal (SIGBUS,  SIGNAL_FUNCTION_CAST bussig);
#endif SYSV_SIGNALS
  longjmp (jmp_env, 2);
}

void segvsig (void) {
  printf ("segv signal recieved.\n");
#ifdef SYSV_SIGNALS
  signal (SIGSEGV, SIGNAL_FUNCTION_CAST segvsig);
#endif SYSV_SIGNALS
  longjmp (jmp_env, 1);
}

#endif USE_SIGNALS


int eval_expression (int argc, char **argv) {
  int i, ok, rn;
  struct macro *mp;
  char msg[80];

  rn = -1;
  /* Macro arg substitution */
  /* Variable substitution */
  argc = substvars (argc, argv);
  if (argc < 1) return (argc);
  /* Expression evaluation? */
  argc = evalexp_grouping (argc, argv);
  if (argc < 1) return (argc);
  if (TRACE < 0) {
    if (maclevel == 0) printf (": "); else
      printf ("%-8.8s.%d.%-3d:", stack[maclevel-1].cargv[0],
              maclevel, Line_nr-1);
    printf ("%s", argv[0]); for (i = 1; i < argc; i++) printf ("_%s", argv[i]);
    printf ("\n");
  }
  ok = 0;
  for (i = 0; !ok && (i < NBUILTINS); i++) {
    if (strcmp (builtins[i].name, argv[0]) == 0) {
#ifdef USE_CHECK
      if (CHECK && (maclevel > 0)) {
        if ((strcmp ("set"  , argv[0]) == 0) || 
            (strcmp ("macro", argv[0]) == 0))
          rn = builtins[i].func (argc, argv);
      }
      else
#endif USE_CHECK
        rn = builtins[i].func (argc, argv);
      sprintf (statusvar.val, "%x", rn);
      ok++;
    }
  }
  for (mp = firstmac; !ok && (mp != NULL); mp = mp ->next) {
    if (strcmp (mp->name, argv[0]) == 0) {
      Cargc = argc;
      maclevel++;
      Line_nr = 1;
      Toexecute = mp -> mac;
      ok++;
    }
  }
  /* Auto macro load */
  if ((!ok) && autoload (argc, argv)) {
    ok++;
    if (strcmp (firstmac->name, argv[0]) != 0) {
      do_abort ("internal error");
      return (-1);
    }
    Cargc = argc;
    maclevel++;
    Line_nr = 1;
    Toexecute = firstmac->mac;
  }
  if (strcmp (argv[0], "init") == 0) ok++;
  if (!ok) {
    sprintf (msg, "invalid command: `%s'", argv[0]);
    do_abort (msg);
    return (-1);
  }
  return (0);
}
              


                                                                   /* main  */
int main (int argc, char **argv) {
  char cmd[1024];
  int i;
#ifdef USE_READLINE
  static char *rls;

  initialize_readline ();
  rls = NULL;
#endif USE_READLINE

#ifdef USE_SIGNALS
  signal (SIGINT,  SIGNAL_FUNCTION_CAST intsig);
  signal (SIGBUS,  SIGNAL_FUNCTION_CAST bussig);
  signal (SIGSEGV, SIGNAL_FUNCTION_CAST segvsig);
#endif USE_SIGNALS
  /* as soon as the fist 'set' is encountered, VERBOSE is set to 0, so: */
  internal_set_int ("VERBOSE", VERBOSE);

  internal_set ("OPSYS", OPSYS);
  fin = stdin;
  for (maclevel = 0; maclevel < MAXDEPTH; maclevel++) {
    for (i = 0; i < MAXARGS; i++)
      Cargv[i] = MY_MALLOC (128);
    Toexecute = NULL;
  }
  maclevel = 0;
  Toexecute = &init;
  batch = 0;
  i = 1;
  if (i < argc)
    if (strcmp (argv [i], "-b") == 0) {
      i++;
      batch = 1;
    }
  if (i < argc) {
    init.line = malloc (256);
    strcpy (init.line, argv [i]);
    i++;
    if (strcmp (init.line + (strlen (init.line)-4), ".mac") == 0)
      init.line[strlen (init.line)-4] = '\0';
    for (; i < argc; i++){
      strcat (init.line, " ");
      strcat (init.line, argv [i]);
    }
  }
#ifdef USE_SIGNALS
  if (setjmp (jmp_env) != 0) {
    maclevel = 0;
  }
#endif USE_SIGNALS

  while (1) {
    while (((Toexecute == NULL) || (Toexecute->next == NULL))&& (maclevel > 0))
      maclevel--;
    if (Toexecute->next == NULL) {
      if (batch) exit (getintvar ("STATUS"));
#ifdef USE_READLINE

      if (rls) free (rls);
      rls =  readline ("> ");
      if (!rls)
        quit (0, Cargv);
      if (*rls)
        add_history (rls);
      else
        continue;
      strcpy (cmd, rls);
#else
      printf ("> ");
      if (fgets (cmd, 1000, fin) == NULL) quit (0, Cargv);
#endif USE_READLINE
    }
    else{
      strcpy (cmd, Toexecute->line);
      Toexecute = Toexecute->next;
    }
    if ((TRACE > 0) && (maclevel>0)) 
      printf ("%-8.8s.%d.%-3d: %s", stack[maclevel-1].cargv[0], maclevel, 
              Line_nr, cmd);
    Line_nr++;
    if (strcmp (cmd, "") == 0) continue;
    for (i = 0; isspace (cmd[i]); i++);
    if (cmd[i] == '#') continue;
    Cargc = my_sscan (cmd, Cargv);
    if (Cargc < 1 ) {
/*       if (Cargc == -1)  */
/*         do_abort ("parse error"); */
      continue;
    }
    for (i = Cargc; i < MAXARGS; Cargv[i++][0] = '\0');
    if (eval_expression (Cargc, Cargv) == -1) 
      getintvar ("evaluation error");
  }
  exit (0);
}
/* end of file control.c */
