static char rcsid[] = "@(#)$Id: headers.c,v 1.15 2001/06/09 10:23:55 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.15 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *****************************************************************************/

#include "headers.h"
#include "hdr_imp.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"header");

static unsigned char * us_str P_((char *str));
static unsigned char * us_str(str)
     char *str;
{
    return (unsigned char *)str;
}


/* MIME header decode routines .........................................   */


static struct string * hdr_decode_from_text P_((header_list_ptr hdr,
						int demime,
						charset_t defcharset));
static struct string * hdr_decode_from_text(hdr,demime,defcharset)
     header_list_ptr hdr;
     int demime;
     charset_t defcharset;
{
    struct string * result = NULL;

    result = hdr_to_string(HDR_TEXT,hdr->body,defcharset,demime);

    return result;
}

static struct string * hdr_decode_from_raw P_((header_list_ptr hdr,
					       int demime, 
					       charset_t defcharset));

static struct string * hdr_decode_from_raw(hdr,demime,defcharset)
     header_list_ptr hdr;
     int demime;
     charset_t defcharset;
{
    struct string * result = NULL;

    result = new_string2(defcharset,us_str(hdr->body));

    return result;
}

static struct string * decode_comment P_((char * token,int demime, 
					  charset_t defcharset));

static struct string * decode_comment(token,demime,defcharset)
     char * token;
     int demime;
     charset_t defcharset;
{
    struct string *result = new_string(defcharset);
    struct string *newresult = NULL;
    struct string *R = NULL;

    int k,j=0;
    char buffer[1000];
    
    for (k = 1; (token[k] != ')' || token[k+1]) &&
	     j < sizeof buffer -1; k++)
	buffer[j++] = token[k];
    buffer[j] = '\0';
    
    R = hdr_to_string(HDR_COMMENT,buffer,defcharset,demime);
    
    /* HACK */
    add_streambyte_to_string(result,'(');
    add_streambyte_to_string(R,')');
    
    /* Now compine strings */
    newresult = cat_strings(result,R,1);
    free_string(&result);
    result = newresult;

    free_string(&R);
    return result;
}


static struct string * hdr_decode_from_comment P_((header_list_ptr hdr,
					       int demime, 
					       charset_t defcharset));

static struct string * hdr_decode_from_comment(hdr,demime,defcharset)
     header_list_ptr hdr;
     int demime;
     charset_t defcharset;
{
    struct string * result = new_string(defcharset);
    int i;

    char ** tokens = rfc822_tokenize(hdr->body);

    for (i = 0; tokens[i]; i++) {
	struct string * R = NULL, * newresult;

	if ('(' == tokens[i][0]) {
	    R = decode_comment(tokens[i],demime,defcharset);
	} else {
	    R = new_string2(defcharset,us_str(tokens[i]));
	}

	/* Now compine strings */
	newresult = cat_strings(result,R,1);
	free_string(&result);
	result = newresult;

	free_string(&R);
    }

    free_rfc822tokenized(tokens);

    return result;
}



static struct string * hdr_decode_from_phrase P_((header_list_ptr hdr,
					       int demime, 
					       charset_t defcharset));
static struct string * hdr_decode_from_phrase(hdr,demime,defcharset)
     header_list_ptr hdr;
     int demime;
     charset_t defcharset;
{
    struct string * result = new_string(defcharset);
    int i;

    char ** tokens = rfc822_tokenize(hdr->body);

    for (i = 0; tokens[i]; i++) {
	struct string * R = NULL, * newresult;
	
	if ('\n' == tokens[i][0] &&
	    !tokens[i+1])
	    break;

	if (whitespace(tokens[i][0]) || '\n' == tokens[i][0]) {
	    R = new_string(defcharset);
	    add_ascii_to_string(R,us_str(tokens[i]));

	} else if ('(' == tokens[i][0]) {
	    R = decode_comment(tokens[i],demime,defcharset);

	} else if ('<' == tokens[i][0]) {
	    int j;
	    
	    R = new_string(defcharset);
	    add_ascii_to_string(R,us_str(tokens[i]));
	    

	    for (j = i+1; tokens[j] && tokens[j][0] != '>'; j++) {
		
		if ('(' == tokens[i][0]) {
		    struct string * R1 = decode_comment(tokens[j],
							demime,defcharset);
		    
		    /* Now combine strings */
		    newresult = cat_strings(result,R1,1);
		    free_string(&R);
		    R = newresult;
		    
		} else
		    add_ascii_to_string(R,us_str(tokens[j]));
		i = j;
	    }
	    
	    if (tokens[j]) {
		add_ascii_to_string(R,us_str(tokens[j]));
		i = j;
	    }
	} else {
	    char * buffer = NULL;
	    int j;
	    
	    for (j = i; tokens[j]; j++) {
		if ('(' == tokens[j][0]) 
		    break;
		else if ('<' == tokens[j][0]) 
		    break;
		else if ('\n' == tokens[j][0] && !tokens[j+1])
		    break;

		i = j;

		if (buffer)
		    buffer = strmcat(buffer," ");
		buffer = strmcat(buffer,tokens[j]);	
	    }

	    R = hdr_to_string(HDR_PHRASE,buffer,defcharset,demime);	    

	    switch (phrase_display_mode) {
	    case 0: 
		break;
	    default:
		/* We do not (backslash) quote displayed (only) text */
		
		add_ascii_to_string(result,us_str("\""));
		add_ascii_to_string(R,us_str("\""));
		break;
	    }

	    free(buffer);
	}

	/* Now combine strings */
	newresult = cat_strings(result,R,1);
	free_string(&result);
	result = newresult;

	free_string(&R);
    }

    free_rfc822tokenized(tokens);

    return result;
}


static struct string * hdr_decode_from_addr P_((header_list_ptr hdr,
					       int demime, 
					       charset_t defcharset));
static struct string * hdr_decode_from_addr(hdr,demime,defcharset)
     header_list_ptr hdr;
     int demime;
     charset_t defcharset;
{
    struct string * result = new_string(defcharset);

    char ** tokens = rfc822_tokenize(hdr->body);
    int tokcount = 0;
    int ptr = 0;
    int bracket = 0;
    int lastptr = 0;

    while(tokens[tokcount])
	tokcount++;

    /* Remove LF from end of header */
    if (tokcount > 0 &&
	   '\n' == tokens[tokcount-1][0])
	tokcount--;
    
    /* Walk through addressess ... */

    for (ptr = 0; ptr < tokcount; ptr = lastptr) {
	
	char **scanned = NULL;
	struct string *comments = NULL;
	struct string *newresult = NULL;
	char tok = '\0';

	look_special_tokens(tokens,":<>,;",ptr,&lastptr,demime,
			    defcharset,&comments,&scanned);


	if (lastptr < tokcount) {
	    tok = tokens[lastptr][0];
	    DPRINT(Debug,25,(&Debug, 
			     "hdr_decode_from_addr: [%d] token=%c (%s)\n",
			     lastptr,tok,tokens[lastptr]));
	} else {
	    tok = '\0';
	    DPRINT(Debug,25,(&Debug,
			     "hdr_decode_from_addr: [%d] token=EOS\n",
			     lastptr));
	}

	if (ptr < lastptr) {
	    struct string *R = NULL;

	    if (bracket <= 0 && 
		( ':' == tok ||
		  '<' == tok)) {
		DPRINT(Debug,25,(&Debug, 
			   "hdr_decode_from_addr: Adding phrase\n"));

		R = scanned_to_phrase(scanned,
				      demime,
				      defcharset);		

		switch (phrase_display_mode) {
		case 0: 
		    break;
		default:
		    /* We do not (backslash) quote displayed (only) text */
		    
		    add_ascii_to_string(result,us_str("\""));
		    add_ascii_to_string(R,us_str("\""));
		    break;
		}

	    } else {
		int z;

		DPRINT(Debug,25,(&Debug, 
			   "hdr_decode_from_addr: Adding address\n"));

		R = new_string(defcharset);
		
		for (z = 0; scanned[z]; z++) {
		    if ('\n' == scanned[z][0] && !scanned[z+1]) {
			/* HACK: Ignore */
			DPRINT(Debug,25,(&Debug, 
				   "hdr_decode_from_addr: Ignoring ending LF\n"));
		    } else
			add_ascii_to_string(R,us_str(scanned[z]));
		}
	    }


	    /* Now compine strings */
	    newresult = cat_strings(result,R,1);
	    free_string(&result);
	    result = newresult;
	    
	    free_string(&R);
	} else {
	    DPRINT(Debug,25,(&Debug, 			     
			     "hdr_decode_from_addr: (empty) \n"));
	}


	if (comments) {
	    DPRINT(Debug,25,(&Debug,
			     "hdr_decode_from_addr: Adding comments\n"));
	    
	    add_ascii_to_string(result,us_str(" ("));

	    /* Now compine strings */
	    newresult = cat_strings(result,comments,1);
	    free_string(&result);
	    result = newresult;

	    add_ascii_to_string(result,us_str(")"));
	}
	
	if (lastptr < tokcount) {
	    switch (tok) {
	    case '<':
		bracket++;

		if (ptr < lastptr)
		    add_ascii_to_string(result,us_str(" ")); 


		DPRINT(Debug,25,(&Debug,
				 "hdr_decode_from_addr: bracket=%d\n",
				 bracket));
		
		break;
	    case '>':
		bracket--;
		DPRINT(Debug,25,(&Debug,
				 "hdr_decode_from_addr: bracket=%d\n",
				 bracket));
		break;
	    }
	    DPRINT(Debug,25,(&Debug,
			     "hdr_decode_from_addr: Adding token %s\n",
			     tokens[lastptr]));
	    add_ascii_to_string(result,us_str(tokens[lastptr]));
	    lastptr++;

	    switch (tok) {
	    case ':':
	    case ',':
		if (bracket <= 0)
		    add_ascii_to_string(result,us_str(" "));
		break;
	    }
	}
	
	if (comments)
	    free_string(&comments);
	if (scanned)
	    free_rfc822tokenized(scanned);
    }

    free_rfc822tokenized(tokens);

    return result;
}


/* ------------------------------------------------------------------------ */

static struct header_info header_types[] = {
    /* From STD 11 (RFC 822): */
    { "Subject",    HDR_TEXT,                  hdr_decode_from_text },
    { "Comments",   HDR_TEXT,                  hdr_decode_from_text },
    { "Return-path",HDR_STRUCTURED,            hdr_decode_from_comment },
    { "Received",   HDR_STRUCTURED,            hdr_decode_from_comment },
    { "Reply-To",   HDR_STRUCTURED|HDR_PHRASE, hdr_decode_from_addr },  
    { "From",       HDR_STRUCTURED|HDR_PHRASE, hdr_decode_from_addr },    
    { "Sender",     HDR_STRUCTURED|HDR_PHRASE, hdr_decode_from_addr },    
    { "Resent-Reply-To", HDR_STRUCTURED|HDR_PHRASE, hdr_decode_from_addr },
    { "Resent-From",     HDR_STRUCTURED|HDR_PHRASE, hdr_decode_from_addr },
    { "Resent-Sender",   HDR_STRUCTURED|HDR_PHRASE, hdr_decode_from_addr },
    { "Date",        HDR_STRUCTURED,             hdr_decode_from_comment },
    { "Resent-Date", HDR_STRUCTURED,             hdr_decode_from_comment },
    { "To",          HDR_STRUCTURED|HDR_PHRASE,  hdr_decode_from_addr }, 
    { "Resent-To",   HDR_STRUCTURED|HDR_PHRASE,  hdr_decode_from_addr }, 
    { "cc",          HDR_STRUCTURED|HDR_PHRASE,  hdr_decode_from_addr }, 
    { "Resent-cc",   HDR_STRUCTURED|HDR_PHRASE,  hdr_decode_from_addr }, 
    { "bcc",         HDR_STRUCTURED|HDR_PHRASE,  hdr_decode_from_addr }, 
    { "Resent-bcc",  HDR_STRUCTURED|HDR_PHRASE,  hdr_decode_from_addr }, 
    { "Message-ID",  HDR_STRUCTURED,             hdr_decode_from_comment },
    { "Resent-Message-ID",  HDR_STRUCTURED,      hdr_decode_from_comment },
    { "In-Reply-To", HDR_STRUCTURED|HDR_PHRASE,  hdr_decode_from_phrase },
    { "References",  HDR_STRUCTURED|HDR_PHRASE,  hdr_decode_from_phrase }, 
    { "Keywords",  HDR_STRUCTURED|HDR_PHRASE,    hdr_decode_from_phrase },  
    { "Encrypted", HDR_STRUCTURED,               hdr_decode_from_comment },   

    /* From MIME (RFC 1521) */
    { "MIME-Version",              HDR_STRUCTURED, hdr_decode_from_comment }, 
    { "Content-Type",              HDR_STRUCTURED, hdr_decode_from_comment },  
    { "Content-Transfer-Encoding", HDR_STRUCTURED, hdr_decode_from_comment },
    { "Content-ID",                HDR_STRUCTURED, hdr_decode_from_comment },  
    { "Content-Description",       HDR_TEXT,       hdr_decode_from_text },
    /* From RFC 1806 */
    { "Content-Disposition",       HDR_STRUCTURED, hdr_decode_from_comment },
    /* From RFC 1864 */
    { "Content-MD5",               0,            hdr_decode_from_raw },
    
    /* mailbox format */
    { "Content-Length",      0,                  hdr_decode_from_raw },
    { "Status",              0,                  hdr_decode_from_raw },

    /* Sendmail */
    { "Full-Name",           HDR_TEXT,           hdr_decode_from_text },
    { "Return-Receipt-To",  HDR_STRUCTURED|HDR_PHRASE, hdr_decode_from_addr }, 
    { "Auto-Submitted",      HDR_STRUCTURED,     hdr_decode_from_comment },
    { "Precedence",          0,                  hdr_decode_from_raw },
    

    /* IDA Sendmail */
    { "X-Charset",           0,                  hdr_decode_from_raw },
    { "X-Char-Esc",          0,                  hdr_decode_from_raw },
    
    /* Unknown source */
    { "Action",              0,                  hdr_decode_from_raw },
    { "Priority",            HDR_STRUCTURED,     hdr_decode_from_comment },
    { "Expires",             HDR_STRUCTURED,     hdr_decode_from_comment },
    { "Importance",          HDR_STRUCTURED,     hdr_decode_from_comment },
    { "Sensitivity",         HDR_STRUCTURED,     hdr_decode_from_comment },

    /* Our non-standard headers */
    { "X-ELM-OSV",                 HDR_STRUCTURED,  hdr_decode_from_comment },
    { "X-Mailer",                  HDR_TEXT,        hdr_decode_from_text },
    { "Content-Name",              HDR_TEXT,        hdr_decode_from_text },
  
    /* Tailer */
    { NULL, 0, hdr_decode_from_raw }
};

/* Grows forever */
static struct hdr_list {
    struct hdr_list * next;
    struct header_info hdr;
} * extra_headers = NULL;
static int extra_header_count = 0;

header_ptr find_header(name, create_flag) 
     CONST char *name;
     int create_flag;
{
    int i;
    struct hdr_list * walk, *last = NULL;
    
    for (i = 0; header_types[i].header; i++) 
	if (istrcmp(name,header_types[i].header) == 0)
	    return &(header_types[i]);
    
    for (walk = extra_headers; 
	 walk != NULL; 
	 last = walk, walk = walk -> next)
	if (istrcmp(name,walk->hdr.header) == 0)
	    return &(walk->hdr);
    
    if (create_flag) {
	walk = (struct hdr_list * ) safe_malloc (sizeof(struct hdr_list));
	
	if (last) last->next = walk;
	else extra_headers   = walk;
	extra_header_count++;
	
	walk->next = NULL;
	walk->hdr.header = strmcpy(NULL,name);
	
	if (strincmp(name,"X-",2) == 0) {
	    /* Default for user defined headers */
	    walk->hdr.flag = HDR_TEXT;     
	    walk->hdr.hdr_decode_from_it =
		hdr_decode_from_text;
	}
	else {
	    /* We don't know it, right? */
	    walk->hdr.flag = 0;           
	    walk->hdr.hdr_decode_from_it =
		hdr_decode_from_raw;
	}	

	return &(walk->hdr);
    }
  
    return NULL;
}

int classify_header(name)
     CONST char *name;
{
    header_ptr P = find_header(name,0);

    if (P)
	return P->flag;
    
    if (strincmp(name,"X-",2) == 0)
	return HDR_TEXT;              /* Default for user defined headers */

    return 0;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */


header_list_ptr locate_header(h,n)
     header_list_ptr h;
     header_ptr n;
{
    header_list_ptr walk;

    for (walk = h; 
	 walk != NULL; 
	 walk = walk -> next_other_header) {
	if (walk -> magic != HEADER_magic)
	    panic("HEADERS PANIC",__FILE__,__LINE__,"locate_header",
		  "Bad magic number",0);
	if (n == walk -> header_name)
	    return walk;
    }
    return NULL;
}

void update_header_list(list,last1, name, value)
     header_list_ptr *list;
     header_list_ptr *last1;
     CONST char * name;
     CONST char * value;
{
    header_list_ptr result = *list, last = *last1;
    header_list_ptr item, last_this = NULL, walk;

    if (result && !last) {
	last = result;

	DPRINT(Debug,9,(&Debug, 
			"update_header_list: Moving 'last' to end of list ... \n"));

	while (last -> next_header)
	    last = last -> next_header;

    }

    item = (struct header_list *) safe_malloc(sizeof (struct header_list));
    item -> header_name = find_header(name,1);
	
    item -> next_header       = NULL;
    item -> next_this_header  = NULL;
    item -> next_other_header = NULL;
    item -> body              = safe_strdup(value);
    item -> magic             = HEADER_magic;
	
    for (walk = result; 
	 walk != NULL; 
	 last_this = walk, walk = walk -> next_other_header) {
	if (walk -> magic != HEADER_magic)
	    panic("HEADERS PANIC",__FILE__,__LINE__,"update_header_list",
		  "Bad magic number",0);
	if (item -> header_name == walk -> header_name)
	    break;
    }
	
    if (!walk) {
	DPRINT(Debug,9,(&Debug,
			"update_header_list: name='%s': (%p) Not in chain!\n",
			item -> header_name->header,
			item -> header_name));
    }

    if (walk) {  
	while (walk->next_this_header != NULL) {
	    if (walk -> magic != HEADER_magic)
		panic("HEADERS PANIC",__FILE__,__LINE__,"update_header_list",
		      "Bad magic number",0);
	    walk = walk->next_this_header;
	}
	    
	walk->next_this_header = item;
	
	DPRINT(Debug,9,(&Debug,
			"update_header_list: name='%s' -- append next_this_header (%s)\n",
			item->header_name->header,
			walk->header_name->header));
	    	
    } else if (last_this) {
	last_this -> next_other_header = item;
	DPRINT(Debug,9,(&Debug,
			"update_header_list: name='%s' -- append next_other_header (%s)\n",
			item->header_name->header,
			last_this->header_name->header));
    }
    
    if (last) {
	last -> next_header  = item;
	
	DPRINT(Debug,9,(&Debug,
			"update_header_list: name='%s' -- append next_header (%s)\n",
			item->header_name->header,
			last->header_name->header));
	
    } else  {
	result               = item;
	DPRINT(Debug,9,(&Debug,
			"update_header_list: name='%s' -- head of next_header\n",
			item->header_name->header));
    }
    last = item;
    
    *list  = result;
    *last1 = last;
}

void delete_headers(hdr)
     header_list_ptr *hdr;
{
    header_list_ptr next = *hdr;
  
    while(next) {
	if (next -> magic != HEADER_magic)
	    panic("HEADERS PANIC",__FILE__,__LINE__,"delete_headers",
		  "Bad magic number",0);
	
	(*hdr)  = next;
	next = next -> next_header;
	
	if((*hdr) -> body) {
	    free((*hdr) -> body);
	    (*hdr) -> body = NULL;
	}
	(*hdr) -> next_header       = NULL;
	(*hdr) -> next_this_header  = NULL;
	(*hdr) -> next_other_header = NULL;
	(*hdr) -> magic             = 0;
	free((void *)(*hdr));
    }
    (*hdr)  = next;
}

header_list_ptr read_folder_headers(read_state_ptr,folder,entry)
     READ_STATE read_state_ptr;
     struct folder_info *folder;
     struct header_rec *entry;
{
    char *buffer;
    int len;

    header_list_ptr result = NULL, last = NULL;
    header_list_ptr tmphdr;

    while (copy_header_folder(folder,read_state_ptr, &buffer,&len)) {
	char * val;
	
	unfold_header(buffer,&len,entry);
	val = strchr(buffer,':');

	if (!val) {
	    panic("HEADERS PANIC",__FILE__,__LINE__,"read_folder_headers",
		  "bad return from copy_header_folder -- ':' not found",0);
	}

	*val = '\0';
	val++;

	while (whitespace(*val))
	    val++;

	no_ret(val);

	update_header_list(&result,&last,buffer,val);

	free(buffer); buffer = NULL;
    }

    if (NULL != (tmphdr = locate_header_by_name(result,"Return-Path"))) {
	/* Required for POP or 
	 * when env_from_source == 2
	 */
	if (!entry->env_from[0]) {
	    return_path_to_env_from(entry,tmphdr->body);
	}
    }	

    if (NULL != (tmphdr = locate_header_by_name(result,"Message-Id"))) {
	strfcpy(entry->messageid, tmphdr->body, 
		sizeof entry->messageid);
    }

    if (NULL != (tmphdr = locate_header_by_name(result,"Content-Length"))) {
	entry->content_length = atol(tmphdr->body);
    }

    if (NULL != (tmphdr = locate_header_by_name(result,"Expires"))) {
	process_expiration_date(tmphdr->body, &(entry->status));
    }

    if (NULL != (tmphdr = locate_header_by_name(result,"Date"))) {
	parse_arpa_date(tmphdr->body, entry);
    }
    
    /* We need accept headers like:
     *    MIME-Version: (MetaSend v1.7) 1.0
     * and
     *    Mime-Version: 1.0 (NeXT Mail 4.2mach_patches v148.2)
     * And if version number is something other accept it but give 
     * warning when message is viewed or replied.
     */
    
    if (NULL != (tmphdr = locate_header_by_name(result,"MIME-Version"))) {

	char ** tokens = rfc822_tokenize(tmphdr->body);

	remove_space_tokenized(tokens);
	
	if (!tokens[0] || 0 != strcmp(tokens[0],"1") ||
	    !tokens[1] || 0 != strcmp(tokens[1],".") ||
	    !tokens[2] || 0 != strcmp(tokens[2],"0") ||
	    tokens[3]) {
	    int i;
	    entry->status |= MIME_MESSAGE|MIME_UNSUPPORTED;
	    DPRINT(Debug,9,(&Debug, 
			    "read_folder_headers: Unsupported mime version '%s'\n",
			    tmphdr->body));
	    DPRINT(Debug,10,(&Debug, 
			    "      tokens:"));
	    for (i = 0; tokens[i]; i++) {
		DPRINT(Debug,10,(&Debug, 
				 " [%d]=%s",i,tokens[i]));
	    }
	    DPRINT(Debug,10,(&Debug, 
			     "\n"));
	} else
	    entry->status |= MIME_MESSAGE;

	if (tmphdr->next_this_header &&
	    0 != istrcmp(tmphdr->body,tmphdr->next_this_header->body) &&
            show_header_errors) {
            lib_error(CATGETS(elm_msg_cat, MeSet, MeParseErrorMIMEVersion,
                              "PARSE ERROR: Several MIME-Version headers!"));
        }

	free_rfc822tokenized(tokens);
    }    

    if (NULL != (tmphdr = locate_header_by_name(result,
						"X-ELM-OSV"))) {
	char value [20];
	
	if (mime_get_param("no-hdr-encoding",value,tmphdr->body,
			   sizeof value)) {
	    int i = atoi(value);
	    DPRINT(Debug,9,(&Debug,"-- no-hdr-encoding=%d\n",i));
	    if (i) 
		entry->status |= NOHDRENCODING;
	}
	
	if (mime_get_param("hdr-charset",value,tmphdr->body,
			   sizeof value)) {
	    entry->header_charset = MIME_name_to_charset(value,1);
	    DPRINT(Debug,9,(&Debug,"-- header-charset=%s\n",
			    entry->header_charset->MIME_name));
	}
    }

    return result;
}

/* Returns ponter to static storage */
char * give_header_name(X)
     header_ptr X;
{
    return X->header;
}

int give_header_flags(X)
     header_ptr X;
{
    return X->flag;
}

struct string * give_decoded_header(hdr,demime,defcharset)
     header_list_ptr hdr;
     int demime;
     charset_t defcharset;
{
    struct string * result = NULL;

    if (hdr -> magic != HEADER_magic)
	panic("HEADERS PANIC",__FILE__,__LINE__,"give_decoded_header",
	      "Bad magic number",0);

    result = hdr->header_name->hdr_decode_from_it(hdr,demime,defcharset);

    return result;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */

