static char rcsid[] = "@(#)$Id: terminal.c,v 1.6 2001/06/03 12:19:22 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.6 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *****************************************************************************/

#include "headers.h"
#include "s_me.h"
#include "cs_imp.h"

DEBUG_VAR(Debug,__FILE__,"charset");

#include <errno.h>
extern int errno;

static enum terminal_map_keyword get_keyword P_((const char *word));

static enum terminal_map_keyword get_keyword(word)
     CONST char *word;
{
    if (0 == istrcmp(word,"ISO-2022"))
	return terminal_iso2022;
    if (0 == istrcmp(word,"PRIVATE"))
	return terminal_private;

    
    return terminal_bad;
}

static char * sStr P_((unsigned char *p));
static char * sStr (p)
     unsigned char *p;
{
    return (char *)p;
}

struct terminal_map_item * load_terminal_map(filename,errors)
     CONST char *filename;
     int *errors;
{
    struct terminal_map_item *result;
    int result_len = 0;
    FILE * f = fopen(filename,"r");
    int max_result = 0;
    int c;
    char buf[LONG_STRING];

    if (!f) {
 	int err = errno;
	DPRINT(Debug,2,(&Debug,"load_terminal_map: %s: %s\n",
			filename,error_description(err)));
	return NULL;
    }

    while(EOF != (c = fgetc(f)))
	if ('\n' == c)
	    max_result++;

    DPRINT(Debug,10,(&Debug,"load_terminal_map: %s, max_result=%d\n",
		     filename,max_result));
    
    if (!max_result) {
	fclose(f);
	return NULL;
    }
    rewind(f);
    
    result = safe_malloc((max_result +1) * sizeof (struct terminal_map_item));
    
    while (result_len < max_result &&
	   fgets(buf,sizeof buf, f) != NULL) {
	char * c = buf,*c1,*c2;
	int l1 = strlen(buf);
	enum terminal_map_keyword kw;

	if ('\n' == buf[l1 -1]) 
	    buf[l1 - 1] = '\0';
	else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLine,
			      "%30s: Too long line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;
	}

	while (l1-- > 0 && whitespace(buf[l1]))
	    buf[l1] = '\0';
	
	c = buf;
	while (*c && whitespace (*c)) /* skip leading whitespace */
	    c++;
	if ('#' == *c)
	    continue;
	if (!*c)
	    continue;

	c1 = strpbrk(c," \t");

	if (!c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}
	*c1 = '\0';
	
	c1++;

	while (*c1 && whitespace (*c1)) /* skip leading whitespace */
	    c1++;
	if (!*c1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}
	    

	c2 = strpbrk(c1," \t");

	if (!c2) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}
	*c2 = '\0';
	
	c2++;

	while (*c2 && whitespace (*c2)) /* skip leading whitespace */
	    c2++;
	if (!*c2) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}
	    
	kw = get_keyword(c1);

	if (terminal_bad == kw) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
			      "%30s: Bad line: %.30s..."),
		      filename,buf);
	    (*errors) ++;
	    break;	    
	}
	
	switch(kw) {
	    char * c3;
	    int size;
	case terminal_iso2022:
	    result[result_len].keyword = kw;
	    
	    if (load_charset_map_info(&result[result_len].
				      value.iso2022.charset,c2))
		result[result_len].value.iso2022.is_def = 
		    ((result[result_len].value.iso2022.charset->flags &
		      SET_nodata) != 0);
	    else {
		result[result_len].value.iso2022.charset = MIME_name_to_charset(c2,0); 
		result[result_len].value.iso2022.is_def = 0;
	    }
	    
	    if (!result[result_len].value.iso2022.charset) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnknownTerminalCharset,
				  "Unknown terminal charset %s (file %s)"),
			  c2,filename);
		(*errors) ++;
		continue;
	    }

	    if (!result[result_len].value.iso2022.charset->map_info) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeCharsetNeedsMap,
				  "Charset %s needs map= defination (file %s)"),
			  result[result_len].value.iso2022.charset->MIME_name ?
			  result[result_len].value.iso2022.charset->MIME_name :
			  "<no MIME name>",
			  filename);
		(*errors) ++;
		continue;
	    }

	    if (!result[result_len].value.iso2022.charset->iso2022_info) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeCharsetNeedsIso2022,
				  "Charset %s needs iso2022 defination (file %s)"),
			  result[result_len].value.iso2022.charset->MIME_name ?
			  result[result_len].value.iso2022.charset->MIME_name :
			  "<no MIME name>",
			  filename);
		(*errors) ++;
		continue;
	    }

	    result[result_len].match = safe_strdup(c);
	    result_len++;
	    break;
	case terminal_private:
	    result[result_len].keyword = kw;

	    c3 = strpbrk(c2," \t");

	    if (!c3) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
				  "%30s: Bad line: %.30s..."),
			  filename,buf);
		(*errors) ++;
		break;	    
	    }
	    *c3 = '\0';
	
	    c3++;

	    while (*c3 && whitespace (*c3)) /* skip leading whitespace */
		c3++;
	    if (!*c3) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLine,
				  "%30s: Bad line: %.30s..."),
			  filename,buf);
		(*errors) ++;
		continue;    
	    }
	    result[result_len].value.private.charset = 
		MIME_name_to_charset(c2,0); 
	    if (!result[result_len].value.private.charset) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeUnknownTerminalCharset,
				  "Unknown terminal charset %s (file %s)"),
			  c2,filename);
		(*errors) ++;
		continue;
	    }
	    size = strlen(c3);
	    result[result_len].value.private.bytes = safe_malloc(size+1);

	    if (!parse_gen_spec(c3,result[result_len].
				value.private.bytes,
				size)) {
		(*errors) ++;
		continue;
	    }
	    result[result_len].value.private.bytes[size] = '\0';
	    result[result_len].match = safe_strdup(c);
	    result_len++;
	    break;
	}
    }

    result[result_len].match   = NULL;
    result[result_len].keyword = terminal_bad;

    DPRINT(Debug,10,(&Debug,
		     "load_terminal_map: %s, result_len=%d\n",
		     filename,result_len));
    
    return result;
}

static int name_ok P_((CONST char *name));
static int name_ok(name)
     CONST char *name;
{
    if ('\0' == *name)
	return 0;

    return strlen(name) == strspn(name,"ABCDEFGHIJKLMNOPQRSTUVXYZ-0123456789");
}


void dump_terminal_map(f,map)
     FILE *f; 
     struct terminal_map_item *map;
{
    struct terminal_map_item *ptr;
    
    for (ptr = map; ptr && ptr->match; ptr++) {
	switch(ptr->keyword) {
	    char *s;

	case terminal_iso2022:
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    fputs("ISO-2022",f);
	    fputc('\t',f);
	    if (ptr->value.iso2022.is_def)
		dump_map_info(f,ptr->value.iso2022.charset);
	    else {
		if (!ptr->value.iso2022.charset->MIME_name) {
		    fputs("# <No MIME name> ",f);
		} else if (name_ok(ptr->value.iso2022.charset->MIME_name)) {
		    fputs(ptr->value.iso2022.charset->MIME_name,f);
		} else {
		    elm_fprintf(f,FRM("%Q;!"),
				ptr->value.iso2022.charset->MIME_name);
		}
	    }
	    break;
	case terminal_private:
	    fputs(ptr->match,f);
	    fputc('\t',f);
	    fputs("PRIVATE",f);
	    fputc('\t',f);
	    fputs(ptr->value.private.charset->MIME_name,f);
	    fputc('\t',f);
	    s = iso2022_codestr(ptr->value.private.bytes,
				strlen(sStr(ptr->value.private.bytes)));
	    if (s) {
		fputs(s,f);
		free(s);
	    }
	    break;
	}
	fputc('\n',f);
    }
}

static struct terminal_map_item * loc_info P_((const char *terminal, 
					       charset_t set));

static struct terminal_map_item * loc_info(terminal,set)
     CONST char *terminal; 
     charset_t set;
{
    int i;

    for (i =  0; i < 2; i++) {
	struct terminal_map_item *ptr = user_terminal_map;

	if (i)
	    ptr = system_terminal_map;

	while (ptr && ptr->match) {
	    char * c = NULL;

	    if (0 == strcmp(ptr->match,terminal) ||
		( (c = strchr(ptr->match,'*')) &&
		  *(c+1) == '\0' &&
		  0 == strncmp(ptr->match,terminal,c - ptr->match))) {

		if (ptr->keyword == terminal_iso2022 &&
		    ptr->value.iso2022.charset == set)
		    return ptr;	    

		if (ptr->keyword == terminal_private &&
		    ptr->value.private.charset == set)
		    return ptr;	    
	    }
	    ptr++;
	}
    }
    return NULL;
}

void terminal_can_switch(terminal,storage,n,max)
     CONST char *terminal;
     charset_t *storage;
     int *n;
     int max;
{

    int i;

    /* Can not switch to other sets if can not
       switch back to display_charset
    */
    if (!loc_info(terminal,display_charset))
	return;

    for (i =  0; i < 2; i++) {
	struct terminal_map_item *ptr = user_terminal_map;
	
	if (i)
	    ptr = system_terminal_map;
	
	while (ptr && ptr->match && *n < max) {
	    char * c = NULL;
	    int j;

	    if (0 == strcmp(ptr->match,terminal) ||
		( (c = strchr(ptr->match,'*')) &&
		  *(c+1) == '\0' &&
		  0 == strncmp(ptr->match,terminal,c - ptr->match))) {
	
		switch (ptr->keyword) {
		case terminal_iso2022:
		    storage[(*n)++] = ptr->value.iso2022.charset;
		    break;

		case terminal_private:
		    storage[(*n)++] = ptr->value.iso2022.charset;
		    break;

		}

		for (j = 0; j < (*n) -1; j++)
		    if (storage[j] == storage[(*n)-1]) {
			(*n)--;  /* Remove duplicate */  
			break;
		    }
	    }
	    ptr++;
	}
    }
}

int terminal_can_switch_to(terminal,set,silent)
     CONST char *terminal; 
     charset_t set;
     int silent;
{

    struct terminal_map_item * a = loc_info(terminal,set);

    if (!a) {
	if (!silent)
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeNoInformationToSwitchTerminal,
			      "No information to switch %s terminal to %s charset!"), 
		      terminal,
		      set->MIME_name ? set->MIME_name : "<no MIME name>");
		      
	return 0;
    }
    
    a = loc_info(terminal,display_charset);

    if (!a) {
	if (!silent)
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeNoInformationToSwitchBack,
			      "No information to switch %s terminal back to %s charset!"), 
		      terminal,
		      display_charset->MIME_name ? display_charset->MIME_name :
		      "<no MIME name>");
	return 0;
    }
    return 1;
}

struct display_settings {
    int                  set_count;
    struct iso2022_setid sets [6];
    int                  bank[4];

    enum iso2022_bank    current_L;   /*  32-127 */
    enum iso2022_bank    current_R;   /* 160-255 */

};

static void debug_display_settings P_((struct display_settings *p));
static void debug_display_settings(p)
     struct display_settings *p;
{
    int i;
    int corrupted = 0;
    DPRINT(Debug,8,(&Debug,
		    "Terminal sets: (%d sets)\n",
		    p->set_count));
    for (i = 0; i < p->set_count; i++) {
	char *s;
	if (i >= sizeof (p->sets) /
	    sizeof (p->sets[0])) {
	    corrupted++;
	    break;
	}

	DPRINT(Debug,8,(&Debug,"    set [%d]:  bank=",i));
	switch(p->sets[i].bank) { 
	case bank_unspecified:  DPRINT(Debug,8,(&Debug, "(none)")); break;
	default:                DPRINT(Debug,8,(&Debug, "G%d   ",
						p->sets[i].bank)); break;
	}
	DPRINT(Debug,8,(&Debug," type="));
	switch(p->sets[i].type) { 
	case iso2022_other: DPRINT(Debug,8,(&Debug,"other")); break;
	case iso2022_94:    DPRINT(Debug,8,(&Debug,"94   ")); break;
	case iso2022_96:    DPRINT(Debug,8,(&Debug,"96   ")); break;
	case iso2022_94x94: DPRINT(Debug,8,(&Debug,"94x94")); break;	
	case iso2022_96x96: DPRINT(Debug,8,(&Debug,"96x96")); break;	
	default:            DPRINT(Debug,8,(&Debug,"?????")); 
	    corrupted++; break;
	}
	s = iso2022_codestr(p->sets[i].bytes, sizeof p->sets[i].bytes);
	DPRINT(Debug,8,(&Debug," bytes=%s\n",s));
	free(s);
    }

    for (i = 0; i < 4; i++) {
	DPRINT(Debug,8,(&Debug,"    bank G%d=",i));;
	switch(p->bank[i]) { 
	case -1:    DPRINT(Debug,8,(&Debug,"(none)")); break;
	default:    DPRINT(Debug,9,(&Debug,"set %d",p->bank[i]));
	    if (p->bank[i] < 0 || p->bank[i] >= p->set_count) {
		corrupted++;
		DPRINT(Debug,8,(&Debug,"   ????"));
	    }
	    break;
	}
	DPRINT(Debug,8,(&Debug,"\n"));
    }
    DPRINT(Debug,8,(&Debug,"       left="));;
    switch(p->current_L) {
    case bank_unspecified:  DPRINT(Debug,8,(&Debug,"(none)")); break;
    default:                DPRINT(Debug,8,(&Debug,"G%d   ",
					    p->current_L)); break;
    }
    DPRINT(Debug,8,(&Debug,"\n"));
    DPRINT(Debug,8,(&Debug,"      right="));;
    switch(p->current_R) {
    case bank_unspecified:  DPRINT(Debug,8,(&Debug,"(none)")); break;
    default:                DPRINT(Debug,8,(&Debug,"G%d   ",
					    p->current_R)); break;
    }
    DPRINT(Debug,8,(&Debug,"\n"));

    if (corrupted)
	panic("STRING PANIC",__FILE__,__LINE__,
	      "debug_display_settings",
#ifdef DEBUG
	      Debug.active >= 8 ? 
	      "Corrupted terminal sets information -- see debugfile" :
#endif
	      "Corrupted terminal sets information",0);
}

static void debug_charset_sets P_((struct setlist  *p));
static void debug_charset_sets(p)
     struct setlist  *p;
{
    int i;

    DPRINT(Debug,8,(&Debug,"(output) charset sets: \n"));
    for (i = 0; 
	 i < sizeof (p->sets)  / sizeof (p->sets[0]) && p->sets[i]; 
	 i++) {
	char *s;
	DPRINT(Debug,8,(&Debug,"    set [%d]:  bank=",i));
	switch(p->sets[i]->bank) { 
	case bank_unspecified:  DPRINT(Debug,8,(&Debug,"(none)")); break;
	default:                DPRINT(Debug,8,(&Debug,"G%d   ",
						p->sets[i]->bank)); break;
	}
	DPRINT(Debug,8,(&Debug," type="));
	switch(p->sets[i]->type) { 
	case iso2022_other: DPRINT(Debug,8,(&Debug,"other")); break;
	case iso2022_94:    DPRINT(Debug,8,(&Debug,"94   ")); break;
	case iso2022_96:    DPRINT(Debug,8,(&Debug,"96   ")); break;
	case iso2022_94x94: DPRINT(Debug,8,(&Debug,"94x94")); break;	
	case iso2022_96x96: DPRINT(Debug,8,(&Debug,"96x96")); break;	
	default:            DPRINT(Debug,8,(&Debug,"?????")); break;
	}
	s = iso2022_codestr(p->sets[i]->bytes, sizeof p->sets[i]->bytes);
	DPRINT(Debug,8,(&Debug," bytes=%s\n",s));
	free(s);
    }    
}

static void reset_display_settings P_((struct display_settings *p));
static void reset_display_settings(p)
     struct display_settings *p;
{
    /* defined in hdrs/defs.h */
    bzero((void *)p,sizeof (struct display_settings));

    p->set_count          = 0;
    p->bank[0]            = -1;
    p->bank[1]            = -1;
    p->bank[2]            = -1;
    p->bank[3]            = -1;
    p->current_L          = bank_unspecified;
    p->current_R          = bank_unspecified;
}


/* Result is MALLOCed */
static char * helper_1 P_((screen_info_p terminal_info,
			   struct iso2022_setid ID,
			   int setnum));
static char * helper_1(terminal_info,ID,setnum)
     screen_info_p terminal_info;
     struct iso2022_setid ID;
{
    char * ret = NULL;

    if (ID.bank != bank_unspecified) {
	if (-1 == terminal_info->bank[ID.bank]) {	       
	    char *c;
	    terminal_info->bank[ID.bank] = setnum;
	    
	    c = iso2022_setid_stream(ID);
	    ret = strmcat(ret,c);
	    free(c);


	    if (terminal_info->current_L  == bank_unspecified &&
		ID.bank    == bank_G0) {
		terminal_info->current_L  = bank_G0;

		c = lock_shift(0,terminal_info->current_L);
		ret = strmcat(ret,c);
		free(c);
	    }

	    if (terminal_info->current_R  == bank_unspecified &&
		ID.type    == iso2022_96) {
		terminal_info->current_R  = ID.bank;

		c = lock_shift(1,terminal_info->current_R);
		ret = strmcat(ret,c);
		free(c);
	    }

	} else if (terminal_info->bank[ID.bank] != setnum) {
	    DPRINT(Debug,4,(&Debug,
			    "terminal_switch_to [helper_1]: Several assignments for bank G%d\n",
			    ID.bank));
	    DPRINT(Debug,8,(&Debug,
			    "     Assigned to set %d -- another candinate set %d\n",
			    terminal_info->bank[ID.bank],setnum));
	}
    } else if (ID.type == iso2022_other) {
	ret = iso2022_setid_stream(ID);
    }

    return ret;
}



/* Result is MALLOCed */
char * terminal_charset_post_init(terminal_info,iso2022_info)
     screen_info_p terminal_info;
     struct setlist          *   iso2022_info;
{
    char * ret = NULL;
    int max,k;
    
    /* FIXME: Usage of this function on string.c unclear!!! */

    static screen_info_p    LAST_COMPLAIN   = NULL;
    static struct setlist * LAST_COMPLAIN_b = NULL;        

    /* Only check that terminal_info includes sames sets
       (on beginning of list) than iso2022_info */
    
    for (max = 0; max < sizeof (iso2022_info->sets)  / 
	     sizeof (iso2022_info->sets[0]) &&
	     iso2022_info->sets[max];
	 max++) {
	
	if (max >= terminal_info->set_count ||
	    terminal_info->sets[max].type != iso2022_info->sets[max]->type ||
	    0 != strncmp(sStr(terminal_info->sets[max].bytes),
			 sStr(iso2022_info->sets[max]->bytes),
			 sizeof (terminal_info->sets[max].bytes))  ||
	    (bank_unspecified           != iso2022_info->sets[max]->bank &&
	     terminal_info->sets[max].bank != iso2022_info->sets[max]->bank)) {
	    if (LAST_COMPLAIN   != terminal_info ||
		LAST_COMPLAIN_b != iso2022_info) {
		LAST_COMPLAIN   = terminal_info;
		LAST_COMPLAIN_b = iso2022_info;
		DPRINT(Debug,8,(&Debug,
				"terminal_charset_post_init: Output charset not current terminal charset\n"));
		DPRINT(Debug,8,(&Debug,
				"            (perhaps subset) -- can't really post init terminal charset\n"));
		if (max >= terminal_info->set_count) {
		    DPRINT(Debug,8,(&Debug,
				    "             Only %d sets on terminal... \n",
				    terminal_info->set_count));
		} else {
		    DPRINT(Debug,8,(&Debug,
				    "             [%d] -- bank G%d on terminal set not match to output charset\n",
				    max, terminal_info->sets[max].bank));
		}
		debug_display_settings(terminal_info);
		debug_charset_sets(iso2022_info);
	    }
	    return NULL;
	}
    }
    LAST_COMPLAIN   = NULL;
    LAST_COMPLAIN_b = NULL;

    /* Remove assignments which does not belong to iso2022_info */
    for (k = 0; k < 4; k++) {
	if (terminal_info->bank[k] >= max) {
	    if (k == terminal_info->current_L)
		terminal_info->current_L = bank_unspecified;
	    if (k == terminal_info->current_R)
		terminal_info->current_R = bank_unspecified;
	    terminal_info->bank[k] = -1;	   
	}
    }
    
    for (k = 0; k < max; k++) {
	char * c;
	struct iso2022_setid ID = terminal_info->sets[k];

	c=helper_1(terminal_info,ID,k);
	if (c) {
	    ret = strmcat(ret,c);
	    free(c);
	}
    }

    if (ret)
	debug_display_settings(terminal_info);

    return ret;
}

/* Result is MALLOCed */
char * terminal_switch_to(terminal,set,terminal_info)
     char *terminal; 
     charset_t set;
     screen_info_p *terminal_info;   
{
    char *ret = NULL;
    struct terminal_map_item * a = loc_info(terminal,set);
    
    if (*terminal_info) {
	free(*terminal_info);
	*terminal_info = NULL;
    }

    if (!a)
	return NULL;

    if (display_charset) {
	struct terminal_map_item * a0 = loc_info(terminal,
						 display_charset);

	if (a0 && a0->keyword == terminal_iso2022 &&
	    display_charset->iso2022_info &&
	    display_charset->iso2022_info->sets[0] &&
	    display_charset->iso2022_info->sets[0]->type == iso2022_other) {
	    ret = iso2022_setid_stream(return_to_iso2022);
	}
    }

    switch(a->keyword) {
    case terminal_iso2022:
	if (set->iso2022_info) {
	    char *c;
	    int i;

	    *terminal_info = safe_malloc(sizeof (struct display_settings));
	    reset_display_settings(*terminal_info);
	    
	    for (i = 0; 
		 i < sizeof (set->iso2022_info->sets)  / 
		     sizeof (set->iso2022_info->sets[0]) &&
		     set->iso2022_info->sets[i];
		 i++) {
		struct iso2022_setid ID = * (set->iso2022_info->sets[i]);

		if (ID.bank == bank_unspecified) {
		    switch(ID.type) {
		    case iso2022_94:
			ID.bank = bank_G0;
			break;
		    case iso2022_94x94:         /* not implemented yet */
			ID.bank = bank_G1;
          
			/* We use bank G2 to compatible of VT3xx terminals */
		    case iso2022_96:  
		    case iso2022_96x96:         /* not implemented yet */
			if (-1 == (*terminal_info)->bank[bank_G2])
			    ID.bank = bank_G2;
			else
			    ID.bank = bank_G3;
		    }
		}

		c=helper_1(*terminal_info,ID,(*terminal_info)->set_count);

		if ((*terminal_info)->set_count >=
		    sizeof ((*terminal_info)->sets) /
		    sizeof ((*terminal_info)->sets[0]))
		    panic("STRING PANIC",__FILE__,__LINE__,
			  "terminal_switch_to",
			  "Too many iso 2022 sets on display charset",0);

		(*terminal_info)->sets[(*terminal_info)->set_count] = ID;
		(*terminal_info)->set_count++;

		if (c) {
		    ret = strmcat(ret,c);
		    free(c);
		}
	    }
	    debug_display_settings(*terminal_info);
	    break;
	case terminal_private:
	    ret = strmcat(ret,sStr(a->value.private.bytes));
	    break;
	}
    }
    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */


