static char rcsid[] = "@(#)$Id: attach_menu.c,v 1.31 2001/06/06 18:09:04 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.31 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "headers.h"
#include "me.h"
#include "s_elm.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mime");

#include <sys/errno.h>

extern int errno;
extern int elm_COLUMNS;

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static int can_save(filename) 
     char *filename;
{
    if (access(filename,ACCESS_EXISTS) < 0) {
	if (errno == ENOENT)
	    return 1;    /* File not exists -- can save ! */
	return 0;     /* Other error -- don't save with that name ! */
    } else
	return 0;    /* File exits -- don't save over ! */
}

static FILE *  get_attachment P_((struct folder_info *mail_folder,
				  mime_t *att));

static FILE *  get_attachment(mail_folder,att)
     struct folder_info *mail_folder;
     mime_t *att;
{
    FILE * result = NULL;
    
    if (att->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"get_attachment",
		   "Bad magic number");
    
    if (att->pathname) {
	if (can_open(att->pathname,"r") != 0) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorReadableUser,
			      "%.50s isn't readable by user!"), 
		      att->pathname);
	    sleep_message();
	    return NULL;
	}
    
	result = fopen(att->pathname,"r");
	if (!result) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantOpenAttach,
			      "Can't open attachment: %.50s"),
		      att->pathname);
	    sleep_message();
	}
    } else {
	result = folder_to_fd(mail_folder,att->offset);
	if (!result) {		
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedSeekAttach,
			      "Failed to seek beginning of attachment!"));
	    sleep_message();
	}
    }
    return result;
}

static void close_attachment P_((struct folder_info *mail_folder, 
				 FILE *tmpfd));

static void close_attachment(mail_folder, tmpfd)
     struct folder_info *mail_folder;
     FILE *tmpfd;
{
    if (!tmpfd)
	return;
    if (folder_to_fd(mail_folder,-1L) != tmpfd)
	fclose(tmpfd);
}

static void attachment_copy P_((mime_t *att,
				FILE *tmpfd, FILE *outfd,
				charset_t defcharset));

static void attachment_copy(att, tmpfd, outfd, defcharset)
     mime_t *att;
     FILE *tmpfd, *outfd;
     charset_t defcharset;
{
    if (att->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attachment_copy",
		   "Bad magic number");
    
    if (att->pathname) {  /* Already decoded */
	char buf[VERY_LONG_STRING];
	int len;
	while (0 < (len = fread(buf,1,sizeof(buf),tmpfd))) {
	    if (fwrite(buf,1,len,outfd) != len) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmWriteErrorAttach,
				  "Write error when copying attachment!"));
		sleep_message();
		break;
	    }
	}
	if (ferror(tmpfd)) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorReading,
			      "Error reading from %.50s"),
		      att->pathname);
	    sleep_message();
	}
    } else { /* Needs decode */
	in_state_t state_in;
	out_state_t state_out;
	charset_t vector[2];
	
	int disp = att -> disposition;
	
	in_state_clear(&state_in, STATE_in_file);
	out_state_clear(&state_out, STATE_out_file);
	
	set_out_state_file(outfd,&state_out);
	state_out.displaying      = FALSE;
	
	vector[0] = display_charset;
	vector[1] = NULL;
	state_out.display_charset = vector;
	set_in_state_file(tmpfd,&state_in);
	
	att -> disposition = DISP_INLINE;  /* Show it ! */
	mime_decode(att,&state_in, &state_out,
		    defcharset);
	att -> disposition = disp;
	
	
	in_state_destroy(&state_in);
	out_state_destroy(&state_out);
    }
}

static void attach_print P_((mime_t *att,
			     charset_t defcharset));


static void attach_print (att, defcharset)
     mime_t *att;
     charset_t defcharset;
{
    char tempfile[STRING];
    char buf[VERY_LONG_STRING];
    
    if (att->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_print",
		   "Bad magic number");

    if (!current_folder)
	return;

    if (!have_printout())
	return;

    elm_sfprintf(tempfile,sizeof tempfile,
		 FRM("%selm.%d"), temp_dir, getpid());

    if (att->type == MIME_TYPE_TEXT && istrcmp(att->subtype,"plain") == 0) {
	FILE *f_out, *f_in;
	int ret;
	
	DPRINT(Debug,3,
	       (&Debug,"attach_print: printing directly\n"));
	
	if (!(f_out = safeopen(tempfile))) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorCreatTempfile,
			      "Error creating tempfile %s"),
		      tempfile);
	    sleep_message();
	    return;
	}

	if (!(f_in = get_attachment(current_folder,att))) {
	    fclose(f_out);
	    unlink(tempfile);
	    return;
	}
	attachment_copy(att,f_in,f_out, defcharset);
	
	(void) elm_chown (tempfile, userid, groupid);
	
	fclose(f_out);
	elm_sfprintf(buf, sizeof buf,
		     FRM(printout), tempfile);
	ret = system_call(buf,0);
	if (ret == 0)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmJobSpooled,
			      "Print job spooled."));
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorPrinting,
			      "Error while printing!"));
	unlink(tempfile);
	close_attachment(current_folder,f_in);
    }
    else if (have_metamail()) {
	DPRINT(Debug,3,
	       (&Debug,"attach_print: printing via metamail: %s\n",
		metamail_path));
	if (att->pathname)
	    elm_sfprintf(buf,sizeof buf,
			 FRM("%s -m Elm -h -b -c %s/%s %s"),
			 metamail_path,TYPE(att->type), att->subtype, 
			 att->pathname);
	else {
	    FILE *fpout, *ZZ;
	    
	    fpout = safeopen(tempfile);

	    if (!fpout) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorCreatTempfile,
				  "Error creating tempfile %s"),
			  tempfile);
		sleep_message();
		return;
	    }
	    
	    (void) elm_chown (tempfile, userid, groupid);
	    ZZ = folder_to_fd(current_folder,att->begin_offset);
	    if (ZZ) {
		while (ftell(ZZ) < att->offset + att->length) {
		    int len = mail_gets(buf,VERY_LONG_STRING,ZZ);
		    if (len <= 0) 
			break; /* Error ? */
		    fwrite(buf,1,len,fpout);
		}
	    }
	    /* Option -z cuses that metamail unlinks tempfile */
	    fclose(fpout);
	    elm_sfprintf(buf,sizeof buf,
			 FRM("%s -m Elm -h -z %s"),
			 metamail_path, tempfile);
	}
	Raw(OFF);
	system_call(buf,SY_ENV_METAMAIL);
	PressAnyKeyToContinue();
	Raw(ON);
    }
    else
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNotKnowPrint,
			  "Don't know how to print this type of data!"));
    
}

static void attach_save P_((mime_t *a, int *redraw));
static void attach_save (a,redraw)
     mime_t *a;
     int *redraw;
{
    char buf[VERY_LONG_STRING], tmp[STRING];
    int bytes=0, err, is_text;
    in_state_t state_in;
    FILE *f_in = NULL;
    int code;
    int br_flags = 0;
    
    struct string         * savefile = NULL;
    struct folder_browser * br       = new_browser(selection_file);
    WRITE_STATE handle =  NULL;

    in_state_clear(&state_in, STATE_in_file);
    
    if (a->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_save",
		   "Bad magic number");
    
    if (a->type == MIME_TYPE_APPLICATION &&
	istrcmp (a->subtype, "octet-stream") == 0) {
	
	/* See if there is a name=<...> field for the default filename */
	tmp[0] = '\0';
	if (a->type_opts &&
	    mime_get_param ("name", tmp, a->type_opts, sizeof(tmp))) {
	    char *p = strrchr (tmp, '/');
	    if (p)
		p++;
	    else
		p = tmp;
	    if (can_save(p)) {
		if (savefile)
		    free_string(&savefile);
		savefile = new_string2(system_charset,s2us(p));
	    }
	}
    }
    
    /* See if there is a filename=<...> field for the default filename */
    tmp[0] = '\0';
    if (a->disposition_opts &&
	mime_get_param ("filename", tmp, a->disposition_opts, sizeof(tmp))) {
	char *p = strrchr (tmp, '/');
	if (p)
	    p++;
	else
	    p = tmp;
	if (can_save(p)) {
	    if (savefile)
		free_string(&savefile);
	    savefile = new_string2(system_charset,s2us(p));
	}
    }
    

    code = file_browser(br,&savefile,redraw,word_save,
			CATGETS(elm_msg_cat, MeSet, MeToFile,
				"To file: "));

    if (!code) {
	ClearLine(elm_LINES-2);
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailNotSaved,
			  "Mail not saved."));
	
	goto clean;
    }

    if (savefile)
	free_string(&savefile);
    savefile = selection_name_dir(br);

    br_flags = give_dir_flags(br);

    DPRINT(Debug,4, (&Debug, 
		     "*** %S have flags:%s%s%s%s%s%s%s\n",
		     savefile,
		     br_flags & BROWSER_NODIR    ?   " NODIR":    "",
		     br_flags & BROWSER_NOFOLDER ?   " NOFOLDER": "",
		     br_flags & BROWSER_MARKED   ?   " MARKED":   "",
		     
		     br_flags & BROWSER_MAILFILE ?   " MAILFILE": "",
		     br_flags & BROWSER_SELECTED ?   " SELECTED": "",
		     br_flags & BROWSER_EXIST    ?   " EXIST"   : "",
		     !br_flags                   ?   " none"    : ""));
	   
	   if ((br_flags & BROWSER_EXIST) == 0) {
	/* Create it now ... */
	if (!create_selection_dir(br)) {
	    ClearLine(elm_LINES-2);
	    goto clean;
	}
    } else {
	/* Confirm overwrite */

	char * msg_buffer = NULL;
	char   answer;

	msg_buffer = elm_message(CATGETS(elm_msg_cat, ElmSet, 
					 ElmConfirmFileOverwrite,
					 "Overwrite existing file `%S'? (%c/%c) "),
				 savefile, 
				 *def_ans_yes, *def_ans_no);

	answer = want_to(msg_buffer, *def_ans_no, elm_LINES-2, 1);
	free(msg_buffer);

	if (answer != *def_ans_yes) {		   
	    ClearLine(elm_LINES-2);
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailNotSaved,
			      "Mail not saved."));
	    goto clean;
	}
    }

    ClearLine(elm_LINES-2);
  
    if (!(f_in = get_attachment(current_folder,a)))  {
	goto clean;
    }

    if (!prepare_write_folder(br,&handle))
	goto clean;
    else {
	out_state_t state_out;
	charset_t charset_vector[2];
	
	out_state_clear(&state_out, STATE_out_dir);

	/* Needed by STATE_out_dir */
	charset_vector[0] = RAW_BUFFER;
	charset_vector[1] = NULL;

	is_text = is_text_type (TYPE(a->type), a->subtype, a->encoding);

	set_in_state_file(f_in,&state_in);

	state_out.display_charset = charset_vector;
	set_out_state_dir(br,handle, &state_out);    
	state_out.prefix = NULL;
	state_out.displaying = 0;
	    
	if (a->encoding == ENCODING_BASE64 && !a->pathname)
	    base64_decode (&state_in, &state_out, a->length, is_text);
	else if (a->encoding == ENCODING_QUOTED && !a->pathname)
	    quoted_printable_decode (&state_in, &state_out, a->length, 
				     is_text);
	else {
	    if (a->encoding != ENCODING_NONE && a->encoding != ENCODING_7BIT &&
		a->encoding != ENCODING_8BIT && a->encoding != ENCODING_BINARY
		&& !a->pathname) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnsupportedEncoding,
				  "Unsupported encoding! Decode manually!"));
		sleep_message();
		
		/* save without decoding ! */
		is_text = 0;  /* We can't suppose it is text before decoding... */
	    }
	    while (bytes < a->length) {
		int chunk = VERY_LONG_STRING;
		int len;
		
		if (chunk > a->length - bytes)
		    chunk = a->length - bytes +1; /* one byte for terminating \0 */
		if ((len = state_getl (buf, chunk, &state_in)) <= 0)
		    break;
		bytes += len;
		if (is_text > 0) { /* replace CRLF with LF */
		    if (len >= 2 && buf[len-2] == '\r' && buf[len-1] == '\n') {
			buf[len-2] = '\n';
			buf[len-1] = '\0';
			len--;
		    }
		}
		state_put(buf,len,&state_out);
	    }
	}
	out_state_destroy(&state_out);
    }
    err = !end_write_folder(br,&handle);
    
    if (err)
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorSaving,
			  "Error saving file!"));
    else
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailSaved,
			  "Mail saved."));
    
 clean:
    in_state_destroy(&state_in);
        
    if (f_in)
	close_attachment(current_folder,f_in);

    if (savefile)
	free_string(&savefile);
    free_dir(&br);
    return;
}

static int attach_info P_((mime_t *ptr));

static void attach_edit (ptr)
     mime_t *ptr;
{
    int savetime;
    struct stat sb;
    char buf[STRING];
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_edit",
		   "Bad magic number");
    
    if (!ptr->pathname)
	return;
    
    if (-1 == stat (ptr->pathname, &sb)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantStat,
			  "Can't stat file!"));
	sleep_message();
	return;
    }
    if (strlen(ptr->pathname) + strlen(editor) > STRING-5)
	return;
    savetime = sb.st_mtime;
    Raw(OFF);
    elm_sfprintf (buf, sizeof buf,
		  FRM("%s %s"), editor, ptr->pathname);
    system_call (buf, 0);
    Raw(ON);
    if (stat (ptr->pathname, &sb) == -1 || sb.st_mtime != savetime)
	(void) attach_info (ptr);  /* update the information on this attachment */
    return;
}

static void attach_viewer (a)
     mime_t *a;
{
    char buf[LONG_STRING];

    struct header_rec tmp;
    FILE *tmpfp = NULL;
    struct stat sb;
    
    /* The next line is required so that mime_t_clear() doesn't call
     * mime_destroy() or free() on bad pointers
     */
    header_zero(&tmp);
    mime_t_clear (&tmp.mime_rec);
    
    if (a->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_viewer",
		   "Bad magic number");
    
    /* So that metapager() doesn't try to do special handling */
    tmp.status = MIME_MESSAGE;  /* Now I test again MIME_MESSAGE
				 * - K E H   <hurtta@dionysos.FMI.FI>
				 */

    tmp.header_charset = display_charset;

#ifdef USE_PGP
    tmp.pgp = 0;
#endif

    if (a->pathname) {
	ClearScreen();     /* Extra clear for attach_parse ... */
	
	if (can_open(a->pathname,"r") != 0) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorReadableUser,
			      "%.50s isn't readable by user!"), 
		      a->pathname);
	    sleep_message();
	    return;
	}

	tmpfp = fopen (a->pathname, "r");
	if (! tmpfp) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldOpenReading,
			      "Could not open file for reading!"));
	    sleep_message();
	    return;
	}
	
	tmp.mime_rec.flags = a->flags;
	tmp.mime_rec.type = a->type;
	strfcpy (tmp.mime_rec.subtype,a->subtype,
		 sizeof tmp.mime_rec.subtype);
	if (a->type_opts) {
	    tmp.mime_rec.type_opts = strmcpy(tmp.mime_rec.type_opts,
					     a->type_opts);
	}
	tmp.mime_rec.notplain = a->notplain;  
	
	tmp.mime_rec.disposition = DISP_INLINE;  /* Show it ! */
	tmp.offset = tmp.mime_rec.offset = 0;
	tmp.mime_rec.encoding = ENCODING_7BIT;
	stat(a->pathname, &sb);
	tmp.content_length = tmp.mime_rec.length = sb.st_size;
	attach_parse(&tmp, tmpfp);
	
    }
    else {
	
	/* Make copy of mime structure: */
	mime_t_copy(&(tmp.mime_rec),a);
	tmp.mime_rec.disposition = DISP_INLINE;  /* Show it ! */
	
	tmp.offset = tmp.mime_rec.offset = a -> offset;
	tmp.content_length = tmp.mime_rec.length = a->length;
#ifdef USE_PGP
	if (headers)
	    tmp.pgp = headers[current-1]->pgp;
#endif
  }

    /* This value is used for selecting external pager versus internal
     * pager.
     */
    if (tmp.lines < 1)
	tmp.lines = tmp.content_length / 60; 

    /* there is nothing to display! */
    if (tmp.mime_rec.length <= 0)
	goto fail;

    if (!mime_notplain(&(tmp.mime_rec)) || !have_metamail()) {
	
	if (tmpfp)
	    metapager (tmpfp, &tmp, FALSE);
	else if (current_folder) {
	    FILE *ZZ = folder_to_fd(current_folder,-1L);
	    if (ZZ) 
		metapager (ZZ,&tmp, FALSE);
	}
    } else {
	/* otherwise call metamail */
	
	if (!current_folder)
	    return;
	
	if (!a->pathname) {
	    char tmpfile[STRING], c;
	    FILE *out,*ZZ;
	    int bytes;
	    
	    elm_sfprintf (tmpfile, sizeof tmpfile,
			  FRM("%selm.%d"), temp_dir, getpid());
	    
	    out = safeopen(tmpfile);
	    
	    if (out == NULL) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorCreatTempfile,
				  "Error creating tempfile %s"),		 
			  tmpfile);
		sleep_message();
		goto fail;
	    }
	    /* copy the headers plus the message to a temp file */
	    ZZ =folder_to_fd(current_folder,a->begin_offset);
	    if (ZZ) {
		bytes = a->begin_offset - a->offset;
		while (bytes < a->length) {
		    c = fgetc (ZZ);
		    fputc (c, out);
		    bytes++;
		}
	    }
	    fclose (out);
	    elm_sfprintf (buf, sizeof buf,
			  FRM("%s -m Elm -p -z %s"), metamail_path,tmpfile);
	    
	    (void) elm_chown (tmpfile, userid, groupid);
	    
	}
	else
	    /* This is the case when we are viewing attachments for an outgoing
	     * message.
	     */
	    elm_sfprintf(buf,sizeof buf,
			 FRM("%s -m Elm -p -b -c %s/%s %s"), 
			 metamail_path,TYPE(a->type), a->subtype, a->pathname);
    

	/* Option -z causes that metamail deletes input file */
	
	Raw (OFF);
	ClearScreen();
	printf ("Executing: %s\n", buf);
	system_call (buf, SY_ENAB_SIGINT|SY_ENV_METAMAIL);
	PressAnyKeyToContinue();
	Raw (ON);
    }
    
 fail:
    mime_t_clear (&(tmp.mime_rec));
    
    if (tmpfp) {
	fclose (tmpfp);
    }
    
    return;
}

/* Check initial attachments */
int Check_attachments() 
{ 
    mime_t * tmp;
    for (tmp = attach_files; tmp; tmp = tmp -> next) {
	int need_enc;
	int is_text;
	char buf[40];
	
	need_enc = attach_info (tmp);
	if (need_enc < 0) {
	    return 0;
	}

	is_text = is_text_type (mime_types[tmp->type], 
				tmp->subtype, tmp->encoding);
	
	if (tmp->type == MIME_TYPE_TEXT && (need_enc & HAVE_8BIT) &&
	    !mime_get_param("charset",buf,tmp->type_opts,sizeof (buf)) &&
	    display_charset && display_charset->MIME_name)
	    add_parameter_t(tmp, "charset", display_charset->MIME_name,0);
	
	is_text = is_text_type (mime_types[tmp->type], 
				tmp->subtype, tmp->encoding);
	
	if (is_text < 0 && (tmp->encoding == ENCODING_QUOTED || 
			    tmp->encoding == ENCODING_BASE64)) {
      	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmStrucredNoEncoding,
			      "%.30s: Structured types don't allow encoding of data."),
		      tmp->pathname);
	    sleep_message();
	    
	    return 0;
	}
    }
    return 1;
}

/* Make initial attachment */
int Attach_it(pathname) 
     char *pathname;
{
    mime_t * tmp;
    
    if (access(pathname,READ_ACCESS) < 0) {
	int err = errno;
	lib_error(FRM("%.45s: %.33s"),
		  pathname,error_description(err));
	return 0;
    }
    
    tmp = (mime_t *) mime_t_alloc ();
    tmp->pathname = strmcpy (tmp->pathname, pathname);
    
    /* Default disposition for attachments: attachment */
    tmp->disposition = DISP_ATTACH;

    tmp->next = attach_files;
    attach_files = tmp;
    return 1;
}

static void forgot_previous P_((mime_t *att));
static void forgot_previous(att)
     mime_t *att;
{
    if (att->pathname) {		    
	if (att->unlink) {
	    if (unlink(att->pathname) == 0) {
		DPRINT(Debug,5, (&Debug, 
				 "attach_modify: Unlinking cache %s\n",
				 att->pathname));
	    }
	}
	
	free (att->pathname);
	att->pathname = NULL;
    }
}

static int attach_modify (att, new)
     mime_t *att;
     int new;
{
    int ret_value = FALSE;

    char buf[STRING];
    int update = TRUE, prompt = TRUE, need_enc = 0;
    int is_text = -1, ch = '\0';
    struct folder_browser * br = new_browser(selection_file);
    
    if (att->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_modify",
		   "Bad magic number");

    DPRINT(Debug,6, (&Debug, "attach_modify: att=%p, new=%d\n",att,new));
    
    buf[0] = '\0';
    
    if (new) {
	/* set the default charset */
	if (display_charset->MIME_name)
	    add_parameter_t (att, "charset", display_charset->MIME_name, 0);

	/* Default disposition for attachments: attachment */
	att->disposition = DISP_ATTACH;
	
	/* force a prompt for a filename */
	prompt = FALSE;
	ch = 'f';
	att->unlink = 0;
    }

main_loop:
  
    /* 1 if is text type (true)
     * 0 if not text type
     * -1 if can't be encoded (ie structured) Message/ or Multpart/
     */
    is_text = is_text_type (TYPE(att->type), att->subtype, att->encoding);

    for (;;) {
	if (update) {
	    struct string * Title = 
		format_string(CATGETS(elm_msg_cat, MeSet, MeAttachTitle,
				      "Attachment Configuration"));      
	    int add = 0;
	    int Width = elm_COLUMNS - 16;
	    if (Width < 40)
		Width = 40;
	    
	    ClearScreen ();

	    print_center(1,Title);
	    free_string(&Title);
	    
	    if (att->dispname) 
		PutLineX(3, 0, 
			 CATGETS(elm_msg_cat, MeSet, MeAttachFilename3,
				 "F)ilename                 : %.*S"),  
			 Width,att->dispname);
	    else if (att->unlink) 
		PutLineX(3, 0, 
			 CATGETS(elm_msg_cat, MeSet, MeAttachFilename1,
				 "Filename                  : %.*s"),  
			 Width,NONULL(att->pathname));
	    else
		PutLineX(3, 0, 
			 CATGETS(elm_msg_cat, MeSet, MeAttachFilename2,
				 "F)ilename                 : %.*s"),  
			 Width,NONULL(att->pathname));

	    if (!att->description)
		att->description = new_string(display_charset);
	    PutLineX(4, 0, 
		     CATGETS(elm_msg_cat, MeSet, MeAttachDescription,
			         "D)escription              : %.*S"),  
		     Width, att->description);

	    PutLineX (5, 0, 
		      CATGETS(elm_msg_cat, MeSet, MeAttachContentType,
			         "Content-T)ype             : %.15s/%.30s%s"), 
		      TYPE(att->type), att->subtype,
		      att->type_opts ? ";" : "");

	    if (att->type_opts) {
		PutLine0 (6+add, 28, att->type_opts);
		add++;
	    }
	    PutLineX(6+add, 0, 
		     CATGETS(elm_msg_cat, MeSet, MeAttachContentTE,
			         "Content-transfer-E)ncoding: %s"), 
		     ENCODING(att->encoding));
     
	    PutLineX (7+add, 0, 
		      CATGETS(elm_msg_cat, MeSet, MeAttachContentTE,
			         "Content-disP)osition      : %.15s%s"), 
		      DISPOSITION(att->disposition),
		      att->disposition_opts ? ";" : "");

	    if (att->disposition_opts) {
		PutLine0 (8+add, 28, att->disposition_opts);
		add++;
	    }
	    
	    if (is_text < 0)
		PutLineX(9+add, 0, 
			 CATGETS(elm_msg_cat, MeSet, MeAttachContentCRLF1,
				 "CRLF-conversions          : structured (direct content-encoding not allowed)"));
	    else if (is_text)
		PutLineX(9+add, 0, 
			 CATGETS(elm_msg_cat, MeSet, MeAttachContentCRLF2,
			         "CRLF-conversions          : Text (line orienteed, convert LF <-> CRLF)"));
	    else 
		PutLineX(9+add, 0, 
			 CATGETS(elm_msg_cat, MeSet, MeAttachContentCRLF3,
			         "CRLF-conversions          : Binary (no conversions)"));

	    update = FALSE;
	    show_last_error();
	}
	
	if (prompt) {
	    PutLineX (elm_LINES-2, 0, 
		      CATGETS(elm_msg_cat, MeSet, MeAttachEnter1,
			      "Enter letter or RETURN to quit: "));
	    ch = ReadCh(REDRAW_MARK);
	    clear_error();
	}
	
	if (ch == '\n' || ch == 'q' || ch == 'Q' || ch == 'x' || ch == 'X') {
	    ret_value = TRUE;
	    goto cleanup;
	} else if (ch == ctrl('L') || ch == REDRAW_MARK)
	    update = TRUE;
	else if (ch == 'f' || ch == 'F') {

	    struct string * tmpname = NULL;
	    int old_is_text = is_text;
	    int loop_count = 0;

	    int code;
	    
	    if (att->unlink && !att->dispname)  {
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmCantChangeFilename,
				  "You can't change filename!"));
		continue;
	    }
	    
	    if (!att->dispname) {
		if (att->pathname)
		    tmpname = new_string2(system_charset,s2us(att->pathname));
	    } else
		tmpname = dup_string(att->dispname);
	    
	    prompt = TRUE;

	    code = file_browser(br,&tmpname,&update,word_read,
				CATGETS(elm_msg_cat, MeSet, 
					MeAttachEnterFilename,
					"Filename: "));

	    if (!code) {
		if (!tmpname || string_len(tmpname) == 0) {
		    if (tmpname)
			free_string(&tmpname);		
		    continue;
		}

	    attach_failure:

		if (tmpname)
		    free_string(&tmpname);		

		update = TRUE;
		
		if (att->pathname) {		    
		    if (att->unlink) {
			if (unlink(att->pathname) == 0) {
			    DPRINT(Debug,5,
				   (&Debug,
				    "attach_modify: Unlinking cache %s\n",
				    att->pathname));
			}
		    }

		    free (att->pathname);
		    att->pathname = NULL;
		}
		if (new) {
		    ret_value = FALSE;
		    goto cleanup;
		} else
		    continue;
	    }

	    update = TRUE;

	    do {
		int br_flags = 0;
		old_is_text = is_text;

		if (tmpname)
		    free_string(&tmpname);

		tmpname = selection_name_dir(br);

		br_flags = give_dir_flags(br);

		    DPRINT(Debug,4, (&Debug, 
				     "*** %S have flags:%s%s%s%s%s%s%s\n",
				     tmpname,
				     br_flags & BROWSER_NODIR    ?   " NODIR":    "",
				     br_flags & BROWSER_NOFOLDER ?   " NOFOLDER": "",
				     br_flags & BROWSER_MARKED   ?   " MARKED":   "",
				     
				     br_flags & BROWSER_MAILFILE ?   " MAILFILE": "",
				     br_flags & BROWSER_SELECTED ?   " SELECTED": "",
				     br_flags & BROWSER_EXIST    ?   " EXIST"   : "",
				     !br_flags                   ?   " none"    : ""));
			   
		if ((br_flags & BROWSER_EXIST) == 0) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeAttachNotExist,
				      "File %S not exist"),
			      tmpname);		
		    goto attach_failure;
		}
	    
		/* Now forgot previous file .... */
		forgot_previous(att);
		
		if (att->dispname)
		    free_string(&att->dispname);
		att->dispname = tmpname;
		tmpname = NULL;
		
		if (!dir_make_ref(br, & (att->pathname), & (att->unlink),
				  is_text)) {
		    goto attach_failure;
		}

		/* Set some information about this attachment */
		need_enc = attach_info (att);
		if (need_enc < 0)
		    continue;
		
		/* 1 if is text type (true)
		 * 0 if not text type
		 * -1 if can't be encoded (ie structured) Message/ or Multpart/
		 */
		is_text = is_text_type (TYPE(att->type), att->subtype, 
					att->encoding);
		    
		if (old_is_text != is_text) {
		    DPRINT(Debug,1,
			   (&Debug,
			    "attach_modify: OPPS! CLRF encoding changed...\n"));
		}
	    } while (old_is_text != is_text && loop_count++ < 5);

	    
	    if (is_text > 0 && (need_enc & HAVE_BINARY)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmWarningBINARY,
				  "Warning: BINARY data? Check Content-Type!"));
	    }

	    if (is_text < 0 && (att->encoding == ENCODING_QUOTED || 
				att->encoding == ENCODING_BASE64)) {
		/* Reconsider encoding ... */
		ch = 'e';
		prompt = FALSE;
		update = TRUE;
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmStructuredNoencoding,
				  "Structured types don't allow encoding of data."));
		sleep_message();
		
		break;
	    } 
	    
	    /* now let the user do what they want */
	    if (new)
		prompt = TRUE;
	    
	}
	else if (ch == 'd' || ch == 'D') {
	    int code = optionally_enter2(&(att->description),
					 elm_LINES-2, 0, 
					 OE_APPEND_CURRENT|OE_REDRAW_MARK,
					 CATGETS(elm_msg_cat, MeSet, 
						 MeAttachEnterDescription,
						 "Description: "));

	    prompt = TRUE;
	    
	    if (REDRAW_MARK == code) {
		update = TRUE;
		prompt = FALSE;
		continue;
	    }
	    if (code != 0)
		continue;
	    
	    update = TRUE;
	}
	else if (ch == 't' || ch == 'T') {
	    int old_is_text = is_text;
	    int loop_count = 0;
	    int code;
	    
	    prompt = TRUE;
	    PutLineX (elm_LINES-2, 0, 
		      CATGETS(elm_msg_cat, MeSet, MeAttachEnterContentType,
			      "Content-Type: "));
	    att -> flags &= ~(MIME_RFC822);
	    att -> flags &= ~(MIME_MIXED);
	    att -> flags &= ~(MIME_DIGEST);
	    elm_sfprintf (buf, sizeof buf,
			  FRM("%.15s/%.30s"), 
			  TYPE(att->type), att->subtype);
	    if (att->type_opts) {
		int l;
		
		strfcat (buf, "; ", sizeof buf);
		l = strlen (buf);
		strfcpy (buf + l, att->type_opts, sizeof (buf) - l);
	    }
	    
	    code = optionally_enter (buf, elm_LINES-2, 14, OE_APPEND_CURRENT|
				     OE_REDRAW_MARK, sizeof buf);
	    if (REDRAW_MARK == code) {
		update = TRUE;
		prompt = FALSE;
		continue;
	    }
	    if (0 != code)
		continue;
	    
	    mime_get_content (buf, att);
	    if (att->type == MIME_TYPE_TEXT && (need_enc & HAVE_8BIT) &&
		!mime_get_param("charset",buf,att->type_opts,sizeof (buf)) &&
		display_charset->MIME_name)
		add_parameter_t(att, "charset", display_charset->MIME_name,0);

	    do {
		old_is_text = is_text;

		/* 1 if is text type (true)
		 * 0 if not text type
		 * -1 if can't be encoded (ie structured) Message/ or Multpart/
		 */
		is_text = is_text_type (mime_types[att->type], 
					att->subtype, att->encoding);
	    
		/* We reload file if it is user suplied */
		if (old_is_text != is_text && att->dispname) {
		    DPRINT(Debug,1,
			   (&Debug,
			    "attach_modify: OPPS! CLRF encoding changed...\n"));		    
		    forgot_previous(att);
		    
		    if (!select_dir_item(br,&(att->dispname))) {
			ch = 'f';
			prompt = FALSE;
			goto main_loop;
		    }
		    
		    if (!dir_make_ref(br, & (att->pathname), & (att->unlink),
				      is_text)) {
			ch = 'f';
			prompt = FALSE;
			goto main_loop;
		    }

		    /* Set some information about this attachment */
		    need_enc = attach_info (att);
		    if (need_enc < 0) {
			ch = 'f';
			prompt = FALSE;
			goto main_loop;
		    }
		}
	    } while (old_is_text != is_text && loop_count++ < 5 &&
		     att->dispname);


	    update = TRUE;
	    if (is_text < 0 && (att->encoding == ENCODING_QUOTED || 
				att->encoding == ENCODING_BASE64)) {
		/* Reconsider encoding ... */
		ch = 'e';
		prompt = FALSE;
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmStructuredNoencoding,
				  "Structured types don't allow encoding of data."));
		sleep_message();
	    } 
	}
	else if (ch == 'p' || ch == 'P') {
	    int code;
	    
	    prompt = TRUE;
	    PutLineX (elm_LINES-2, 0, 
		      CATGETS(elm_msg_cat, MeSet, MeAttachEnterContentDisp,
			      "Content-Disposition: "));
	    strfcpy (buf, DISPOSITION(att->disposition), sizeof buf);
	    if (att->disposition_opts) {
		int l;

		strfcat (buf,"; ", sizeof buf);
		l = strlen (buf);
		strfcpy (buf + l, att->disposition_opts, sizeof (buf) - l);
	    }
	    
	    code = optionally_enter (buf, elm_LINES-2, 21, 
				     OE_REDRAW_MARK|OE_APPEND_CURRENT,
				     sizeof buf);
	    if (REDRAW_MARK == code) {
		update = TRUE;
		prompt = FALSE;
		continue;
	    }
	    if (0 != code)
		continue;
	    mime_get_disposition (buf, att);
	    update = TRUE;
	}
	else if (ch == 'e' || ch == 'E') {
	    prompt = TRUE;
	    PutLineX (elm_LINES-2, 0, 
		      CATGETS(elm_msg_cat, MeSet, MeAttachEnterContentTE,
			      "Content-Transfer-Encoding: "));

	    Centerline (elm_LINES-1, "<SPACE> for next value, <RETURN> to accept.");
	    for (;;) {
		MoveCursor (elm_LINES-2, 27);
		CleartoEOLN ();

#define NEXT_ENCODING  { \
			   att->encoding++; \
			   if (att->encoding > 5) att->encoding = 1; \
			   continue; \
		       }

#ifndef USE_8BITMIME
		if (allow_no_encoding < 1) {
		    if (att->encoding == ENCODING_8BIT) {  
			/* Mailer won't support ! */
			/* TRY next encosing instead */
			NEXT_ENCODING;
		    }
		}
#endif
	
#ifndef USE_BINARYMIME
		if (allow_no_encoding < 2) {
		    if (att->encoding == ENCODING_BINARY) {  
			/* Mailer won't support ! */
			/* TRY next encoding instead */
			NEXT_ENCODING;
		    }
		}
#endif

		/* Don't allow 7bit if the file contains 8bit chars... 
		 * 7bit encoding is allowed if file includes control 
		 * characters 
		 */
		if (att->encoding == ENCODING_7BIT && (need_enc & HAVE_8BIT)) {
		    NEXT_ENCODING;
		}
		/* Don't allow 7bit or 8bit if the file required binary
		 * encoding according of Mime Standard.
		 */
		if ((att->encoding == ENCODING_7BIT || 
		     att->encoding == ENCODING_8BIT)
		    && (need_enc & HAVE_BINARY)) {
		    NEXT_ENCODING;
		}
		
		/* Don't allow encoding for Multipart/ and Message/ 
		 * See Mime Standard. Be carefull that don't create
		 * infinitive loop! */

		if (is_text < 0) {
		    static int again = 0;    /* Prevent looping */
		    if (att->encoding == ENCODING_QUOTED || 
			att->encoding == ENCODING_BASE64) {
			if (again == att->encoding) {
			    lib_error(CATGETS(elm_msg_cat, ElmSet,
					      ElmStructuredLeaf,
					      "Structured types must be encoded in leaf type!"));
			    sleep_message();
			    
			    /* prompt for new content-type */
			    prompt = FALSE;
			    ch = 't';
			    break;
			} else {
			    if (!again)
				again = att->encoding;
			    NEXT_ENCODING;
			}
		    } else
			again = 0;
		}

		Write_to_screen (FRM("%s"),ENCODING(att->encoding));
		ch = ReadCh(REDRAW_MARK);
		if (ch == '\n')
		    break;
		else if (ch == ' ') {
		    NEXT_ENCODING;
		}
		else if (ch == REDRAW_MARK) {
		    update = TRUE;
		    prompt = FALSE;
		    ch =  'e';
		    
		    break;
		}
	    }

#undef NEXT_ENCODING

	    ClearLine (elm_LINES-1);
	    /* 1 if is text type (true)
	     * 0 if not text type
	     * -1 if can't be encoded (ie structured) Message/ or Multpart/
	     */
	    is_text = is_text_type (mime_types[att->type], att->subtype, 
				    att->encoding);
	    update = TRUE;
	}
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmUnknownCommand2,			
			      "Unknown command."));
    }
    ret_value = FALSE;

 cleanup:
    if (br)
	free_dir(&br);

    DPRINT(Debug,5,
	   (&Debug,"attach_modify=%d\n",ret_value));
    return ret_value;
}

static void mime_guess_content_type (ptr)
     mime_t *ptr;
{
    /* This routine tries to guess the content-type of an attachment by looking
     * at the suffix of ptr->pathname.  It first looks to see if it can find
     * an entry in ~/.elm/mime.types, then in "system_mime_types", and failing
     * that, to a small list of builtin definitions.
     */
    int i, found = FALSE;
    char *p, *c, buf[LONG_STRING];
    FILE *fp = NULL;
    
    DPRINT(Debug,3, (&Debug, "mime_guess_content_type: pathname=%s\n",
		     ptr->pathname));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_guess_content_type",
		   "Bad magic number");
    
    /* Set the "filename" field for content-disposition */
    p = strrchr (ptr->pathname, '/');
    if (p)
	p++;
    else
	p = ptr->pathname;
    buf[0] = '\0';
    add_parameter (buf, "filename", p, sizeof (buf), 0);
    ptr->disposition_opts = strmcpy (ptr->disposition_opts, buf);
    
    /* Try to guess the content type of the data by the filename extension */
    p = strrchr (ptr->pathname, '.');
    if (! p)
	return;
    p++;
    for (i = 0; i < 3; i++) {
	DPRINT(Debug,3,
	       (&Debug, 
		"mime_guess_content_type: searching \"%s\", i=%d\n",
		p,i));
	if (i == 0) {
	    
	    /* First try the user's mime.types file */
	    
	    fp = fopen (user_mime_types, "r");
	    if (!fp)
		continue;
	}
	else if (i == 1) {
	    
	    /* If it wasn't there, try the system file... */
      
	    fp = fopen (system_mime_types, "r");
	    if (! fp)
		continue;
	}
	else {
	    /* Couldn't find user or system mime.types file,
	     * use these defaults...
	     */
	    if (istrcmp (p, "ps") == 0) {
		ptr->type = MIME_TYPE_APPLICATION;
		strfcpy (ptr->subtype, "postscript", sizeof ptr->subtype);
		if (ptr->type_opts) {
		    free (ptr->type_opts);
		    ptr->type_opts = NULL;
		}
	    }
	    else if (istrcmp (p, "gif") == 0 || istrcmp (p, "jpeg") == 0 ||
		     istrcmp (p, "tiff") == 0) {
		ptr->type = MIME_TYPE_IMAGE;
		strfcpy (ptr->subtype, p, sizeof ptr->subtype);
		if (ptr->type_opts) {
		    free (ptr->type_opts);
		    ptr->type_opts = NULL;
		}
	    }
	    DPRINT(Debug,3,
		   (&Debug, 
		    "mime_guess_content_type: built-in default \"%s\" as \"%s/%s\"\n", 
		    p, TYPE(ptr->type), ptr->subtype));
	}
	
	if (i < 2) {
	    int l = strlen(p);
	    while (fgets (buf, LONG_STRING, fp) != NULL) {
		int l1 = strlen(buf);
		
		if ('\n' == buf[l1 -1]) 
		    buf[l1 - 1] = '\0';
		else {
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeMimeTypeLine,
				      "mime.types: Too long line: %.30s..."),
			      buf);
		    break;
		}
		while (l1-- > 0 && whitespace(buf[l1]))
		    buf[l1] = '\0';
		
		c = buf;
		while (*c && whitespace ( *c)) /* skip leading whitespace */
		    c++;
		if (*c == '#') /* Skip comments */
		    continue;
		if (! *c)
		    continue;
		if (strincmp (c, p, l) == 0 &&
		    whitespace(c[l])) {
		    mime_get_content (c + l + 1, ptr);
		    DPRINT(Debug,3,
			   (&Debug,
			    "mime_guess_content_type: user defined \"%s\" as \"%s/%s\"\n", 
			    p, TYPE(ptr->type), ptr->subtype));
		    found = TRUE;
		    break;
		}
	    }
	    fclose (fp);
	    if (found)
		break;
	}
    }
    return;
}

static int attach_info (ptr)
     mime_t *ptr;
{
    struct stat sb;
    FILE *fp;
    int need_enc;
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_info",
		   "Bad magic number");

    if (stat (ptr->pathname, &sb) == -1) {
	if (errno == ENOENT)
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmFileNotExist,			
			      "That file %.20s does not exist!"),
		      ptr->pathname);
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmCantStatFile,			
			      "Could not stat file %.20s!"),
		      ptr->pathname);
	sleep_message();
	return (-1);
    }

    ptr->length = sb.st_size;
    
    if (can_open(ptr->pathname,"r") != 0) {
	lib_error(CATGETS(elm_msg_cat, ElmSet,
			  ElmNotReadableByUser,			
			  "%.50s isn't readable by user!"), 
		  ptr->pathname);
	ptr->pathname[0] = '\0';
	sleep_message();
	return (-1);
    }
    
    mime_guess_content_type (ptr);
    
    /* Figure out what the default encoding is... */
    
    lib_transient(CATGETS(elm_msg_cat, ElmSet,
			  ElmCheckingEncoding,			
			  "Checking %s..."), 
		  ptr->pathname);
    fp = fopen (ptr->pathname, "r");
    if (!fp) {
	lib_error(CATGETS(elm_msg_cat, ElmSet,
			  ElmCantOpenFile,			
			  "Can't open %s!"), 
		  ptr->pathname);
	sleep_message();
	return -1;
    }
    
    need_enc = needs_encoding (fp);
    
    if (need_enc & HAVE_CTRL)
	ptr->encoding = (need_enc & HAVE_BINARY) 
	    ? ENCODING_BASE64 : ENCODING_QUOTED;
    else if (need_enc & HAVE_BINARY) { 
	/* HAVE_BINARY, but not HAVE_CTRL so that have long lines! */
#ifdef USE_BINARYMIME
	ptr->encoding = ENCODING_BINARY;
#else
	ptr->encoding = ENCODING_QUOTED;
#endif
    }
    else if (need_enc & HAVE_8BIT) {
#ifdef USE_8BITMIME
	ptr->encoding = ENCODING_8BIT;
#else
	ptr->encoding = ENCODING_QUOTED;
#endif
    }
    fclose (fp);
    clear_error();  /* Remove reading ... -message */
    
    DPRINT(Debug,3, (&Debug, 
		     "attach_info: need_enc=%d, encoding=%d, pathname=%s\n",
		     need_enc,ptr->encoding,ptr->pathname));
    return (need_enc);
}

static void attach_header P_((mime_t *mt,
			      int num, int is_cur, int offset, int use_desc));

static void attach_header (mt, num, is_cur, use_desc, offset)
     mime_t *mt;
     int num, is_cur, offset, use_desc;
{
    /* Displays a header for a menu for the specified attachment. */
    char *Encoding = "???";
 
    char buf[LONG_STRING], buf2[LONG_STRING];
    int len, len2;
    int Width = elm_COLUMNS;     /* Protect arbitary big values of COLUMNS */
    if (Width > sizeof (buf)-1)
	Width = sizeof (buf)-1;
    
    if (mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_header",
		   "Bad magic number");

    Encoding = ENCODING(mt->encoding);

    
    /* TODO: FIXME */
    if (use_desc && mt->description) 
	elm_sfprintf (buf, sizeof buf,
		      FRM("%4d %-30.30S (%d) "),
		      num,
		      mt->description,
		      mt->length);
    else    
	elm_sfprintf (buf, sizeof buf,
		      FRM("%4d %-30.30s (%d) "),
		      num,
		      NONULL(mt->pathname),
		      mt->length);

    if (mt->length < 10)
	strfcat (buf, "   ", sizeof buf);
    else if (mt->length < 100)
	strfcat (buf, "  ", sizeof buf);
    else if (mt->length < 1000)
	strfcat (buf, " ", sizeof buf);
    elm_sfprintf (buf2, sizeof buf2,
		  FRM("%.15s/%.30s"), TYPE(mt->type), mt->subtype);
    strfcat (buf, buf2, sizeof buf);
    len = Width - strlen (buf);
    len2 = strlen (Encoding) + 3;
    if (len2 > len) {
	buf[Width-len2] = '\0';
	len = len2 = 0;
    }
    elm_sfprintf (buf2, sizeof buf2,
		  FRM("%s %*.*s[%s]"), buf, len-len2, len-len2, "", Encoding);
    
    if (is_cur && arrow_cursor) {
	buf2[0] = '-';
	buf2[1] = '>';
    }
    
    if (is_cur && has_highlighting && ! arrow_cursor) 
	StartInverse();
    PutLine0 (offset, 0, buf2);
    if (is_cur && has_highlighting && ! arrow_cursor) 
	EndInverse();
}

static void write_num P_((int num, int is_cur, int offset));

static void write_num(num, is_cur, offset)
     int num, is_cur,offset;
{
    char buf[10];
    
    elm_sfprintf (buf,sizeof buf,
		  FRM("%4d"),num);
    
    if (is_cur) {
	buf[0] = '-';
	buf[1] = '>';
    }
    PutLine0 (offset, 0, buf);
}

static void attach_low_menu P_((int rdonly));
static void attach_low_menu(rdonly)
     int rdonly;
{
    struct string * buf1 = NULL;

    if (rdonly) {
	buf1 = 
	    format_string(CATGETS(elm_msg_cat, MeSet, MeAttachMenuLowRd,
				  "p)rint, s)ave, v)iew subparts, q)uit"));
				     
    } else {
	buf1 = 
	    format_string(CATGETS(elm_msg_cat, MeSet, MeAttachMenuLow,
				  "a)dd, e)dit, d)elete, m)odify, p)rint, s)ave, v)iew subparts, q)uit"));
    }
    
    print_center(elm_LINES-1, buf1);

    free_string(&buf1);
}

mime_t * attach_menu (mt, rdonly, defcharset)
     mime_t *mt;
     int rdonly;
     charset_t defcharset;
{
    /* A generic attachment menu.  "rdonly" controls whether or not the list
     * of attachments "mt" may be edited.
     */
    
    mime_t **ptrs = NULL, *tmp, *ret = NULL;
    int ptr_len = 0, ptr_max = 0, i, cur = 0, offset = 3, ch;
    int update = TRUE;
    char buf[STRING];
    int top = 0;
    
    if (mt && mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"attach_menu",
		   "Bad magic number");
    
    /* Generate an array of pointers so it is easier to work with. */
    while (mt) {
	if (mt->magic != MIME_magic)
	    mime_panic(__FILE__,__LINE__,"attach_menu",
		       "Bad magic number (next chain)");
	
	if (ptr_len == ptr_max)
	    ptrs = (mime_t **) DynamicArray ((void **)ptrs, sizeof (mime_t *), 
					     &ptr_len, 5);
	ptrs[ptr_max++] = mt;
	mt = mt->next;
    }

    for (;;) {
	if (cur < top || cur >= top + elm_LINES-3-offset) {
	    if (cur < top) 
		top -= elm_LINES-3-offset;
	    if (cur >= top + elm_LINES-3-offset) 
		top += elm_LINES-3-offset;
	    if (top >= ptr_max)
		top = ptr_max - elm_LINES +3 +offset;
	    if (top < 0)
		top = 0;
	    update = TRUE;
	}
	if (update) {
	    ClearScreen ();
	    elm_sfprintf (buf, sizeof buf,
			  CATGETS(elm_msg_cat, ElmSet, ElmAttachMenu,
				  "Attachment Menu (%d attachments)"), 
			  ptr_max);
	    Centerline (1, buf);
	    
	    attach_low_menu(rdonly);

	    for (i = top; i < ptr_max && i < top + elm_LINES -3 - offset; i++)
		attach_header (ptrs[i], i + 1, i == cur, rdonly, offset + i - top);
	    update = FALSE;
	    show_last_error(); 
	}
	ClearLine (elm_LINES-2);
	PutLineX (elm_LINES-2, 0, 
		  CATGETS(elm_msg_cat, MeSet, MeAttachMenuPrompt,
			  "Attachments: "));
	ch = ReadCh (REDRAW_MARK|READCH_CURSOR);
	clear_error(); /* Clear the error message (from buffer also) */
	switch (ch) {
	case '-':
	case LEFT_MARK:
	case PAGEUP_MARK:
	    if ( ptr_max > 0 ) {
		attach_header (ptrs[cur], cur+1, FALSE, rdonly, 
			       offset + cur - top);
		cur -= elm_LINES -3;
		if (cur < 0)
		    cur = 0;
	    }
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmNoAttachments,
				  "There are no attachments!"));
	    break;
	case '+':
	case RIGHT_MARK:
	case PAGEDOWN_MARK:
	    if ( ptr_max > 0 ) {
		attach_header (ptrs[cur], cur+1, FALSE, rdonly, 
			       offset + cur - top);
		cur += elm_LINES -3;
		if (cur > ptr_max - 1) 
		    cur = ptr_max - 1;
	    }
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmNoAttachments,
				  "There are no attachments!"));
	    break;
	case 's':
	    if (ptr_max > 0) {
		attach_save (ptrs[cur],&update);
		if (!update)
		    attach_low_menu(rdonly);
	    } else
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmNoAttachments,			
				  "There are no attachments!"));
	    break;
	case ' ':
	case '\n':
	    if (ptr_max > 0) {
		attach_viewer (ptrs[cur]);
		update = TRUE;
	    }
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmNoAttachments,			
				  "There are no attachments!"));
	    break;
	case 'p':
	    if (ptr_max > 0)
		attach_print(ptrs[cur], defcharset);
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmNoAttachments,			
				  "There are no attachments!"));
	    
	    break;
	case 'v': /* Perhaps it is better that attachment meny shows whole
		   * structure -- but this is temporary hack.... */
	    if (ptr_max == 0) {
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmNoAttachments,			
				  "There are no attachments!"));
		break;
	    }
	    if (ptrs[cur]->parts) {
		attach_menu(ptrs[cur]->parts,TRUE, defcharset);
		update = TRUE;
	    }
	    break;
	case 'e':
	    if (! rdonly) {
		if (ptr_max == 0) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet,
				      ElmNoAttachments,			
				      "There are no attachments!"));
		    break;
		}
		attach_edit (ptrs[cur]);
		update = TRUE;
	    }
	    break;
	case 'd':
	    if (! rdonly) {
		if (ptr_max == 0) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet,
				      ElmNoAttachments,			
				      "There are no attachments!"));
		    break;
		}
		if (question_me) {
		    for(;;) {
			PutLine0(elm_LINES-2, 0, "Are you sure? (y/n): y");
			MoveCursor(elm_LINES-2, 21);
			ch = ReadCh(0);
			if (ch == 'y' || ch == '\n' || ch == 'n')
			    break;
		    }
		    ClearLine(elm_LINES-2);
		    if (ch == 'n')
			break;
		}
	    delete_it:
		/* List will be rebuild in exit */
		ptrs[cur]->next = NULL;
		mime_destroy (ptrs[cur]);
		ptrs[cur] = NULL;
		/* Shift the rest of the pointers down by one. */
		for (i = cur + 1; i < ptr_max; i++)
		    ptrs[i-1] = ptrs[i];
		ptrs[ptr_max-1] = NULL;
		ptr_max--;
		if ( cur >= ptr_max && cur > 0 ) cur = ptr_max-1;
		update = TRUE;
	    }
	    break;
	case 'a':
	    if (! rdonly) {
		tmp = (mime_t *) mime_t_alloc ();
		if (attach_modify (tmp, TRUE)) {
		    if (ptr_len == ptr_max)
			ptrs = (mime_t **) DynamicArray ((void **)ptrs, 
							 sizeof (mime_t *), 
							 &ptr_len, 5);
		    ptrs[ptr_max++] = tmp;
		}
		else {
		    /* List will be rebuild in exit */
		    tmp->next = NULL;
		    mime_destroy(tmp);
		}
		update = TRUE;
	    }
	    break;
	case 'm':
	    if (! rdonly) {
		if (ptr_max == 0) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet,
				      ElmNoAttachments,			
				      "There are no attachments!"));
		    break;
		}
		attach_modify (ptrs[cur], FALSE);
		/* If there is not pathname it is otherwise assumed to be 
		 * part from mailfile...!
		 */
		if (ptrs[cur]->pathname == NULL)
		    goto delete_it;
		update = TRUE;
	    }
	    break;
	case 'j':
	case 'J':
	case 'n':
	case DOWN_MARK:
	case ctrl('N'):
	    if (cur >= ptr_max - 1) {
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmLastAttachment,			
				  "You are on the last attachment!"));
		break;
	    }
	    if (arrow_cursor)
		write_num(cur+1,FALSE,offset + cur - top);
	    else
		attach_header (ptrs[cur], cur+1, FALSE, rdonly, 
			       offset + cur - top);
	    cur++;
	    if (cur < top + elm_LINES -3 - offset) {
		if (arrow_cursor)
		    write_num(cur+1,TRUE,offset + cur - top);
		else
		    attach_header (ptrs[cur], cur+1, TRUE, rdonly, 
				   offset + cur - top);
	    }
	    break;
	case 'k':
	case 'K':
	case ctrl('K'):
	case UP_MARK:
	    if (cur == 0) {
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmFirstAttachment,			
				  "You are on the first attachment!"));
		break;
	    }
	    if (arrow_cursor) 
		write_num(cur+1,FALSE,offset + cur - top);
	    else
		attach_header (ptrs[cur], cur+1, FALSE, rdonly, 
			       offset + cur - top);
	    cur--;
	    if (cur >= top) {
		if (arrow_cursor)
		    write_num(cur+1,TRUE,offset + cur - top);
		else	  
		    attach_header (ptrs[cur], cur+1, TRUE, rdonly, 
				   offset + cur - top);
	    }
	    break;
	case 'i':
	    if (! rdonly)
		break;
	    /* else fall through to next statement! */
	case 'q':
	case 'x':
	    if (ptrs == NULL)
		return NULL;
	    if (! rdonly) {
		/* The attachments might have been edited, so rebuild the 
		   list */
		if (ptr_max > 0) {
		    ret = tmp = ptrs[0];
		    
		    if (ret->magic != MIME_magic)
			mime_panic(__FILE__,__LINE__,"attach_menu",
				   "Bad magic number (ptrs[0])");
		    
		    for (i = 1; i < ptr_max; i++) {
			
			if (ptrs[i]->magic != MIME_magic)
			    mime_panic(__FILE__,__LINE__,"attach_menu",
				       "Bad magic number (ptrs[..])");
			
			tmp->next = ptrs[i];
			tmp = tmp->next;
		    }
		    tmp->next = NULL;
		}
		else
		    ret = NULL;
	    }
	    else
		ret = mt;
	    free (ptrs);
	    return ret;
	case ctrl('F'):
	    forget_passphrase();
	case ctrl('L'):
	case REDRAW_MARK:
	    update = TRUE;
	    break;
	default:
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmUnknownCommand3,		       
			      "Unknown command: %c"), 
		      ch);
	}
    }
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */
