static char rcsid[] = "@(#)$Id: string.c,v 1.23 2001/06/16 18:56:13 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.23 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@ozone.FMI.FI>
 *****************************************************************************/

#include "headers.h"
#include "s_me.h"
#include "cs_imp.h"

DEBUG_VAR(Debug,__FILE__,"charset");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}
static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}
static CONST char *cus2s P_((const unsigned char *str));
static CONST char *cus2s(str) 
     CONST unsigned char *str;
{
    return (CONST char *)str;
}



int charset_valid_magic(magic)
     charset_t magic;
{
    if (!magic) {
	DPRINT(Debug,1,(&Debug,
			"charset_valid_magic=0: magic=NULL\n"));	      	
	return 0;
    }
    
    if (&cs_ascii    == magic->charset_type ||
	&cs_unknown  == magic->charset_type ||
	&cs_onebyte  == magic->charset_type ||
	&cs_iso646   == magic->charset_type ||
	&cs_utf8     == magic->charset_type ||
	&cs_utf7     == magic->charset_type ||
	&cs_imap     == magic->charset_type) {

	if (magic->map_info &&
	    magic->map_info->map_type != magic->charset_type) {

	    DPRINT(Debug,1,(&Debug,
			    "charset_valid_magic=0: magic=%p (map_info = %p)\n",
			    magic,magic->map_info));	      
	    
	    return 0;
	}
	return 1;
    }

    DPRINT(Debug,1,(&Debug,
		    "charset_valid_magic=0: magic=%p (charset_type = %p)\n",
		    magic,magic->charset_type));	      
  
    return 0;
}

int verify_string(str)
     struct string *str;
{
     
     if (!str) {
	 DPRINT(Debug,1,(&Debug,"verify_string=0:  FAILURE  str = NULL\n"));
	 return 0;
     }

     if (!str->string_type) {
	 DPRINT(Debug,1,(&Debug,
			 "verify_string=0:  FAILURE  str->string_type = NULL  (str=%p)\n",
			 str));
	  return 0;
     }

     if (!charset_valid_magic(str->string_type)) {
	 DPRINT(Debug,1,(&Debug,
			 "verify_string=0:  FAILURE  Bad magic  (str=%p)\n",
			 str));
       return 0;
     }
     return 1;
}

static struct string * malloc_string P_((charset_t set));
static struct string * malloc_string(set) 
     charset_t set;
{
    struct string * res = safe_malloc(sizeof (struct string));

    /* defined in hdrs/defs.h */
    bzero((void *)res,sizeof (struct string));

    if (!charset_valid_magic(set)) 
	panic("STRING PANIC",__FILE__,__LINE__,"malloc_string",
	      "Bad magic number (set)",0);

    /* Initialize associated map if not done already ... */       
    if (set->map_info && !set->map_info->map_initialized)
	set->map_info->map_init_it(set->map_info);

    res->string_type = set;
    res->p = safe_malloc(sizeof (struct str_private_data));
    /* defined in hdrs/defs.h */
    bzero((void *)res->p,sizeof (struct str_private_data));
    res->p->len    = 0;
    res->p->state  = NULL;

    res->string_type->charset_type->cs_init_it(res);
    return res;
}

/* ------------------------------------------------------------------------- */

struct string * new_string(set)
     charset_t set;
{
    struct string *res;

    DPRINT(Debug,60,(&Debug,"new_string(set=%p '%s')\n",
		     set,
		     set->MIME_name ? set->MIME_name : "<none>"));

    res = malloc_string(set);
    
    DPRINT(Debug,60,(&Debug,"new_string=%p  (type=%p '%s')\n",
		     res,
		     res->string_type->charset_type,
		     res->string_type->charset_type->type_name));

    return res;
}

struct string * new_string2 P_((charset_t set, 
				CONST unsigned char *data));
struct string * new_string2(set,data)
     charset_t set; 
     CONST unsigned char *data;
{
    struct string * res;

    DPRINT(Debug,60,(&Debug,
	       "new_string2: set=%p '%s'\n",
	       set,
	       set->MIME_name ? set->MIME_name : "<none>"));
    if (set == display_charset || set == system_charset) {
	DPRINT(Debug,61,(&Debug,
			 "new_string2 -- data=%s\n",data));
    }

    res = malloc_string(set);
    res->string_type->charset_type->
	cs_add_streambytes_to_it(res,
				 strlen(cus2s(data)),
				 data);

    DPRINT(Debug,60,(&Debug,
		     "new_string2=%p  (type=%p '%s'): len=%d\n",
		     res,
		     res->string_type->charset_type,
		     res->string_type->charset_type->type_name,
		     res->p->len));
    return res;
}

void free_string(str)
     struct string **str;
{

    if (!charset_valid_magic((*str)->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"free_string",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,"free_string(*str=%p) ('%s'; type=%p '%s')\n",
		     (*str),
		     (*str)->string_type->MIME_name ? 
		     (*str)->string_type->MIME_name : "<none>",
		     (*str)->string_type->charset_type,
		     (*str)->string_type->charset_type->type_name));
    
    if ((*str)->p->state && 
	(*str)->p->state->charset != (*str)->string_type)
	panic("STRING PANIC",__FILE__,__LINE__,"free_string",
	      "Bad magic number (state charset)",0);

    (*str)->string_type->charset_type->cs_free_it(*str);

    free((*str)->p);
    (*str)->p = NULL;
    (*str)->string_type = NULL;

    free(*str);
    *str = NULL;
}

int add_streambyte_to_string(str,ch)
     struct string *str; 
     int ch;
{
    int r;

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambyte_to_string",
	      "Bad magic number (string type)",0);

    if (str->p->state && 
	str->p->state->charset != str->string_type)
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambyte_to_string",
	      "Bad magic number (state charset)",0);

    DPRINT(Debug,60,(&Debug,
		     "add_streambyte_to_string: str=%p ('%s'; type=%p '%s')\n",
		     str,	       
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    if (str->string_type == display_charset || 
	str->string_type == system_charset) {
	DPRINT(Debug,61,(&Debug,
			 "add_streambyte_to_string -- ch=%d '%c'\n",ch,ch));    
    } else {
	DPRINT(Debug,61,(&Debug,
			 "add_streambyte_to_string -- ch=%d\n",ch));    
    }


    r = str->string_type->charset_type->cs_add_streambyte_to_it(str,ch);
    
      DPRINT(Debug,60,(&Debug,
		       "add_streambyte_to_string=%d\n",
		       r));

    return r;
}

int add_streambytes_to_string(str,count,data)
     struct string *str; 
     int count; 
     CONST unsigned char *data;
{
    int r;

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambytes_to_string",
	      "Bad magic number (string type)",0);

    if (str->p->state && 
	str->p->state->charset != str->string_type)
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambytes_to_string",
	      "Bad magic number (state charset)",0);

    DPRINT(Debug,60,(&Debug,
		     "add_streambytes_to_string: str=%p, count=%d, data=%p ('%s'; type=%p '%s')\n",
		     str,count,data,	       
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    if (str->string_type == display_charset || 
	str->string_type == system_charset) {
	DPRINT(Debug,61,(&Debug,
			 "add_streambytes_to_string -- data='%.*s'\n",
			 count,data));    
    } 

    r = str->string_type->charset_type->cs_add_streambytes_to_it(str,
								 count,data);

    DPRINT(Debug,60,(&Debug,"add_streambytes_to_string=%d\n",r));

    return r;
}

static CONST unsigned char * csUs P_((const char *str));
static CONST unsigned char * csUs(str)
     CONST char *str;
{
    return (CONST unsigned char *)str;
}

static CONST char * cUss P_((const unsigned char *str));
static CONST char * cUss(str)
     CONST unsigned char *str;
{
    return (CONST char *)str;
}

static void add_name_to_string P_((struct string *str,
				   CONST unsigned char *ascii));

static void add_name_to_string(str,ascii)
     struct string *str; 
     CONST unsigned char *ascii;
{
    int l = strlen(cUss(ascii));
    int i;
    uint16 * vector = safe_malloc ( (l) * sizeof (uint16));

    /* Ascii and UNICODE have save codes ... */
    
    for (i = 0; i < l ; i++)
	/* ascii string is unsigned ... */
	if (ascii[i] < 128) 
	    vector[i] = ascii[i];
	else
	    vector[i] = '?';
    
    str->string_type->charset_type->cs_add_unicodedata_to_it(str,l,vector);
    free(vector);
}

void add_unicode_to_string(str,len,vector)
     struct string *str;
     int len;
     CONST uint16 *vector;
{
    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"add_aunicode_to_string",
	      "Bad magic number (string type)",0);

    if (len < 0)
	panic("STRING PANIC",__FILE__,__LINE__,"add_aunicode_to_string",
	      "Bad vector len",0);

    DPRINT(Debug,60,(&Debug,
		     "add_unicode_to_string(%p,len=%d)  ('%s'; type=%p '%s')\n",
		     str,len,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    str->string_type->charset_type->cs_add_unicodedata_to_it(str,len,vector);
    
    DPRINT(Debug,60,(&Debug,
		     "add_unicode_to_string: len = %d\n",str->p->len));	     
}

struct string * cat_strings(str1,str2,printind)
     CONST struct string *str1; 
     CONST struct string *str2;
     int printind;
{
    struct string *ret = NULL;

    if (!charset_valid_magic(str1->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad magic number (string type)",0);

    if (!charset_valid_magic(str2->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,
		     "cat_strings(%p,%p) ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
	       str1,str2,
	       str1->string_type->MIME_name ? 
	       str1->string_type->MIME_name : "<none>",
	       str1->string_type->charset_type,
	       str1->string_type->charset_type->type_name,
	       str1->p->len,
	       str2->string_type->MIME_name ? 
	       str2->string_type->MIME_name : "<none>",
	       str2->string_type->charset_type,
	       str2->string_type->charset_type->type_name,
	       str2->p->len));

    /* Either one is empty, produce other,
       but not if other is not printable and printind is set.
     */
    if (str1->p->len == 0 &&
	(!printind || 
	 0 != (charset_properties(str2 -> string_type) & CS_mapping))) {
	ret = malloc_string(str2 -> string_type);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);
    } else if (str2->p->len == 0 &&
	       (!printind || 
		0 != (charset_properties(str1 -> string_type) & CS_mapping))) {
	ret = malloc_string(str1 -> string_type);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);

    } else if (str1 -> string_type == str2 -> string_type) {
	/* 1) Same CHARSET */

	ret = malloc_string(str1 -> string_type);

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);

    } else if (str1 -> string_type -> charset_type == 
	       str2 -> string_type -> charset_type &&
	       charset_superset_of(str1 -> string_type,str2 -> string_type)) {
	/* 2)  CHARSET of str1 is superset of CHARSET of str2 */

	ret = malloc_string(str1 -> string_type);

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);

    } else if (str1 -> string_type -> charset_type == 
	       str2 -> string_type -> charset_type &&
	       charset_superset_of(str2 -> string_type,str1 -> string_type)) {
	/* 3)  CHARSET of str2 is superset of CHARSET of str1 */

	ret = malloc_string(str2 -> string_type);

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);

    } else if (printind &&
	       0 != (charset_properties(str1 -> string_type) & CS_mapping) &&
	       0 == (charset_properties(str2 -> string_type) & CS_mapping) &&
	       str2 -> string_type -> MIME_name) {
	/* Add just str1 and indicate that 2 is not mappable */
	
	ret = malloc_string(str1 -> string_type);

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);
	add_name_to_string(ret,csUs("[?"));
	add_name_to_string(ret,csUs(str2 -> string_type -> MIME_name));
	add_name_to_string(ret,csUs("?]"));
    } else if (printind &&
	       0 != (charset_properties(str2 -> string_type) & CS_mapping) &&
	       0 == (charset_properties(str1 -> string_type) & CS_mapping) &&
	       str1 -> string_type -> MIME_name) {
	/* Add just str2 and indicate that 1 is not mappable */
	
	ret = malloc_string(str2 -> string_type);

	add_name_to_string(ret,csUs("[?"));
	add_name_to_string(ret,csUs(str2 -> string_type -> MIME_name));
	add_name_to_string(ret,csUs("?]"));
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);
    } else {
	/* 4) Unrelated CHARSETs -- so we try use display charset instead */
	int i;
	int l = 0;
	charset_t  target_set = display_charset;
 	uint16 * vector = safe_malloc ( (str1->p->len + str2->p->len) *
					sizeof (uint16));


	if (0 == (charset_properties(target_set) & CS_mapping)) {
	    DPRINT(Debug,60,(&Debug,
		       "cat_strings: Display charset do not have mapping info, using ASCII as target set\n"));
	    target_set = ASCII_SET;
	}

	ret = malloc_string(target_set);

	for (i = 0; i < str1->p->len; i++) {
	    int found;
	    uint16 c = 
		str1->string_type->charset_type->
		cs_give_unicode_from_it(str1,i,&found);
	    vector[l++] = c;
	}

	for (i = 0; i < str2->p->len; i++) {
	    int found;
	    uint16 c = 
		str2->string_type->charset_type->
		cs_give_unicode_from_it(str2,i,&found);
	    vector[l++] = c;
	}

	ret->string_type->charset_type->cs_add_unicodedata_to_it(ret,l,vector);
	free(vector);
    }

    DPRINT(Debug,60,(&Debug,
		     "cat_strings=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));

    return ret;
}

int string_len(str)
     struct string *str;
{
    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"string_len",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,"string_len(%p)=%d ('%s'; type=%p '%s')\n",
		     str,str->p->len,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    return str->p->len;
}

int string_cmp(str1,str2,unknown_val)
     struct string *str1; 
     struct string *str2;
     int unknown_val;
{
    int ret = 0;
    int not_found = 0;
    int i;

    if (!charset_valid_magic(str1->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad magic number (string type)",0);

    if (!charset_valid_magic(str2->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,
		     "string_cmp(%p,%p,unknown_val=%d) ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
		     str1,str2,unknown_val,
		     str1->string_type->MIME_name ? 
		     str1->string_type->MIME_name : "<none>",
		     str1->string_type->charset_type,
		     str1->string_type->charset_type->type_name,
		     str1->p->len,
		     str2->string_type->MIME_name ? 
		     str2->string_type->MIME_name : "<none>",
		     str2->string_type->charset_type,
		     str2->string_type->charset_type->type_name,
		     str2->p->len));


    /* Make unicode compare first to make comparision to stable */

    for (i = 0; i < str1->p->len && i < str2->p->len; i++) {
	int found1,found2;
	uint16 c1 = 
	    str1->string_type->charset_type->
	    cs_give_unicode_from_it(str1,i,&found1);
	uint16 c2 = 
	    str2->string_type->charset_type->
	    cs_give_unicode_from_it(str2,i,&found2);

	if (!found1 || !found2) {
	    ret = 0;
	    not_found = 1;
	    break;
	} else if (c1 < c2) 
	    ret = -1;
	else if (c1 > c2)
	    ret = 1;

	if (ret != 0)	    
	    break;
    }

    DPRINT(Debug,61,(&Debug,
		     "string_cmp: i=%d, str1 len=%d, str2 len=%d, not_found=%d\n",
		     i,str1->p->len,str2->p->len,not_found));

    if (!not_found) {
	if (0 == ret && i < str1->p->len)
	    ret = 1;
	if (0 == ret && i < str2->p->len)
	    ret = -1;
	
	DPRINT(Debug,61,(&Debug,
		   "string_cmp: unicode cmp ret=%d\n",ret)); 
    }

    if (0 == ret && str1 -> string_type == str2 -> string_type) {
	ret = str1 -> string_type->charset_type->cs_cmp_it(str1,str2);
	DPRINT(Debug,61,(&Debug,
			 "string_cmp: cs_cmp_it ret=%d\n",ret)); 
	
    } else if (0 == ret && not_found) {
	DPRINT(Debug,61,(&Debug,"string_cmp: Unable to compare values!\n"));
	ret = unknown_val;
    }

    DPRINT(Debug,60,(&Debug,"string_cmp=%d\n",ret));
    return ret;
}

struct string * dup_string(str)
     CONST struct string *str;
{
    struct string *ret;
    
    DPRINT(Debug,60,(&Debug,"dup_string: str=%p ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    ret = malloc_string(str -> string_type);
    ret->string_type->charset_type->cs_add_intdata_to_it(ret,str);

    DPRINT(Debug,60,(&Debug,"dup_string=%p ('%s'; type=%p '%s')\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name));

    return ret;
}

struct string * convert_string(set,str,printind)
     charset_t set;
     CONST struct string *str;
     int printind;
{
    struct string *ret = malloc_string(set);

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,
		     "convert_string(set=%p '%s',str=%p)  ('%s', type=%p '%s', len=%d)\n",
		     set,
		     set->MIME_name ? set->MIME_name : "<none>",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
    

    if (set == str->string_type) {
	/* 1) Same charset */
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str);
    } else if (printind &&
	       0 != (charset_properties(set) & CS_mapping) &&
	       0 == (charset_properties(str -> string_type) & CS_mapping) &&
	       str -> string_type -> MIME_name) {
	/* Just indicate that str is not mappable */
	
	add_name_to_string(ret,csUs("[?"));
	add_name_to_string(ret,csUs(str -> string_type -> MIME_name));
	add_name_to_string(ret,csUs("?]"));

    } else {
	/* 2) Different charset */
	int i;
	int l = 0;

	uint16 * vector = safe_malloc ( (str->p->len) * sizeof (uint16));
	for (i = 0; i < str->p->len; i++) {
	    int found;
	    uint16 c = 
		str->string_type->charset_type->
		cs_give_unicode_from_it(str,i,&found);
	    vector[l++] = c;
	}

	ret->string_type->charset_type->cs_add_unicodedata_to_it(ret,l,vector);
	free(vector);
    }

    DPRINT(Debug,60,(&Debug,
		     "convert_string=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));

    return ret;
}

unsigned char *stream_from_string(str, printable,terminal)
     CONST struct string *str;
     int printable;
     screen_info_p terminal; 
{
    int reslen;
    unsigned char * ret;
    char *ret0 = NULL;
    
    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"stream_from_string",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,"stream_from_string: ret=%p ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    if (terminal && str->string_type->iso2022_info) 
	ret0 = terminal_charset_post_init(terminal,
					  str->string_type->iso2022_info);


    /* FIXME: What terminal_charset_post_init is supposed to do?
     *        if it initalization for return string then it's
     *        values should be catenated before actual string returned
     *
     * ???????????? 
     */
    	    
    ret = str->string_type->charset_type->cs_stream_from_it(str,
							    printable,
							    terminal,
							    &reslen);
    DPRINT(Debug,60,(&Debug,
		     "stream_from_string -- reslen=%d\n",reslen));

    if (str->string_type == display_charset) {
	DPRINT(Debug,61,(&Debug,
			 "stream_from_string -- ret='%s'%s\n",
			 ret,
			 ret0 ? " (without post init)" : ""));
    }


    if (ret0) {
	int i;
	DPRINT(Debug,8,(&Debug,"Post init terminal charset %s (",
			str->string_type->MIME_name ? 
			str->string_type->MIME_name : 
			"<no MIME name>"));
	for (i = 0; ret0[i]; i++) {
	    DPRINT(Debug,8,(&Debug,"%s%02X",
			    i ? " " : "", ret0[i]));
	}
	DPRINT(Debug,8,(&Debug,")\n"));

	ret = s2us(strmcat(us2s(ret),ret0));
	free(ret0);
    }

    DPRINT(Debug,60,(&Debug,"stream_from_string=%p\n",
	       ret));

    return ret;
}

/* result is malloced */
void bytestream_from_string(str,res,reslen)
     const struct string *str;
     char **res;
     int *reslen;
{
    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"bytestream_from_string",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,
		     "bytestream_from_string: str=%p ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));


    *res = us2s(str->string_type->charset_type->cs_stream_from_it(str,0,NULL,
								  reslen));

    DPRINT(Debug,60,(&Debug,
		     "bytestream_from_string: *res=%p, reslen=%d\n",
		     *res,reslen));
    
}


unsigned char *streamclip_from_string(str,pos,len,terminal)
     CONST struct string *str;
     int *pos; 
     int len; 
     screen_info_p terminal;   
{
    int p = *pos;
    unsigned char * ret;
    char *ret0 = NULL;

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"streamclip_from_string",
	      "Bad magic number (string type)",0);
	
    if (terminal && str->string_type->iso2022_info) 
	ret0 = terminal_charset_post_init(terminal,
					  str->string_type->iso2022_info);

    /* FIXME: What terminal_charset_post_init is supposed to do?
     *        if it initalization for return string then it's
     *        values should be catenated before actual string returned
     *
     * ???????????? 
     */
    
    DPRINT(Debug,60,(&Debug,
		     "streamclip_from_string(%p,%d,%d)    ('%s'; type=%p '%s')\n",
		     str,p,len,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    
    ret = 
	str->string_type->charset_type->cs_streamclip_from_it(str,pos,len,
							      terminal);

    if (str->string_type == display_charset) {
	DPRINT(Debug,61,(&Debug,
			 "streamclip_from_string -- ret='%s'%s\n",
			 ret,
			 ret0 ? " (without post init)" : ""));
    }

    if (ret0) {
	int i;
	DPRINT(Debug,8,(&Debug,
			"Post init terminal charset %s (",
			str->string_type->MIME_name ? 
		  str->string_type->MIME_name : 
			"<no MIME name>"));
	for (i = 0; ret0[i]; i++) {
	    DPRINT(Debug,8,(&Debug,"%s%02X",
			    i ? " " : "", ret0[i]));
	}
	DPRINT(Debug,8,(&Debug,")\n"));
	
	ret = s2us(strmcat(us2s(ret),ret0));
	free(ret0);
    }
    
    DPRINT(Debug,60,(&Debug,
		     "streamclip_from_string [%d(..%d)] =%p ('%s'; type=%p '%s')\n",
		     p,*pos,ret,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    
    return ret;
}

struct string *clip_from_string(str,pos,len)
     CONST struct string *str;
     int *pos; 
     int len;
{
    struct string *ret = malloc_string(str->string_type);
    
    DPRINT(Debug,60,(&Debug,
		     "clip_from_string(%p,%d,%d)  ('%s'; type=%p '%s')\n",
		     str,*pos,len,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    str->string_type->charset_type->cs_clip_from_it(ret,str,pos,len);

    DPRINT(Debug,60,(&Debug,
		     "clip_from_string=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));
    
    return ret;
}

struct string *ascify_string(str)
     CONST struct string *str;
{
    int can_ascii;
    charset_t ascii_ptr = MIME_name_to_charset("US-ASCII",0);
    struct string *ret;

    if (!ascii_ptr)
	panic("STRING PANIC",__FILE__,__LINE__,"ascify_string",
	      "US-ASCII not found",0);

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"ascify_string",
	      "Bad magic number (string type)",0);

        DPRINT(Debug,60,(&Debug,
			 "ascify_string(%p) ('%s'; type=%p '%s', len=%d)\n",
			 str,
			 str->string_type->MIME_name ? 
			 str->string_type->MIME_name : "<none>",
			 str->string_type->charset_type,
			 str->string_type->charset_type->type_name,
			 str->p->len));

    can_ascii = str->string_type->charset_type->cs_can_ascii_it(str);

    if (can_ascii) {
	int i;
	int l = 0;
	uint16 * vector = safe_malloc ( (str->p->len) * sizeof (uint16));

	ret = malloc_string(ascii_ptr);
	

	for (i = 0; i < str->p->len; i++) {
	    int found;
	    uint16 c = 
		str->string_type->charset_type->
		cs_give_unicode_from_it(str,i,&found);
	    vector[l++] = c;
	}
	
	ret->string_type->charset_type->cs_add_unicodedata_to_it(ret,l,vector);
	free(vector);
    } else {
	ret = malloc_string(str -> string_type);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str);   
    }

    DPRINT(Debug,60,(&Debug,
		     "ascify_string=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));

    return ret;
}

int can_ascii_string(str)
     CONST struct string *str;
{
    int ret;
    
    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"can_ascii_string",
	      "Bad magic number (string type)",0);
    
    ret= str->string_type->charset_type->cs_can_ascii_it(str);
    
    DPRINT(Debug,60,(&Debug,
		     "can_ascii_string(%p)=%d ('%s'; type=%p '%s', len=%d)\n",
		     str,ret,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
    
    return ret;
}

void add_ascii_to_string(str,ascii)
     struct string *str; 
     CONST unsigned char *ascii;
{

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"add_ascii_to_string",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,
		     "add_ascii_to_string(%p)  ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    
    DPRINT(Debug,61,(&Debug,
		     "add_ascii_to_string -- ascii=%s\n",ascii));

    add_name_to_string(str,ascii);

    DPRINT(Debug,60,(&Debug,
		     "add_ascii_to_string: len = %d\n",str->p->len));	   
}

void fill_ascii_to_string(str,count,ascii)
     struct string *str; 
     int count; 
     unsigned int ascii;
{
    int i;
    uint16 * vector;

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
	      "Bad magic number (string type)",0);
    if (count < 1)
	panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
	      "bad count",0);

    if (str->p->state) {
	if (str->p->state->charset != str->string_type)
	    panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
		  "Bad magic number (state charset)",0);
	free_state_internal(&(str->p->state));
    }

    vector = safe_malloc ( count * sizeof (uint16));

    /* Ascii and UNICODE have save codes ... */
    
    for (i = 0; i < count ; i++)
	/* ascii is unsigned ... */
	if (ascii < 128) 
	    vector[i] = ascii;
	else
	    vector[i] = '?';  /* XXX */


    str->string_type->charset_type->cs_add_unicodedata_to_it(str,count,vector);
    free(vector);

    DPRINT(Debug,60,(&Debug,
		     "fill_ascii_to_string(%p,%d,%d)  ('%s'; type=%p '%s'): len=%d\n",
		     str,count,ascii,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
}

struct string *skip_ascii_head_from_string(str,ascii,ignore_case)
     CONST struct string *str;
     CONST unsigned char *ascii;
     int ignore_case;
{
    struct string *ret = malloc_string(str->string_type);
    int X = 0;
    int x;

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"skip_ascii_head_from_string",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,
		     "skip_ascii_head_from_string(%p) ('%s'; type=%p '%s', len=%d)\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
    DPRINT(Debug,60,(&Debug,
		     "skip_ascii_head_from_string -- ascii=%s\n",ascii));
    
    while(1) {
	int i;
	CONST unsigned char * p = ascii;

	DPRINT(Debug,61,(&Debug,
			 "skip_ascii_head_from_string -- starting pos %d\n",
			 X));


	for (i = X;  
	     /* ascii string is unsigned ... */
	     *p && *p < 128 && i < str->p->len; 
	     i++, p++) {

	    int found;
	    uint16 c =
		str->string_type->charset_type->
		cs_give_unicode_from_it(str,i,&found);
	    unsigned char c1 = *p;

	    if (!found) {
		DPRINT(Debug,61,(&Debug,
				 "skip_ascii_head_from_string -- failing at %d -- no unicode\n",
				 i));
		break;
	    }
	    if (ignore_case) {
		c = unicode_ch(c,UOP_lowercase);
		if (!c) {	
		    DPRINT(Debug,61,(&Debug,
				     "skip_ascii_head_from_string -- failing at %d -- no lowercasing\n",
				     i));
		    break;
		}
	    }

	    if (ignore_case && c1 >= 0x41 && c1 <= 0x5A) {
	        c1 = ( c1 - 0x41) + 0x61;
	    }

	    if (c1 != c) {
		DPRINT(Debug,61,(&Debug,
				 "skip_ascii_head_from_string -- failing at %d -- mismatch %02X (%c) <> %04X\n",
				 i,c1,c1,c));
		break;
	    }
	}
	if (*p) {
	    DPRINT(Debug,61,(&Debug,
			     "skip_ascii_head_from_string -- hit end at %d\n",
			     i));
	    break;
	}
       
	/* Try next sequence */
	X = i;
	DPRINT(Debug,61,(&Debug,
			 "skip_ascii_head_from_string --  got %d\n",
			 X));
    }	    

    x = X;
    str->string_type->charset_type->cs_clip_from_it(ret,str,&X,str->p->len);

    DPRINT(Debug,60,(&Debug,
		     "skip_ascii_head_from_string=%p  ('%s'; type=%p '%s'): len=%d (clip=%d..%d)\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len,x,X));

    return ret;
}

int find_pattern_from_string(str,pattern,ignore_case)
     CONST struct string *str;
     CONST struct string *pattern;
     int ignore_case;
{
    int ret = -1;

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad magic number (string type)",0);

    if (!charset_valid_magic(pattern->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,
		     "find_pattern_from_string(%p,%p,%d) ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
		     str,pattern,ignore_case,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len,
		     pattern->string_type->MIME_name ? 
		     pattern->string_type->MIME_name : "<none>",
		     pattern->string_type->charset_type,
		     pattern->string_type->charset_type->type_name,
		     pattern->p->len));


    if (str -> string_type == pattern -> string_type)
	ret = str->string_type->charset_type->
	    cs_find_pattern_from_it(str,pattern,ignore_case);

    /* -1 indicates that unicode values should be used on comparision ... */

    if (ret < 0) {
	int l1 = 0;
	uint16 * vector1 = safe_malloc ( (str->p->len) * sizeof (uint16));
	int l2 = 0;
	uint16 * vector2 = safe_malloc ( (pattern->p->len) * sizeof (uint16));
	int i,j;
	
	ret = 0;
	
	for (i = 0;  i < str->p->len; i++) {
	    int found;
            uint16 c =
                str->string_type->charset_type->
		cs_give_unicode_from_it(str,i,&found);

	    /* TODO: Should lowercase ANY unicode value ... */

	    if (!found)
		c = MAPPING_NONE;
	    else if (ignore_case) {
		c = unicode_ch(c,UOP_lowercase);
		if (!c)
		    c = MAPPING_NONE;
	    }
	    vector1[l1++] = c;
	}

	for (i = 0;  i < pattern->p->len; i++) {
	    int found;
            uint16 c =
                pattern->string_type->charset_type->
		cs_give_unicode_from_it(pattern,i,&found);

	    /* TODO: Should lowercase ANY unicode value ... */

	    if (!found)
		c = MAPPING_NONE;
	    else if (ignore_case) { 
		c = unicode_ch(c,UOP_lowercase);
		if (!c)
		    c = MAPPING_NONE;
	    }
	    vector2[l2++] = c;
	}

	for (i = 0; i < l1; ) {
	    CONST int s = i + 1;
	    for (j = 0; j < l2 && i < l1; j++,i++) {
		if (vector1[i] != vector2[j])
		    break;
		/* MAPPING_NONE matches newer ... */
		if (MAPPING_NONE == vector1[i] ||
		    MAPPING_NONE == vector2[j])
		    break;
	    }
	    if (j >= l2) {
		DPRINT(Debug,63,(&Debug,
				 "cs_find_pattern_from_string=1 MATCH\n"));

		ret = 1;
		break;
	    }
	    i = s;
	}	     

	if (!ret) {
	    DPRINT(Debug,63,(&Debug,
			     "cs_find_pattern_from_string=0 MATCH\n"));
	}
	
	free(vector1);
	free(vector2);
    }
								      
    DPRINT(Debug,60,(&Debug,
		     "find_pattern_from_string=%d\n",ret));

    return ret;
}


static int string_match_part P_((const struct string * name, int name_X,
				 const struct string * pat, int pat_X));

static int string_match_part(name,name_X,pat,pat_X)
     CONST struct string * name;
     int name_X;
     CONST struct string * pat;
     int pat_X;
{
    int r = 1;

    while (pat_X < pat->p->len) {
	int pat_found = 0;
	uint16 code_pat  = 
	    pat->string_type->charset_type->
	    cs_give_unicode_from_it(pat,
				    pat_X,
				    &pat_found);
	
	if (pat_found && 
	    0x002A /* '*' */ == code_pat) {
	    
	    if (name_X < name->p->len) {
		if (string_match_part(name,name_X,pat,pat_X+1)) {
		    /* Tail matches */
		
		    goto succeed;
		}
		/* Try match to next starting position */
		
		name_X++;
		
	    } else {
		/* '*' matches to empty string */
		pat_X++;
	    }	    
	} else if (name_X < name->p->len) {
	    int name_found = 0;
	    uint16 code_name = 
		name->string_type->charset_type->
		cs_give_unicode_from_it(name,
					name_X,
					&name_found);
	    
	    
	    if (pat_found &&
		0x003F /* '?' */ == code_pat) {

		/* Accept any character */
		name_X++;
		pat_X++;
		
	    } else if (pat_found && name_found &&
		       code_pat == code_name) {
		
		/* Accept character */
		name_X++;
		pat_X++;
	    } else {
		r = 0;

		DPRINT(Debug,55,(&Debug,
			   "-- Match fail on pat_X=%d (%04X) name_X=%d (%04X)\n",
			   pat_X,code_pat,name_X,code_name));
		break;
	    }		
	} else {
	    r = 0;
	    DPRINT(Debug,55,(&Debug,
			     "-- Match fail on pat_X=%d name_X=%d (end of name)\n",
			     pat_X,name_X));
	    break;	    
	}  
    }
    
    if (name_X != name->p->len ||
	pat_X != pat->p->len) {

	DPRINT(Debug,55,(&Debug,
			 "-- Match fail on pat_X=%d name_X=%d (failure?)\n",
			 pat_X,name_X));

	r = 0;
    }
 succeed:
    return r;
}

int string_match(name,pat)
     CONST struct string * name;
     CONST struct string * pat;
{
    int r = 0;

    if (!charset_valid_magic(name->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad magic number (string type)",0);

    if (!charset_valid_magic(pat->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,
		     "string_match(%p,%p) ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
		     name,pat,
		     name->string_type->MIME_name ? 
		     name->string_type->MIME_name : "<none>",
		     name->string_type->charset_type,
		     name->string_type->charset_type->type_name,
		     name->p->len,
		     pat->string_type->MIME_name ? 
		     pat->string_type->MIME_name : "<none>",
		     pat->string_type->charset_type,
		     pat->string_type->charset_type->type_name,
		     pat->p->len));

    r = string_match_part(name,0,pat,0);
    
    DPRINT(Debug,60,(&Debug,
		     "string_match=%d\n",r));

    return r;
}


uint16 give_unicode_from_string(str,pos)
     CONST struct string *str;
     int pos;
{
    int found;
    uint16 ret;

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"give_unicode_from_string",
	      "Bad magic number (string type)",0);

    ret = str->string_type->charset_type->
	cs_give_unicode_from_it(str,pos, &found);

    DPRINT(Debug,60,(&Debug,
		     "give_unicode_from_string(%p,%d)=%d  ('%s'; type=%p '%s')%s\n",
		     str,pos,ret,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     found ? "" : " [NOT FOUND]"));
    
    return ret;
}

void remove_control(str)
     CONST struct string *str;
{

    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"remove_control",
	      "Bad magic number (string type)",0);

    if (str->p->state) {
	if (str->p->state->charset != str->string_type)
	    panic("STRING PANIC",__FILE__,__LINE__,"remove_control",
		  "Bad magic number (state charset)",0);
	free_state_internal(&(str->p->state));
    }

    DPRINT(Debug,60,(&Debug,
		     "remove_control(%p)   ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    
    str->string_type->charset_type->cs_remove_control_it(str);	
}

void add_state_to_string(str,ch)
     struct string *str; 
     struct charset_state *ch;
{
    if (!charset_valid_magic(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
	      "Bad magic number (string type)",0);
    
    if (!charset_valid_magic(ch->charset))
	panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
	      "Bad magic number (charset)",0);

    if (str->p->state) {
	if (str->p->state->charset != str->string_type)
	    panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
		  "Bad magic number (state charset)",0);
	free_state_internal(&(str->p->state));
    }

    DPRINT(Debug,60,(&Debug,
		     "add_state_to_it(%p,%p)   ('%s'; type=%p '%s', '%s'; type=%p '%s')\n",
		     str,ch,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     ch->charset->MIME_name ? 
		     ch->charset->MIME_name : "<none>",
		     ch->charset->charset_type,
		     ch->charset->charset_type->type_name));

    if (ch->charset == str->string_type)
	str->string_type->charset_type->cs_add_state_to_it(str,ch);
    else {
	int found;
	uint16 val = 
	    ch->charset->charset_type->cs_give_unicode_from_s_it(ch,&found);
	
	str->string_type->charset_type->
	    cs_add_unicodedata_to_it(str,1,&val);
    }
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 * End:
 */

