#ifndef __ADDRESS_H__
#define __ADDRESS_H__ 1
/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2002, 2003, 2004 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

*/
/****************************************************************************
 *    INTERFACE REQUIRED HEADERS
 ****************************************************************************/

#include "memblock.h"
#include "memchunk.h"
#include "rstring.h"

/****************************************************************************
 *    INTERFACE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/

/**
 * email_kind describes one of the syntax, that can be used in an email:
 *
 *  KIND_UNKNOWN         -- there was no attempt to guess the type yet
 *  KIND_EMAIL_IN_ANGLES -- the "modern" syntax described in RFC 2822
 *                          example: rzyjontko <rzyj@op.pl>
 *  KIND_NAME_AS_COMMENT -- the obsolete syntax described in RFC 822
 *                          example: rzyj@op.pl (rzyjontko)
 *  KIND_PURE_EMAIL      -- just an email addres
 *                          example: rzyj@op.pl
 *  KIND_NO_EMAIL        -- possibly only user name for local delivery
 *                          example: rzyj
 *  KIND_STRANGE         -- email with something else, possibly syntax error
 *                          example: rzyjontko < rzyj@op.pl >
 */
enum email_kind {
        KIND_UNKNOWN         = 0,
        KIND_EMAIL_IN_ANGLES = 1,
        KIND_NAME_AS_COMMENT = 2,
        KIND_PURE_EMAIL      = 3,
        KIND_NO_EMAIL        = 4,
        KIND_STRANGE         = 5
};

enum email_sex {
        SEX_UNKNOWN = 0,
        SEX_MALE    = 1,
        SEX_FEMALE  = 2
};

enum yes_no {
        NO  = 0,
        YES = 1
};

#define A_SHIFT_SEX 3
#define A_MASK_SEX  (3 << A_SHIFT_SEX)

#define A_SHIFT_OFFICIAL 5
#define A_MASK_OFFICIAL (1 << A_SHIFT_OFFICIAL)

#define A_SHIFT_FOREIGN 6
#define A_MASK_FOREIGN (1 << A_SHIFT_FOREIGN)

/****************************************************************************
 *    INTERFACE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/

/**
 * index    -- for use in atable module
 * full     -- when parsing a message, this is the email, as appearing in
 *             a header field (To, From, etc.), otherwise it holds a string
 *             that may be put in a header field
 * email    -- address specifier of a message (see RFC2822)
 * name     -- display name of the address (see RFC2822), but maybe empty,
 *             in which case, there should be used email address
 * first    -- first name of the display name
 * last     -- last name of the display name
 * initials -- first letters of each word in name
 * groups   -- list of groups, that this address belongs to
 *
 *****   user flags   *****
 * kind     -- see type comment above
 * sex      --
 * official --
 * foreign  --
 *****   implementation flags   ****
 * abook          -- true if this address is a part of addressbook
 * own            -- true if this address is defined in one of smtp_acc
 * atomic_name    -- true if display name is an ATOM in sense of RFC2822
 * quotes_in_name -- true if there is a double quote sign inside of name
 */
typedef struct addr {
        int   index;
        char *full;
        char *email;
        char *name;
        char *first;
        char *last;
        char *initials;
        union {
                struct {
                        enum email_kind kind           : 3;
                        enum email_sex  sex            : 2;
                        enum yes_no     official       : 1;
                        enum yes_no     foreign        : 1;
                        enum yes_no     abook          : 1;
                        enum yes_no     own            : 1;
                        enum yes_no     atomic_name    : 1;
                        enum yes_no     quotes_in_name : 1;
                } bits;
                unsigned value;
        } flags;

        rstring_t *groups;
} address_t;

/****************************************************************************
 *    INTERFACE DATA DECLARATIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE FUNCTION PROTOTYPES
 ****************************************************************************/

extern void address_init (void);
extern void address_free_resources (void);

extern address_t *address_empty (void);
extern address_t *address_from_string (const char *str);
extern address_t *address_complete (address_t *addr);
extern address_t *address_find (const char *str);
extern address_t *address_read (memchunk_t *chunk);
extern void       address_dump (address_t *addr, memchunk_t *chunk);

extern char *address_name (address_t *addr);
extern char *address_wrote_format (address_t *addr);
extern int   address_cmp (const address_t *a, const address_t *b);
extern void  address_for_all (void (*fun)(address_t *));
extern void  address_reset_indexes (void);

/****************************************************************************
 *    INTERFACE OBJECT CLASS DEFINITIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE TRAILING HEADERS
 ****************************************************************************/
/****************************************************************************
 *
 *    END HEADER address.h
 *
 ****************************************************************************/
#endif
