#
# Copyright (c) 2010 The NetBSD Foundation, Inc.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND
# CONTRIBUTORS ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
# INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS BE LIABLE FOR ANY
# DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
# GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
# IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
# OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
# IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#

# TODO: The name of the binary should be set during the build to let tests work
# when automake's program renaming features are used.
SERVICES='services'

common_head() {
    # Once atf 0.8 is released, descriptions will be optional.  Do not bother
    # to set redundant texts for them, as we will kill this anyway soon. 
    atf_set "descr" "XXX"
    atf_set "require.progs" "${SERVICES}"
}

check_showed_usage() {
    if ! grep "Usage: ${SERVICES}" stderr >/dev/null; then
        cat stderr
        atf_fail "Expected usage message, but it was not shown or is invalid"
    fi
}

check_error() {
    if ! grep "^${SERVICES}: $1" stderr >/dev/null; then
        cat stderr
        atf_fail "Expected error message \"$1\" not found in stderr"
    fi
}

check_usage_error() {
    local message="${1}"; shift
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" "${@}"
    check_showed_usage
    check_error ".*${message}"
}

check_no_file() {
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" \
        -K non-existent "${@}"
    check_error "Cannot find the services file 'non-existent'"
}

compare_files() {
    if [ "${1}" = - ]; then
        cat >expected
        atf_check -s eq:0 -o file:expected -e empty cat "${2}"
    else
        atf_check -s eq:0 -o file:"${1}" -e empty cat "${2}"
    fi
}

create_shell() {
    local name="${1:-binary}"
    touch "${name}"
    chmod +x "${name}"
    echo $(pwd)/"${name}"
}

# -------------------------------------------------------------------------
# Generic command line tests.
# -------------------------------------------------------------------------

atf_test_case usage
usage_head() { common_head; }
usage_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}"
    atf_check -s eq:0 -o ignore -e empty grep "${SERVICES}.*add" stderr
    atf_check -s eq:0 -o ignore -e empty grep "${SERVICES}.*check" stderr
    atf_check -s eq:0 -o ignore -e empty grep "${SERVICES}.*list" stderr
    atf_check -s eq:0 -o ignore -e empty grep "${SERVICES}.*remove" stderr
}

atf_test_case no_args
no_args_head() { common_head; }
no_args_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}"
    check_showed_usage
}

atf_test_case unknown_command
unknown_command_head() { common_head; }
unknown_command_body() {
    check_usage_error "Unknown command 'foo'" foo
}

atf_test_case unknown_command_with_args
unknown_command_with_args_head() { common_head; }
unknown_command_with_args_body() {
    check_usage_error "Unknown command 'foo'" foo arg1 arg2
}

# -------------------------------------------------------------------------
# Add command.
# -------------------------------------------------------------------------

atf_test_case add_bad_args
add_bad_args_head() { common_head; }
add_bad_args_body() {
    check_usage_error "'add'.*two or more arguments" add
    check_usage_error "'add'.*two or more arguments" add arg1
}

atf_test_case add_invalid_name
add_invalid_name_head() { common_head; }
add_invalid_name_body() {
    # Relies on check_invalid_name for proper validation.
    # In this test, we just make sure that the verification code is used.

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        add "#foo" 1234/tcp
    check_error "Invalid service name '#foo'"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        add "foo bar" 1234/tcp
    check_error "Invalid service name 'foo bar'"

    [ -f "${SERVICES}" ] && \
        atf_fail "services file should not have been created"
}

atf_test_case add_invalid_port
add_invalid_port_head() { common_head; }
add_invalid_port_body() {
    # Relies on check_invalid_port for proper validation.
    # In this test, we just make sure that the verification code is used.

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        add foo "1234 tcp"
    check_error "Invalid service port '1234 tcp'"

    [ -f "${SERVICES}" ] && \
        atf_fail "services file should not have been created"
}

atf_test_case add_duplicate
add_duplicate_head() { common_head; }
add_duplicate_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" add echo 7/udp
    check_error ".*'echo 7/udp'.*already.*database"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" add ftp 21/tcp
    check_error ".*'ftp 21/tcp'.*already.*database"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" add foo 13/tcp
    check_error ".*'13/tcp'.*already.*database"
}

atf_test_case add_some
add_some_head() { common_head; }
add_some_body() {
    cp "$(atf_get_srcdir)/services.gold" db
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" -K db add a 1/a
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" -K db add ok 5/ok
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" -K db add a2b 999/udp
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" -K db add a2b 999/tcp \
        alias1 alias2

    cp "$(atf_get_srcdir)/services.gold" expdb
    echo "a 1/a" >>expdb
    echo "ok 5/ok" >>expdb
    echo "a2b 999/udp" >>expdb
    echo "a2b 999/tcp alias1 alias2" >>expdb
    compare_files expdb db
}

atf_test_case add_create_fail
add_create_fail_head() { common_head; }
add_create_fail_body() {
    mkdir dir; chmod 555 dir
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K dir/db \
        add foo 1234/tcp
    check_error "Failed to create"
}

atf_test_case add_change_fail
add_change_fail_head() { common_head; }
add_change_fail_body() {
    touch db; chmod 444 db
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K db \
        add bar 4321/udp
    check_error "Failed to add"
}

# -------------------------------------------------------------------------
# Check command.
# -------------------------------------------------------------------------

atf_test_case check_bad_args
check_bad_args_head() { common_head; }
check_bad_args_body() {
    check_usage_error "'check'.*one to two" check
    check_usage_error "'check'.*one to two" check arg1 arg2 arg3
}

atf_test_case check_no_file
check_no_file_head() { common_head; }
check_no_file_body() {
    check_no_file check ssh 22/tcp
}

atf_test_case check_invalid_name
check_invalid_name_head() { common_head; }
check_invalid_name_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        check "" 1234/tcp
    check_error "Invalid service name ''"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        check "#foo" 1234/tcp
    check_error "Invalid service name '#foo'"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        check "foo bar" 1234/tcp
    check_error "Invalid service name 'foo bar'"
}

atf_test_case check_invalid_port
check_invalid_port_head() { common_head; }
check_invalid_port_body() {
    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        check foo 1234
    check_error "Invalid service port '1234'"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        check foo tcp
    check_error "Invalid service port 'tcp'"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        check foo "1234 tcp"
    check_error "Invalid service port '1234 tcp'"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        check foo "1234/"
    check_error "Invalid service port '1234/'"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        check foo "/tcp"
    check_error "Invalid service port '/tcp'"
}

atf_test_case check_true
check_true_head() { common_head; }
check_true_body() {
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check echo

    atf_check -s eq:0 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check 9/udp

    atf_check -s eq:0 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check ftp 21/udp

    atf_check -s eq:0 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check ftp 21/tcp

    atf_check -s eq:0 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check name 42/tcp

    atf_check -s eq:0 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check mixed1 9999/tcp
}

atf_test_case check_false
check_false_head() { common_head; }
check_true_body() {
    atf_check -s eq:1 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check ftp-data

    atf_check -s eq:1 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check 23/tcp

    atf_check -s eq:1 -o empty -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" check mixed1 9999/udp
}

# -------------------------------------------------------------------------
# List command.
# -------------------------------------------------------------------------

atf_test_case list_bad_args
list_bad_args_head() { common_head; }
list_bad_args_body() {
    check_usage_error "'list'.*zero arguments" list arg1
}

atf_test_case list_no_file
list_no_file_head() { common_head; }
list_no_file_body() {
    check_no_file list
}

atf_test_case list_some
list_some_head() { common_head; }
list_some_body() {
    cat >expout <<EOF
daytime 13/tcp
daytime 13/udp
discard 9/tcp
discard 9/udp
echo 7/tcp
echo 7/udp
foobar 1234/unknown bogus
ftp 21/tcp
ftp 21/udp
mixed1 9999/tcp
name 42/tcp nameserver
name 42/udp nameserver
EOF
    atf_check -s eq:0 -o file:expout -e empty "${SERVICES}" \
        -K "$(atf_get_srcdir)/services.gold" list
}

# -------------------------------------------------------------------------
# Remove command.
# -------------------------------------------------------------------------

atf_test_case remove_bad_args
remove_bad_args_head() { common_head; }
remove_bad_args_body() {
    check_usage_error "'remove'.*two arguments" remove
    check_usage_error "'remove'.*two arguments" remove arg1 arg2 arg3
}

atf_test_case remove_no_file
remove_no_file_head() { common_head; }
remove_no_file_body() {
    check_no_file remove ftp 21/udp
}

atf_test_case remove_invalid_name
remove_invalid_name_head() { common_head; }
remove_invalid_name_body() {
    # Relies on check_invalid_name for proper validation.
    # In this test, we just make sure that the verification code is used.

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        remove "#foo" 1234/tcp
    check_error "Invalid service name '#foo'"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        remove "foo bar" 1234/tcp
    check_error "Invalid service name 'foo bar'"
}

atf_test_case remove_invalid_port
remove_invalid_port_head() { common_head; }
remove_invalid_port_body() {
    # Relies on check_invalid_port for proper validation.
    # In this test, we just make sure that the verification code is used.

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K non-existent \
        remove foo "1234 tcp"
    check_error "Invalid service port '1234 tcp'"
}

atf_test_case remove_missing
remove_missing_head() { common_head; }
remove_missing_body() {
    cp "$(atf_get_srcdir)/services.gold" db

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K db remove \
        discard 9/foo
    check_error ".*'discard 9/foo'.*not.*database"

    atf_check -s eq:1 -o empty -e save:stderr "${SERVICES}" -K db remove \
        mixed1 99/tcp
    check_error ".*'mixed1 99/tcp'.*not.*database"

    cmp -s "$(atf_get_srcdir)/services.gold" db || atf_fail "services was" \
        "modified when it should not have been"
}

atf_test_case remove_some
remove_some_head() { common_head; }
remove_some_body() {
    cp "$(atf_get_srcdir)/services.gold" db
    chmod +x db # This is to check that permissions are kept.
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" -K db remove echo 7/tcp
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" -K db remove ftp 21/udp

    cat "$(atf_get_srcdir)/services.gold" | grep -v 7/tcp | grep -v 21/udp >db2
    compare_files db2 db

    test -x db || atf_fail "services file permissions were not respected"
}

atf_test_case remove_last
remove_last_head() { common_head; }
remove_last_body() {
    cat >db <<EOF
one 1/foo alias # comment
EOF
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" -K db remove one 1/foo
    test -f db && atf_fail "Empty services file left behind"
}

atf_test_case remove_commented
remove_commented_head() { common_head; }
remove_commented_body() {
    cat >db <<EOF
ftp 21/tcp
#ftp 21/tcp
EOF
    cat >db2 <<EOF
#ftp 21/tcp
EOF
    atf_check -s eq:0 -o empty -e empty "${SERVICES}" -K db remove ftp 21/tcp
    compare_files db db2
}

# -------------------------------------------------------------------------
# Main.
# -------------------------------------------------------------------------

atf_init_test_cases() {
    atf_add_test_case usage
    atf_add_test_case no_args
    atf_add_test_case unknown_command
    atf_add_test_case unknown_command_with_args

    atf_add_test_case add_bad_args
    atf_add_test_case add_invalid_name
    atf_add_test_case add_invalid_port
    atf_add_test_case add_duplicate
    atf_add_test_case add_some
    atf_add_test_case add_create_fail
    atf_add_test_case add_change_fail

    atf_add_test_case check_bad_args
    atf_add_test_case check_no_file
    atf_add_test_case check_invalid_name
    atf_add_test_case check_invalid_port
    atf_add_test_case check_true
    atf_add_test_case check_false

    atf_add_test_case list_bad_args
    atf_add_test_case list_no_file
    atf_add_test_case list_some

    atf_add_test_case remove_bad_args
    atf_add_test_case remove_no_file
    atf_add_test_case remove_invalid_name
    atf_add_test_case remove_invalid_port
    atf_add_test_case remove_missing
    atf_add_test_case remove_some
    atf_add_test_case remove_last
    atf_add_test_case remove_commented
}

# vim: syntax=sh:expandtab:shiftwidth=4:softtabstop=4
