/* ========================================================================== */
/* Operating system API */
/*
 * Include this as the first header in a C source file that should access the
 * operating system. A C++ source file must never include this header.
 */

#ifdef __cplusplus
#  error The POSIX.1-1990 API has no C++ binding!
#endif  /* __cplusplus */


/* ========================================================================== */
/* Configuration (must be 1st) */

#include "config.h"


/* ========================================================================== */
/* Tell system headers that they should be POSIX conformant (must be 2nd) */

#ifdef HDR_POSIX_H
   /* This must always be the first header that is included */
#  error Never include this header more than once!
#endif  /* NDR_POSIX_H */
#define HDR_POSIX_H  1

/*
 * Check twice before requesting another API, this can have many side effects.
 * It is obvious that older APIs lack support for some features, but newer APIs
 * also have features removed (because they where deprecated). In both cases we
 * must provide replacements for (different) missing features.
 *
 * See also: ../INSTALL.HP-UX
 */

#if CFG_USE_POSIX_API >= 202405L
#  define _POSIX_C_SOURCE  202405L
#elif CFG_USE_POSIX_API == 200809L
#  define _POSIX_C_SOURCE  200809L
#elif CFG_USE_POSIX_API == 200112L
   /*
    * Attention:
    * At least POSIX.1g-2000 is required for IP6 option
    */
#  define _POSIX_C_SOURCE  200112L
#else  /* CFG_USE_POSIX_API */
   /*
    * Attention:
    * POSIX.1-1990 is not sufficient because we need the POSIX.1b-1993 Realtime
    * and POSIX.1c-1996 thread extensions
    */
#  define _POSIX_C_SOURCE  199506L
#endif  /* CFG_USE_POSIX_API */

#if CFG_USE_XSI
#  if CFG_USE_POSIX_API == 202405L
      /*
       * Attention:
       * The POSIX version must be not greater than 202405 if this value is 800:
       * <https://pubs.opengroup.org/onlinepubs/9799919799/functions/V2_chap02.html#tag_16_02_01_02>
       */
#     define _XOPEN_SOURCE  800
#  elif CFG_USE_POSIX_API == 200809L
      /*
       * Attention:
       * The POSIX version must be not greater than 200809 if this value is 700:
       * <https://pubs.opengroup.org/onlinepubs/9699919799/functions/V2_chap02.html#tag_15_02_01_02>
       */
#     define _XOPEN_SOURCE  700
#  elif CFG_USE_POSIX_API == 200112L
      /*
       * Attention:
       * The POSIX version must be not greater than 200112 if this value is 600:
       * <http://pubs.opengroup.org/onlinepubs/007904975/functions/xsh_chap02_02.html#tag_02_02_01_02>
       */
#     define _XOPEN_SOURCE  600
#  else  /* CFG_USE_POSIX_API */
      /*
       * Attention:
       * The POSIX version must be not greater than 199506 if this value is 500:
       * <http://pubs.opengroup.org/onlinepubs/7908799/xns/compilation.html>
       */
#     define _XOPEN_SOURCE  500
#  endif  /* CFG_USE_POSIX_API */
#endif  /* CFG_USE_XSI */

/*
 * This definition allows linking objects compiled for XNS and BSD sockets:
 * <http://docstore.mik.ua/manuals/hp-ux/en/B2355-60130/xopen_networking.7.html>
 */
#if defined(__hpux) && CFG_USE_XSI
#  define _HPUX_ALT_XOPEN_SOCKET_API  1
#endif   /* defined(__hpux) && CFG_USE_XSI */

#include <sys/types.h>
#include <pthread.h>
#include <stdlib.h>
#include <stddef.h>
#include <limits.h>
#include <errno.h>
#include <unistd.h>


/* ========================================================================== */
/* POSIX.1b realtime support */

#include <time.h>

#define api_posix_struct_timespec  struct timespec

#define api_posix_nanosleep  nanosleep

#if CFG_USE_FSC
#  define api_posix_fsync  fsync
#else  /* CFG_USE_FSC */
int  api_posix_fsync(int);
#endif  /* CFG_USE_FSC */


/* ========================================================================== */
/* POSIX.1c thread support */

#define API_POSIX_PTHREAD_MUTEX_INITIALIZER  PTHREAD_MUTEX_INITIALIZER
#define API_POSIX_PTHREAD_COND_INITIALIZER  PTHREAD_COND_INITIALIZER
#define API_POSIX_PTHREAD_CANCEL_DISABLE  PTHREAD_CANCEL_DISABLE

#define api_posix_pthread_t  pthread_t
#define api_posix_pthread_mutex_t  pthread_mutex_t
#define api_posix_pthread_cond_t  pthread_cond_t

#define api_posix_pthread_create  pthread_create
#define api_posix_pthread_cancel  pthread_cancel
#define api_posix_pthread_setcancelstate  pthread_setcancelstate
#define api_posix_pthread_join  pthread_join
#define api_posix_pthread_self  pthread_self
#define api_posix_pthread_equal  pthread_equal
#define api_posix_pthread_sigmask  pthread_sigmask
#define api_posix_pthread_cleanup_push  pthread_cleanup_push
#define api_posix_pthread_cleanup_pop  pthread_cleanup_pop
#define api_posix_pthread_mutex_lock  pthread_mutex_lock
#define api_posix_pthread_mutex_trylock  pthread_mutex_trylock
#define api_posix_pthread_mutex_unlock  pthread_mutex_unlock
#define api_posix_pthread_cond_wait  pthread_cond_wait
#define api_posix_pthread_cond_signal  pthread_cond_signal


/* ========================================================================== */
/* POSIX.2 support */

#include <stdio.h>

#define api_posix_popen  popen
#define api_posix_pclose  pclose


/* ========================================================================== */
/* System limits */

#if CFG_HDR_INTTYPES_H
#  include <inttypes.h>
#endif  /* CFG_HDR_INTTYPES_H */

#define API_POSIX_INT_MAX  INT_MAX
#define API_POSIX_UINT_MAX  UINT_MAX
#define API_POSIX_LONG_MAX  LONG_MAX
#define API_POSIX_ULONG_MAX  ULONG_MAX
#define API_POSIX_SSIZE_MAX  SSIZE_MAX
#ifdef SIZE_MAX
#  define API_POSIX_SIZE_MAX  SIZE_MAX
#else  /* SIZE_MAX */
#  define API_POSIX_SIZE_MAX  ((size_t) API_POSIX_SSIZE_MAX * (size_t) 2)
#endif  /* SIZE_MAX */


/* ========================================================================== */
/* Environment and return values */

/* Pointer to array with environment variables */
extern char**  environ;
#define api_posix_environ  environ

/* Standard return values for processes */
#define API_POSIX_EXIT_SUCCESS  EXIT_SUCCESS
#define API_POSIX_EXIT_FAILURE  EXIT_FAILURE


/* ========================================================================== */
/* System information */

#include <sys/utsname.h>

#define api_posix_struct_utsname  struct utsname

#define api_posix_uname  uname


/* ========================================================================== */
/* System errors */

#define API_POSIX_EACCES  EACCES
#define API_POSIX_EAFNOSUPPORT  EAFNOSUPPORT
#define API_POSIX_EAGAIN  EAGAIN
#define API_POSIX_EBADF   EBADF
#define API_POSIX_ECONNREFUSED  ECONNREFUSED
#define API_POSIX_EEXIST  EEXIST
#define API_POSIX_EINTR  EINTR
#define API_POSIX_EINPROGRESS  EINPROGRESS
#define API_POSIX_EINVAL  EINVAL
#define API_POSIX_EIO  EIO
#define API_POSIX_ENOENT  ENOENT
#define API_POSIX_ENOMEM  ENOMEM
#define API_POSIX_ENOPROTOOPT  ENOPROTOOPT
#define API_POSIX_ENOSYS  ENOSYS
#define API_POSIX_ENOTSOCK  ENOTSOCK
#define API_POSIX_EOVERFLOW  EOVERFLOW

#define api_posix_errno  errno


/* ========================================================================== */
/* Signal handling */

#include <signal.h>

#define api_posix_sig_atomic_t  sig_atomic_t

#define api_posix_struct_sigaction  struct sigaction

#define API_POSIX_SIGHUP  SIGHUP
#define API_POSIX_SIGINT  SIGINT
#define API_POSIX_SIGPIPE  SIGPIPE
#define API_POSIX_SIGQUIT  SIGQUIT
#define API_POSIX_SIGTERM  SIGTERM

#define API_POSIX_SIG_DFL  SIG_DFL
#define API_POSIX_SIG_IGN  SIG_IGN

#define API_POSIX_SIG_BLOCK  SIG_BLOCK
#define API_POSIX_SIG_SETMASK  SIG_SETMASK

#define api_posix_sigset_t  sigset_t

#define api_posix_kill  kill
#define api_posix_sigemptyset  sigemptyset
#define api_posix_sigaddset  sigaddset
#define api_posix_sigaction  sigaction


/* ========================================================================== */
/* Process handling */

#include <sys/wait.h>

#define api_posix_pid_t  pid_t

#define API_POSIX_WIFEXITED  WIFEXITED
#define API_POSIX_WEXITSTATUS  WEXITSTATUS

#define API_POSIX_WNOHANG  WNOHANG

#define api_posix_fork  fork
#define api_posix_execl  execl
#define api_posix_execlp  execlp
#define api_posix_waitpid  waitpid
#define api_posix_getpid  getpid


/* ========================================================================== */
/* Time */

#define api_posix_time_t  time_t
#define api_posix_struct_tm  struct tm

#define api_posix_time  time

#define api_posix_gmtime_r  gmtime_r
#define api_posix_localtime_r  localtime_r

/* The %z conversion was included into POSIX.1 since Issue 6 */
#if CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI
#  define api_posix_strftime  strftime
#endif  /* CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI */

#if CFG_USE_XSI
#  include <sys/time.h>

#  define api_posix_time_t  time_t
#  define api_posix_suseconds_t  suseconds_t
#  define api_posix_struct_timeval  struct timeval
#else  /* CFG_USE_XSI */
#  define api_posix_suseconds_t  long int

struct api_posix_timeval
{
  api_posix_time_t  tv_sec;
  api_posix_suseconds_t  tv_usec;
};

#  define api_posix_struct_timeval  struct api_posix_timeval
#endif  /* CFG_USE_XSI */


/* ========================================================================== */
/* Memory management */

#if CFG_USE_MEMDEBUG
void*  api_posix_malloc(size_t);
void*  api_posix_realloc(void*, size_t);
void  api_posix_free(void*);
#else  /* CFG_USE_MEMDEBUG */
#  define api_posix_malloc  malloc
#  define api_posix_realloc  realloc
#  define api_posix_free  free
#endif  /* CFG_USE_MEMDEBUG */


/* ========================================================================== */
/* Random numbers */

#if CFG_USE_XSI
#  define api_posix_srandom  srandom
#  define api_posix_random  random
#else  /* CFG_USE_XSI */
void  api_posix_srandom(unsigned int);
long int  api_posix_random(void);
#endif  /* CFG_USE_XSI */


/* ========================================================================== */
/* File & directory handling */

#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>

#define api_posix_off_t  off_t

#define API_POSIX_SEEK_SET  SEEK_SET

#define API_POSIX_FD_CLOEXEC  FD_CLOEXEC

#define API_POSIX_O_NONBLOCK  O_NONBLOCK
#define API_POSIX_O_CREAT  O_CREAT
#define API_POSIX_O_TRUNC  O_TRUNC
#define API_POSIX_O_RDONLY  O_RDONLY
#define API_POSIX_O_WRONLY  O_WRONLY
#define API_POSIX_O_RDWR  O_RDWR
#define API_POSIX_O_EXCL  O_EXCL

#define API_POSIX_S_IRWXU  S_IRWXU
#define API_POSIX_S_IRWXG  S_IRWXG
#define API_POSIX_S_IRWXO  S_IRWXO
#define API_POSIX_S_IRUSR  S_IRUSR
#define API_POSIX_S_IWUSR  S_IWUSR
#define API_POSIX_S_IXUSR  S_IXUSR
#define API_POSIX_S_IRGRP  S_IRGRP
#define API_POSIX_S_IWGRP  S_IWGRP
#define API_POSIX_S_IROTH  S_IROTH
#define API_POSIX_S_IWOTH  S_IWOTH

#define API_POSIX_F_GETFL  F_GETFL
#define API_POSIX_F_SETFL  F_SETFL
#define API_POSIX_F_WRLCK  F_WRLCK
#define API_POSIX_F_SETLK  F_SETLK
#define API_POSIX_F_SETFD  F_SETFD

#define API_POSIX_STDIN_FILENO  STDIN_FILENO
#define API_POSIX_STDOUT_FILENO  STDOUT_FILENO

#define API_POSIX_DIR  DIR

#define API_POSIX_S_ISREG  S_ISREG
#define API_POSIX_S_ISDIR  S_ISDIR

#define api_posix_mode_t  mode_t
#define api_posix_struct_flock  struct flock
#define api_posix_struct_stat  struct stat
#define api_posix_struct_dirent  struct dirent

#define api_posix_dup2  dup2
#define api_posix_pipe  pipe
#define api_posix_unlink  unlink
#define api_posix_rename  rename
#define api_posix_open  open
#define api_posix_fdopen  fdopen
#define api_posix_close  close
#define api_posix_lseek  lseek
#define api_posix_read  read
#define api_posix_write  write
#define api_posix_fcntl  fcntl
#define api_posix_fgetc  fgetc
#define api_posix_mkdir  mkdir
#define api_posix_rmdir  rmdir
#define api_posix_opendir  opendir
#define api_posix_readdir  readdir
#define api_posix_closedir  closedir
#define api_posix_stat  stat

#if CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI
#  define api_posix_ftruncate  ftruncate
#  define api_posix_lstat  lstat
#  define api_posix_symlink  symlink
#endif  /* CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI */

/* Included into POSIX.1 since Issue 7 */
#if CFG_USE_POSIX_API >= 200809 || CFG_USE_XSI
#  define api_posix_mkstemp  mkstemp
#else  /* CFG_USE_POSIX_API >= 200809 || CFG_USE_XSI */
int  api_posix_mkstemp(char*);
#endif  /* CFG_USE_POSIX_API >= 200809 || CFG_USE_XSI */

#if CFG_USE_POSIX_API >= 200809
#  define  api_posix_scandir  scandir
#else  /* CFG_USE_POSIX_API >= 200809 */
int  api_posix_scandir(const char*, api_posix_struct_dirent***,
                       int (*sel)(api_posix_struct_dirent*),
                       int (*compar)(const api_posix_struct_dirent**,
                                     const api_posix_struct_dirent**));
#endif  /* CFG_USE_POSIX_API >= 200809 */


/* ========================================================================== */
/* System limits */

#  define API_POSIX_PC_NAME_MAX  _PC_NAME_MAX
#  define API_POSIX_PC_PATH_MAX  _PC_PATH_MAX

#  define api_posix_pathconf  pathconf


/* ========================================================================== */
/* NLS */

#if CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI
#  include <locale.h>

#  define API_POSIX_LC_ALL  LC_ALL
#  define API_POSIX_LC_CTYPE  LC_CTYPE
#  define API_POSIX_LC_MESSAGES  LC_MESSAGES

#  define api_posix_setlocale  setlocale
#endif  /* CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI */

#if  CFG_USE_POSIX_API >= 200809 || CFG_USE_XSI
#  include <nl_types.h>
#  include <langinfo.h>

#  define API_POSIX_NL_MSGMAX  NL_MSGMAX
#  define API_POSIX_NL_CAT_LOCALE  NL_CAT_LOCALE

#  define api_posix_nl_catd  nl_catd

#  define api_posix_nl_langinfo  nl_langinfo
#  define api_posix_catopen  catopen
#  define api_posix_catgets  catgets
#  define api_posix_catclose  catclose
#endif  /*  CFG_USE_POSIX_API >= 200809 || CFG_USE_XSI */

/* These functions are part of POSIX.1 since Issue 7 */
#if  CFG_USE_POSIX_API >= 200809
#  include <strings.h>

#  define API_POSIX_LC_CTYPE_MASK  LC_CTYPE_MASK

#  define api_posix_locale_t  locale_t

#  define api_posix_newlocale  newlocale
#  define api_posix_freelocale  freelocale
#  define api_posix_strcasecmp_l  strcasecmp_l
#  define api_posix_strncasecmp_l  strncasecmp_l
#else  /*  CFG_USE_POSIX_API >= 200809 */
#  define API_POSIX_LC_CTYPE_MASK  0

typedef int  api_posix_locale_t;

api_posix_locale_t  api_posix_newlocale(int, const char*, api_posix_locale_t);
void  api_posix_freelocale(api_posix_locale_t);
int  api_posix_strcasecmp_l(const char*, const char*, api_posix_locale_t);
int  api_posix_strncasecmp_l(const char*, const char*, size_t,
                             api_posix_locale_t);
#endif  /* CFG_USE_POSIX_API >= 200809 */


/* ========================================================================== */
/* BSD socket interface (XNS) */

/*
 * If the system report the IP6 option as available, we expect that it provides
 * XNS too (logical OR)
 */
#if CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI || CFG_USE_IP6
#  include <netinet/in.h>
#  include <arpa/inet.h>

#  define API_POSIX_AF_UNSPEC   AF_UNSPEC
#  define API_POSIX_AF_INET     AF_INET
#  if CFG_USE_IP6
#    define API_POSIX_AF_INET6  AF_INET6
#  endif   /* CFG_USE_IP6 */

#  define API_POSIX_MSG_PEEK  MSG_PEEK

#  define api_posix_ssize_t  ssize_t

/* Some systems that claim to be SUSv2 conformant don't define 'in_addr_t' */
#  if CFG_HDR_INTTYPES_H
#     include <inttypes.h>
typedef uint16_t  api_posix_in_port_t;
typedef uint32_t  api_posix_in_addr_t;
#  else  /* CFG_HDR_INTTYPES_H */
typedef CFG_UINT16_TYPE  api_posix_in_port_t;
typedef CFG_UINT32_TYPE  api_posix_in_addr_t;
#  endif  /* CFG_HDR_INTTYPES_H */

#  define api_posix_struct_in_addr  struct in_addr
#  define api_posix_struct_sockaddr_in  struct sockaddr_in

#  if CFG_USE_IP6
#     define api_posix_struct_in6_addr  struct in6_addr
#     define api_posix_struct_sockaddr_in6  struct sockaddr_in6
#  endif   /* CFG_USE_IP6 */

/* Bug in NetBSD: Declarations not exposed for POSIX.1-2001 without XSI */
#  define api_posix_htons  htons
#  define api_posix_htonl  htonl
#  define api_posix_ntohs  ntohs
#  define api_posix_ntohl  ntohl

#  define api_posix_inet_addr  inet_addr
#  if CFG_USE_POSIX_API >= 200112
#     define api_posix_inet_pton  inet_pton
#  else  /* CFG_USE_POSIX_API >= 200112 */
int  api_posix_inet_pton(int, const char*, void*);
#  endif  /* CFG_USE_POSIX_API >= 200112 */

#  include <sys/socket.h>

#  define api_posix_sa_family_t  sa_family_t
#  define api_posix_socklen_t  socklen_t

/*
 * 'struct sockaddr_storage' is broken on AIX (see documentation)
 * Currently there is no workaround provided by this module
 */
#  define api_posix_struct_sockaddr_storage  struct sockaddr_storage
#  define api_posix_struct_sockaddr  struct sockaddr

#  define  api_posix_struct_servent  struct servent
#  define  api_posix_struct_hostent  struct hostent

#  define  api_posix_getservbyname  getservbyname

#  if  CFG_USE_POSIX_API >= 200809
      /*
       * gethostbyname() is marked obsolete in POSIX.1 Issue 6
       * and was removed with Issue 7.
       * No replacement is implemented, use getaddrinfo() instead.
       */
#  else  /* CFG_USE_POSIX_API >= 200809 */
#     define  api_posix_gethostbyname  gethostbyname
#  endif  /* CFG_USE_POSIX_API >= 200809 */

#  define  api_posix_socket  socket

#  define API_POSIX_SOCK_STREAM  SOCK_STREAM

#  define api_posix_connect  connect

#  define API_POSIX_SOL_SOCKET  SOL_SOCKET

#  define API_POSIX_SO_ERROR     SO_ERROR
#  define API_POSIX_SO_RCVTIMEO  SO_RCVTIMEO
#  define API_POSIX_SO_SNDTIMEO  SO_SNDTIMEO

#  define api_posix_getsockopt  getsockopt
#  define api_posix_setsockopt  setsockopt

#  define api_posix_send  send
#  define api_posix_recv  recv

#  include <poll.h>

#  define API_POSIX_POLLIN   POLLIN
#  define API_POSIX_POLLOUT  POLLOUT
#  define API_POSIX_POLLERR  POLLERR

#  define api_posix_struct_pollfd  struct pollfd

#  define  api_posix_poll  poll
#else  /* CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI || CFG_USE_IP6 */
#  /* Use wrappers for original BSD socket API */
#  define BSD_FOR_POSIX
#  include "bsd.h"
#endif  /* CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI || CFG_USE_IP6 */


/* ========================================================================== */
/* Address information */

#include <netdb.h>

/*
 * If the system report the IP6 option as available, we expect that it provides
 * the POSIX.1g API too (logical OR)
 *
 * Attention: This subsystem is emulated using 'gethostbyname()':
 * This function was marked obsolete by POSIX.1-2001 and was eventually removed
 * in POSIX.1-2008 => The emulation can no longer be used since POSIX.1-2008
 * (even for IPv4).
 */
#if CFG_USE_POSIX_API >= 200112 || CFG_USE_IP6
#  define API_POSIX_AI_ADDRCONFIG  AI_ADDRCONFIG

#  define API_POSIX_EAI_AGAIN     EAI_AGAIN
#  define API_POSIX_EAI_BADFLAGS  EAI_BADFLAGS
#  define API_POSIX_EAI_FAIL      EAI_FAIL
#  define API_POSIX_EAI_FAMILY    EAI_FAMILY
#  define API_POSIX_EAI_MEMORY    EAI_MEMORY
#  define API_POSIX_EAI_NONAME    EAI_NONAME
#  define API_POSIX_EAI_OVERFLOW  EAI_OVERFLOW
#  define API_POSIX_EAI_SERVICE   EAI_SERVICE
#  define API_POSIX_EAI_SOCKTYPE  EAI_SOCKTYPE
#  define API_POSIX_EAI_SYSTEM    EAI_SYSTEM

#  define api_posix_struct_addrinfo  struct addrinfo

#  define api_posix_freeaddrinfo  freeaddrinfo
#  define api_posix_getaddrinfo  getaddrinfo
#  define api_posix_gai_strerror  gai_strerror
#else  /* CFG_USE_POSIX_API >= 200112 || CFG_USE_IP6 */
#  define API_POSIX_AI_ADDRCONFIG  0x0001

#  define API_POSIX_EAI_AGAIN     -1
#  define API_POSIX_EAI_BADFLAGS  -2
#  define API_POSIX_EAI_FAIL      -3
#  define API_POSIX_EAI_FAMILY    -4
#  define API_POSIX_EAI_MEMORY    -5
#  define API_POSIX_EAI_NONAME    -6
#  define API_POSIX_EAI_OVERFLOW  -7
#  define API_POSIX_EAI_SERVICE   -8
#  define API_POSIX_EAI_SOCKTYPE  -9
#  define API_POSIX_EAI_SYSTEM    -10

struct api_posix_addrinfo
{
   int  ai_flags;
   int  ai_family;
   int  ai_socktype;
   int  ai_protocol;
   api_posix_socklen_t  ai_addrlen;
   api_posix_struct_sockaddr*  ai_addr;
   char*  ai_canonname;
   struct api_posix_addrinfo*  ai_next;
};

#  define api_posix_struct_addrinfo  struct api_posix_addrinfo

void  api_posix_freeaddrinfo(api_posix_struct_addrinfo*);
int  api_posix_getaddrinfo(const char*, const char*,
                           const api_posix_struct_addrinfo*,
                           api_posix_struct_addrinfo**);
const char*  api_posix_gai_strerror(int);
#endif  /* CFG_USE_POSIX_API >= 200112 || CFG_USE_IP6 */


/* ========================================================================== */
/* Standard I/O */

#include <stdarg.h>
#include <string.h>
#include <stdio.h>

/*
 * Available since POSIX.1-2001.
 * Attention: The version in SUSv2 is not sufficient (different semantics)!
 */
#if CFG_USE_POSIX_API >= 200112
#  define api_posix_snprintf  snprintf
#else  /* CFG_USE_POSIX_API >= 200112 */
int  api_posix_snprintf(char*, size_t, const char*, ...);
#endif  /* CFG_USE_POSIX_API >= 200112 */

/* Part of POSIX.1 since Issue 7 */
#if CFG_USE_POSIX_API >= 200809
#  define api_posix_getline  getline
#else  /* CFG_USE_POSIX_API >= 200809 */
api_posix_ssize_t  api_posix_getline(char**, size_t*, FILE*);
#endif  /* CFG_USE_POSIX_API >= 200809 */


/* ========================================================================== */
/* Regular expression matching */

#if CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI || CFG_USE_CLB
#  include <regex.h>

#  define API_POSIX_REG_EXTENDED  REG_EXTENDED
#  define API_POSIX_REG_ICASE     REG_ICASE
#  define API_POSIX_REG_NOSUB     REG_NOSUB
#  define API_POSIX_REG_NEWLINE   REG_NEWLINE

#  define api_posix_regex_t  regex_t

#  define api_posix_regcomp  regcomp
#  define api_posix_regerror  regerror
#  define api_posix_regexec  regexec
#  define api_posix_regfree  regfree
#endif  /* CFG_USE_POSIX_API >= 200112 || CFG_USE_XSI || CFG_USE_CLB */


/* ========================================================================== */
/* Codeset conversion */

/*
 * Implementations before POSIX.1-2024 may not be good enough.
 *
 * The definition from SUSv2 is ambiguous:
 * <https://pubs.opengroup.org/onlinepubs/007908799/xsh/iconv.html>
 * Different semantics are used in header and definition!
 *
 * Attention:
 * The version from POSIX.1-2001 looks similar to the one from POSIX.1-2024:
 * <https://pubs.opengroup.org/onlinepubs/007904975/functions/iconv.html>
 * But there are operating systems that falsely claim POSIX.1-2001 conformance,
 * e.g. NetBSD before version 10 (released in 2024).
 */
#if CFG_USE_POSIX_API >= 202405L
#  include <iconv.h>

#  define api_posix_iconv_open  iconv_open
#  define api_posix_iconv_close  iconv_close
#  define api_posix_iconv  iconv
#endif  /* CFG_USE_POSIX_API >= 202405L */


/* EOF */
