/* ========================================================================== */
/*! \file
 * \brief Test of \c enc_timestamp_decode() implementation
 *
 * Copyright (c) 2012-2024 by the developers. See the LICENSE file for details.
 */


/* ========================================================================== */
/* Include headers */

#include "posix.h"  /* Include this first because of feature test macros */

#include <stdio.h>

#include "encoding.h"
#include "test.h"
#include "test_timestamp.h"


/* ========================================================================== */
/*! \addtogroup TEST */
/*! @{ */


/* ========================================================================== */
/*! \brief Test \c enc_timestamp_decode() implementation
 *
 * The following cases are tested:
 * - One second after epoche with timezone "+0000"
 * - One second after epoche with timezone "GMT"
 * - First second of Y2K using timezone "Z"
 * - Optional day of week and timezone "-0400"
 * - Optional day of week and timezone "+0200"
 * - Obsolete timezone "PDT"
 * - Leap year, leap second and timezone "+0131" (with nonzero minutes)
 * - Time of day without optional seconds and obsolete timezone "UT"
 *
 * \return
 * - \c EXIT_SUCCESS on success
 * - \c EXIT_FAILURE on error
 */

int  test_timestamp(void)
{
#define TS_NUM  (size_t) 8  /* Number of timestamps to test */
   static const char*  ts[TS_NUM] =
   {
      "1 Jan 1970 00:00:01 +0000",
      "1 Jan 1970 00:00:01 GMT",
      "1 Mar 2000 00:00:00 Z",
      "Wed, 12 Jun 2013 21:56:34 -0400",
      "Thu, 27 Jun 2013 00:01:26 +0200",
      "01 Jan 1999 22:30:59 PDT",
      "29 Feb 2036 22:30:60 +0131",
      "31 Dec 1980 23:30 UT"
   };
   static const core_time_t  t[TS_NUM] =  {
      1U,
      1U,
      951868800U,
      1371088594U,
      1372284086U,
      915255059U,
      2087931600U,
      347153400U
   };
   int  res = API_POSIX_EXIT_SUCCESS;
   int  rv;
   size_t  i;

   for(i = 0; i < TS_NUM; ++i)
   {
      rv = enc_timestamp_decode(ts[i]);
      if(!rv)
      {
         print_error("Timestamp decoder failed");
         res = API_POSIX_EXIT_FAILURE;
         break;
      }
      if(t[i] != rv)
      {
         print_error("Result is not correct");
         /* For debugging */
         fprintf(stderr, TEST_TAB "Timestamp: \"%s\"\n", ts[i]);
         fprintf(stderr, TEST_TAB "Result is: \"%lu\"\n",
                 (unsigned long int) rv);
         fprintf(stderr, TEST_TAB "Should be: \"%lu\"\n",
                 (unsigned long int) t[i]);
         res = API_POSIX_EXIT_FAILURE;
         break;
      }
   }

   return(res);
}


/*! @} */

/* EOF */
