/*
 * Base 64 decoding functions
 *
 * SPDX-FileType: SOURCE
 * SPDX-FileCopyrightText: Michael Bäuerle
 * SPDX-License-Identifier: BSD-2-Clause
 */

#include <limits.h>

#include "libbasexx-0/basexx_ebcdic.h"


#define BXX0_I_ASCII_TO_EBCDIC_SIZE  128U
#define BXX0_I_EBCDIC_TO_ASCII_SIZE  256U


/* ========================================================================== */
/*
 * Base 64 alphabet mapping table
 *
 * "c" must be an EBCDIC encoded Base 64 Alphabet character.
 *
 * Returns Base 64 alphabet US-ASCII character.
 * Returns 0x6F ("?") for Non-Alphabet characters.
 */
static unsigned char bxx0_i_ebcdic_to(const unsigned char c)
{
    /*
     * Elements of this table must be independent of the execution character
     * set (which "c-char" in "character-constant" from C90 is not).
     */
    static const unsigned char ascii_to_ebcdic[BXX0_I_ASCII_TO_EBCDIC_SIZE] =
    {
        /* 0x0X */
        0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
        0x6F, 0x6F, 0x25, 0x6F, 0x6F, 0x0D, 0x6F, 0x6F,
        /* 0x1X */
        0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
        0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
        /* 0x2X */
        0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F,
        0x6F, 0x6F, 0x6F, 0x4E, 0x6B, 0x60, 0x6F, 0x61,
        /* 0x3X */
        0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7,
        0xF8, 0xF9, 0x6F, 0x6F, 0x6F, 0x7E, 0x6F, 0x6F,
        /* 0x4X */
        0x6F, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7,
        0xC8, 0xC9, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6,
        /* 0x5X */
        0xD7, 0xD8, 0xD9, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6,
        0xE7, 0xE8, 0xE9, 0x6F, 0x6F, 0x6F, 0x6F, 0x6D,
        /* 0x6X */
        0x6F, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
        0x88, 0x89, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
        /* 0x7X */
        0x97, 0x98, 0x99, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6,
        0xA7, 0xA8, 0xA9, 0x6F, 0x6F, 0x6F, 0x6F, 0x6F
    };

    /* Check would be required for 256 elements too if (CHAR_BIT > 8) */
    return (BXX0_I_ASCII_TO_EBCDIC_SIZE <= c) ? 0x6F : ascii_to_ebcdic[c];

}


/* ========================================================================== */
/*
 * Base 64 alphabet mapping table
 *
 * "c" must be an EBCDIC encoded Base 64 Alphabet character.
 *
 * Returns Base 64 alphabet US-ASCII character.
 * Returns 0x3F ("?") for Non-Alphabet characters.
 */
static unsigned char bxx0_i_ebcdic_from(const unsigned char c)
{
    /*
     * Elements of this table must be independent of the execution character
     * set (which "c-char" in "character-constant" from C90 is not).
     */
    static const unsigned char ebcdic_to_ascii[BXX0_I_EBCDIC_TO_ASCII_SIZE] =
    {
        /* 0x0X */
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x0D, 0x3F, 0x3F,
        /* 0x1X */
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0x2X */
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x0A, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0x3X */
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0x4X */
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x2B, 0x3F,
        /* 0x5X */
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0x6X */
        0x2D, 0x2F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x2C, 0x3F, 0x5F, 0x3F, 0x3F,
        /* 0x7X */
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3D, 0x3F,
        /* 0x8X */
        0x3F, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
        0x68, 0x69, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0x9X */
        0x3F, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70,
        0x71, 0x72, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0xAX */
        0x3F, 0x3F, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
        0x79, 0x7A, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0xBX */
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0xCX */
        0x3F, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0xDX */
        0x3F, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
        0x51, 0x52, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0xEX */
        0x3F, 0x3F, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
        0x59, 0x5A, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
        /* 0xFX */
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F
    };

#if (8 < CHAR_BIT)
    return (BXX0_I_EBCDIC_TO_ASCII_SIZE <= c) ? 0x3F : ebcdic_to_ascii[c];
#else
    return ebcdic_to_ascii[c];
#endif

}


/* ========================================================================== */
void bxx0_ebcdic_to(unsigned char *buf, const size_t len)
{
    size_t i = 0;

    for (; len > i; ++i)
        buf[i] = bxx0_i_ebcdic_to(buf[i]);
}


/* ========================================================================== */
void bxx0_ebcdic_from(unsigned char *buf, const size_t len)
{
    size_t i = 0;

    for (; len > i; ++i)
        buf[i] = bxx0_i_ebcdic_from(buf[i]);
}
