/* freezetag - id3 Editor for Linux
 * Copyright (C), 1999, Jeremy Katz <katzj@linuxpower.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#include "freezetag.h"
#include "genre.h"

GtkWidget *window, *songbox, *artistbox, *albumbox, *yearbox, *commentbox, *genrebox;
gint song_flag = 0, artist_flag = 0, album_flag = 0, year_flag = 0, comments_flag = 0, genre_flag = 0;

/* things which need to be flagged if they change */
enum flags{ SONG, ARTIST, ALBUM, YEAR, COMMENTS, GENRE };

gchar *initial_text(GtkWidget *clist, gint column) {
	GList *list;
	gchar *tmp, *first;
	gint same = 1;

	list = GTK_CLIST(clist)->selection;
	gtk_clist_get_text( GTK_CLIST(clist), (gint) list->data, column, &first);
	list = list->next;
	while(list) {
		gtk_clist_get_text( GTK_CLIST(clist), (gint) list->data, column, &tmp);
		list = list->next;
		if(strcmp(first, tmp) != 0) {
			same = 0;
			break;
		}
	}

	if (same == 1) {
		return g_strstrip(first);
	} else {
		return (gchar *) "";
	}
}


void change_flag (GtkWidget *widget, gpointer data) {
	switch ( (gint) data ) {
		case SONG:
			song_flag = 1;
			break;
		case ARTIST:
			artist_flag = 1;
			break;
		case ALBUM:
			album_flag = 1;
			break;
		case YEAR:
			year_flag = 1;
			break;
		case COMMENTS:
			comments_flag = 1;
			break;
		case GENRE:
			genre_flag = 1;
			break;
		default:
#ifdef DEBUG
			g_print("Whoops... shouldn't get to this part of change_flag\n");
#endif
			break;
	}
}


void write_tags(GtkWidget *widget, gpointer data) {
	GList *filenames;
	gchar *tmp, *tmp2;
	int fd, length,i, genre;
	gchar *song, *artist, *album, *year, *comments;
	char tag[4];
	char fulltag = "TAG                                                                                                                             ";
	extern GtkWidget *filelist;

	if ( ! ((song_flag != 0) || (artist_flag != 0) || (album_flag != 0) || (year_flag != 0) || (comments_flag != 0) || (genre_flag != 0) )) {
		dialog_destroy( NULL, GTK_WIDGET (window) );
		return;
	}


	/* The following will do this if they have changed
	 * 1) get the string from the text entry box
	 * 2) make a filler string of spaces of 30 - strlen
	 * 3) cat strings together
	 */
	if (song_flag) {
		tmp = gtk_entry_get_text( GTK_ENTRY(songbox) );
		tmp2 = g_strnfill( 30 - strlen(tmp), ' ');
		song = g_strconcat(tmp, tmp2, NULL);
		g_free(tmp2);
	}

	if (artist_flag) {
		tmp = gtk_entry_get_text( GTK_ENTRY(artistbox) );
		tmp2 = g_strnfill( 30 - strlen(tmp), ' ');
		artist = g_strconcat(tmp, tmp2, NULL);
		g_free(tmp2);
	}

	if (album_flag) {
		tmp = gtk_entry_get_text( GTK_ENTRY(albumbox) );
		tmp2 = g_strnfill( 30 - strlen(tmp), ' ');
		album = g_strconcat(tmp, tmp2, NULL);
		g_free(tmp2);
	}

	if (year_flag) {
		tmp = gtk_entry_get_text( GTK_ENTRY(yearbox) );
		tmp2 = g_strnfill( 4 - strlen(tmp), ' ');
		year = g_strconcat(tmp, tmp2, NULL);
		g_free(tmp2);
	}

	if (comments_flag) {
		tmp = gtk_entry_get_text( GTK_ENTRY(commentbox) );
		tmp2 = g_strnfill( 30 - strlen(tmp), ' ');
		comments = g_strconcat(tmp, tmp2, NULL);
		g_free(tmp2);
	}

	if (genre_flag) {
		tmp = gtk_entry_get_text( GTK_ENTRY( GTK_COMBO(genrebox)->entry) );
		for(i = 0; i < genre_count; i++) {
			if ( strcasecmp(tmp, genre_table[i]) == 0) {
				genre = i;
				break;
			}
		}
	}

	filenames = data;
	while(filenames) {
		if( (fd = open(filenames->data, O_RDWR) ) < 0) {
			g_warning("Can't open file %s\n", filenames->data);
			filenames = filenames->next;
			continue;
		}

		lseek(fd, -128, SEEK_END);
		read(fd, tag, 3);
		if ( (tag[0] != 'T') || (tag[1] != 'A') || (tag[2] != 'G') ) {
			lseek(fd, 0, SEEK_END);
			write(fd, fulltag, 128);
		}

		if (song_flag) {
			lseek(fd, -125, SEEK_END);
			write(fd, song, 30);
#ifdef DEBUG
			g_print("Wrote song %s\n", song);
#endif
		}

		if (artist_flag) {
			lseek(fd, -95, SEEK_END);
			write(fd, artist, 30);
#ifdef DEBUG
			g_print("Wrote artist %s\n", artist);
#endif
		}


		if (album_flag) {
			lseek(fd, -65, SEEK_END);
			write(fd, album, 30);
#ifdef DEBUG
			g_print("Wrote album %s\n", album);
#endif
		}


		if (year_flag) {
			lseek(fd, -35, SEEK_END);
			write(fd, year, 4);
#ifdef DEBUG
			g_print("Wrote year %s\n", year);
#endif
		}

		if (comments_flag) {
			lseek(fd, -31, SEEK_END);
			write(fd, comments, 30);
#ifdef DEBUG
			g_print("Wrote comments %s\n", comments);
#endif
		}

		if ( genre_flag) {
			lseek(fd, -1, SEEK_END);
			tmp[0] = (char) genre;
			write(fd, tmp, 1);
#ifdef DEBUG
			g_print("Wrote genre %d\n", tmp[0]);
#endif
		}

		close(fd);
		filenames = filenames->next;
	}

	g_list_free(filenames);

	/* free any of the ones which changed */
	if (song_flag) g_free(song);
	if (artist_flag) g_free(artist);
	if (album_flag) g_free(album);
	if (year_flag) g_free(year);

	dialog_destroy( NULL, GTK_WIDGET (window) );
	refresh_filelist( GTK_OBJECT(filelist), GTK_CLIST(filelist) );
}


void edit_box(GtkWidget *clist) {
	GtkWidget *vbox, *bbox, *ok, *cancel, *songlabel, *artistlabel, *albumlabel, *yearlabel, *commentlabel, *genrelabel, *song, *artist, *album, *year, *comment, *genre;
	GList *list = NULL, *filenames = NULL;
	gchar *tmp;
	int i;

	song_flag = 0;
	artist_flag = 0;
	album_flag = 0;
	year_flag = 0;
	comments_flag = 0;
	genre_flag = 0;

	window = gtk_window_new (GTK_WINDOW_DIALOG);
	gtk_widget_set_usize( GTK_WIDGET (window), 300,300);
	
	vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add( GTK_CONTAINER (window), vbox);
	gtk_widget_show(vbox);

	bbox = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_end( GTK_BOX (vbox), bbox, TRUE, FALSE, 10);
	gtk_widget_show(bbox);

	ok = gtk_button_new_with_label("Ok");
	gtk_box_pack_start( GTK_BOX (bbox), ok, TRUE, TRUE, 5);
	gtk_widget_show(ok);

	cancel = gtk_button_new_with_label("Cancel");
	gtk_box_pack_start( GTK_BOX (bbox), cancel, TRUE, TRUE, 5);
	gtk_widget_show(cancel);

	gtk_signal_connect(GTK_OBJECT (cancel), "clicked", GTK_SIGNAL_FUNC (dialog_destroy), GTK_OBJECT (window) );

	/* Whee... this gets messy here due to large number of 
	 * entry boxes... oh well, there's probably a better way
	 * to do it */

	/* Song Name Section */
	song = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start( GTK_BOX(vbox), song, TRUE, FALSE, 5);
	gtk_widget_show(song);
	songlabel = gtk_label_new("Title");
	gtk_box_pack_start( GTK_BOX (song), songlabel, TRUE, FALSE, 5);
	gtk_widget_show(songlabel);
	songbox = gtk_entry_new_with_max_length(30);
	gtk_widget_set_usize ( GTK_WIDGET (songbox), 200, 20);
	gtk_box_pack_start( GTK_BOX (song), songbox, TRUE, FALSE, 5);
	gtk_widget_show(songbox);

	/* Artist Section */
	artist = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start( GTK_BOX(vbox), artist, TRUE, FALSE, 5);
	gtk_widget_show(artist);
	artistlabel = gtk_label_new("Artist");
	gtk_box_pack_start( GTK_BOX (artist), artistlabel, TRUE, FALSE, 5);
	gtk_widget_show(artistlabel);
	artistbox = gtk_entry_new_with_max_length(30);
	gtk_widget_set_usize ( GTK_WIDGET (artistbox), 200, 20);
	gtk_box_pack_start( GTK_BOX (artist), artistbox, TRUE, FALSE, 5);
	gtk_widget_show(artistbox);

	/* Album Section */
	album = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start( GTK_BOX(vbox), album, TRUE, FALSE, 5);
	gtk_widget_show(album);
	albumlabel = gtk_label_new("Album");
	gtk_box_pack_start( GTK_BOX (album), albumlabel, TRUE, FALSE, 5);
	gtk_widget_show(albumlabel);
	albumbox = gtk_entry_new_with_max_length(30);
	gtk_widget_set_usize ( GTK_WIDGET (albumbox), 200, 20);
	gtk_box_pack_start( GTK_BOX (album), albumbox, TRUE, FALSE, 5);
	gtk_widget_show(albumbox);

	/* Year Section */
	year = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start( GTK_BOX(vbox), year, TRUE, FALSE, 5);
	gtk_widget_show(year);
	yearlabel = gtk_label_new("Year");
	gtk_box_pack_start( GTK_BOX (year), yearlabel, TRUE, FALSE, 5);
	gtk_widget_show(yearlabel);
	yearbox = gtk_entry_new_with_max_length(4);
	gtk_widget_set_usize ( GTK_WIDGET (yearbox), 200, 20);
	gtk_box_pack_start( GTK_BOX (year), yearbox, TRUE, FALSE, 5);
	gtk_widget_show(yearbox);

	/* Comment Section */
	comment = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start( GTK_BOX(vbox), comment, TRUE, FALSE, 5);
	gtk_widget_show(comment);
	commentlabel = gtk_label_new("Comment");
	gtk_box_pack_start( GTK_BOX (comment), commentlabel, TRUE, FALSE, 5);
	gtk_widget_show(commentlabel);
	commentbox = gtk_entry_new_with_max_length(30);
	gtk_widget_set_usize ( GTK_WIDGET (commentbox), 200, 20);
	gtk_box_pack_start( GTK_BOX (comment), commentbox, TRUE, FALSE, 5);
	gtk_widget_show(commentbox);

	/* Genre Section */
	genre = gtk_hbox_new(FALSE, 10);
	gtk_box_pack_start( GTK_BOX(vbox), genre, TRUE, FALSE, 5);
	gtk_widget_show(genre);
	genrelabel = gtk_label_new("Genre");
	gtk_box_pack_start( GTK_BOX (genre), genrelabel, TRUE, FALSE, 5);
	gtk_widget_show(genrelabel);
	genrebox = gtk_combo_new();
	list = g_list_append(list, (gchar *) " ");
	for(i = 0; i < genre_count; i++) {
		list = g_list_append(list, genre_table[i]);
	}
	gtk_combo_set_popdown_strings( GTK_COMBO (genrebox), list );
	g_list_free(list);
	gtk_combo_set_use_arrows( GTK_COMBO (genrebox), TRUE);
	gtk_combo_set_case_sensitive( GTK_COMBO (genrebox), FALSE);
	gtk_box_pack_start( GTK_BOX (genre), genrebox, TRUE, FALSE, 5);
	gtk_widget_show(genrebox);

	
	list = GTK_CLIST(clist)->selection;
	while(list) {
		gtk_clist_get_text( GTK_CLIST(clist), (gint) list->data, 0, &tmp);
		filenames = g_list_append( filenames, tmp);
		list = list->next;
	}
	g_list_free(list);

	/* abstraction is good... this should work to set the stuff in either
	 * of two conditions - a) one file selected, b) multiple files
	 * selected which share common attributes
	 */
	gtk_entry_set_text( GTK_ENTRY(songbox), initial_text( GTK_WIDGET(clist), 1) );
	gtk_entry_set_text( GTK_ENTRY(artistbox), initial_text( GTK_WIDGET(clist), 2) );
	gtk_entry_set_text( GTK_ENTRY(albumbox), initial_text( GTK_WIDGET(clist), 3) );
	gtk_entry_set_text( GTK_ENTRY(yearbox), initial_text( GTK_WIDGET(clist), 4) );
	gtk_entry_set_text( GTK_ENTRY(commentbox), initial_text( GTK_WIDGET(clist), 5) );
	gtk_entry_set_text( GTK_ENTRY( GTK_COMBO (genrebox)->entry ), initial_text( GTK_WIDGET(clist), 6) );
	
	

	gtk_signal_connect( GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(write_tags), filenames);
	gtk_signal_connect( GTK_OBJECT(GTK_ENTRY(songbox)), "activate", GTK_SIGNAL_FUNC(write_tags), filenames);
	gtk_signal_connect( GTK_OBJECT(GTK_ENTRY(artistbox)), "activate", GTK_SIGNAL_FUNC(write_tags), filenames);
	gtk_signal_connect( GTK_OBJECT(GTK_ENTRY(albumbox)), "activate", GTK_SIGNAL_FUNC(write_tags), filenames);
	gtk_signal_connect( GTK_OBJECT(GTK_ENTRY(yearbox)), "activate", GTK_SIGNAL_FUNC(write_tags), filenames);
	gtk_signal_connect( GTK_OBJECT(GTK_ENTRY(commentbox)), "activate", GTK_SIGNAL_FUNC(write_tags), filenames);
	gtk_signal_connect( GTK_OBJECT(GTK_ENTRY(GTK_COMBO(genrebox)->entry)), "activate", GTK_SIGNAL_FUNC(write_tags), filenames);


	/* Need the callbacks so we know when something has changed */
	gtk_signal_connect( GTK_OBJECT( songbox ), "changed", GTK_SIGNAL_FUNC (change_flag), (gint *) SONG );
	gtk_signal_connect( GTK_OBJECT( artistbox ), "changed", GTK_SIGNAL_FUNC (change_flag), (gint *) ARTIST );
	gtk_signal_connect( GTK_OBJECT( albumbox ), "changed", GTK_SIGNAL_FUNC (change_flag), (gint *) ALBUM );
	gtk_signal_connect( GTK_OBJECT( yearbox ), "changed", GTK_SIGNAL_FUNC (change_flag), (gint *) YEAR );
	gtk_signal_connect( GTK_OBJECT( commentbox ), "changed", GTK_SIGNAL_FUNC (change_flag), (gint *) COMMENTS );
	gtk_signal_connect( GTK_OBJECT( GTK_COMBO (genrebox)->entry ), "changed", GTK_SIGNAL_FUNC (change_flag), (gint *) GENRE );

	gtk_widget_show(window);
	return;
}
