/*	$Id: gkrellm-est.c,v 1.2 2004/07/10 16:44:11 cube Exp $	*/

/*
 *  Copyright (c) 2004 Quentin Garnier.
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *  3. All advertising materials mentioning features or use of this software
 *     must display the following acknowledgement:
 *         This product includes software developed by Quentin Garnier.
 *  4. Neither the name of The NetBSD Foundation nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 * 
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
#include <sys/param.h>
#include <sys/sysctl.h>
#include <errno.h>

#include <gkrellm2/gkrellm.h>

static void	gkrellm_est_create(GtkWidget *, gint);
static void	gkrellm_est_update();
GkrellmMonitor	*gkrellm_init_plugin();
static void	gkrellm_est_config_tab(GtkWidget *);

static int32_t	est_curfreq();
static void	est_change_freq(int);

static gint	est_cb_expose_event(GtkWidget *, GdkEventExpose *);
static gint	est_cb_button_press_event(GtkWidget *, GdkEventButton *);

#define	GKRELLM_EST_NAME	"gkrellm-est"

static GkrellmMonitor gkrellm_est = {
	GKRELLM_EST_NAME,		/* name */
	0,				/* id */
	gkrellm_est_create,		/* create_monitor */
	gkrellm_est_update,		/* update_monitor */
	gkrellm_est_config_tab,		/* create_config */
	NULL,				/* apply_config */
	NULL,				/* save_user_config */
	NULL,				/* load_user_config */
	GKRELLM_EST_NAME,		/* config_keyword */
	NULL,				/* undef2 */
	NULL,				/* undef1 */
	NULL,				/* private */
	MON_BATTERY,			/* insert_before_id */
	NULL,				/* handle */
	NULL,				/* path */
};

static char		est_frequencies[MAXPATHLEN];
static uint32_t		est_freqs[10];
static int		est_nfreqs;

static int		est_freq_mib[CTL_MAXNAME];
static size_t		est_freq_namelen = CTL_MAXNAME;
static int		est_targ_mib[CTL_MAXNAME];
static size_t		est_targ_namelen = CTL_MAXNAME;

static gint		est_style_id;

GkrellmMonitor *
gkrellm_init_plugin()
{
	char *freqs, *freq, *p;
	int i;
	size_t buflen = MAXPATHLEN;

	if (sysctlbyname("machdep.est.frequency.available",
	    (void *)est_frequencies, &buflen, NULL, 0) == -1)
		return NULL;

	if (sysctlnametomib("machdep.est.frequency.target", est_targ_mib,
	    &est_targ_namelen) == -1 ||
	    sysctlnametomib("machdep.est.frequency.current", est_freq_mib,
	    &est_freq_namelen) == -1)
		return NULL;

	/* Parse frequencies */
	freqs = strdup(est_frequencies);
	est_nfreqs = 0;
	for (i = 0, p = freqs; i < 10 && p != NULL; i++) {
		freq = strsep(&p, " ");
		if (*freq == '\0')
			continue;
		est_freqs[est_nfreqs++] = atoi(freq);
	}
	if (freqs != NULL)
		free(freqs);

	est_style_id = gkrellm_add_meter_style(&gkrellm_est, GKRELLM_EST_NAME);
	return &gkrellm_est;
}

static int32_t
est_curfreq()
{
	int32_t r;
	size_t l = sizeof(r);

	if (sysctl(est_freq_mib, est_freq_namelen, (void *)&r, &l, NULL,
	    0) == -1)
		return (-1);

	return (r);
}

static void
est_change_freq(int dir)
{
	int i;
	int32_t cur;

	if ((est_nfreqs < 2) || ((cur = est_curfreq()) == -1))
		return;

	for (i = 0; i < 10; i++)
		if (est_freqs[i] == cur)
			break;
	if (i >= 10)
		return;

	i += dir;
	if (i < 0)
		i = 0;
	else if (i >= 10)
		i = 9;

	(void)sysctl(est_targ_mib, est_targ_namelen, NULL, NULL, &est_freqs[i],
		sizeof(est_freqs[0]));
}

static GkrellmPanel *est_panel;
static GkrellmDecal *est_decal;

static gint
est_cb_expose_event(GtkWidget *widget, GdkEventExpose *ev)
{
	if (widget == est_panel->drawing_area)
		gdk_draw_drawable(widget->window, gkrellm_draw_GC(1),
		    est_panel->pixmap, ev->area.x, ev->area.y, ev->area.x,
		    ev->area.y, ev->area.width, ev->area.height);
	return (FALSE);
}

static gint
est_cb_button_press_event(GtkWidget *widget, GdkEventButton *ev)
{
	/* Left or right buttons */
	if (ev->button == 1 || ev->button == 3)
		est_change_freq(ev->button - 2);

	return (FALSE);
}

static void
gkrellm_est_create(GtkWidget *vbox, gint first_create)
{
	GkrellmStyle *style;

	if (first_create)
		est_panel = gkrellm_panel_new0();
	style = gkrellm_meter_style(est_style_id);
	est_decal = gkrellm_create_decal_text(est_panel, "8000",
	    gkrellm_meter_textstyle(est_style_id), style, -1, -1, -1);
	gkrellm_panel_configure(est_panel, NULL, style);
	gkrellm_panel_configure_add_height(est_panel, 1);
	gkrellm_panel_create(vbox, &gkrellm_est, est_panel);

	if (first_create) {
		g_signal_connect(G_OBJECT(est_panel->drawing_area),
		    "expose_event", G_CALLBACK(est_cb_expose_event), NULL);
		g_signal_connect(G_OBJECT(est_panel->drawing_area),
		    "button_press_event", G_CALLBACK(est_cb_button_press_event), NULL);
	}

	gkrellm_draw_decal_text(est_panel, est_decal, "test123", -1);
	gkrellm_draw_panel_layers(est_panel);
}

static void
gkrellm_est_update()
{
	char buf[6];
	snprintf(buf, 6, "%d", est_curfreq());
	gkrellm_draw_decal_text(est_panel, est_decal, buf, -1);
	gkrellm_draw_panel_layers(est_panel);
}

static const char gkrellm_est_about[] =
	"gkrellm-est version 20040710\n\n"
	"GKrellM plug-in to display current processor frequency\n"
	"and let the user change it by clicking the Krell.\n\n"
	"It only works on EST-capable processors running\n"
	"NetBSD-current (from April, 30th or newer).\n\n"
	"Copyright (c) Quentin Garnier, 2004.\n"
	"http://cubidou.nerim.net/gkrellm-est/\n";

static void
gkrellm_est_config_tab(GtkWidget *vbox)
{
	GtkWidget *tabs, *about, *label;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(vbox), tabs, TRUE, TRUE, 0);

	about = gtk_label_new(gkrellm_est_about);
	label = gtk_label_new("About");
	gtk_notebook_append_page(GTK_NOTEBOOK(tabs), about, label);
}
