/*----------------------------------------------------------------------------
  Simple class support routines such as list manipulation.
----------------------------------------------------------------------------*/
#include "dbddr.h"

void (*dbDesignDestructorHook)(dbDesign);
dbDesign _dbFirstFreeDesign, _dbTempDesign;
utHeapRef _dbDesignHeap;
void (*dbNetlistDestructorHook)(dbNetlist);
dbNetlist _dbFirstFreeNetlist, _dbTempNetlist;
utHeapRef _dbNetlistHeap;
void (*dbNetDestructorHook)(dbNet);
dbNet _dbFirstFreeNet, _dbTempNet;
utHeapRef _dbNetHeap;
void (*dbInstDestructorHook)(dbInst);
dbInst _dbFirstFreeInst, _dbTempInst;
utHeapRef _dbInstHeap;
void (*dbMportDestructorHook)(dbMport);
dbMport _dbFirstFreeMport, _dbTempMport;
utHeapRef _dbMportHeap;
void (*dbPortDestructorHook)(dbPort);
dbPort _dbFirstFreePort, _dbTempPort;
utHeapRef _dbPortHeap;
void (*dbMbusDestructorHook)(dbMbus);
dbMbus _dbFirstFreeMbus, _dbTempMbus;
utHeapRef _dbMbusHeap;
void (*dbBusDestructorHook)(dbBus);
dbBus _dbFirstFreeBus, _dbTempBus;
utHeapRef _dbBusHeap;
void (*dbHierDestructorHook)(dbHier);
dbHier _dbFirstFreeHier, _dbTempHier;
utHeapRef _dbHierHeap;
void (*dbGlobalDestructorHook)(dbGlobal);
dbGlobal _dbFirstFreeGlobal, _dbTempGlobal;
utHeapRef _dbGlobalHeap;
void (*dbAttrDestructorHook)(dbAttr);
dbAttr _dbFirstFreeAttr, _dbTempAttr;
utHeapRef _dbAttrHeap;
void (*dbRootDestructorHook)(dbRoot);
dbRoot _dbFirstFreeRoot, _dbTempRoot;
utHeapRef _dbRootHeap;
void (*dbDevspecDestructorHook)(dbDevspec);
dbDevspec _dbFirstFreeDevspec, _dbTempDevspec;
utHeapRef _dbDevspecHeap;

/*----------------------------------------------------------------------------
  Add the Netlist to the tail of the list on the Design.
----------------------------------------------------------------------------*/
void dbDesignInsertNetlist(
    dbDesign Design,
    dbNetlist _Netlist)
{
    utInsertHtblEntry(dbDesignGetNetlists(Design), &(_Netlist)->h);
    dbNetlistSetDesign(_Netlist, Design);
}

/*----------------------------------------------------------------------------
  Remove the Netlist from the list on the Design.
----------------------------------------------------------------------------*/
void dbDesignRemoveNetlist(
    dbDesign Design,
    dbNetlist _Netlist)
{
    utDeleteHtblEntry(&(Design)->Netlists, &(_Netlist)->h);
    dbNetlistSetDesign(_Netlist, dbDesignNull);
}

/*----------------------------------------------------------------------------
  Append the Netlist to the tail of the list on the Design.
----------------------------------------------------------------------------*/
void dbDesignAppendNetlist(
    dbDesign Design,
    dbNetlist _Netlist)
{
    utAppendHtblEntry(dbDesignGetNetlists(Design), &(_Netlist)->h);
    dbNetlistSetDesign(_Netlist, Design);
}

/*----------------------------------------------------------------------------
  Add the Global to the tail of the list on the Design.
----------------------------------------------------------------------------*/
void dbDesignInsertGlobal(
    dbDesign Design,
    dbGlobal _Global)
{
    utInsertHtblEntry(dbDesignGetGlobals(Design), &(_Global)->h);
    dbGlobalSetDesign(_Global, Design);
}

/*----------------------------------------------------------------------------
  Remove the Global from the list on the Design.
----------------------------------------------------------------------------*/
void dbDesignRemoveGlobal(
    dbDesign Design,
    dbGlobal _Global)
{
    utDeleteHtblEntry(&(Design)->Globals, &(_Global)->h);
    dbGlobalSetDesign(_Global, dbDesignNull);
}

/*----------------------------------------------------------------------------
  Append the Global to the tail of the list on the Design.
----------------------------------------------------------------------------*/
void dbDesignAppendGlobal(
    dbDesign Design,
    dbGlobal _Global)
{
    utAppendHtblEntry(dbDesignGetGlobals(Design), &(_Global)->h);
    dbGlobalSetDesign(_Global, Design);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Design.
------------------------------------------------------------------------*/
static utBlockRef buildDesignBlock(void)
{
    utBlockRef block = utcBlock();
    dbDesign Design;
    U16 x, sDesign;

    utaHeapBlock(_dbDesignHeap, block);
    sDesign = UTBLOCKSIZE/sizeof(struct _dbDesign);
    _dbFirstFreeDesign = (dbDesign)utgBlockMem(block);
    Design = _dbFirstFreeDesign;
    for(x = 0; x < sDesign; x++) {
        _dbDesignNextFree(Design) = Design + 1;
        Design++;
    }
    _dbDesignNextFree(--Design) = dbDesignNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Design.
----------------------------------------------------------------------------*/
dbDesign dbDesignAlloc(void)
{
    dbDesign Design;

    if(_dbFirstFreeDesign == dbDesignNull) {
        buildDesignBlock();
    }
    Design = _dbFirstFreeDesign;
    _dbFirstFreeDesign = _dbDesignNextFree(Design);
    memset((void *)Design, 0, sizeof(struct _dbDesign));
    utInitHtbl(&Design->Netlists);
    utInitHtbl(&Design->Globals);
    return Design;
}

/*----------------------------------------------------------------------------
  Free a Design.
----------------------------------------------------------------------------*/
void dbDesignFree(
    dbDesign Design)
{
    utdArray(utfHtblArray(dbDesignGetNetlists(Design)));
    utdArray(utfHtblArray(dbDesignGetGlobals(Design)));
    _dbDesignNextFree(Design) = _dbFirstFreeDesign;
    _dbFirstFreeDesign = Design;
}

/*----------------------------------------------------------------------------
  Destroy a Design.
----------------------------------------------------------------------------*/
void dbDesignDestroy(
    dbDesign Design)
{
    if(dbDesignDestructorHook != NULL) {
        dbDesignDestructorHook(Design);
    }
    {
        dbNetlist _Netlist;
        dbSafeForeachDesignNetlist(Design, _Netlist) {
            dbNetlistDestroy(_Netlist);
        } dbEndSafeForeachDesignNetlist;
    }
    {
        dbGlobal _Global;
        dbSafeForeachDesignGlobal(Design, _Global) {
            dbGlobalDestroy(_Global);
        } dbEndSafeForeachDesignGlobal;
    }
    if(dbDesignGetAttr(Design) != dbAttrNull) {
        dbAttrDestroy(dbDesignGetAttr(Design));
    }
    {
        dbRoot parentRoot = dbDesignGetRoot(Design);
        if(parentRoot != dbRootNull) {
            dbRootRemoveDesign(parentRoot, Design);
        }
    }
    dbDesignFree(Design);
}

/*----------------------------------------------------------------------------
  Add the Net to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistInsertNet(
    dbNetlist Netlist,
    dbNet _Net)
{
    utInsertHtblEntry(dbNetlistGetNets(Netlist), &(_Net)->h);
    dbNetSetNetlist(_Net, Netlist);
}

/*----------------------------------------------------------------------------
  Remove the Net from the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistRemoveNet(
    dbNetlist Netlist,
    dbNet _Net)
{
    utDeleteHtblEntry(&(Netlist)->Nets, &(_Net)->h);
    dbNetSetNetlist(_Net, dbNetlistNull);
}

/*----------------------------------------------------------------------------
  Append the Net to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistAppendNet(
    dbNetlist Netlist,
    dbNet _Net)
{
    utAppendHtblEntry(dbNetlistGetNets(Netlist), &(_Net)->h);
    dbNetSetNetlist(_Net, Netlist);
}

/*----------------------------------------------------------------------------
  Add the Mport to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistInsertMport(
    dbNetlist Netlist,
    dbMport _Mport)
{
    utInsertHtblEntry(dbNetlistGetMports(Netlist), &(_Mport)->h);
    dbMportSetNetlist(_Mport, Netlist);
}

/*----------------------------------------------------------------------------
  Remove the Mport from the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistRemoveMport(
    dbNetlist Netlist,
    dbMport _Mport)
{
    utDeleteHtblEntry(&(Netlist)->Mports, &(_Mport)->h);
    dbMportSetNetlist(_Mport, dbNetlistNull);
}

/*----------------------------------------------------------------------------
  Append the Mport to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistAppendMport(
    dbNetlist Netlist,
    dbMport _Mport)
{
    utAppendHtblEntry(dbNetlistGetMports(Netlist), &(_Mport)->h);
    dbMportSetNetlist(_Mport, Netlist);
}

/*----------------------------------------------------------------------------
  Add the Inst to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistInsertInst(
    dbNetlist Netlist,
    dbInst _Inst)
{
    utInsertHtblEntry(dbNetlistGetInsts(Netlist), &(_Inst)->h);
    dbInstSetNetlist(_Inst, Netlist);
}

/*----------------------------------------------------------------------------
  Remove the Inst from the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistRemoveInst(
    dbNetlist Netlist,
    dbInst _Inst)
{
    utDeleteHtblEntry(&(Netlist)->Insts, &(_Inst)->h);
    dbInstSetNetlist(_Inst, dbNetlistNull);
}

/*----------------------------------------------------------------------------
  Append the Inst to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistAppendInst(
    dbNetlist Netlist,
    dbInst _Inst)
{
    utAppendHtblEntry(dbNetlistGetInsts(Netlist), &(_Inst)->h);
    dbInstSetNetlist(_Inst, Netlist);
}

/*----------------------------------------------------------------------------
  Add the Mbus to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistInsertMbus(
    dbNetlist Netlist,
    dbMbus _Mbus)
{
    utInsertHtblEntry(dbNetlistGetMbuss(Netlist), &(_Mbus)->h);
    dbMbusSetNetlist(_Mbus, Netlist);
}

/*----------------------------------------------------------------------------
  Remove the Mbus from the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistRemoveMbus(
    dbNetlist Netlist,
    dbMbus _Mbus)
{
    utDeleteHtblEntry(&(Netlist)->Mbuss, &(_Mbus)->h);
    dbMbusSetNetlist(_Mbus, dbNetlistNull);
}

/*----------------------------------------------------------------------------
  Append the Mbus to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistAppendMbus(
    dbNetlist Netlist,
    dbMbus _Mbus)
{
    utAppendHtblEntry(dbNetlistGetMbuss(Netlist), &(_Mbus)->h);
    dbMbusSetNetlist(_Mbus, Netlist);
}

/*----------------------------------------------------------------------------
  Add the Bus to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistInsertBus(
    dbNetlist Netlist,
    dbBus _Bus)
{
    utInsertHtblEntry(dbNetlistGetBuss(Netlist), &(_Bus)->h);
    dbBusSetNetlist(_Bus, Netlist);
}

/*----------------------------------------------------------------------------
  Remove the Bus from the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistRemoveBus(
    dbNetlist Netlist,
    dbBus _Bus)
{
    utDeleteHtblEntry(&(Netlist)->Buss, &(_Bus)->h);
    dbBusSetNetlist(_Bus, dbNetlistNull);
}

/*----------------------------------------------------------------------------
  Append the Bus to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistAppendBus(
    dbNetlist Netlist,
    dbBus _Bus)
{
    utAppendHtblEntry(dbNetlistGetBuss(Netlist), &(_Bus)->h);
    dbBusSetNetlist(_Bus, Netlist);
}

/*----------------------------------------------------------------------------
  Add the ExternalInst to the head of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistInsertExternalInst(
    dbNetlist Netlist,
    dbInst _Inst)
{
    dbInstSetnextNetlistExternalInst(_Inst, dbNetlistGetfirstExternalInst(Netlist));
    dbNetlistSetfirstExternalInst(Netlist, _Inst);
    if(dbNetlistGetlastExternalInst(Netlist) == dbInstNull) {
        dbNetlistSetlastExternalInst(Netlist, _Inst);
    } else{
        dbInstSetprevNetlistExternalInst(dbInstGetnextNetlistExternalInst(_Inst), _Inst);
    }
    dbInstSetInternalNetlist(_Inst, Netlist);
}

/*----------------------------------------------------------------------------
  Remove the ExternalInst from the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistRemoveExternalInst(
    dbNetlist Netlist,
    dbInst _Inst)
{
    dbInst nextInst = dbInstGetnextNetlistExternalInst(_Inst);
    dbInst prevInst = dbInstGetprevNetlistExternalInst(_Inst);

    if(prevInst == dbInstNull && dbNetlistGetfirstExternalInst(Netlist) != _Inst) {
        /* Not in list */
        return;
    }
    if(prevInst != dbInstNull) {
        dbInstSetnextNetlistExternalInst(prevInst, nextInst);
    } else {
        dbNetlistSetfirstExternalInst(Netlist, nextInst);
    }
    if(nextInst != dbInstNull) {
        dbInstSetprevNetlistExternalInst(nextInst, prevInst);
    } else {
        dbNetlistSetlastExternalInst(Netlist, prevInst);
    }
    dbInstSetprevNetlistExternalInst(_Inst, dbInstNull);
    dbInstSetnextNetlistExternalInst(_Inst, dbInstNull);
    dbInstSetInternalNetlist(_Inst, dbNetlistNull);
}

/*----------------------------------------------------------------------------
  Add the ExternalInst to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void dbNetlistAppendExternalInst(
    dbNetlist Netlist,
    dbInst _Inst)
{
    if(dbNetlistGetlastExternalInst(Netlist) == dbInstNull) {
        dbNetlistSetfirstExternalInst(Netlist, _Inst);
    } else {
        dbInstSetnextNetlistExternalInst(dbNetlistGetlastExternalInst(Netlist), _Inst);
        dbInstSetprevNetlistExternalInst(_Inst, dbNetlistGetlastExternalInst(Netlist));
    }
    dbNetlistSetlastExternalInst(Netlist, _Inst);
    dbInstSetInternalNetlist(_Inst, Netlist);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Netlist.
------------------------------------------------------------------------*/
static utBlockRef buildNetlistBlock(void)
{
    utBlockRef block = utcBlock();
    dbNetlist Netlist;
    U16 x, sNetlist;

    utaHeapBlock(_dbNetlistHeap, block);
    sNetlist = UTBLOCKSIZE/sizeof(struct _dbNetlist);
    _dbFirstFreeNetlist = (dbNetlist)utgBlockMem(block);
    Netlist = _dbFirstFreeNetlist;
    for(x = 0; x < sNetlist; x++) {
        _dbNetlistNextFree(Netlist) = Netlist + 1;
        Netlist++;
    }
    _dbNetlistNextFree(--Netlist) = dbNetlistNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Netlist.
----------------------------------------------------------------------------*/
dbNetlist dbNetlistAlloc(void)
{
    dbNetlist Netlist;

    if(_dbFirstFreeNetlist == dbNetlistNull) {
        buildNetlistBlock();
    }
    Netlist = _dbFirstFreeNetlist;
    _dbFirstFreeNetlist = _dbNetlistNextFree(Netlist);
    memset((void *)Netlist, 0, sizeof(struct _dbNetlist));
    utInitHtbl(&Netlist->Nets);
    utInitHtbl(&Netlist->Mports);
    utInitHtbl(&Netlist->Insts);
    utInitHtbl(&Netlist->Mbuss);
    utInitHtbl(&Netlist->Buss);
    return Netlist;
}

/*----------------------------------------------------------------------------
  Free a Netlist.
----------------------------------------------------------------------------*/
void dbNetlistFree(
    dbNetlist Netlist)
{
    utdArray(utfHtblArray(dbNetlistGetNets(Netlist)));
    utdArray(utfHtblArray(dbNetlistGetMports(Netlist)));
    utdArray(utfHtblArray(dbNetlistGetInsts(Netlist)));
    utdArray(utfHtblArray(dbNetlistGetMbuss(Netlist)));
    utdArray(utfHtblArray(dbNetlistGetBuss(Netlist)));
    _dbNetlistNextFree(Netlist) = _dbFirstFreeNetlist;
    _dbFirstFreeNetlist = Netlist;
}

/*----------------------------------------------------------------------------
  Destroy a Netlist.
----------------------------------------------------------------------------*/
void dbNetlistDestroy(
    dbNetlist Netlist)
{
    if(dbNetlistDestructorHook != NULL) {
        dbNetlistDestructorHook(Netlist);
    }
    {
        utExtension child, nextChild;
        for(child = dbNetlistGetfirstExtension(Netlist); child != NULL;
            child = nextChild) {
            nextChild = child->next;
            child->functionTable->destroy((void *)child);
        }
    }
    {
        dbNet _Net;
        dbSafeForeachNetlistNet(Netlist, _Net) {
            dbNetDestroy(_Net);
        } dbEndSafeForeachNetlistNet;
    }
    {
        dbMport _Mport;
        dbSafeForeachNetlistMport(Netlist, _Mport) {
            dbMportDestroy(_Mport);
        } dbEndSafeForeachNetlistMport;
    }
    {
        dbInst _Inst;
        dbSafeForeachNetlistInst(Netlist, _Inst) {
            dbInstDestroy(_Inst);
        } dbEndSafeForeachNetlistInst;
    }
    {
        dbMbus _Mbus;
        dbSafeForeachNetlistMbus(Netlist, _Mbus) {
            dbMbusDestroy(_Mbus);
        } dbEndSafeForeachNetlistMbus;
    }
    {
        dbBus _Bus;
        dbSafeForeachNetlistBus(Netlist, _Bus) {
            dbBusDestroy(_Bus);
        } dbEndSafeForeachNetlistBus;
    }
    if(dbNetlistGetHier(Netlist) != dbHierNull) {
        dbHierDestroy(dbNetlistGetHier(Netlist));
    }
    {
        dbInst _Inst;
        dbSafeForeachNetlistExternalInst(Netlist, _Inst) {
            dbInstDestroy(_Inst);
        } dbEndSafeForeachNetlistExternalInst;
    }
    if(dbNetlistGetAttr(Netlist) != dbAttrNull) {
        dbAttrDestroy(dbNetlistGetAttr(Netlist));
    }
    {
        dbDesign parentDesign = dbNetlistGetDesign(Netlist);
        if(parentDesign != dbDesignNull) {
            dbDesignRemoveNetlist(parentDesign, Netlist);
        }
    }
    dbNetlistFree(Netlist);
}

/*----------------------------------------------------------------------------
  Add the Port to the head of the list on the Net.
----------------------------------------------------------------------------*/
void dbNetInsertPort(
    dbNet Net,
    dbPort _Port)
{
    dbPortSetnextNetPort(_Port, dbNetGetfirstPort(Net));
    dbNetSetfirstPort(Net, _Port);
    if(dbNetGetlastPort(Net) == dbPortNull) {
        dbNetSetlastPort(Net, _Port);
    } else{
        dbPortSetprevNetPort(dbPortGetnextNetPort(_Port), _Port);
    }
    dbPortSetNet(_Port, Net);
}

/*----------------------------------------------------------------------------
  Remove the Port from the list on the Net.
----------------------------------------------------------------------------*/
void dbNetRemovePort(
    dbNet Net,
    dbPort _Port)
{
    dbPort nextPort = dbPortGetnextNetPort(_Port);
    dbPort prevPort = dbPortGetprevNetPort(_Port);

    if(prevPort == dbPortNull && dbNetGetfirstPort(Net) != _Port) {
        /* Not in list */
        return;
    }
    if(prevPort != dbPortNull) {
        dbPortSetnextNetPort(prevPort, nextPort);
    } else {
        dbNetSetfirstPort(Net, nextPort);
    }
    if(nextPort != dbPortNull) {
        dbPortSetprevNetPort(nextPort, prevPort);
    } else {
        dbNetSetlastPort(Net, prevPort);
    }
    dbPortSetprevNetPort(_Port, dbPortNull);
    dbPortSetnextNetPort(_Port, dbPortNull);
    dbPortSetNet(_Port, dbNetNull);
}

/*----------------------------------------------------------------------------
  Add the Port to the tail of the list on the Net.
----------------------------------------------------------------------------*/
void dbNetAppendPort(
    dbNet Net,
    dbPort _Port)
{
    if(dbNetGetlastPort(Net) == dbPortNull) {
        dbNetSetfirstPort(Net, _Port);
    } else {
        dbPortSetnextNetPort(dbNetGetlastPort(Net), _Port);
        dbPortSetprevNetPort(_Port, dbNetGetlastPort(Net));
    }
    dbNetSetlastPort(Net, _Port);
    dbPortSetNet(_Port, Net);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Net.
------------------------------------------------------------------------*/
static utBlockRef buildNetBlock(void)
{
    utBlockRef block = utcBlock();
    dbNet Net;
    U16 x, sNet;

    utaHeapBlock(_dbNetHeap, block);
    sNet = UTBLOCKSIZE/sizeof(struct _dbNet);
    _dbFirstFreeNet = (dbNet)utgBlockMem(block);
    Net = _dbFirstFreeNet;
    for(x = 0; x < sNet; x++) {
        _dbNetNextFree(Net) = Net + 1;
        Net++;
    }
    _dbNetNextFree(--Net) = dbNetNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Net.
----------------------------------------------------------------------------*/
dbNet _dbNetAlloc(void)
{
    dbNet Net;

    if(_dbFirstFreeNet == dbNetNull) {
        buildNetBlock();
    }
    Net = _dbFirstFreeNet;
    _dbFirstFreeNet = _dbNetNextFree(Net);
    memset((void *)Net, 0, sizeof(struct _dbNet));
    return Net;
}

/*----------------------------------------------------------------------------
  Free a Net.
----------------------------------------------------------------------------*/
void _dbNetFree(
    dbNet Net)
{
    _dbNetNextFree(Net) = _dbFirstFreeNet;
    _dbFirstFreeNet = Net;
}

/*----------------------------------------------------------------------------
  Destroy a Net.
----------------------------------------------------------------------------*/
void dbNetDestroy(
    dbNet Net)
{
    if(dbNetDestructorHook != NULL) {
        dbNetDestructorHook(Net);
    }
    {
        dbPort _Port;
        dbSafeForeachNetPort(Net, _Port) {
            dbNetRemovePort(Net, _Port);
        } dbEndSafeForeachNetPort;
    }
    if(dbNetGetAttr(Net) != dbAttrNull) {
        dbAttrDestroy(dbNetGetAttr(Net));
    }
    {
        dbNetlist parentNetlist = dbNetGetNetlist(Net);
        if(parentNetlist != dbNetlistNull) {
            dbNetlistRemoveNet(parentNetlist, Net);
        }
    }
    {
        dbBus parentBus = dbNetGetBus(Net);
        if(parentBus != dbBusNull) {
            dbBusRemoveNet(parentBus, Net);
        }
    }
    {
        dbGlobal parentGlobal = dbNetGetGlobal(Net);
        if(parentGlobal != dbGlobalNull) {
            dbGlobalRemoveNet(parentGlobal, Net);
        }
    }
    dbNetFree(Net);
}

/*----------------------------------------------------------------------------
  Add the Port to the head of the list on the Inst.
----------------------------------------------------------------------------*/
void dbInstInsertPort(
    dbInst Inst,
    dbPort _Port)
{
    dbPortSetnextInstPort(_Port, dbInstGetfirstPort(Inst));
    dbInstSetfirstPort(Inst, _Port);
    if(dbInstGetlastPort(Inst) == dbPortNull) {
        dbInstSetlastPort(Inst, _Port);
    } else{
        dbPortSetprevInstPort(dbPortGetnextInstPort(_Port), _Port);
    }
    dbPortSetInst(_Port, Inst);
}

/*----------------------------------------------------------------------------
  Remove the Port from the list on the Inst.
----------------------------------------------------------------------------*/
void dbInstRemovePort(
    dbInst Inst,
    dbPort _Port)
{
    dbPort nextPort = dbPortGetnextInstPort(_Port);
    dbPort prevPort = dbPortGetprevInstPort(_Port);

    if(prevPort == dbPortNull && dbInstGetfirstPort(Inst) != _Port) {
        /* Not in list */
        return;
    }
    if(prevPort != dbPortNull) {
        dbPortSetnextInstPort(prevPort, nextPort);
    } else {
        dbInstSetfirstPort(Inst, nextPort);
    }
    if(nextPort != dbPortNull) {
        dbPortSetprevInstPort(nextPort, prevPort);
    } else {
        dbInstSetlastPort(Inst, prevPort);
    }
    dbPortSetprevInstPort(_Port, dbPortNull);
    dbPortSetnextInstPort(_Port, dbPortNull);
    dbPortSetInst(_Port, dbInstNull);
}

/*----------------------------------------------------------------------------
  Add the Port to the tail of the list on the Inst.
----------------------------------------------------------------------------*/
void dbInstAppendPort(
    dbInst Inst,
    dbPort _Port)
{
    if(dbInstGetlastPort(Inst) == dbPortNull) {
        dbInstSetfirstPort(Inst, _Port);
    } else {
        dbPortSetnextInstPort(dbInstGetlastPort(Inst), _Port);
        dbPortSetprevInstPort(_Port, dbInstGetlastPort(Inst));
    }
    dbInstSetlastPort(Inst, _Port);
    dbPortSetInst(_Port, Inst);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Inst.
------------------------------------------------------------------------*/
static utBlockRef buildInstBlock(void)
{
    utBlockRef block = utcBlock();
    dbInst Inst;
    U16 x, sInst;

    utaHeapBlock(_dbInstHeap, block);
    sInst = UTBLOCKSIZE/sizeof(struct _dbInst);
    _dbFirstFreeInst = (dbInst)utgBlockMem(block);
    Inst = _dbFirstFreeInst;
    for(x = 0; x < sInst; x++) {
        _dbInstNextFree(Inst) = Inst + 1;
        Inst++;
    }
    _dbInstNextFree(--Inst) = dbInstNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Inst.
----------------------------------------------------------------------------*/
dbInst _dbInstAlloc(void)
{
    dbInst Inst;

    if(_dbFirstFreeInst == dbInstNull) {
        buildInstBlock();
    }
    Inst = _dbFirstFreeInst;
    _dbFirstFreeInst = _dbInstNextFree(Inst);
    memset((void *)Inst, 0, sizeof(struct _dbInst));
    return Inst;
}

/*----------------------------------------------------------------------------
  Free a Inst.
----------------------------------------------------------------------------*/
void _dbInstFree(
    dbInst Inst)
{
    _dbInstNextFree(Inst) = _dbFirstFreeInst;
    _dbFirstFreeInst = Inst;
}

/*----------------------------------------------------------------------------
  Destroy a Inst.
----------------------------------------------------------------------------*/
void dbInstDestroy(
    dbInst Inst)
{
    if(dbInstDestructorHook != NULL) {
        dbInstDestructorHook(Inst);
    }
    {
        dbPort _Port;
        dbSafeForeachInstPort(Inst, _Port) {
            dbPortDestroy(_Port);
        } dbEndSafeForeachInstPort;
    }
    if(dbInstGetAttr(Inst) != dbAttrNull) {
        dbAttrDestroy(dbInstGetAttr(Inst));
    }
    {
        dbNetlist parentNetlist = dbInstGetNetlist(Inst);
        if(parentNetlist != dbNetlistNull) {
            dbNetlistRemoveInst(parentNetlist, Inst);
        }
    }
    {
        dbHier parentHier = dbInstGetHier(Inst);
        if(parentHier != dbHierNull) {
            dbHierRemoveInst(parentHier, Inst);
        }
    }
    {
        dbNetlist parentNetlist = dbInstGetInternalNetlist(Inst);
        if(parentNetlist != dbNetlistNull) {
            dbNetlistRemoveExternalInst(parentNetlist, Inst);
        }
    }
    {
        dbMbus parentMbus = dbInstGetExternalMbus(Inst);
        if(parentMbus != dbMbusNull) {
            dbMbusSetFlagInst(parentMbus, dbInstNull);
        }
    }
    dbInstFree(Inst);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Mport.
------------------------------------------------------------------------*/
static utBlockRef buildMportBlock(void)
{
    utBlockRef block = utcBlock();
    dbMport Mport;
    U16 x, sMport;

    utaHeapBlock(_dbMportHeap, block);
    sMport = UTBLOCKSIZE/sizeof(struct _dbMport);
    _dbFirstFreeMport = (dbMport)utgBlockMem(block);
    Mport = _dbFirstFreeMport;
    for(x = 0; x < sMport; x++) {
        _dbMportNextFree(Mport) = Mport + 1;
        Mport++;
    }
    _dbMportNextFree(--Mport) = dbMportNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Mport.
----------------------------------------------------------------------------*/
dbMport _dbMportAlloc(void)
{
    dbMport Mport;

    if(_dbFirstFreeMport == dbMportNull) {
        buildMportBlock();
    }
    Mport = _dbFirstFreeMport;
    _dbFirstFreeMport = _dbMportNextFree(Mport);
    memset((void *)Mport, 0, sizeof(struct _dbMport));
    return Mport;
}

/*----------------------------------------------------------------------------
  Free a Mport.
----------------------------------------------------------------------------*/
void _dbMportFree(
    dbMport Mport)
{
    _dbMportNextFree(Mport) = _dbFirstFreeMport;
    _dbFirstFreeMport = Mport;
}

/*----------------------------------------------------------------------------
  Destroy a Mport.
----------------------------------------------------------------------------*/
void dbMportDestroy(
    dbMport Mport)
{
    if(dbMportDestructorHook != NULL) {
        dbMportDestructorHook(Mport);
    }
    if(dbMportGetAttr(Mport) != dbAttrNull) {
        dbAttrDestroy(dbMportGetAttr(Mport));
    }
    {
        dbNetlist parentNetlist = dbMportGetNetlist(Mport);
        if(parentNetlist != dbNetlistNull) {
            dbNetlistRemoveMport(parentNetlist, Mport);
        }
    }
    {
        dbMbus parentMbus = dbMportGetMbus(Mport);
        if(parentMbus != dbMbusNull) {
            dbMbusRemoveMport(parentMbus, Mport);
        }
    }
    dbMportFree(Mport);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Port.
------------------------------------------------------------------------*/
static utBlockRef buildPortBlock(void)
{
    utBlockRef block = utcBlock();
    dbPort Port;
    U16 x, sPort;

    utaHeapBlock(_dbPortHeap, block);
    sPort = UTBLOCKSIZE/sizeof(struct _dbPort);
    _dbFirstFreePort = (dbPort)utgBlockMem(block);
    Port = _dbFirstFreePort;
    for(x = 0; x < sPort; x++) {
        _dbPortNextFree(Port) = Port + 1;
        Port++;
    }
    _dbPortNextFree(--Port) = dbPortNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Port.
----------------------------------------------------------------------------*/
dbPort _dbPortAlloc(void)
{
    dbPort Port;

    if(_dbFirstFreePort == dbPortNull) {
        buildPortBlock();
    }
    Port = _dbFirstFreePort;
    _dbFirstFreePort = _dbPortNextFree(Port);
    memset((void *)Port, 0, sizeof(struct _dbPort));
    return Port;
}

/*----------------------------------------------------------------------------
  Free a Port.
----------------------------------------------------------------------------*/
void _dbPortFree(
    dbPort Port)
{
    _dbPortNextFree(Port) = _dbFirstFreePort;
    _dbFirstFreePort = Port;
}

/*----------------------------------------------------------------------------
  Destroy a Port.
----------------------------------------------------------------------------*/
void dbPortDestroy(
    dbPort Port)
{
    if(dbPortDestructorHook != NULL) {
        dbPortDestructorHook(Port);
    }
    {
        dbNet parentNet = dbPortGetNet(Port);
        if(parentNet != dbNetNull) {
            dbNetRemovePort(parentNet, Port);
        }
    }
    {
        dbInst parentInst = dbPortGetInst(Port);
        if(parentInst != dbInstNull) {
            dbInstRemovePort(parentInst, Port);
        }
    }
    dbPortFree(Port);
}

/*----------------------------------------------------------------------------
  Add the Mport to the end of the array on the Mbus.
----------------------------------------------------------------------------*/
void dbMbusInsertMport(
    dbMbus Mbus,
    dbMport _Mport)
{
    if(dbMbusGetusedMports(Mbus) == dbMbusGetnumMports(Mbus)) {
        dbMbusReallocMports(Mbus, dbMbusGetnumMports(Mbus) +
            (dbMbusGetnumMports(Mbus) >> 1) + 1);
    }
    dbMportSetMbusIndex(_Mport, dbMbusGetusedMports(Mbus));
    dbMbusSetiMport(Mbus, dbMbusGetusedMports(Mbus)++, _Mport);
    dbMportSetMbus(_Mport, Mbus);
}

/*----------------------------------------------------------------------------
  Remove the Mport from the array on the Mbus.
----------------------------------------------------------------------------*/
void dbMbusRemoveMport(
    dbMbus Mbus,
    dbMport _Mport)
{
    U32 xMport = dbMportGetMbusIndex(_Mport);

    if(xMport != U32_MAX) {
        dbMbusSetiMport(Mbus, xMport, dbMbusGetiMport(Mbus,
            --dbMbusGetusedMports(Mbus)));
        dbMportSetMbusIndex(_Mport, U32_MAX);
    }
    dbMportSetMbus(_Mport, dbMbusNull);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Mbus.
------------------------------------------------------------------------*/
static utBlockRef buildMbusBlock(void)
{
    utBlockRef block = utcBlock();
    dbMbus Mbus;
    U16 x, sMbus;

    utaHeapBlock(_dbMbusHeap, block);
    sMbus = UTBLOCKSIZE/sizeof(struct _dbMbus);
    _dbFirstFreeMbus = (dbMbus)utgBlockMem(block);
    Mbus = _dbFirstFreeMbus;
    for(x = 0; x < sMbus; x++) {
        _dbMbusNextFree(Mbus) = Mbus + 1;
        Mbus++;
    }
    _dbMbusNextFree(--Mbus) = dbMbusNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Mbus.
----------------------------------------------------------------------------*/
dbMbus dbMbusAlloc(void)
{
    dbMbus Mbus;

    if(_dbFirstFreeMbus == dbMbusNull) {
        buildMbusBlock();
    }
    Mbus = _dbFirstFreeMbus;
    _dbFirstFreeMbus = _dbMbusNextFree(Mbus);
    memset((void *)Mbus, 0, sizeof(struct _dbMbus));
    return Mbus;
}

/*----------------------------------------------------------------------------
  Free a Mbus.
----------------------------------------------------------------------------*/
void dbMbusFree(
    dbMbus Mbus)
{
    if(dbMbusGetnumMports(Mbus) != 0) {
        dbMbusFreeMports(Mbus);
    }
    _dbMbusNextFree(Mbus) = _dbFirstFreeMbus;
    _dbFirstFreeMbus = Mbus;
}

/*----------------------------------------------------------------------------
  Destroy a Mbus.
----------------------------------------------------------------------------*/
void dbMbusDestroy(
    dbMbus Mbus)
{
    if(dbMbusDestructorHook != NULL) {
        dbMbusDestructorHook(Mbus);
    }
    {
        dbMport _Mport;
        dbSafeForeachMbusMport(Mbus, _Mport) {
            dbMbusRemoveMport(Mbus, _Mport);
        } dbEndSafeForeachMbusMport;
    }
    if(dbMbusGetFlagInst(Mbus) != dbInstNull) {
        dbInstSetExternalMbus(dbMbusGetFlagInst(Mbus), dbMbusNull);
    }
    {
        dbNetlist parentNetlist = dbMbusGetNetlist(Mbus);
        if(parentNetlist != dbNetlistNull) {
            dbNetlistRemoveMbus(parentNetlist, Mbus);
        }
    }
    dbMbusFree(Mbus);
}

/*----------------------------------------------------------------------------
  Add the Net to the end of the array on the Bus.
----------------------------------------------------------------------------*/
void dbBusInsertNet(
    dbBus Bus,
    dbNet _Net)
{
    if(dbBusGetusedNets(Bus) == dbBusGetnumNets(Bus)) {
        dbBusReallocNets(Bus, dbBusGetnumNets(Bus) +
            (dbBusGetnumNets(Bus) >> 1) + 1);
    }
    dbNetSetBusIndex(_Net, dbBusGetusedNets(Bus));
    dbBusSetiNet(Bus, dbBusGetusedNets(Bus)++, _Net);
    dbNetSetBus(_Net, Bus);
}

/*----------------------------------------------------------------------------
  Remove the Net from the array on the Bus.
----------------------------------------------------------------------------*/
void dbBusRemoveNet(
    dbBus Bus,
    dbNet _Net)
{
    U32 xNet = dbNetGetBusIndex(_Net);

    if(xNet != U32_MAX) {
        dbBusSetiNet(Bus, xNet, dbBusGetiNet(Bus,
            --dbBusGetusedNets(Bus)));
        dbNetSetBusIndex(_Net, U32_MAX);
    }
    dbNetSetBus(_Net, dbBusNull);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Bus.
------------------------------------------------------------------------*/
static utBlockRef buildBusBlock(void)
{
    utBlockRef block = utcBlock();
    dbBus Bus;
    U16 x, sBus;

    utaHeapBlock(_dbBusHeap, block);
    sBus = UTBLOCKSIZE/sizeof(struct _dbBus);
    _dbFirstFreeBus = (dbBus)utgBlockMem(block);
    Bus = _dbFirstFreeBus;
    for(x = 0; x < sBus; x++) {
        _dbBusNextFree(Bus) = Bus + 1;
        Bus++;
    }
    _dbBusNextFree(--Bus) = dbBusNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Bus.
----------------------------------------------------------------------------*/
dbBus dbBusAlloc(void)
{
    dbBus Bus;

    if(_dbFirstFreeBus == dbBusNull) {
        buildBusBlock();
    }
    Bus = _dbFirstFreeBus;
    _dbFirstFreeBus = _dbBusNextFree(Bus);
    memset((void *)Bus, 0, sizeof(struct _dbBus));
    return Bus;
}

/*----------------------------------------------------------------------------
  Free a Bus.
----------------------------------------------------------------------------*/
void dbBusFree(
    dbBus Bus)
{
    if(dbBusGetnumNets(Bus) != 0) {
        dbBusFreeNets(Bus);
    }
    _dbBusNextFree(Bus) = _dbFirstFreeBus;
    _dbFirstFreeBus = Bus;
}

/*----------------------------------------------------------------------------
  Destroy a Bus.
----------------------------------------------------------------------------*/
void dbBusDestroy(
    dbBus Bus)
{
    if(dbBusDestructorHook != NULL) {
        dbBusDestructorHook(Bus);
    }
    {
        dbNet _Net;
        dbSafeForeachBusNet(Bus, _Net) {
            dbBusRemoveNet(Bus, _Net);
        } dbEndSafeForeachBusNet;
    }
    {
        dbNetlist parentNetlist = dbBusGetNetlist(Bus);
        if(parentNetlist != dbNetlistNull) {
            dbNetlistRemoveBus(parentNetlist, Bus);
        }
    }
    dbBusFree(Bus);
}

/*----------------------------------------------------------------------------
  Add the Hier to the tail of the list on the Hier.
----------------------------------------------------------------------------*/
void dbHierInsertHier(
    dbHier Hier,
    dbHier _Hier)
{
    utInsertHtblEntry(dbHierGetHiers(Hier), &(_Hier)->h);
    dbHierSetHier(_Hier, Hier);
}

/*----------------------------------------------------------------------------
  Remove the Hier from the list on the Hier.
----------------------------------------------------------------------------*/
void dbHierRemoveHier(
    dbHier Hier,
    dbHier _Hier)
{
    utDeleteHtblEntry(&(Hier)->Hiers, &(_Hier)->h);
    dbHierSetHier(_Hier, dbHierNull);
}

/*----------------------------------------------------------------------------
  Append the Hier to the tail of the list on the Hier.
----------------------------------------------------------------------------*/
void dbHierAppendHier(
    dbHier Hier,
    dbHier _Hier)
{
    utAppendHtblEntry(dbHierGetHiers(Hier), &(_Hier)->h);
    dbHierSetHier(_Hier, Hier);
}

/*----------------------------------------------------------------------------
  Add the Inst to the head of the list on the Hier.
----------------------------------------------------------------------------*/
void dbHierInsertInst(
    dbHier Hier,
    dbInst _Inst)
{
    dbInstSetnextHierInst(_Inst, dbHierGetfirstInst(Hier));
    dbHierSetfirstInst(Hier, _Inst);
    if(dbHierGetlastInst(Hier) == dbInstNull) {
        dbHierSetlastInst(Hier, _Inst);
    } else{
        dbInstSetprevHierInst(dbInstGetnextHierInst(_Inst), _Inst);
    }
    dbInstSetHier(_Inst, Hier);
}

/*----------------------------------------------------------------------------
  Remove the Inst from the list on the Hier.
----------------------------------------------------------------------------*/
void dbHierRemoveInst(
    dbHier Hier,
    dbInst _Inst)
{
    dbInst nextInst = dbInstGetnextHierInst(_Inst);
    dbInst prevInst = dbInstGetprevHierInst(_Inst);

    if(prevInst == dbInstNull && dbHierGetfirstInst(Hier) != _Inst) {
        /* Not in list */
        return;
    }
    if(prevInst != dbInstNull) {
        dbInstSetnextHierInst(prevInst, nextInst);
    } else {
        dbHierSetfirstInst(Hier, nextInst);
    }
    if(nextInst != dbInstNull) {
        dbInstSetprevHierInst(nextInst, prevInst);
    } else {
        dbHierSetlastInst(Hier, prevInst);
    }
    dbInstSetprevHierInst(_Inst, dbInstNull);
    dbInstSetnextHierInst(_Inst, dbInstNull);
    dbInstSetHier(_Inst, dbHierNull);
}

/*----------------------------------------------------------------------------
  Add the Inst to the tail of the list on the Hier.
----------------------------------------------------------------------------*/
void dbHierAppendInst(
    dbHier Hier,
    dbInst _Inst)
{
    if(dbHierGetlastInst(Hier) == dbInstNull) {
        dbHierSetfirstInst(Hier, _Inst);
    } else {
        dbInstSetnextHierInst(dbHierGetlastInst(Hier), _Inst);
        dbInstSetprevHierInst(_Inst, dbHierGetlastInst(Hier));
    }
    dbHierSetlastInst(Hier, _Inst);
    dbInstSetHier(_Inst, Hier);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Hier.
------------------------------------------------------------------------*/
static utBlockRef buildHierBlock(void)
{
    utBlockRef block = utcBlock();
    dbHier Hier;
    U16 x, sHier;

    utaHeapBlock(_dbHierHeap, block);
    sHier = UTBLOCKSIZE/sizeof(struct _dbHier);
    _dbFirstFreeHier = (dbHier)utgBlockMem(block);
    Hier = _dbFirstFreeHier;
    for(x = 0; x < sHier; x++) {
        _dbHierNextFree(Hier) = Hier + 1;
        Hier++;
    }
    _dbHierNextFree(--Hier) = dbHierNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Hier.
----------------------------------------------------------------------------*/
dbHier dbHierAlloc(void)
{
    dbHier Hier;

    if(_dbFirstFreeHier == dbHierNull) {
        buildHierBlock();
    }
    Hier = _dbFirstFreeHier;
    _dbFirstFreeHier = _dbHierNextFree(Hier);
    memset((void *)Hier, 0, sizeof(struct _dbHier));
    utInitHtbl(&Hier->Hiers);
    return Hier;
}

/*----------------------------------------------------------------------------
  Free a Hier.
----------------------------------------------------------------------------*/
void dbHierFree(
    dbHier Hier)
{
    utdArray(utfHtblArray(dbHierGetHiers(Hier)));
    _dbHierNextFree(Hier) = _dbFirstFreeHier;
    _dbFirstFreeHier = Hier;
}

/*----------------------------------------------------------------------------
  Destroy a Hier.
----------------------------------------------------------------------------*/
void dbHierDestroy(
    dbHier Hier)
{
    if(dbHierDestructorHook != NULL) {
        dbHierDestructorHook(Hier);
    }
    {
        dbHier _Hier;
        dbSafeForeachHierHier(Hier, _Hier) {
            dbHierDestroy(_Hier);
        } dbEndSafeForeachHierHier;
    }
    {
        dbInst _Inst;
        dbSafeForeachHierInst(Hier, _Inst) {
            dbInstDestroy(_Inst);
        } dbEndSafeForeachHierInst;
    }
    {
        dbHier parentHier = dbHierGetHier(Hier);
        if(parentHier != dbHierNull) {
            dbHierRemoveHier(parentHier, Hier);
        }
    }
    dbHierFree(Hier);
}

/*----------------------------------------------------------------------------
  Add the Net to the head of the list on the Global.
----------------------------------------------------------------------------*/
void dbGlobalInsertNet(
    dbGlobal Global,
    dbNet _Net)
{
    dbNetSetnextGlobalNet(_Net, dbGlobalGetfirstNet(Global));
    dbGlobalSetfirstNet(Global, _Net);
    if(dbGlobalGetlastNet(Global) == dbNetNull) {
        dbGlobalSetlastNet(Global, _Net);
    } else{
        dbNetSetprevGlobalNet(dbNetGetnextGlobalNet(_Net), _Net);
    }
    dbNetSetGlobal(_Net, Global);
}

/*----------------------------------------------------------------------------
  Remove the Net from the list on the Global.
----------------------------------------------------------------------------*/
void dbGlobalRemoveNet(
    dbGlobal Global,
    dbNet _Net)
{
    dbNet nextNet = dbNetGetnextGlobalNet(_Net);
    dbNet prevNet = dbNetGetprevGlobalNet(_Net);

    if(prevNet == dbNetNull && dbGlobalGetfirstNet(Global) != _Net) {
        /* Not in list */
        return;
    }
    if(prevNet != dbNetNull) {
        dbNetSetnextGlobalNet(prevNet, nextNet);
    } else {
        dbGlobalSetfirstNet(Global, nextNet);
    }
    if(nextNet != dbNetNull) {
        dbNetSetprevGlobalNet(nextNet, prevNet);
    } else {
        dbGlobalSetlastNet(Global, prevNet);
    }
    dbNetSetprevGlobalNet(_Net, dbNetNull);
    dbNetSetnextGlobalNet(_Net, dbNetNull);
    dbNetSetGlobal(_Net, dbGlobalNull);
}

/*----------------------------------------------------------------------------
  Add the Net to the tail of the list on the Global.
----------------------------------------------------------------------------*/
void dbGlobalAppendNet(
    dbGlobal Global,
    dbNet _Net)
{
    if(dbGlobalGetlastNet(Global) == dbNetNull) {
        dbGlobalSetfirstNet(Global, _Net);
    } else {
        dbNetSetnextGlobalNet(dbGlobalGetlastNet(Global), _Net);
        dbNetSetprevGlobalNet(_Net, dbGlobalGetlastNet(Global));
    }
    dbGlobalSetlastNet(Global, _Net);
    dbNetSetGlobal(_Net, Global);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Global.
------------------------------------------------------------------------*/
static utBlockRef buildGlobalBlock(void)
{
    utBlockRef block = utcBlock();
    dbGlobal Global;
    U16 x, sGlobal;

    utaHeapBlock(_dbGlobalHeap, block);
    sGlobal = UTBLOCKSIZE/sizeof(struct _dbGlobal);
    _dbFirstFreeGlobal = (dbGlobal)utgBlockMem(block);
    Global = _dbFirstFreeGlobal;
    for(x = 0; x < sGlobal; x++) {
        _dbGlobalNextFree(Global) = Global + 1;
        Global++;
    }
    _dbGlobalNextFree(--Global) = dbGlobalNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Global.
----------------------------------------------------------------------------*/
dbGlobal _dbGlobalAlloc(void)
{
    dbGlobal Global;

    if(_dbFirstFreeGlobal == dbGlobalNull) {
        buildGlobalBlock();
    }
    Global = _dbFirstFreeGlobal;
    _dbFirstFreeGlobal = _dbGlobalNextFree(Global);
    memset((void *)Global, 0, sizeof(struct _dbGlobal));
    return Global;
}

/*----------------------------------------------------------------------------
  Free a Global.
----------------------------------------------------------------------------*/
void _dbGlobalFree(
    dbGlobal Global)
{
    _dbGlobalNextFree(Global) = _dbFirstFreeGlobal;
    _dbFirstFreeGlobal = Global;
}

/*----------------------------------------------------------------------------
  Destroy a Global.
----------------------------------------------------------------------------*/
void dbGlobalDestroy(
    dbGlobal Global)
{
    if(dbGlobalDestructorHook != NULL) {
        dbGlobalDestructorHook(Global);
    }
    {
        dbNet _Net;
        dbSafeForeachGlobalNet(Global, _Net) {
            dbGlobalRemoveNet(Global, _Net);
        } dbEndSafeForeachGlobalNet;
    }
    {
        dbDesign parentDesign = dbGlobalGetDesign(Global);
        if(parentDesign != dbDesignNull) {
            dbDesignRemoveGlobal(parentDesign, Global);
        }
    }
    dbGlobalFree(Global);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Attr.
------------------------------------------------------------------------*/
static utBlockRef buildAttrBlock(void)
{
    utBlockRef block = utcBlock();
    dbAttr Attr;
    U16 x, sAttr;

    utaHeapBlock(_dbAttrHeap, block);
    sAttr = UTBLOCKSIZE/sizeof(struct _dbAttr);
    _dbFirstFreeAttr = (dbAttr)utgBlockMem(block);
    Attr = _dbFirstFreeAttr;
    for(x = 0; x < sAttr; x++) {
        _dbAttrNextFree(Attr) = Attr + 1;
        Attr++;
    }
    _dbAttrNextFree(--Attr) = dbAttrNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Attr.
----------------------------------------------------------------------------*/
dbAttr _dbAttrAlloc(void)
{
    dbAttr Attr;

    if(_dbFirstFreeAttr == dbAttrNull) {
        buildAttrBlock();
    }
    Attr = _dbFirstFreeAttr;
    _dbFirstFreeAttr = _dbAttrNextFree(Attr);
    memset((void *)Attr, 0, sizeof(struct _dbAttr));
    return Attr;
}

/*----------------------------------------------------------------------------
  Free a Attr.
----------------------------------------------------------------------------*/
void _dbAttrFree(
    dbAttr Attr)
{
    _dbAttrNextFree(Attr) = _dbFirstFreeAttr;
    _dbFirstFreeAttr = Attr;
}

/*----------------------------------------------------------------------------
  Destroy a Attr.
----------------------------------------------------------------------------*/
void dbAttrDestroy(
    dbAttr Attr)
{
    if(dbAttrDestructorHook != NULL) {
        dbAttrDestructorHook(Attr);
    }
    if(dbAttrGetNextAttr(Attr) != dbAttrNull) {
        dbAttrDestroy(dbAttrGetNextAttr(Attr));
    }
    dbAttrFree(Attr);
}

/*----------------------------------------------------------------------------
  Add the Design to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void dbRootInsertDesign(
    dbRoot Root,
    dbDesign _Design)
{
    utInsertHtblEntry(dbRootGetDesigns(Root), &(_Design)->h);
    dbDesignSetRoot(_Design, Root);
}

/*----------------------------------------------------------------------------
  Remove the Design from the list on the Root.
----------------------------------------------------------------------------*/
void dbRootRemoveDesign(
    dbRoot Root,
    dbDesign _Design)
{
    utDeleteHtblEntry(&(Root)->Designs, &(_Design)->h);
    dbDesignSetRoot(_Design, dbRootNull);
}

/*----------------------------------------------------------------------------
  Append the Design to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void dbRootAppendDesign(
    dbRoot Root,
    dbDesign _Design)
{
    utAppendHtblEntry(dbRootGetDesigns(Root), &(_Design)->h);
    dbDesignSetRoot(_Design, Root);
}

/*----------------------------------------------------------------------------
  Add the Devspec to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void dbRootInsertDevspec(
    dbRoot Root,
    dbDevspec _Devspec)
{
    utInsertHtblEntry(dbRootGetDevspecs(Root), &(_Devspec)->h);
    dbDevspecSetRoot(_Devspec, Root);
}

/*----------------------------------------------------------------------------
  Remove the Devspec from the list on the Root.
----------------------------------------------------------------------------*/
void dbRootRemoveDevspec(
    dbRoot Root,
    dbDevspec _Devspec)
{
    utDeleteHtblEntry(&(Root)->Devspecs, &(_Devspec)->h);
    dbDevspecSetRoot(_Devspec, dbRootNull);
}

/*----------------------------------------------------------------------------
  Append the Devspec to the tail of the list on the Root.
----------------------------------------------------------------------------*/
void dbRootAppendDevspec(
    dbRoot Root,
    dbDevspec _Devspec)
{
    utAppendHtblEntry(dbRootGetDevspecs(Root), &(_Devspec)->h);
    dbDevspecSetRoot(_Devspec, Root);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Root.
------------------------------------------------------------------------*/
static utBlockRef buildRootBlock(void)
{
    utBlockRef block = utcBlock();
    dbRoot Root;
    U16 x, sRoot;

    utaHeapBlock(_dbRootHeap, block);
    sRoot = UTBLOCKSIZE/sizeof(struct _dbRoot);
    _dbFirstFreeRoot = (dbRoot)utgBlockMem(block);
    Root = _dbFirstFreeRoot;
    for(x = 0; x < sRoot; x++) {
        _dbRootNextFree(Root) = Root + 1;
        Root++;
    }
    _dbRootNextFree(--Root) = dbRootNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Root.
----------------------------------------------------------------------------*/
dbRoot dbRootAlloc(void)
{
    dbRoot Root;

    if(_dbFirstFreeRoot == dbRootNull) {
        buildRootBlock();
    }
    Root = _dbFirstFreeRoot;
    _dbFirstFreeRoot = _dbRootNextFree(Root);
    memset((void *)Root, 0, sizeof(struct _dbRoot));
    utInitHtbl(&Root->Designs);
    utInitHtbl(&Root->Devspecs);
    return Root;
}

/*----------------------------------------------------------------------------
  Free a Root.
----------------------------------------------------------------------------*/
void dbRootFree(
    dbRoot Root)
{
    utdArray(utfHtblArray(dbRootGetDesigns(Root)));
    utdArray(utfHtblArray(dbRootGetDevspecs(Root)));
    _dbRootNextFree(Root) = _dbFirstFreeRoot;
    _dbFirstFreeRoot = Root;
}

/*----------------------------------------------------------------------------
  Destroy a Root.
----------------------------------------------------------------------------*/
void dbRootDestroy(
    dbRoot Root)
{
    if(dbRootDestructorHook != NULL) {
        dbRootDestructorHook(Root);
    }
    {
        dbDesign _Design;
        dbSafeForeachRootDesign(Root, _Design) {
            dbDesignDestroy(_Design);
        } dbEndSafeForeachRootDesign;
    }
    {
        dbDevspec _Devspec;
        dbSafeForeachRootDevspec(Root, _Devspec) {
            dbDevspecDestroy(_Devspec);
        } dbEndSafeForeachRootDevspec;
    }
    dbRootFree(Root);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Devspec.
------------------------------------------------------------------------*/
static utBlockRef buildDevspecBlock(void)
{
    utBlockRef block = utcBlock();
    dbDevspec Devspec;
    U16 x, sDevspec;

    utaHeapBlock(_dbDevspecHeap, block);
    sDevspec = UTBLOCKSIZE/sizeof(struct _dbDevspec);
    _dbFirstFreeDevspec = (dbDevspec)utgBlockMem(block);
    Devspec = _dbFirstFreeDevspec;
    for(x = 0; x < sDevspec; x++) {
        _dbDevspecNextFree(Devspec) = Devspec + 1;
        Devspec++;
    }
    _dbDevspecNextFree(--Devspec) = dbDevspecNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Devspec.
----------------------------------------------------------------------------*/
dbDevspec dbDevspecAlloc(void)
{
    dbDevspec Devspec;

    if(_dbFirstFreeDevspec == dbDevspecNull) {
        buildDevspecBlock();
    }
    Devspec = _dbFirstFreeDevspec;
    _dbFirstFreeDevspec = _dbDevspecNextFree(Devspec);
    memset((void *)Devspec, 0, sizeof(struct _dbDevspec));
    return Devspec;
}

/*----------------------------------------------------------------------------
  Free a Devspec.
----------------------------------------------------------------------------*/
void dbDevspecFree(
    dbDevspec Devspec)
{
    if(dbDevspecGetnumStrings(Devspec) != 0) {
        dbDevspecFreeStrings(Devspec);
    }
    _dbDevspecNextFree(Devspec) = _dbFirstFreeDevspec;
    _dbFirstFreeDevspec = Devspec;
}

/*----------------------------------------------------------------------------
  Destroy a Devspec.
----------------------------------------------------------------------------*/
void dbDevspecDestroy(
    dbDevspec Devspec)
{
    if(dbDevspecDestructorHook != NULL) {
        dbDevspecDestructorHook(Devspec);
    }
    {
        dbRoot parentRoot = dbDevspecGetRoot(Devspec);
        if(parentRoot != dbRootNull) {
            dbRootRemoveDevspec(parentRoot, Devspec);
        }
    }
    dbDevspecFree(Devspec);
}

/*----------------------------------------------------------------------------
  Initialize memory for the db&&.
----------------------------------------------------------------------------*/
void dbDDRStart(void)
{
    _dbFirstFreeDesign = dbDesignNull;
    _dbDesignHeap = utcHeap();
    buildDesignBlock();
    _dbFirstFreeNetlist = dbNetlistNull;
    _dbNetlistHeap = utcHeap();
    buildNetlistBlock();
    _dbFirstFreeNet = dbNetNull;
    _dbNetHeap = utcHeap();
    buildNetBlock();
    _dbFirstFreeInst = dbInstNull;
    _dbInstHeap = utcHeap();
    buildInstBlock();
    _dbFirstFreeMport = dbMportNull;
    _dbMportHeap = utcHeap();
    buildMportBlock();
    _dbFirstFreePort = dbPortNull;
    _dbPortHeap = utcHeap();
    buildPortBlock();
    _dbFirstFreeMbus = dbMbusNull;
    _dbMbusHeap = utcHeap();
    buildMbusBlock();
    _dbFirstFreeBus = dbBusNull;
    _dbBusHeap = utcHeap();
    buildBusBlock();
    _dbFirstFreeHier = dbHierNull;
    _dbHierHeap = utcHeap();
    buildHierBlock();
    _dbFirstFreeGlobal = dbGlobalNull;
    _dbGlobalHeap = utcHeap();
    buildGlobalBlock();
    _dbFirstFreeAttr = dbAttrNull;
    _dbAttrHeap = utcHeap();
    buildAttrBlock();
    _dbFirstFreeRoot = dbRootNull;
    _dbRootHeap = utcHeap();
    buildRootBlock();
    _dbFirstFreeDevspec = dbDevspecNull;
    _dbDevspecHeap = utcHeap();
    buildDevspecBlock();
}

/*----------------------------------------------------------------------------
  Free memory for the db&&.
----------------------------------------------------------------------------*/
void dbDDRStop(void)
{
    utFreeHeap(_dbDesignHeap);
    utFreeHeap(_dbNetlistHeap);
    utFreeHeap(_dbNetHeap);
    utFreeHeap(_dbInstHeap);
    utFreeHeap(_dbMportHeap);
    utFreeHeap(_dbPortHeap);
    utFreeHeap(_dbMbusHeap);
    utFreeHeap(_dbBusHeap);
    utFreeHeap(_dbHierHeap);
    utFreeHeap(_dbGlobalHeap);
    utFreeHeap(_dbAttrHeap);
    utFreeHeap(_dbRootHeap);
    utFreeHeap(_dbDevspecHeap);
}

