#include <string.h>
#include "db.h"
#include "schext.h"
#include "cirext.h"
#include "vrext.h"
#include "geext.h"

extern char *config_dir;

/*--------------------------------------------------------------------------------------------------
  Set the current design.
--------------------------------------------------------------------------------------------------*/
int set_current_design(
    char *designName)
{
    dbDesign design = dbRootFindDesign(dbTheRoot, utSymCreate(designName));

    if(design == dbDesignNull) {
        utWarning("set_current_design: Unable to locate design %s in the database", designName);
        return 0;
    }
    dbCurrentDesign = design;
    dbCurrentNetlist = dbDesignGetRootNetlist(design);
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Set the current netlist.
--------------------------------------------------------------------------------------------------*/
int set_current_netlist(
    char *netlistName)
{
    dbNetlist netlist;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("set_current_netlist: no current design in the database");
        return 0;
    }
    netlist = dbDesignFindNetlist(dbCurrentDesign, utSymCreate(netlistName));
    if(netlist == dbNetlistNull) {
        utWarning("set_current_netlist: Unable to locate netlist %s in the database", netlistName);
        return 0;
    }
    dbCurrentNetlist = netlist;
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Set the root netlist.
--------------------------------------------------------------------------------------------------*/
int set_root_netlist(
    char *netlistName)
{
    dbNetlist netlist;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("set_root_netlist: no current design in the database");
        return 0;
    }
    netlist = dbDesignFindNetlist(dbCurrentDesign, utSymCreate(netlistName));
    if(netlist == dbNetlistNull) {
        utWarning("set_root_netlist: Unable to locate netlist %s in the database", netlistName);
        return 0;
    }
    dbDesignSetRootNetlist(dbCurrentDesign, netlist);
    dbCurrentNetlist = netlist;
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Create a net in the current netlist, if it does not already exist.
--------------------------------------------------------------------------------------------------*/
void create_net(
    char *netName)
{
    utSym netSym = utSymCreate(netName);

    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("create_net: no current netlist in the database");
        return;
    }
    if(dbNetlistFindNet(dbCurrentNetlist, netSym) == dbNetNull) {
        dbNetCreate(dbCurrentNetlist, netSym);
    }
}

/*--------------------------------------------------------------------------------------------------
  Merge two nets in the current netlist.
--------------------------------------------------------------------------------------------------*/
void merge_net_into_net(
    char *sourceNetName,
    char *destNetName)
{
    dbNet sourceNet, destNet;
    utSym sourceNetSym = utSymCreate(sourceNetName);
    utSym destNetSym = utSymCreate(destNetName);

    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("merge_net_into_net: no current netlist in the database");
        return;
    }
    sourceNet = dbNetlistFindNet(dbCurrentNetlist, sourceNetSym);
    if(sourceNet == dbNetNull) {
        utWarning("merge_nets: no net named %s in the current netlist", sourceNetName);
        return;
    }
    destNet = dbNetlistFindNet(dbCurrentNetlist, destNetSym);
    if(destNet == dbNetNull) {
        utWarning("merge_net_into_net: no net named %s in the current netlist", destNetName);
        return;
    }
    dbMergeNetIntoNet(sourceNet, destNet);
}

/*--------------------------------------------------------------------------------------------------
  Get the current design.
--------------------------------------------------------------------------------------------------*/
char *get_current_design(void)
{
    if(dbCurrentDesign == dbDesignNull) {
        utWarning("get_current_design: no current design in the database");
        return "";
    }
    return dbDesignGetName(dbCurrentDesign);
}

/*--------------------------------------------------------------------------------------------------
  Read a spice netlist.
--------------------------------------------------------------------------------------------------*/
int read_spice_netlist(
    char *designName,
    char *spiceFileName)
{
    dbCurrentDesign = cirReadDesign(designName, spiceFileName);
    if(dbCurrentDesign == dbDesignNull) {
        utWarning("Unable to read SPICE file %s", spiceFileName);
        return 0;
    }
    dbCurrentNetlist = dbDesignGetRootNetlist(dbCurrentDesign);
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Add a component directory to the component path.
--------------------------------------------------------------------------------------------------*/
void component_library(
    char *dirName)
{
    char *newPath;
  
    if(strcmp(dbGschemComponentPath, ".")) {
        newPath = utSprintf(".:%s:%s", dirName, dbGschemComponentPath + 2);
    } else {
        newPath = utSprintf(".:%s", dirName);
    }
    utResizeArray(dbGschemComponentPath, strlen(newPath) + 1);
    strcpy(dbGschemComponentPath, newPath);
}

/*--------------------------------------------------------------------------------------------------
  Add a schematic source directory to the source path.
--------------------------------------------------------------------------------------------------*/
void source_library(
    char *dirName)
{
    char *newPath;
  
    if(strcmp(dbGschemSourcePath, ".")) {
        newPath = utSprintf(".:%s:%s", dirName, dbGschemSourcePath + 2);
    } else {
        newPath = utSprintf(".:%s", dirName);
    }
    utResizeArray(dbGschemSourcePath, strlen(newPath) + 1);
    strcpy(dbGschemSourcePath, newPath);
}

/*--------------------------------------------------------------------------------------------------
  reset the source library to ".".
--------------------------------------------------------------------------------------------------*/
void reset_source_library(void)
{
    strcpy(dbGschemSourcePath, ".");
}

/*--------------------------------------------------------------------------------------------------
  reset the compoent library to ".".
--------------------------------------------------------------------------------------------------*/
void reset_component_library(void)
{
    strcpy(dbGschemComponentPath, ".");
}

/*--------------------------------------------------------------------------------------------------
  Set the current SPICE target.
--------------------------------------------------------------------------------------------------*/
void set_spice_target(
    char *spiceType)
{
    dbSpiceTargetType target = dbFindSpiceTargetFromName(spiceType);

    if(target == DB_NULL_SPICE_TYPE) {
        utWarning("Unknown spice type '%s': try one of pspice, hspice, tclspice, or ltspice",
            spiceType);
        return;
    }
    dbSpiceTarget = target;
}

/*--------------------------------------------------------------------------------------------------
  Add a new device type for spice netlisting.
--------------------------------------------------------------------------------------------------*/
void add_spice_device(
    char *device)
{
    dbDevspec devspec = dbFindCurrentDevspec();
    char *newDeviceString;
  
    newDeviceString = utSprintf("%s%s\n", dbDevspecGetString(devspec), device);
    dbDevspecReallocStrings(devspec, strlen(newDeviceString) + 1);
    strcpy(dbDevspecGetString(devspec), newDeviceString);
}

/*--------------------------------------------------------------------------------------------------
  Write out the symbol for the schematic.
--------------------------------------------------------------------------------------------------*/
int create_default_symbol(
    char *schemName)
{
    int passed;
  
    if(!utAccess(schemName, "r")) {
        utWarning("Unable to read schematic %s", schemName);
        return 0;
    }
    passed = schGenerateSymbolFile(schemName);
    if(!passed) {
        return 0;
    }
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Read a schematic netlist into the database.
--------------------------------------------------------------------------------------------------*/
int read_schematic(
    char *designName,
    char *schemName)
{
    if(!utAccess(schemName, "r")) {
        utWarning("Unable to read schematic %s", schemName);
        return 0;
    }
    dbCurrentDesign = schReadSchematic(designName, schemName);
    if(dbCurrentDesign == dbDesignNull) {
        return 0;
    }
    dbCurrentNetlist = dbDesignGetRootNetlist(dbCurrentDesign);
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Write a spice netlist.
--------------------------------------------------------------------------------------------------*/
int write_spice_netlist(
    char *spiceFileName,
    int includeTopLevelPorts,
    int maxLineLength)
{
    bool passed;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("write_spice_netlist: No current design in the database");
        return 0;
    }
    passed = cirWriteDesign(dbCurrentDesign, spiceFileName, (bool)includeTopLevelPorts,
        maxLineLength);
    if(!passed) {
        return 0;
    }
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Write a Verilog netlist.
--------------------------------------------------------------------------------------------------*/
int write_verilog_netlist(
    char *verilogFileName)
{
    bool passed;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("write_verilog_netlist: No current design in the database");
        return 0;
    }
    passed = vrWriteDesign(dbCurrentDesign, verilogFileName);
    if(!passed) {
        return 0;
    }
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Read a Verilog netlist.
--------------------------------------------------------------------------------------------------*/
int read_verilog_netlist(
    char *designName,
    char *verilogFileName)
{
    dbCurrentDesign = vrReadDesign(designName, verilogFileName);
    if(dbCurrentDesign == dbDesignNull) {
        utWarning("Unable to read Verilog file %s", verilogFileName);
        return 0;
    }
    dbCurrentNetlist = dbDesignGetRootNetlist(dbCurrentDesign);
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Cause primitive components to be generated into actual library devices.
--------------------------------------------------------------------------------------------------*/
int generate_devices(void)
{
    if(dbCurrentDesign == dbDesignNull) {
        utWarning("genereate_devices: No design present in database");
        return  0;
    }
    if(geGenerateDevices()) {
        return 1;
    }
    return 0;
}

/*--------------------------------------------------------------------------------------------------
  Connvert power instances into globals.
--------------------------------------------------------------------------------------------------*/
void convert_power_insts_to_globals(void)
{
    if(dbCurrentDesign == dbDesignNull) {
        utWarning("convert_power_insts_to_globals: no current design");
    } else {
        dbDesignConvertPowerInstsToGlobals(dbCurrentDesign);
    }
}

/*--------------------------------------------------------------------------------------------------
  Thread a global signal from the root netlist all the way down to where it is used.  Try to use
  mport names that match the global name.
--------------------------------------------------------------------------------------------------*/
void thread_global_through_hierarchy(
    char *globalName,
    int createTopLevelPorts)
{
    dbGlobal global;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("thread_global_through_hierarchy: no current design");
    } else {
        global = dbDesignFindGlobal(dbCurrentDesign, utSymCreate(globalName));
        if(global == dbGlobalNull) {
            utWarning("thread_globals_through_hierarchy: no global '%s' found in design",
                globalName);
        } else {
            dbThreadGlobalThroughHierarchy(global, createTopLevelPorts);
        }
    }
}

/*--------------------------------------------------------------------------------------------------
  Thread global signals from the root netlist all the way down to where they are used.  Try to use
  mport names that match the global names.
--------------------------------------------------------------------------------------------------*/
void thread_globals_through_hierarchy(
    int createTopLevelPorts)
{
    if(dbCurrentDesign == dbDesignNull) {
        utWarning("thread_globals_through_hierarchy: no current design");
    } else {
        dbThreadGlobalsThroughHierarchy(dbCurrentDesign, createTopLevelPorts);
    }
}

/*--------------------------------------------------------------------------------------------------
  Rename the global.
--------------------------------------------------------------------------------------------------*/
void rename_global(
    char *globalName,
    char *newGlobalName)
{
    dbGlobal global;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("rename_global: no current design");
    } else {
        global = dbDesignFindGlobal(dbCurrentDesign, utSymCreate(globalName));
        if(global == dbGlobalNull) {
            utWarning("rename_global: no global named %s in current design", globalName);
        } else {
            dbGlobalRename(global, utSymCreate(newGlobalName));
        }
    }
}

/*--------------------------------------------------------------------------------------------------
  Rename the netlist.
--------------------------------------------------------------------------------------------------*/
void rename_netlist(
    char *netlistName,
    char *newNetlistName)
{
    dbNetlist netlist;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("rename_netlist: no current design");
    } else {
        netlist = dbDesignFindNetlist(dbCurrentDesign, utSymCreate(netlistName));
        if(netlist == dbNetlistNull) {
            utWarning("rename_netlist: no netlist named %s in current design", netlistName);
        } else {
            dbNetlistRename(netlist, utSymCreate(newNetlistName));
        }
    }
}

/*--------------------------------------------------------------------------------------------------
  Set all names in the netlist to  upper case.
--------------------------------------------------------------------------------------------------*/
int make_netlists_upper_case(
    char *designName)
{
    dbDesign design = dbRootFindDesign(dbTheRoot, utSymCreate(designName));

    if(design == dbDesignNull) {
        utWarning("make_netlist_upper_case: could not find design %s", designName);
        return 0;
    }
    dbDesignMakeNetlistNamesUpperCase(design);
    return 1;
}

/*--------------------------------------------------------------------------------------------------
  Set an attribute on a netlist.
--------------------------------------------------------------------------------------------------*/
void set_netlist_value(
    char *netlistName,
    char *propName,
    char *value)
{
    dbNetlist netlist;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("set_netlist_value: no current design");
    } else {
        netlist = dbDesignFindNetlist(dbCurrentDesign, utSymCreate(netlistName));
        if(netlist == dbNetlistNull) {
            utWarning("set_netlist_value: no netlist named %s in current design", netlistName);
        } else {
            dbNetlistSetValue(netlist, utSymCreate(propName), utSymCreate(value));
        }
    }
}

/*--------------------------------------------------------------------------------------------------
  Get an attribute of an netlist.
--------------------------------------------------------------------------------------------------*/
char *get_netlist_value(
    char *netlistName,
    char *propName)
{
    dbNetlist netlist;
    utSym value;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("get_netlist_value: no current design");
        return "";
    }
    netlist = dbDesignFindNetlist(dbCurrentDesign, utSymCreate(netlistName));
    if(netlist == dbNetlistNull) {
        utWarning("get_netlist_value: no netlist named %s in current design", netlistName);
        return "";
    }
    value = dbNetlistGetValue(netlist, utSymCreate(propName));
    if(value == utSymNull) {
        return "";
    }
    return utSymGetName(value);
}

/*--------------------------------------------------------------------------------------------------
  Set an attribute on a inst.
--------------------------------------------------------------------------------------------------*/
void set_inst_value(
    char *instName,
    char *propName,
    char *value)
{
    dbInst inst;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("set_inst_value: no current design");
        return;
    }
    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("set_inst_value: no current netlist in the database");
        return;
    }
    inst = dbNetlistFindInst(dbCurrentNetlist, utSymCreate(instName));
    if(inst == dbInstNull) {
        utWarning("set_inst_value: no inst named %s in current netlist", instName);
    } else {
        dbInstSetValue(inst, utSymCreate(propName), utSymCreate(value));
    }
}

/*--------------------------------------------------------------------------------------------------
  Get an attribute of an inst.
--------------------------------------------------------------------------------------------------*/
char *get_inst_value(
    char *instName,
    char *propName)
{
    dbInst inst;
    utSym value;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("get_inst_value: no current design");
        return "";
    }
    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("get_inst_value: no current netlist in the database");
        return "";
    }
    inst = dbNetlistFindInst(dbCurrentNetlist, utSymCreate(instName));
    if(inst == dbInstNull) {
        utWarning("get_inst_value: no inst named %s in current netlist", instName);
        return "";
    }
    value = dbInstGetValue(inst, utSymCreate(propName));
    if(value == utSymNull) {
        return "";
    }
    return utSymGetName(value);
}

/*--------------------------------------------------------------------------------------------------
  Set an attribute on a net.
--------------------------------------------------------------------------------------------------*/
void set_net_value(
    char *netName,
    char *propName,
    char *value)
{
    dbNet net;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("set_net_value: no current design");
        return;
    }
    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("set_net_value: no current netlist in the database");
        return;
    }
    net = dbNetlistFindNet(dbCurrentNetlist, utSymCreate(netName));
    if(net == dbNetNull) {
        utWarning("set_net_value: no net named %s in current netlist", netName);
    } else {
        dbNetSetValue(net, utSymCreate(propName), utSymCreate(value));
    }
}

/*--------------------------------------------------------------------------------------------------
  Get an attribute of an net.
--------------------------------------------------------------------------------------------------*/
char *get_net_value(
    char *netName,
    char *propName)
{
    dbNet net;
    utSym value;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("get_net_value: no current design");
        return "";
    }
    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("get_net_value: no current netlist in the database");
        return "";
    }
    net = dbNetlistFindNet(dbCurrentNetlist, utSymCreate(netName));
    if(net == dbNetNull) {
        utWarning("get_net_value: no net named %s in current netlist", netName);
        return "";
    }
    value = dbNetGetValue(net, utSymCreate(propName));
    if(value == utSymNull) {
        return "";
    }
    return utSymGetName(value);
}

/*--------------------------------------------------------------------------------------------------
  Explode array instances into individual instances.
--------------------------------------------------------------------------------------------------*/
void explode_instance_arrays(void)
{
    if(dbCurrentDesign == dbDesignNull) {
        utWarning("explode_instance_arrays: no current design");
        return;
    }
    dbDesignExplodeArrayInsts(dbCurrentDesign);
}

/*--------------------------------------------------------------------------------------------------
  Get the first instance of the current netlist.
--------------------------------------------------------------------------------------------------*/
char *get_first_inst(void)
{
    dbInst inst;

    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("get_first_inst: no current netlist in the database");
        return "";
    }
    dbForeachNetlistInst(dbCurrentNetlist, inst) {
        if(dbInstGetType(inst) != DB_FLAG) {
            return dbInstGetName(inst);
        }
    } dbEndForeachNetlistInst;
    return "";
}

/*--------------------------------------------------------------------------------------------------
  Get the next instance of the current netlist.
--------------------------------------------------------------------------------------------------*/
char *get_next_inst(
    char *instName)
{
    dbInst inst;

    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("get_next_inst: no current netlist in the database");
        return "";
    }
    inst = dbNetlistFindInst(dbCurrentNetlist, utSymCreate(instName));
    if(inst == dbInstNull) {
        utWarning("get_next_inst: no instance named %s in the database", instName);
        return "";
    }
    utDo {
        inst = dbInstGetnextNetlistInst(inst);
    } utWhile(inst != dbInstNull) {
        if(dbInstGetType(inst) != DB_FLAG) {
            return dbInstGetName(inst);
        }
    } utRepeat;
    return "";
}

/*--------------------------------------------------------------------------------------------------
  Get the first net of the current netlist.
--------------------------------------------------------------------------------------------------*/
char *get_first_net(void)
{
    dbNet net;

    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("get_first_net: no current netlist in the database");
        return "";
    }
    net = dbNetlistGetfirstNet(dbCurrentNetlist);
    if(net == dbNetNull) {
        return "";
    }
    return dbNetGetName(net);
}

/*--------------------------------------------------------------------------------------------------
  Get the next net of the current netlist.
--------------------------------------------------------------------------------------------------*/
char *get_next_net(
    char *netName)
{
    dbNet net;

    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("get_next_net: no current netlist in the database");
        return "";
    }
    net = dbNetlistFindNet(dbCurrentNetlist, utSymCreate(netName));
    if(net == dbNetNull) {
        utWarning("get_next_net: no net named %s in the database", netName);
        return "";
    }
    net = dbNetGetnextNetlistNet(net);
    if(net == dbNetNull) {
        return "";
    }
    return dbNetGetName(net);
}

/*--------------------------------------------------------------------------------------------------
  Get the first netlist of the current design.
--------------------------------------------------------------------------------------------------*/
char *get_first_netlist(void)
{
    dbNetlist netlist;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("get_first_netlist: no current design in the database");
        return "";
    }
    dbForeachDesignNetlist(dbCurrentDesign, netlist) {
        if(dbNetlistGetType(netlist) == DB_SUBCIRCUIT) {
            return dbNetlistGetName(netlist);
        }
    } dbEndForeachDesignNetlist;
    return "";
}

/*--------------------------------------------------------------------------------------------------
  Get the next netlist of the current design.
--------------------------------------------------------------------------------------------------*/
char *get_next_netlist(
    char *netlistName)
{
    dbNetlist netlist;

    if(dbCurrentDesign == dbDesignNull) {
        utWarning("get_next_netlist: no current design in the database");
        return "";
    }
    netlist = dbDesignFindNetlist(dbCurrentDesign, utSymCreate(netlistName));
    if(netlist == dbNetlistNull) {
        utWarning("get_next_netlist: no netlist named %s in the database", netlistName);
        return "";
    }
    utDo {
        netlist = dbNetlistGetnextDesignNetlist(netlist);
    } utWhile(netlist != dbNetlistNull) {
        if(dbNetlistGetType(netlist) == DB_SUBCIRCUIT) {
            return dbNetlistGetName(netlist);
        }
    } utRepeat;
    return "";
}

/*--------------------------------------------------------------------------------------------------
  Get the internal netlist name of the instance.
--------------------------------------------------------------------------------------------------*/
char *get_inst_internal_netlist(
    char *instName)
{
    dbInst inst;

    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("get_inst_internal_netlist: no current netlist in the database");
        return "";
    }
    inst = dbNetlistFindInst(dbCurrentNetlist, utSymCreate(instName));
    if(inst == dbInstNull) {
        utWarning("get_inst_internal_netlist: no instance named %s in the database", instName);
        return "";
    }
    return dbNetlistGetName(dbInstGetInternalNetlist(inst));
}

/*--------------------------------------------------------------------------------------------------
  Determine if a net by the name exists in the current netlist.
--------------------------------------------------------------------------------------------------*/
int net_exists(
    char *netName)
{
    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("net_exists: no current netlist in the database");
        return false;
    }
    return dbNetlistFindNet(dbCurrentNetlist, utSymCreate(netName)) != dbNetNull;
}

/*--------------------------------------------------------------------------------------------------
  Determine if a inst by the name exists in the current netlist.
--------------------------------------------------------------------------------------------------*/
int inst_exists(
    char *instName)
{
    if(dbCurrentNetlist == dbNetlistNull) {
        utWarning("inst_exists: no current netlist in the database");
        return false;
    }
    return dbNetlistFindInst(dbCurrentNetlist, utSymCreate(instName)) != dbInstNull;
}
