/*----------------------------------------------------------------------------
  Simple class support routines such as list manipulation.
----------------------------------------------------------------------------*/
#include "vrddr.h"

struct _utFunctionTable vrNetlistFunctionTable = {utSymNull, vrNetlistDestroy};
utSym vrNetlistClassName;
vrNetlist _vrFirstFreeNetlist, _vrTempNetlist;
utHeapRef _vrNetlistHeap;
void (*vrIdecDestructorHook)(vrIdec);
vrIdec _vrFirstFreeIdec, _vrTempIdec;
utHeapRef _vrIdecHeap;
void (*vrParamDestructorHook)(vrParam);
vrParam _vrFirstFreeParam, _vrTempParam;
utHeapRef _vrParamHeap;
void (*vrConnDestructorHook)(vrConn);
vrConn _vrFirstFreeConn, _vrTempConn;
utHeapRef _vrConnHeap;
void (*vrDefparamDestructorHook)(vrDefparam);
vrDefparam _vrFirstFreeDefparam, _vrTempDefparam;
utHeapRef _vrDefparamHeap;
void (*vrPathDestructorHook)(vrPath);
vrPath _vrFirstFreePath, _vrTempPath;
utHeapRef _vrPathHeap;

/*----------------------------------------------------------------------------
  Add the Idec to the head of the list on the Netlist.
----------------------------------------------------------------------------*/
void vrNetlistInsertIdec(
    vrNetlist Netlist,
    vrIdec _Idec)
{
    vrIdecSetnextNetlistIdec(_Idec, vrNetlistGetfirstIdec(Netlist));
    vrNetlistSetfirstIdec(Netlist, _Idec);
    if(vrNetlistGetlastIdec(Netlist) == vrIdecNull) {
        vrNetlistSetlastIdec(Netlist, _Idec);
    }
    vrIdecSetNetlist(_Idec, Netlist);
}

/*----------------------------------------------------------------------------
  Remove the Idec from the list on the Netlist.
----------------------------------------------------------------------------*/
void vrNetlistRemoveIdec(
    vrNetlist Netlist,
    vrIdec _Idec)
{
    vrIdec nextIdec, prevIdec = vrIdecNull;

    for(nextIdec = vrNetlistGetfirstIdec(Netlist);
        nextIdec != vrIdecNull && nextIdec != _Idec;
        nextIdec = vrIdecGetnextNetlistIdec(nextIdec)) {
        prevIdec = nextIdec;
    }
    if(nextIdec == vrIdecNull) {
        /* Not in list */
        return;
    }
    if(prevIdec != vrIdecNull) {
        vrIdecSetnextNetlistIdec(prevIdec, vrIdecGetnextNetlistIdec(_Idec));
    } else {
        vrNetlistSetfirstIdec(Netlist, vrIdecGetnextNetlistIdec(_Idec));
    }
    if(vrNetlistGetlastIdec(Netlist) == _Idec) {
        vrNetlistSetlastIdec(Netlist, prevIdec);
    }
    vrIdecSetnextNetlistIdec(_Idec, vrIdecNull);
    vrIdecSetNetlist(_Idec, vrNetlistNull);
}

/*----------------------------------------------------------------------------
  Add the Idec to the tail of the list on the Netlist.
----------------------------------------------------------------------------*/
void vrNetlistAppendIdec(
    vrNetlist Netlist,
    vrIdec _Idec)
{
    if(vrNetlistGetlastIdec(Netlist) == vrIdecNull) {
        vrNetlistSetfirstIdec(Netlist, _Idec);
    } else {
        vrIdecSetnextNetlistIdec(vrNetlistGetlastIdec(Netlist), _Idec);
    }
    vrNetlistSetlastIdec(Netlist, _Idec);
    vrIdecSetNetlist(_Idec, Netlist);
}

/*----------------------------------------------------------------------------
  Add the Defparam to the head of the list on the Netlist.
----------------------------------------------------------------------------*/
void vrNetlistInsertDefparam(
    vrNetlist Netlist,
    vrDefparam _Defparam)
{
    vrDefparamSetnextNetlistDefparam(_Defparam, vrNetlistGetfirstDefparam(Netlist));
    vrNetlistSetfirstDefparam(Netlist, _Defparam);
    vrDefparamSetNetlist(_Defparam, Netlist);
}

/*----------------------------------------------------------------------------
  Remove the Defparam from the list on the Netlist.
----------------------------------------------------------------------------*/
void vrNetlistRemoveDefparam(
    vrNetlist Netlist,
    vrDefparam _Defparam)
{
    vrDefparam nextDefparam, prevDefparam = vrDefparamNull;

    for(nextDefparam = vrNetlistGetfirstDefparam(Netlist);
        nextDefparam != vrDefparamNull && nextDefparam != _Defparam;
        nextDefparam = vrDefparamGetnextNetlistDefparam(nextDefparam)) {
        prevDefparam = nextDefparam;
    }
    if(nextDefparam == vrDefparamNull) {
        /* Not in list */
        return;
    }
    if(prevDefparam != vrDefparamNull) {
        vrDefparamSetnextNetlistDefparam(prevDefparam, vrDefparamGetnextNetlistDefparam(_Defparam));
    } else {
        vrNetlistSetfirstDefparam(Netlist, vrDefparamGetnextNetlistDefparam(_Defparam));
    }
    vrDefparamSetnextNetlistDefparam(_Defparam, vrDefparamNull);
    vrDefparamSetNetlist(_Defparam, vrNetlistNull);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Netlist.
------------------------------------------------------------------------*/
static utBlockRef buildNetlistBlock(void)
{
    utBlockRef block = utcBlock();
    vrNetlist Netlist;
    U16 x, sNetlist;

    utaHeapBlock(_vrNetlistHeap, block);
    sNetlist = UTBLOCKSIZE/sizeof(struct _vrNetlist);
    _vrFirstFreeNetlist = (vrNetlist)utgBlockMem(block);
    Netlist = _vrFirstFreeNetlist;
    for(x = 0; x < sNetlist; x++) {
        _vrNetlistNextFree(Netlist) = Netlist + 1;
        Netlist++;
    }
    _vrNetlistNextFree(--Netlist) = vrNetlistNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Netlist.
----------------------------------------------------------------------------*/
vrNetlist vrNetlistAlloc(dbNetlist super)
{
    vrNetlist Netlist;

    if(_vrFirstFreeNetlist == vrNetlistNull) {
        buildNetlistBlock();
    }
    Netlist = _vrFirstFreeNetlist;
    _vrFirstFreeNetlist = _vrNetlistNextFree(Netlist);
    memset((void *)Netlist, 0, sizeof(struct _vrNetlist));
    Netlist->_extension.next = dbNetlistGetfirstExtension(super);
    dbNetlistSetfirstExtension(super, &Netlist->_extension);
    Netlist->_extension.functionTable = &vrNetlistFunctionTable;
    Netlist->_super = super;
    return Netlist;
}

/*----------------------------------------------------------------------------
  Free a Netlist.
----------------------------------------------------------------------------*/
void vrNetlistFree(
    vrNetlist Netlist)
{
    dbNetlistSetfirstExtension(vrNetlistSuper(Netlist), Netlist->_extension.next);
    _vrNetlistNextFree(Netlist) = _vrFirstFreeNetlist;
    _vrFirstFreeNetlist = Netlist;
}

/*----------------------------------------------------------------------------
  Destroy a Netlist.
----------------------------------------------------------------------------*/
void vrNetlistDestroy(
    vrNetlist Netlist)
{
    {
        vrIdec _Idec;
        vrSafeForeachNetlistIdec(Netlist, _Idec) {
            vrIdecDestroy(_Idec);
        } vrEndSafeForeachNetlistIdec;
    }
    {
        vrDefparam _Defparam;
        vrSafeForeachNetlistDefparam(Netlist, _Defparam) {
            vrDefparamDestroy(_Defparam);
        } vrEndSafeForeachNetlistDefparam;
    }
    vrNetlistFree(Netlist);
}

/*----------------------------------------------------------------------------
  Add the Param to the head of the list on the Idec.
----------------------------------------------------------------------------*/
void vrIdecInsertParam(
    vrIdec Idec,
    vrParam _Param)
{
    vrParamSetnextIdecParam(_Param, vrIdecGetfirstParam(Idec));
    vrIdecSetfirstParam(Idec, _Param);
    if(vrIdecGetlastParam(Idec) == vrParamNull) {
        vrIdecSetlastParam(Idec, _Param);
    }
    vrParamSetIdec(_Param, Idec);
}

/*----------------------------------------------------------------------------
  Remove the Param from the list on the Idec.
----------------------------------------------------------------------------*/
void vrIdecRemoveParam(
    vrIdec Idec,
    vrParam _Param)
{
    vrParam nextParam, prevParam = vrParamNull;

    for(nextParam = vrIdecGetfirstParam(Idec);
        nextParam != vrParamNull && nextParam != _Param;
        nextParam = vrParamGetnextIdecParam(nextParam)) {
        prevParam = nextParam;
    }
    if(nextParam == vrParamNull) {
        /* Not in list */
        return;
    }
    if(prevParam != vrParamNull) {
        vrParamSetnextIdecParam(prevParam, vrParamGetnextIdecParam(_Param));
    } else {
        vrIdecSetfirstParam(Idec, vrParamGetnextIdecParam(_Param));
    }
    if(vrIdecGetlastParam(Idec) == _Param) {
        vrIdecSetlastParam(Idec, prevParam);
    }
    vrParamSetnextIdecParam(_Param, vrParamNull);
    vrParamSetIdec(_Param, vrIdecNull);
}

/*----------------------------------------------------------------------------
  Add the Param to the tail of the list on the Idec.
----------------------------------------------------------------------------*/
void vrIdecAppendParam(
    vrIdec Idec,
    vrParam _Param)
{
    if(vrIdecGetlastParam(Idec) == vrParamNull) {
        vrIdecSetfirstParam(Idec, _Param);
    } else {
        vrParamSetnextIdecParam(vrIdecGetlastParam(Idec), _Param);
    }
    vrIdecSetlastParam(Idec, _Param);
    vrParamSetIdec(_Param, Idec);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Idec.
------------------------------------------------------------------------*/
static utBlockRef buildIdecBlock(void)
{
    utBlockRef block = utcBlock();
    vrIdec Idec;
    U16 x, sIdec;

    utaHeapBlock(_vrIdecHeap, block);
    sIdec = UTBLOCKSIZE/sizeof(struct _vrIdec);
    _vrFirstFreeIdec = (vrIdec)utgBlockMem(block);
    Idec = _vrFirstFreeIdec;
    for(x = 0; x < sIdec; x++) {
        _vrIdecNextFree(Idec) = Idec + 1;
        Idec++;
    }
    _vrIdecNextFree(--Idec) = vrIdecNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Idec.
----------------------------------------------------------------------------*/
vrIdec _vrIdecAlloc(void)
{
    vrIdec Idec;

    if(_vrFirstFreeIdec == vrIdecNull) {
        buildIdecBlock();
    }
    Idec = _vrFirstFreeIdec;
    _vrFirstFreeIdec = _vrIdecNextFree(Idec);
    memset((void *)Idec, 0, sizeof(struct _vrIdec));
    return Idec;
}

/*----------------------------------------------------------------------------
  Free a Idec.
----------------------------------------------------------------------------*/
void _vrIdecFree(
    vrIdec Idec)
{
    _vrIdecNextFree(Idec) = _vrFirstFreeIdec;
    _vrFirstFreeIdec = Idec;
}

/*----------------------------------------------------------------------------
  Destroy a Idec.
----------------------------------------------------------------------------*/
void vrIdecDestroy(
    vrIdec Idec)
{
    if(vrIdecDestructorHook != NULL) {
        vrIdecDestructorHook(Idec);
    }
    {
        vrParam _Param;
        vrSafeForeachIdecParam(Idec, _Param) {
            vrParamDestroy(_Param);
        } vrEndSafeForeachIdecParam;
    }
    {
        vrNetlist parentNetlist = vrIdecGetNetlist(Idec);
        if(parentNetlist != vrNetlistNull) {
            vrNetlistRemoveIdec(parentNetlist, Idec);
        }
    }
    vrIdecFree(Idec);
}

/*----------------------------------------------------------------------------
  Add the Conn to the head of the list on the Param.
----------------------------------------------------------------------------*/
void vrParamInsertConn(
    vrParam Param,
    vrConn _Conn)
{
    vrConnSetnextParamConn(_Conn, vrParamGetfirstConn(Param));
    vrParamSetfirstConn(Param, _Conn);
    if(vrParamGetlastConn(Param) == vrConnNull) {
        vrParamSetlastConn(Param, _Conn);
    }
    vrConnSetParam(_Conn, Param);
}

/*----------------------------------------------------------------------------
  Remove the Conn from the list on the Param.
----------------------------------------------------------------------------*/
void vrParamRemoveConn(
    vrParam Param,
    vrConn _Conn)
{
    vrConn nextConn, prevConn = vrConnNull;

    for(nextConn = vrParamGetfirstConn(Param);
        nextConn != vrConnNull && nextConn != _Conn;
        nextConn = vrConnGetnextParamConn(nextConn)) {
        prevConn = nextConn;
    }
    if(nextConn == vrConnNull) {
        /* Not in list */
        return;
    }
    if(prevConn != vrConnNull) {
        vrConnSetnextParamConn(prevConn, vrConnGetnextParamConn(_Conn));
    } else {
        vrParamSetfirstConn(Param, vrConnGetnextParamConn(_Conn));
    }
    if(vrParamGetlastConn(Param) == _Conn) {
        vrParamSetlastConn(Param, prevConn);
    }
    vrConnSetnextParamConn(_Conn, vrConnNull);
    vrConnSetParam(_Conn, vrParamNull);
}

/*----------------------------------------------------------------------------
  Add the Conn to the tail of the list on the Param.
----------------------------------------------------------------------------*/
void vrParamAppendConn(
    vrParam Param,
    vrConn _Conn)
{
    if(vrParamGetlastConn(Param) == vrConnNull) {
        vrParamSetfirstConn(Param, _Conn);
    } else {
        vrConnSetnextParamConn(vrParamGetlastConn(Param), _Conn);
    }
    vrParamSetlastConn(Param, _Conn);
    vrConnSetParam(_Conn, Param);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Param.
------------------------------------------------------------------------*/
static utBlockRef buildParamBlock(void)
{
    utBlockRef block = utcBlock();
    vrParam Param;
    U16 x, sParam;

    utaHeapBlock(_vrParamHeap, block);
    sParam = UTBLOCKSIZE/sizeof(struct _vrParam);
    _vrFirstFreeParam = (vrParam)utgBlockMem(block);
    Param = _vrFirstFreeParam;
    for(x = 0; x < sParam; x++) {
        _vrParamNextFree(Param) = Param + 1;
        Param++;
    }
    _vrParamNextFree(--Param) = vrParamNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Param.
----------------------------------------------------------------------------*/
vrParam _vrParamAlloc(void)
{
    vrParam Param;

    if(_vrFirstFreeParam == vrParamNull) {
        buildParamBlock();
    }
    Param = _vrFirstFreeParam;
    _vrFirstFreeParam = _vrParamNextFree(Param);
    memset((void *)Param, 0, sizeof(struct _vrParam));
    return Param;
}

/*----------------------------------------------------------------------------
  Free a Param.
----------------------------------------------------------------------------*/
void _vrParamFree(
    vrParam Param)
{
    _vrParamNextFree(Param) = _vrFirstFreeParam;
    _vrFirstFreeParam = Param;
}

/*----------------------------------------------------------------------------
  Destroy a Param.
----------------------------------------------------------------------------*/
void vrParamDestroy(
    vrParam Param)
{
    if(vrParamDestructorHook != NULL) {
        vrParamDestructorHook(Param);
    }
    {
        vrConn _Conn;
        vrSafeForeachParamConn(Param, _Conn) {
            vrConnDestroy(_Conn);
        } vrEndSafeForeachParamConn;
    }
    {
        vrIdec parentIdec = vrParamGetIdec(Param);
        if(parentIdec != vrIdecNull) {
            vrIdecRemoveParam(parentIdec, Param);
        }
    }
    vrParamFree(Param);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Conn.
------------------------------------------------------------------------*/
static utBlockRef buildConnBlock(void)
{
    utBlockRef block = utcBlock();
    vrConn Conn;
    U16 x, sConn;

    utaHeapBlock(_vrConnHeap, block);
    sConn = UTBLOCKSIZE/sizeof(struct _vrConn);
    _vrFirstFreeConn = (vrConn)utgBlockMem(block);
    Conn = _vrFirstFreeConn;
    for(x = 0; x < sConn; x++) {
        _vrConnNextFree(Conn) = Conn + 1;
        Conn++;
    }
    _vrConnNextFree(--Conn) = vrConnNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Conn.
----------------------------------------------------------------------------*/
vrConn _vrConnAlloc(void)
{
    vrConn Conn;

    if(_vrFirstFreeConn == vrConnNull) {
        buildConnBlock();
    }
    Conn = _vrFirstFreeConn;
    _vrFirstFreeConn = _vrConnNextFree(Conn);
    memset((void *)Conn, 0, sizeof(struct _vrConn));
    return Conn;
}

/*----------------------------------------------------------------------------
  Free a Conn.
----------------------------------------------------------------------------*/
void _vrConnFree(
    vrConn Conn)
{
    _vrConnNextFree(Conn) = _vrFirstFreeConn;
    _vrFirstFreeConn = Conn;
}

/*----------------------------------------------------------------------------
  Destroy a Conn.
----------------------------------------------------------------------------*/
void vrConnDestroy(
    vrConn Conn)
{
    if(vrConnDestructorHook != NULL) {
        vrConnDestructorHook(Conn);
    }
    {
        vrParam parentParam = vrConnGetParam(Conn);
        if(parentParam != vrParamNull) {
            vrParamRemoveConn(parentParam, Conn);
        }
    }
    vrConnFree(Conn);
}

/*----------------------------------------------------------------------------
  Add the Path to the head of the list on the Defparam.
----------------------------------------------------------------------------*/
void vrDefparamInsertPath(
    vrDefparam Defparam,
    vrPath _Path)
{
    vrPathSetnextDefparamPath(_Path, vrDefparamGetfirstPath(Defparam));
    vrDefparamSetfirstPath(Defparam, _Path);
    if(vrDefparamGetlastPath(Defparam) == vrPathNull) {
        vrDefparamSetlastPath(Defparam, _Path);
    }
    vrPathSetDefparam(_Path, Defparam);
}

/*----------------------------------------------------------------------------
  Remove the Path from the list on the Defparam.
----------------------------------------------------------------------------*/
void vrDefparamRemovePath(
    vrDefparam Defparam,
    vrPath _Path)
{
    vrPath nextPath, prevPath = vrPathNull;

    for(nextPath = vrDefparamGetfirstPath(Defparam);
        nextPath != vrPathNull && nextPath != _Path;
        nextPath = vrPathGetnextDefparamPath(nextPath)) {
        prevPath = nextPath;
    }
    if(nextPath == vrPathNull) {
        /* Not in list */
        return;
    }
    if(prevPath != vrPathNull) {
        vrPathSetnextDefparamPath(prevPath, vrPathGetnextDefparamPath(_Path));
    } else {
        vrDefparamSetfirstPath(Defparam, vrPathGetnextDefparamPath(_Path));
    }
    if(vrDefparamGetlastPath(Defparam) == _Path) {
        vrDefparamSetlastPath(Defparam, prevPath);
    }
    vrPathSetnextDefparamPath(_Path, vrPathNull);
    vrPathSetDefparam(_Path, vrDefparamNull);
}

/*----------------------------------------------------------------------------
  Add the Path to the tail of the list on the Defparam.
----------------------------------------------------------------------------*/
void vrDefparamAppendPath(
    vrDefparam Defparam,
    vrPath _Path)
{
    if(vrDefparamGetlastPath(Defparam) == vrPathNull) {
        vrDefparamSetfirstPath(Defparam, _Path);
    } else {
        vrPathSetnextDefparamPath(vrDefparamGetlastPath(Defparam), _Path);
    }
    vrDefparamSetlastPath(Defparam, _Path);
    vrPathSetDefparam(_Path, Defparam);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Defparam.
------------------------------------------------------------------------*/
static utBlockRef buildDefparamBlock(void)
{
    utBlockRef block = utcBlock();
    vrDefparam Defparam;
    U16 x, sDefparam;

    utaHeapBlock(_vrDefparamHeap, block);
    sDefparam = UTBLOCKSIZE/sizeof(struct _vrDefparam);
    _vrFirstFreeDefparam = (vrDefparam)utgBlockMem(block);
    Defparam = _vrFirstFreeDefparam;
    for(x = 0; x < sDefparam; x++) {
        _vrDefparamNextFree(Defparam) = Defparam + 1;
        Defparam++;
    }
    _vrDefparamNextFree(--Defparam) = vrDefparamNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Defparam.
----------------------------------------------------------------------------*/
vrDefparam _vrDefparamAlloc(void)
{
    vrDefparam Defparam;

    if(_vrFirstFreeDefparam == vrDefparamNull) {
        buildDefparamBlock();
    }
    Defparam = _vrFirstFreeDefparam;
    _vrFirstFreeDefparam = _vrDefparamNextFree(Defparam);
    memset((void *)Defparam, 0, sizeof(struct _vrDefparam));
    return Defparam;
}

/*----------------------------------------------------------------------------
  Free a Defparam.
----------------------------------------------------------------------------*/
void _vrDefparamFree(
    vrDefparam Defparam)
{
    _vrDefparamNextFree(Defparam) = _vrFirstFreeDefparam;
    _vrFirstFreeDefparam = Defparam;
}

/*----------------------------------------------------------------------------
  Destroy a Defparam.
----------------------------------------------------------------------------*/
void vrDefparamDestroy(
    vrDefparam Defparam)
{
    if(vrDefparamDestructorHook != NULL) {
        vrDefparamDestructorHook(Defparam);
    }
    {
        vrPath _Path;
        vrSafeForeachDefparamPath(Defparam, _Path) {
            vrPathDestroy(_Path);
        } vrEndSafeForeachDefparamPath;
    }
    {
        vrNetlist parentNetlist = vrDefparamGetNetlist(Defparam);
        if(parentNetlist != vrNetlistNull) {
            vrNetlistRemoveDefparam(parentNetlist, Defparam);
        }
    }
    vrDefparamFree(Defparam);
}

/*------------------------------------------------------------------------
  Allocate a new memory block for the Path.
------------------------------------------------------------------------*/
static utBlockRef buildPathBlock(void)
{
    utBlockRef block = utcBlock();
    vrPath Path;
    U16 x, sPath;

    utaHeapBlock(_vrPathHeap, block);
    sPath = UTBLOCKSIZE/sizeof(struct _vrPath);
    _vrFirstFreePath = (vrPath)utgBlockMem(block);
    Path = _vrFirstFreePath;
    for(x = 0; x < sPath; x++) {
        _vrPathNextFree(Path) = Path + 1;
        Path++;
    }
    _vrPathNextFree(--Path) = vrPathNull;
    return block;
}

/*----------------------------------------------------------------------------
  Allocate and initialize a new Path.
----------------------------------------------------------------------------*/
vrPath _vrPathAlloc(void)
{
    vrPath Path;

    if(_vrFirstFreePath == vrPathNull) {
        buildPathBlock();
    }
    Path = _vrFirstFreePath;
    _vrFirstFreePath = _vrPathNextFree(Path);
    memset((void *)Path, 0, sizeof(struct _vrPath));
    return Path;
}

/*----------------------------------------------------------------------------
  Free a Path.
----------------------------------------------------------------------------*/
void _vrPathFree(
    vrPath Path)
{
    _vrPathNextFree(Path) = _vrFirstFreePath;
    _vrFirstFreePath = Path;
}

/*----------------------------------------------------------------------------
  Destroy a Path.
----------------------------------------------------------------------------*/
void vrPathDestroy(
    vrPath Path)
{
    if(vrPathDestructorHook != NULL) {
        vrPathDestructorHook(Path);
    }
    {
        vrDefparam parentDefparam = vrPathGetDefparam(Path);
        if(parentDefparam != vrDefparamNull) {
            vrDefparamRemovePath(parentDefparam, Path);
        }
    }
    vrPathFree(Path);
}

/*----------------------------------------------------------------------------
  Initialize memory for the vr&&.
----------------------------------------------------------------------------*/
void vrDDRStart(void)
{
    vrNetlistClassName = utSymCreate("vrNetlist");
    vrNetlistFunctionTable.className = vrNetlistClassName;
    _vrFirstFreeNetlist = vrNetlistNull;
    _vrNetlistHeap = utcHeap();
    buildNetlistBlock();
    _vrFirstFreeIdec = vrIdecNull;
    _vrIdecHeap = utcHeap();
    buildIdecBlock();
    _vrFirstFreeParam = vrParamNull;
    _vrParamHeap = utcHeap();
    buildParamBlock();
    _vrFirstFreeConn = vrConnNull;
    _vrConnHeap = utcHeap();
    buildConnBlock();
    _vrFirstFreeDefparam = vrDefparamNull;
    _vrDefparamHeap = utcHeap();
    buildDefparamBlock();
    _vrFirstFreePath = vrPathNull;
    _vrPathHeap = utcHeap();
    buildPathBlock();
}

/*----------------------------------------------------------------------------
  Free memory for the vr&&.
----------------------------------------------------------------------------*/
void vrDDRStop(void)
{
    utFreeHeap(_vrNetlistHeap);
    utFreeHeap(_vrIdecHeap);
    utFreeHeap(_vrParamHeap);
    utFreeHeap(_vrConnHeap);
    utFreeHeap(_vrDefparamHeap);
    utFreeHeap(_vrPathHeap);
}

