// Package code is used to analyse Go code packages.
// It can find out all the implementation relations between package-level type.
package code

import (
	"container/list"
	"fmt"
	"go/ast"
	"go/token"
	"go/types"
	"log"
	"os"
	"strings"
	//"runtime/debug"

	"golang.org/x/tools/go/types/typeutil"
)

// The analysis steps.
const (
	SubTask_PreparationDone = iota
	SubTask_NFilesParsed
	SubTask_ParsePackagesDone
	SubTask_CollectPackages
	SubTask_CollectModules
	SubTask_CollectExamples
	SubTask_SortPackagesByDependencies
	SubTask_CollectDeclarations
	SubTask_CollectRuntimeFunctionPositions
	//SubTask_ConfirmTypeSources
	SubTask_CollectMoreTypes
	SubTask_CollectSelectors
	SubTask_CheckCollectedSelectors
	SubTask_FindImplementations
	SubTask_RegisterInterfaceMethodsForTypes
	SubTask_MakeStatistics
	SubTask_CollectSourceFiles
	SubTask_CollectObjectReferences
	SubTask_CacheSourceFiles
)

type ToolchainInfo struct {
	// Three paths
	Root, Src, Cmd string

	// A commit hash or something like "go1.16".
	// ToDo: now lso might be blank, but need some handling ...
	Version string
}

// CodeAnalyzer holds all the analysis results and functionalities.
type CodeAnalyzer struct {
	modulesByPath       map[string]*Module // including stdModule
	nonToolchainModules []Module           // not including stdModule and std/cmd module
	stdModule           *Module
	wdModule            *Module // working diretory module. It might be the cmd toolchain module, or nil if modules feature is off.

	//stdPackages  map[string]struct{}
	packageTable map[string]*Package
	packageList  []*Package
	builtinPkg   *Package

	// This one is removed now. We should use FileSet.PositionFor.
	//sourceFileLineOffsetTable map[string]int32

	//=== Will be fulfilled in analyse phase ===

	allSourceFiles map[string]*SourceFileInfo
	//sourceFile2PackageTable  map[string]SourceFile
	//sourceFile2PackageTable         map[string]*Package
	//generatedFile2OriginalFileTable map[string]string

	exampleFileSet *token.FileSet

	// *types.Type -> *TypeInfo
	lastTypeIndex       uint32
	ttype2TypeInfoTable typeutil.Map
	allTypeInfos        []*TypeInfo

	instantiatedTypes        *list.List
	numSeenInstantiatedTypes uint32

	//>> 1.18, fake underlying for instantiated types
	// Always nil for 1.17-.
	//blankInterface *TypeInfo
	//<<

	// ToDo: it looks this table is not much useful.
	//
	// Package-level declared type names.
	lastTypeNameIndex uint32
	allTypeNameTable  map[string]*TypeName // not including instantiated ones

	// ToDo: need a better implementation.
	typeMethodsContributingToTypeImplementations map[[4]string]struct{}

	// Position info of some runtime functions.
	runtimeFuncPositions map[string]token.Position

	// Refs of unnamed types, type names, variables, functions, ...
	// Why not put []RefPos in TypeInfo, Variable, ...?
	//refPositions map[interface{}][]RefPos

	// ToDo: some TopN lists
	stats Stats

	// Identifer references (ToDo: need optimizations)
	objectRefs map[types.Object][]Identifier

	// Not concurrent safe.
	tempTypeLookup map[uint32]struct{}

	//
	forbidRegisterTypes bool // for debug

	debug bool
}

// WorkingDirectoryModule returns the module at the working directory.
// It might be nil.
func (d *CodeAnalyzer) WorkingDirectoryModule() *Module {
	return d.wdModule
}

// ModuleByPath returns the module corresponding the specified path.
func (d *CodeAnalyzer) ModuleByPath(path string) *Module {
	return d.modulesByPath[path]
}

// IterateModule iterates all modules and passes them to the specified callback f.
func (d *CodeAnalyzer) IterateModule(f func(*Module)) {
	if d.stdModule != nil {
		f(d.stdModule)
	}
	if d.wdModule != nil {
		f(d.wdModule)
	}
	for i := range d.nonToolchainModules {
		m := &d.nonToolchainModules[i]
		if m != d.wdModule {
			f(m)
		}
	}
}

func (d *CodeAnalyzer) regObjectReference(obj types.Object, fileInfo *SourceFileInfo, id *ast.Ident) {
	if d.objectRefs == nil {
		d.objectRefs = map[types.Object][]Identifier{} // ToDo: estimate an initial minimum capacity
	}

	ids := d.objectRefs[obj]
	if ids == nil {
		ids = make([]Identifier, 0, 2) // ToDo: estimate an initial minimum capacity. How?
	}
	ids = append(ids, Identifier{FileInfo: fileInfo, AstIdent: id})
	d.objectRefs[obj] = ids
}

// ObjectReferences returns all the references to the given object.
func (d *CodeAnalyzer) ObjectReferences(obj types.Object) []Identifier {
	ids := d.objectRefs[obj]
	if ids == nil {
		return nil
	}
	dups := make([]Identifier, len(ids))
	copy(dups, ids)
	return dups
}

// Please reset it after using.
func (d *CodeAnalyzer) tempTypeLookupTable() map[uint32]struct{} {
	if d.tempTypeLookup == nil {
		d.tempTypeLookup = make(map[uint32]struct{}, 1024)
	}
	return d.tempTypeLookup
}

func (d *CodeAnalyzer) resetTempTypeLookupTable() {
	// the gc compiler optimize this
	for k := range d.tempTypeLookup {
		delete(d.tempTypeLookup, k)
	}
}

// NumPackages returns packages count.
func (d *CodeAnalyzer) NumPackages() int {
	return len(d.packageList)
}

// PackageAt returns the packages at specified index i.
func (d *CodeAnalyzer) PackageAt(i int) *Package {
	return d.packageList[i]
}

// PackageByPath returns the packages corresponding the specified path.
func (d *CodeAnalyzer) PackageByPath(path string) *Package {
	return d.packageTable[path]
}

// IsStandardPackage returns whether or not the given package is a standard package.
func (d *CodeAnalyzer) IsStandardPackage(pkg *Package) bool {
	return pkg.module == d.stdModule
}

// IsStandardPackageByPath returns whether or not the package specified by the path is a standard package.
func (d *CodeAnalyzer) IsStandardPackageByPath(path string) bool {
	pkg, ok := d.packageTable[path]
	return ok && d.IsStandardPackage(pkg)
}

// BuiltinPackge returns the builtin package.
func (d *CodeAnalyzer) BuiltinPackge() *Package {
	return d.builtinPkg
}

// NumSourceFiles returns the source files count.
func (d *CodeAnalyzer) NumSourceFiles() int {
	//return len(d.sourceFile2PackageTable) // including generated files and non-go files
	return len(d.allSourceFiles) // including generated files and non-go files
}

// Must be callsed after stats.FilesWithGenerateds is confirmed.
func (d *CodeAnalyzer) buildSourceFileTable() {
	d.allSourceFiles = make(map[string]*SourceFileInfo, d.stats.FilesWithGenerateds)
	for _, pkg := range d.packageList {
		for i := range pkg.SourceFiles {
			f := &pkg.SourceFiles[i]
			d.allSourceFiles[pkg.Path+"/"+f.AstBareFileName()] = f
		}
	}
}

// SourceFile returns the source file coresponding the specified file name.
// pkgFile: pkg.Path + "/" + file.BareFilename
func (d *CodeAnalyzer) SourceFile(pkgFile string) *SourceFileInfo {
	return d.allSourceFiles[pkgFile]
}

func (d *CodeAnalyzer) ExampleFileSet() *token.FileSet {
	return d.exampleFileSet
}

//func (d *CodeAnalyzer) SourceFile2Package(path string) (*Package, bool) {
//	srcFile, ok := d.sourceFile2PackageTable[path]
//	return srcFile, ok
//}

//func (d *CodeAnalyzer) SourceFileLineOffset(path string) int {
//	return int(d.sourceFileLineOffsetTable[path])
//}

//func (d *CodeAnalyzer) OriginalGoSourceFile(filename string) string {
//	if f, ok := d.generatedFile2OriginalFileTable[filename]; ok {
//		return f
//	}
//	return filename
//}

// RuntimeFunctionCodePosition returns the position of the specified runtime function f.
func (d *CodeAnalyzer) RuntimeFunctionCodePosition(f string) token.Position {
	return d.runtimeFuncPositions[f]
}

// RuntimePackage returns the runtime package.
func (d *CodeAnalyzer) RuntimePackage() *Package {
	return d.PackageByPath("runtime")
}

// Id1 builds an id from the specified package and identifier name.
// The result is the same as go/types.Id.
func (d *CodeAnalyzer) Id1(p *types.Package, name string) string {
	if p == nil {
		p = d.builtinPkg.PPkg.Types
	}

	return types.Id(p, name)
}

// Id1b builds an id from the specified package and identifier name.
// The result is almost the same as go/types.Id.
func (d *CodeAnalyzer) Id1b(pkg *Package, name string) string {
	if pkg == nil {
		return d.Id1(nil, name)
	}

	return d.Id1(pkg.PPkg.Types, name)
}

// ToDo: avoid string concating by using a struct{pkg; name} for Id and Id2 functions.

// Id2 builds an id from the specified package and identifier name.
func (d *CodeAnalyzer) Id2(p *types.Package, name string) string {
	if p == nil {
		p = d.builtinPkg.PPkg.Types
	}

	return p.Path() + "." + name
}

// Id2b builds an id from the specified package and identifier name.
func (d *CodeAnalyzer) Id2b(pkg *Package, name string) string {
	if pkg == nil {
		return d.Id2(nil, name)
	}

	return d.Id2(pkg.PPkg.Types, name)
}

// Declared functions.
//func (d *CodeAnalyzer) RegisterFunction(f *Function) {
//	// meaningful?
//}

// RegisterTypeName registers a TypeName.
// Only origin ones should be registered.
func (d *CodeAnalyzer) RegisterTypeName(tn *TypeName) {
	if d.allTypeNameTable == nil {
		d.allTypeNameTable = make(map[string]*TypeName, 4096)
	}
	tn.index = d.lastTypeNameIndex
	d.lastTypeNameIndex++
	if name := tn.Name(); name != "_" {
		//d.allTypeNameTable[tn.Id()] = tn
		// Unify the id generations.
		// !!! For builtin types, tn.TypeName.Pkg() == nil
		//d.allTypeNameTable[types.Id(tn.TypeName.Pkg(), tn.TypeName.Name())] = tn

		//d.allTypeNameTable[types.Id(tn.Pkg.PPkg.Types, tn.TypeName.Name())] = tn
		d.allTypeNameTable[d.Id2b(tn.Pkg, tn.TypeName.Name())] = tn

		//log.Println(">>>", types.Id(tn.Pkg.PPkg.Types, tn.TypeName.Name()))
	}
}

func (d *CodeAnalyzer) registerInstantiatedType(t *TypeInfo, typeArgs []TypeExpr) {

	switch tt := t.TT.(type) {
	case *types.Named:
		ot := d.RegisterType(originType(tt))
		//t.Origin = ot
		if ot.TypeName == nil {
			panic("ot.TypeName == nil")
		}
		//t.TypeName = ot.TypeName // already set in registeringType
		if t.TypeName == nil || true {
			// see: https://github.com/go101/golds/issues/52
			//      I haven't get why a named type has not a type name now.
			// panic("t.TypeName == nil")

			if len(os.Getenv("GoldsIssue52")) > 0 {
				log.Printf("Issue#52: registerInstantiatedType:\n\tntt=%v\n\tntt.Obj()=%v\n\tot.TypeName=%v\n\n", tt, tt.Obj(), ot.TypeName)
			}
			t.TypeName = ot.TypeName
		}
	case *types.Alias:
		// ToDo: now Go support alias generic types, things become more complicated ...
		return
	default:
		log.Printf("????? %T\n\t%v", t.TT, t.TT)
		panic("should not")
	}

	t.Instantiated = &InstantiatedInfo{
		TypeArgs: typeArgs,
	}

	if d.instantiatedTypes == nil {
		d.instantiatedTypes = list.New()
	}
	d.instantiatedTypes.PushBack(t)
	d.numSeenInstantiatedTypes++
}

// RegisterType registers a go/types.Type as TypeInfo.
func (d *CodeAnalyzer) RegisterType(t types.Type) *TypeInfo {
	return d.registeringType(t, true)
}

// LookForType trys to find out the TypeInfo registered for the spefified types.Type.
func (d *CodeAnalyzer) LookForType(t types.Type) *TypeInfo {
	return d.registeringType(t, false)
}

func (d *CodeAnalyzer) registeringType(tt types.Type, createOnNonexist bool) *TypeInfo {
	if tt == nil {
		tt = types.Typ[types.Invalid]
	}

	typeInfo, _ := d.ttype2TypeInfoTable.At(tt).(*TypeInfo)
	if typeInfo == nil && createOnNonexist {
		if d.forbidRegisterTypes {
			if d.debug {
				log.Printf("================================= %v, %T", tt, tt)
			}
		}

		//d.lastTypeIndex++ // the old design (1-based)
		typeInfo = &TypeInfo{TT: tt, index: d.lastTypeIndex}
		d.lastTypeIndex++ // the new design (0-based)

		d.ttype2TypeInfoTable.Set(tt, typeInfo)
		if d.allTypeInfos == nil {
			d.allTypeInfos = make([]*TypeInfo, 0, 8192)

			// The old design ensure all type index > 0,
			// which maight be an unnecessary design.
			//d.allTypeInfos = append(d.allTypeInfos, nil)
		}
		d.allTypeInfos = append(d.allTypeInfos, typeInfo)

		switch tt := tt.(type) {
		case *types.Named:
			// ToDo: Use types.Identical instead of comparison operator?
			//if ott := originType(tt); ott != tt {
			//	typeInfo.Origin = d.RegisterType(ott)
			//} else {
			//	typeInfo.Origin = typeInfo
			//}
			if ott := originType(tt); ott != tt {
				origin := d.RegisterType(ott)

				// origin.TypeName may be nil now, for
				// the origin type might declared later.
				// So also need to try to set typeInfo.TypeName
				// again in collectDirectSelectorsForSourceType,
				// confirmInstantiatedInfoForTypeConstraints, and so on.
				if origin.TypeName != nil {
					//panic("should not")
					typeInfo.TypeName = origin.TypeName
				}
			}

			//typeInfo.Name = tt.Obj().Name()

			underlying := d.RegisterType(tt.Underlying())
			typeInfo.Underlying = underlying
			//underlying.Underlying = underlying // already done

			//numNameds++
			//if _, ok := tt.Underlying().(*types.Interface); ok {
			//	numNamedInterfaces++
			//}
		//>> 1.18
		// No need to be special, use default behavior.
		// Just need to be care in findImplementations.
		// ToDo: reject to register such types.
		//case *typesTypeParam:
		//	typeInfo.Underlying = typeInfo
		//<<
		default:
			typeInfo.Underlying = typeInfo
		}

		switch tt.Underlying().(type) {
		case *types.Pointer:
			// The exception is to avoid infinite pointer depth.
		case *types.Interface:
			// Pointers of interfaces are not important.
		default:
			// *T might have methods if T is neigher an interface nor pointer type.
			d.RegisterType(types.NewPointer(tt))
		}
	}
	return typeInfo
}

// RetrieveTypeName trys to retrieve the TypeName from a TypeInfo.
func (d *CodeAnalyzer) RetrieveNamedType(t *TypeInfo) (*TypeInfo, bool) {
	if tn := t.TypeName; tn != nil {
		return t, false
	}

	if ptt, ok := t.TT.(*types.Pointer); ok {
		bt := d.RegisterType(ptt.Elem())
		if btn := bt.TypeName; btn != nil {
			return bt, true
		}

		if _, ok := bt.TT.(*types.Named); ok {
			//	//>> 1.18 (temp handling, need more handling. bug#33)
			//	if ott := originType(ntt); ott != ntt {
			//		ot := d.RegisterType(ott)
			//		if otn := ot.TypeName; otn != nil {
			//			log.Println("===============", ott)
			//			return otn, true
			//		}
			//	}
			//	//<<

			panic("named types must has a type name (*)")
		}
	}

	if _, ok := t.TT.(*types.Named); ok {
		panic("named types must has a type name")
	}

	return nil, false
}

// Methods contribute to type implementations.
// The key is typeIndex << 32 | methodIndex.
// typeIndex must be the index of a non-interface type.
//
// typeMethodsContributingToTypeImplementations map[uint64]]struct{}
//
//func (d *CodeAnalyzer) registerTypeMethodContributingToTypeImplementations(typeIndex, methodIndex uint32) {
//	if d.typeMethodsContributingToTypeImplementations == nil {
//		d.typeMethodsContributingToTypeImplementations = make(map[int64]struct{}, d.lastTypeIndex*3)
//	}
//	key := uint64(typeIndex)<<32 | uint64(methodIndex)
//	d.typeMethodsContributingToTypeImplementations[key] = struct{}{}
//}
//
//// typeIndex must be the index of a non-interface type.
//func (d *CodeAnalyzer) CheckTypeMethodContributingToTypeImplementations(typeIndex, methodIndex uint32) bool {
//	key := uint64(typeIndex)<<32 | uint64(methodIndex)
//	_, ok := d.typeMethodsContributingToTypeImplementations[key]
//	return ok
//}

func (d *CodeAnalyzer) registerTypeMethodContributingToTypeImplementations(pkg, typ, methodPkg, method string) {
	if d.typeMethodsContributingToTypeImplementations == nil {
		d.typeMethodsContributingToTypeImplementations = make(map[[4]string]struct{}, d.lastTypeIndex*3)
	}
	d.typeMethodsContributingToTypeImplementations[[4]string{pkg, typ, methodPkg, method}] = struct{}{}
}

// CheckTypeMethodContributingToTypeImplementations checks whether or not a method implements some interface methods.
func (d *CodeAnalyzer) CheckTypeMethodContributingToTypeImplementations(pkg, typ, methodPkg, method string) bool {
	_, ok := d.typeMethodsContributingToTypeImplementations[[4]string{pkg, typ, methodPkg, method}]
	return ok
}

// CleanImplements returns a clean list of the implementions for a TypeInfo.
func (d *CodeAnalyzer) CleanImplements(self *TypeInfo, includingUnnamed bool) []Implementation {
	// remove:
	// * self
	// * unnameds whose underlied names are also in the list (or are self)
	// The ones in internal packages are kept.

	typeLookupTable := d.tempTypeLookupTable()
	defer d.resetTempTypeLookupTable()

	if itt, ok := self.TT.Underlying().(*types.Interface); ok {
		typeLookupTable[self.index] = struct{}{}
		ut := d.RegisterType(itt)
		typeLookupTable[ut.index] = struct{}{}
	}

	implements := make([]Implementation, 0, len(self.Implements))
	for _, impl := range self.Implements {
		it := impl.Interface
		if it.TypeName == nil {
			continue
		}
		if _, ok := typeLookupTable[it.index]; ok {
			continue
		}
		typeLookupTable[it.index] = struct{}{}
		//ut := d.RegisterType(it.TT.Underlying())
		ut := it.Underlying
		typeLookupTable[ut.index] = struct{}{}
		implements = append(implements, impl)
	}

	if includingUnnamed {
		for _, impl := range self.Implements {
			it := impl.Interface
			if it.TypeName != nil {
				continue
			}
			if _, ok := typeLookupTable[it.index]; ok {
				continue
			}
			implements = append(implements, impl)
		}
	}

	return implements
}

// iterateTypenames is used in registerFunctionForInvolvedTypeNames
// to build AsInputsOf and AsOutputsOf lists.
func (d *CodeAnalyzer) iterateTypenames(typeLiteral ast.Expr, pkg *Package, onTypeName func(*TypeInfo)) {
	switch node := typeLiteral.(type) {
	default:
		panic(fmt.Sprintf("unexpected ast expression. %T : %v", node, node))

	case *astIndexExpr:
		typeArgs := d.astIndexExprToTypeArgs(pkg, node)
		tt := pkg.PPkg.TypesInfo.TypeOf(typeLiteral)
		typeInfo := d.RegisterType(tt)
		d.registerInstantiatedType(typeInfo, typeArgs)

		d.iterateTypenames(node.X, pkg, onTypeName) // to find the origin type
	case *astIndexListExpr:
		typeArgs := d.astIndexListExprToTypeArgs(pkg, node)
		tt := pkg.PPkg.TypesInfo.TypeOf(typeLiteral)
		typeInfo := d.RegisterType(tt)
		d.registerInstantiatedType(typeInfo, typeArgs)

		d.iterateTypenames(node.X, pkg, onTypeName) // to find the origin type
	case *ast.Ident:
		// ToDo: The latter two are added since Go 1.18.
		//       To avoid returning too much weak-related results,
		//       the indexes types in the lattet two are ignored now.

		tt := pkg.PPkg.TypesInfo.TypeOf(node)
		if tt == nil {
			// ToDo: good?
			if pkg.Path == "unsafe" {
				return
			}
			//log.Println("??? type of node is nil:", node.Name, pkg.Path)
			return
		}
		switch t := tt.(type) {
		default:
			// not interested

			// log.Printf("%T, %v", tt, tt)

			// ToDo: it might be an alias to an unnamed type
			//       To also collect functions for such aliases.

			return // not interested
		case *typesTypeParam: // 1.18
			//log.Printf("%T, %v", tt, tt)
			return // not interested
		case *types.Basic:
		case *types.Named:
			//>> 1.18
			tt = originType(t) // should be no-op here
			//<<
		}
		typeInfo := d.RegisterType(tt)
		if typeInfo.TypeName == nil {
			//panic("not a named type")
			return
		}
		onTypeName(typeInfo)
	case *ast.SelectorExpr: // ToDo: merge with *ast.IndexExpr, *ast.IndexListExpr branch?
		d.iterateTypenames(node.Sel, pkg, onTypeName)
	case *ast.ParenExpr:
		d.iterateTypenames(node.X, pkg, onTypeName)
	case *ast.StarExpr:
		d.iterateTypenames(node.X, pkg, onTypeName)
	case *ast.ArrayType:
		d.iterateTypenames(node.Elt, pkg, onTypeName)
	case *ast.Ellipsis: // ...Ele
		d.iterateTypenames(node.Elt, pkg, onTypeName)
	case *ast.MapType:
		d.iterateTypenames(node.Key, pkg, onTypeName)
		d.iterateTypenames(node.Value, pkg, onTypeName)
	case *ast.ChanType:
		d.iterateTypenames(node.Value, pkg, onTypeName)

	// ToDo: To avoid returning too much weak-related results,
	//       the following types are ignored now.
	case *ast.FuncType:
	case *ast.StructType:
	case *ast.InterfaceType:
	}
}

// I some forget the meaningfulness of this method.
// It looks this method is to collect complete method lists for types,
// which is important to calculate implementation relations.
//
// As Go 1.18 introduces custom generics, two new possible type exprssion nodes
// (ast.IndexExpr and ast.IndexListExpr, both denote named types) are added,
// so the "unnamed" word in the method name is not accurate now.
func (d *CodeAnalyzer) lookForAndRegisterUnnamedInterfaceAndStructTypes(typeLiteral ast.Expr, pkg *Package) {
	// ToDo: move this func to package level?
	var reg = func(n ast.Expr) *TypeInfo {
		tv := pkg.PPkg.TypesInfo.Types[n]
		return d.RegisterType(tv.Type)
	}

	switch node := typeLiteral.(type) {

	//>> 1.18, ToDo
	// If the underlying type of reg(node) is an interfae type,
	// need to find a way to get the ast.Node for the interface type.
	// Need an inverse-lookup from the IndexExpr/IndexListExpr to the generics declaration
	// so that the ast.Node could be found.
	// So, need one more pass:
	// 1. collect all the instances of each generic type, (in a loop "for range allPkgs")
	// 2. in another loop "for range allPkgs", handle all variable types:
	//    "for range pkg.PackageAnalyzeResult.AllVariables"
	// The WriteAstType function might also need some tweaks,
	// by replace the type parameters with type arguments,
	// but might also not (maybe WriteAstType doesn't require to do this).
	//
	// Maybe, at least for some situations,
	// registerDirectFields and registerExplicitlySpecifiedMethods don't need
	// set the ast.Node for Field and Method structs, but the need to handle the cases
	// in which AstFunc or AstField is nil. The ast nodes are used in fields/method listing.

	case *astIndexExpr:
		//log.Printf("======== node = %v, %#v", node, node)
		t := reg(node)
		typeArgs := d.astIndexExprToTypeArgs(pkg, node)
		d.registerInstantiatedType(t, typeArgs)

		//>> 1.18, ToDo
		// How to get the instanced ast.Node, with the specified TypeParams?
		//if t.Underlying is interface or struct {
		//	d.registerDirectFields needs ast info
		//    d.registerExplicitlySpecifiedMethods needs ast info
		//}
		//<<

		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(node.Index, pkg)
	case *astIndexListExpr:
		//log.Printf("======== node = %v, %#v", node, node)
		t := reg(node)
		typeArgs := d.astIndexListExprToTypeArgs(pkg, node)
		d.registerInstantiatedType(t, typeArgs)

		//>> 1.18, ToDo
		// How to get the instanced ast.Node, with the specified TypeParams?
		//if t.Underlying is interface or struct {
		//	d.registerDirectFields needs ast info
		//    d.registerExplicitlySpecifiedMethods needs ast info
		//}
		//<<

		for _, index := range node.Indices {
			d.lookForAndRegisterUnnamedInterfaceAndStructTypes(index, pkg)
		}
	case *ast.BinaryExpr, *ast.UnaryExpr:
		// The two may be used as type constraints.
	//<<
	default:
		panic(fmt.Sprintf("unexpected ast expression. %T : %v", node, node))
	case *ast.BadExpr:
		log.Println("encounter BadExpr:", node)
		return
	case *ast.Ident, *ast.SelectorExpr:
		// named types and basic types will be registered from other routes.
		return
	case *ast.ParenExpr:
		reg(node)
		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(node.X, pkg)
	case *ast.StarExpr:
		reg(node)
		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(node.X, pkg)
	case *ast.ArrayType:
		reg(node)
		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(node.Elt, pkg)
	case *ast.Ellipsis: // ...Ele
		reg(node)
		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(node.Elt, pkg)
	case *ast.ChanType:
		reg(node)
		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(node.Value, pkg)
	case *ast.FuncType:
		reg(node)
		for _, field := range node.Params.List {
			d.lookForAndRegisterUnnamedInterfaceAndStructTypes(field.Type, pkg)
		}
		if node.Results != nil {
			for _, field := range node.Results.List {
				d.lookForAndRegisterUnnamedInterfaceAndStructTypes(field.Type, pkg)
			}
		}
	case *ast.MapType:
		reg(node)
		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(node.Key, pkg)
		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(node.Value, pkg)
	case *ast.StructType:
		d.registerDirectFields(reg(node), node, pkg)
		return
	case *ast.InterfaceType:
		d.registerExplicitlySpecifiedMethods(reg(node), node, pkg)
		return
	}
}

func (d *CodeAnalyzer) registerDirectFields(typeInfo *TypeInfo, astStructNode *ast.StructType, pkg *Package) {
	if (typeInfo.attributes & directSelectorsCollected) != 0 {
		return
	}
	typeInfo.attributes |= directSelectorsCollected

	register := func(field *Field) {
		if field.Name == "-" { // ??? ToDo: forget what does this mean?
			panic("impossible")
		}
		// ToDo, ToDo2: the handling of ".Pkg" here might be some problematic.
		// 1. The ".Pkg" field is always set in the callers of this function.
		// 2. Is the "sel.Id" calculation for builtinPkg+unexportedField right?
		if !token.IsExported(field.Name) {
			field.Pkg = pkg
			if pkg == d.builtinPkg {
				field.Pkg = nil
			}
		}
		if typeInfo.DirectSelectors == nil {
			typeInfo.DirectSelectors = make([]*Selector, 0, 16)
		}
		sel := &Selector{
			Id:    d.Id1b(field.Pkg, field.Name),
			Field: field,
		}
		typeInfo.DirectSelectors = append(typeInfo.DirectSelectors, sel)
	}

	for _, field := range astStructNode.Fields.List {
		//var id string
		var fieldName string
		var isStar = false

		for node := field.Type; ; {
			switch expr := node.(type) {
			default:
				if len(field.Names) == 0 {
					panic("not an embedded field but should be. type: " + fmt.Sprintf("%T", expr))
				}
			//>> ToDo: Go 1.18
			case *astIndexExpr:
				tt := pkg.PPkg.TypesInfo.TypeOf(expr)
				t := d.RegisterType(tt)
				typeArgs := d.astIndexExprToTypeArgs(pkg, expr)
				d.registerInstantiatedType(t, typeArgs)

				node = expr.X
				continue
			case *astIndexListExpr:
				tt := pkg.PPkg.TypesInfo.TypeOf(expr)
				t := d.RegisterType(tt)
				typeArgs := d.astIndexListExprToTypeArgs(pkg, expr)
				d.registerInstantiatedType(t, typeArgs)

				node = expr.X
				continue
			//<<
			case *ast.Ident:
				//id = d.Id1b(pkg, expr.Name) // incorrect for builtin typenames

				//tn := pkg.PPkg.TypesInfo.Uses[expr]
				//id = d.Id2(tn.Pkg(), expr.Name)
				fieldName = expr.Name
			case *ast.SelectorExpr:
				//srcObj := pkg.PPkg.TypesInfo.ObjectOf(expr.X.(*ast.Ident))
				//srcPkg := srcObj.(*types.PkgName)
				//id = d.Id2(srcPkg.Imported(), expr.Sel.Name)
				fieldName = expr.Sel.Name
			case *ast.StarExpr:
				if isStar && len(field.Names) == 0 {
					panic("bad embedded field **T.")
				}

				node = expr.X
				isStar = true
				continue
			case *ast.ParenExpr:
				node = expr.X
				continue
			}

			break
		}

		d.lookForAndRegisterUnnamedInterfaceAndStructTypes(field.Type, pkg)

		var fieldTT = pkg.PPkg.TypesInfo.TypeOf(field.Type)
		var fieldTypeInfo = d.RegisterType(fieldTT)

		var tag string
		if field.Tag != nil {
			tag = field.Tag.Value
		}

		if len(field.Names) == 0 {

			//tn := d.allTypeNameTable[id]
			//if tn == nil {
			//	panic("TypeName for " + id + " not found")
			//}
			//
			//fieldTypeInfo := tn.Named
			//if fieldTypeInfo == nil {
			//	fieldTypeInfo = tn.Alias.Denoting
			//}
			//
			// fieldName := tn.Name()

			//tv := pkg.PPkg.TypesInfo.Types[field.Type]

			// ToDo: if field.Type is an interface or struct, or pointer to interface or struct, collect direct selectors.
			// or even disassemble any complex types and look for struct and interface types.

			//fieldTypeInfo := d.RegisterType(tv.Type)

			//>> 1.18 (temp handling, need more handling. bug#33)
			//   need a replaceDirectSelectors(...)
			//if ntt, ok := tv.Type.(*types.Named); ok {
			//	if ott := originType(ntt); ntt != ott {
			//		ot := d.RegisterType(ott)
			//		fieldTypeInfo.DirectSelectors = ot.DirectSelectors
			//	}
			//}
			//<<

			//if tt, ok := tv.Type.(*types.Named); ok {
			//	log.Println("================", tt)
			//	t := fieldTypeInfo
			//	log.Println(tt)
			//	log.Println(t.DirectSelectors)
			//	log.Println(tt.TypeArgs(), tt.TypeParams())
			//	log.Println(tt.Origin())
			//	var ot = d.RegisterType(tt.Origin())
			//	log.Println(ot.DirectSelectors)
			//	log.Println(tt.Origin().TypeArgs(), tt.Origin().TypeParams())
			//}

			//>>
			// ToDo:
			// Field type might present as a type alias to a pointer type.
			// So the calculation for isStar might be wrong above?
			// But it looks the EmbedMode_Indirect and EmbedMode_Direct
			// enums are not effectively used, so ...
			//<<

			embedMode := EmbedMode_Direct
			if isStar {
				//fieldTypeInfo = d.RegisterType(types.NewPointer(fieldTypeInfo.TT))
				embedMode = EmbedMode_Indirect
			}

			typeInfo.EmbeddingFields++

			register(&Field{
				Pkg:  pkg,
				Name: fieldName,
				Type: fieldTypeInfo,
				Mode: embedMode,
				Tag:  tag,

				astStruct: astStructNode,
				AstField:  field,
			})

			continue
		}

		//tv := pkg.PPkg.TypesInfo.Types[field.Type]

		// ToDo: if field.Type is an interface or struct, or pointer to interface or struct, collect direct selectors.
		// or even disassemble any complex types and look for struct and interface types.

		//fieldTypeInfo := d.RegisterType(tv.Type)

		for _, ident := range field.Names {
			if ident.Name == "_" {
				continue
			}

			register(&Field{
				Pkg:  pkg,
				Name: ident.Name,
				Type: fieldTypeInfo,
				Mode: EmbedMode_None,
				Tag:  tag,

				astStruct: astStructNode,
				AstField:  field,
			})
		}
	}
}

func (d *CodeAnalyzer) registerParameterAndResultTypes(astFunc *ast.FuncType, pkg *Package) {
	//log.Println("=========================", f.Pkg.Path, f.Name())

	if astFunc.Params != nil {
		for _, fld := range astFunc.Params.List {
			tt := pkg.PPkg.TypesInfo.TypeOf(fld.Type)
			if tt == nil {
				log.Println("tt is nil!")
				continue
			}
			d.RegisterType(tt)
		}
	}

	if astFunc.Results != nil {
		for _, fld := range astFunc.Results.List {
			tt := pkg.PPkg.TypesInfo.TypeOf(fld.Type)
			if tt == nil {
				log.Println("tt is nil!")
				continue
			}
			d.RegisterType(tt)
		}
	}
}

// ToDo: also register function variables?
// This function is to ensure that the selectors of unnamed types are all confirmed before comfirming selectors for all types.
func (d *CodeAnalyzer) registerUnnamedInterfaceAndStructTypesFromParametersAndResults(astFunc *ast.FuncType, pkg *Package) {
	//log.Println("=========================", f.Pkg.Path, f.Name())

	if astFunc.Params != nil {
		for _, fld := range astFunc.Params.List {
			d.lookForAndRegisterUnnamedInterfaceAndStructTypes(fld.Type, pkg)
		}
	}

	if astFunc.Results != nil {
		for _, fld := range astFunc.Results.List {
			d.lookForAndRegisterUnnamedInterfaceAndStructTypes(fld.Type, pkg)
		}
	}
}

// id must be a *ast.Ident or *ast.ast.SelectorExpr
func (d *CodeAnalyzer) lookForUnnamedSourceExpr(pkg *Package, id ast.Expr, forAliasOnly bool) *ast.InterfaceType {
	//obj := pkg.PPkg.TypesInfo.ObjectOf(id)
	//_ = obj.(types.TypeName)
	//tn := d.allTypeNameTable[d.Id2(obj.Pkg(), obj.Name())]
	//if tn == nil {
	//	panic("should not")
	//}

	// ToDo:
	return nil
}

// ToDo: now interface{Error() string} and interface{error} will be viewed as one TypeInfo,
//
//	which is true from Go senmatics view, but might be not good from code analysis view.
//
// typeInfo must be an unnamed interface type.
func (d *CodeAnalyzer) registerExplicitlySpecifiedMethods(typeInfo *TypeInfo, astInterfaceNode *ast.InterfaceType, pkg *Package) {
	//if (typeInfo.attributes & directSelectorsCollected) != 0 {
	//	return
	//}

	// The logic of the above three lines is not right as it looks.
	// In the current go/* packages implementation, "interface{A}" and "type A interface {M{}}"
	// will be viewed as identical types. If they are passed by the above order to this function, ...
	// So now the above three lines are disabled.

	// Another detail is that, unlike embedded fields, the embedded type names in an interface type can be identical.

	typeInfo.attributes |= directSelectorsCollected

	registerMethod := func(method *Method) {
		if method.Name == "-" {
			panic("impossible")
		}
		if !token.IsExported(method.Name) {
			method.Pkg = pkg
			if pkg == d.builtinPkg {
				method.Pkg = nil
			}
		}

		if typeInfo.DirectSelectors == nil {
			typeInfo.DirectSelectors = make([]*Selector, 0, 16)
		}

		newId := d.Id1b(method.Pkg, method.Name)

		//for _, old := range typeInfo.DirectSelectors {
		//	// A method and some field names can be the same.
		//	if old.Id == newId && old.Method != nil {
		//		// See the comment at the starting of the function.
		//
		//		// ToDo: this loop is not effecient,
		//		//       try to find a new efficient way ...
		//
		//		// ToDo: is this if-block possible to be entered? (update: surely)
		//		//       If not, then the loop is useless.
		//
		//		return
		//	}
		//}

		sel := &Selector{
			Id:     newId,
			Method: method,
		}
		typeInfo.DirectSelectors = append(typeInfo.DirectSelectors, sel)
	}

	// ToDo: since Go 1.14, two same name interface names can be embedded together.
	//       But it looks this is not a problem.
	// ToDo: since Go 1.18, any type may be embedded in an interface,
	//       furthmore, more embedding forms: ~T and T1 | T2, ...
	//       But up to 1.20, only an interface types (either unnamed or named)
	//       will make contributions to the method set.
	registerField := func(field *Field) {
		//>> since 1.18, it is possible now
		//if field.Name == "-" {
		//	panic("impossible")
		//}
		//if !token.IsExported(field.Name) {
		//	field.Pkg = pkg
		//	if pkg == d.builtinPkg {
		//		field.Pkg = nil
		//	}
		//}
		//
		//newId := d.Id1b(field.Pkg, field.Name)
		//<<

		if typeInfo.DirectSelectors == nil {
			typeInfo.DirectSelectors = make([]*Selector, 0, 16)
		}

		// Two embedded types in an interface type can have an identical name.
		//for _, old := range typeInfo.DirectSelectors {
		//	if old.Id == newId {
		//		// See the comment at the starting of the function.
		//		return
		//	}
		//}
		// Why is the above block commented away?
		// Because interface {Name(); Name; Name} is valid code.

		sel := &Selector{
			//>> since 1.18
			//Id:    newId,
			//<<
			Field: field,
		}
		typeInfo.DirectSelectors = append(typeInfo.DirectSelectors, sel)
	}

	//log.Println("!!!!!! registerExplicitlySpecifiedMethods:", typeInfo)

	for _, method := range astInterfaceNode.Methods.List {
		// method is a *ast.Field.
		if len(method.Names) == 0 {
			// embed interface type (anonymous field)

			//>> 1.18
			//
			// Now, it is also possible
			// * an unnamed type
			// * an instantiated type (ast.IndexExpr, ast.IndexListExpr)
			// * a type union (ast.BinaryExpr)
			// * ~aType (ast.UnaryExpr)
			//
			// And note that an Ident and SelectorExpr might not
			// denote interface.
			//
			// Update to Go 1.19, we may only need to consider
			// IndexExpr/IndexListExpr/Ident/SelectorExpr which
			// denote interface types.
			//<<

			//var id string
			var astInterfaceType *ast.InterfaceType
			var typeArgs []TypeExpr

			var expr = method.Type

			fieldTT := pkg.PPkg.TypesInfo.TypeOf(expr)
			if fieldTT == nil {
				panic("should not")
			}
			if _, ok := fieldTT.Underlying().(*types.Interface); !ok {
				// Up to Go 1.20, the following cases may be safely ignored:
				// 1. T, where T is a non-interfaces
				// 2. ~T
				// 3. T1 | T2

				//ToDo: 1.18. Up to now, this is valid.
				continue
			}

			var name string
			var expr1 = expr

		NextToFindName:
			switch expr2 := expr1.(type) {
			default:
				//>> 1.18
				panic(fmt.Sprintf("not a valid embedding interface type name: %#v", method))
				//continue
				//<<

			case *ast.InterfaceType: // since 1.18
				astInterfaceType = expr2
			case *astIndexExpr: // since 1.18
				typeArgs = d.astIndexExprToTypeArgs(pkg, expr2)
				expr1 = expr2.X // X is either Ident or SelectorExpr
				goto NextToFindName
			case *astIndexListExpr: // since 1.18
				typeArgs = d.astIndexListExprToTypeArgs(pkg, expr2)
				expr1 = expr2.X // X is either Ident or SelectorExpr
				goto NextToFindName
			case *ast.Ident:
				//ttn := pkg.PPkg.TypesInfo.Uses[expr2]
				//id = d.Id2(ttn.Pkg(), ttn.Name())
				name = expr2.Name
			case *ast.SelectorExpr:
				//srcObj := pkg.PPkg.TypesInfo.ObjectOf(expr2.X.(*ast.Ident))
				//srcPkg := srcObj.(*types.PkgName)
				//id = d.Id2(srcPkg.Imported(), expr2.Sel.Name)
				name = expr2.Sel.Name
			}

			//>> since 1.18, instantiated types are not in this table.
			//tn := d.allTypeNameTable[id]
			//if tn == nil {
			//	panic("TypeName for " + id + " not found")
			//}
			//
			//
			////fieldTypeInfo := tn.Named
			////if fieldTypeInfo == nil {
			////	fieldTypeInfo = tn.Alias.Denoting
			////}
			//fieldTypeInfo := tn.Denoting

			fieldTypeInfo := d.RegisterType(fieldTT)
			if _, ok := fieldTypeInfo.Underlying.TT.(*types.Interface); !ok {
				continue // ToDo: up to Go 1.20, this is okay.
			}

			if astInterfaceType != nil {
				//d.registerExplicitlySpecifiedMethods(fieldTypeInfo, astInterfaceType, pkg)
				// Register the items of unnamed embedded interfaces on the embedding one directly.
				d.registerExplicitlySpecifiedMethods(typeInfo, astInterfaceType, pkg)
				continue
			} else if typeArgs != nil {
				d.registerInstantiatedType(fieldTypeInfo, typeArgs)
			}
			//<<

			embedMode := EmbedMode_Direct

			//if strings.Index(id, "image") >= 0 {
			//log.Println("!!!!!!!!!!! ", id, fieldTypeInfo)
			//}

			registerField(&Field{
				Pkg:  pkg,
				Name: name,
				Type: fieldTypeInfo,
				Mode: embedMode,

				//AstInterface: astInterfaceNode,
				AstField: method,
			})

		} else {
			//log.Println("   method")
			if len(method.Names) != 1 {
				panic(fmt.Sprint("number of method names is not 1: ", len(method.Names), method.Names))
			}

			ident := method.Names[0]
			if ident.Name == "_" {
				continue
			}

			//mtt := pkg.PPkg.TypesInfo.Types[method.Type].Type
			mtt := pkg.PPkg.TypesInfo.TypeOf(method.Type)
			if mtt == nil {
				log.Printf("===> method [%s] has no type!", ident.Name)
				panic("should not")
			}
			methodTypeInfo := d.RegisterType(mtt)

			if pkg == d.builtinPkg && ident.Name == "Error" {
				// The special handling is to correctly find all implementations of the builtin "error" type.
				errorUnderlyingType := types.Universe.Lookup("error").(*types.TypeName).Type().Underlying().(*types.Interface)
				// ToDo: optimization: the result should be cached
				methodTypeInfo = d.RegisterType(errorUnderlyingType.Method(0).Type())
			}

			m := &Method{
				Pkg:  pkg,
				Name: ident.Name,
				Type: methodTypeInfo,

				PointerRecv: false,

				//AstInterface: astInterfaceNode,
				AstField: method,
			}
			//>> 1.18, ToDo
			//m.Parameterized = checkParameterized(m.Type)
			//<<

			registerMethod(m)

			astFunc, ok := method.Type.(*ast.FuncType)
			if !ok {
				panic("should not")
			}
			d.registerUnnamedInterfaceAndStructTypesFromParametersAndResults(astFunc, pkg)
			d.registerParameterAndResultTypes(astFunc, pkg)
		}
	}
	//log.Println("       registerExplicitlySpecifiedMethods:", len(typeInfo.DirectSelectors))
}

// ToDo: to loop parameter and result lists and use AST to constract custom methods.
func (d *CodeAnalyzer) tryToRegisterExplicitlyDeclaredMethod(f *Function) (isMethod bool) {
	if f.Func == nil {
		return
	}

	sig := f.Func.Type().(*types.Signature)
	recv := sig.Recv()
	if recv == nil {
		return
	}

	isMethod, methodName := true, f.Func.Name()
	if methodName == "-" {
		return
	}

	d.registerUnnamedInterfaceAndStructTypesFromParametersAndResults(f.AstDecl.Type, f.Pkg)
	d.registerParameterAndResultTypes(f.AstDecl.Type, f.Pkg)

	recvTT := recv.Type()
	var baseTT *types.Named
	var ptrRecv bool
	switch tt := recvTT.(type) {
	case *types.Named:
		baseTT = tt
		ptrRecv = false
	case *types.Pointer:
		baseTT = tt.Elem().(*types.Named)
		ptrRecv = true
	default:
		panic("impossible")
	}

	// ToDo: using sig.Params() and sig.Results() instead of f.Func.Type()

	//>> 1.18
	baseTT = originType(baseTT)
	// As explicit methods are always declared for orign types,
	// the above line is actually a no-op.
	//<<

	typeInfo := d.RegisterType(baseTT)
	if typeInfo.DirectSelectors == nil {
		selectors := make([]*Selector, 0, 16)
		typeInfo.DirectSelectors = selectors
	}
	method := &Method{
		Pkg:         f.Pkg, // ToDo: research why must set it?
		Name:        methodName,
		Type:        d.RegisterType(f.Func.Type()),
		PointerRecv: ptrRecv,
		AstFunc:     f.AstDecl,
	}
	//>> 1.18, ToDo
	//method.Parameterized = checkParameterized(method.Type)
	//<<
	if !token.IsExported(methodName) {
		method.Pkg = f.Pkg
		if method.Pkg == d.builtinPkg {
			method.Pkg = nil
		}
	}
	sel := &Selector{
		Id:     d.Id1b(method.Pkg, method.Name),
		Method: method,
	}
	typeInfo.DirectSelectors = append(typeInfo.DirectSelectors, sel)
	return
}

// ToDo: also register function variables?
// Return parameter and result counts.
// func (d *CodeAnalyzer) registerFunctionForInvolvedTypeNames(f *Function) (ins, outs int, lastResultIsError bool) {
func (d *CodeAnalyzer) registerFunctionForInvolvedTypeNames(f FunctionResource) (ins, outs int, lastResultIsError bool) {
	// ToDo: unepxorted function should also reged,
	//       but then they should be filtered out when in listing.
	//notToReg := !f.Exported()

	//fType := f.AstDecl.Type
	fType := f.AstFuncType()

	//log.Println("=========================", f.Pkg.Path, f.Name())

	if fType.Params != nil {
		for _, fld := range fType.Params.List {
			if n := len(fld.Names); n == 0 {
				ins++
			} else {
				ins += n
			}
			//if notToReg {
			//	continue
			//}
			//d.iterateTypenames(fld.Type, f.Package(), func(t *TypeInfo) {
			d.iterateTypenames(fld.Type, f.AstPackage(), func(t *TypeInfo) {
				if t.TypeName == nil {
					panic("shoud not")
				}
				//if t.Pkg == nil {
				//	log.Println("================", f.Pkg.Path, t.TT)
				//}
				if t.TypeName.Pkg.Path == "builtin" {
					return
				}
				if t.AsInputsOf == nil {
					t.AsInputsOf = make([]ValueResource, 0, 4)
				}
				t.AsInputsOf = append(t.AsInputsOf, f.(ValueResource))
			})
		}
	}

	if fType.Results != nil {
		for _, fld := range fType.Results.List {
			lastResultIsError = false

			if n := len(fld.Names); n == 0 {
				outs++
			} else {
				outs += n
			}
			//if notToReg {
			//	continue
			//}
			//d.iterateTypenames(fld.Type, f.Package(), func(t *TypeInfo) {
			d.iterateTypenames(fld.Type, f.AstPackage(), func(t *TypeInfo) {
				if t.TypeName == nil {
					panic("shoud not")
				}
				//if t.Pkg == nil {
				//	log.Println("================", f.Pkg.Path, t.TT)
				//}
				if t.TypeName.Pkg.Path == "builtin" {
					if t.TypeName.Name() == "error" {
						lastResultIsError = true
					}
					return
				}
				if t.AsOutputsOf == nil {
					t.AsOutputsOf = make([]ValueResource, 0, 4)
				}
				t.AsOutputsOf = append(t.AsOutputsOf, f.(ValueResource))
			})
		}
	}

	return
}

func (d *CodeAnalyzer) registerValueForItsTypeName(res ValueResource, valueSpec *ast.ValueSpec) {
	t := res.TypeInfo(d)
	if t.TypeName != nil {
		goto Register
	}

	// ToDo: also consider []T, [..]T, chan T, map[K]T, ... ?
	switch tt := t.TT.(type) {
	case *types.Pointer:
		t = d.RegisterType(tt.Elem())
		// ToDo: also register if an unnamed type has some aliases
		if t.TypeName != nil {
			goto Register
		}
	}

	return

Register:

	if t.AsTypesOf == nil {
		t.AsTypesOf = make([]ValueResource, 0, 4)
	}
	t.AsTypesOf = append(t.AsTypesOf, res)

	ot := t.TypeName.Denoting
	if ot != t {
		if ot.AsTypesOf == nil {
			ot.AsTypesOf = make([]ValueResource, 0, 4)
		}
		ot.AsTypesOf = append(ot.AsTypesOf, res)

		if t.Instantiated == nil {
			// Try to confirm the instantiated info.
			// ToDo: but it is not always easy (though always possible I think).

		}
	}
}

// BuildMethodSignatureFromFuncObject builds the signature for function object.
//func (d *CodeAnalyzer) BuildMethodSignatureFromFuncObject(funcObj *types.Func) MethodSignature {
//	funcSig, ok := funcObj.Type().(*types.Signature)
//	if !ok {
//		panic(funcObj.Id() + "'s type is not types.Signature")
//	}
//
//	methodName, pkgImportPath := funcObj.Id(), ""
//	if !token.IsExported(methodName) {
//		pkgImportPath = funcObj.Pkg().Path
//	}
//
//	return d.BuildMethodSignatureFromFunctionSignature(funcSig, methodName, pkgImportPath)
//}

// BuildMethodSignatureFromFunctionSignature  builds the signature for method function object.
// pkgImportPath should be only passed for unexported method names.
func (d *CodeAnalyzer) BuildMethodSignatureFromFunctionSignature(funcSig *types.Signature, methodName string, pkgImportPath string) MethodSignature {
	if pkgImportPath != "" {
		if token.IsExported(methodName) {
			//panic("bad argument: " + pkgImportPath + "." + methodName)
			// ToDo: handle this case gracefully.
			//log.Println("bad argument: " + pkgImportPath + "." + methodName)
			pkgImportPath = ""
		} else if pkgImportPath == "builtin" {
			log.Println("bad argument: " + pkgImportPath + "." + methodName)
			pkgImportPath = ""
		}
	}

	var b strings.Builder
	var writeTypeIndex = func(index uint32) {
		b.WriteByte(byte((index >> 24) & 0xFF))
		b.WriteByte(byte((index >> 16) & 0xFF))
		b.WriteByte(byte((index >> 8) & 0xFF))
		b.WriteByte(byte((index >> 0) & 0xFF))
	}

	params, results := funcSig.Params(), funcSig.Results()

	n := 4 * (params.Len() + results.Len())
	b.Grow(n)

	//inouts := make([]byte, n)
	//cursor := 0
	for i := params.Len() - 1; i >= 0; i-- {
		typeIndex := d.RegisterType(params.At(i).Type()).index
		//binary.LittleEndian.PutUint32(inouts[cursor:], typeIndex)
		//cursor += 4
		writeTypeIndex(typeIndex)
	}
	for i := results.Len() - 1; i >= 0; i-- {
		typeIndex := d.RegisterType(results.At(i).Type()).index
		//binary.LittleEndian.PutUint32(inouts[cursor:], typeIndex)
		//cursor += 4
		writeTypeIndex(typeIndex)
	}

	counts := params.Len()<<16 | results.Len()
	if funcSig.Variadic() {
		counts = -counts
	}

	//log.Println("?? full name:", funcObj.Id())
	//log.Println("   counts:   ", counts)
	//log.Println("   inouts:   ", inouts)

	return MethodSignature{
		//InOutTypes:          string(inouts),
		InOutTypes:          b.String(),
		NumInOutAndVariadic: counts,
		Name:                methodName,
		Pkg:                 pkgImportPath,
	}
}
