/* ----------------------------------------------------------------------
 * deformability scan
 * Copyright (C) 2000 January Weiner III
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 ---------------------------------------------------------------------- */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>

#include "genpak.h"
#include "gp_getopt.h"

#define VERSION "0.01"
#define PROGNAME "gp_deform"

char *progname ;

typedef struct {
	gzFile *in ;
	int window ;
	FILE *out ; } opt_s ;

static double params[4][4]={
	{2.9, 2.3, 2.1, 1.6}, 
	{9.8, 6.1, 12.1, 2.1},
	{4.5, 4.0, 6.1, 2.3},
	{6.3, 2.1, 2.3, 2.9}} ;

/*
 *
 */

int main(int argc, char *argv[])
{
	extern int optind ;
	extern char *optarg ;
	int width = 70 ; /* width with which the sequence gets formatted */
	opt_s options ;
	sekw *inseq, *outseq ;

	int c;
	char message[100] ;

	progname = argv[0] ;
	textdomain("genpak") ;

	options.window = 1 ;

	while ((c = gp_getopt(argc, argv, "w:Hqdvh")) != EOF)
		switch(c) {
		case 'w':
			if( sscanf(optarg, "%i", &options.window) < 1)
				gp_error("cannot read parameter ''%s'' for option -w", optarg) ;
			else 
				if(debug) gp_warn("window is %i", options.window) ;
			break ;
		case 'H':
			html = TRUE ;
			break ;
		case 'q':
			quiet = TRUE ;
			break ;
		case 'v':
			fprintf(stderr,_("%s version %s\n"),progname,VERSION) ;
			exit(EXIT_SUCCESS) ;
			break ;
		case 'd':
			debug = TRUE ;
			gp_warn(_("Running in debug mode")) ;
			break ;
		case 'h':
			Help() ;
			break ;
		default:
			gp_error(_("Type '%s -h' for help"),progname) ;
			break;
		}


/* open the file pointer to read the sequences 
 * from: standard input or a file provided? */
	if(optind >= argc) options.in = stdin ;
	else options.in = gp_file_open(argv[optind],"r") ;

/* opening the file pointer to write the output: 
 * standard output or file provided? */
	optind++ ;

	if(optind >= argc) options.out = stdout ;
	else options.out = gp_file_open(argv[optind],"wb") ;

	while( (inseq = gp_seq_read(options.in)) ) {
		if(debug) gp_warn("processing %s", inseq->name) ;
		deform(inseq, &options) ;

	}
	
	if(html) gp_warn_print_all(options.out) ;
	fclose(options.out) ;
	gzclose(options.in) ;
	return EXIT_SUCCESS ;
}


int deform(sekw *s, opt_s *o) {

	long l ;
	int i, j, n, c[128] ;
	double sum ;

	c['A'] = 0 ; c['C'] = 1 ; c['G'] = 2 ; c['T'] = 3 ; c['U'] = 3 ;

	if(o->window - 1 > s->leng) {
		gp_warn("sequence %s too short", s->name) ;
		return EXIT_FAILURE ;
	}

	fprintf(o->out, "%s\t", s->name) ;	
	for(l = 0 ; l < s->leng - 1 - o->window ; l++) {

		for(n = l, sum = 0 ; n < l + o->window ; n++) {
			i = c[s->sequ[n]] ;
			j = c[s->sequ[n + 1]] ;
			sum += params[i][j] ;
		}

		sum /= o->window ;

		fprintf(o->out, "%.2f\t", sum) ;
	}

	fprintf(o->out, "\n") ;




	return EXIT_SUCCESS ;
}

/* Standard mesage */
void Help()
{
	fprintf(stdout,_(""
	"\n"
	"%s, v. %s - deformability scan"
	"\n"
	"  Usage:\n"
	"     %s [options] [ input file ] [ output file ]\n"
	"\n"
	"  Options:\n"
	"     -w <value> : set the averaging window size to <value>\n"
	"     -v         : print version information & exit\n"
	"     -h         : print this help screen & exit\n"
	"     -q         : quiet, suppress error messages\n\n"),
	PROGNAME,VERSION,progname);
	exit(EXIT_SUCCESS);
}


			
