/* ----------------------------------------------------------------------
 * Frequencies of nucleotide trimers (AAA,AAC,AAG...TTT) in a DNA sequence
 * Copyright (C) 2000 January Weiner III
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 ---------------------------------------------------------------------- */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <time.h>

#include "genpak.h"
#include "gp_getopt.h"

#define VERSION "template file"
#define PROGNAME "gp_dimer"

char *progname ;

typedef struct {
	FILE *in ;
	FILE *out ; } opt_s ;

/*
 *
 */

int GetFtable(sekw* s, double ***t,opt_s o) ;
int PrintFtable(double ***t, sekw* s,opt_s o) ;
int ZeroFtable(double ***t) ;
double ***AllocateTable(int i, int j, int k) ;

int main(int argc, char *argv[])
{
	extern int optind ;
	extern char *optarg ;
	int width = 70 ; /* width with which the sequence gets formatted */
	opt_s options ;
	sekw *inseq, *outseq ;
	double ***ftable ; /* here we store the frequencies */

	int c;
	char message[100] ;
	int errflg = 0 ;

	progname = argv[0] ;

	while ((c = gp_getopt(argc, argv, "Hqdvh")) != EOF)
		switch(c) {
		case 'H':
			html = TRUE ;
			break ;
		case 'q':
			quiet = TRUE ;
			break ;
		case 'v':
			fprintf(stderr,"%s version %s\n",progname,VERSION) ;
			exit(0) ;
			break ;
		case 'd':
			debug = TRUE ;
			gp_warn("Running in debug mode") ;
			break ;
		case 'h':
			Help() ;
			break ;
		default:
			errflg++ ;
			break;
		}


	if(errflg) gp_error("Type '%s -h' for help",progname) ;

/* open the file pointer to read the sequences 
 * from: standard input or a file provided? */
	if(optind >= argc) options.in = stdin ;
	else options.in = gp_file_open(argv[optind],"r") ;

/* opening the file pointer to write the output: 
 * standard output or file provided? */
	optind++ ;

	if(optind >= argc) options.out = stdout ;
	else options.out = gp_file_open(argv[optind],"wb") ;

	ftable = AllocateTable(4,4,4) ;
	PrintHeader(options) ;

	while((inseq = gp_seq_read_fragment(options.in,0,0,0)) != NULL) {

		ZeroFtable(ftable) ;
		GetFtable(inseq,ftable,options) ;
		PrintFtable(ftable,inseq,options) ;


	}
	
	if(html) gp_warn_print_all(options.out) ;
	fclose(options.out) ;
	fclose(options.in) ;
	return(0);
}


/* Prepends output with a header */

int PrintHeader(opt_s o) {

	int i,j,k ;
	char C[4] = { 'A','C','G','T'} ;

	fprintf(o.out,"Name\t") ;

	for(i = 0;i<4;i++)
		for(j = 0;j<4;j++)
			for(k = 0;k<4;k++)
			fprintf(o.out,"%c%c%c\t",C[i],C[j],C[k]) ;

	fprintf(o.out,"\n") ;


	return EXIT_SUCCESS ;

}


/* Zeroes the frequency table */

int ZeroFtable(double ***t) {

	int i,j,k ;

	for(i = 0;i<4;i++)
		for(j = 0;j<4;j++)
			for(k = 0;k<4;k++)
				t[i][j][k] = 0.0 ;

	return EXIT_SUCCESS ;

}


/* Allocates a table of i x j elements */

double ***AllocateTable(int i, int j, int k) {


	int n,m ;
	double ***ret ;

	ret = calloc(i,sizeof(*ret)) ;
	for(n = 0;n<i;n++) {
		ret[n] = calloc(j,sizeof(*ret[n])) ;
		for(m = 0;m<j;m++)
			ret[n][m] = calloc(k,sizeof(*ret[n][m])) ;
	}

	return(ret) ;

}



/* Prints the frequency table */
int PrintFtable(double ***t, sekw* s, opt_s o) {
	int i,j,k ;

	fprintf(o.out,"%s\t",s->name) ;

	for(i = 0;i<4;i++)
		for(j = 0;j<4;j++)
			for(k = 0;k<4;k++)
				fprintf(o.out,"%.0f\t",t[i][j][k]) ;
	fprintf(o.out,"\n") ;
		

	return EXIT_SUCCESS ;

}


/* counts the dimers and stores information in the frequency table */
int GetFtable(sekw *s, double ***t,opt_s o) {

	int i,j,k ;
	long p ;
	char c1,c2,c3 ;
	
	int C[128] ;

	C['A'] = 0 ; C['C'] = 1 ; C['G'] = 2 ; C['T'] = 3 ; C['U'] = 3 ;

	for(p = 0;p<s->leng-2;p++) {
		c1= toupper(s->sequ[p]) ;
		c2= toupper(s->sequ[p+1]) ;
		c3= toupper(s->sequ[p+2]) ;
		i = C[c1] ; j = C[c2] ; k = C[c3] ;
		t[i][j][k]++ ;

	}

	return EXIT_SUCCESS ;


}

/* Standard mesage */

void Help()
{
printf("\n");
printf("%s, v. %s- count frequencies of nucleotide triplets",PROGNAME,VERSION);
printf("\n");
printf("  Usage:\n");
printf("     %s [options] [ input file ] [ output file ]\n",progname);
printf("\n");
printf("  Options:\n");
printf("     -v       : print version information & exit\n");
printf("     -h       : print this help screen & exit\n");
printf("     -q       : quiet, suppress error messages\n\n");
exit(0);
}


			
