//Copyright (C) 2000 - 2003 Thomas Schank 
// 
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or (at your option) any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA. 

#include "protocol.h"

namespace gpspoint2 {
using namespace std;
using namespace gpspoint2;


Protocol::Protocol(void)
{
   garmin_waypoint = NULL;
   garmin_trackheader = NULL;
   garmin_trackpoint = NULL;
   garmin_routeheader = NULL;
   garmin_datetime = NULL;

   clear();
}

void Protocol::clear(void)
{
   linkProtocol=0, cmndProtocol=0,
   wptProtocol=0, wptDataProtocol=0,
   rteProtocol=0, rteHeaderProtocol=0,rteDataProtocol=0,
   trkProtocol=0, trkHeaderProtocol=0,trkDataProtocol=0,
   prxProtocol=0, prxDataProtocol=0,
   almProtocol=0, almDataProtocol=0;

   dateTimeTransfer= false;
   waypointTransfer= false;
   trackTransfer= false;
   trackHeader= false;
   
   if(garmin_datetime!=NULL) delete garmin_datetime, garmin_datetime = NULL;
   if(garmin_waypoint!=NULL) delete garmin_waypoint, garmin_waypoint = NULL;
   if(garmin_routeheader!=NULL) delete garmin_routeheader, garmin_routeheader = NULL;
   if(garmin_trackheader!=NULL) delete garmin_trackheader, garmin_trackheader= NULL;
   if(garmin_trackpoint!=NULL) delete garmin_trackpoint, garmin_trackpoint= NULL;
}


int Protocol::openPort(void)
{
   int err = Link::openPort();   
   if(err<0) return err;

   Packet p_id_req;
   p_id_req.pid=254;
   p_id_req.length=2;

   sendPacket(p_id_req);

   Packet p_id;
   getPacket(p_id);

   pdt.process(p_id);

   product_ID = pdt.product_ID();
   softwareVersion = pdt.softwareVersion();
   productDescription = pdt.productDescription();

#ifdef HAVE_WINDOWS_H
   setCapabilitiesByTable();

   if(linkProtocol==0){
      Packet p_capabil;
      if(getPacket(p_capabil)>0)
         setCapabilitiesByPacket(p_capabil);}
#endif 


#ifdef HAVE_TERMIOS_H
   Packet p_capabil;

   if(getPacket(p_capabil)>0)
      setCapabilitiesByPacket(p_capabil);
   else // use database
      setCapabilitiesByTable();
#endif

// ******************************************************* set Data Types

   if(linkProtocol) {
      garmin_datetime = new D600_Date_Time_Type;   
      dateTimeTransfer = true; }

   switch(wptDataProtocol) {   // Set Waypoint Data Type 
      case 103:
         garmin_waypoint = new D103_Wpt_Type;
         waypointTransfer = true;
         break;
      case 108:
         garmin_waypoint = new D108_Wpt_Type;
         waypointTransfer = true;
         break;
      case 109:
         garmin_waypoint = new D108_Wpt_Type;
         waypointTransfer = true;
         break;
      default:
         garmin_waypoint = new Wpt_Type;
         waypointTransfer= false;
         break; }

   switch(rteProtocol){
      case 200:
         sendRouteLinkData = false;
         break;
      case 201:
         sendRouteLinkData = true;
         break;
      default:
         sendRouteLinkData = false;
         break;}
   
   switch(rteHeaderProtocol) {
      case 200:
         garmin_routeheader = new D200_Rte_Hdr_Type;
         routeHeaderTransfer = true;
         break;
      case 201:
         garmin_routeheader = new D201_Rte_Hdr_Type;
         routeHeaderTransfer = true;
         break;
      case 202:
         garmin_routeheader = new D202_Rte_Hdr_Type;
         routeHeaderTransfer = true;
         break;
      default:
         garmin_routeheader= new Rte_Hdr_Type;
         routeHeaderTransfer = false; 
         break; }

   switch(trkHeaderProtocol) {   // Track Header
      case 310:
         garmin_trackheader= new D310_Trk_Hdr_Type;
         trackHeader=true;
         break;
      default:
         garmin_trackheader= new Trk_Hdr_Type;
         trackHeader=false;
         break; }

   switch(trkDataProtocol) {
      case 300:
         garmin_trackpoint = new D300_Trk_Point_Type;
         trackTransfer = true;
         break;
      case 301:
         garmin_trackpoint = new D301_Trk_Point_Type;
         trackTransfer = true;
         break;
      default:
         garmin_trackpoint = new Trk_Point_Type;
         trackTransfer = false;
         break; }

   return 1;

};



void Protocol::setCapabilitiesByTable(void)
{
//   #include "garmin_units_array.h"
	
	extern g_word  gpsunit[100][15];
   
   for(int i=0; i<100 && gpsunit[i][0]!=0; i++)
   {
      if( (gpsunit[i][0]==product_ID  )
         && (gpsunit[i][1]==0 || softwareVersion >= gpsunit[i][1]))
      {
         linkProtocol = gpsunit[i][2];
         cmndProtocol = gpsunit[i][3];

         wptProtocol= gpsunit[i][4];
         wptDataProtocol = gpsunit[i][5];
 
         rteProtocol = gpsunit[i][6];
         rteHeaderProtocol = gpsunit[i][7];
         rteDataProtocol = gpsunit[i][8];

         trkProtocol = gpsunit[i][9];
         trkHeaderProtocol = 0;
         trkDataProtocol = gpsunit[i][10];
         
         prxProtocol = gpsunit[i][11];
         prxDataProtocol = gpsunit[i][12];

         almProtocol = gpsunit[i][13];
         almDataProtocol = gpsunit[i][14];
      }
   }

}


void Protocol::setCapabilitiesByPacket(Packet p)
{

   struct Protocol_Data_Type
   {
      g_byte   tag;
      g_word   data;
   };

   Protocol_Data_Type   pdt[80];

   for (unsigned int i=0; i<80 && i<(p.length*3) ; i++) {
      pdt[i].tag=p.data[i*3];
      pdt[i].data=p.data[i*3+1]+256*p.data[i*3+2];}
      // 256 * should take care of the endianess problem here


   for(int i=0;i<80;i++)
   {
      if( pdt[i].tag=='L' && pdt[i].data==(1)) linkProtocol=pdt[i].data;
      if( pdt[i].tag=='L' && pdt[i].data==(2)) linkProtocol=pdt[i].data;
      if( pdt[i].tag=='A' && pdt[i].data==(10)) cmndProtocol=pdt[i].data;
      if( pdt[i].tag=='A' && pdt[i].data==(11)) cmndProtocol=pdt[i].data;

      if( pdt[i].tag=='A' && pdt[i].data>=100 && pdt[i].data<=199) { // Waypoints
         wptProtocol=pdt[i].data;
         wptDataProtocol=pdt[i+1].data; }

      if( pdt[i].tag=='A' && pdt[i].data>=200 && pdt[i].data<=299) { // Route
         rteProtocol=pdt[i].data;
         rteHeaderProtocol=pdt[i+1].data;
         rteDataProtocol=pdt[i+2].data;   }

      if( pdt[i].tag=='A' && pdt[i].data>=300 && pdt[i].data<=300) { // Track 300
         trkProtocol=pdt[i].data;
         trkDataProtocol=pdt[i+1].data; }

      if( pdt[i].tag=='A' && pdt[i].data>=301 && pdt[i].data<=399) {// Track >300
         trkProtocol=pdt[i].data;
         trkHeaderProtocol=pdt[i+1].data;
         trkDataProtocol=pdt[i+2].data;   }

      if( pdt[i].tag=='A' && pdt[i].data>=400 && pdt[i].data<=499) { // Proximity
         prxProtocol=pdt[i].data;
         prxDataProtocol=pdt[i+1].data; }

      if( pdt[i].tag=='A' && pdt[i].data>=500 && pdt[i].data<=599 ){// Almanac
         almProtocol=pdt[i].data;
         almDataProtocol=pdt[i+1].data; }
   }
}

const DeviceInfo *Protocol::getInfo() {
  static DeviceInfo *devinfo = new DeviceInfo();
  
  if(openPort() >= 0) {
    *devinfo = DeviceInfo(pdt, product_ID, softwareVersion, productDescription,
           waypointTransfer,
           routeHeaderTransfer && waypointTransfer,
           trackTransfer, linkProtocol, cmndProtocol,
           wptProtocol, wptDataProtocol, rteProtocol,
           rteHeaderProtocol, rteDataProtocol, trkProtocol,
           trkHeaderProtocol, trkDataProtocol, prxProtocol,
           prxDataProtocol, almProtocol, almDataProtocol);
    return devinfo;
  }
  closePort();
  return 0;
}


void Protocol::printTest(void)
{
   
   cerr    << endl;
   cerr  << "\tProduct ID:           "   <<  product_ID << endl;
   cerr  << "\tSoftware Version:     "   <<  softwareVersion << endl;
   cerr  << "\tProduct Descpription: " <<  productDescription << endl;
   cerr  << endl;
   cerr  << "\tLink Protokol:          " << linkProtocol << endl;
   cerr  << "\tCommand Protokol:       " << cmndProtocol << endl;
   cerr  << "\tWaypoint Protokol:      " << wptProtocol << endl;
   cerr  << "\tWaypoint Data Protocol: " << wptDataProtocol << endl;
   cerr  << "\tRoute Protokol:         " << rteProtocol << endl;
   cerr  << "\tRoute Data Protocol:    " << rteDataProtocol << endl;
   cerr  << "\tTrack Protokol:         " << trkProtocol << endl;
   cerr  << "\tTrack Header Protocol:  " << trkHeaderProtocol<< endl;
   cerr  << "\tTrack Data Protocol:    " << trkDataProtocol << endl;
   cerr  << "\tAlmanac Protokol:       " << almProtocol << endl;
   cerr  << "\tAlmanac Data Protocol:  " << almDataProtocol << endl;
   cerr  << endl;

   cerr  << "\tWaypoint Up- & Download : " ; 
      if(waypointTransfer)  cerr << "YES" ; else cerr << "NO" ; cerr << endl;
   cerr  << "\tRoute Up- & Download : " ; 
      if(routeHeaderTransfer && waypointTransfer)  cerr << "YES" ; else cerr << "NO" ; cerr << endl;
   cerr  << "\tTrack Up- & Download : " ; 
      if(trackTransfer)  cerr << "YES" ; else cerr << "NO" ; cerr << endl;
   cerr  << endl;
}

}
