//==============================================================================================*/
//  vector.h                                                                       Font3D       */
//----------------------------------------------------------------------------------------------*/
//                                                                                              */
//  Copyright (c) 1994-1996 by Todd A. Prater                                   Version 1.60    */
//  All rights reserved.                                                                        */
//                                                                                              */
//----------------------------------------------------------------------------------------------
//
//  Permission to copy and distribute Font3D in its entirety, for noncommercial purposes,
//  is hereby granted without fee, provided that this license information and copyright 
//  notice appear in all copies. 
//
//  If you redistribute Font3D, the entire contents of this distribution must be distributed,
//  including the readme.txt, and register.txt files, and the complete set of documentation,
//  both ASCII text, and PostScript files. 
//
//  The software may be modified for your own purposes, but modified versions may not be
//  distributed without prior consent of the author.
//
//  This software is provided 'asis', without any express or implied warranty.  In no event
//  will the author be held liable for any damages arising from the use of this software.  
//
//  If you would like to do something with Font3D that this copyright prohibits (such as 
//  distributing it with a commercial product, using portions of the source in some other
//  program, distributing registered copies, etc.), please contact the author (preferably
//  via email).  Arrangements can probably be worked out.
//
//----------------------------------------------------------------------------------------------*/
//                                                                                              */
//   A vector class.                                                                            */
//                                                                                              */
//   Constructors:                                                                              */
//                                                                                              */
//      vector(void)                                                                            */
//      vector(double,double,double)                                                            */
//      vector(vector&)                                                                         */
//                                                                                              */
//   Member Operators:                                                                          */
//                                                                                              */
//      +=     Assign Sum                                                                       */
//      -=     Assign Difference                                                                */
//      *=     Assign Product                                                                   */
//      /=     Assign Quotient                                                                  */
//                                                                                              */
//   Non-Member Operators:                                                                      */
//                                                                                              */
//      +      Sum            Binary      (vector + vector) -> vector                           */
//      -      Difference     Binary      (vector - vector) -> vector                           */
//      *      Scale          Binary      (vector * double) -> vector                           */
//      *      Scale          Binary      (double * vector) -> vector                           */
//      /      Scale          Binary      (vector / double) -> vector                           */
//      %      Dot Product    Binary      (vector % vector) -> double                           */
//      ^      Cross Product  Binary      (vector ^ vector) -> vector                           */
//      ~      Normalize      Unary       (~vector)         -> vector                           */
//      -      Negative       Unary       (-vector)         -> vector                           */
//                                                                                              */
//      ==     Equal to       Binary      (vector == vector) -> INT                             */
//      !=     Not equal to   Binary      (vector !- vector) -> INT                             */
//                                                                                              */
//      >>     Insertion      Binary      (ostream& >> vector&) -> ostream&                     */
//                                                                                              */
//   Non-Member Functions:                                                                      */
//                                                                                              */
//      dist(vector x, vector y) ...... Returns the distance between x and y.                   */
//      midpoint(vector x, vector y) .. Returns the midpoint between x and y.                   */
//      length(vector x) .............. Returns the length of x.                                */
//                                                                                              */
//                                                                                              */
//==============================================================================================*/

#ifndef __VECTOR_H__
#define __VECTOR_H__

#include <math.h>
#include <iostream.h>

   class vector
   {
     public:  double  x,y,z;

              vector ();
              vector (double _x, double _y, double _z);
              vector (const vector& v);

              vector& operator += (vector v);
              vector& operator -= (vector v);
              vector& operator *= (double f);
              vector& operator /= (double f);
   };


   /*_____ CONSTRUCTORS _____*/


   inline vector::vector(void)
   {
     x=0;y=0;z=0;
   }

   inline vector::vector(double _x, double _y, double _z)
   {
     x=_x;y=_y;z=_z;
   }

   inline vector::vector(const vector& v)
   {
     x=v.x;y=v.y;z=v.z;
   }


   /*_____ NON-MEMBER FUNCTIONS _____*/


   inline vector operator + (vector v1, vector v2)
   {
     return vector(v1.x+v2.x,v1.y+v2.y,v1.z+v2.z);
   }

   inline vector operator - (vector v1, vector v2)
   {
     return vector(v1.x-v2.x,v1.y-v2.y,v1.z-v2.z);
   }

   inline vector operator * (vector v, double f)
   {
     return vector(v.x*f,v.y*f,v.z*f);
   }

   inline vector operator * (double f, vector v)
   {
     return vector(v.x*f,v.y*f,v.z*f);
   }

   inline vector operator / (vector v, double f)
   {
     return vector(v.x/f,v.y/f,v.z/f);
   }

   inline double operator % (vector v1, vector v2)
   {
     return (v1.x*v2.x+v1.y*v2.y+v1.z*v2.z);
   }

   inline vector operator ^ (vector v1, vector v2)
   {
     double  tx = v1.y*v2.z - v1.z*v2.y;
     double  ty = v1.z*v2.x - v1.x*v2.z;
     double  tz = v1.x*v2.y - v1.y*v2.x;
     return  vector(tx,ty,tz);
   }

   inline vector operator ~ (vector v)
   {
     double  mag = sqrt(v.x*v.x + v.y*v.y + v.z*v.z);
     double  tx = v.x/mag;
     double  ty = v.y/mag;
     double  tz = v.z/mag;
     return  vector(tx,ty,tz);
   }

   inline vector operator - (vector v)
   {
     return vector(-v.x,-v.y,-v.z);
   }

   inline int operator == (vector v1, vector v2)
   {
     return (v1.x==v2.x&&v1.y==v2.y&&v1.z==v2.z);
   }

   inline int operator != (vector v1, vector v2)
   {
     return (v1.x!=v2.x&&v1.y!=v2.y&&v1.z!=v2.z);
   }


   inline double dist (vector v1, vector v2)
   {
     vector d = v2-v1;
     return (sqrt(d%d));
   }

   inline vector midpoint (vector v1, vector v2)
   {
      return ((v1+v2)/2);
   }

   inline double length (vector v)
   {
      return (sqrt(v.x*v.x+v.y*v.y+v.z*v.z));
   }

   /*_____ MEMBER SHORTHAND OPERATORS _____*/

   inline vector& vector::operator += (vector v)
   {
     x+=v.x; y+=v.y; z+=v.z;
     return *this;
   }

   inline vector& vector::operator -= (vector v)
   {
     x-=v.x; y-=v.y; z-=v.z;
     return *this;
   }

   inline vector& vector::operator *= (double f)
   {
     x*=f; y*=f; z*=f;
     return *this;
   }

   inline vector& vector::operator /= (double f)
   {
     x/=f; y/=f; z/=f;
     return *this;
   }

   inline ostream& operator << (ostream& s, vector v)
   {
      s<<"<"<<v.x<<","<<v.y<<","<<v.z<<">";
      return s;
   }


#endif
