/*
 * ksi.c
 * stand alone ksi interpreter
 *
 * Note that this file can be linked with GNU readline library and so,
 * it is released under GPL as required by GNU readline library.
 *
 * Copyright (C) 1997-2010, ivan demakov
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        Ivan Demakov <ksion@users.sourceforge.net>
 * Creation date: Sun Aug 17 20:55:20 1997
 *
 */

#include <ksi_type.h>
#include <ksi_int.h>
#include <ksi_printf.h>
#include <ksi_util.h>

#include <signal.h>

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#endif

#ifdef _MSC_VER
#  include <io.h>
#endif


#if defined(WIN32) || defined(OS2) || defined(MSDOS) || defined(__CYGWIN32__) || defined(__CYGWIN__)

#  define EXE_SUFFIX ".exe"
#  define IS_DIR(x)  ((x) == '/' || (x) == '\\' || (x) == ':')

#else

#  define EXE_SUFFIX ".static"
#  define IS_DIR(x)  ((x) == '/')

#endif


struct Ksi
{
    int interactive, booted;
    ksi_obj input, output, error;
    char *app_name, *prompt;

    ksi_env top_level_env;
    ksi_obj intr_prim, intr_evt, read_prim, write_prim;
    int do_intr;

#if HAVE_LIBREADLINE
    ksi_obj complit_list;
    char *history_file;
    int history_size;
    char *readline;
#endif
};


static struct Ksi *ksi;


#if HAVE_LIBREADLINE

typedef int Function ();

extern char *readline (char *prompt);
extern int  rl_parse_and_bind (char *line);
extern int  rl_variable_bind (char *var, char *val);

extern void add_history (char *line);
extern int  read_history (char *filename);
extern int  write_history (char *filename);
extern int  append_history (int nelements, char *filename);
extern void *remove_history (int which);
extern int  where_history ();
extern int  history_search (char *string, int direction);
extern int  history_truncate_file (char *filename, int nlines);
extern void stifle_history (int max);

extern char *rl_readline_name;
extern char *rl_basic_word_break_characters;
extern char *rl_basic_quote_characters;
extern char *rl_completer_word_break_characters;
extern char *rl_completer_quote_characters;
extern Function *rl_completion_entry_function;
extern Function *rl_event_hook;


static const char*
rl_name (ksi_port x)
{
    return "readline";
}

static int
rl_close (ksi_port x)
{
    return 0;
}

static int
rl_read (ksi_port x, char *buf, int num)
{
    int i;

    while (ksi->readline == 0) {
        ksi->readline = readline (ksi->prompt);
        if (ksi->readline == 0)
            return 0;

        if (ksi->readline[0] == 0) {
            free (ksi->readline);
            ksi->readline = 0;
        } else {
            add_history (ksi->readline);
        }
    }

    for (i = 0; i < num; i++) {
        if (ksi->readline[i] != 0) {
            buf[i] = ksi->readline[i];
        } else {
            free (ksi->readline);
            ksi->readline = 0;
            buf[i++] = '\n';
            break;
        }
    }

    if (ksi->readline) {
        num = strlen (ksi->readline) - i;
        memmove (ksi->readline, ksi->readline + i, num + 1);
    }

    return i;
}

static int
rl_write (ksi_port x, const char* buf, int num)
{
    return num;
}

static int
rl_ready (ksi_port x)
{
    return 1;
}

static int
rl_flush (ksi_port x)
{
    return 0;
}

static int
rl_fd (ksi_port x)
{
    return -1;
}

static struct Ksi_Port_Ops rl_port_ops =
{
    rl_name,
    rl_close,
    rl_read,
    rl_write,
    rl_ready,
    rl_ready,
    rl_flush,
    rl_fd,
    rl_fd
};

static ksi_obj
make_rl_port (void)
{
    struct Ksi_Port *port;
    port = (struct Ksi_Port*) ksi_malloc (sizeof *port);
    port->o.itag = KSI_TAG_PORT;
    port->ops = &rl_port_ops;
    port->input = 1;
    return (ksi_obj) port;
}


static char*
ksi_compliter (char *text, int num)
{
    if (!text)
        text = "";

    if (num == 0)
        ksi->complit_list = ksi_abbrev (text, strlen (text));

    if (ksi->complit_list != ksi_nil) {
        char *ptr = ksi_aprintf ("%s", ksi_obj2str(KSI_CAR (ksi->complit_list)));
        int len = strlen (ptr);
        ksi->complit_list = KSI_CDR (ksi->complit_list);

        if (ptr[0] == '#' && ptr[1] == '|') {
            ptr[len - 1] = '\0';
            ptr += 2;
        } else if (ptr[0] == '#' && ptr[1] == ':' && ptr[2] == '|') {
            ptr[len - 1] = '\0';
            ptr += 3;
        }

        text = (char*) malloc (len + 1);
        memcpy (text, ptr, len + 1);
        return text;
    }

    return 0;
}

static int
idle_work ()
{
    if (ksi_wait_event (ksi_long2num(0)) == ksi_false)
        return ksi_gcollect (0);
    return 0;
}

static void
init_readline (void)
{
    /* Allow conditional parsing of the ~/.inputrc file. */
    rl_readline_name = ksi->app_name;

    rl_basic_word_break_characters = " \n\t()[]{}'`,;\"";
    rl_completion_entry_function = (Function*) ksi_compliter;
    rl_event_hook = idle_work;

    stifle_history (ksi->history_size);
    read_history (ksi->history_file);
}

#endif


static ksi_obj
intr_proc (ksi_obj evt)
{
    if (ksi->do_intr)
        ksi_quit ();
    return ksi_void;
}

static ksi_obj
exit_proc (void)
{
    char *ptr;

    if (ksi->interactive && ksi->booted) {
#if HAVE_LIBREADLINE
        write_history (ksi->history_file);
        history_truncate_file (ksi->history_file, ksi->history_size);
#endif
        ptr = ksi_aprintf ("; EXIT\n");
        ksi_port_write (ksi->error, ptr, strlen(ptr));
        ksi_flush_port (ksi->error);
    }
    return ksi_void;
}

static ksi_obj
read_proc (ksi_obj input)
{
    return ksi_read (input);
}

static ksi_obj
write_proc (ksi_obj val)
{
    const char *buf = ksi_obj2str(val);
    ksi_port_write(ksi->output, "\n", 1);
    ksi_port_write(ksi->output, buf, strlen(buf));
    ksi_port_write(ksi->output, "\n", 1);
    ksi_flush_port (ksi->output);
    return ksi_void;
}

static ksi_obj
error_proc (ksi_obj exn)
{
    if (KSI_EXN_P (exn)) {
        const char *msg;
        if (KSI_EXN_VAL (exn) != ksi_void) {
            if (KSI_EXN_SRC (exn) != ksi_void) {
                msg = ksi_aprintf("%s (errval: %s) at %s", KSI_STR_PTR (KSI_EXN_MSG (exn)), ksi_obj2str(KSI_EXN_VAL (exn)), ksi_obj2name(KSI_EXN_SRC (exn)));
            } else {
                msg = ksi_aprintf("%s (errval: %s)", KSI_STR_PTR (KSI_EXN_MSG (exn)), ksi_obj2str(KSI_EXN_VAL (exn)));
            }
        } else {
            if (KSI_EXN_SRC (exn) != ksi_void) {
                msg = ksi_aprintf("%s at %s", KSI_STR_PTR (KSI_EXN_MSG (exn)), ksi_obj2name(KSI_EXN_SRC (exn)));
            } else {
                msg = KSI_STR_PTR (KSI_EXN_MSG (exn));
            }
        }
        ksi_errlog_msg (ERRLOG_ERROR, msg);
    }
    return ksi_void;
}

static void
repl_proc (void)
{
    double t1 = 0.0, t2 = 0.0, t3 = 0.0, t4 = 0.0, t5 = 0.0, t6 = 0.0;
    ksi_obj val;
    const char *buf;

    for (;;) {
#if !HAVE_LIBREADLINE
        if (ksi->interactive) {
            ksi_port_write (ksi->output, ksi->prompt, strlen(ksi->prompt));
            ksi_flush_port (ksi->output);
        }
#endif

        ksi->do_intr = 0;
        val = ksi_apply_1_with_catch (ksi->read_prim, ksi->input);
        ksi->do_intr = 1;

        if (val == ksi_eof)
            ksi_exit(0);

        if (KSI_EXN_P (val)) {
            val = error_proc (val);
        } else {
            if (ksi->interactive) {
                t5 = ksi_real_time ();
                t3 = ksi_eval_time ();
                t1 = ksi_cpu_time ();
            }

            val = ksi_eval_with_catch (val, ksi->top_level_env);

            if (KSI_EXN_P (val)) {
                val = error_proc (val);
            }

            if (ksi->interactive) {
                t2 = ksi_cpu_time ();
                t4 = ksi_eval_time ();
                t6 = ksi_real_time ();

                if (val != ksi_void) {
                    val = ksi_apply_1_with_catch (ksi->write_prim, val);
                    if (KSI_EXN_P (val)) {
                        val = error_proc (val);
                    }
                }

                buf = ksi_aprintf ("\n; cpu/eval/real time: %.4f/%.4f/%.4f sec\n; heap/free size: %d/%d bytes\n",
                                   t2-t1, t4-t3, t6-t5,
                                   ksi_get_heap_size (), ksi_get_heap_free ());
                ksi_port_write (ksi->error, buf, strlen(buf));
                ksi_flush_port (ksi->error);
            }
        }
    }
}

static ksi_obj
ksi_interactive (ksi_obj x)
{
    int i = ksi->interactive;
    if (x)
        ksi->interactive = (x == ksi_false ? 0 : 1);
    return i ? ksi_true : ksi_false;
}

int
main (int argc, char* argv[])
{
    ksi_obj val;
    ksi_env env;
    char *ptr, *app;
    int i;

    ptr = argv [0];
    for (i = strlen (ptr); i > 0; --i) {
        if (IS_DIR (ptr[i-1])) {
            ptr += i;
            break;
        }
    }

    app = (char*) alloca (strlen (ptr) + 1);
    strcpy (app, ptr);
#ifdef EXE_SUFFIX
    if (ksi_has_suffix (app, EXE_SUFFIX))
        app [strlen (app) - strlen (EXE_SUFFIX)] = '\0';
#endif

    ksi_init(&argc);
    ksi_init_std_ports(0, 1, 2);

    ksi = (struct Ksi*) ksi_malloc_eternal(sizeof *ksi);
    ksi->interactive   = isatty(0);
    ksi->booted        = 0;
    ksi->input         = ksi_current_input_port();
    ksi->output        = ksi_current_output_port();
    ksi->error         = ksi_current_error_port();
    ksi->app_name      = app;
    ksi->prompt        = ksi_aprintf("%s> ", app);
    ksi->top_level_env = ksi_top_level_env();

    env = ksi_get_lib_env("ksi", "core", "system", 0);
    ksi_defun("interactive", ksi_interactive, KSI_CALL_ARG1, 0, env);
    ksi_defsym("application", ksi_str02string(app), env);

    /* build arg-list */
    val = ksi_nil;
    for (i = argc; --i > 0; )
        val = ksi_cons(ksi_str02string(argv[i]), val);
    val = ksi_cons(ksi_str02string(app), val);
    ksi_defsym("argv", val, env);

    ksi->read_prim = (ksi_obj) ksi_new_prim(0, read_proc, KSI_CALL_ARG1, 1);
    ksi->write_prim = (ksi_obj) ksi_new_prim(0, write_proc, KSI_CALL_ARG1, 1);

    if (ksi->interactive) {
        ksi->intr_prim = (ksi_obj) ksi_new_prim(0, intr_proc, KSI_CALL_ARG1, 1);
        ksi->intr_evt  = ksi_signal_event(ksi_long2num (SIGINT), ksi->intr_prim);

#if HAVE_LIBREADLINE
        ptr = getenv ("HISTSIZE");
        ksi->history_file = ksi_expand_file_name("~/.ksi_history");
        ksi->history_size = (ptr ? atoi(ptr) : 200);
        ksi->complit_list = ksi_nil;
        ksi->input        = make_rl_port();
        init_readline();
#endif

        ksi_add_exit_handler ((ksi_obj) ksi_new_prim(0, exit_proc, KSI_CALL_ARG0, 0));
    }

    if (argc >= 2 && strcmp(argv[1], "-b") == 0) {
        if (argc >= 3) {
            ksi_load_boot_file(argv[2], ksi->top_level_env);
        }
    } else {
        ksi_load_boot_file("Boot.scm", ksi->top_level_env);
    }

#ifndef WIN32
    if (ksi->intr_evt)
        ksi_start_event (ksi->intr_evt);
#endif

    ksi->booted = 1;
    repl_proc();
    ksi_term();
    exit(0);
}

/* End of code */
