/*
 * ksi_char.c
 * chars
 *
 * Copyright (C) 1997-2010, ivan demakov
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        ivan demakov <ksion@users.sourceforge.net>
 * Creation date: Fri Jan 24 23:11:51 1997
 * Last Update:   Fri Jan 22 19:59:09 2010
 *
 */

#include "ksi_int.h"
#include "ksi_printf.h"

#include <locale.h>


const char *ksi_char_names[] =
{
  "#\\nul","#\\soh","#\\stx","#\\etx","#\\eot","#\\enq","#\\ack","#\\bel",
  "#\\bs", "#\\ht", "#\\nl", "#\\vt", "#\\np", "#\\cr", "#\\so", "#\\si",
  "#\\dle","#\\dc1","#\\dc2","#\\dc3","#\\dc4","#\\nak","#\\syn","#\\etb",
  "#\\can", "#\\em","#\\sub","#\\esc", "#\\fs", "#\\gs", "#\\rs", "#\\us",
  "#\\space", "#\\newline", "#\\tab", "#\\backspace",
  "#\\return", "#\\page", "#\\escape", "#\\del",
  "#\\null", /* for end null char in char_nums */
  NULL
};

static unsigned char ksi_char_code[] =
    "\000\001\002\003\004\005\006\007"
    "\010\011\012\013\014\015\016\017"
    "\020\021\022\023\024\025\026\027"
    "\030\031\032\033\034\035\036\037"
    " \n\t\b\r\f\033\177";

#define KSI_CHAR_NUMS (sizeof (ksi_char_code))

ksi_obj
ksi_int2char (unsigned int code)
{
    ksi_char x = ksi_malloc(sizeof *x);
    x->o.itag = KSI_TAG_CHAR;
    x->code = code;
    return (ksi_obj) x;
}


ksi_obj
ksi_str2char (const char* name, int len)
{
    int i, j, n;

    if (len > 2 && name[0] == '#' && name[1] == '\\') {
        name += 2;
        len -= 2;
    }

    for (i = 0; i < KSI_CHAR_NUMS; ++i) {
        if (strlen (ksi_char_names[i]) == len + 2) {
            for (j = 0; j < len; j++)
                if (tolower (ksi_char_names[i][j+2]) != tolower (name[j]))
                    goto next;
            return ksi_int2char (ksi_char_code[i]);
        }
    next:
        ;
    }

    if (name[0] == 'o' || name[0] == '0') {
        for (n = i = 0; ++i < len; )
            if ('0' <= name[i] && name[i] <= '7')
                n = n * 8 + name[i] - '0';
            else
                return ksi_false;

        return ksi_int2char (n);
    }

    if (name[0] == 'x' || name[0] == 'h') {
        for (n = i = 0; ++i < len; )
            if ('0' <= name[i] && name[i] <= '9')
                n = n * 16 + name[i] - '0';
            else if ('a' <= name[i] && name[i] <= 'f')
                n = n * 16 + (name[i] - 'a' + 10);
            else if ('A' <= name[i] && name[i] <= 'F')
                n = n * 16 + (name[i] - 'A' + 10);
            else
                return ksi_false;

        return ksi_int2char (n);
    }

    return ksi_false;
}

const char*
ksi_char2str (ksi_char x)
{
    int i, c;
    char* ptr;

    c = KSI_CHAR_CODE(x);
    for (i = 0; i < KSI_CHAR_NUMS; ++i)
        if (ksi_char_code[i] == c)
            return ksi_char_names[i];

    if (isgraph(c)) {
        ptr = (char*) ksi_malloc_data(4);
        ptr[0] = '#'; ptr[1] = '\\'; ptr[2] = (char) c; ptr[3] = '\0';
    } else {
        ptr = ksi_aprintf("#\\x%02x", c);
    }

    return ptr;
}

ksi_obj
ksi_char_p (ksi_obj x)
{
    return KSI_CHAR_P(x) ? ksi_true : ksi_false;
}


#define CONCAT(a,b) a##b

#define DEF_COMP(op, test, name)                                        \
ksi_obj                                                                 \
CONCAT (ksi_char_, op) (int ac, ksi_obj* av)                            \
{                                                                       \
    int c1, c2, i;                                                      \
    if (ac < 1)                                                         \
        return ksi_true;                                                \
    KSI_CHECK (av[0], KSI_CHAR_P (av[0]), name ": invalid char");       \
    c1 = KSI_CHAR_CODE (av[0]);                                         \
    for (i = 1; i < ac; i++) {                                          \
        KSI_CHECK (av[i], KSI_CHAR_P (av[i]), name ": invalid char");   \
        c2 = KSI_CHAR_CODE (av[i]);                                     \
        if (!(test))                                                    \
            return ksi_false;                                           \
        c1 = c2;                                                        \
    }                                                                   \
    return ksi_true;                                                    \
}

#ifdef HAVE_STRCOLL
static inline int chrcmp (char c1, char c2)
{
    char s1[2], s2[2];
    s1[0] = c1; s1[1] = '\0';
    s2[0] = c2; s2[1] = '\0';
    return strcoll (s1, s2);
}

#  define CHRCMP(c1, op, c2) (chrcmp ((c1), (c2)) op 0)
#else
#  define CHRCMP(c1, op, c2) ((c1) op (c2))
#endif

DEF_COMP (eq_p, c1 == c2, "char=?")
DEF_COMP (lt_p, CHRCMP (c1, <, c2), "char<?")
DEF_COMP (gt_p, CHRCMP (c1, >, c2), "char>?")
DEF_COMP (le_p, CHRCMP (c1, <=, c2), "char<=?")
DEF_COMP (ge_p, CHRCMP (c1, >=, c2), "char>=?")
DEF_COMP (ci_eq_p, tolower (c1) == tolower (c2), "char=?")
DEF_COMP (ci_lt_p, CHRCMP (tolower (c1), <, tolower (c2)), "char-ci<?")
DEF_COMP (ci_gt_p, CHRCMP (tolower (c1), >, tolower (c2)), "char-ci>?")
DEF_COMP (ci_le_p, CHRCMP (tolower (c1), <=, tolower (c2)), "char-ci<=?")
DEF_COMP (ci_ge_p, CHRCMP (tolower (c1), >=, tolower (c2)), "char-ci>=?")


#define DEF_TEST(op, test, name)                                \
ksi_obj                                                         \
CONCAT (ksi_char_, op) (ksi_obj o)                              \
{                                                               \
    int c;                                                      \
    KSI_CHECK (o, KSI_CHAR_P (o), name ": invalid char");       \
    c = KSI_CHAR_CODE (o);                                      \
    if (test)                                                   \
        return ksi_true;                                        \
    return ksi_false;                                           \
}

DEF_TEST (digit_p, isdigit (c), "char-numeric?")
DEF_TEST (space_p, isspace (c), "char-whitespace?")
DEF_TEST (alpha_p, isalpha (c), "char-alphabetic?")
DEF_TEST (upper_p, isupper (c), "char-upper-case?")
DEF_TEST (lower_p, islower (c), "char-lower-case?")


ksi_obj
ksi_char2integer (ksi_obj x)
{
    KSI_CHECK(x, KSI_CHAR_P(x), "char->integer: invalid char");
    return ksi_long2num(KSI_CHAR_CODE(x));
}

ksi_obj
ksi_integer2char (ksi_obj x)
{
    unsigned i = 0;

    KSI_CHECK(x, ksi_integer_p(x) == ksi_true, "integer->char: invalid integer");
    i = ksi_num2ulong(x, "integer->char");
    return ksi_int2char(i);
}

ksi_obj
ksi_char_upcase (ksi_obj x)
{
    int c;
    KSI_CHECK(x, KSI_CHAR_P(x), "char-upcase: invalid char");
    c = KSI_CHAR_CODE(x);
    return ksi_int2char(toupper(c));
}

ksi_obj
ksi_char_downcase (ksi_obj x)
{
    int c;
    KSI_CHECK(x, KSI_CHAR_P(x), "char-downcase: invalid char");
    c = KSI_CHAR_CODE(x);
    return ksi_int2char(tolower(c));
}

ksi_obj
ksi_setlocale (ksi_obj category, ksi_obj locale)
{
    const char *p = 0;
    int c = 0;

    if (KSI_SYM_P(category))
        p = KSI_SYM_PTR(category);
    else if (KSI_KEY_P(category))
        p = KSI_KEY_PTR(category);
    else
        ksi_exn_error(ksi_assertion_s, category, "setlocale: invalid category in arg1");

    if (strcasecmp(p, "LC_ALL") == 0)
        c = LC_ALL;
    else if (strcasecmp(p, "LC_COLLATE") == 0)
        c = LC_COLLATE;
    else if (strcasecmp(p, "LC_CTYPE") == 0)
        c = LC_CTYPE;
    else if (strcasecmp(p, "LC_MONETARY") == 0)
        c = LC_MONETARY;
    else if (strcasecmp(p, "LC_NUMERIC") == 0)
        c = LC_NUMERIC;
    else if (strcasecmp(p, "LC_TIME") == 0)
        c = LC_TIME;
    else
        ksi_exn_error(ksi_assertion_s, category, "setlocale: invalid category in arg1");

    if (!locale)
        p = 0;
    else if (KSI_STR_P(locale))
        p = KSI_STR_PTR(locale);
    else
        ksi_exn_error(ksi_assertion_s, locale, "setlocale: invalid locale in arg2");

    p = setlocale(c, p);

    return (p ? ksi_str02string(p) : ksi_false);
}


 /* End of code */
