/*
 * ksi_port.c
 * ports
 *
 * Copyright (C) 1997-2010, ivan demakov
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        Ivan Demakov <ksion@users.sourceforge.net>
 * Creation date: Fri Jan 24 21:48:27 1997
 * Last Update:   Fri Jan 22 20:38:07 2010
 *
 */

#include "ksi_port.h"
#include "ksi_int.h"
#include "ksi_env.h"
#include "ksi_evt.h"
#include "ksi_klos.h"
#include "ksi_printf.h"


#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#ifdef HAVE_FCNTL_H
#  include <fcntl.h>
#endif

#ifdef _MSC_VER
#  include <io.h>
#endif

#ifdef HAVE_SYS_STAT_H
#  include <sys/stat.h>
#endif

#if defined(HAVE_SYS_POLL_H)
#  include <sys/poll.h>
#elif defined(HAVE_SYS_SELECT_H)
#  include <sys/select.h>
#endif

#ifndef F_OK
#  define F_OK 0
#endif

#ifndef O_ACCMODE
#  define O_ACCMODE (O_RDONLY | O_WRONLY | O_RDWR)
#endif

#ifndef EWOULDBLOCK
#  ifdef EAGAIN
#    define EWOULDBLOCK EAGAIN
#  endif
#endif


#define CONCAT(a,b) a##b
#define STR_BUFSZ 256
#define MAX_BUFSZ 4096


struct Ksi_FdEvent
{
  struct Ksi_Event evt;
  struct Ksi_FdPort *port;
  void *ed;
};


/* ---------- null port ---------------------------------------------- */

static const char*
nul_name (ksi_port x)
{
  return "null";
}

static int
nul_close (ksi_port x)
{
  return 0;
}

static int
nul_read (ksi_port x, char* buf, int max_len)
{
  return 0;
}

static int
nul_write (ksi_port x, const char* buf, int len)
{
  return len;
}

static int
nul_ready (ksi_port x)
{
  return 1;
}

static int
nul_flush (ksi_port x)
{
  return 0;
}

static int
nul_fd (ksi_port x)
{
  return -1;
}

static struct Ksi_Port_Ops nul_port_ops =
{
  nul_name,
  nul_close,
  nul_read,
  nul_write,
  nul_ready,
  nul_ready,
  nul_flush,
  nul_fd,
  nul_fd
};

ksi_port
ksi_new_nul_port (void)
{
  struct Ksi_FdPort *x = (struct Ksi_FdPort*) ksi_malloc (sizeof *x);
  bzero (x, sizeof *x);

  x->kp.o.itag            = KSI_TAG_PORT;
  x->kp.ops             = &nul_port_ops;
  x->kp.input           = 1;
  x->kp.output          = 1;
  x->fd                 = -1;

  return (ksi_port) x;
}


/* ---------- string port -------------------------------------------- */

static const char*
str_name (ksi_port x)
{
  struct Ksi_StringPort* port = (struct Ksi_StringPort*) x;
  int i = 0, pos = 0;
  char buf[40];

  while (pos < sizeof (buf) - 3) {
    int c = KSI_STR_PTR (port->str)[i];
    switch (c) {
    case '\n': buf[pos++] = '\\'; c = 'n'; break;
    case '\r': buf[pos++] = '\\'; c = 'r'; break;
    case '\b': buf[pos++] = '\\'; c = 'b'; break;
    case '\t': buf[pos++] = '\\'; c = 't'; break;
    case '\f': buf[pos++] = '\\'; c = 'f'; break;
    case '\a': buf[pos++] = '\\'; c = 'a'; break;
    case '\v': buf[pos++] = '\\'; c = 'v'; break;
    }
    if (c < ' ')
      break;
    buf[pos++] = c;
    i++;
  }
  buf[pos] = '\0';

  return ksi_aprintf ("string %s", buf);
}

static int
str_close (ksi_port x)
{
  struct Ksi_StringPort* port = (struct Ksi_StringPort*) x;
  port->kp.closed = 1;
  port->kp.input  = 0;
  port->kp.output = 0;
  return 0;
}

static int
str_read (ksi_port x, char* buf, int num)
{
  struct Ksi_StringPort* port = (struct Ksi_StringPort*) x;
  char* ptr = KSI_STR_PTR (port->str);
  int i = 0, len = KSI_STR_LEN (port->str);

  while (i < num && port->pos < len)
    buf[i++] = ptr[port->pos++];

  return i;
}

static int
str_write (ksi_port x, const char* buf, int num)
{
  struct Ksi_StringPort* port = (struct Ksi_StringPort*) x;
  char* ptr = KSI_STR_PTR (port->str);
  int i = 0, len = KSI_STR_LEN (port->str);

  if (port->size == 0 || port->pos + num >= port->size) {
    int sz = (port->pos + num + STR_BUFSZ) & ~(STR_BUFSZ-1);
    char* p = (char*) ksi_malloc_data (sz);
    if (len) memcpy (p, ptr, len);

    port->size = sz;
    KSI_STR_PTR (port->str) = ptr = p;
  }

  while (i < num)
    ptr[port->pos++] = buf[i++];
  ptr[port->pos] = '\0';
  KSI_STR_LEN (port->str) = port->pos;
  return i;
}

static struct Ksi_Port_Ops str_port_ops =
{
  str_name,
  str_close,
  str_read,
  str_write,
  nul_ready,
  nul_ready,
  nul_flush,
  nul_fd,
  nul_fd
};

ksi_port
ksi_new_str_port (ksi_string str)
{
    struct Ksi_StringPort* port;

    if (str)
        KSI_CHECK((ksi_obj)str, KSI_STR_P(str), "ksi_new_str_port: invalid string");
    else
        str = (ksi_string) ksi_make_string(0, 0);

    port = ksi_malloc(sizeof *port);
    port->kp.o.itag = KSI_TAG_PORT;
    port->kp.ops = &str_port_ops;
    port->kp.last_write_char = '\n';
    port->str = (ksi_string) str;
    port->size = KSI_STR_LEN(str);

    return (ksi_port) port;
}


/* ---------- file port ---------------------------------------------- */

#define FILE_P(x) (KSI_PORT_P (x) && (((ksi_port) x) -> is_fd))

#ifdef WIN32
#  define fd_close(p)     (p->close ? p->close(p)     : close(p->fd))
#  define fd_read(p,b,l)  (p->read  ? p->read(p,b,l)  : read(p->fd,b,l))
#  define fd_write(p,b,l) (p->write ? p->write(p,b,l) : write(p->fd,b,l))
#else
#  define fd_close(p)     close(p->fd)
#  define fd_read(p,b,l)  read(p->fd,b,l)
#  define fd_write(p,b,l) write(p->fd,b,l)
#endif

static int
file_set_async (char *proc, struct Ksi_FdPort *port, int async)
{
#if defined(WIN32)
  {
    if (port->set_async == 0) {
      if (proc)
        ksi_exn_error ("i/o", (ksi_obj) port, "%s: async mode not supported", proc);
      return -1;
    }
    if (port->set_async (port, async)) {
      if (proc)
        ksi_exn_error ("i/o", (ksi_obj) port, "%s: %s", proc, port->error (port));
      return -1;
    }

    return 0;
  }
#elif defined(O_NONBLOCK)
  {
    int flg = fcntl (port->fd, F_GETFL, 0);
    if (flg == -1) {
      if (proc)
        ksi_exn_error ("i/o", (ksi_obj) port, "%s: %m", proc);
      return -1;
    }

    if (async)
      flg |= O_NONBLOCK;
    else
      flg &= ~O_NONBLOCK;

    if (fcntl (port->fd, F_SETFL, flg) == -1) {
      if (proc)
        ksi_exn_error ("i/o", (ksi_obj) port, "%s: %m", proc);
      return -1;
    }

    return 0;
  }
#else
  if (proc)
    ksi_exn_error ("i/o", (ksi_obj) port, "%s: async mode not supported", proc);
  return -1;
#endif
}

static int
file_r_ready (char *proc, struct Ksi_FdPort *port)
{
#if defined(WIN32)
  if (port->r_ready) {
    int res = port->r_ready (port, 0);
    if (res < 0 && proc) {
      ksi_exn_error ("i/o", (ksi_obj) port, "%s: %s", proc, port->error (port));
    }
    return res;
  }

  if (port->kp.is_tty) {
    long  h = _get_osfhandle (port->fd);
    DWORD r = WaitForSingleObject ((HANDLE) h, 0);
    return (r == WAIT_OBJECT_0 ? 1 : 0);
  }

  return 1;

#elif defined(HAVE_POLL)

  struct pollfd fds;
  int res;

again:
  fds.fd      = port->fd;
  fds.events  = POLLIN;
  fds.revents = 0;

  if ((res = poll (&fds, 1, 0)) < 0) {
    if (errno == EINTR)
      goto again;
    if (proc)
      ksi_exn_error ("i/o", (ksi_obj) port, "%s: %m", proc);
  }

  return res; /* (fds.revents & POLLIN) != 0 */

#elif defined(HAVE_SELECT)

  struct timeval timeout;
  fd_set r_set;
  int res;

  if (port->fd >= FD_SETSIZE)
    return 1;

again:
  FD_ZERO (&r_set);
  FD_SET (port->fd, &r_set);

  timeout.tv_sec  = 0;
  timeout.tv_usec = 0;

  if ((res = select (port->fd+1, &r_set, 0, 0, &timeout)) < 0) {
    if (errno == EINTR)
      goto again;
    if (proc)
      ksi_exn_error ("i/o", (ksi_obj) port, "%s: %m", proc);
  }

  return res; /*FD_ISSET (sock->fd, &r_set);*/

#else

  return 1;

#endif
}

static int
file_w_ready (char *proc, struct Ksi_FdPort *port, int wait)
{
#if defined(WIN32)

  if (port->w_ready) {
    int res = port->w_ready (port, wait);
    if (res < 0 && proc) {
      ksi_exn_error ("i/o", (ksi_obj) port, "%s: %s", proc, port->error (port));
    }
    return res;
  }

  return 1;

#elif defined(HAVE_POLL)

  struct pollfd fds;
  int res;

again:
  fds.fd      = port->fd;
  fds.events  = POLLOUT;
  fds.revents = 0;

  if ((res = poll (&fds, 1, 0)) < 0) {
    if (errno == EINTR)
      goto again;
    if (proc)
      ksi_exn_error ("i/o", (ksi_obj) port, "%s: %m", proc);
  }

  return res; /* (fds.revents & POLLOUT) != 0 */

#elif defined(HAVE_SELECT)

  struct timeval timeout;
  fd_set w_set;
  int res;

  if (port->fd >= FD_SETSIZE)
    return 1;

again:
  FD_ZERO (&w_set);
  FD_SET (port->fd, &w_set);

  timeout.tv_sec  = 0;
  timeout.tv_usec = 0;

  if ((res = select (port->fd+1, 0, &w_set, 0, wait ? 0 : &timeout)) < 0) {
    if (errno == EINTR)
      goto again;
    if (proc)
      ksi_exn_error ("i/o", (ksi_obj) port, "%s: %m", proc);
  }

  return res; /*FD_ISSET (port->fd, &w_set);*/

#else

  return 1;

#endif
}


static const char*
fevt_name (ksi_event x)
{
  return "async-output";
}

static void
fevt_init (ksi_event x)
{
}

static void
fevt_setup (ksi_event x)
{
  struct Ksi_FdEvent *e = (struct Ksi_FdEvent*) x;

#ifdef WIN32
  e->ed = ksi_wait_timer ((ksi_event) e, 0.1, 1);
#else
  e->ed = ksi_wait_output ((ksi_event) e, e->port->fd, 1);
#endif
}

static void
fevt_cancel (ksi_event x)
{
  struct Ksi_FdEvent *e = (struct Ksi_FdEvent*) x;

  if (e->ed) {
#ifdef WIN32
    ksi_cancel_timer ((ksi_event) e, e->ed);
#else
    ksi_cancel_output ((ksi_event) e, e->ed);
#endif
    e->ed = 0;
  }
}

static int
fevt_invoke (ksi_event x, void *data)
{
  struct Ksi_FdEvent *e = (struct Ksi_FdEvent *) x;
  int n;

  if (e->port->w_num == 0)
    return 1;

  if ((n = file_w_ready (0, e->port, 0)) == 0)
    return 0;
  if (n > 0 && (n = fd_write (e->port, e->port->w_buf, e->port->w_num)) > 0) {
    if ((e->port->w_num -= n) > 0)
      memmove (e->port->w_buf, e->port->w_buf + n, e->port->w_num);
  }

  if (n < 0) {
    const char *err;
#ifdef WIN32
    if (e->port->error)
      err = e->port->error (e->port);
    else
#endif
      err = strerror (errno);

    x->result = ksi_make_exn ("i/o", (ksi_obj) e->port, ksi_aprintf ("write-char: %s", err), 0);
    return 1;
  }

  return (!e->port->w_num);
}

static struct Ksi_Event_Tag fevt_ops =
{
  fevt_name,
  fevt_init,
  fevt_setup,
  fevt_cancel,
  fevt_invoke,
};

static ksi_obj
make_fd_event (struct Ksi_FdPort *port)
{
  struct Ksi_FdEvent *x;

  x = (struct Ksi_FdEvent*) ksi_malloc (sizeof *x);
  x->evt.o.itag   = KSI_TAG_EVENT;
  x->evt.ops    = &fevt_ops;
  x->evt.state  = ksi_data->sym_inactive;
  x->evt.action = ksi_data->void_proc;
  x->evt.result = ksi_void;
  x->port       = port;
  x->ed         = 0;

  return (ksi_obj) x;
}

static const char*
file_name (ksi_port x)
{
  struct Ksi_FdPort* port = (struct Ksi_FdPort*) x;
  return port->name;
}

static int
file_close (ksi_port x)
{
  int res = 0;
  struct Ksi_FdPort *port = (struct Ksi_FdPort*) x;

  if (port->evt) {
    ksi_stop_event (port->evt);
    port->evt = 0;
  }
  if (port->kp.async)
    res = file_set_async (0, port, 0);

  if (res == 0 && port->w_num)
    res = fd_write (port, port->w_buf, port->w_num);

  ksi_free (port->r_buf);
  ksi_free (port->w_buf);

  if (fd_close (port) < 0)
    res = -1;

  port->kp.input  = 0;
  port->kp.output = 0;
  port->kp.closed = 1;
  port->fd        = -1;
  port->r_buf     = 0;
  port->w_buf     = 0;
  port->evt       = 0;
  port->r_pos     = 0;
  port->r_len     = 0;
  port->w_num     = 0;
  port->w_size    = 0;

  if (res < 0) {
    const char *err;
#ifdef MSWIN32
    if (port->error)
      err = port->error (port);
    else
#endif
      err = strerror (errno);

    ksi_exn_error ("i/o", (ksi_obj) x, "close-port: %s", err);
  }

  return 0;
}

static int
file_read (ksi_port x, char* buf, int len)
{
  struct Ksi_FdPort* port = (struct Ksi_FdPort*) x;
  int num;

  if (port->kp.unbuf) {
    if ((num = fd_read (port, buf, len)) < 0) {
      const char *err;
    error:
#ifdef WIN32
      if (port->error)
        err = port->error (port);
      else
#endif
        err = strerror (errno);

      ksi_exn_error ("i/o", (ksi_obj) port, "read-char: %s", err);
    }

    return num;
  }

  for (num = 0; len > 0 && port->r_pos < port->r_len; num++, len--)
    *buf++ = port->r_buf[port->r_pos++];

  if (len >= port->pg_size) {
    int i, n = len - (len % port->pg_size);
    if ((i = fd_read (port, buf, n)) < 0) {
#ifdef EWOULDBLOCK
      if (num > 0 && errno == EWOULDBLOCK)
        return num;
#endif
      goto error;
    }

    if (i < n)
      return num + i;

    len -= i; buf += i; num += i;
  }

  if (len > 0) {
    int i;

    if (!port->r_buf)
      port->r_buf = (char*) ksi_malloc_data (port->pg_size);

    if ((i = fd_read (port, port->r_buf, port->pg_size)) < 0) {
#ifdef EWOULDBLOCK
      if (num > 0 && errno == EWOULDBLOCK)
        return num;
#endif
      goto error;
    }

    port->r_len = i;
    port->r_pos = 0;
    for (/**/; len > 0 && port->r_pos < port->r_len; num++, len--)
      *buf++ = port->r_buf[port->r_pos++];
  }

  return num;
}

static int
file_write (ksi_port x, const char* buf, int len)
{
  struct Ksi_FdPort* port = (struct Ksi_FdPort*) x;
  int num, nl_num;

  if (!port->kp.async && port->kp.unbuf) {
    if ((num = fd_write (port, buf, len)) < 0) {
      const char *err;
    error:
#ifdef WIN32
      if (port->error)
        err = port->error (port);
      else
#endif
        err = strerror (errno);

      ksi_exn_error ("i/o", (ksi_obj) port, "write-char: %s", err);
    }

    return num;
  }

  if (port->w_buf == 0) {
    port->w_buf  = (char*) ksi_malloc_data (port->pg_size);
    port->w_size = port->pg_size;
  }

  nl_num = 0;
  for (num = 0; len > 0 && port->w_num < port->w_size; num++, len--)
    if ((port->w_buf[port->w_num++] = *buf++) == '\n')
      nl_num = port->w_num;

  /* возможные варианты:
   * 1) записали все в буфер и буфер не заполнен
   * 2) записали все в буфер и буфер заполнен
   * 3) что-то осталось и буфер заполнен
   */

  if (port->w_num == port->w_size) {
    int n;
    if (port->kp.async && !file_w_ready ("write-char", port, 0))
      n = 0;
    else if ((n = fd_write (port, port->w_buf, port->w_num)) < 0)
      goto error;

    nl_num = 0;
    if (n == port->w_num)
      port->w_num = 0;
    else {
      port->w_num -= n;
      memmove (port->w_buf, port->w_buf + n, port->w_num);

      /* добавляем в буфер все что осталось */
      if (len > 0) {
        int s = port->w_num + len;
        if (s > port->w_size) {
          s += port->pg_size - 1;
          s -= s % port->pg_size;
          port->w_buf  = (char*) ksi_realloc (port->w_buf, s);
          port->w_size = s;
        }

        memcpy (port->w_buf + port->w_num, buf, len);
        port->w_num += len;
        num += len; len = 0;
      }
    }
  }

  /* возможные варианты:
   * 1) записали все в буфер
   * 2) что-то осталось и буфер пуст
   */

  if (len >= port->w_size) {
    int i, n = len - (len % port->w_size);
    if (port->kp.async && !file_w_ready ("write-char", port, 0))
      i = 0;
    else if ((i = fd_write (port, buf, n)) < 0)
      goto error;

    nl_num = 0;
    num += i, buf += i, len -= i;
    if (i < n) {
      /* добавляем остатки в буфер */
      int s = port->w_num + len;
      if (s > port->w_size) {
        s += port->pg_size - 1;
        s -= s % port->pg_size;
        port->w_buf  = (char*) ksi_realloc (port->w_buf, s);
        port->w_size = s;
      }

      memcpy (port->w_buf + port->w_num, buf, len);
      port->w_num += len;
      num += len; len = 0;
    }
  }

  /* возможные варианты:
   * 1) записали все в буфер
   * 2) данных осталось меньше размера буфера и буфер пуст
   */

  if (len > 0) {
    for (nl_num = -1; len > 0; num++, len--)
      if ((port->w_buf[port->w_num++] = *buf++) == '\n')
        nl_num = port->w_num;
  }

  /* возможные варианты:
   * 1) что-то есть в буфере
   * 2) буфер пуст
   */

  if (port->kp.async) {
    if (port->w_num) {
      if (!port->evt)
        port->evt = make_fd_event (port);
      ksi_start_event (port->evt);
    } else if (port->evt) {
      ksi_stop_event (port->evt);
    }
  } else if (port->kp.linebuf && nl_num > 0) {
    if ((nl_num = fd_write (port, port->w_buf, nl_num)) < 0)
      goto error;

    if ((port->w_num -= nl_num) > 0)
      memmove (port->w_buf, port->w_buf + nl_num, port->w_num);
  }

  return num;
}

static int
file_input_ready (ksi_port x)
{
  struct Ksi_FdPort* port = (struct Ksi_FdPort*) x;

  if (port->r_pos < port->r_len)
    return 1;

  return file_r_ready ("char-ready?", port);
}

static int
file_output_ready (ksi_port x)
{
  struct Ksi_FdPort* port = (struct Ksi_FdPort*) x;

  if (port->w_num) {
    if (port->kp.async)
      return 0;
  }

  return file_w_ready ("port-ready?", port, 0);
}

static int
file_flush (ksi_port x)
{
  struct Ksi_FdPort* port = (struct Ksi_FdPort*) x;

  if (port->evt)
    ksi_stop_event (port->evt);

  if (port->kp.async) {
    if (file_set_async (0, port, 0) != 0) {
      const char *err;
    error:
#ifdef WIN32
      if (port->error)
        err = port->error (port);
      else
#endif
        err = strerror (errno);

      ksi_exn_error ("i/o", (ksi_obj) port, "flush-port: %s", err);
    }
  }

  while (port->w_num) {
    int n = fd_write (port, port->w_buf, port->w_num);
    if (n < 0)
      goto error;

    if ((port->w_num -= n) != 0)
      memmove (port->w_buf, port->w_buf + n, port->w_num);
  }

  if (port->kp.async)
    if (file_set_async (0, port, 1) != 0)
      goto error;


  return 0;
}

static int
file_in_fd (ksi_port x)
{
  struct Ksi_FdPort* port = (struct Ksi_FdPort*) x;

  if (port->r_pos >= port->r_len) {
#if defined(MSWIN32)
    if (port->kp.is_tty)
      return _get_osfhandle (port->fd);
#else
    return port->fd;
#endif
  }

  return -1;
}

static int
file_out_fd (ksi_port x)
{
  struct Ksi_FdPort* port = (struct Ksi_FdPort*) x;

  if (port->w_num == 0) {
#if defined(MSWIN32)
    return -1;
#else
    return port->fd;
#endif
  }

  return -1;
}

static struct Ksi_Port_Ops file_port_ops =
{
  file_name,
  file_close,
  file_read,
  file_write,
  file_input_ready,
  file_output_ready,
  file_flush,
  file_in_fd,
  file_out_fd
};

static void
file_finalizer (void* obj, void* data)
{
  int n = 0;
  struct Ksi_FdPort *p = (struct Ksi_FdPort*) obj;

  if (p->kp.output) {
    char *buf = p->w_buf;
    while (p->w_num) {
      if (!p->kp.async || (n = file_w_ready (0, p, 1))) {
        if (n < 0 || (n = fd_write (p, buf, p->w_num)) < 0)
          break;
        buf += n;
        p->w_num -= n;
      }
    }
  }

  if (p->evt) {
    ksi_stop_event (p->evt);
    p->evt = 0;
  }
  if (!p->kp.is_ext)
    fd_close (p);
}

ksi_port
ksi_new_fd_port (int fd, const char* name, int no_init)
{
  struct Ksi_FdPort *port;
  struct stat st;
  char *type;

  if (!no_init)
    fstat (fd, &st);

  port = ksi_malloc(sizeof *port);
  port->kp.o.itag = KSI_TAG_PORT;
  port->kp.ops = &file_port_ops;
  port->kp.is_fd = 1;
  port->fd = fd;

  if (no_init) {
    port->name = ksi_aprintf ("%s", name);
  } else {
    port->kp.is_tty  = (isatty (fd) ? 1 : 0);
#ifdef S_ISDIR
    port->kp.is_dir  = (S_ISDIR  (st.st_mode) ? 1 : 0);
#endif
#ifdef S_ISCHR
    port->kp.is_chr  = (S_ISCHR  (st.st_mode) ? 1 : 0);
#endif
#ifdef S_ISBLK
    port->kp.is_blk  = (S_ISBLK  (st.st_mode) ? 1 : 0);
#endif
#ifdef S_ISREG
    port->kp.is_reg  = (S_ISREG  (st.st_mode) ? 1 : 0);
#endif
#ifdef S_ISFIFO
    port->kp.is_fifo = (S_ISFIFO (st.st_mode) ? 1 : 0);
#endif
#ifdef S_ISLNK
    port->kp.is_lnk  = (S_ISLNK  (st.st_mode) ? 1 : 0);
#endif
#ifdef S_ISSOCK
    port->kp.is_sock = (S_ISSOCK (st.st_mode) ? 1 : 0);
#endif
    port->kp.linebuf = (port->kp.is_tty ? 1 : 0);

    if (port->kp.is_tty)
      type = "tty";
    else if (port->kp.is_dir)
      type = "dir";
    else if (port->kp.is_chr)
      type = "char-device";
    else if (port->kp.is_blk)
      type = "block-device";
    else if (port->kp.is_fifo)
      type = "fifo";
    else if (port->kp.is_lnk)
      type = "link";
    else if (port->kp.is_sock)
      type = "socket";
    else
      type = "file";

    if (name) {
      port->name = ksi_aprintf ("%s %s", type, name);
      if (port->kp.is_reg)
        port->kp.good_name = 1;
    } else {
      port->name = ksi_aprintf ("%s-fd %d", type, fd);
    }
  }

#ifdef HAVE_STRUCT_STAT_ST_BLKSIZE
  port->pg_size = (port->kp.is_reg ? st.st_blksize : MAX_BUFSZ);
#else
  port->pg_size = MAX_BUFSZ;
#endif

  ksi_register_finalizer (port, file_finalizer, 0);
  return (ksi_port) port;
}


/* ---------- common port utils -------------------------------------- */

int
ksi_port_putc (ksi_port port, int ch)
{
  char buf[1];
  buf[0] = (char) ch;
  return ksi_port_write ((ksi_obj) port, buf, 1);
}

int
ksi_port_getc (ksi_port port)
{
  char buf[1];
  int n = ksi_port_read ((ksi_obj) port, buf, 1);

  if (n <= 0)
    return -1;

  return ((unsigned char) buf[0]);
}

void
ksi_port_ungetc (ksi_port port, int c)
{
    if (port->unread_num >= sizeof (port->unread_chars))
        ksi_exn_error(ksi_assertion_s, 0, "ksi_port_ungetc: ungetc buffer overflow");

    port->unread_chars[(int) port->unread_num++] = (char) c;

    if (c == '\n') {
        port->read_line -= 1;
        /* port->read_pos = sorry, position lost */
    } else if (c == '\t') {
        port->read_pos -= 8;
    } else {
        port->read_pos -= 1;
    }
}

int
ksi_port_read (ksi_obj port_obj, char* ptr, int len)
{
  int i, num;
  ksi_port port;
  KSI_CHECK (port_obj, KSI_INPUT_PORT_P (port_obj), "read-port: invalid inpput port in arg1");

  port = (ksi_port) port_obj;
  for (num = 0; port->unread_num && len > 0; num++, ptr++, len--) {
    *ptr = port->unread_chars[(int) --port->unread_num];
    if (*ptr == '\n') {
      port->read_line += 1;
      port->read_pos   = 0;
    } else if (*ptr == '\t')
      port->read_pos += 8;
    else
      port->read_pos += 1;
  }

  if (len > 0 && ((len = port->ops->read (port, ptr, len)) > 0)) {
    num += len;
    for (i = 0; i < len; i++) {
      if (ptr[i] == '\n') {
        port->read_line += 1;
        port->read_pos   = 0;
      } else if (ptr[i] == '\t') {
        port->read_pos += 8;
      } else {
        port->read_pos += 1;
      }
    }
  }

  return num;
}

int
ksi_port_write (ksi_obj port_obj, const char* ptr, int len)
{
  ksi_port port;
  KSI_CHECK (port_obj, KSI_OUTPUT_PORT_P (port_obj), "write-port: invalid output port in arg1");

  port = (ksi_port) port_obj;
  if (len > 0) {
    len = port->ops->write (port, ptr, len);
    if (len > 0)
      port->last_write_char = ptr [len - 1];
  }

  return len;
}

ksi_obj
ksi_close_port (ksi_obj port)
{
  KSI_CHECK (port, KSI_PORT_P (port), "close-port: invalid port in arg1");
  ((ksi_port) port) -> ops -> close ((ksi_port) port);
  ksi_cancel_port_events ((ksi_port) port);
  return ksi_void;
}

ksi_obj
ksi_flush_port (ksi_obj port)
{
  KSI_CHECK (port, KSI_PORT_P (port), "flush-port: invalid port in arg1");
  ((ksi_port) port) -> ops -> flush ((ksi_port) port);
  return ksi_void;
}

ksi_obj
ksi_null_port ()
{
  return (ksi_obj) ksi_data->null_port;
}

ksi_obj
ksi_current_input_port ()
{
  return ksi_int_data ? (ksi_obj) ksi_int_data->input_port : (ksi_obj) ksi_data->null_port;
}

ksi_obj
ksi_current_output_port ()
{
  return ksi_int_data ? (ksi_obj) ksi_int_data->output_port : (ksi_obj) ksi_data->null_port;
}

ksi_obj
ksi_current_error_port ()
{
  return ksi_int_data ? (ksi_obj) ksi_int_data->error_port : (ksi_obj) ksi_data->null_port;
}

#define DEF_SET(port, name, test)                                       \
ksi_obj                                                                 \
CONCAT(ksi_set_current_, port) (ksi_obj new_port)                       \
{                                                                       \
    if (ksi_int_data) {                                                 \
        ksi_obj old = (ksi_obj) ksi_int_data->port;                     \
        if (!test (new_port))                                           \
            ksi_exn_error (ksi_assertion_s, new_port, name ": invalid port"); \
        ksi_int_data->port = (ksi_port) new_port;                       \
        return old;                                                     \
    } else {                                                            \
        return (ksi_obj) ksi_data->null_port;                           \
    }                                                                   \
}

DEF_SET (input_port, "set-current-input-port", KSI_INPUT_PORT_P)
DEF_SET (output_port, "set-current-output-port", KSI_OUTPUT_PORT_P)
DEF_SET (error_port, "set-current-error-port", KSI_OUTPUT_PORT_P)


/* ---------- internal port utils ------------------------------------ */

static int
str2mode (const char *mode, const char *proc)
{
  int flags = 0;

  switch (*mode) {
  case 'r': flags = O_RDONLY;                      break;
  case 'w': flags = O_WRONLY | O_CREAT | O_TRUNC;  break;
  case 'a': flags = O_WRONLY | O_CREAT | O_APPEND; break;
  default:
    ksi_exn_error ("range", ksi_str02string (mode), "%s: invalid open mode", proc);
  }

  while (*++mode)
    switch (*mode) {
    case '+':
      flags = (flags & ~(O_RDONLY | O_WRONLY)) | O_RDWR;
      break;
#ifdef O_TEXT
    case 't':
      flags |= O_TEXT;
      break;
#endif
#ifdef O_BINARY
    case 'b':
      flags |= O_BINARY;
      break;
#endif
    }

  return flags;
}

static ksi_port
set_port_flags (ksi_port port, int flags, const char *mode)
{
  if (flags == O_RDWR) {
    port->input  = 1;
    port->output = 1;
  } else if (flags == O_RDONLY) {
    port->input  = 1;
    port->output = 0;
  } else {
    port->input  = 0;
    port->output = 1;
  }

  while (*++mode) {
    if (*mode == '0')
      port->unbuf = 1;
    else if (*mode == 'l')
      port->linebuf = 1;
  }

  return port;
}

ksi_port
ksi_open_fd_port_int (const char* fname, const char* mode, const char* proc)
{
  int fd, flags;
  ksi_port port;

  flags = str2mode (mode, proc);
  fd = open (fname, flags, 0666);
  if (fd < 0) {
    ksi_exn_error ("system", ksi_str02string (fname), "%s: %s", proc, strerror (errno));
  }

  port = ksi_new_fd_port (fd, fname, 0);
  return set_port_flags (port, flags & O_ACCMODE, mode);
}


/* ---------- external port utils ------------------------------------ */

ksi_obj
ksi_open_string (ksi_obj str, ksi_obj mode)
{
  int flags;
  ksi_port port;

  KSI_CHECK (str, KSI_STR_P (str), "open-string: invalid string in arg1");
  KSI_CHECK (mode, KSI_STR_P (mode), "open-string: invalid string in arg2");

  flags = str2mode (KSI_STR_PTR (mode), "open-string");
  if ((flags & O_ACCMODE) != O_RDONLY)
    KSI_CHECK (str, KSI_M_STR_P (str), "open-string: const string in arg1");

  port = ksi_new_str_port ((ksi_string) str);
  if (flags & O_APPEND) {
    struct Ksi_StringPort *sp = (struct Ksi_StringPort*) port;
    sp->pos = sp->size;
  }

  return (ksi_obj) set_port_flags (port, flags & O_ACCMODE, "");
}

ksi_obj
ksi_port_string (ksi_obj port)
{
  KSI_CHECK (port, (KSI_PORT_P (port) && (((ksi_port) port) -> ops == &str_port_ops)),
                  "port-string: invalid string-port in arg1");

  return (ksi_obj) (((struct Ksi_StringPort*) port) -> str);
}

ksi_port
ksi_make_fd_port (int fd, const char *fname, const char *mode)
{
  ksi_port port;
#if defined(F_GETFL)
  int fd_flags = fcntl (fd, F_GETFL, 0);
#else
  int fd_flags = O_RDWR;
#endif
  int md_flags = (mode ? str2mode (mode, "ksi_make_fd_port") : fd_flags);

  KSI_CHECK (ksi_long2num (fd), fd_flags >= 0, "ksi_make_fd_port: invalid fd");

  fd_flags &= O_ACCMODE;
  md_flags &= O_ACCMODE;
  if (fd_flags != O_RDWR && fd_flags != md_flags) {
    /* if fd_flags == O_RDWR, then any access allowed;
     * else fd_flags and md_flags can be O_RDONLY or O_WRONLY,
     * and they should be equal to allow requested access.
     */
    ksi_exn_error ("misc", ksi_long2num (fd), "ksi_make_fd_port: mode %s not available on fd", mode);
  }

  port = ksi_new_fd_port (fd, fname, 0);
  port->is_ext = 1;
  return set_port_flags (port, md_flags, mode);
}

ksi_obj
ksi_open_fd_port (const char* fname, const char* mode)
{
  return (ksi_obj) ksi_open_fd_port_int (fname, mode, "ksi_open_fd_port");
}

ksi_obj
ksi_open_file (ksi_obj fname, ksi_obj fmode)
{
  const char *mode, *name;

  name = ksi_mk_filename (fname, "open-file");

  KSI_CHECK (fmode, KSI_STR_P (fmode), "open-file: invalid string in arg2");
  mode = KSI_STR_PTR (fmode);

  return (ksi_obj) ksi_open_fd_port_int (name, mode, "open-file");
}


/* ---------- scheme port utils -------------------------------------- */

ksi_obj
ksi_port_p (ksi_obj port)
{
  return (KSI_PORT_P (port) ? ksi_true : ksi_false);
}

ksi_obj
ksi_input_port_p (ksi_obj port)
{
  return (KSI_INPUT_PORT_P (port) ? ksi_true : ksi_false);
}

ksi_obj
ksi_output_port_p (ksi_obj port)
{
  return (KSI_OUTPUT_PORT_P (port) ? ksi_true : ksi_false);
}

ksi_obj
ksi_eof_object_p (ksi_obj x)
{
  return (x == ksi_eof ? ksi_true : ksi_false);
}

ksi_obj
ksi_read_char (ksi_obj p)
{
  int c;

  if (!p)
    p = ksi_current_input_port ();

  KSI_CHECK (p, KSI_INPUT_PORT_P (p), "read-char: invalid input port");

  c = ksi_port_getc ((ksi_port) p);
  if (c < 0)
    return ksi_eof;

  return ksi_int2char (c);
}

ksi_obj
ksi_peek_char (ksi_obj p)
{
  int c;

  if (!p)
    p = ksi_current_input_port ();

  KSI_CHECK (p, KSI_INPUT_PORT_P (p), "peek-char: invalid input port");

  c = ksi_port_getc ((ksi_port) p);
  if (c < 0)
    return ksi_eof;

  ksi_port_ungetc ((ksi_port) p, c);
  return ksi_int2char (c);
}

ksi_obj
ksi_char_ready_p (ksi_obj p)
{
  ksi_obj res = ksi_false;

  if (!p)
    p = ksi_current_input_port ();

  KSI_CHECK (p, KSI_INPUT_PORT_P (p), "char-ready?: invalid input port");

  if (((ksi_port) p) -> unread_num)
    res = ksi_true;
  else if (((ksi_port) p) -> ops -> input_ready ((ksi_port) p))
    res = ksi_true;

  return res;
}

ksi_obj
ksi_port_ready_p (ksi_obj p)
{
  if (!p)
    p = ksi_current_output_port ();

  KSI_CHECK (p, KSI_OUTPUT_PORT_P (p), "port-ready?: invalid output port");

  if (((ksi_port) p) -> ops -> output_ready ((ksi_port) p))
    return ksi_true;

  return ksi_false;
}

ksi_obj
ksi_write_char (ksi_obj o, ksi_obj p)
{
  if (!p)
    p = ksi_current_output_port ();

  KSI_CHECK (o, KSI_CHAR_P (o), "write-char: invalid char in arg1");
  KSI_CHECK (p, KSI_OUTPUT_PORT_P (p), "write-char: invalid output port in arg2");

  ksi_port_putc ((ksi_port) p, KSI_CHAR_CODE (o));
  return ksi_void;
}

ksi_obj
ksi_newline (ksi_obj p)
{
  if (!p)
    p = ksi_current_output_port ();

  KSI_CHECK (p, KSI_OUTPUT_PORT_P (p), "newline: invalid output port");

  ksi_port_putc ((ksi_port) p, '\n');
  return ksi_void;
}

ksi_obj
ksi_write (ksi_obj o, ksi_obj p)
{
  int i;
  const char* ptr;
  ksi_obj x;

  if (!p)
    p = ksi_current_output_port ();

  KSI_CHECK (p, KSI_OUTPUT_PORT_P (p), "write: invalid output port");

  if (!o)
    goto genobj;

  if (KSI_VEC_P (o)) {
    ksi_port_write (p, "#(", 2);
    for (i = 0; ;) {
      KSI_CHECK_EVENTS;
      if (i < KSI_VEC_LEN (o))
        ksi_write (KSI_VEC_REF (o, i), p);
      ++i;
      if (i >= KSI_VEC_LEN (o))
        break;
      ksi_port_putc ((ksi_port) p, ' ');
    }
    ksi_port_putc ((ksi_port) p, ')');
    return ksi_void;
  } else if (KSI_PAIR_P (o)) {
    if (KSI_PAIR_P (KSI_CDR (o)) && KSI_CDR (KSI_CDR (o)) == ksi_nil) {
      if (KSI_CAR (o) == ksi_data->sym_quote) {
        ksi_port_putc ((ksi_port) p, '\'');
        ksi_write (KSI_CAR (KSI_CDR (o)), p);
        return ksi_void;
      }
      if (KSI_CAR (o) == ksi_data->sym_quasiquote) {
        ksi_port_putc ((ksi_port) p, '`');
        ksi_write (KSI_CAR (KSI_CDR (o)), p);
        return ksi_void;
      }
      if (KSI_CAR (o) == ksi_data->sym_unquote) {
        ksi_port_putc ((ksi_port) p, ',');
        ksi_write (KSI_CAR (KSI_CDR (o)), p);
        return ksi_void;
      }
      if (KSI_CAR (o) == ksi_data->sym_unquote_splicing) {
        ksi_port_putc ((ksi_port) p, ',');
        ksi_port_putc ((ksi_port) p, '@');
        ksi_write (KSI_CAR (KSI_CDR (o)), p);
        return ksi_void;
      }
    }

    ksi_port_putc ((ksi_port) p, '(');
    x = o;
    while (1) {
      KSI_CHECK_EVENTS;
      ksi_write (KSI_CAR (o), p);
      o = KSI_CDR (o);
      if (o == ksi_nil)
        break;
      if (!KSI_PAIR_P (o)) {
        ksi_port_write (p, " . ", 3);
        ksi_write (o, p);
        break;
      }
      ksi_port_putc ((ksi_port) p, ' ');

      ksi_write (KSI_CAR (o), p);
      o = KSI_CDR (o);
      if (o == ksi_nil)
        break;
      if (!KSI_PAIR_P (o)) {
        ksi_port_write (p, " . ", 3);
        ksi_write (o, p);
        break;
      }
      ksi_port_putc ((ksi_port) p, ' ');

      x = KSI_CDR (x);
      if (o == x) {
        ksi_port_write (p, ". . .", 5);
        break;
      }
    }

    ksi_port_putc ((ksi_port) p, ')');
    return ksi_void;
  } else if (KSI_INST_P (o)) {
    ksi_write_inst (o, p, 1);
    return ksi_void;
  }

genobj:

  ptr = ksi_obj2str (o);
  ksi_port_write (p, ptr, strlen (ptr));

  return ksi_void;
}

ksi_obj
ksi_display (ksi_obj o, ksi_obj p)
{
  if (!p)
    p = ksi_current_output_port ();

  KSI_CHECK (p, KSI_OUTPUT_PORT_P (p), "display: invalid port");

  if (!o)
    goto genobj;

  if (KSI_CHAR_P (o)) {
    ksi_port_putc ((ksi_port) p, KSI_CHAR_CODE (o));
  } else {
    int len, i;
    const char* ptr;
    ksi_obj x;

    if (!o)
      goto genobj;

    if (KSI_SYM_P (o)) {
      ptr = KSI_SYM_PTR (o);
      len = KSI_SYM_LEN (o);
    } else if (KSI_STR_P (o)) {
      ptr = KSI_STR_PTR (o);
      len = KSI_STR_LEN (o);
    } else if (KSI_VEC_P (o)) {
      ksi_port_write (p, "#(", 2);
      for (i = 0; ;) {
        KSI_CHECK_EVENTS;
        if (i < KSI_VEC_LEN (o))
          ksi_display (KSI_VEC_REF (o, i), p);
        ++i;
        if (i >= KSI_VEC_LEN (o))
          break;
        ksi_port_putc ((ksi_port) p, ' ');
      }
      ksi_port_putc ((ksi_port) p, ')');
      return ksi_void;
    } else if (KSI_PAIR_P (o)) {
      if (KSI_PAIR_P (KSI_CDR (o)) && KSI_CDR (KSI_CDR (o)) == ksi_nil) {
        if (KSI_CAR (o) == ksi_data->sym_quote) {
          ksi_port_putc ((ksi_port) p, '\'');
          ksi_display (KSI_CAR (KSI_CDR (o)), p);
          return ksi_void;
        }
        if (KSI_CAR (o) == ksi_data->sym_quasiquote) {
          ksi_port_putc ((ksi_port) p, '`');
          ksi_display (KSI_CAR (KSI_CDR (o)), p);
          return ksi_void;
        }
        if (KSI_CAR (o) == ksi_data->sym_unquote) {
          ksi_port_putc ((ksi_port) p, ',');
          ksi_display (KSI_CAR (KSI_CDR (o)), p);
          return ksi_void;
        }
        if (KSI_CAR (o) == ksi_data->sym_unquote_splicing) {
          ksi_port_putc ((ksi_port) p, ',');
          ksi_port_putc ((ksi_port) p, '@');
          ksi_display (KSI_CAR (KSI_CDR (o)), p);
          return ksi_void;
        }
      }

      ksi_port_putc ((ksi_port) p, '(');
      x = o;
      while (1) {
        KSI_CHECK_EVENTS;
        ksi_display (KSI_CAR (o), p);
        o = KSI_CDR (o);
        if (o == ksi_nil)
          break;
        if (!KSI_PAIR_P (o)) {
          ksi_port_write (p, " . ", 3);
          ksi_display (o, p);
          break;
        }

        ksi_port_putc ((ksi_port) p, ' ');
        ksi_display (KSI_CAR (o), p);
        o = KSI_CDR (o);
        if (o == ksi_nil)
          break;
        if (!KSI_PAIR_P (o)) {
          ksi_port_write (p, " . ", 3);
          ksi_display (o, p);
          break;
        }

        ksi_port_putc ((ksi_port) p, ' ');
        x = KSI_CDR (x);
        if (o == x) {
          ksi_port_write (p, ". . .", 5);
          break;
        }
      }
      ksi_port_putc ((ksi_port) p, ')');
      return ksi_void;
    } else if (KSI_INST_P (o)) {
      ksi_write_inst (o, p, 0);
      return ksi_void;
    } else {
    genobj:
      ptr = ksi_obj2str (o);
      len = strlen (ptr);
    }

    ksi_port_write (p, ptr, len);
  }

  return ksi_void;
}

ksi_obj
ksi_read_block (ksi_obj x, ksi_obj s)
{
  int n;
  char *buf;

  KSI_CHECK (x, KSI_INPUT_PORT_P (x), "read-block: invalid input port in arg1");
  KSI_CHECK (s, !s || KSI_EINT_P (s), "read-block: invalid integer in arg2");
  n = (s ? ksi_num2long (s, "read-block") : MAX_BUFSZ);
  KSI_CHECK (s, n >= 0, "read-block: size out of range");

  if (n == 0)
    return ksi_str2string (0, 0);

  if (n >= MAX_BUFSZ)
    buf = (char*) ksi_malloc_data (n);
  else
    buf = (char*) alloca (n);

  n = ((ksi_port) x) -> ops -> read ((ksi_port) x, buf, n);
  if (n <= 0)
    return ksi_eof;
  return ksi_str2string (buf, n);
}

ksi_obj
ksi_write_block (ksi_obj x, ksi_obj s)
{
  char *buf;
  int  num;

  KSI_CHECK (x, KSI_OUTPUT_PORT_P (x), "write-block: invalid output port in arg1");
  KSI_CHECK (s, KSI_STR_P (s), "write-block: invalid string in arg2");

  buf = KSI_STR_PTR (s);
  num = KSI_STR_LEN (s);

  num = ((ksi_port) x) -> ops -> write ((ksi_port) x, buf, num);
  return ksi_long2num (num);
}


ksi_obj
ksi_set_async_mode (ksi_obj x, ksi_obj async)
{
  struct Ksi_FdPort *port = (struct Ksi_FdPort*) x;
  int async_flag = (async == ksi_false ? 0 : 1);

  KSI_CHECK (x, FILE_P (x), "set-async-mode: invalid port in arg1");

  file_set_async ("set-async-mode", port, async_flag);
  port->kp.async = async_flag;

  return ksi_void;
}


/* End of code */
