/*
 * ksi_time.c
 * time related function
 *
 * Copyright (C) 1997-2010, ivan demakov
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        Ivan Demakov <ksion@users.sourceforge.net>
 * Creation date: Mon Nov 17 05:38:36 1997
 * Last Update:   Fri Jan 22 21:42:17 2010
 *
 */

#include "ksi_int.h"
#include "ksi_util.h"


ksi_obj
ksi_scm_current_time (void)
{
  time_t t = time (0);
  return ksi_long2num (t);
}

ksi_obj
ksi_scm_cpu_time (void)
{
  double d = ksi_cpu_time ();
  return ksi_double2num (d);
}

ksi_obj
ksi_scm_eval_time (void)
{
  double d = ksi_eval_time ();
  return ksi_double2num (d);
}

ksi_obj
ksi_scm_real_time (void)
{
  double d = ksi_real_time ();
  return ksi_double2num (d);
}


static ksi_obj
tm2obj (struct tm *bt)
{
  ksi_obj v = (ksi_obj) ksi_alloc_vector (9, KSI_TAG_VECTOR);
  KSI_VEC_REF (v, 0) = ksi_long2num (bt->tm_sec);
  KSI_VEC_REF (v, 1) = ksi_long2num (bt->tm_min);
  KSI_VEC_REF (v, 2) = ksi_long2num (bt->tm_hour);
  KSI_VEC_REF (v, 3) = ksi_long2num (bt->tm_mday);
  KSI_VEC_REF (v, 4) = ksi_long2num (bt->tm_mon);
  KSI_VEC_REF (v, 5) = ksi_long2num (bt->tm_year);
  KSI_VEC_REF (v, 6) = ksi_long2num (bt->tm_wday);
  KSI_VEC_REF (v, 7) = ksi_long2num (bt->tm_yday);
  KSI_VEC_REF (v, 8) = ksi_long2num (bt->tm_isdst);

  return v;
}

static int
obj2tm (ksi_obj v, struct tm *bt, const char *name)
{
  if (KSI_VEC_P (v) && KSI_VEC_LEN (v) == 9
      && KSI_EINT_P (KSI_VEC_REF (v, 1))
      && KSI_EINT_P (KSI_VEC_REF (v, 2))
      && KSI_EINT_P (KSI_VEC_REF (v, 3))
      && KSI_EINT_P (KSI_VEC_REF (v, 4))
      && KSI_EINT_P (KSI_VEC_REF (v, 5))
      && KSI_EINT_P (KSI_VEC_REF (v, 6))
      && KSI_EINT_P (KSI_VEC_REF (v, 7))
      && KSI_EINT_P (KSI_VEC_REF (v, 8)))
    {
      bzero (bt, sizeof *bt);

      bt->tm_sec   = ksi_num2long (KSI_VEC_REF (v, 0), name);
      bt->tm_min   = ksi_num2long (KSI_VEC_REF (v, 1), name);
      bt->tm_hour  = ksi_num2long (KSI_VEC_REF (v, 2), name);
      bt->tm_mday  = ksi_num2long (KSI_VEC_REF (v, 3), name);
      bt->tm_mon   = ksi_num2long (KSI_VEC_REF (v, 4), name);
      bt->tm_year  = ksi_num2long (KSI_VEC_REF (v, 5), name);
      bt->tm_wday  = ksi_num2long (KSI_VEC_REF (v, 6), name);
      bt->tm_yday  = ksi_num2long (KSI_VEC_REF (v, 7), name);
      bt->tm_isdst = ksi_num2long (KSI_VEC_REF (v, 8), name);

      return 1;
    }

  return 0;
}

ksi_obj
ksi_scm_gmtime (ksi_obj arg_time)
{
  time_t it;
  struct tm tm;

  if (arg_time) {
    KSI_CHECK (arg_time, KSI_RATIONAL_P (arg_time), "gmtime: invalid real number as time in arg1");
    it = (time_t) ksi_num2long (arg_time, "gmtime");
  } else {
    it = time(0);
  }

  if (ksi_gmtime (it, &tm))
    return tm2obj (&tm);

  return ksi_false;
}

ksi_obj
ksi_scm_localtime (ksi_obj arg_time)
{
  time_t it;
  struct tm tm;

  if (arg_time) {
    KSI_CHECK (arg_time, KSI_REAL_P (arg_time), "localtime: invalid real number as time in arg1");
    it = (time_t) ksi_num2long (arg_time, "localtime");
  } else {
    it = time(0);
  }

  if (ksi_localtime (it, &tm))
    return tm2obj (&tm);

  return ksi_false;
}

ksi_obj
ksi_strftime (ksi_obj format, ksi_obj time)
{
  struct tm bt;
  char *buf;
  int size, ok = obj2tm (time, &bt, "strtime");

  KSI_CHECK (format, KSI_STR_P (format), "strftime: invalid string in arg1");
  KSI_CHECK (time, ok, "strftime: invalid broken-time in arg2");

  size = 64;
  buf = (char*) ksi_malloc_data (size);
  while (strftime (buf, size, KSI_STR_PTR (format), &bt) >= size) {
    ksi_free (buf);
    size *= 2;
    buf = (char*) ksi_malloc_data (size);
  }

  return ksi_str02string (buf);
}

ksi_obj
ksi_mktime (ksi_obj time)
{
  time_t    it;
  struct tm bt;
  int ok = obj2tm (time, &bt, "mktime");

  KSI_CHECK (time, ok, "mktime: invalid broken-time");
  it = mktime (&bt);
  return (it == ((time_t) -1) ? ksi_false : ksi_long2num (it));
}


 /* End of code */
