/*
 * Copyright (c) 2004, 2005, 2018, 2023 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <getopt.h>
#include <string.h>
#include <ifaddrs.h>
#include <inttypes.h>
#include <time.h>

#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <net/if.h>

#include "config.h"
#include "packets.h"
#include "mdd.h"

int quiet = 0;
int debug = 0;
int verbose = 0;

static char *humanize_offset(off_t, char *, char *);

int main(int argc, char **argv)
{
	char ch;
	int fl;
	int i;
	int net_block_size = NET_BLOCK_SIZE;
	int write_block_size = WRITE_BLOCK_SIZE;
	off_t start = 0;
	off_t sendmax = -1;
	int block_size = 0;
	int do_sender = 0;
	int do_sniffer = 0;
	int do_stat = 0;
	int do_receiver = 0;
	struct timeval timeout = { TIMEOUT_SEC, TIMEOUT_USEC };
	char *file_name = NULL;
	in_addr_t group_addr;
	unsigned int ifindex = 0;
	unsigned char ttl = 1;

	while((ch = getopt(argc, argv,
			   "S:C:qdvr:s:f:b:t:T:w:W:i:")) != (char)-1) {
		switch (ch) {
		case 'S':
			start = (off_t)atoll(optarg);
			break;
		case 'C':
			sendmax = (off_t)atoll(optarg);
			break;
		case 'q':
			quiet = 1;
			break;
		case 'd':
			debug = 1;
			verbose = 1;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'r':
			do_receiver = 1;
			if ((inet_pton(AF_INET, optarg, &group_addr)) != 1) {
				printf("Invalid address \"%s\"\n", optarg);
				exit(-1);
			}
			break;
		case 's':
			do_sender = 1;
			if ((inet_pton(AF_INET, optarg, &group_addr)) != 1) {
				printf("Invalid address \"%s\"\n", optarg);
				exit(-1);
			}
			break;
		case 'f':
			file_name = optarg;
			break;
		case 'b':
			block_size = atoi(optarg);
			break;
		case 't':
			timeout.tv_usec = atol(optarg);
			timeout.tv_sec = timeout.tv_usec / 1000000;
			timeout.tv_usec = timeout.tv_usec % 1000000;
			break;
		case 'T':
			ttl = (unsigned char)atoi(optarg);
			break;
		case 'w':
			if ((inet_pton(AF_INET, optarg, &group_addr)) != 1) {
				printf("Invalid address \"%s\"\n", optarg);
				exit(-1);
			}
			do_sniffer = 1;
			break;
		case 'W':
			if ((inet_pton(AF_INET, optarg, &group_addr)) != 1) {
				printf("Invalid address \"%s\"\n", optarg);
				exit(-1);
			}
			do_stat = 1;
			break;
		case 'i':
			if ((ifindex = if_nametoindex(optarg)) == 0) {
				printf("Invalid interface \"%s\"\n", optarg);
				exit(-1);
			}
			break;
		default:
			printf("Usage: mdd -w addr [-d]\n");
			printf("       mdd -W addr [-d]\n");
			printf("       mdd -s addr [-d|-v|-q] [-T ttl] "
			       "[-b blocksize] [-S start] [-C sendmax] "
			       "-f file \n");
			printf("       mdd -r addr [-d|-v|-q] [-T ttl] "
			       "[-b blocksize] [-t timeout] "
			       "[-S start] -f file \n");
			exit(-1);
			break;
		}
	}

	/*
	 * If stdin/stdout/stderr are hooked to the network,
	 * avoid blocking on them when reporting progression
	 */
	for (i = 0; i <= 2; i++) {
		if ((fl = fcntl(i, F_GETFL, NULL)) != -1)
			(void)fcntl(i, F_SETFL, fl | O_NONBLOCK);
	}


	if (do_sender) {
		if (block_size != 0)
			net_block_size = block_size;

		if (verbose) {
			printf("Running as sender, ");
			printf("file name: \"%s\", ", file_name);
			printf("block size: %d\n", net_block_size);
			if (start != 0)
				printf("start at byte: %" PRId64 "\n", start);
			if (sendmax != -1)
				printf("send max bytes: %" PRId64 "\n",sendmax);
		}
		sender(file_name, start, sendmax, net_block_size,
		    group_addr, ttl, ifindex);
		printf("\n");
		return 0;
	}

	if (do_sniffer) {
		if (verbose)
			printf("Running as sniffer\n");
		sniffer(group_addr, 0, ifindex);
		printf("\n");
		return 0;
	}

	if (do_stat) {
		if (verbose)
			printf("Collecting statistics\n");
		sniffer(group_addr, 1, ifindex);
		printf("\n");
		return 0;
	}

	if (do_receiver) {
		/* Else: receiver */
		if (block_size != 0)
			write_block_size = block_size;

		if (verbose) {
			printf("Running as receiver, ");
			printf("file name: \"%s\", ", file_name);
			printf("timeout: %g.%06ld sec\n",
			    difftime(timeout.tv_sec, 0),
			    (long)timeout.tv_usec);
			printf("block size: %d\n",  write_block_size);
			if (start != 0)
				printf("start at byte: %" PRId64 "\n", start);
		}
		receiver(file_name, start, write_block_size,
			 timeout, group_addr, ttl, ifindex);
		printf("\n");
		return 0;
	}

	printf("Either -r, -s or -w is required\n");
	return -1;
}

static in_addr_t
local_addr_if(unsigned int ifindex)
{
	in_addr_t addr = INADDR_ANY;
	struct ifaddrs *ifa_list, *ifa;
	char name[IFNAMSIZ];

	if (if_indextoname(ifindex, name) == NULL) {
		perror("if_indextoname failed");
		exit (-1);
	}

	if (getifaddrs(&ifa_list) == -1) {
		perror("getifaddrs failed");
		exit(-1);
	}

	for (ifa = ifa_list; ifa != NULL; ifa = ifa->ifa_next) {
		struct sockaddr_in *sin;

		if (strcmp(ifa->ifa_name, name) != 0)
			continue;

		if (ifa->ifa_addr->sa_family != AF_INET)
			continue;

		sin = (struct sockaddr_in *)ifa->ifa_addr;
		addr = sin->sin_addr.s_addr;
		break;
	}

	freeifaddrs(ifa_list);

	return addr;
}

static in_addr_t
local_addr_dst(in_addr_t dst_addr)
{
	struct sockaddr_in src;
	struct sockaddr_in dst;
	int sock = -1;
	socklen_t socklen;

	bzero(&src, sizeof(src));
#ifndef __linux__
	src.sin_len = sizeof(src);
#endif
	src.sin_family = AF_INET;

#ifndef __linux__
	dst.sin_len = sizeof(dst);
#endif
	dst.sin_family = AF_INET;
	dst.sin_port = htons(DATA_SEND_PORT);
	dst.sin_addr.s_addr = dst_addr;

	if ((sock = socket(dst.sin_family, SOCK_DGRAM, 0)) == -1) {
		perror("Cannot open socket");
		goto err;
	}

	if (connect(sock, (struct sockaddr *)&dst, sizeof(dst)) != 0) {
		perror("connect failed");
		goto err;
	}

	socklen = sizeof(src);
	if (getsockname(sock, (struct sockaddr *)&src, &socklen) != 0) {
		perror("getsockname failed");
		goto err;
	}

err:
	close(sock);
	return src.sin_addr.s_addr;
}

in_addr_t
local_addr(in_addr_t dst_addr, unsigned int ifindex)
{
	in_addr_t addr;

	if (ifindex != 0)
		addr = local_addr_if(ifindex);
	else
		addr = local_addr_dst(dst_addr);

	return addr;
}

int
set_multicast_if(int s, in_addr_t *src_addr)
{
	if (debug) {
		struct in_addr sa;

		sa.s_addr = *src_addr;
		printf("setting IP_MULTICAST_IF for %s\n",
		       inet_ntoa(sa));
	}

	if ((setsockopt(s, IPPROTO_IP, IP_MULTICAST_IF,
			src_addr, sizeof(*src_addr))) == -1) {
		perror("cannot set IP_MULTICAST_IF");
		return -1;
	}

	return 0;
}

#define MAXSTRLEN 80

static char *
humanize_offset(off_t offset, char *str, char *unit)
{
	off_t b, kb, mb, gb, tb;

	kb = offset / 1024;
	b = offset % 1024;

	mb = kb / 1024;
	kb = kb % 1024;

	gb = mb / 1024;
	mb = mb % 1024;

	tb = gb / 1024;
	gb = gb % 1024;

	if (tb != 0) {
		snprintf(str, MAXSTRLEN,
		    "%" PRId64 ".%03" PRId64 " T%s",
		    tb, gb * 1000 / 1024, unit);
		return str;
	}

	if (gb != 0) {
		snprintf(str, MAXSTRLEN,
		    "%" PRId64 ".%03" PRId64 " G%s",
		    gb, mb * 1000 / 1024, unit);
		return str;
	}

	if (mb != 0) {
		snprintf(str, MAXSTRLEN,
		    "%" PRId64 ".%03" PRId64 " M%s",
		    mb, kb * 1000 / 1024, unit);
		return str;
	}

	snprintf(str, MAXSTRLEN,
	    "%" PRId64 ".%03" PRId64 " k%s",
	    kb, b * 1000 / 1024, unit);

	return str;
}

static void
humanize_time(time_t ts, char *str)
{
	int sec;
	int min;
	int hour;

	hour = ts / 3600;
	sec = ts % 3600;
	min = sec / 60;
	sec = sec % 60;

	snprintf(str, MAXSTRLEN, "ETA %d:%02d:%02d", hour, min, sec);

	return;
}

static int
comp_throughput(off_t offset, off_t last_offset,
		struct timeval *tv, struct timeval *last_tv)
{
	double throughput;
	struct timeval diff_tv;
	double diff_usec;

	timersub(tv, last_tv, &diff_tv);
	diff_usec = (1000000 * diff_tv.tv_sec) + diff_tv.tv_usec;

	if (diff_usec <= 0)
		return 0;

	throughput = 1000000 * (offset - last_offset) / diff_usec;
	return (int)throughput;
}

void
print_report(off_t offset, off_t file_size)
{
	struct timeval now;
	static struct timeval next_report = { 0, 0 };
	static struct timeval last_report = { 0, 0 };
	static struct timeval start = { 0, 0 };
	static off_t last_offset = 0;
	struct timeval interval = { REPORT_INTERVAL_SEC, REPORT_INTERVAL_USEC };

	gettimeofday(&now, NULL);
	if (start.tv_sec == 0) {
		start.tv_sec = now.tv_sec;
		start.tv_usec = now.tv_usec;
	}

	if (timercmp(&now, &next_report, >)) {
		char offset_str[MAXSTRLEN + 1] = "";
		char throughput_str[MAXSTRLEN + 1] = "";
		char remain_str[MAXSTRLEN + 1] = "";
		char report_str[MAXSTRLEN + 1] = "";
		int throughput;

		humanize_offset(offset, offset_str, "B");

		throughput = comp_throughput(offset, last_offset,
					     &now, &last_report);
		humanize_offset(throughput, throughput_str, "B/s");

		if (file_size > 0) {
			int throughput_all;
			time_t remain;

			throughput_all = comp_throughput(offset, 0,
							 &now, &start);

			if (throughput_all > 0) {
				remain = (file_size - offset) / throughput_all;
				humanize_time(remain, remain_str);
			}
		}

		snprintf(report_str, MAXSTRLEN,
			 "Progress: %10s    \t%10s    \t%s    ",
			 offset_str, throughput_str, remain_str);

		printf("%s\r", report_str);
		fflush(stdout);

		timeradd(&interval, &now, &next_report);

		last_report.tv_sec = now.tv_sec;
		last_report.tv_usec = now.tv_usec;

		last_offset = offset;
	}

	return;
}
