/*
 * Copyright (c) 2004, 2005, 2018, 2023 Emmanuel Dreyfus
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *        This product includes software developed by Emmanuel Dreyfus
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <inttypes.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <netdb.h>

#include <sys/types.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/stat.h>

#include <arpa/inet.h>

#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>

#ifdef __linux__
#include <stdio_ext.h>
#define fpurge(x) __fpurge(x)
#endif

#include "mdd.h"
#include "config.h"
#include "packets.h"

#define REASON_TIMEOUT 	1
#define REASON_MISS	2
static void resend_req(int, off_t, struct sockaddr_in, int);

void
receiver(char *file_name, off_t start, size_t block_size,
         struct timeval timeout,
	 in_addr_t group_addr, unsigned char ttl, unsigned int ifindex)
{
	int fd, sd, rd;
	struct sockaddr_in send_addr;
	struct sockaddr_in recv_addr;
	ssize_t written;
	off_t offset;
	struct mdd_packet *packet = { 0 };
	char *write_buf;
	off_t write_offset;
	int write_remain;
	size_t write_size;
	char *write_ptr;
	size_t write_free;
	in_addr_t src_addr;
	size_t packet_size;
	int finished;
	mode_t mode;
	int water_mark;
	int on;
	fd_set fs;
	struct ip_mreq ms;
	struct ip_mreq mr;
	int ioerror;
	size_t write_buflen;

	mode = S_IWUSR | S_IRUSR;
	if ((fd = open(file_name, O_RDWR | O_CREAT, mode)) == -1) {
		printf("Open \"%s\" failed: %s\n", file_name, strerror(errno));
		exit(-1);
	}

	if ((sd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1) {
		printf("Cannot create UDP/IP socket: %s\n", strerror(errno));
		exit(-1);
	}

	on = 1;
	if ((setsockopt(sd, SOL_SOCKET, SO_REUSEPORT, &on, sizeof(on))) == -1) {
		perror("cannot set SO_REUSEPORT");
		exit(-1);
	}

#ifdef notanymore
	if ((setsockopt(sd, SOL_SOCKET, SO_BROADCAST, &on, sizeof(on))) == -1) {
		perror("cannot use broadcast");
		exit(-1);
	}
#endif
	if (debug)
		printf("Setting TTL to %d\n", (int)ttl);
        if ((setsockopt(sd, IPPROTO_IP, IP_MULTICAST_TTL,
            &ttl, sizeof(ttl))) == -1) {
                perror("cannot set IP_MULTICAST_TTL");
                exit(-1);
        }

	src_addr = local_addr(group_addr, ifindex);

	(void)memset(&send_addr, 0, sizeof(send_addr));
	send_addr.sin_family = AF_INET;
#ifndef __linux__
	send_addr.sin_len = sizeof(send_addr);
#endif
	send_addr.sin_addr.s_addr = src_addr;
	send_addr.sin_port = htons(RESEND_REQ_PORT);
	if ((bind(sd, (struct sockaddr *)&send_addr,
	    sizeof(send_addr))) == -1) {
		printf("Cannot bind to UDP port %d: %s\n",
		    RESEND_REQ_PORT, strerror(errno));
		exit(-1);
	}

	if (set_multicast_if(sd, &src_addr) != 0) {
		perror("set_multicast_if for sender socket failed");
	}

	water_mark = sizeof(struct mdd_packet);
	if ((setsockopt(sd, SOL_SOCKET, SO_SNDLOWAT,
	    &water_mark, sizeof(water_mark))) == -1) {
		perror("cannot set send low water mark");
		exit(-1);
	}

	bzero(&mr, sizeof(mr));
	ms.imr_multiaddr.s_addr = group_addr;
	ms.imr_interface.s_addr = src_addr;
	if ((setsockopt(sd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
	    &ms, sizeof(ms))) == -1) {
		perror("cannot set IP_ADD_MEMBERSHIP");
		exit(-1);
	}

	if ((rd = socket(PF_INET, SOCK_DGRAM, IPPROTO_UDP)) == -1) {
		printf("Cannot create UDP/IP socket: %s\n", strerror(errno));
		exit(-1);
	}


	on = 1;
	if ((setsockopt(rd, SOL_SOCKET, SO_REUSEPORT, &on, sizeof(on))) == -1) {
		perror("cannot set SO_REUSEPORT");
		exit(-1);
	}
	(void)memset(&recv_addr, 0, sizeof(recv_addr));
	recv_addr.sin_family = AF_INET;
#ifndef __linux__
	recv_addr.sin_len = sizeof(recv_addr);
#endif
	recv_addr.sin_addr.s_addr = INADDR_ANY;
	recv_addr.sin_port = htons(DATA_SEND_PORT);
	if ((bind(rd, (struct sockaddr *)&recv_addr,
	    sizeof(recv_addr))) == -1) {
		printf("Cannot bind to UDP port %d: %s\n",
		    DATA_SEND_PORT, strerror(errno));
		exit(-1);
	}

	if (set_multicast_if(rd, &src_addr) != 0) {
		perror("set_multicast_if for reciever socket failed");
	}

	water_mark = sizeof(struct mdd_packet);
	if ((setsockopt(rd, SOL_SOCKET, SO_RCVLOWAT,
	    &water_mark, sizeof(water_mark))) == -1) {
		perror("cannot set receive low water mark");
		exit(-1);
	}

	bzero(&mr, sizeof(mr));
	mr.imr_multiaddr.s_addr = group_addr;
	mr.imr_interface.s_addr = src_addr;
	if ((setsockopt(rd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
	    &mr, sizeof(mr))) == -1) {
		perror("cannot set IP_ADD_MEMBERSHIP");
		exit(-1);
	}

	packet_size = IP_MAXPACKET;
	if ((packet = malloc(packet_size)) == NULL) {
		perror("Cannot allocate buffer");
		exit(-1);
	}

	write_buflen = block_size + packet_size;
	if ((write_buf = malloc(write_buflen)) == NULL) {
		perror("Cannot allocate buffer");
		exit(-1);
	}
	write_ptr = write_buf;
	write_free = write_buflen;
	write_offset = 0;

	offset = 0;
	finished = 0;
	ioerror = 0;
	send_addr.sin_addr.s_addr = group_addr;
	do {
		off_t mdd_offset;
		size_t mdd_data_size;
		int mdd_operation;

		FD_ZERO(&fs);
		FD_SET(rd, &fs);
		if ((select(rd + 1, &fs, NULL, NULL, &timeout)) == -1) {
			perror("select failed");
			continue;
		}
		if ((!FD_ISSET(rd, &fs) ||
		    (recvfrom(rd, packet, packet_size, 0, NULL, 0)) == -1)) {
			resend_req(sd, offset, send_addr, REASON_TIMEOUT);
			continue;
		}

		mdd_offset = le64toh(packet->mdd_offset);
		mdd_data_size = (size_t)le64toh(packet->mdd_data_size);
		mdd_operation = le32toh(packet->mdd_operation);

		if (debug) {
			printf("received "
			       "offset = %" PRId64 ", "
			       "size = %zd, "
			       "operation = %d, "
			       "expect offset %" PRId64 "\n",
			       mdd_offset,
			       mdd_data_size,
			       mdd_operation,
			       offset);
		}

		if (!(mdd_operation & OP_DATA_DUMP)) {
			printf("\nUnexpected operation %d\n", mdd_operation);
			continue;
		}

		if (mdd_operation & OP_SHUTDOWN) {
			if (debug)
				printf("finishing...\n");
			finished = 1;
		}

		if (mdd_offset > offset) {
			resend_req(sd, offset, send_addr, REASON_MISS);
			finished = 0;
			continue;
		}
		if (mdd_offset < offset) {
			continue;
		}

		memcpy(write_ptr, &packet->mdd_data, mdd_data_size);
		write_ptr += mdd_data_size;
		write_free -= mdd_data_size;
		offset += mdd_data_size;

		if (!debug && !verbose && !quiet)
			print_report(offset, 0);

		/*
		 * If we received more than block_size we can write,
		 * else we loop receiving. Of course if we finished,
		 * we still write even if we have less than block_size.
		 */
		if (!finished && (write_buflen - write_free < block_size))
			continue;

		if (debug)
			printf("committed to write %zd bytes\n",
			    write_buflen - write_free);

		/*
		 * Loop writing block_size of data until
		 * the remaining is smaller than block_size.
		 * If block_size is bigger than the packet size
		 * (the usual case) we should do only one write.
		 */
		do {
			/*
			 * We write block_size except if there is less
			 * to write (finished case)
			 */
			if (write_buflen - write_free < block_size)
				write_size = write_buflen - write_free;
			else
				write_size = block_size;

			if (debug)
				printf("writing %zd bytes at "
				       "offset %" PRId64 "\n",
				    write_size, write_offset);

			(void)lseek(fd, start + write_offset, SEEK_SET);
			written = write(fd, write_buf, write_size);
			if (written != write_size) {
				printf("\nWrite failed "
				       "(offset %" PRId64 "): %s ",
				    write_offset, strerror(errno));
				written = 0;
				finished = 0;

				if (errno == EIO)
					printf("(I/O errors: %d)", ++ioerror);
				printf("\n");

				if (ioerror > 20) {
					printf("Too many I/O errors, ");
					printf("giving up!\n");
					printf("Hit return to continue\n");
					finished = 1;
					fpurge(stdin);
					getc(stdin);
					break;
				}
			}

			/*
			 * If we have some extra data beyond write_size,
			 * report it to the beginning of the buffer
			 */
			write_remain = write_buflen - write_free - write_size;
			if (write_remain > 0) {
				bcopy(write_buf + write_size,
				    write_buf, write_remain);
			} else {
				write_remain = 0;
			}

			if (debug)
				printf("remaining: %d bytes\n", write_remain);

			write_offset += write_size;
			write_ptr = write_buf + write_remain;
			write_free = write_buflen - write_remain;

		} while (write_buflen - write_free >= block_size);
	} while (!finished);

	if (verbose)
		printf("receiver finished\n");

	if ((setsockopt(sd, IPPROTO_IP, IP_DROP_MEMBERSHIP,
	    &ms, sizeof(ms))) == -1) {
		perror("cannot set IP_DROP_MEMBERSHIP");
	}

	if ((setsockopt(rd, IPPROTO_IP, IP_DROP_MEMBERSHIP,
	    &mr, sizeof(mr))) == -1) {
		perror("cannot set IP_DROP_MEMBERSHIP");
	}
	return;
}

#define ST_RUNNING 0
#define ST_RESEND_REQ_SENT 1
static void
resend_req(int sd, off_t offset, struct sockaddr_in send_addr, int reason)
{
	struct mdd_packet packet = { 0 };
	struct timeval now;
	struct timeval wait = { RESEND_WAIT_SEC, RESEND_WAIT_USEC };
	static struct timeval time_to_resend = { 0, 0 };
	static int status = ST_RUNNING;
	static off_t last_requested;

	if ((status == ST_RESEND_REQ_SENT) && (offset != last_requested))
		status = ST_RUNNING;

	if (status == ST_RUNNING) {
		(void)gettimeofday(&now, NULL);
		timeradd(&now, &wait, &time_to_resend);

		packet.mdd_operation = htole32(OP_RESEND_REQ);
		packet.mdd_offset = htole64(offset);
		if (verbose) {
			printf("resend request at %" PRId64 ": ", offset);
			switch(reason) {
			case REASON_TIMEOUT:
				printf("timeout\n");
				break;
			case REASON_MISS:
				printf("packet miss\n");
				break;
			default:
				printf("unknown reason\n");
				break;
			}
		}
		if (sendto(sd, &packet, sizeof(struct mdd_packet),
		    0, (struct sockaddr *)&send_addr,
		    sizeof(send_addr)) != sizeof(struct mdd_packet)) {
			printf("\nResend request to %s failed "
			       "(%" PRId64 "): %s\n",
			    inet_ntoa(send_addr.sin_addr),
			    offset, strerror(errno));
			status = ST_RUNNING;
		} else {
			last_requested = offset;
			status = ST_RESEND_REQ_SENT;
		}
	}
	if (status == ST_RESEND_REQ_SENT) {
		(void)gettimeofday(&now, NULL);
		if (timercmp(&now, &time_to_resend, >)) {
			status = ST_RUNNING;
			if (debug)
				printf("multiple timeout\n");
		}
	}
	return;
}
