(******************************************************************************)
(*                                                                            *)
(*                                    Menhir                                  *)
(*                                                                            *)
(*   Copyright Inria. All rights reserved. This file is distributed under     *)
(*   the terms of the GNU General Public License version 2, as described in   *)
(*   the file LICENSE.                                                        *)
(*                                                                            *)
(******************************************************************************)

open Range
open Located
open Syntax

(* -------------------------------------------------------------------------- *)
(* -------------------------------------------------------------------------- *)

(* Function composition. *)

let (>>) f g x =
  g (f x)

(* -------------------------------------------------------------------------- *)

(* Constructors for the new rule syntax. *)

(* [return pos x] is the semantic action [{x}]. *)

(* No %prec annotation or attribute can be supplied when this form is used. *)

let return pos x : seq_expression =
  let semvars = StringSet.singleton x in
  let action = Action.make 0 semvars [] (IL.EVar x) in
  let raw_action _ _ = action in
  locate pos (EAction (XATraditional raw_action, None, []))

(* -------------------------------------------------------------------------- *)
(* -------------------------------------------------------------------------- *)

(* Converting the new syntax to the old syntax. *)

(* The new syntax is organized in several levels: choice expressions, sequence
   expressions, symbol expressions, action expressions. The code below reflects
   this organization. *)

(* -------------------------------------------------------------------------- *)

(* When a [~] pattern appears at the top level in [~ = e1; e2] and furthermore
   the expression [e1] is a symbol [x1], then this is considered a pun -- that
   is, [~] is sugar for [x1]. We resolve these puns in a first pass, before we
   check that patterns are linear, so a linearity violation that involves [~]
   will be correctly caught. *)

(* There can still remain [~] patterns after puns are resolved, but they stand
   for fresh variables and cannot cause a linearity violation. *)

let rec resolve_puns (e : seq_expression) : seq_expression =
  map (fun e ->
    match e with
    | ECons (SemPatTilde pos, (ESymbol (x1, [], _) as e1), e2)
      when ParserAux.valid_ocaml_identifier x1 ->
        (* This is a pun. Resolve it. *)
        let x1 = locate pos (value x1) in (* optional *)
        ECons (SemPatVar x1, e1, resolve_puns e2)
    | ECons (p1, e1, e2) ->
        ECons (p1, e1, resolve_puns e2)
    | ESingleton _
    | EAction _ ->
        e
  ) e

(* -------------------------------------------------------------------------- *)

(* Checking that a new-syntax pattern is linear,
   i.e., that no variable is bound twice. *)

(* We first build a mapping of variables to the places where they are bound,
   then check that every list in the image of this mapping is a singleton
   list. *)

let check_linearity (ps : pattern list) =

  let rec build (m : ranges StringMap.t) (p : pattern) =
    match p with
    | SemPatVar x ->
        let x = value x
        and pos = position x in
        StringMap.multiple_add x pos m
    | SemPatWildcard
    | SemPatTilde _ ->
        m
    | SemPatTuple ps ->
        List.fold_left build m ps
  in

  let m = List.fold_left build StringMap.empty ps in
  m |> StringMap.iter @@ fun x positions ->
  if List.length positions > 1 then
    Report.Just.error positions
      "The variable %s is bound several times." x

let rec patterns (e : seq_expression) : pattern list =
  let e = value e in
  match e with
  | ECons (p, _, e) ->
      p :: patterns e
  | ESingleton _
  | EAction _ ->
      []

let check_linearity : seq_expression -> unit =
  patterns >> check_linearity

(* -------------------------------------------------------------------------- *)

(* Determining whether a pattern contains a [~] subpattern. *)

let rec tilde_used positions (p : pattern) =
  match p with
  | SemPatVar _
  | SemPatWildcard ->
      positions
  | SemPatTilde pos ->
      pos :: positions
  | SemPatTuple ps ->
      List.fold_left tilde_used positions ps

(* Emitting a warning when a [~] subpattern has been used but the sequence
   expression ends in something other than a point-free semantic action. *)

let tilde_used_warning positions =
  let n = List.length positions in
  if n > 0 then
    let variables_have, tpatterns, wpatterns =
      if n = 1 then "variable has", "a ~ pattern", "a wildcard pattern _"
      else "variables have", "~ patterns", "wildcard patterns _"
    in
    Report.Just.warning positions
      "%s nameless %s been introduced by %s,\n\
       yet this sequence does not end in a point-free semantic action <...>.\n\
       Perhaps %s should be used instead."
      (MString.count n) variables_have tpatterns wpatterns

(* -------------------------------------------------------------------------- *)

(* Converting a new-syntax pattern to an IL pattern. *)

(* Here, [x1] is the variable that holds the semantic value; it is typically
   named [_1], [_2], etc. When we encounter a [~] pattern, we convert it to a
   fresh name, using [x1] as a basis in the generation of this fresh name. *)

let pattern (x1 : identifier) (p : pattern) : IL.pattern =

  let c = ref 1 in
  let fresh () = Printf.sprintf "%s_%d" x1 (MInt.postincrement c) in

  let rec pattern p =
    match p with
    | SemPatVar x ->
        IL.PVar (value x)
    | SemPatWildcard ->
        IL.PWildcard
    | SemPatTilde _ ->
        IL.PVar (fresh())
    | SemPatTuple [] ->
        IL.PUnit
    | SemPatTuple [p] ->
        pattern p
    | SemPatTuple ps ->
        IL.PTuple (List.map pattern ps)

  in
  pattern p

(* [bv accu p] accumulates the bound variables of a pattern [p] produced by
   the above function. The ordering is significant; variables must be
   accumulated left to right (so we get a reversed list). *)

let rec bv accu p =
  match p with
  | IL.PVar x ->
      x :: accu
  | IL.PWildcard ->
      accu
  | IL.PUnit ->
      accu
  | IL.PTuple ps ->
      List.fold_left bv accu ps
  | _ ->
      (* Impossible; not produced above. *)
      assert false

(* -------------------------------------------------------------------------- *)

(* Extracting the attributes of a symbol expression. *)

let attributes (e : symbol_expression) : attributes =
  match e with
  | ESymbol (_, _, attrs) ->
      attrs

(* -------------------------------------------------------------------------- *)

(* As we descend into a sequence expression and prepare to build a production,
   we maintain a context of the following form. *)

type context = {

  (* The position of the complete sequence expression. *)
  pos: Range.range;

  (* The prefix of the production's right-hand side that has been built so far.
     This is reversed list of producers. Every producer carries an identifier,
     which is either user-supplied or auto-generated. *)
  producers: producer list;

  (* The user-supplied names under which the semantic values are known. Not
     every semantic value has such a name, as the user can supply no pattern,
     a wildcard pattern, or a tuple pattern; in either of these cases, there
     is no name for the semantic value. This is a reversed list. Its length
     is equal to the length of the list [producers] above. *)
  uxs: identifier option list;

  (* A set of (independent) bindings that must be wrapped around the semantic
     action. These are typically bindings of the form [let p = x in ...]. *)
  bindings: action -> action;

  (* A tuple of variables that the user has bound, either explicitly or via
     the [~] notation. This is a reversed list. Its length is unrelated to
     the length of the above lists, because one semantic value can be matched
     against a pattern that binds zero, one, or more variables. Once complete,
     this tuple becomes the argument to a point-free semantic action. *)
  tuple: identifier list;

  (* A list of positions indicating where [~] patterns appear. This flag is
     maintained as we descend into a [seq_expression] whose puns have been
     resolved already. Thus, when this list is nonempty, we expect that this
     [seq_expression] ends with a point-free semantic action; otherwise, there
     is no point in using [~], and the user could have used [_] instead. We
     issue a warning if the [seq_expression] does not end with a point-free
     semantic action. *)
  tilde_used: ranges;

}

(* The empty context. *)

let empty pos : context =
  {
    pos;
    producers  = [];
    uxs        = [];
    bindings   = (fun a -> a);
    tuple      = [];
    tilde_used = [];
  }

(* Recording a user-supplied identifier. *)

let user (x : identifier located) : identifier option =
  Some (value x)

let auto : identifier option =
  None

(* Accessing the producers. *)

let producers context : producer list =
  List.rev context.producers

(* Accessing the user-supplied identifiers. *)

let uxs context : identifier option array =
  Array.of_list (List.rev context.uxs)

(* Accessing the tuple. *)

let tuple context : identifier list =
  List.rev context.tuple

(* -------------------------------------------------------------------------- *)

(* OCaml variables for semantic values. *)

(* We do not use a fresh name generator. Instead, we use our [context]
   to generate names of the form [_1], [_2], etc., corresponding to the
   current index in the production that we are building. *)

let semvar context : identifier =
  let i = List.length context.producers + 1 in
  Printf.sprintf "_%d" i

(* -------------------------------------------------------------------------- *)

(* Converting a symbol expression to a parameter. *)

let rec parameter (e : symbol_expression) : parameter =
  match e with
  | ESymbol (sym, args, _attrs) ->
      (* Converting a symbol expression is easy. Note, however, that the
         arguments [args] are arbitrary expressions. *)
      Parameter.apply sym (List.map nested_parameter args)

(* Converting an arbitrary expression to a parameter. *)

and nested_parameter (e : expression) : parameter =
  match value e with
  | EChoice [ Branch (branch, _) ] -> (
      match value branch with
      | ESingleton e ->
          (* A choice with only one branch, whose body is a trivial sequence
             consisting of just a symbol, is simplified on the fly. This is
             important, as it allows us to avoid falling into the default case
             below, where an anonymous rule is generated. For example, when we
             have an application [f(x)] of a parameterized symbol [f] to a
             symbol [x], we don't want an anonymous rule to be generated for
             [x]. That would be wasteful and (more seriously) could cause the
             grammar-expansion-termination check to fail. *)
          parameter e
      | _ ->
          choice_expression e
    )
  | EChoice _ ->
      choice_expression e

(* By default, a choice expression is converted to an anonymous rule. *)

and choice_expression (e : expression) : parameter =
  let branches = productions e in
  ParamAnonymous (locate (position e) branches)

(* Converting the head of a sequence, a pair [x = e1] of a variable [x] and
   a symbol expression [e1], to a producer. *)

and producer x (e1 : symbol_expression) : producer =
  x, parameter e1, attributes e1

(* Converting the head of a sequence, a pair [p = e1] of a pattern [p] and
   a symbol expression [e1], to a context extension. *)

and extend (p : pattern) (e1 : symbol_expression) (context : context) : context =
  match p with
  | SemPatVar x1 ->
      (* The variable [x1] is bound to the semantic value of [e1]. *)
      (* Puns have been resolved already, so they are handled by this code. *)
      { pos        =                       context.pos;
        producers  = producer x1 e1     :: context.producers;
        uxs        = user x1            :: context.uxs;
        bindings   =                       context.bindings;
        tuple      = value x1 :: context.tuple;
        tilde_used = context.tilde_used }
  | _ ->
      (* An arbitrary pattern [p] is used. We bind a variable [x1] to the
         semantic value of [e1], and wrap the semantic action in a binding
         [let p = x1 in ...]. Any [~] subpatterns within [p] are translated to
         fresh identifiers. *)
      let x1 = semvar context in
      let tilde_used = tilde_used context.tilde_used p in
      let p : IL.pattern = pattern x1 p in
      let bvp = bv [] p in
      let binding = Action.bind bvp p x1 in
      let x1 = locate dummy x1 in
      { pos        =                   context.pos;
        producers  = producer x1 e1 :: context.producers;
        uxs        = auto           :: context.uxs;
        bindings   = binding        >> context.bindings;
        tuple      = bv context.tuple p;
        tilde_used }

(* Converting a sequence expression to a production. *)

and production_aux
  (context : context)
  (e : seq_expression)
  (level : production_level)
: parameterized_branch =
  let e, pos = value e, position e in
  match e with

  | ECons (p, e1, e2) ->
      (* A sequence expression [p = e1; e2]. Based on [p] and [e1], extend the
         context, then continue with [e2]. *)
      production_aux (extend p e1 context) e2 level

  | EAction (XATraditional raw_action, prec, attrs) ->
      (* An action expression. This is the end of the sequence. *)
      tilde_used_warning context.tilde_used;
      (* Check that the semantic action seems well-formed. *)
      let action = raw_action `DollarsDisallowed (uxs context) in
      (* Build and return a complete production. *)
      {
        pb_position         = context.pos;
        pb_producers        = producers context;
        pb_action           = context.bindings action;
        pb_prec_annotation  = prec;
        pb_production_level = level;
        pb_attributes       = attrs;
      }

  | EAction (XAPointFree oid, prec, attrs) ->
      (* A point-free semantic action, containing an OCaml identifier [id]
         between angle brackets. This is syntactic sugar for a traditional
         semantic action containing an application of [id] to a tuple of the
         semantic values that have been assigned a name by the user. *)

      (* As a special case, if [oid] is [None], then we must not build
         an application node -- we simply build a tuple. *)

     (* [id] is actually a fragment, not just a string, and this matters when
        there is an OCaml error (e.g., [id] is undeclared, or ill-typed).
        The fragment contains source code location information which allows
        the error to be reported in the source file. *)

      (* Build the tuple as an IL expression. *)
      let semvars = StringSet.of_list (tuple context) in
      let tuple = ILConstruction.(etuple (evars (tuple context))) in
      (* Build an application of [id] to this tuple. *)
      (* We abuse the abstract syntax of IL and build an application node,
         regardless of whether [id] is a (possibly qualified) value, a (possibly
         qualified) data constructor, a polymorphic variant constructor, etc. *)
      let e =
        match oid with
        | Some id ->
            IL.EApp (IL.ETextual id, [tuple])
        | None ->
            tuple
      in
      (* Build a traditional semantic action. *)
      let action = Action.make 0 semvars [] e in
      let raw_action _ _ = action in
      let e = EAction (XATraditional raw_action, prec, attrs) in
      let e = locate pos e in
      (* Reset [tilde_used], to avoid triggering the warning
         via our recursive call. *)
      let context = { context with tilde_used = [] } in
      (* Done. *)
      production_aux context e level

  | ESingleton e ->
      tilde_used_warning context.tilde_used;
      (* When a symbol expression [e] appears as the last element of a sequence,
         this is understood as syntactic sugar for [x = e; {x}], where [x] is a
         fresh variable. *)
      (* Another option would be to view it as sugar for [~ = e; <>]. That would
         also make sense, but would depart from the lambda-calculus convention
         that in a sequence [e1; e2; e3] only the value of the last expression
         is returned. *)
      (* No %prec annotation can be supplied when this sugar is used. *)
      (* No attribute can be supplied when this sugar is used. *)
      let x = semvar context in
      let e = ECons (SemPatVar (locate pos x), e, return pos x) in
      let e = locate pos e in
      production_aux context e level

and production (Branch (e, level) : branch) =
  let e = resolve_puns e in
  check_linearity e;
  let pos = position e in
  production_aux (empty pos) e level

and productions (e : expression) : parameterized_branch list =
  match value e with
  | EChoice branches ->
      List.map production branches

(* -------------------------------------------------------------------------- *)

(* Converting a new rule to an old rule. *)

let rule (rule : rule) : parameterized_rule =
  {
    pr_public      = rule.rule_public;
    pr_inline      = rule.rule_inline;
    pr_nt          = value rule.rule_lhs;
    pr_positions   = [ position rule.rule_lhs ];
    pr_attributes  = rule.rule_attributes;
    pr_parameters  = List.map value rule.rule_formals;
    pr_branches    = productions rule.rule_rhs;
    pr_merge       = rule.rule_merge;
  }
