/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#define MGL_PREFIX
#include "config.h"
#include "mgl2.h"
#include "draw_engine.h"
#include <stdlib.h>

#ifdef USE_LOCAL_MEMSET
#define memset	mgl_memset
#endif
#ifdef USE_LOCAL_MEMMOVE
#define memmove	mgl_memmove
#endif

#ifndef NULL
#define NULL ((void *)0)
#endif

static struct draw_engine dec3kc;

#define lp_color	(s->_pen_color.local_color)

unsigned short *dec3kc_cmap;
int *dec3kc_rcmap;
int *dec3kc_rcmap_alt;

static inline int conv_from_dec3kc(int c0) {
	int c1,col;
	c1 =  ((c0 & 0xf800) >> (11-8) )
	   || ((c0 & 0x07e0) >> ( 7-4) )
	   || ((c0 & 0x001f) >> ( 1-0) ) ;

	col = dec3kc_rcmap[c1];

	if (col < 0) {
		col = dec3kc_rcmap_alt[c1];
	}
	return col;
}

static inline unsigned short conv_to_dec3kc(int col) {
	return dec3kc_cmap[col & 0xfff];
}

#define DEC3KC_REVERSE(c) conv_to_dec3kc(conv_from_dec3kc(c) ^ 0xf)


static struct screen *dec3kc_create_memscreen(int xs,int ys,char *bitmap,int op) {
	struct screen *ret;
	int wb;
	int type = STK_GENERIC_FULLCOLOR | ST_ALLOCED;

	wb = xs * 2;
	ret = de_create_memscreen(xs,ys,bitmap,op);
	if (!ret) return NULL;
	if (!bitmap) {
		bitmap = (char *)malloc(ys * wb);
        	if(!bitmap){
			perror("malloc");
			return (struct screen *)NULL;
        	}
		type |= ST_ALLOCED_BITMAP;
		memset(bitmap,0,ys * wb);
	}

	ret->type |= type;
	ret->de = &dec3kc;
	ret->wbytes = wb;
	ret->bitmap = bitmap;
	return ret;
}

static struct screen *dec3kc_create_subscreen(struct screen *org, int x, int y,
                                int xs, int ys,int opt) {
	struct screen *ret;

	ret = de_create_subscreen(org,x,y,xs,ys,opt);
	if (!ret) return NULL;
	if (opt & CSS_AS_MEMSCREEN) {
		ret->type &= ~ST_SUBSCREEN;
		ret->bitmap = org->bitmap + (ret->off_y * ret->wbytes)
			+ (ret->off_x *2);
		ret->off_y = 0;
		ret->off_x = 0;
	}
	return ret;
}

static void dec3kc_free_screen(struct screen *s) {
	de_free_screen(s);
}


static int dec3kc_get_pixel(struct screen *s,int x, int y,int op) {
	unsigned char *p;
	int c0,c1,col,ret;

	if (x < 0 || x >= s->width
	   || y < 0 || y >= s->height) return 0;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x * 2;
	col = conv_from_dec3kc(*(unsigned short *)p);

	ret = col |= COLOR_DITHER;
	if ((op & BLT_MASKING) && ((ret & 0xfff) == (op & BLT_MASKCOL))) {
		ret = COLOR_TRANSPARENT;
//printf("dec3kc_get_pixel (%d %d) = %08x\n",x,y,ret);
	}
	return ret;
}

static void dec3kc_put_pixel(struct screen *s,int x, int y, int col) {
	char *p;
	if (col & COLOR_TRANSPARENT) return;
	if (col < 0) return;

	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x * 2;

	*(unsigned short *)p = conv_to_dec3kc(col);
	return;
}

static void dec3kc_get_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir,int op
		,struct draw_engine *self) {
	char *p;
	int i;
	int d;
	int r;
	int len4;
	unsigned short *pp;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x*2;
	pp = (unsigned short *)p;

	switch(dir) {
	case DIR_NORTH:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = conv_from_dec3kc(*(unsigned short *)p)
			   | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p += 2;
		} else for (i=0; i<length; i++) {
			*buf++ = conv_from_dec3kc(*(unsigned short *)p)
				 | COLOR_DITHER;
			p += 2;
		}
		break;
	case DIR_WEST:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = conv_from_dec3kc(*(unsigned short *)p)
			 | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p -= s->wbytes;
		} else for (i=0; i<length; i++) {
			*buf++ = conv_from_dec3kc(*(unsigned short *)p)
				 | COLOR_DITHER;
			p -= s->wbytes;
		}
		break;
	case DIR_SOUTH:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = conv_from_dec3kc(*(unsigned short *)p)
			 | COLOR_DITHER;
			if ((r & 0xfff) == (op & BLT_MASKCOL)) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p -= 2;
		} else for (i=0; i<length; i++) {
			*buf++ = conv_from_dec3kc(*(unsigned short *)p)
				 | COLOR_DITHER;
			p -= 2;
		}
		break;
	case DIR_EAST:
		if (op & BLT_MASKING) for (i=0; i<length; i++) {
			r = conv_from_dec3kc(*(unsigned short *)p)
			 | COLOR_DITHER;
			if ((op & BLT_MASKING) && ((r & 0xfff)== (op & BLT_MASKCOL))) {
				r = COLOR_TRANSPARENT;
			}
			*buf++ = r;
			p += s->wbytes;
		} else for (i=0; i<length; i++) {
			*buf++ = conv_from_dec3kc(*(unsigned short *)p)
				 | COLOR_DITHER;
			p += s->wbytes;
		}
		break;
	}
}

static void dec3kc_put_pixstream(struct screen *s,int x, int y,int *buf,int length,int dir
		,struct draw_engine *self) {
	char *p;
	int col,mask;
	int d,i;
	int dx,dy;

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x*2;

	switch(dir) {
	case DIR_NORTH:
	    for (i=0; i<length;i++,x++) {
		col = conv_to_dec3kc(*buf++);
		if (!(col & COLOR_TRANSPARENT)) {
		        *(unsigned short *)p = col;
		}
		p += 2;
	    }
	    break;
	case DIR_WEST:
	    for (i=0; i<length;i++) {
		col = conv_to_dec3kc(*buf++);
		if (!(col & COLOR_TRANSPARENT)) {
		        *(unsigned short *)p = col;
		}
		p -= s->wbytes;
	    }
	    break;
	case DIR_SOUTH:
	    for (i=0; i<length;i++,x--) {
		col = conv_to_dec3kc(*buf++);
		if (!(col & COLOR_TRANSPARENT)) {
		        *(unsigned short *)p = col;
		}
	        p -= 2;
	    }
	    break;
	case DIR_EAST:
	    for (i=0; i<length;i++) {
		col = conv_to_dec3kc(*buf++);
		if (!(col & COLOR_TRANSPARENT)) {
		        *(unsigned short *)p = col;
		}
		p += s->wbytes;
	    }
	    break;
	}
	return;
}

static void dec3kc_set_color(struct screen *s,int col) {
	lp_color = conv_to_dec3kc(col);
	return;
}

static void dec3kc_draw_pixel(struct screen *s,int x, int y) {
	char *p;
	int mask;

	CLIP_POINT(s,x,y);

	x += s->off_x;
	y += s->off_y;
	if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	} else {
		p = (char *)s->bitmap;
	}
	p += y * s->wbytes + x * 2;

	if (pen_color == COLOR_REVERSE)
		*(unsigned short *)p = DEC3KC_REVERSE(*(unsigned short *)p);
	else
		*(unsigned short *)p = lp_color;
      return;
}

static void dec3kc_draw_line_horizontal(struct screen *s,int x1, int y1, int x2, int y2,struct draw_engine *self) {
	char *p;
	int wp;
	int i;
	int off_x,off_y;

	CLIP_HLINE(s,x1,y1,x2,y2);
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;

	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	wp = (x2 + 1 - x1);
	if (wp) {
	     if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	     } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + x1 * 2;

	    if (pen_color == COLOR_REVERSE) {
		for (i=0; i< wp; i++,p+=2) {
		    *(unsigned short *)p = DEC3KC_REVERSE(*(unsigned short *)p);
		}
	    } else {
		mgl_memset2(p, lp_color, wp);
	    }
	}
	s->need_clipping++;
    return;
}

static void dec3kc_clear_screen(struct screen *s,struct draw_engine *self) {
	int x1,y1,x2,y2;
	int off_x,off_y;
	char *p,*pp;
	int wp;
	int i;
	int y;

	x1 = 0;
	x2 = s->width-1;
	y1 = 0;
	y2 = s->height-1;
	s->need_clipping--;

	off_x = s->off_x;
	off_y = s->off_y;
	x1 += off_x;
	x2 += off_x;
	y1 += off_y;
	y2 += off_y;

	wp = (x2 + 1 - x1);
	if (wp) {
	    if (s->type & ST_SUBSCREEN) {
		p = ((struct screen *)(s->bitmap))->bitmap;
		s->wbytes = 
			((struct screen *)(s->bitmap))->wbytes;
	    } else {
		p = (char *)s->bitmap;
	    }
	    p += y1 * s->wbytes + x1*2;
	    for (y=y1; y<=y2; y++) {
		if (pen_color == COLOR_REVERSE) {
		    pp = p;
		    for (i=0; i< wp; i++,pp+=2) {
			*(unsigned short *)pp = 
			            DEC3KC_REVERSE(*(unsigned short *)pp);
		    }
		} else {
		    mgl_memset2(p, lp_color, wp);
		} 
		p += s->wbytes;
	    }
	}
	s->need_clipping++;
}

static void dec3kc_bitblt_copy(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		dp = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		dp = (char *)dst->bitmap;
	}
	if (src->type & ST_SUBSCREEN) {
		sp = ((struct screen *)(src->bitmap))->bitmap;
		src->wbytes = 
			((struct screen *)(src->bitmap))->wbytes;
	} else {
		sp = (char *)src->bitmap;
	}
	dp += dy * dst->wbytes + dx*2;
	sp += sy * src->wbytes + sx*2;
	len = xsize*2;

	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp += dst->wbytes;
		sp += src->wbytes;
	}
	return;
}

static void dec3kc_bitblt_scroll_forward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		p = (char *)dst->bitmap;
	}
	dp = p + dy * dst->wbytes + dx*2;
	sp = p + sy * dst->wbytes + sx*2;
	len = xsize*2;
	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp += dst->wbytes;
		sp += dst->wbytes;
	}
	return;
}

static void dec3kc_bitblt_scroll_backward(struct screen *dst, int dx, int dy
	, struct screen *src, int sx, int sy, int xsize, int ysize, int op
	, struct draw_engine *self) {
	char *p,*dp,*sp;
	int i,len;

	dx += dst->off_x;
	dy += dst->off_y;
	sx += src->off_x;
	sy += src->off_y;
	if (dst->type & ST_SUBSCREEN) {
		p = ((struct screen *)(dst->bitmap))->bitmap;
		dst->wbytes = 
			((struct screen *)(dst->bitmap))->wbytes;
	} else {
		p = (char *)dst->bitmap;
	}
	dy += ysize - 1;
	sy += ysize - 1;
	dp = p + dy * dst->wbytes + dx*2;
	sp = p + sy * dst->wbytes + sx*2;
	len = xsize*2;
	for (i=0; i< ysize; i++) {
		memmove(dp,sp,len);
		dp -= dst->wbytes;
		sp -= dst->wbytes;
	}
	return;
}
void dec3kc_init(int type) {
	dec3kc._create_subscreen = dec3kc_create_subscreen;
	dec3kc._free_screen = dec3kc_free_screen;
	dec3kc._put_pixel = dec3kc_put_pixel;
	dec3kc._get_pixel = dec3kc_get_pixel;

	dec3kc._set_color = dec3kc_set_color;
	dec3kc._draw_pixel = dec3kc_draw_pixel;

	//dec3kc._draw_line = dec3kc_draw_line;

	//dec3kc._draw_line_vertical = dec3kc_draw_line_vertical;
	dec3kc._draw_line_horizontal = dec3kc_draw_line_horizontal;

	//dec3kc._clear_screen = dec3kc_clear_screen;

	dec3kc._get_pixstream = dec3kc_get_pixstream;
	dec3kc._put_pixstream = dec3kc_put_pixstream;
	//dec3kc._put_pixstream_rect = dec3kc_put_pixstream__rect;

	//dec3kc._bitblt = dec3kc_bitblt;
	dec3kc._bitblt_scroll_forward = dec3kc_bitblt_scroll_forward;
	dec3kc._bitblt_scroll_backward = dec3kc_bitblt_scroll_backward;
	dec3kc._bitblt_copy = dec3kc_bitblt_copy;
	//dec3kc._bitblt_reserved_mask = dec3kc_bitblt_reserved_mask;
	//dec3kc._bitblt_reserved_masktile = dec3kc_bitblt_reserver_masktile;
	//dec3kc._bitblt_gen = dec3kc_bitblt_gen;
	setup_draw_engine(&dec3kc,0);
	_create_memscreen[type] = dec3kc_create_memscreen;
}
