/* MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include "config.h"

enum color_type {
	COLTYPE_3k,
	COLTYPE_192,
	COLTYPE_16,
	COLTYPE_4,
} color_type;

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <dlfcn.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <sys/fcntl.h>
#include <sys/time.h>
#include <sys/uio.h>
#include <sys/param.h>
#include <sys/ioctl.h>

#include <errno.h>
#include <sys/stat.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

#ifdef SUPPORT_X_KEYMOUSE
#include <X11/extensions/XTest.h>
#endif


#define W_TITLE      "MGL Emulator"
#define I_TITLE      "MGL Emulator"
#define R_NAME       "MGL Emulator"
#define R_CLASS      "MGL"

#define WHITE      (mgl_getenv("MGL_WHITE")    ?mgl_getenv("MGL_WHITE")    :"#b8cca3")
#define LIGHTGRAY  (mgl_getenv("MGL_LIGHTGRAY")?mgl_getenv("MGL_LIGHTGRAY"):"#95a685")
#define DARKGRAY   (mgl_getenv("MGL_DARKGRAY") ?mgl_getenv("MGL_DARKGRAY") :"#565f4c")
#define BLACK      (mgl_getenv("MGL_BLACK")    ?mgl_getenv("MGL_BLACK")    :"#171914")

#define PIXMAP_W     32
#define PIXMAP_H     32

static Display        *display;
static Window         main_win;
static XEvent         event;
static XTextProperty  w_title_property;  /* window name as a property */
static XTextProperty  i_title_property;  /* window name as a property */
//static Pixmap         pixmap;
static XImage         *ximage;
static XColor         coltab[4096];
static int alloc_color_3k();
static int alloc_color_192();
static int alloc_color_16();
static int alloc_color_4();
static GC             gc;
//static GC             fg_gc;
//static GC             bg_gc;
static char           *ximage_data;
static char           *w_title;
static char           *i_title;
static char           r_name[] = R_NAME;
static char           r_class[] = R_CLASS;
static int            bitsperpixel;
static XImage         *rotated_ximage;

static int can_disp = 0;
#include "mgl2.h"
#include "event_man.h"
#include "mglkey.h"
#include "keymap.h"

int min_keycode;
#define USE_KEYMAP

static struct event_manager emx11;
static int emx11_get_key(int);
static int emx11_key_select(int,fd_set *,int);
static void emx11_refresh();
static void emx11_term();
static int emx11_set_keymode(int);
static void emx11_set_icon(char *icon,char *name);

static int emx11_key_mode = 0;
static int keymap[256];
static void create_keymap();

static int            x_rotated = DIR_NORTH;

#define CURSOR XC_exchange
#define GET_EVENT (ButtonPressMask | KeyPressMask | KeyReleaseMask \
		| ExposureMask | ButtonReleaseMask | PointerMotionMask)

#ifndef MGLDIR
#  define MGLDIR "."
#endif

//extern int pen_color; /* 3: white 2: lightgray 1: darkgray 0: black */

/* internal function */
int (*im_readp)(char *ibuf, int ibufsize);

void im_load(char *name);
extern void put_key(int c);
extern void put_key_xy(struct virtual_key *v,int c,int x,int y);

static int setup_physical_screen();
extern void rp_wrap();

extern long long millitime(void);
static int key_conv(KeySym key);

void *im_handler;

extern int r_rect;

static void x8_put_pixstream(int x,int y,int *buf,int xs,int dir,int op);
static void x16_put_pixstream(int x,int y,int *buf,int xs,int dir,int op);
static void x24_put_pixstream(int x,int y,int *buf,int xs,int dir,int op);
static void x32_put_pixstream(int x,int y,int *buf,int xs,int dir,int op);
static void emx11_mouse_proc(int mode,int dx,int dy);
static int last_x,last_y;
static int x_root,y_root;

extern int mgl_screen_type;
extern int mgl_screen_realtype;
extern int mgl_screen_realwidth;
extern int mgl_screen_bpp;

int emx11_init(int debug) {
int i;
XSizeHints            xsh;
XClassHint            xch;
XWMHints              xwm;
XSetWindowAttributes  xswa;
XGCValues             gcv;
XPixmapFormatValues   *pmf;
int                   bytes;
int                   rotated_bytes;
int                   num;
int		      depth;
    /****** ǥץ쥤Υץ *******/
    display = XOpenDisplay(NULL);
    if(!display){
	fprintf(stderr,"XOpenDisplay fail\n");
        return (-1);
    }

    /****** EMUɥκ *******/
    xswa.background_pixel    = WhitePixel(display, DefaultScreen(display));
    xswa.border_pixel        = BlackPixel(display, DefaultScreen(display));

    main_win = XCreateWindow(display,
               DefaultRootWindow(display),
               0, 0,
              (unsigned int)SCREEN_WIDTH,
              (unsigned int)SCREEN_HEIGHT,
               1,
               DefaultDepth(display, DefaultScreen(display)),
               InputOutput,
               DefaultVisual(display, DefaultScreen(display)),
               CWBackPixel | CWBorderPixel,
               &xswa);

    /* wm size hints */
    xsh.flags  = (PBaseSize | USSize | PMinSize | PMaxSize);
    xsh.width       = SCREEN_WIDTH;
    xsh.height      = SCREEN_HEIGHT;
    xsh.base_width  = SCREEN_WIDTH;
    xsh.base_height = SCREEN_HEIGHT;
    xsh.min_width   = SCREEN_WIDTH;
    xsh.min_height  = SCREEN_HEIGHT;
    xsh.max_width   = SCREEN_WIDTH;
    xsh.max_height  = SCREEN_HEIGHT;

    xch.res_name = r_name;
    xch.res_class = r_class;

    xwm.flags = InputHint;
    xwm.input = True;

    w_title = (char *)malloc(strlen(W_TITLE) + 1);
    if(!w_title){
        perror("malloc");
        return 0;
    }
    strcpy(w_title, W_TITLE);

    /* create window_title_property */
    if(!XStringListToTextProperty(&w_title, 1, &w_title_property)){
        fprintf(stderr,
                "%s: structure w_title_property can't be allocated.\n",
                w_title);
	goto err1;
    }

    i_title = (char *)malloc(strlen(I_TITLE) + 1);
    if(!i_title){
        perror("malloc");
	goto err1;
    }
    strcpy(i_title, I_TITLE);

    /* create icon_title_property */
    if(!XStringListToTextProperty(&i_title, 1, &i_title_property)){
        fprintf(stderr,
                "%s: structure i_title_property can't be allocated.\n",
                i_title);
	goto err2;
    }

    XSetWMProperties(display, main_win, &w_title_property, &i_title_property,
                     NULL, 0, &xsh, &xwm, &xch);

    /*** ƻ뤹륤٥Ȥλؼ ***/
    XSelectInput(display, main_win, GET_EVENT);

    depth = DefaultDepth(display, DefaultScreen(display));
    {
	int max_color = MAX_COLOR;
	char *s;
	int tmp;
	if (s = mgl_getenv("MGL_MAXCOLOR")) {
		if ((sscanf(s,"%d",&tmp) == 1) && (tmp < max_color)) {
//printf("MAX_COLOR change %d to %d \n",MAX_COLOR,tmp);
			max_color = tmp;
		}
	}
	if ((max_color > 192) && (depth >= 15)) {
		color_type = COLTYPE_3k;
	} else if ((max_color > 16) && (depth >= 8)) {
		color_type = COLTYPE_192;
	} else if ((max_color > 4) && (depth >= 8)) {
		color_type = COLTYPE_16;
	} else {
		color_type = COLTYPE_4;
	}
    }
    switch (color_type) {
#if (MAX_COLOR > 192)
    case COLTYPE_3k:
    	if (alloc_color_3k()) {
		goto err2;
    	}
	break;
#endif
#if (MAX_COLOR > 16)
    case COLTYPE_192:
    	if (alloc_color_192()) {
		goto err2;
    	}
	break;
#endif
#if (MAX_COLOR > 4)
    case COLTYPE_16:
    	if (alloc_color_16()) {
		goto err2;
    	}
	break;
#endif
    case COLTYPE_4:
    	if (alloc_color_4()) {
		goto err2;
    	}
	break;
    }

    gcv.function= GXcopy;
    gc = XCreateGC(display, main_win, GCFunction, &gcv);

    switch(depth) {
    case 15:
        bytes = SCREEN_WIDTH * 16 / 8;
        rotated_bytes = SCREEN_HEIGHT * 16 / 8;
        break;
    case 4: /* VGA16 */
        bytes = SCREEN_WIDTH;
        rotated_bytes = SCREEN_HEIGHT;
        break;
    case 8:
    case 16:
        bytes = SCREEN_WIDTH *
                DefaultDepth(display, DefaultScreen(display)) / 8;
        rotated_bytes = SCREEN_HEIGHT *
                        DefaultDepth(display, DefaultScreen(display)) / 8;
        break;
    case 24:
    case 32:
        /* number of supported pixmap formats */
        pmf = XListPixmapFormats(display, &num);
        if(!pmf){
            fprintf(stderr, "supported pixmap formats can't get\n");
	    goto err2;
        }
        bitsperpixel = 0;
        for(i = 0 ; i < num ; i++){
            if(pmf[i].depth == DefaultDepth(display, DefaultScreen(display))){
                bitsperpixel = pmf[i].bits_per_pixel;
                break;
            }
        }
        XFree ((char *) pmf);
        if(!bitsperpixel){
            fprintf(stderr, "this pixmap formats can't supported\n");
	    goto err2;
        }
        bytes = SCREEN_WIDTH * bitsperpixel / 8;
        /* 640 padding no need ? */
        rotated_bytes = SCREEN_HEIGHT * bitsperpixel / 8;
        /* 240 padding no need ? */
        break;
    default:
        fprintf(stderr, "Unsupport Depth\n");
	goto err2;
    }
    ximage_data = (char *)malloc(bytes * SCREEN_HEIGHT);
    if(!ximage_data){
        perror("malloc");
	goto err2;
    }

    /* XImage¤Τ */
    ximage = XCreateImage(display,
             DefaultVisual(display, DefaultScreen(display)),
             DefaultDepth(display, DefaultScreen(display)),
             ZPixmap, 0, ximage_data,
            (unsigned int)SCREEN_WIDTH,
            (unsigned int)SCREEN_HEIGHT,
             BitmapPad(display),
             bytes);
    if(ximage == 0){
        fprintf(stderr, "XImage can't created.\n");
	goto err3;
    }
    /* XImage¤Τ */
    rotated_ximage = XCreateImage(display,
                     DefaultVisual(display, DefaultScreen(display)),
                     DefaultDepth(display, DefaultScreen(display)),
                     ZPixmap, 0, ximage_data,
                    (unsigned int)SCREEN_HEIGHT,
                    (unsigned int)SCREEN_WIDTH,
                     BitmapPad(display),
                     rotated_bytes);
    if(rotated_ximage == 0){
        fprintf(stderr, "XImage can't created.\n");
	goto err4;
    }

    /**** ɥĥդ *****/
    XMapWindow(display, main_win);
    XSync(display, 0);

    if (!mgl_screen_type) {
        if ((color_type == COLTYPE_3k)
		&& _create_memscreen[STK_GENERIC_FULLCOLOR]) {
	    _create_memscreen[STK_NATIVE]
		= _create_memscreen[STK_GENERIC_FULLCOLOR];
	    mgl_screen_type = STK_GENERIC_FULLCOLOR;
	    mgl_screen_bpp = 16;
	} else if ((color_type == COLTYPE_3k || color_type == COLTYPE_192)
		&& _create_memscreen[STK_GENERIC_192COLOR]) {
	    _create_memscreen[STK_NATIVE]
		= _create_memscreen[STK_GENERIC_192COLOR];
	    mgl_screen_type = STK_GENERIC_192COLOR;
	    mgl_screen_bpp = 8;
	} else if ((color_type == COLTYPE_3k || color_type == COLTYPE_192
		|| color_type == COLTYPE_16)
		&& _create_memscreen[STK_GENERIC_16COLOR]) {
	    _create_memscreen[STK_NATIVE]
		= _create_memscreen[STK_GENERIC_16COLOR];
	    mgl_screen_type = STK_GENERIC_16COLOR;
	    mgl_screen_bpp = 4;
	} else {
	    _create_memscreen[STK_NATIVE] 
		= _create_memscreen[STK_GENERIC_4COLOR];
	    mgl_screen_type = STK_GENERIC_4COLOR;
	    mgl_screen_bpp = 2;
	}
	mgl_screen_realtype = mgl_screen_type;
    }
    if (setup_physical_screen(debug)) {
	fprintf(stderr,"setup_physical_screen fail\n");
	return (-1);
    }

    create_keymap();
    mgl_mouse_proc = emx11_mouse_proc;

	emx11._key_select = emx11_key_select;
	emx11._get_key = emx11_get_key;
	emx11._refresh = emx11_refresh;
	emx11._term = emx11_term;
	emx11._set_keymode = emx11_set_keymode;
	emx11._set_icon = emx11_set_icon;
	mgl_em = &emx11;
    vk_init();
    rp_wrap(physical_screen);

   can_disp = 1;

   return 0;

err4:
        XFree(rotated_ximage);
err3:
        XDestroyImage(ximage);
err2:
        free(i_title);
err1:
        free(w_title);
	fprintf(stderr,"can't initialize X\n");
	return -1;
}

extern char *mgl_screen_addr;

extern struct screen *physical_screen,*current_screen;

static int setup_physical_screen(int debug) {

	physical_screen = _create_memscreen[STK_NATIVE](
		SCREEN_WIDTH,SCREEN_HEIGHT,mgl_screen_addr,0);
	current_screen = physical_screen;

	mgl_screen_realwidth = physical_screen->wbytes * 
					   8 / mgl_screen_bpp;
	return 0;
}


static void emx11_term(void) {
    /****** 񸻤β *******/

    XDestroyImage(ximage);
    XFreeGC(display, gc);

    free(w_title);
    free(i_title);

    XFree(rotated_ximage);

    /****** ǥץ쥤Υ *******/
    XCloseDisplay(display);

    can_disp = 0;
}

static int emx11_set_keymode(int mode) {
	emx11_key_mode = mode;
	return 0;
}

static int emx11_get_key(int time_out) {
	fd_set fds;
	int c;

	if (key_buf_cnt) {
		return get_key_nodelay();
	}
	FD_ZERO( &fds );
        FD_SET(0, &fds);
	if (time_out >= 0) {
		time_out *= 100;
	}
	emx11_key_select(1, &fds, time_out);
	if (key_buf_cnt) {
		return get_key_nodelay();
	}
	return (-1);
}

extern long long millitime();
static void xevent_check();
static int    _key_select(int nfds, fd_set *readfds, int timeout);

static int emx11_key_select(nfds, readfds, timeout)
int    nfds;
fd_set *readfds;
int timeout;
{
	int ret;
	fd_set local;
	int ticks;
	long long s,e,t;
	int x,y,b;

	t = s = millitime();
	emx11_refresh();	/* auto matically refresh */
	if (timeout < 0) {
		e = 0x7fffffffffffffffLL;
		ticks = 150;
	} else if (timeout < 150) {
		e = s + timeout;
		ticks = timeout;
	} else {
		e = s + timeout;
		ticks = 150;
	}
	for (;;) {
		local = *readfds;
		if (delayed_key && (t > dk_limit)) {
			put_key(-1);
		}
		ret = _key_select(nfds,&local,ticks);
		t = millitime();
		if (t > e) break;
		if (ret != 0) break;
		if (ticks >= 150) xevent_check();
	}
//printf("key_select %d -> %d (ret = %d)\n",timeout,(int)(millitime() - s),ret);
	*readfds = local;
	return ret;
}


static int _key_select(int nfds, fd_set *readfds, int timeout) {
    int  ret;
    struct timeval to,*top;
    int xfd = -1;

    if (display) {
        xfd = ConnectionNumber(display);
    }
    if (key_buf_cnt) {
	timeout = 0;
    }
    FD_CLR(0, readfds);
    if (xfd > 0) {
	if (xfd+1 > nfds) {
		nfds = xfd+1;
	}
	FD_SET(xfd, readfds);
    }

    top = &to;
    to.tv_sec = timeout / 1000;
    to.tv_usec = (timeout % 1000) * 1000;

    ret = select(nfds, readfds, 0, 0, top);

    if ((ret < 0)) {
        /* error or time out */
        return ret;
    }
    if ((xfd > 0) && FD_ISSET(xfd,readfds)) {
	if (ret > 0) ret--;
    	xevent_check();
        FD_CLR(xfd, readfds);
    }
    if (key_buf_cnt) {
	ret += 1;
        FD_SET(0, readfds);
    }
    return ret;
}

static int key_conv(KeySym key) {
    switch(key){
    case XK_Shift_L:
    case XK_Shift_R:
    case XK_Control_L:
    case XK_Control_R:
    case XK_Caps_Lock:
    case XK_Shift_Lock:
    case XK_Meta_L:
    case XK_Meta_R:
    case XK_Alt_L:
    case XK_Alt_R:
      return 1;
    case XK_Home:	put_key(MK_HOME); return 1;
    case XK_End:	put_key(MK_END); return 1;
    case XK_Up:		put_key(MK_UP); return 1;
    case XK_Down:	put_key(MK_DOWN); return 1;
    case XK_Right:	put_key(MK_RIGHT); return 1;
    case XK_Left:	put_key(MK_LEFT); return 1;
    case XK_Page_Up:	put_key(MK_PAGE_UP); return 1;
    case XK_Page_Down:	put_key(MK_PAGE_DOWN); return 1;
    case XK_Insert:	put_key(MK_INS); return 1;
    case XK_F1:		put_key(MK_F1); return 1;
    case XK_F2:		put_key(MK_F2); return 1;
    case XK_F3:		put_key(MK_F3); return 1;
    case XK_F4:		put_key(MK_F4); return 1;
    case XK_F5:		put_key(MK_F5); return 1;
    case XK_F6:		put_key(MK_F6); return 1;
    case XK_F7:		put_key(MK_F7); return 1;
    case XK_F8:		put_key(MK_F8); return 1;
    case XK_F9:		put_key(MK_F9); return 1;
    case XK_F10:	put_key(MK_F10); return 1;
    case XK_F11:	put_key(MK_F11); return 1;
    case XK_F12:	put_key(MK_F12); return 1;
    }
    return 0;
}

static int mod_conv(int mod,int key,int press,int *is_modifier) {
	int mgl_mod = 0;
	int ismod = 0;	

	if ( mod & 1) { /* Shift */
		mgl_mod |= MGL_SKM_SHIFT;
	}
	if ( mod & 2) { /* Caps_Lock */
	}
	if ( mod & 4) { /* Control */
		mgl_mod |= MGL_SKM_CTRL;
	}
	if ( mod & 8) { /* ALT */
		mgl_mod |= MGL_SKM_ALT;
	}

	if (press) {
	    switch(key) {
	    case XK_Shift_L:
	    case XK_Shift_R:
			ismod = 1;
			mgl_mod |= MGL_SKM_SHIFT;
			break;
	    case XK_Control_L:
	    case XK_Control_R:
			ismod = 1;
			mgl_mod |= MGL_SKM_CTRL;
			break;
	    case XK_Alt_L:
	    case XK_Alt_R:
			ismod = 1;
			mgl_mod |= MGL_SKM_ALT;
			break;
	    }
	} else {
	    switch(key) {
	    case XK_Shift_L:
	    case XK_Shift_R:
			ismod = 1;
			mgl_mod &= ~MGL_SKM_SHIFT;
			break;
	    case XK_Control_L:
	    case XK_Control_R:
			ismod = 1;
			mgl_mod &= ~MGL_SKM_CTRL;
			break;
	    case XK_Alt_L:
	    case XK_Alt_R:
			ismod = 1;
			mgl_mod &= ~MGL_SKM_ALT;
			break;
	    }
	}
	if (is_modifier) *is_modifier = ismod;
	return mgl_mod;
}

static void xevent_check() {
    int             ret_key;
    char            string[10];
    XSizeHints      xsh;
    KeySym          key;
    XComposeStatus  cs;
    int modifier;
    if (display) {
	while (XCheckMaskEvent(display, GET_EVENT,&event)) {
	    int x = event.xbutton.x;
	    int y = event.xbutton.y;
	    int tmp;
	    switch (x_rotated) {
	     default:
	     case DIR_NORTH:
		break;
	     case DIR_EAST:
		tmp = y;
		y = physical_screen->height - 1 - x;
		x = tmp;
		break;
	     case DIR_SOUTH:
		x = physical_screen->width - 1 - x;
		y = physical_screen->height- 1 - y;
		break;
	     case DIR_WEST:
		tmp = physical_screen->width - 1 - y;
		y = x;
		x = tmp;
		break;
	    }

       	    switch(event.type) {
       	    case Expose:
		if (r_rect == 1) {
       			emx11_refresh();
		}
		r_rect = 1;

		{
		    int t_minx, t_maxx, t_miny, t_maxy;

		    t_minx = event.xexpose.x;
		    t_miny = event.xexpose.y;
		    t_maxx = event.xexpose.x + event.xexpose.width - 1;
		    t_maxy = event.xexpose.y + event.xexpose.height - 1;

		    switch (x_rotated) {
		    default:
		    case DIR_NORTH:
			r_minx = t_minx;
			r_maxx = t_maxx;
			r_miny = t_miny;
			r_maxy = t_maxy;
			break;
		    case DIR_EAST:
			r_minx = t_miny;
			r_maxx = t_maxy;
			r_miny = physical_screen->height- 1 - t_maxx;
			r_maxy = physical_screen->height- 1 - t_minx;
			break;
		    case DIR_SOUTH:
			r_maxx = physical_screen->width - 1 - t_minx;
			r_minx = physical_screen->width - 1 - t_maxx;
			r_maxy = physical_screen->height- 1 - t_miny;
			r_miny = physical_screen->height- 1 - t_maxy;
			break;
		    case DIR_WEST:
			r_minx = physical_screen->width - 1 - t_maxy;
			r_maxx = physical_screen->width - 1 - t_miny;
			r_miny = t_minx;
			r_maxy = t_maxx;
			break;
		    };
		}
//printf("expose %d %d %d %d\n",r_minx,r_maxx,r_miny,r_maxy);
       		emx11_refresh();
		break;
       	    case KeyRelease:
		if (emx11_key_mode & MGL_SK_RAW) {
			put_key(mgl_key_conv_raw((((XKeyEvent *)&event)->keycode
				- min_keycode) | 0x80));
		}
#ifdef USE_KEYMAP
		else {
			int c;
			c = mgl_key_conv_raw((((XKeyEvent *)&event)->keycode
				- min_keycode) | 0x80);
			c = mgl_key_trans(c & 0x7f, c & 0x80);
			if (c >= 0) {
				if (emx11_key_mode & MGL_SK_EXTRANSLATED) {
					put_key(c);
				} else if (!(c & MGL_SKM_NOTICE)) {
					put_key(c & ~MGL_SKM_MASK);
				}
			}
		}
#else
		else if (emx11_key_mode & MGL_SK_EXTRANSLATED) {
			int is_modifier;
       			ret_key = XLookupString((XKeyEvent *)&event, string, 1, &key, &cs);
			modifier = mod_conv((((XKeyEvent *)&event)->state) & 0xF,key,0,&is_modifier);
			if (is_modifier)
                        	put_key(MGL_SKM_NOTICE | modifier);
		}
#endif
		break;
       	    case KeyPress:
		if (emx11_key_mode & MGL_SK_RAW) {
			put_key(mgl_key_conv_raw(((XKeyEvent *)&event)->keycode
				- min_keycode));
			break;
		}
#ifdef USE_KEYMAP
		else {
			int c;
			c = mgl_key_conv_raw(((XKeyEvent *)&event)->keycode
				- min_keycode);
			c = mgl_key_trans(c & 0x7f, c & 0x80);
			if (c >= 0) {
				if (emx11_key_mode & MGL_SK_EXTRANSLATED) {
					put_key(c);
				} else if (!(c & MGL_SKM_NOTICE)) {
					put_key(c & ~MGL_SKM_MASK);
				}
			}
		}
#else
#ifdef SOLARIS
                for(ret_key = 0 ; ret_key < 10 ; ret_key++){
                    string[ret_key] = 0x00;
                }
#endif /* SOLARIS */
       		ret_key = XLookupString((XKeyEvent *)&event, string, 1, &key, &cs);
		modifier = mod_conv((((XKeyEvent *)&event)->state) & 0xF,key,1,NULL);
                if(!key_conv(key)) {
		    if (string[0]) {
       			put_key((string[0]&0xff) | modifier);
                    }else{
                        if(key == 0x20){
                            put_key((('@' & 0x1f)) | modifier);
                        } else {
			}
		    }
		} else if (emx11_key_mode & MGL_SK_EXTRANSLATED) {
                        put_key(MGL_SKM_NOTICE | modifier);
		}
#endif
		break;
	    case MotionNotify:
		vk_mouse_move(x,y);
	    	x_root = event.xbutton.x_root;
	    	y_root = event.xbutton.y_root;
		last_x = x;
		last_y = y;
		break;
	    case ButtonRelease:
                if(event.xbutton.button == Button1) {
			if (mgl_modifier_status & (MGL_SKM_MENU|MGL_SKM_RMENU)) {
				vk_mouse_move(x,y);
			} else {
				vk_mouse_up(x,y);
			}
		}
	    	x_root = event.xbutton.x_root;
	    	y_root = event.xbutton.y_root;
		last_x = x;
		last_y = y;
		break;
       	    case ButtonPress:
                if(event.xbutton.button == Button1) {
			if (mgl_modifier_status & (MGL_SKM_MENU|MGL_SKM_RMENU)) {
				vk_mouse_move(x,y);
			} else {
				vk_mouse_down(x,y);
			}
	    		x_root = event.xbutton.x_root;
	    		y_root = event.xbutton.y_root;
			last_x = x;
			last_y = y;
			break;
		}else
                if(event.xbutton.button == Button2){
                    /*  ->  ->  ->  */
                    x_rotated = (x_rotated + 1) % 4;
                }else
                if(event.xbutton.button == Button3){
                    /*  ->  ->  ->  */
                    x_rotated = (x_rotated + 3) % 4;
                }else{
                    break;
                }
                switch(x_rotated){
                case DIR_NORTH:
                case DIR_SOUTH:
                    xsh.flags  = (PBaseSize | USSize | PMinSize | PMaxSize);
                    xsh.width       = SCREEN_WIDTH;
                    xsh.height      = SCREEN_HEIGHT;
                    xsh.base_width  = SCREEN_WIDTH;
                    xsh.base_height = SCREEN_HEIGHT;
                    xsh.min_width   = SCREEN_WIDTH;
                    xsh.min_height  = SCREEN_HEIGHT;
                    xsh.max_width   = SCREEN_WIDTH;
                    xsh.max_height  = SCREEN_HEIGHT;
                    break;
                case DIR_WEST:
                case DIR_EAST:
                    xsh.flags  = (PBaseSize | USSize | PMinSize | PMaxSize);
                    xsh.width       = SCREEN_HEIGHT;
                    xsh.height      = SCREEN_WIDTH;
                    xsh.base_width  = SCREEN_HEIGHT;
                    xsh.base_height = SCREEN_WIDTH;
                    xsh.min_width   = SCREEN_HEIGHT;
                    xsh.min_height  = SCREEN_WIDTH;
                    xsh.max_width   = SCREEN_HEIGHT;
                    xsh.max_height  = SCREEN_WIDTH;
                    break;
                default:
                    x_rotated = DIR_NORTH;
                    xsh.flags  = (PBaseSize | USSize | PMinSize | PMaxSize);
                    xsh.width       = SCREEN_WIDTH;
                    xsh.height      = SCREEN_HEIGHT;
                    xsh.base_width  = SCREEN_WIDTH;
                    xsh.base_height = SCREEN_HEIGHT;
                    xsh.min_width   = SCREEN_WIDTH;
                    xsh.min_height  = SCREEN_HEIGHT;
                    xsh.max_width   = SCREEN_WIDTH;
                    xsh.max_height  = SCREEN_HEIGHT;
                    break;
                }
                XSetWMNormalHints(display, main_win, &xsh);
                XResizeWindow(display, main_win, xsh.width, xsh.height);
                r_rect = 2;
       		emx11_refresh();
		break;
       	    }
	}
    }
}

static void emx11_refresh() {
	int depth;
	int y;
	int xs;
	int minx=0,maxx=0,miny=0,maxy=0; /* write rect */
	int buf[1024];

	if (!can_disp) return;
	if (!r_rect) {
		return;
	}
//printf("refresh %d (%d,%d) - (%d,%d)\n",r_rect,r_minx,r_miny,r_maxx,r_maxy);
	if (r_rect == 2) {
		r_minx = 0;
		r_miny = 0;
		r_maxx = physical_screen->width-1;
		r_maxy = physical_screen->height-1;
	}
	/* Check */
	if(r_minx < 0){
		r_minx = 0;
	}
	if(r_miny < 0){
		r_miny = 0;
	}
	if(r_maxx > physical_screen->width-1){
		r_maxx = physical_screen->width-1;
	}
	if(r_maxy > physical_screen->height-1){
		r_maxy = physical_screen->height-1;
	}
//printf("emx11_refresh (%d %d)-(%d-%d)\n",r_minx,r_miny,r_maxx,r_maxy);
	push_screen(physical_screen);

	depth = (DefaultDepth(display, DefaultScreen(display)));
	if ((depth == 24) || (depth == 32)) {
		depth = bitsperpixel;
	}
	switch (x_rotated) {
	default:
	case DIR_NORTH:
	    minx = r_minx;
	    maxx = r_maxx;
	    miny = r_miny;
	    maxy = r_maxy;
	    xs = maxx - minx + 1;
	    for (y=miny; y<=maxy; y++) {
		get_pixstream(r_minx,r_miny++,buf,xs,DIR_NORTH,0);
		switch(depth) {
		case 4:
		case 8:
			x8_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 15:
		case 16:
			x16_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 24:
			x24_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 32:
			x32_put_pixstream(minx,y,buf,xs,0,0);
			break;
		}
	    }
	    break;
	case DIR_EAST:
	    minx = physical_screen->height - 1 - r_maxy;
	    maxx = physical_screen->height - 1 - r_miny;
	    miny = r_minx;
	    maxy = r_maxx;
	    xs = maxx - minx + 1;
	    for (y=miny; y<=maxy; y++) {
		get_pixstream(r_minx++,r_maxy,buf,xs,DIR_WEST,0);
		switch(depth) {
		case 4:
		case 8:
			x8_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 15:
		case 16:
			x16_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 24:
			x24_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 32:
			x32_put_pixstream(minx,y,buf,xs,0,0);
			break;
		}
	    }
	    break;
	case DIR_SOUTH:
	    maxx = physical_screen->width - 1 - r_minx;
	    minx = physical_screen->width - 1 - r_maxx;
	    maxy = physical_screen->height- 1 - r_miny;
	    miny = physical_screen->height- 1 - r_maxy;
	    xs = maxx - minx + 1;
	    for (y=miny; y<=maxy; y++) {
		get_pixstream(r_maxx,r_maxy--,buf,xs,DIR_SOUTH,0);
		switch(depth) {
		case 4:
		case 8:
			x8_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 15:
		case 16:
			x16_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 24:
			x24_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 32:
			x32_put_pixstream(minx,y,buf,xs,0,0);
			break;
		}
	    }
	    break;
	case DIR_WEST:
	    minx = r_miny;
	    maxx = r_maxy;
	    miny = physical_screen->width - 1 -r_maxx;
	    maxy = physical_screen->width - 1 -r_minx;
	    xs = maxx - minx + 1;
	    for (y=miny; y<=maxy; y++) {
		get_pixstream(r_maxx--,r_miny,buf,xs,DIR_EAST,0);
		switch(depth) {
		case 4:
		case 8:
			x8_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 15:
		case 16:
			x16_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 24:
			x24_put_pixstream(minx,y,buf,xs,0,0);
			break;
		case 32:
			x32_put_pixstream(minx,y,buf,xs,0,0);
			break;
		}
	    }
	    break;
	}

	pop_screen();
	/* XImage¤Τ򥦥ɥž */
	if ((x_rotated & 1) == 0) {
	    XPutImage(display, main_win, gc, ximage,
		minx, miny,
		minx, miny,
		maxx - minx +1,
		maxy - miny +1);
	} else {
	    XPutImage(display, main_win, gc, rotated_ximage,
		minx, miny,
		minx, miny,
		maxx - minx +1,
		maxy - miny +1);
	}
	XSync(display, 0);
	r_rect = r_minx = r_miny = r_maxx = r_maxy = 0;
}


#include "mglcol.h"

#define R_MAX	0xb8
#define R_MIN	0x17
#define G_MAX	0xcc
#define G_MIN	0x19
#define B_MAX	0xc3
#define B_MIN	0x14

static int alloc_color_4() {
    Status                status;
    XColor		      tmpc;

    /* 0: white 1: lightgray 2: darkgray 3: black */
    status = XAllocNamedColor(display,
                              DefaultColormap(display, DefaultScreen(display)),
                              WHITE, &coltab[3], &tmpc);
    if(!status){
        fprintf(stderr, "white color can't created\n");
	goto err2;
    }

    status = XAllocNamedColor(display,
                              DefaultColormap(display, DefaultScreen(display)),
                              LIGHTGRAY, &coltab[2], &tmpc);
    if(!status){
        fprintf(stderr, "lightgray color can't created\n");
	goto err2;
    }

    status = XAllocNamedColor(display,
                              DefaultColormap(display, DefaultScreen(display)),
                              DARKGRAY, &coltab[1], &tmpc);
    if(!status){
        fprintf(stderr, "darkgray color can't created\n");
	goto err2;
    }

    status = XAllocNamedColor(display,
                              DefaultColormap(display, DefaultScreen(display)),
                              BLACK, &coltab[0], &tmpc);
    if(!status){
        fprintf(stderr, "black color can't created\n");
	goto err2;
    }
    return 0;
err2:
    return 1;
}

static int alloc_color_16() {
    XColor         tmp;
    char rgb[24];
    int i,j;
    int hue,sat,bri;
    int c,c2,r,g,b;
    int status; 
    int alloc_num = 0;
    for (i=0; i<16; i++) {
	  c = i;
	  c2 = mc_to_rgb(c);
	  unpackRGB(c2,r,g,b);
#ifdef DULLY_DISPLAY
	  r = (R_MAX-R_MIN)*r /0xf + R_MIN;
	  g = (G_MAX-G_MIN)*g /0xf + G_MIN;
	  b = (B_MAX-B_MIN)*b /0xf + B_MIN;
#else
	  r = ((r << 4) | r) & 0xff;
	  g = ((g << 4) | g ) & 0xff;
	  b = ((b << 4) | b ) & 0xff;
#endif
	  sprintf(rgb,"#%02x%02x%02x",r,g,b);

          status = XAllocNamedColor(display,
                      DefaultColormap(display, DefaultScreen(display)),
                      rgb, &coltab[i], &tmp);
	  if (!status) {
        	fprintf(stderr, "color can't allocate %d/192\n",alloc_num);
		return -1;
	  }
	  alloc_num++;
    }
    return 0;
}

static int alloc_color_192() {
    XColor         tmp;
    char rgb[24];
    int i,j;
    int hue,sat,bri;
    int c,c2,r,g,b;
    int status; 
    int alloc_num = 0;
    for (i=0; i<192; i++) {
	  c = CONV_FROM_COL192(i);
	  unpackMC(c,hue,sat,bri);
	  if ((sat == 0) && (hue != 0)) {
/* skipping 44 col */
		continue;
	  }
	  if (sat > bri) {
/* skipping 72 col */
		continue;
	  }
	  c2 = mc_to_rgb(c);
	  unpackRGB(c2,r,g,b);
#ifdef DULLY_DISPLAY
	  r = (R_MAX-R_MIN)*r /0xf + R_MIN;
	  g = (G_MAX-G_MIN)*g /0xf + G_MIN;
	  b = (B_MAX-B_MIN)*b /0xf + B_MIN;
#else
	  r = ((r << 4) | r) & 0xff;
	  g = ((g << 4) | g ) & 0xff;
	  b = ((b << 4) | b ) & 0xff;
#endif
	  sprintf(rgb,"#%02x%02x%02x",r,g,b);

          status = XAllocNamedColor(display,
                      DefaultColormap(display, DefaultScreen(display)),
                      rgb, &coltab[i], &tmp);
	  if (!status) {
        	fprintf(stderr, "color can't allocate %d/192\n",alloc_num);
		return -1;
	  }
	  alloc_num++;
    }
    for (i=0; i<192; i++) {
	  c = CONV_FROM_COL192(i);
	  unpackMC(c,hue,sat,bri);
	  if ((sat == 0) && (hue != 0)) {
		hue = 0;
		c2 = packMC(hue,sat,bri);
		j = CONV_TO_COL192(c2);
		coltab[i] = coltab[j];
	  }
	  if (sat > bri) {
		sat = bri;
		c2 = packMC(hue,sat,bri);
		j = CONV_TO_COL192(c2);
		coltab[i] = coltab[j];
	  }
    }
    for (; i<256; i++) {
	coltab[i] = coltab[0];
    }
    return 0;
}

static int alloc_color_3k() {
    XColor         tmp;
    char rgb[24];
    int i,j;
    int hue,sat,bri;
    int c,c2,r,g,b;
    int status; 
    int alloc_num = 0;
    for (i=0; i<1024*3; i++) {
	  c = i;
	  unpackMC(c,hue,sat,bri);
	  c2 = mc_to_rgb(c);
	  unpackRGB(c2,r,g,b);
#ifdef DULLY_DISPLAY
	  r = (R_MAX-R_MIN)*r /0xf + R_MIN;
	  g = (G_MAX-G_MIN)*g /0xf + G_MIN;
	  b = (B_MAX-B_MIN)*b /0xf + B_MIN;
#else
	  r = ((r << 4) | r) & 0xff;
	  g = ((g << 4) | g ) & 0xff;
	  b = ((b << 4) | b ) & 0xff;
#endif
	  sprintf(rgb,"#%02x%02x%02x",r,g,b);

          status = XAllocNamedColor(display,
                      DefaultColormap(display, DefaultScreen(display)),
                      rgb, &coltab[i], &tmp);
	  if (!status) {
        	fprintf(stderr, "color can't allocate %d/3192\n",alloc_num);
		return -1;
	  }
	  alloc_num++;
    }
    for (; i<4096; i++) {
	coltab[i] = coltab[0];
    }
    return 0;
}

static void x8_put_pixstream(int x,int y,int *buf,int xs,int dir,int op) {
	int i;
	char *pcolor;
	int pixel;

	if (!(x_rotated & 1))
		i = y*physical_screen->width + x;
	else
		i = y*physical_screen->height + x;

        pcolor = (unsigned char *)&ximage_data[i];
	switch(color_type) {
#if (MAX_COLOR > 192)
	case COLTYPE_3k:
	    for (x=0; x<xs; x++,pcolor++) {
              	*pcolor = coltab[buf[x] & 0xfff].pixel;
	     }
	     break;
#endif
#if (MAX_COLOR > 16)
	case COLTYPE_192:
	    for (x=0; x<xs; x++,pcolor++) {
		pixel = CONV_TO_COL192(buf[x] & 0xfff);
              	*pcolor = coltab[pixel].pixel;
	     }
	     break;
#endif
#if (MAX_COLOR > 4)
	case COLTYPE_16:
	    for (x=0; x<xs; x++,pcolor++) {
		pixel = buf[x] & 0xf;
              	*pcolor = coltab[pixel].pixel;
	     }
	     break;
#endif
	case COLTYPE_4:
	    for (x=0; x<xs; x++,pcolor++) {
		pixel = CONV_TO_COL4(buf[x] & 0xfff);
              	*pcolor = coltab[pixel].pixel;
	     }
	     break;
	}
}

static void x16_put_pixstream(int x,int y,int *buf,int xs,int dir,int op) {
	int i;
	unsigned short *pcolor;
	int pixel;

	if (!(x_rotated & 1))
		i = y*physical_screen->width + x;
	else
		i = y*physical_screen->height + x;

        pcolor = (unsigned short *)&ximage_data[i * 2];
	switch(color_type) {
#if (MAX_COLOR > 192)
	case COLTYPE_3k:
	    for (x=0; x<xs; x++,pcolor++) {
              	*pcolor = coltab[buf[x] & 0xfff].pixel;
	     }
	     break;
#endif
#if (MAX_COLOR > 16)
	case COLTYPE_192:
	    for (x=0; x<xs; x++,pcolor++) {
		pixel = CONV_TO_COL192(buf[x] & 0xfff);
              	*pcolor = coltab[pixel].pixel;
	     }
	     break;
#endif
#if (MAX_COLOR > 4)
	case COLTYPE_16:
	    for (x=0; x<xs; x++,pcolor++) {
		pixel = buf[x] & 0xf;
              	*pcolor = coltab[pixel].pixel;
	     }
	     break;
#endif
	case COLTYPE_4:
	    for (x=0; x<xs; x++,pcolor++) {
		pixel = CONV_TO_COL4(buf[x] & 0xfff);
              	*pcolor = coltab[pixel].pixel;
	     }
	     break;
	}
}

static void x24_put_pixstream(int x,int y,int *buf,int xs,int dir,int op) {
	int i;
	char *pcolor;
	int pixel;

	if (!(x_rotated & 1))
		i = y*physical_screen->width + x;
	else
		i = y*physical_screen->height + x;

        pcolor = (unsigned char *)&ximage_data[i * 3];

	if(ImageByteOrder(display)) {
	    switch(color_type) {
#if (MAX_COLOR > 192)
	    case COLTYPE_3k:
		for (x=0; x<xs; x++) {
              		*pcolor++ = coltab[buf[x] & 0xfff].red;
              		*pcolor++ = coltab[buf[x] & 0xfff].green;
              		*pcolor++ = coltab[buf[x] & 0xfff].blue;
	         }
	         break;
#endif
#if (MAX_COLOR > 16)
	    case COLTYPE_192:
		for (x=0; x<xs; x++) {
			pixel = CONV_TO_COL192(buf[x] & 0xfff);
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].blue;
	         }
	         break;
#endif
#if (MAX_COLOR > 4)
	    case COLTYPE_16:
		for (x=0; x<xs; x++) {
			pixel = buf[x] & 0xf;
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].blue;
	         }
	         break;
#endif
	    case COLTYPE_4:
		for (x=0; x<xs; x++) {
			pixel = CONV_TO_COL4(buf[x] & 0xfff);
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].blue;
	         }
	         break;
	    }
	} else {
	    switch(color_type) {
#if (MAX_COLOR > 192)
	    case COLTYPE_3k:
		for (x=0; x<xs; x++) {
              		*pcolor++ = coltab[buf[x] & 0xfff].blue;
              		*pcolor++ = coltab[buf[x] & 0xfff].green;
              		*pcolor++ = coltab[buf[x] & 0xfff].red;
	         }
	         break;
#endif
#if (MAX_COLOR > 16)
	    case COLTYPE_192:
		for (x=0; x<xs; x++) {
			pixel = CONV_TO_COL192(buf[x] & 0xfff);
              		*pcolor++ = coltab[pixel].blue;
              		*pcolor++ = coltab[pixel].green;
             		*pcolor++ = coltab[pixel].red;
	         }
	         break;
#endif
#if (MAX_COLOR > 4)
	    case COLTYPE_16:
		for (x=0; x<xs; x++) {
			pixel = buf[x] & 0xf;
              		*pcolor++ = coltab[pixel].blue;
              		*pcolor++ = coltab[pixel].green;
             		*pcolor++ = coltab[pixel].red;
	         }
	         break;
#endif
	    case COLTYPE_4:
		for (x=0; x<xs; x++) {
			pixel = CONV_TO_COL4(buf[x] & 0xfff);
              		*pcolor++ = coltab[pixel].blue;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].red;
	         }
	         break;
	    }
	}
}

static void x32_put_pixstream(int x,int y,int *buf,int xs,int dir,int op) {
	int i;
	char *pcolor;
	int pixel;

	if (!(x_rotated & 1))
		i = y*physical_screen->width + x;
	else
		i = y*physical_screen->height + x;

        pcolor = (unsigned char *)&ximage_data[i * 4];

	if(ImageByteOrder(display)) {
	    switch(color_type) {
#if (MAX_COLOR > 192)
	    case COLTYPE_3k:
		for (x=0; x<xs; x++) {
              		*pcolor++ = 0;
              		*pcolor++ = coltab[buf[x] & 0xfff].red;
              		*pcolor++ = coltab[buf[x] & 0xfff].green;
              		*pcolor++ = coltab[buf[x] & 0xfff].blue;
	         }
	         break;
#endif
#if (MAX_COLOR > 16)
	    case COLTYPE_192:
		for (x=0; x<xs; x++) {
			pixel = CONV_TO_COL192(buf[x] & 0xfff);
              		*pcolor++ = 0;
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].blue;
	         }
	         break;
#endif
#if (MAX_COLOR > 4)
	    case COLTYPE_16:
		for (x=0; x<xs; x++) {
			pixel = buf[x] & 0xf;
              		*pcolor++ = 0;
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].blue;
	         }
	         break;
#endif
	    case COLTYPE_4:
		for (x=0; x<xs; x++) {
			pixel = CONV_TO_COL4(buf[x] & 0xfff);
              		*pcolor++ = 0;
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].blue;
	         }
	         break;
	    }
	} else {
	    switch(color_type) {
#if (MAX_COLOR > 192)
	    case COLTYPE_3k:
		for (x=0; x<xs; x++) {
              		*pcolor++ = coltab[buf[x] & 0xfff].blue;
              		*pcolor++ = coltab[buf[x] & 0xfff].green;
              		*pcolor++ = coltab[buf[x] & 0xfff].red;
              		*pcolor++ = 0;
	         }
	         break;
#endif
#if (MAX_COLOR > 16)
	    case COLTYPE_192:
		for (x=0; x<xs; x++) {
			pixel = CONV_TO_COL192(buf[x] & 0xfff);
              		*pcolor++ = coltab[pixel].blue;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = 0;
	         }
	         break;
#endif
#if (MAX_COLOR > 4)
	    case COLTYPE_16:
		for (x=0; x<xs; x++) {
			pixel = buf[x] & 0xf;
              		*pcolor++ = coltab[pixel].blue;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = 0;
	         }
	         break;
#endif
	    case COLTYPE_4:
		for (x=0; x<xs; x++) {
			pixel = CONV_TO_COL4(buf[x] & 0xfff);
              		*pcolor++ = coltab[pixel].blue;
              		*pcolor++ = coltab[pixel].green;
              		*pcolor++ = coltab[pixel].red;
              		*pcolor++ = 0;
	         }
	         break;
	    }
	}
}

static void emx11_set_icon(char *icon,char *name) {
    int s1,s2;
    if (name) {
	if (w_title) free(w_title);
	if (i_title) free(i_title);
	w_title = malloc(strlen(name)+1);
	i_title = malloc(strlen(name)+1);
	if (w_title && i_title) {
		strcpy(w_title,name);
		strcpy(i_title,name);
    		s1 = XStringListToTextProperty(&w_title,1,&w_title_property);
    		s2 = XStringListToTextProperty(&i_title, 1, &i_title_property);
		if (s1 && s2) {
    			XSetWMProperties(display, main_win
				, &w_title_property, &i_title_property
                     		,NULL, 0, NULL, NULL, NULL);
		}
	}
    }
}

static void emx11_mouse_proc(int mode,int dx,int dy) {
#ifdef SUPPORT_X_KEYMOUSE
	if (mode == 0) { /* down */
		vk_mouse_down(last_x,last_y);
	} else if (mode == 1) { /* move */
	    int x = x_root + dx;
	    int y = y_root + dy;
	    XTestFakeMotionEvent(display,DefaultScreen(display),x,y,0);
	} else if (mode == 2) { /* up */
		vk_mouse_up(last_x,last_y);
	}
#endif
}

static int
sym_convert(int sym) {
	switch(sym) {
		case XK_1:		return '1';
		case XK_2:		return '2';
		case XK_3:		return '3';
		case XK_4:		return '4';
		case XK_5:		return '5';
		case XK_6:		return '6';
		case XK_7:		return '7';
		case XK_8:		return '8';
		case XK_9:		return '9';
		case XK_0:		return '0';
		case XK_a:		return 'a';
		case XK_b:		return 'b';
		case XK_c:		return 'c';
		case XK_d:		return 'd';
		case XK_e:		return 'e';
		case XK_f:		return 'f';
		case XK_g:		return 'g';
		case XK_h:		return 'h';
		case XK_i:		return 'i';
		case XK_j:		return 'j';
		case XK_k:		return 'k';
		case XK_l:		return 'l';
		case XK_m:		return 'm';
		case XK_n:		return 'n';
		case XK_o:		return 'o';
		case XK_p:		return 'p';
		case XK_q:		return 'q';
		case XK_r:		return 'r';
		case XK_s:		return 's';
		case XK_t:		return 't';
		case XK_u:		return 'u';
		case XK_v:		return 'v';
		case XK_w:		return 'w';
		case XK_x:		return 'x';
		case XK_y:		return 'y';
		case XK_z:		return 'z';
		case XK_A:		return 'A';
		case XK_B:		return 'B';
		case XK_C:		return 'C';
		case XK_D:		return 'D';
		case XK_E:		return 'E';
		case XK_F:		return 'F';
		case XK_G:		return 'G';
		case XK_H:		return 'H';
		case XK_I:		return 'I';
		case XK_J:		return 'J';
		case XK_K:		return 'K';
		case XK_L:		return 'L';
		case XK_M:		return 'M';
		case XK_N:		return 'N';
		case XK_O:		return 'O';
		case XK_P:		return 'P';
		case XK_Q:		return 'Q';
		case XK_R:		return 'R';
		case XK_S:		return 'S';
		case XK_T:		return 'T';
		case XK_U:		return 'U';
		case XK_V:		return 'V';
		case XK_W:		return 'W';
		case XK_X:		return 'X';
		case XK_Y:		return 'Y';
		case XK_Z:		return 'Z';

		case XK_Caps_Lock:	return MGL_SKM_CAPS;
		case XK_Shift_L:	return MGL_SKM_SHIFT;
		case XK_Shift_R:	return MGL_SKM_RSHIFT;
		case XK_Alt_L:		return MGL_SKM_ALT;
		case XK_Alt_R:		return MGL_SKM_RALT;
		case XK_Control_L:	return MGL_SKM_CTRL;
		case XK_Control_R:	return MGL_SKM_RCTRL;

		case XK_exclam:		return '!';
		case XK_quotedbl:	return '"';
		case XK_numbersign:	return '#';
		case XK_dollar:		return '$';
		case XK_percent:	return '%';
		case XK_ampersand:	return '&';
		case XK_apostrophe:	return '\'';
		case XK_asterisk:	return '*';
		case XK_parenleft:	return '(';
		case XK_parenright:	return ')';
		case XK_minus:		return '-';
		case XK_plus:		return '+';
		case XK_equal:		return '=';
		case XK_asciicircum:	return '^';
		case XK_asciitilde:	return '~';
		case XK_at:		return '@';
		case XK_bracketleft:	return '[';
		case XK_bracketright:	return ']';
		case XK_braceleft:	return '{';
		case XK_braceright:	return '}';
		case XK_semicolon:	return ';';
		case XK_colon:		return ':';
		case XK_comma:		return ',';
		case XK_less:		return '<';
		case XK_period:		return '.';
		case XK_greater:	return '>';
		case XK_slash:		return '/';
		case XK_question:	return '?';
		case XK_space:		return ' ';
		case XK_bar:		return '|';
		case XK_backslash:	return '\\';
		case XK_underscore:	return '_';
		case XK_grave:		return '`';

		case XK_Escape:		return '\e';
		case XK_BackSpace:	return '\b';
		case XK_Tab:		return '\t';
		case XK_Return:		return '\r';

		case XK_F1:		return MK_F1;
		case XK_F2:		return MK_F2;
		case XK_F3:		return MK_F3;
		case XK_F4:		return MK_F4;
		case XK_F5:		return MK_F5;
		case XK_F6:		return MK_F6;
		case XK_F7:		return MK_F7;
		case XK_F8:		return MK_F8;
		case XK_F9:		return MK_F9;
		case XK_F10:		return MK_F10;
		case XK_F11:		return MK_F11;
		case XK_F12:		return MK_F12;
		case XK_Left:		return MK_LEFT;
		case XK_Right:		return MK_RIGHT;
		case XK_Up:		return MK_UP;
		case XK_Down:		return MK_DOWN;
		case XK_Insert:		return MK_INS;
		case XK_Delete:		return 0x7f;
		case XK_Prior:		return MK_PAGE_UP;
		case XK_Next:		return MK_PAGE_DOWN;
		case XK_End:		return MK_END;

		case XK_KP_0:		return MKE_0;
		case XK_KP_1:		return MKE_1;
		case XK_KP_2:		return MKE_2;
		case XK_KP_3:		return MKE_3;
		case XK_KP_4:		return MKE_4;
		case XK_KP_5:		return MKE_5;
		case XK_KP_6:		return MKE_6;
		case XK_KP_7:		return MKE_7;
		case XK_KP_8:		return MKE_8;
		case XK_KP_9:		return MKE_9;
		case XK_KP_Add:		return MKE_PLUS;
		case XK_KP_Subtract:	return MKE_MINUS;
		case XK_KP_Multiply:	return MKE_MULT;
		case XK_KP_Divide:	return MKE_DIV;
		case XK_Break:		return MKE_BREAK;
		case XK_Zenkaku_Hankaku: return MKE_HANZEN;
		case XK_Kanji:		return MKE_KANJI;
		case XK_Henkan_Mode:	return MKE_HENKAN;
		case XK_Muhenkan:	return MKE_MUHENKAN;
		case XK_Num_Lock:	return MK_NUMLOCK;
		case XK_Scroll_Lock:	return MKE_SCRLOCK;
		case XK_Select:		return MKE_SELECT;
		case XK_KP_Delete:	return MK_DEL;
		case XK_KP_Home:	return MKE_HOME;
		case XK_Print:		return MKE_PRINT;
		case XK_Pause:		return MKE_PAUSE;
		case XK_KP_End:		return MKE_END;
		case XK_KP_Enter:	return MKE_RETURN;
		case XK_KP_Up:		return MKE_UP;
		case XK_KP_Down:	return MKE_DOWN;
		case XK_KP_Right:	return MKE_RIGHT;
		case XK_KP_Left:	return MKE_LEFT;
		case XK_KP_Prior:	return MKE_PAGE_UP;
		case XK_KP_Next:	return MKE_PAGE_DOWN;
		case XK_KP_Insert:	return MKE_INS;
		case XK_Execute:	return MKE_EXECUTE;
		case XK_KP_Begin:	return MKE_BEGIN;

		case XK_F13:		return MKE_F13;
		case XK_F14:		return MKE_F14;
		case XK_F15:		return MKE_F15;
		case XK_F16:		return MKE_F16;
		case XK_F17:		return MKE_F17;
		case XK_F18:		return MKE_F18;
		case XK_F19:		return MKE_F19;
		case XK_F20:		return MKE_F20;
	}
	return MK_NONE;
}

static void create_keymap() {
	KeySym *sy;
	int i,n;
	int max_keycode;
	int sym1,sym2,sym3;

	mk_init();
	XDisplayKeycodes(display,&min_keycode,&max_keycode);

	for (i=min_keycode; i<= max_keycode; i++) {
                if((i - min_keycode) >= 128){
                        fprintf(stderr, "Warning! X keymap is defined over 128 chars.\n");
                        break;
                }

		sy = XGetKeyboardMapping(display,i,1,&n);
		if (!sy) {
			continue;
		}
		sym1 = MK_NONE;
		sym2 = MK_NONE;
		sym3 = MK_NONE;

		sym1 = sym_convert(sy[0]);
		sym2 = sym_convert(sy[1]);
		if (sym2 == MK_NONE) sym2 = sym1;

		if ((sym1 == MK_F9) && (sym2 == MK_F9)) {
			sym1 = MKE_SWITCH_WINDOW;
		}
		if ((sym1 == MK_F10) && (sym2 == MK_F10)) {
			sym1 = MKE_SWITCH_FOCUS;
		}
		if ((sym1 == MKE_KANJI) && (sym2 == MKE_KANJI)) {
			sym1 = sym2 = MKE_SWITCH_WINDOW;
		}
		if ((sym1 == MKE_HENKAN) && (sym2 == MKE_HENKAN)) {
			sym1 = sym2 = MKE_SWITCH_WINDOW;
		}
		if ((sym1 == MKE_MUHENKAN) && (sym2 == MKE_MUHENKAN)) {
			sym1 = sym2 = MKE_SWITCH_FOCUS;
		}
		if ((sym1 != MK_NONE) && (sym1 & MGL_SKM_MASK)) {
			sym3 = sym1;
		}
		mgl_keymap[i-min_keycode][0] = sym1;
		mgl_keymap[i-min_keycode][1] = sym2;
		mgl_keymap[i-min_keycode][2] = sym3;
	}

	/* for XFree86 ?? */
	if (mgl_keymap[115 - min_keycode][0] == MK_NONE) {
		mgl_keymap[115 - min_keycode][0] = MGL_SKM_MENU;
		mgl_keymap[115 - min_keycode][1] = MGL_SKM_MENU;
	}
	if (mgl_keymap[116 - min_keycode][0] == MK_NONE) {
		mgl_keymap[116 - min_keycode][0] = MGL_SKM_RMENU;
		mgl_keymap[116 - min_keycode][1] = MGL_SKM_RMENU;
	}


#if 0
mk_dump_keymap("xkeymap");
#endif
	mk_create_raw_keymap();
#if 0
mk_dump_keymap("mglmap ");
#endif
}
