/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#include <unistd.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <linux/fb.h>
#include <linux/vt.h>
#include <fcntl.h>

#ifndef MAP_FAILED
#define MAP_FAILED  ((void *)-1)
#endif

#ifdef SUPPORT_LINUXFB_CFB2
# include "fb2.h"
#endif
#ifdef SUPPORT_LINUXFB_CFB4
# include "fb4.h"
#endif
#ifdef SUPPORT_LINUXFB_CFB8
# include "fb8.h"
#endif
#ifdef SUPPORT_LINUXFB_CFB16
# include "fb16.h"
#endif

#include "mgl_md.h"

#define USE_KEYMAP

#if ( MGL_MACHINE == MGL_MACHINE_MG_LINUX )
#  include "md_mobilegear.h"
#  undef MGL_NO_MOUSE
#endif

#ifdef MGL_MOUSE_MG2
#  include <linux/tpanel.h>
#  include "tptrans.h"
#  include "md_linuxvr.h"
#endif

#ifdef MGL_MOUSE_PSION5
#define UNIFY_PLAIN_AND_ALTGR
#  define _LINUX_TYPES_H
#  include <asm/arch/touch_psion.h>
#  include "md_psion5.h"
#endif

#ifdef MGL_NO_MOUSE
const static int same_format=0;
#define MD_PUT_SCANSEGMENT(dst_y,dst_x,src,nbytes)
static int show_mouse=0;
#endif


static struct fb_fix_screeninfo fbinfo_fix;
static struct fb_var_screeninfo fbinfo_var;
static struct fb_cmap*          fbinfo_cmap;
static int                      fbinfo_device;
static char*                    fbinfo_name;
static void*                    fbinfo_buffer;

static struct fb_var_screeninfo graph_fbinfo;
static struct fb_fix_screeninfo graph_fbinfo_fix;
static struct fb_cmap*           graph_cmap;
static struct fb_var_screeninfo text_fbinfo;
static struct fb_fix_screeninfo text_fbinfo_fix;
static struct fb_cmap*           text_cmap;

static struct fb_var_screeninfo* current_fbinfo;
static struct fb_fix_screeninfo* current_fbinfo_fix;

static int init_cmap_color(struct fb_cmap* cmap, int size) {
    int i;
    int c,c2,r,g,b;

    for (i=0; i<size; i++) {
	switch(size){
	  case 2:
	    c = CONV_FROM_COL2(i); break;
	  case 4:
	    c = CONV_FROM_COL4(i); break;
	  case 16:
	    c = CONV_FROM_COL16(i); break;
	  case 192:
	    c = CONV_FROM_COL192(i); break;
  	  default:
	    printf("fatal error! invalid argument.\n");
	    return -1;
	};
	c2 = mc_to_rgb(c);
	unpackRGB(c2,r,g,b);

	r |= r << 4; r |= r << 8;
	g |= g << 4; g |= g << 8;
	b |= b << 4; b |= b << 8;

	cmap->red[i] = r;
	cmap->green[i] = g;
	cmap->blue[i] = b;
    }
    return 0;
}

static struct fb_cmap* malloc_cmap(int size){

    struct fb_cmap* p = malloc(sizeof(struct fb_cmap));

    if(p == NULL) return p;

    if(NULL == (p->red = malloc(sizeof(u_int16_t) * size))){
	free(p);
	return NULL;
    }
    if(NULL == (p->green = malloc(sizeof(u_int16_t) * size))){
	free(p->red);
	free(p);
	return NULL;
    }
    if(NULL == (p->blue = malloc(sizeof(u_int16_t) * size))){
	free(p->red);
	free(p->green);
	free(p);
	return NULL;
    }
    p->start = 0;
    p->len = size;
    p->transp = NULL;
    return p;
}
static void free_cmap(struct fb_cmap* cmap){
    free(cmap->red);
    free(cmap->green);
    free(cmap->blue);
    free(cmap->transp);
    free(cmap);
}

static struct fb_cmap* get_cmap(){
    struct fb_cmap* cmap;
    int size;

    if(current_fbinfo_fix->visual != FB_VISUAL_PSEUDOCOLOR){ return NULL; }

    switch(current_fbinfo->bits_per_pixel){
      case 1:
	size = 2; break;
      case 2:
	size = 4; break;
      case 4:
	size = 16; break;
      case 8:
	size = 256; break;
      default:
	return NULL; break;
    };
    cmap = malloc_cmap(size);

    if (0 > ioctl(fbinfo_device, FBIOGETCMAP, cmap)) {
	perror("ioctl(FBIOGETCMAP)");
	return NULL;
    }

    return cmap;
}

static int restore_cmap(struct fb_cmap* cmap){
    int size;

    if (!cmap) return 0;
    if(current_fbinfo_fix->visual != FB_VISUAL_PSEUDOCOLOR){ return 0; }

    if (0 > ioctl(fbinfo_device, FBIOPUTCMAP, cmap)) {
	perror("ioctl(FBIOPUTCMAP)");
	return -1;
    }
    return 0;
}

static struct fb_cmap* put_cmap(){
    int size;
    struct fb_cmap* cmap;

printf("put_cmap bpp %d %d\n",current_fbinfo->bits_per_pixel,
		current_fbinfo_fix->visual);
if (current_fbinfo_fix->visual == FB_VISUAL_DIRECTCOLOR) {
    char buf[65536*4*3 + 128];
    cmap = buf;

    if (0 == ioctl(fbinfo_device, FBIOGETCMAP, cmap)) {
	printf("can getcmap !! len = %d\n",cmap->len);
    }
}
    if(current_fbinfo_fix->visual != FB_VISUAL_PSEUDOCOLOR){ return NULL; }

    switch(current_fbinfo->bits_per_pixel){
      default:
	return NULL; break;

      case 1:
	size = 2; break;
      case 2:
	size = 4; break;
      case 4:
	size = 16; break;
      case 8:
	if(current_fbinfo->grayscale){
	    size = 16;
	}else{
	    size = 192;
	}
	break;
    };
    if(NULL == (cmap = malloc_cmap(size))){ return NULL; }
    init_cmap_color(cmap, size);

    if (0 > ioctl(fbinfo_device, FBIOPUTCMAP, cmap)) {
	perror("ioctl(FBIOPUTCMAP)");
	free_cmap(cmap);
	return NULL;
    }
    return cmap;
}

static int get_fbinfo(){
    struct fb_cmap* cmap;

    fbinfo_device = -1;
    if (NULL != (fbinfo_name = getenv("FRAMEBUFFER"))){
	fbinfo_device = open(fbinfo_name, O_RDWR);
    }
    if (fbinfo_device < 0){
	fbinfo_name = "/dev/fb0";
	fbinfo_device = open(fbinfo_name, O_RDWR);
    }
    if (fbinfo_device < 0){
	perror("open /dev/fb0");
	return -1;
    }

    /* printf("success fbinfo_device(%d) for %s\n", fbinfo_device, fbinfo_name); */

    if (0 > ioctl(fbinfo_device, FBIOGET_FSCREENINFO, &fbinfo_fix)) {
	perror("ioctl(FBIOGET_FSCREENINFO)");
	return -1;
    }
    if (0 > ioctl(fbinfo_device, FBIOGET_VSCREENINFO, &fbinfo_var)) {
	perror("ioctl(FBIOGET_VSCREENINFO)");
	return -1;
    }

    text_fbinfo = fbinfo_var;
    text_fbinfo_fix = fbinfo_fix;
    current_fbinfo = &fbinfo_var;
    current_fbinfo_fix = &fbinfo_fix;
    if(!text_cmap) { text_cmap = get_cmap(); }

    return 0;
}

static int set_fbinfo(int bpp){

    struct fb_var_screeninfo var;
    struct fb_cmap* cmap;

    var = fbinfo_var;
    var.bits_per_pixel = bpp;
    var.xoffset = 0;
    var.yoffset = 0;

    /* for too large yres_virtaul ??? */
    if (var.yres_virtual > var.yres) {
    	var.yres_virtual = var.yres;
    }

    if (0 > ioctl(fbinfo_device, FBIOPUT_VSCREENINFO, &var)){
	    return -1;
    }
    if (0 > ioctl(fbinfo_device, FBIOGET_VSCREENINFO, &graph_fbinfo)) {
	perror("ioctl(FBIOGET_VSCREENINFO)");
	return -1;
    }
    if (0 > ioctl(fbinfo_device, FBIOGET_FSCREENINFO, &graph_fbinfo_fix)) {
	perror("ioctl(FBIOGET_FSCREENINFO)");
	return -1;
    }
    current_fbinfo = &graph_fbinfo;
    current_fbinfo_fix = &graph_fbinfo_fix;

printf("%d.red  %d,%d\n",bpp,graph_fbinfo.red.offset,graph_fbinfo.red.length);
printf("%d.gren %d,%d\n",bpp,graph_fbinfo.green.offset,graph_fbinfo.green.length);
printf("%d.blue %d,%d\n",bpp,graph_fbinfo.blue.offset,graph_fbinfo.blue.length);
    if(!graph_cmap) {
	graph_cmap = put_cmap();
    }

    return 0;
}

static int open_frame_buffer(){

    unsigned long offset;
    unsigned long mm_size;

    offset = (unsigned long)fbinfo_fix.smem_start & ( getpagesize() - 1);
    mm_size = (fbinfo_fix.smem_len + getpagesize() -1 + offset)
             / getpagesize() * getpagesize();
    
    fbinfo_buffer = (unsigned char *) mmap((caddr_t)0, mm_size,
					   PROT_READ | PROT_WRITE,
					   MAP_SHARED | MAP_FILE,
					   fbinfo_device,
					   0);
    if (fbinfo_buffer == MAP_FAILED) {
	perror("mmap");
	return -1;
    }
    fbinfo_buffer += offset;
    return 0;
}

static int restore_text_fbinfo(){

    if (0 > ioctl(fbinfo_device, FBIOPUT_VSCREENINFO, &text_fbinfo)){
	perror("ioctl(FBIOPUT_VSCREENINFO)");
	return -1;
    }
#if 0
    if (0 > ioctl(fbinfo_device, FBIOGET_FSCREENINFO, &fbinfo_fix)) {
	perror("ioctl(FBIOGET_FSCREENINFO)");
	return -1;
    }
    if (0 > ioctl(fbinfo_device, FBIOGET_VSCREENINFO, text_fbinfo)){
	perror("ioctl(FBIOGET_VSCREENINFO)");
	return -1;
    }
#endif
    current_fbinfo = &text_fbinfo;
    current_fbinfo_fix = &text_fbinfo_fix;
    restore_cmap(text_cmap);

    return 0;
}
static int restore_graphics_fbinfo(){

    if (0 > ioctl(fbinfo_device, FBIOPUT_VSCREENINFO, &graph_fbinfo)){
	perror("ioctl(FBIOPUT_VSCREENINFO)");
	return -1;
    }
#if 0
    if (0 > ioctl(fbinfo_device, FBIOGET_FSCREENINFO, &fbinfo_fix)) {
	perror("ioctl(FBIOGET_FSCREENINFO)");
	return -1;
    }
    if (0 > ioctl(fbinfo_device, FBIOGET_VSCREENINFO, &fbinfo_var)){
	perror("ioctl(FBIOGET_VSCREENINFO)");
	return -1;
    }
#endif
    current_fbinfo = &graph_fbinfo;
    current_fbinfo_fix = &graph_fbinfo_fix;
    restore_cmap(graph_cmap);

    return 0;
}

static int restore_fbinfo(){

    unsigned long offset;
    unsigned long mm_size;

    offset = (unsigned long)fbinfo_fix.smem_start & ( getpagesize() - 1);
    mm_size = (fbinfo_fix.smem_len + getpagesize() -1 + offset)
             / getpagesize() * getpagesize();
    
    fbinfo_buffer -= offset;

    if(0 != munmap(fbinfo_buffer, mm_size)){
	perror("munmap");
	return -1;
    }

    if (0 > ioctl(fbinfo_device, FBIOPUT_VSCREENINFO, &fbinfo_var)){
	perror("ioctl(FBIOPUT_VSCREENINFO)");
	return -1;
    }
    current_fbinfo = &fbinfo_var;
    current_fbinfo_fix = &fbinfo_fix;
    restore_cmap(text_cmap);

#if 0
    if (0 > ioctl(fbinfo_device, FBIOGET_FSCREENINFO, &fbinfo_fix)) {
	/* perror("ioctl(FBIOGET_FSCREENINFO)"); */
	return -1;
    }
    if (0 > ioctl(fbinfo_device, FBIOGET_VSCREENINFO, &fbinfo_var)) {
	/* perror("ioctl(FBIOGET_VSCREENINFO)"); */
	return -1;
    }
#endif
    return 0;
}

static inline void MD_GRAPH_MODE() {
    restore_graphics_fbinfo();
}

static inline void MD_TEXT_MODE() {
    restore_text_fbinfo();
}

static int MD_INIT() {
    int not_init = -1;
    static struct vt_stat vt_status;

    printf("md_linuxfb init\n");

    get_fbinfo();

    if (mgl_keyboard_reopen) {
	mgl_keyboard_fd = fbinfo_device;
    }

    /* sanity check for changing console. */
    if(0 > ioctl(mgl_keyboard_fd, VT_GETSTATE, &vt_status)){
	perror("ioctl(VT_GETSTATE)");
	return -1;
    }

#ifdef SUPPORT_LINUXFB_CFB16
    if(not_init) not_init = set_fbinfo(16);
#endif
#ifdef SUPPORT_LINUXFB_CFB8
    if(not_init) not_init = set_fbinfo(8);
#endif
#ifdef SUPPORT_LINUXFB_CFB4
    if(not_init) not_init = set_fbinfo(4);
#endif
#ifdef SUPPORT_LINUXFB_CFB2
    if(not_init) not_init = set_fbinfo(2);
#endif

    if(not_init){
	printf("cannot init fb\n");
	return -1;
    }

    open_frame_buffer();

    printf("current fb %u x %u @ %u bpp\n",
	   graph_fbinfo.xres_virtual,
	   graph_fbinfo.yres_virtual,
	   graph_fbinfo.bits_per_pixel );

    depth = graph_fbinfo.bits_per_pixel;
    rotated = 0;
    SCREEN_WIDTH  = graph_fbinfo.xres_virtual;
    SCREEN_HEIGHT = graph_fbinfo.yres_virtual;
    if (graph_fbinfo.xres_virtual > graph_fbinfo.xres) 
	   SCREEN_WIDTH = graph_fbinfo.xres;
    if (graph_fbinfo.yres_virtual > graph_fbinfo.yres) 
	   SCREEN_HEIGHT = graph_fbinfo.yres;

    switch(depth){
#ifdef SUPPORT_LINUXFB_CFB2
    case 2:
	if(graph_fbinfo.red.msb_right){
	    fb2_setup(fbinfo_buffer, graph_fbinfo_fix.line_length, ORD2_NORMAL, COL2_REVERSE);
	}else{
	    fb2_setup(fbinfo_buffer, graph_fbinfo_fix.line_length, ORD2_REVERSE, COL2_REVERSE);
	}
	break;
#endif
#ifdef SUPPORT_LINUXFB_CFB4
    case 4:
	fb4_setup(fbinfo_buffer, graph_fbinfo_fix.line_length,
		  graph_fbinfo.red.msb_right ? ORD4_NORMAL : ORD4_REVERSE,
		  COL4_REVERSE );
	break;
#endif
#ifdef SUPPORT_LINUXFB_CFB8
    case 8:
	fb8_setup(fbinfo_buffer, graph_fbinfo_fix.line_length, 0);
	break;
#endif
#ifdef SUPPORT_LINUXFB_CFB16
    case 16:
	if(graph_fbinfo.green.length == 5){
	    fb16_setup(fbinfo_buffer, graph_fbinfo_fix.line_length, ORD_RGB);
	}else{
	    fb16_setup(fbinfo_buffer, graph_fbinfo_fix.line_length, ORD_R5G6B5);
	}
	break;
#endif
    default:
	printf("fatal error! invalid depth.\n");
	return -1;
    };

#ifdef MGL_MOUSE_PSION5
    if(0 > open_touch()){
	printf("cannot open touch screen.\n");
    }
#endif

#ifdef MGL_MOUSE_MG2
    if(0 > open_touch()){
	printf("cannot open touch screen.\n");
    }
    else{
        printf("open touch screen.\n");
    }
#endif

#ifdef MD_CALIBRATE_CTRL
    mgl_calibrate_ctrl = MD_CALIBRATE_CTRL;
#endif
    return 0;
}

static inline void MD_TERM() {

#ifdef MGL_MOUSE_PSION5
    close_touch();
#endif
#ifdef MGL_MOUSE_MG2
    close_touch();
#endif
    restore_fbinfo();
}


/*----------------------------------------------------------------------*/
/* ޥ ط */

#ifdef MGL_NO_MOUSE

static int MD_MOUSE_EVENT() { return 0;}
static void MD_SET_MOUSEXY(int x,int y) { }
static inline int MD_MOUSE_X() { return 0;}
static inline int MD_MOUSE_Y() { return 0;}
static inline int MD_MOUSE_BUTTON() { return 0;}

#endif

#ifdef SUPPORT_LINUXFB_CFB2
#  define MD_PUT_PIXSTREAM2	fb2_put_pixstream
#else
  static void MD_PUT_PIXSTREAM2(int x,int y,int *buf,int xs,int dir,int op) { }
#endif

#ifdef SUPPORT_LINUXFB_CFB4
#  define MD_PUT_PIXSTREAM4	fb4_put_pixstream
#else
  static void MD_PUT_PIXSTREAM4(int x,int y,int *buf,int xs,int dir,int op) { }
#endif

#ifdef SUPPORT_LINUXFB_CFB8
#  define MD_PUT_PIXSTREAM8	fb8_put_pixstream
#else
  static void MD_PUT_PIXSTREAM8(int x,int y,int *buf,int xs,int dir,int op) { }
#endif

#ifdef SUPPORT_LINUXFB_CFB16
#  define MD_PUT_PIXSTREAM16	fb16_put_pixstream
#else
  static void MD_PUT_PIXSTREAM16(int x,int y,int *buf,int xs,int dir,int op) { }
#endif


/*----------------------------------------------------------------------*/
/* 󥽡ؤ */

#define MD_PROC_CONSOLE md_proc_console

static int md_proc_console(int key){
    static struct vt_stat vt_status;

    if(MKE_CONSOLE <= key) key -= MKE_CONSOLE;
    if(key == 0) return 0;                     /* try to change myself */
    if(key < 1 || 10 < key) return -1;

    if(0 > ioctl(mgl_keyboard_fd, VT_GETSTATE, &vt_status)) return -1;

    if(key == vt_status.v_active) return 0;    /* try to change myself */

    /* fprintf(stderr, "change console from %d to %d in 0x%04x\r\n",
     * vt_status.v_active, key, vt_status.v_state);
     */

    if(vt_status.v_state & (1 << key)){
	if(0 > ioctl(mgl_keyboard_fd, VT_ACTIVATE, key)){
	    return -1;
	}
	// ioctl(mgl_keyboard_fd, VT_WAITACTIVE, vt_status.v_active);
    }
    return 0;
}


/*----------------------------------------------------------------------*/
/* Хå饤ȡLCDĴ */

#ifdef FBIOPUT_BACKLIGHT

    static int backlight_level = 1;

#define MD_PROC_BACKLIGHT md_proc_backlight
static int md_proc_backlight(int key) {

    int bl_stat = -1;
    if (0 > ioctl(fbinfo_device, FBIOGET_BACKLIGHT,  &bl_stat)) {
        return -1;
    }
    if (bl_stat == 0) /* off */ {
        bl_stat = backlight_level; /* resotre stat */
    } else {
        backlight_level = bl_stat; /* save stat */
        bl_stat = 0; /* change off */
    }
    if (0 > ioctl(fbinfo_device, FBIOPUT_BACKLIGHT, bl_stat)) return -1;
    return 0;
}

#define MD_PROC_BRIGHTNESS md_proc_brightness
static int md_proc_brightness(int key) {
    int bl_stat;

    if (0 > ioctl(fbinfo_device, FBIOGET_BACKLIGHT,  &bl_stat)) {
        return -1;
    }

    if(key == MKE_BRIGHTNESS_UP) {
        if (bl_stat == 0 ) bl_stat = backlight_level - 1;
        if (0 > ioctl(fbinfo_device, FBIOPUT_BACKLIGHT,  bl_stat+1)) return -1;
        return 0;
    }
    if(key == MKE_BRIGHTNESS_DOWN) {
        if (bl_stat == 1) return 0; /* do nothing */
        if (bl_stat == 0) bl_stat = backlight_level + 1;
        if (0 > ioctl(fbinfo_device, FBIOPUT_BACKLIGHT, bl_stat-1)) return -1;
        return 0;
    }
    return 0;
}

#endif /* FBIOPUT_BACKLIGHT */

#ifdef FBIOPUT_CONTRAST

#define MD_PROC_CONTRAST md_proc_contrast
static int md_proc_contrast(int key) {
    /* static */ int current_contrast = -1;

    // fprintf(stderr, "md_proc_contrast(0x%0x) current ... ", key);

    if (0 > ioctl(fbinfo_device, FBIOGET_CONTRAST,  &current_contrast)){
	current_contrast = -1;
	return -1;
    }

    // fprintf(stderr, "%d\r\n", current_contrast);

    switch(key){
     case 0:
     case MKE_CONTRAST_DOWN:
	{
	    if(current_contrast == 0) return 0;
	    current_contrast --;
	    if (0 > ioctl(fbinfo_device, FBIOPUT_CONTRAST,  current_contrast)) return -1;
	    break;
	}
     case 1:
     case MKE_CONTRAST_UP:
	{
	    current_contrast ++;
	    if (0 > ioctl(fbinfo_device, FBIOPUT_CONTRAST,  current_contrast)) return -1;
	    break;
	}
     default:
	break;
    };
    return 0;
}

#endif /* FBIOPUT_CONTRAST */

