/*
 * MGL -- MobileGear Graphic Library -
 * Copyright (C) 1998, 1999
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      Yukihiko Sano (yukihiko@yk.rim.or.jp)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KOJI SUZUKI AND YUKIHIKO SANO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifdef __NetBSD__
#define USE_KEYMAP
#define SAVE_FONT
#endif
#ifdef __FreeBSD__
#define USE_KEYMAP
#define SAVE_FONT
#undef USE_OD_METHOD
#endif

static int show_mouse=1;
const static int same_format=0;
#define MD_PUT_SCANSEGMENT(dst_y,dst_x,src,nbytes)

static int mouse_fd = -1;

#ifdef __FreeBSD__
#include <machine/cpufunc.h>
#include <machine/console.h>
#include <machine/mouse.h>
#endif

#ifdef __NetBSD__
#include <machine/pio.h>
#include <dev/wscons/wsconsio.h>
#include <dev/wscons/wsdisplay_usl_io.h>
#define MOUSEDEV	"/dev/wsmouse0"
#define USE_SELECT	/* use select() for key/mouse */
#endif

static void MD_MOUSE_CLOSE();
static int MD_MOUSE_OPEN();

void portoutb(unsigned short p, int data) {
    outb(p, data);
}

void portoutw(unsigned short p, int data) {
    outw(p, data);
}

int portinb(unsigned short p) {
    return inb(p);
}

#define ATTR_ADDR	0x3c0
#define ATTR_DATA	0x3c1
#define MISC_OADDR	0x3c2

#define	SEQ_ADDR	0x3c4
#define	SEQ_DATA	0x3c5

#define PEL_IADDR	0x3c7
#define PEL_OADDR	0x3c8
#define PEL_DATA	0x3c9

#define MISC_IADDR	0x3cc

#define GRAPH_ADDR  	0x3ce
#define GRAPH_DATA	0x3cf

#define CRT_ADDR	0x3d4
#define CRT_DATA	0x3d5
#define ST1_ADDR	0x3da



#ifdef SAVE_FONT
#define FONT_SIZE	0x2000
static char font_data1[FONT_SIZE]; /* text font data ??? */
static char font_data2[FONT_SIZE]; /* text font data ??? */
#endif
static char *fb;		/* frame buffer */

static char text_pel[64*3];
static char graph_pel[64*3];
static char cmap[256];

void set_plane(int pl) {
     portoutb(SEQ_ADDR, 0x02);
     portoutb(SEQ_DATA, 0x01<<pl);
     portoutw(GRAPH_ADDR, 0x04 | (pl << 8));
}

static void pel_save(char *buf) {
  int i;

  portoutb(PEL_IADDR,0);
  for (i=0; i<64; i++) {
     *buf++ = portinb(PEL_DATA);
     *buf++ = portinb(PEL_DATA);
     *buf++ = portinb(PEL_DATA);
  }
#ifdef PEL_DEBUG
  {
	buf -= 64*3;
	printf("pel_save\n");
	for (i=0; i<64; i++) {
		printf("%d: %3d %3d %3d\n",i,buf[0],buf[1],buf[2]);
		buf += 3;
	}
  }
#endif
}

static void pel_load(char *buf) {
  int i;
  portoutb(PEL_OADDR,0);
  for (i=0; i<64; i++) {
     portoutb(PEL_DATA,*buf++);
     portoutb(PEL_DATA,*buf++);
     portoutb(PEL_DATA,*buf++);
  }
#ifdef PEL_DEBUG
  {
	char xbuf[64*3];
	pel_save(xbuf);
	buf = xbuf;
	printf("pel_load\n");
	for (i=0; i<64; i++) {
		printf("%d: %3d %3d %3d\n",i,buf[0],buf[1],buf[2]);
		buf += 3;
	}
  }
#endif
}

#ifdef __FreeBSD__
static int fbsd_setgraph(int fd,int in_init) {
	if (in_init) {
		return ioctl(fd,SW_CG640x480,0);
	}
	return ioctl(fd,M_CG640x480,0);
}

static int fbsd_settext(int fd,int in_term) {
	if (in_term) {
		return ioctl(fd, _IO('S', M_VGA_C80x25) ,0);
	}
	return ioctl(fd,M_VGA_C80x25,0);
}
#endif

struct vga_regs {
	unsigned char crt[25];
	unsigned char att[21];
	unsigned char gra[9];
	unsigned char seq[5];
	unsigned char mis;
};

const struct vga_regs graph_regs = {
	{0x5f,0x4f,0x50,0x82,0x54,0x80,0x0b,0x3e,0x00,0x40,
	0x00,0x00,0x00,0x00,0xff,0xff,0xea,0x8c,0xdf,0x28,
	0x00,0xe7,0x04,0xe3,0xff},
	{0x00,0x01,0x02,0x03,0x04,0x05,0x14,0x07,0x38,0x39,
	0x3a,0x3b,0x3c,0x3d,0x3e,0x3f,0x01,0x00,0x0f,0x00,
	0x00},
	{0x00,0x00,0x00,0x00,0x00,0x00,0x05,0x0f,0xff},
	{0x03,0x01,0x0f,0x00,0x06},
	0xe3
};

struct vga_regs text_regs;

static int cur_crt_mode = 0;

int disable_video = 0;

int vga_disable_video() {
	int ret;

	if (!disable_video) {
		portinb(ST1_ADDR);
		portoutb(ATTR_ADDR, 0x00);
	}
	ret = disable_video;
	disable_video = 1;
	return ret;
}

void vga_enable_video(int old_mode) {
	if (old_mode == 0) {
		/* enable video */
		portinb(ST1_ADDR);
		portoutb(ATTR_ADDR, 0x20);
		disable_video = 0;
	}
}

static int vga_setregs(const struct vga_regs *r) {
	int i;
	int mod;

	mod = vga_disable_video();

	portoutb(MISC_OADDR, portinb(MISC_IADDR)&1 | r->mis & ~1);

	portoutb(SEQ_ADDR, 0x00);
	portoutb(SEQ_DATA, 0x01);

	for (i=1; i< sizeof(r->seq); i++) {
		portoutb(SEQ_ADDR, i);
		portoutb(SEQ_DATA, r->seq[i]);
	}

	portoutb(SEQ_ADDR, 0x00);
	portoutb(SEQ_DATA, 0x03);

	portoutb(CRT_ADDR, 0x11);
	portoutb(CRT_DATA, portinb(CRT_DATA) & 0x7f);

	for (i=0; i< sizeof(r->crt); i++) {
		portoutb(CRT_ADDR, i);
		portoutb(CRT_DATA, r->crt[i]);
	}

	for (i=0; i< sizeof(r->gra); i++) {
		portoutb(GRAPH_ADDR, i);
		portoutb(GRAPH_DATA, r->gra[i]);
	}

	for (i=0; i< sizeof(r->att); i++) {
		portinb(ST1_ADDR);
		portoutb(ATTR_ADDR, i);
		portoutb(ATTR_ADDR, r->att[i]);
	}
	vga_enable_video(mod);
	return 0;
}

static int vga_getregs(struct vga_regs *r) {
	int i;
	int mod;

	mod = vga_disable_video();

	for (i=0; i< sizeof(r->crt); i++) {
		portoutb(CRT_ADDR, i);
		r->crt[i] = portinb(CRT_DATA);
	}

	for (i=0; i< sizeof(r->att); i++) {
		portinb(ST1_ADDR);
		portoutb(ATTR_ADDR, i);
		r->att[i] = portinb(ATTR_DATA);
	}

	for (i=0; i< sizeof(r->gra); i++) {
		portoutb(GRAPH_ADDR, i);
		r->gra[i] = portinb(GRAPH_DATA);
	}
	for (i=0; i< sizeof(r->seq); i++) {
		portoutb(SEQ_ADDR, i);
		r->seq[i] = portinb(SEQ_DATA);
	}
	r->mis = portinb(MISC_IADDR);

	vga_enable_video(mod);
	return 0;
}

#ifdef VGA_DEBUG
static int vga_dumpregs(struct vga_regs *r) {
	int i;
	printf("crt");
	for (i=0; i< sizeof(r->crt); i++) {
		if (i % 10 == 0) printf("\n");
		printf("%02x,",r->crt[i]);
	}
	printf("\n");

	printf("att");
	for (i=0; i< sizeof(r->att); i++) {
		if (i % 10 == 0) printf("\n");
		printf("%02x,",r->att[i]);
	}
	printf("\n");

	printf("gra");
	for (i=0; i< sizeof(r->gra); i++) {
		if (i % 10 == 0) printf("\n");
		printf("%02x,",r->gra[i]);
	}
	printf("\n");

	printf("seq");
	for (i=0; i< sizeof(r->seq); i++) {
		if (i % 10 == 0) printf("\n");
		printf("%02x,",r->seq[i]);
	}
	printf("\n");
	printf("mis %02x\n",r->mis);
}
#endif

static int vga_setgraph(int fd,int in_init) {
	int mod;

	if (cur_crt_mode == 1) return 0;

	mod = vga_disable_video();

	if (in_init) {
		vga_getregs(&text_regs);
#ifdef VGA_DEBUG
		vga_dumpregs(&text_regs);
#endif
		portoutb(MISC_OADDR, portinb(MISC_IADDR) | 1);
	}
	vga_setregs(&graph_regs);

	vga_enable_video(mod);
	cur_crt_mode = 1;
	return 0;
}

static int vga_settext(int fd,int in_term) {
	int mod;
	if (cur_crt_mode == 0) return 0;

	mod = vga_disable_video();
	vga_setregs(&text_regs);
	vga_enable_video(mod);

	cur_crt_mode = 0;
	return 0;
}

#ifdef USE_OD_METHOD
#define od_settext	fbsd_settext
#define od_setgraph	fbsd_setgraph
#else
#define od_settext	vga_settext
#define od_setgraph	vga_setgraph
#endif

static void MD_GRAPH_MODE() {
#ifdef SAVE_FONT
        set_plane(2);
        memcpy(font_data1,fb,FONT_SIZE); /* save font data */
        set_plane(3);
        memcpy(font_data2,fb,FONT_SIZE); /* save font data */
#endif
        od_setgraph(mgl_keyboard_fd,0);
        pel_load(graph_pel);
}

static void MD_TEXT_MODE() {
#ifdef SAVE_FONT
        set_plane(2);
        memcpy(fb,font_data1,FONT_SIZE); /* load font data */
        set_plane(3);
        memcpy(fb,font_data2,FONT_SIZE); /* load font data */
#endif
        pel_load(text_pel);
	od_settext(mgl_keyboard_fd,0);
}

static int MD_INIT() {
	int fd;
	int offset;
	int i,c;
	char *p;
	char *devname;

printf("md_vga init\n");
#if (defined(__FreeBSD__) && __FreeBSD__ <= 3)
	fd = open("/dev/vga",O_RDWR|O_NDELAY);
	offset = 0;
#else
	fd = open("/dev/mem",O_RDWR|O_NDELAY);
	offset = 0xa0000;
#endif
	if (fd < 0) {
		perror("open /dev/mem");
		return -1;
	}
	fb = mmap(0, 0x10000, PROT_READ|PROT_WRITE,MAP_FILE|MAP_SHARED,
			fd, offset);

	if ((long)fb < 0) {
		perror("mmap");
		return -1;
	}
	//mgl_keyboard_fd = kbd_fd;
	if (ioctl(mgl_keyboard_fd, KDENABIO, 0) < 0) {
		fprintf(stderr,"fd = %d\n",mgl_keyboard_fd);
		perror("KDENABIO");
		return -1;
	}
	ioctl(mgl_keyboard_fd, VT_WAITACTIVE, 0);

	ioctl(mgl_keyboard_fd, KDSETMODE, KD_GRAPHICS);
	if (od_setgraph(mgl_keyboard_fd,1) == 0) {
		SCREEN_WIDTH = 640;
		SCREEN_HEIGHT = 480;
		depth = 8;
	} else {
		ioctl(mgl_keyboard_fd, KDSETMODE, KD_TEXT);
		return -1;
	}
  outb(SEQ_ADDR, 0x02);
  outb(SEQ_DATA, 0x0f);
  outb(0x3c6, 0xff);
        pel_save(text_pel);

   {
	int i,c,h,s,b;
#define Pel(n,red,green,blue) p = graph_pel + (n*3);\
	p[0] = red; p[1] = green; p[2] = blue;

	Pel( 0,  0, 0, 0);
	Pel( 1, 21,21,21);
	Pel( 2, 42,42,42);
	Pel( 3, 63,63,63);

	Pel( 4, 31, 0, 0);
	Pel( 5, 63, 0, 0);
	Pel( 6, 31,31, 0);
	Pel(20, 31,31, 0); /* Why ??? */
	Pel( 7, 63,63, 0);

	Pel(56,  0,31, 0);
	Pel(57,  0,63, 0);
	Pel(58,  0,31,31);
	Pel(59,  0,63,63);
	Pel(60,  0, 0,31);
	Pel(61,  0, 0,63);
	Pel(62, 31, 0,31);
	Pel(63, 63, 0,63);
	for (i=0; i<256; i++) {
		if ((i>= 192) || ((i & 0x0c) == 0)) {
			cmap[i] = i & 0x3;
		} else if ((i & 0x03) == 0x03) {
			h = (i & 0xf0) >> 4;
			h = (h + 12+1)%12;
			h /= 2;
			cmap[i] = (h * 2 + 1 + 4) & 0xf;
		} else if ((i & 0x03) == 0x00) {
			cmap[i] = 0;
		} else {
			h = (i & 0xf0) >> 4;
			h = (h + 12+1)%12;
			h /= 2;
			cmap[i] = (h * 2 + 0 + 4) & 0xf;
		}
	}
   }
	MD_GRAPH_MODE();
	MD_MOUSE_OPEN();
printf("md_init success\n");
#ifdef VGA_DEBUG
{
struct vga_regs r;
vga_getregs(&r);
vga_dumpregs(&r);
}
#endif
	return 0;
}

static void MD_TERM() {
printf("md_term\n");
#ifdef SAVE_FONT
        set_plane(2);
        memcpy(fb,font_data1,FONT_SIZE); /* load font data */
        set_plane(3);
        memcpy(fb,font_data2,FONT_SIZE); /* load font data */
#endif
        pel_load(text_pel);
	od_settext(mgl_keyboard_fd, 1);
	ioctl(mgl_keyboard_fd, KDSETMODE, KD_TEXT);
	munmap(fb,0x10000);
	ioctl(mgl_keyboard_fd, KDDISABIO, 0);
	MD_MOUSE_CLOSE();
printf("md_term done\n");
}

static void MD_PUT_PIXSTREAM4(int x,int y,int *buf,int xs,int dir,int op) {
	char pbuf[4][640/8];
	int offset,len;
	int c,pos,i;

	offset = y * 640/8 + x/8;
	len = xs/8;
	pos = 0;

	while (xs > 0) {
	    int dat1,dat2,dat3,dat4;
	    dat1 = dat2 = dat3 = dat4 = 0;
	    for (i=0; i<8; i++) {
		dat1 <<= 1;
		dat2 <<= 1;
		dat3 <<= 1;
		dat4 <<= 1;
		c = (*buf++) & 0xf;
		xs --;
		if (c & 1) dat1 |= 1;
		if (c & 2) dat2 |= 1;
		if (c & 4) dat3 |= 1;
		if (c & 8) dat4 |= 1;
	    }
	    pbuf[0][pos] = dat1;
	    pbuf[1][pos] = dat2;
	    pbuf[2][pos] = dat3;
	    pbuf[3][pos] = dat4;
	    pos++;
	}
	for (i=0; i<4; i++) {
		set_plane(i);
		memcpy(fb+offset,pbuf[i],len);
	}
}

static void MD_PUT_PIXSTREAM2(int x,int y,int *buf,int xs,int dir,int op) {
}

static void MD_PUT_PIXSTREAM8(int x,int y,int *buf,int xs,int dir,int op) {
	char pbuf[4][640/8];
	int offset,len;
	int c,pos,i;

	offset = y * 640/8 + x/8;
	len = xs/8;
	pos = 0;

	while (xs > 0) {
	    int dat1,dat2,dat3,dat4;
	    dat1 = dat2 = dat3 = dat4 = 0;
	    for (i=0; i<8; i++) {
		dat1 <<= 1;
		dat2 <<= 1;
		dat3 <<= 1;
		dat4 <<= 1;
		c = cmap[CONV_TO_COL192(*buf)];
		buf++;
		xs --;
		if (c & 1) dat1 |= 1;
		if (c & 2) dat2 |= 1;
		if (c & 4) dat3 |= 1;
		if (c & 8) dat4 |= 1;
	    }
	    pbuf[0][pos] = dat1;
	    pbuf[1][pos] = dat2;
	    pbuf[2][pos] = dat3;
	    pbuf[3][pos] = dat4;
	    pos++;
	}
	for (i=0; i<4; i++) {
		set_plane(i);
		memcpy(fb+offset,pbuf[i],len);
	}
}

static void MD_PUT_PIXSTREAM16(int x,int y,int *buf,int xs,int dir,int op) {
}

static int m_x,m_y,m_b;

#ifdef __FreeBSD__
static int MD_MOUSE_OPEN() {
	mouse_fd = open("/dev/sysmouse",O_RDONLY);
	if (mouse_fd < 0) {
		printf("warning cannot open sysmouse\n");
	}
	return (mouse_fd >=0)?0:-1;
}
static void MD_MOUSE_CLOSE() {
	if (mouse_fd >= 0) {
		close(mouse_fd);
		mouse_fd = -1;
	}
}

static int MD_MOUSE_EVENT() {
	mousestatus_t s[10];
	int r;
	if (mouse_fd < 0) return 0;
	
	if ( ioctl(mouse_fd,MOUSE_GETSTATUS,s) == 0) {
		if ((s->dx == 0) && (s->dy == 0) && (s->button == s->obutton))
			return 0;
		m_x += s->dx;
		m_y += s->dy;
		m_b = s->button;
		return 1;
	}
	return 0;
}

#endif

#ifdef __NetBSD__
static int MD_MOUSE_OPEN() {
#ifdef USE_SELECT
    if ((mouse_fd = open(MOUSEDEV, O_RDWR )) < 0)
	return -1;
#else
    if ((mouse_fd = open(MOUSEDEV, O_RDWR | O_NDELAY )) < 0)
	return -1;
#endif
    return 0;
}

static void MD_MOUSE_CLOSE(void) {
    if (mouse_fd == -1)
	return;
    close(mouse_fd);
    mouse_fd = -1;
}

#ifdef USE_SELECT
static int MD_MOUSE_EVENT() {
    struct wscons_event ev[16];
    int i,n,c,d,r;
    fd_set fds;
    struct timeval to;

	if (mouse_fd < 0) return 0;

	c = 0;
retry:
	FD_ZERO(&fds);
	FD_SET(mouse_fd,&fds);
	to.tv_sec = 0;
	to.tv_usec = 0;
	if (select(mouse_fd+1,&fds,0,0,&to) != 1) return c;
	r = read(mouse_fd,ev,sizeof(ev));
	if (r < 0) {
    		close(mouse_fd);
		mouse_fd = -1;
		return 0;
	}
	n = r/sizeof(ev[0]);
//printf("event count = %d \n",n);
	for (i=0; i<n; i++) {
	    switch(ev[i].type) {
		case WSCONS_EVENT_MOUSE_UP:
//printf("MOUSE_UP\n");
			if  (ev[i].value == 0) {
				m_b = 0;
				c++;
			}
			break;
		case WSCONS_EVENT_MOUSE_DOWN:
//printf("MOUSE_DOWN\n");
			if (ev[i].value == 0) {
				m_b = 1;
				c++;
			}
			if (i+1 == n) goto retry;
			break;
		case WSCONS_EVENT_MOUSE_DELTA_X:
			d = ev[i].value;
			m_x += d;
			if (m_x < 0) m_x = 0;
			if (m_x >= SCREEN_WIDTH) m_x = SCREEN_WIDTH - 1;
			c++;
			break;
		case WSCONS_EVENT_MOUSE_DELTA_Y:
			d = ev[i].value;
			m_y -= d;
			if (m_y < 0) m_y = 0;
			if (m_y >= SCREEN_HEIGHT) m_y = SCREEN_HEIGHT - 1;
			c++;
			break;
		case WSCONS_EVENT_MOUSE_ABSOLUTE_X:
			m_x = ev[i].value;
//printf("MOUSE_ABSX %d\n",m_x);
			c++;
			break;
		case WSCONS_EVENT_MOUSE_ABSOLUTE_Y:
			m_y = ev[i].value;
//printf("MOUSE_ABSY %d\n",m_y);
			c++;
			break;
	     }
	}
	return c;
}
#else /* NDELAY version */
static int MD_MOUSE_EVENT() {
    struct wscons_event ev[16];
    int i,n,c,d,r;

	if (mouse_fd < 0) return 0;

	r = read(mouse_fd,ev,sizeof(ev));
	if (r < 0) {
    		close(mouse_fd);
		mouse_fd = -1;
		return 0;
	}
	n = r/sizeof(ev[0]);
printf("event count = %d \n",n);
	c = 0;
	for (i=0; i<n; i++) {
	    switch(ev[i].type) {
		case WSCONS_EVENT_MOUSE_UP:
			if  (ev[i].value == 0) {
printf("MOUSE_UP\n");
				m_b = 0;
				c++;
			}
			break;
		case WSCONS_EVENT_MOUSE_DOWN:
			if (ev[i].value == 0) {
printf("MOUSE_DOWN\n");
				m_b = 1;
				c++;
			}
			break;
		case WSCONS_EVENT_MOUSE_DELTA_X:
			d = ev[i].value;
			m_x += d;
			if (m_x < 0) m_x = 0;
			if (m_x >= SCREEN_WIDTH) m_x = SCREEN_WIDTH - 1;
			c++;
			break;
		case WSCONS_EVENT_MOUSE_DELTA_Y:
			d = ev[i].value;
			m_y += d;
			if (m_y < 0) m_y = 0;
			if (m_y >= SCREEN_HEIGHT) m_y = SCREEN_HEIGHT - 1;
			c++;
			break;
		case WSCONS_EVENT_MOUSE_ABSOLUTE_X:
			m_x = ev[i].value;
printf("MOUSE_ABSX %d\n",m_x);
			c++;
			break;
		case WSCONS_EVENT_MOUSE_ABSOLUTE_Y:
			m_y = ev[i].value;
printf("MOUSE_ABSY %d\n",m_y);
			c++;
			break;
	     }
	}
	return c;
}
#endif
#endif /* __NetBSD__ */

/* common */
#define MD_MOUSE_X()		(m_x)
#define MD_MOUSE_Y()		(m_y)
#define MD_MOUSE_BUTTON()	(m_b)
#define MD_SET_MOUSEXY(x,y)	do { m_x = x; m_y = y; } while (0)


#define MD_PROC_CONSOLE md_proc_console
static int md_proc_console(int key){
	int active;

	if (MKE_CONSOLE <= key)
		key -= MKE_CONSOLE;
	if (key == 0)
		return 0;	/* try to change myself */
	if (key < 1 || 10 < key)
		return -1;

	if (ioctl(mgl_keyboard_fd, VT_GETACTIVE, &active) < 0)
		return -1;

	if (key == active)
		return 0;	/* try to change myself */

	fprintf(stderr, "change console from %d to %d \r\n",
		active, key);

	if (ioctl(mgl_keyboard_fd, VT_ACTIVATE, key) < 0){
		return -1;
	}
	return 0;
}
