/*
 * NINE -- the alternative keyboard function using mouse or touchpad
 *    (keycode is determined by the order passing unvisible NINE buttons)
 *
 * Copyright (C) 2000, 2001
 *	Kazuma Arino (kazuma@sola.c.u-tokyo.ac.jp)
 *      Koji Suzuki (suz@at.sakura.ne.jp)
 *      
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY KAZUMA ARINO ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */
#include "mtostr.h"
#include <stdio.h>

static int plot_line(int *buf,int max_cnt,int x1, int y1, int x2, int y2);

static int min_x,max_x;
static int min_y,max_y;
static int rx,ry;
static int first_x,first_y;
static int last_x,last_y;

#define RESULT_LENGTH 1024
#define BUF_LENGTH RESULT_LENGTH
char result_buf[RESULT_LENGTH];
static int buf[BUF_LENGTH];
static int buf_cnt;
static int oldx,oldy;

void mtostr_record(int x,int y) {
   x <<= SCALE_SFT;
   y <<= SCALE_SFT;
   if (x < min_x) min_x = x;
   if (x > max_x) max_x = x;
   if (y < min_y) min_y = y;
   if (y > max_y) max_y = y;
   buf_cnt += plot_line(buf + buf_cnt,RESULT_LENGTH - buf_cnt
                                        ,oldx,oldy,x,y);
   oldx = x;
   oldy = y;
}

void mtostr_init(int x,int y){
   x <<= SCALE_SFT;
   y <<= SCALE_SFT;
   oldx = x;
   oldy = y;
   min_x = x;
   max_x = x;
   min_y = y;
   max_y = y;
   first_x = x;
   first_y = y;
   buf_cnt = 0;
   buf[buf_cnt++] = x;
   buf[buf_cnt++] = y;
}

void mtostr_finish() {
   last_x = buf[buf_cnt-2];
   last_y = buf[buf_cnt-1];

   if (max_x - min_x <  (max_y - min_y)/2) {
      int x,r;
      x = (max_x + min_x)/2;
      r = (max_y - min_y)/4;
      max_x = x + r;
      min_x = x - r;
   }
   else if (max_y - min_y < (max_x - min_x)/2) {
      int y,r;
      y = (max_y + min_y)/2;
      r = (max_x - min_x)/4;
      max_y = y + r;
      min_y = y - r;
   }
   rx = (max_x - min_x)/8;
   ry = (max_y - min_y)/8;
}

static
calc_area(int n,int *xp,int *yp, int *xsp, int *ysp) {
	int xs,ys;
	int x,y;
	xs = (max_x - min_x)/2;
	ys = (max_y - min_y)/2;
	x = min_x + (((n-1) % 3) * xs) - xs/2;
	y = min_y + (((n-1) / 3) * ys) - ys/2;
	*xp = x;
	*yp = y;
	*xsp = xs;
	*ysp = ys;
}

static
calc_center(int n,int *cxp,int *cyp) {
	int xs,ys;
	int x,y;
	calc_area(n, &x,&y,&xs,&ys);
	*cxp = x + xs/2;
	*cyp = y + ys/2;
}

static
is_startstop(int n) {
	int xs,ys;
	int x,y;
	calc_area(n, &x,&y,&xs,&ys);

	if ((x <= first_x) && (first_x < x + xs)
	   && (y <= first_y) && (first_y < y + ys))
		return BUTTON_ALL;
	if ((x <= last_x) && (last_x < x + xs)
	   && (y <= last_y) && (last_y < y + ys))
		return BUTTON_ALL;
	return BUTTON_SMALL;
}

#define max(x,y) (x>y ? x : y)

static
which_area(int cx,int cy) {
	int xs,ys,x,y,i,j;
	xs = max((max_x - min_x)/2,1);
	ys = max((max_y - min_y)/2,1);
	x = min_x - xs/2;
	y = min_y - ys/2;
	i = (cx - x)/xs;
	j = (cy - y)/ys;
	

	if (i < 0) i= 0;
	if (i > 2) i = 2;
	if (j < 0) j = 0;
	if (j > 2) j = 2;
	return (1 + i + j * 3);
}

void
mtostr_button_type(int *l) {
   int i;
   for (i=1; i<= 9; i++) {
      l[i] = is_startstop(i);
   }
   if (l[1] && l[3]) l[8] = BUTTON_LARGE;
   else if (l[3] && l[9]) l[4] = BUTTON_LARGE;
   else if (l[9] && l[7]) l[2] = BUTTON_LARGE;
   else if (l[7] && l[1]) l[6] = BUTTON_LARGE;
   else if (l[1] && l[9]) { l[7] = BUTTON_ALL; l[3] = BUTTON_ALL;}
   else if (l[3] && l[7]) { l[1] = BUTTON_ALL; l[9] = BUTTON_ALL;}

   if (l[5] == BUTTON_SMALL) {
	l[5] = BUTTON_MEDIUM;
   }
}

void mtostr_area(int i,int *xp,int *yp,int *rxp,int *ryp) {
	calc_center(i,xp,yp);
	*xp >>= SCALE_SFT;
	*yp >>= SCALE_SFT;
	*rxp = rx >> SCALE_SFT;
	*ryp = ry >> SCALE_SFT;
}

int mtostr_buf(int *bp, int max) {
	int i;
	for (i=0; (i<max) && (i < buf_cnt); i++) {
		*bp++ = buf[i] >> SCALE_SFT;
	}
	return i;
}


#define in_rect(x,y,w,h,cx,cy) (cx > x) && (cx < x+w) && (cy > y) \
				&& (cy < y +h )

static
int in_circle(int x1,int y1,int x2,int y2,int x,int y){
   int a2,b2,cx,cy; /* a^2,b^2 , x^2/a^2 + y^2/b^2 = 1*/
#ifdef DEBUG
   printf("x=%d y = %d, x1=%d,x2=%d,y1=%d,y2=%d\n",
	  x/SCALE,y/SCALE,x1/SCALE,x2/SCALE,y1/SCALE,y2/SCALE);
#endif
   a2 = (x2 -x1)*(x2-x1)>>SCALE_SFT+2;
   b2 = (y2-y1)*(y2-y1)>>SCALE_SFT+2;
   cx = (x2+x1)>>1;
   cy = (y2+y1)>>1;
   return (((x-cx)*(x-cx)>>SCALE_SFT)*b2>>SCALE_SFT)
	 +(((y-cy)*(y-cy)>>SCALE_SFT)*a2>>SCALE_SFT)
	 < a2 * b2>>SCALE_SFT;
}

static
int is_in(int i,int flag,int x,int y){
   int cx,cy;

   calc_center(i,&cx,&cy);

   switch(flag) {
   case BUTTON_ALL:
        return in_rect(cx- rx*2, cy -ry*2, rx*4+1,ry*4+1, x, y);
   case BUTTON_LARGE:
	return in_circle(cx - rx*2, cy - ry*2, cx + rx*2, cy + rx*2,x,y);
   case BUTTON_MEDIUM:
	return in_circle(cx - rx*5/4,cy - ry*5/4, cx + rx*5/4, cy + rx*5/4,x,y);
   case BUTTON_SMALL:
	return in_circle(cx - rx, cy - ry, cx + rx, cy + rx,x,y);
   }
}

static
int translate(int *buf,int cnt) {
   int l[10];
   int i,old,x,y;
   char *temp;

   mtostr_button_type(l);
   old = 0;
   temp = result_buf;
   fprintf(stderr,"translate!!\n");
   if(cnt <= 2){
      temp[0] = '5';
      temp[1] = '\0';
      return ;
   }
   while(cnt > 0){
      x = *buf++;
      y = *buf++;
      cnt -= 2;
	i = which_area(x,y);
	if(is_in(i,l[i],x,y)) {
	    if(old != i){
	       *temp++ = i+ '0';
	       old = i;
	    }
	}
   }
   fprintf(stderr,"end translate!!\n");
   *temp = '\0';
}

char* mtostr_translate(){
   if(translate(buf,buf_cnt)){
      return result_buf;
   }
   return NULL;
}

#define ABS(a) (((a)<0) ? -(a) : (a))

static int plot_line(int *buf,int max_cnt,int x1, int y1, int x2, int y2) {
    int dx = x2 - x1;
    int dy = y2 - y1;
    int ax = ABS(dx) << 1;
    int ay = ABS(dy) << 1;
    int sx = (dx >= 0) ? SCALE : -SCALE;
    int sy = (dy >= 0) ? SCALE : -SCALE;

    int x = x1;
    int y = y1;
    int ret = 0;

    if (ax > ay) {
	int d = ay - (ax >> 1);
	while (x != x2) {
	    if ((max_cnt > 2) && ((x1 != x) || (y1 != y))) {
	        *buf++ = x;
	        *buf++ = y;
		max_cnt -= 2;
		ret += 2;
	    }

	    if (d > 0 || (d == 0 && sx == 1)) {
		y += sy;
		d -= ax;
	    }
	    x += sx;
	    d += ay;
	}
    } else {
	int d = ax - (ay >> 1);
	while (y != y2) {
	    if ((max_cnt > 2) && ((x1 != x) || (y1 != y))) {
	        *buf++ = x;
	        *buf++ = y;
		max_cnt -= 2;
		ret += 2;
	    }

	    if (d > 0 || (d == 0 && sy == 1)) {
		x += sx;
		d -= ay;
	    }
	    y += sy;
	    d += ax;
	}
    }
   if ((max_cnt > 2) && ((x1 != x) || (y1 != y))) {
        *buf++ = x;
        *buf++ = y;
	max_cnt -= 2;
	ret += 2;
  }
    return ret;
}
