/*
 * SPDX-License-Identifier: GPL-3.0-only
 * MuseScore-Studio-CLA-applies
 *
 * MuseScore Studio
 * Music Composition & Notation
 *
 * Copyright (C) 2021 MuseScore Limited
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
#include "symnames.h"

#include "containers.h"
#include "translation.h"

#include "log.h"

using namespace muse;
using namespace mu;
using namespace mu::engraving;

std::map<AsciiStringView, SymId> SymNames::s_nameToSymIdHash = {};

AsciiStringView SymNames::nameForSymId(SymId id)
{
    return s_symNames.at(size_t(id));
}

const muse::TranslatableString& SymNames::userNameForSymId(SymId id)
{
    return s_symUserNames.at(size_t(id));
}

String SymNames::translatedUserNameForSymId(SymId id)
{
    return userNameForSymId(id).translated();
}

SymId SymNames::symIdByName(const AsciiStringView& name, SymId def)
{
    if (s_nameToSymIdHash.empty()) {
        loadNameToSymIdHash();
    }
    return muse::value(s_nameToSymIdHash, name, def);
}

SymId SymNames::symIdByName(const String& name, SymId def)
{
    ByteArray ba = name.toAscii();
    return symIdByName(AsciiStringView(ba.constChar()), def);
}

SymId SymNames::symIdByOldName(const AsciiStringView& oldName)
{
    return muse::value(s_oldNameToSymIdHash, oldName, SymId::noSym);
}

SymId SymNames::symIdByUserName(const String& userName)
{
    size_t index = 0;
    for (const muse::TranslatableString& n : s_symUserNames) {
        if (userName == n.translated()) {
            return static_cast<SymId>(index);
        }
        index++;
    }
    return SymId::noSym;
}

void SymNames::loadNameToSymIdHash()
{
    TRACEFUNC; // Should be called 0 or 1 times.

    for (size_t i = 0; i < s_symNames.size(); ++i) {
        s_nameToSymIdHash.insert({ s_symNames[i], static_cast<SymId>(i) });
    }
}

// must be in sync with SymId in symid_p.h
// should be in sync with s_symUserNames further down

constexpr const std::array<AsciiStringView, size_t(SymId::lastSym) + 1> SymNames::s_symNames { {
    "noSym",

    // DO NOT edit the SMuFL standard symbol names (see below) manually!
    // They are generated from tools/fonttools/smufl2sym.{bat,sh} and then
    // copied into this file!

    // SMuFL standard symbol names {{{
    "4stringTabClef",
    "6stringTabClef",
    "accSagittal11LargeDiesisDown",
    "accSagittal11LargeDiesisUp",
    "accSagittal11MediumDiesisDown",
    "accSagittal11MediumDiesisUp",
    "accSagittal11v19LargeDiesisDown",
    "accSagittal11v19LargeDiesisUp",
    "accSagittal11v19MediumDiesisDown",
    "accSagittal11v19MediumDiesisUp",
    "accSagittal11v49CommaDown",
    "accSagittal11v49CommaUp",
    "accSagittal143CommaDown",
    "accSagittal143CommaUp",
    "accSagittal17CommaDown",
    "accSagittal17CommaUp",
    "accSagittal17KleismaDown",
    "accSagittal17KleismaUp",
    "accSagittal19CommaDown",
    "accSagittal19CommaUp",
    "accSagittal19SchismaDown",
    "accSagittal19SchismaUp",
    "accSagittal1MinaDown",
    "accSagittal1MinaUp",
    "accSagittal1TinaDown",
    "accSagittal1TinaUp",
    "accSagittal23CommaDown",
    "accSagittal23CommaUp",
    "accSagittal23SmallDiesisDown",
    "accSagittal23SmallDiesisUp",
    "accSagittal25SmallDiesisDown",
    "accSagittal25SmallDiesisUp",
    "accSagittal2MinasDown",
    "accSagittal2MinasUp",
    "accSagittal2TinasDown",
    "accSagittal2TinasUp",
    "accSagittal35LargeDiesisDown",
    "accSagittal35LargeDiesisUp",
    "accSagittal35MediumDiesisDown",
    "accSagittal35MediumDiesisUp",
    "accSagittal3TinasDown",
    "accSagittal3TinasUp",
    "accSagittal49LargeDiesisDown",
    "accSagittal49LargeDiesisUp",
    "accSagittal49MediumDiesisDown",
    "accSagittal49MediumDiesisUp",
    "accSagittal49SmallDiesisDown",
    "accSagittal49SmallDiesisUp",
    "accSagittal4TinasDown",
    "accSagittal4TinasUp",
    "accSagittal55CommaDown",
    "accSagittal55CommaUp",
    "accSagittal5CommaDown",
    "accSagittal5CommaUp",
    "accSagittal5TinasDown",
    "accSagittal5TinasUp",
    "accSagittal5v11SmallDiesisDown",
    "accSagittal5v11SmallDiesisUp",
    "accSagittal5v13LargeDiesisDown",
    "accSagittal5v13LargeDiesisUp",
    "accSagittal5v13MediumDiesisDown",
    "accSagittal5v13MediumDiesisUp",
    "accSagittal5v19CommaDown",
    "accSagittal5v19CommaUp",
    "accSagittal5v23SmallDiesisDown",
    "accSagittal5v23SmallDiesisUp",
    "accSagittal5v49MediumDiesisDown",
    "accSagittal5v49MediumDiesisUp",
    "accSagittal5v7KleismaDown",
    "accSagittal5v7KleismaUp",
    "accSagittal6TinasDown",
    "accSagittal6TinasUp",
    "accSagittal7CommaDown",
    "accSagittal7CommaUp",
    "accSagittal7TinasDown",
    "accSagittal7TinasUp",
    "accSagittal7v11CommaDown",
    "accSagittal7v11CommaUp",
    "accSagittal7v11KleismaDown",
    "accSagittal7v11KleismaUp",
    "accSagittal7v19CommaDown",
    "accSagittal7v19CommaUp",
    "accSagittal8TinasDown",
    "accSagittal8TinasUp",
    "accSagittal9TinasDown",
    "accSagittal9TinasUp",
    "accSagittalAcute",
    "accSagittalDoubleFlat",
    "accSagittalDoubleFlat11v49CUp",
    "accSagittalDoubleFlat143CUp",
    "accSagittalDoubleFlat17CUp",
    "accSagittalDoubleFlat17kUp",
    "accSagittalDoubleFlat19CUp",
    "accSagittalDoubleFlat19sUp",
    "accSagittalDoubleFlat23CUp",
    "accSagittalDoubleFlat23SUp",
    "accSagittalDoubleFlat25SUp",
    "accSagittalDoubleFlat49SUp",
    "accSagittalDoubleFlat55CUp",
    "accSagittalDoubleFlat5CUp",
    "accSagittalDoubleFlat5v11SUp",
    "accSagittalDoubleFlat5v19CUp",
    "accSagittalDoubleFlat5v23SUp",
    "accSagittalDoubleFlat5v7kUp",
    "accSagittalDoubleFlat7CUp",
    "accSagittalDoubleFlat7v11CUp",
    "accSagittalDoubleFlat7v11kUp",
    "accSagittalDoubleFlat7v19CUp",
    "accSagittalDoubleSharp",
    "accSagittalDoubleSharp11v49CDown",
    "accSagittalDoubleSharp143CDown",
    "accSagittalDoubleSharp17CDown",
    "accSagittalDoubleSharp17kDown",
    "accSagittalDoubleSharp19CDown",
    "accSagittalDoubleSharp19sDown",
    "accSagittalDoubleSharp23CDown",
    "accSagittalDoubleSharp23SDown",
    "accSagittalDoubleSharp25SDown",
    "accSagittalDoubleSharp49SDown",
    "accSagittalDoubleSharp55CDown",
    "accSagittalDoubleSharp5CDown",
    "accSagittalDoubleSharp5v11SDown",
    "accSagittalDoubleSharp5v19CDown",
    "accSagittalDoubleSharp5v23SDown",
    "accSagittalDoubleSharp5v7kDown",
    "accSagittalDoubleSharp7CDown",
    "accSagittalDoubleSharp7v11CDown",
    "accSagittalDoubleSharp7v11kDown",
    "accSagittalDoubleSharp7v19CDown",
    "accSagittalFlat",
    "accSagittalFlat11LDown",
    "accSagittalFlat11MDown",
    "accSagittalFlat11v19LDown",
    "accSagittalFlat11v19MDown",
    "accSagittalFlat11v49CDown",
    "accSagittalFlat11v49CUp",
    "accSagittalFlat143CDown",
    "accSagittalFlat143CUp",
    "accSagittalFlat17CDown",
    "accSagittalFlat17CUp",
    "accSagittalFlat17kDown",
    "accSagittalFlat17kUp",
    "accSagittalFlat19CDown",
    "accSagittalFlat19CUp",
    "accSagittalFlat19sDown",
    "accSagittalFlat19sUp",
    "accSagittalFlat23CDown",
    "accSagittalFlat23CUp",
    "accSagittalFlat23SDown",
    "accSagittalFlat23SUp",
    "accSagittalFlat25SDown",
    "accSagittalFlat25SUp",
    "accSagittalFlat35LDown",
    "accSagittalFlat35MDown",
    "accSagittalFlat49LDown",
    "accSagittalFlat49MDown",
    "accSagittalFlat49SDown",
    "accSagittalFlat49SUp",
    "accSagittalFlat55CDown",
    "accSagittalFlat55CUp",
    "accSagittalFlat5CDown",
    "accSagittalFlat5CUp",
    "accSagittalFlat5v11SDown",
    "accSagittalFlat5v11SUp",
    "accSagittalFlat5v13LDown",
    "accSagittalFlat5v13MDown",
    "accSagittalFlat5v19CDown",
    "accSagittalFlat5v19CUp",
    "accSagittalFlat5v23SDown",
    "accSagittalFlat5v23SUp",
    "accSagittalFlat5v49MDown",
    "accSagittalFlat5v7kDown",
    "accSagittalFlat5v7kUp",
    "accSagittalFlat7CDown",
    "accSagittalFlat7CUp",
    "accSagittalFlat7v11CDown",
    "accSagittalFlat7v11CUp",
    "accSagittalFlat7v11kDown",
    "accSagittalFlat7v11kUp",
    "accSagittalFlat7v19CDown",
    "accSagittalFlat7v19CUp",
    "accSagittalFractionalTinaDown",
    "accSagittalFractionalTinaUp",
    "accSagittalGrave",
    "accSagittalShaftDown",
    "accSagittalShaftUp",
    "accSagittalSharp",
    "accSagittalSharp11LUp",
    "accSagittalSharp11MUp",
    "accSagittalSharp11v19LUp",
    "accSagittalSharp11v19MUp",
    "accSagittalSharp11v49CDown",
    "accSagittalSharp11v49CUp",
    "accSagittalSharp143CDown",
    "accSagittalSharp143CUp",
    "accSagittalSharp17CDown",
    "accSagittalSharp17CUp",
    "accSagittalSharp17kDown",
    "accSagittalSharp17kUp",
    "accSagittalSharp19CDown",
    "accSagittalSharp19CUp",
    "accSagittalSharp19sDown",
    "accSagittalSharp19sUp",
    "accSagittalSharp23CDown",
    "accSagittalSharp23CUp",
    "accSagittalSharp23SDown",
    "accSagittalSharp23SUp",
    "accSagittalSharp25SDown",
    "accSagittalSharp25SUp",
    "accSagittalSharp35LUp",
    "accSagittalSharp35MUp",
    "accSagittalSharp49LUp",
    "accSagittalSharp49MUp",
    "accSagittalSharp49SDown",
    "accSagittalSharp49SUp",
    "accSagittalSharp55CDown",
    "accSagittalSharp55CUp",
    "accSagittalSharp5CDown",
    "accSagittalSharp5CUp",
    "accSagittalSharp5v11SDown",
    "accSagittalSharp5v11SUp",
    "accSagittalSharp5v13LUp",
    "accSagittalSharp5v13MUp",
    "accSagittalSharp5v19CDown",
    "accSagittalSharp5v19CUp",
    "accSagittalSharp5v23SDown",
    "accSagittalSharp5v23SUp",
    "accSagittalSharp5v49MUp",
    "accSagittalSharp5v7kDown",
    "accSagittalSharp5v7kUp",
    "accSagittalSharp7CDown",
    "accSagittalSharp7CUp",
    "accSagittalSharp7v11CDown",
    "accSagittalSharp7v11CUp",
    "accSagittalSharp7v11kDown",
    "accSagittalSharp7v11kUp",
    "accSagittalSharp7v19CDown",
    "accSagittalSharp7v19CUp",
    "accSagittalUnused1",
    "accSagittalUnused2",
    "accSagittalUnused3",
    "accSagittalUnused4",
    "accdnCombDot",
    "accdnCombLH2RanksEmpty",
    "accdnCombLH3RanksEmptySquare",
    "accdnCombRH3RanksEmpty",
    "accdnCombRH4RanksEmpty",
    "accdnDiatonicClef",
    "accdnLH2Ranks16Round",
    "accdnLH2Ranks8Plus16Round",
    "accdnLH2Ranks8Round",
    "accdnLH2RanksFullMasterRound",
    "accdnLH2RanksMasterPlus16Round",
    "accdnLH2RanksMasterRound",
    "accdnLH3Ranks2Plus8Square",
    "accdnLH3Ranks2Square",
    "accdnLH3Ranks8Square",
    "accdnLH3RanksDouble8Square",
    "accdnLH3RanksTuttiSquare",
    "accdnPull",
    "accdnPush",
    "accdnRH3RanksAccordion",
    "accdnRH3RanksAuthenticMusette",
    "accdnRH3RanksBandoneon",
    "accdnRH3RanksBassoon",
    "accdnRH3RanksClarinet",
    "accdnRH3RanksDoubleTremoloLower8ve",
    "accdnRH3RanksDoubleTremoloUpper8ve",
    "accdnRH3RanksFullFactory",
    "accdnRH3RanksHarmonium",
    "accdnRH3RanksImitationMusette",
    "accdnRH3RanksLowerTremolo8",
    "accdnRH3RanksMaster",
    "accdnRH3RanksOboe",
    "accdnRH3RanksOrgan",
    "accdnRH3RanksPiccolo",
    "accdnRH3RanksTremoloLower8ve",
    "accdnRH3RanksTremoloUpper8ve",
    "accdnRH3RanksTwoChoirs",
    "accdnRH3RanksUpperTremolo8",
    "accdnRH3RanksViolin",
    "accdnRH4RanksAlto",
    "accdnRH4RanksBassAlto",
    "accdnRH4RanksMaster",
    "accdnRH4RanksSoftBass",
    "accdnRH4RanksSoftTenor",
    "accdnRH4RanksSoprano",
    "accdnRH4RanksTenor",
    "accdnRicochet2",
    "accdnRicochet3",
    "accdnRicochet4",
    "accdnRicochet5",
    "accdnRicochet6",
    "accdnRicochetStem2",
    "accdnRicochetStem3",
    "accdnRicochetStem4",
    "accdnRicochetStem5",
    "accdnRicochetStem6",
    "accidental1CommaFlat",
    "accidental1CommaSharp",
    "accidental2CommaFlat",
    "accidental2CommaSharp",
    "accidental3CommaFlat",
    "accidental3CommaSharp",
    "accidental4CommaFlat",
    "accidental5CommaSharp",
    "accidentalArrowDown",
    "accidentalArrowUp",
    "accidentalBakiyeFlat",
    "accidentalBakiyeSharp",
    "accidentalBracketLeft",
    "accidentalBracketRight",
    "accidentalBuyukMucennebFlat",
    "accidentalBuyukMucennebSharp",
    "accidentalCombiningCloseCurlyBrace",
    "accidentalCombiningLower17Schisma",
    "accidentalCombiningLower19Schisma",
    "accidentalCombiningLower23Limit29LimitComma",
    "accidentalCombiningLower29LimitComma",
    "accidentalCombiningLower31Schisma",
    "accidentalCombiningLower37Quartertone",
    "accidentalCombiningLower41Comma",
    "accidentalCombiningLower43Comma",
    "accidentalCombiningLower47Quartertone",
    "accidentalCombiningLower53LimitComma",
    "accidentalCombiningOpenCurlyBrace",
    "accidentalCombiningRaise17Schisma",
    "accidentalCombiningRaise19Schisma",
    "accidentalCombiningRaise23Limit29LimitComma",
    "accidentalCombiningRaise29LimitComma",
    "accidentalCombiningRaise31Schisma",
    "accidentalCombiningRaise37Quartertone",
    "accidentalCombiningRaise41Comma",
    "accidentalCombiningRaise43Comma",
    "accidentalCombiningRaise47Quartertone",
    "accidentalCombiningRaise53LimitComma",
    "accidentalCommaSlashDown",
    "accidentalCommaSlashUp",
    "accidentalDoubleFlat",
    "accidentalDoubleFlatArabic",
    "accidentalDoubleFlatEqualTempered",
    "accidentalDoubleFlatOneArrowDown",
    "accidentalDoubleFlatOneArrowUp",
    "accidentalDoubleFlatReversed",
    "accidentalDoubleFlatThreeArrowsDown",
    "accidentalDoubleFlatThreeArrowsUp",
    "accidentalDoubleFlatTurned",
    "accidentalDoubleFlatTwoArrowsDown",
    "accidentalDoubleFlatTwoArrowsUp",
    "accidentalDoubleSharp",
    "accidentalDoubleSharpArabic",
    "accidentalDoubleSharpEqualTempered",
    "accidentalDoubleSharpOneArrowDown",
    "accidentalDoubleSharpOneArrowUp",
    "accidentalDoubleSharpThreeArrowsDown",
    "accidentalDoubleSharpThreeArrowsUp",
    "accidentalDoubleSharpTwoArrowsDown",
    "accidentalDoubleSharpTwoArrowsUp",
    "accidentalEnharmonicAlmostEqualTo",
    "accidentalEnharmonicEquals",
    "accidentalEnharmonicTilde",
    "accidentalFilledReversedFlatAndFlat",
    "accidentalFilledReversedFlatAndFlatArrowDown",
    "accidentalFilledReversedFlatAndFlatArrowUp",
    "accidentalFilledReversedFlatArrowDown",
    "accidentalFilledReversedFlatArrowUp",
    "accidentalFiveQuarterTonesFlatArrowDown",
    "accidentalFiveQuarterTonesSharpArrowUp",
    "accidentalFlat",
    "accidentalFlatArabic",
    "accidentalFlatEqualTempered",
    "accidentalFlatLoweredStockhausen",
    "accidentalFlatOneArrowDown",
    "accidentalFlatOneArrowUp",
    "accidentalFlatRaisedStockhausen",
    "accidentalFlatRepeatedLineStockhausen",
    "accidentalFlatRepeatedSpaceStockhausen",
    "accidentalFlatThreeArrowsDown",
    "accidentalFlatThreeArrowsUp",
    "accidentalFlatTurned",
    "accidentalFlatTwoArrowsDown",
    "accidentalFlatTwoArrowsUp",
    "accidentalHabaFlatQuarterToneHigher",
    "accidentalHabaFlatThreeQuarterTonesLower",
    "accidentalHabaQuarterToneHigher",
    "accidentalHabaQuarterToneLower",
    "accidentalHabaSharpQuarterToneLower",
    "accidentalHabaSharpThreeQuarterTonesHigher",
    "accidentalHalfSharpArrowDown",
    "accidentalHalfSharpArrowUp",
    "accidentalJohnston13",
    "accidentalJohnston31",
    "accidentalJohnstonDown",
    "accidentalJohnstonEl",
    "accidentalJohnstonMinus",
    "accidentalJohnstonPlus",
    "accidentalJohnstonSeven",
    "accidentalJohnstonUp",
    "accidentalKomaFlat",
    "accidentalKomaSharp",
    "accidentalKoron",
    "accidentalKucukMucennebFlat",
    "accidentalKucukMucennebSharp",
    "accidentalLargeDoubleSharp",
    "accidentalLowerOneSeptimalComma",
    "accidentalLowerOneTridecimalQuartertone",
    "accidentalLowerOneUndecimalQuartertone",
    "accidentalLowerTwoSeptimalCommas",
    "accidentalLoweredStockhausen",
    "accidentalNarrowReversedFlat",
    "accidentalNarrowReversedFlatAndFlat",
    "accidentalNatural",
    "accidentalNaturalArabic",
    "accidentalNaturalEqualTempered",
    "accidentalNaturalFlat",
    "accidentalNaturalLoweredStockhausen",
    "accidentalNaturalOneArrowDown",
    "accidentalNaturalOneArrowUp",
    "accidentalNaturalRaisedStockhausen",
    "accidentalNaturalReversed",
    "accidentalNaturalSharp",
    "accidentalNaturalThreeArrowsDown",
    "accidentalNaturalThreeArrowsUp",
    "accidentalNaturalTwoArrowsDown",
    "accidentalNaturalTwoArrowsUp",
    "accidentalOneAndAHalfSharpsArrowDown",
    "accidentalOneAndAHalfSharpsArrowUp",
    "accidentalOneQuarterToneFlatFerneyhough",
    "accidentalOneQuarterToneFlatStockhausen",
    "accidentalOneQuarterToneSharpFerneyhough",
    "accidentalOneQuarterToneSharpStockhausen",
    "accidentalOneThirdToneFlatFerneyhough",
    "accidentalOneThirdToneSharpFerneyhough",
    "accidentalParensLeft",
    "accidentalParensRight",
    "accidentalQuarterFlatEqualTempered",
    "accidentalQuarterSharpEqualTempered",
    "accidentalQuarterToneFlat4",
    "accidentalQuarterToneFlatArabic",
    "accidentalQuarterToneFlatArrowUp",
    "accidentalQuarterToneFlatFilledReversed",
    "accidentalQuarterToneFlatNaturalArrowDown",
    "accidentalQuarterToneFlatPenderecki",
    "accidentalQuarterToneFlatStein",
    "accidentalQuarterToneFlatVanBlankenburg",
    "accidentalQuarterToneSharp4",
    "accidentalQuarterToneSharpArabic",
    "accidentalQuarterToneSharpArrowDown",
    "accidentalQuarterToneSharpBusotti",
    "accidentalQuarterToneSharpNaturalArrowUp",
    "accidentalQuarterToneSharpStein",
    "accidentalQuarterToneSharpWiggle",
    "accidentalRaiseOneSeptimalComma",
    "accidentalRaiseOneTridecimalQuartertone",
    "accidentalRaiseOneUndecimalQuartertone",
    "accidentalRaiseTwoSeptimalCommas",
    "accidentalRaisedStockhausen",
    "accidentalReversedFlatAndFlatArrowDown",
    "accidentalReversedFlatAndFlatArrowUp",
    "accidentalReversedFlatArrowDown",
    "accidentalReversedFlatArrowUp",
    "accidentalSharp",
    "accidentalSharpArabic",
    "accidentalSharpEqualTempered",
    "accidentalSharpLoweredStockhausen",
    "accidentalSharpOneArrowDown",
    "accidentalSharpOneArrowUp",
    "accidentalSharpOneHorizontalStroke",
    "accidentalSharpRaisedStockhausen",
    "accidentalSharpRepeatedLineStockhausen",
    "accidentalSharpRepeatedSpaceStockhausen",
    "accidentalSharpReversed",
    "accidentalSharpSharp",
    "accidentalSharpThreeArrowsDown",
    "accidentalSharpThreeArrowsUp",
    "accidentalSharpTwoArrowsDown",
    "accidentalSharpTwoArrowsUp",
    "accidentalSims12Down",
    "accidentalSims12Up",
    "accidentalSims4Down",
    "accidentalSims4Up",
    "accidentalSims6Down",
    "accidentalSims6Up",
    "accidentalSori",
    "accidentalTavenerFlat",
    "accidentalTavenerSharp",
    "accidentalThreeQuarterTonesFlatArabic",
    "accidentalThreeQuarterTonesFlatArrowDown",
    "accidentalThreeQuarterTonesFlatArrowUp",
    "accidentalThreeQuarterTonesFlatCouper",
    "accidentalThreeQuarterTonesFlatGrisey",
    "accidentalThreeQuarterTonesFlatTartini",
    "accidentalThreeQuarterTonesFlatZimmermann",
    "accidentalThreeQuarterTonesSharpArabic",
    "accidentalThreeQuarterTonesSharpArrowDown",
    "accidentalThreeQuarterTonesSharpArrowUp",
    "accidentalThreeQuarterTonesSharpBusotti",
    "accidentalThreeQuarterTonesSharpStein",
    "accidentalThreeQuarterTonesSharpStockhausen",
    "accidentalTripleFlat",
    "accidentalTripleSharp",
    "accidentalTwoThirdTonesFlatFerneyhough",
    "accidentalTwoThirdTonesSharpFerneyhough",
    "accidentalUpsAndDownsDown",
    "accidentalUpsAndDownsLess",
    "accidentalUpsAndDownsMore",
    "accidentalUpsAndDownsUp",
    "accidentalWilsonMinus",
    "accidentalWilsonPlus",
    "accidentalWyschnegradsky10TwelfthsFlat",
    "accidentalWyschnegradsky10TwelfthsSharp",
    "accidentalWyschnegradsky11TwelfthsFlat",
    "accidentalWyschnegradsky11TwelfthsSharp",
    "accidentalWyschnegradsky1TwelfthsFlat",
    "accidentalWyschnegradsky1TwelfthsSharp",
    "accidentalWyschnegradsky2TwelfthsFlat",
    "accidentalWyschnegradsky2TwelfthsSharp",
    "accidentalWyschnegradsky3TwelfthsFlat",
    "accidentalWyschnegradsky3TwelfthsSharp",
    "accidentalWyschnegradsky4TwelfthsFlat",
    "accidentalWyschnegradsky4TwelfthsSharp",
    "accidentalWyschnegradsky5TwelfthsFlat",
    "accidentalWyschnegradsky5TwelfthsSharp",
    "accidentalWyschnegradsky6TwelfthsFlat",
    "accidentalWyschnegradsky6TwelfthsSharp",
    "accidentalWyschnegradsky7TwelfthsFlat",
    "accidentalWyschnegradsky7TwelfthsSharp",
    "accidentalWyschnegradsky8TwelfthsFlat",
    "accidentalWyschnegradsky8TwelfthsSharp",
    "accidentalWyschnegradsky9TwelfthsFlat",
    "accidentalWyschnegradsky9TwelfthsSharp",
    "accidentalXenakisOneThirdToneSharp",
    "accidentalXenakisTwoThirdTonesSharp",
    "analyticsChoralmelodie",
    "analyticsEndStimme",
    "analyticsHauptrhythmus",
    "analyticsHauptstimme",
    "analyticsInversion1",
    "analyticsNebenstimme",
    "analyticsStartStimme",
    "analyticsTheme",
    "analyticsTheme1",
    "analyticsThemeInversion",
    "analyticsThemeRetrograde",
    "analyticsThemeRetrogradeInversion",
    "arpeggiato",
    "arpeggiatoDown",
    "arpeggiatoUp",
    "arrowBlackDown",
    "arrowBlackDownLeft",
    "arrowBlackDownRight",
    "arrowBlackLeft",
    "arrowBlackRight",
    "arrowBlackUp",
    "arrowBlackUpLeft",
    "arrowBlackUpRight",
    "arrowOpenDown",
    "arrowOpenDownLeft",
    "arrowOpenDownRight",
    "arrowOpenLeft",
    "arrowOpenRight",
    "arrowOpenUp",
    "arrowOpenUpLeft",
    "arrowOpenUpRight",
    "arrowWhiteDown",
    "arrowWhiteDownLeft",
    "arrowWhiteDownRight",
    "arrowWhiteLeft",
    "arrowWhiteRight",
    "arrowWhiteUp",
    "arrowWhiteUpLeft",
    "arrowWhiteUpRight",
    "arrowheadBlackDown",
    "arrowheadBlackDownLeft",
    "arrowheadBlackDownRight",
    "arrowheadBlackLeft",
    "arrowheadBlackRight",
    "arrowheadBlackUp",
    "arrowheadBlackUpLeft",
    "arrowheadBlackUpRight",
    "arrowheadOpenDown",
    "arrowheadOpenDownLeft",
    "arrowheadOpenDownRight",
    "arrowheadOpenLeft",
    "arrowheadOpenRight",
    "arrowheadOpenUp",
    "arrowheadOpenUpLeft",
    "arrowheadOpenUpRight",
    "arrowheadWhiteDown",
    "arrowheadWhiteDownLeft",
    "arrowheadWhiteDownRight",
    "arrowheadWhiteLeft",
    "arrowheadWhiteRight",
    "arrowheadWhiteUp",
    "arrowheadWhiteUpLeft",
    "arrowheadWhiteUpRight",
    "articAccentAbove",
    "articAccentBelow",
    "articAccentStaccatoAbove",
    "articAccentStaccatoBelow",
    "articLaissezVibrerAbove",
    "articLaissezVibrerBelow",
    "articMarcatoAbove",
    "articMarcatoBelow",
    "articMarcatoStaccatoAbove",
    "articMarcatoStaccatoBelow",
    "articMarcatoTenutoAbove",
    "articMarcatoTenutoBelow",
    "articSoftAccentAbove",
    "articSoftAccentBelow",
    "articSoftAccentStaccatoAbove",
    "articSoftAccentStaccatoBelow",
    "articSoftAccentTenutoAbove",
    "articSoftAccentTenutoBelow",
    "articSoftAccentTenutoStaccatoAbove",
    "articSoftAccentTenutoStaccatoBelow",
    "articStaccatissimoAbove",
    "articStaccatissimoBelow",
    "articStaccatissimoStrokeAbove",
    "articStaccatissimoStrokeBelow",
    "articStaccatissimoWedgeAbove",
    "articStaccatissimoWedgeBelow",
    "articStaccatoAbove",
    "articStaccatoBelow",
    "articStressAbove",
    "articStressBelow",
    "articTenutoAbove",
    "articTenutoAccentAbove",
    "articTenutoAccentBelow",
    "articTenutoBelow",
    "articTenutoStaccatoAbove",
    "articTenutoStaccatoBelow",
    "articUnstressAbove",
    "articUnstressBelow",
    "augmentationDot",
    "barlineDashed",
    "barlineDotted",
    "barlineDouble",
    "barlineFinal",
    "barlineHeavy",
    "barlineHeavyHeavy",
    "barlineReverseFinal",
    "barlineShort",
    "barlineSingle",
    "barlineTick",
    "beamAccelRit1",
    "beamAccelRit10",
    "beamAccelRit11",
    "beamAccelRit12",
    "beamAccelRit13",
    "beamAccelRit14",
    "beamAccelRit15",
    "beamAccelRit2",
    "beamAccelRit3",
    "beamAccelRit4",
    "beamAccelRit5",
    "beamAccelRit6",
    "beamAccelRit7",
    "beamAccelRit8",
    "beamAccelRit9",
    "beamAccelRitFinal",
    "brace",
    "bracket",
    "bracketBottom",
    "bracketTop",
    "brassBend",
    "brassDoitLong",
    "brassDoitMedium",
    "brassDoitShort",
    "brassFallLipLong",
    "brassFallLipMedium",
    "brassFallLipShort",
    "brassFallRoughLong",
    "brassFallRoughMedium",
    "brassFallRoughShort",
    "brassFallSmoothLong",
    "brassFallSmoothMedium",
    "brassFallSmoothShort",
    "brassFlip",
    "brassHarmonMuteClosed",
    "brassHarmonMuteStemHalfLeft",
    "brassHarmonMuteStemHalfRight",
    "brassHarmonMuteStemOpen",
    "brassJazzTurn",
    "brassLiftLong",
    "brassLiftMedium",
    "brassLiftShort",
    "brassLiftSmoothLong",
    "brassLiftSmoothMedium",
    "brassLiftSmoothShort",
    "brassMuteClosed",
    "brassMuteHalfClosed",
    "brassMuteOpen",
    "brassPlop",
    "brassScoop",
    "brassSmear",
    "brassValveTrill",
    "breathMarkComma",
    "breathMarkSalzedo",
    "breathMarkTick",
    "breathMarkUpbow",
    "bridgeClef",
    "buzzRoll",
    "cClef",
    "cClef8vb",
    "cClefArrowDown",
    "cClefArrowUp",
    "cClefChange",
    "cClefCombining",
    "cClefReversed",
    "cClefSquare",
    "caesura",
    "caesuraCurved",
    "caesuraShort",
    "caesuraSingleStroke",
    "caesuraThick",
    "chantAccentusAbove",
    "chantAccentusBelow",
    "chantAuctumAsc",
    "chantAuctumDesc",
    "chantAugmentum",
    "chantCaesura",
    "chantCclef",
    "chantCirculusAbove",
    "chantCirculusBelow",
    "chantConnectingLineAsc2nd",
    "chantConnectingLineAsc3rd",
    "chantConnectingLineAsc4th",
    "chantConnectingLineAsc5th",
    "chantConnectingLineAsc6th",
    "chantCustosStemDownPosHigh",
    "chantCustosStemDownPosHighest",
    "chantCustosStemDownPosMiddle",
    "chantCustosStemUpPosLow",
    "chantCustosStemUpPosLowest",
    "chantCustosStemUpPosMiddle",
    "chantDeminutumLower",
    "chantDeminutumUpper",
    "chantDivisioFinalis",
    "chantDivisioMaior",
    "chantDivisioMaxima",
    "chantDivisioMinima",
    "chantEntryLineAsc2nd",
    "chantEntryLineAsc3rd",
    "chantEntryLineAsc4th",
    "chantEntryLineAsc5th",
    "chantEntryLineAsc6th",
    "chantEpisema",
    "chantFclef",
    "chantIctusAbove",
    "chantIctusBelow",
    "chantLigaturaDesc2nd",
    "chantLigaturaDesc3rd",
    "chantLigaturaDesc4th",
    "chantLigaturaDesc5th",
    "chantOriscusAscending",
    "chantOriscusDescending",
    "chantOriscusLiquescens",
    "chantPodatusLower",
    "chantPodatusUpper",
    "chantPunctum",
    "chantPunctumCavum",
    "chantPunctumDeminutum",
    "chantPunctumInclinatum",
    "chantPunctumInclinatumAuctum",
    "chantPunctumInclinatumDeminutum",
    "chantPunctumLinea",
    "chantPunctumLineaCavum",
    "chantPunctumVirga",
    "chantPunctumVirgaReversed",
    "chantQuilisma",
    "chantSemicirculusAbove",
    "chantSemicirculusBelow",
    "chantStaff",
    "chantStaffNarrow",
    "chantStaffWide",
    "chantStrophicus",
    "chantStrophicusAuctus",
    "chantStrophicusLiquescens2nd",
    "chantStrophicusLiquescens3rd",
    "chantStrophicusLiquescens4th",
    "chantStrophicusLiquescens5th",
    "chantVirgula",
    "clef15",
    "clef8",
    "clefChangeCombining",
    "coda",
    "codaSquare",
    "conductorBeat2Compound",
    "conductorBeat2Simple",
    "conductorBeat3Compound",
    "conductorBeat3Simple",
    "conductorBeat4Compound",
    "conductorBeat4Simple",
    "conductorLeftBeat",
    "conductorRightBeat",
    "conductorStrongBeat",
    "conductorUnconducted",
    "conductorWeakBeat",
    "controlBeginBeam",
    "controlBeginPhrase",
    "controlBeginSlur",
    "controlBeginTie",
    "controlEndBeam",
    "controlEndPhrase",
    "controlEndSlur",
    "controlEndTie",
    "csymAccidentalDoubleFlat",
    "csymAccidentalDoubleSharp",
    "csymAccidentalFlat",
    "csymAccidentalNatural",
    "csymAccidentalSharp",
    "csymAccidentalTripleFlat",
    "csymAccidentalTripleSharp",
    "csymAlteredBassSlash",
    "csymAugmented",
    "csymBracketLeftTall",
    "csymBracketRightTall",
    "csymDiagonalArrangementSlash",
    "csymDiminished",
    "csymHalfDiminished",
    "csymMajorSeventh",
    "csymMinor",
    "csymParensLeftTall",
    "csymParensLeftVeryTall",
    "csymParensRightTall",
    "csymParensRightVeryTall",
    "curlewSign",
    "daCapo",
    "dalSegno",
    "daseianExcellentes1",
    "daseianExcellentes2",
    "daseianExcellentes3",
    "daseianExcellentes4",
    "daseianFinales1",
    "daseianFinales2",
    "daseianFinales3",
    "daseianFinales4",
    "daseianGraves1",
    "daseianGraves2",
    "daseianGraves3",
    "daseianGraves4",
    "daseianResidua1",
    "daseianResidua2",
    "daseianSuperiores1",
    "daseianSuperiores2",
    "daseianSuperiores3",
    "daseianSuperiores4",
    "doubleLateralRollStevens",
    "doubleTongueAbove",
    "doubleTongueBelow",
    "dynamicCombinedSeparatorColon",
    "dynamicCombinedSeparatorHyphen",
    "dynamicCombinedSeparatorSlash",
    "dynamicCombinedSeparatorSpace",
    "dynamicCrescendoHairpin",
    "dynamicDiminuendoHairpin",
    "dynamicFF",
    "dynamicFFF",
    "dynamicFFFF",
    "dynamicFFFFF",
    "dynamicFFFFFF",
    "dynamicForte",
    "dynamicFortePiano",
    "dynamicForzando",
    "dynamicHairpinBracketLeft",
    "dynamicHairpinBracketRight",
    "dynamicHairpinParenthesisLeft",
    "dynamicHairpinParenthesisRight",
    "dynamicMF",
    "dynamicMP",
    "dynamicMessaDiVoce",
    "dynamicMezzo",
    "dynamicNiente",
    "dynamicNienteForHairpin",
    "dynamicPF",
    "dynamicPP",
    "dynamicPPP",
    "dynamicPPPP",
    "dynamicPPPPP",
    "dynamicPPPPPP",
    "dynamicPiano",
    "dynamicRinforzando",
    "dynamicRinforzando1",
    "dynamicRinforzando2",
    "dynamicSforzando",
    "dynamicSforzando1",
    "dynamicSforzandoPianissimo",
    "dynamicSforzandoPiano",
    "dynamicSforzato",
    "dynamicSforzatoFF",
    "dynamicSforzatoPiano",
    "dynamicZ",
    "elecAudioChannelsEight",
    "elecAudioChannelsFive",
    "elecAudioChannelsFour",
    "elecAudioChannelsOne",
    "elecAudioChannelsSeven",
    "elecAudioChannelsSix",
    "elecAudioChannelsThreeFrontal",
    "elecAudioChannelsThreeSurround",
    "elecAudioChannelsTwo",
    "elecAudioIn",
    "elecAudioMono",
    "elecAudioOut",
    "elecAudioStereo",
    "elecCamera",
    "elecDataIn",
    "elecDataOut",
    "elecDisc",
    "elecDownload",
    "elecEject",
    "elecFastForward",
    "elecHeadphones",
    "elecHeadset",
    "elecLineIn",
    "elecLineOut",
    "elecLoop",
    "elecLoudspeaker",
    "elecMIDIController0",
    "elecMIDIController100",
    "elecMIDIController20",
    "elecMIDIController40",
    "elecMIDIController60",
    "elecMIDIController80",
    "elecMIDIIn",
    "elecMIDIOut",
    "elecMicrophone",
    "elecMicrophoneMute",
    "elecMicrophoneUnmute",
    "elecMixingConsole",
    "elecMonitor",
    "elecMute",
    "elecPause",
    "elecPlay",
    "elecPowerOnOff",
    "elecProjector",
    "elecReplay",
    "elecRewind",
    "elecShuffle",
    "elecSkipBackwards",
    "elecSkipForwards",
    "elecStop",
    "elecTape",
    "elecUSB",
    "elecUnmute",
    "elecUpload",
    "elecVideoCamera",
    "elecVideoIn",
    "elecVideoOut",
    "elecVolumeFader",
    "elecVolumeFaderThumb",
    "elecVolumeLevel0",
    "elecVolumeLevel100",
    "elecVolumeLevel20",
    "elecVolumeLevel40",
    "elecVolumeLevel60",
    "elecVolumeLevel80",
    "fClef",
    "fClef15ma",
    "fClef15mb",
    "fClef8va",
    "fClef8vb",
    "fClefArrowDown",
    "fClefArrowUp",
    "fClefChange",
    "fClefReversed",
    "fClefTurned",
    "fermataAbove",
    "fermataBelow",
    "fermataLongAbove",
    "fermataLongBelow",
    "fermataLongHenzeAbove",
    "fermataLongHenzeBelow",
    "fermataShortAbove",
    "fermataShortBelow",
    "fermataShortHenzeAbove",
    "fermataShortHenzeBelow",
    "fermataVeryLongAbove",
    "fermataVeryLongBelow",
    "fermataVeryShortAbove",
    "fermataVeryShortBelow",
    "figbass0",
    "figbass1",
    "figbass2",
    "figbass2Raised",
    "figbass3",
    "figbass4",
    "figbass4Raised",
    "figbass5",
    "figbass5Raised1",
    "figbass5Raised2",
    "figbass5Raised3",
    "figbass6",
    "figbass6Raised",
    "figbass6Raised2",
    "figbass7",
    "figbass7Diminished",
    "figbass7Raised1",
    "figbass7Raised2",
    "figbass8",
    "figbass9",
    "figbass9Raised",
    "figbassBracketLeft",
    "figbassBracketRight",
    "figbassCombiningLowering",
    "figbassCombiningRaising",
    "figbassDoubleFlat",
    "figbassDoubleSharp",
    "figbassFlat",
    "figbassNatural",
    "figbassParensLeft",
    "figbassParensRight",
    "figbassPlus",
    "figbassSharp",
    "figbassTripleFlat",
    "figbassTripleSharp",
    "fingering0",
    "fingering0Italic",
    "fingering1",
    "fingering1Italic",
    "fingering2",
    "fingering2Italic",
    "fingering3",
    "fingering3Italic",
    "fingering4",
    "fingering4Italic",
    "fingering5",
    "fingering5Italic",
    "fingering6",
    "fingering6Italic",
    "fingering7",
    "fingering7Italic",
    "fingering8",
    "fingering8Italic",
    "fingering9",
    "fingering9Italic",
    "fingeringALower",
    "fingeringCLower",
    "fingeringELower",
    "fingeringILower",
    "fingeringLeftBracket",
    "fingeringLeftBracketItalic",
    "fingeringLeftParenthesis",
    "fingeringLeftParenthesisItalic",
    "fingeringMLower",
    "fingeringMultipleNotes",
    "fingeringOLower",
    "fingeringPLower",
    "fingeringQLower",
    "fingeringRightBracket",
    "fingeringRightBracketItalic",
    "fingeringRightParenthesis",
    "fingeringRightParenthesisItalic",
    "fingeringSLower",
    "fingeringSeparatorMiddleDot",
    "fingeringSeparatorMiddleDotWhite",
    "fingeringSeparatorSlash",
    "fingeringSubstitutionAbove",
    "fingeringSubstitutionBelow",
    "fingeringSubstitutionDash",
    "fingeringTLower",
    "fingeringTUpper",
    "fingeringXLower",
    "flag1024thDown",
    "flag1024thUp",
    "flag128thDown",
    "flag128thUp",
    "flag16thDown",
    "flag16thUp",
    "flag256thDown",
    "flag256thUp",
    "flag32ndDown",
    "flag32ndUp",
    "flag512thDown",
    "flag512thUp",
    "flag64thDown",
    "flag64thUp",
    "flag8thDown",
    "flag8thUp",
    "flagInternalDown",
    "flagInternalUp",
    "fretboard3String",
    "fretboard3StringNut",
    "fretboard4String",
    "fretboard4StringNut",
    "fretboard5String",
    "fretboard5StringNut",
    "fretboard6String",
    "fretboard6StringNut",
    "fretboardFilledCircle",
    "fretboardO",
    "fretboardX",
    "functionAngleLeft",
    "functionAngleRight",
    "functionBracketLeft",
    "functionBracketRight",
    "functionDD",
    "functionDLower",
    "functionDUpper",
    "functionEight",
    "functionFUpper",
    "functionFive",
    "functionFour",
    "functionGLower",
    "functionGUpper",
    "functionGreaterThan",
    "functionILower",
    "functionIUpper",
    "functionKLower",
    "functionKUpper",
    "functionLLower",
    "functionLUpper",
    "functionLessThan",
    "functionMLower",
    "functionMUpper",
    "functionMinus",
    "functionNLower",
    "functionNUpper",
    "functionNUpperSuperscript",
    "functionNine",
    "functionOne",
    "functionPLower",
    "functionPUpper",
    "functionParensLeft",
    "functionParensRight",
    "functionPlus",
    "functionRLower",
    "functionRepetition1",
    "functionRepetition2",
    "functionRing",
    "functionSLower",
    "functionSSLower",
    "functionSSUpper",
    "functionSUpper",
    "functionSeven",
    "functionSix",
    "functionSlashedDD",
    "functionTLower",
    "functionTUpper",
    "functionThree",
    "functionTwo",
    "functionVLower",
    "functionVUpper",
    "functionZero",
    "gClef",
    "gClef15ma",
    "gClef15mb",
    "gClef8va",
    "gClef8vb",
    "gClef8vbCClef",
    "gClef8vbOld",
    "gClef8vbParens",
    "gClefArrowDown",
    "gClefArrowUp",
    "gClefChange",
    "gClefLigatedNumberAbove",
    "gClefLigatedNumberBelow",
    "gClefReversed",
    "gClefTurned",
    "glissandoDown",
    "glissandoUp",
    "graceNoteAcciaccaturaStemDown",
    "graceNoteAcciaccaturaStemUp",
    "graceNoteAppoggiaturaStemDown",
    "graceNoteAppoggiaturaStemUp",
    "graceNoteSlashStemDown",
    "graceNoteSlashStemUp",
    "guitarBarreFull",
    "guitarBarreHalf",
    "guitarClosePedal",
    "guitarFadeIn",
    "guitarFadeOut",
    "guitarGolpe",
    "guitarHalfOpenPedal",
    "guitarLeftHandTapping",
    "guitarOpenPedal",
    "guitarRightHandTapping",
    "guitarShake",
    "guitarString0",
    "guitarString1",
    "guitarString10",
    "guitarString11",
    "guitarString12",
    "guitarString13",
    "guitarString2",
    "guitarString3",
    "guitarString4",
    "guitarString5",
    "guitarString6",
    "guitarString7",
    "guitarString8",
    "guitarString9",
    "guitarStrumDown",
    "guitarStrumUp",
    "guitarVibratoBarDip",
    "guitarVibratoBarScoop",
    "guitarVibratoStroke",
    "guitarVolumeSwell",
    "guitarWideVibratoStroke",
    "handbellsBelltree",
    "handbellsDamp3",
    "handbellsEcho1",
    "handbellsEcho2",
    "handbellsGyro",
    "handbellsHandMartellato",
    "handbellsMalletBellOnTable",
    "handbellsMalletBellSuspended",
    "handbellsMalletLft",
    "handbellsMartellato",
    "handbellsMartellatoLift",
    "handbellsMutedMartellato",
    "handbellsPluckLift",
    "handbellsSwing",
    "handbellsSwingDown",
    "handbellsSwingUp",
    "handbellsTablePairBells",
    "handbellsTableSingleBell",
    "harpMetalRod",
    "harpPedalCentered",
    "harpPedalDivider",
    "harpPedalLowered",
    "harpPedalRaised",
    "harpSalzedoAeolianAscending",
    "harpSalzedoAeolianDescending",
    "harpSalzedoDampAbove",
    "harpSalzedoDampBelow",
    "harpSalzedoDampBothHands",
    "harpSalzedoDampLowStrings",
    "harpSalzedoFluidicSoundsLeft",
    "harpSalzedoFluidicSoundsRight",
    "harpSalzedoIsolatedSounds",
    "harpSalzedoMetallicSounds",
    "harpSalzedoMetallicSoundsOneString",
    "harpSalzedoMuffleTotally",
    "harpSalzedoOboicFlux",
    "harpSalzedoPlayUpperEnd",
    "harpSalzedoSlideWithSuppleness",
    "harpSalzedoSnareDrum",
    "harpSalzedoTamTamSounds",
    "harpSalzedoThunderEffect",
    "harpSalzedoTimpanicSounds",
    "harpSalzedoWhistlingSounds",
    "harpStringNoiseStem",
    "harpTuningKey",
    "harpTuningKeyGlissando",
    "harpTuningKeyHandle",
    "harpTuningKeyShank",
    "indianDrumClef",
    "kahnBackChug",
    "kahnBackFlap",
    "kahnBackRiff",
    "kahnBackRip",
    "kahnBallChange",
    "kahnBallDig",
    "kahnBrushBackward",
    "kahnBrushForward",
    "kahnChug",
    "kahnClap",
    "kahnDoubleSnap",
    "kahnDoubleWing",
    "kahnDrawStep",
    "kahnDrawTap",
    "kahnFlam",
    "kahnFlap",
    "kahnFlapStep",
    "kahnFlat",
    "kahnFleaHop",
    "kahnFleaTap",
    "kahnGraceTap",
    "kahnGraceTapChange",
    "kahnGraceTapHop",
    "kahnGraceTapStamp",
    "kahnHeel",
    "kahnHeelChange",
    "kahnHeelClick",
    "kahnHeelDrop",
    "kahnHeelStep",
    "kahnHeelTap",
    "kahnHop",
    "kahnJumpApart",
    "kahnJumpTogether",
    "kahnKneeInward",
    "kahnKneeOutward",
    "kahnLeap",
    "kahnLeapFlatFoot",
    "kahnLeapHeelClick",
    "kahnLeftCatch",
    "kahnLeftCross",
    "kahnLeftFoot",
    "kahnLeftToeStrike",
    "kahnLeftTurn",
    "kahnOverTheTop",
    "kahnOverTheTopTap",
    "kahnPull",
    "kahnPush",
    "kahnRiff",
    "kahnRiffle",
    "kahnRightCatch",
    "kahnRightCross",
    "kahnRightFoot",
    "kahnRightToeStrike",
    "kahnRightTurn",
    "kahnRip",
    "kahnRipple",
    "kahnScrape",
    "kahnScuff",
    "kahnScuffle",
    "kahnShuffle",
    "kahnSlam",
    "kahnSlap",
    "kahnSlideStep",
    "kahnSlideTap",
    "kahnSnap",
    "kahnStamp",
    "kahnStampStamp",
    "kahnStep",
    "kahnStepStamp",
    "kahnStomp",
    "kahnStompBrush",
    "kahnTap",
    "kahnToe",
    "kahnToeClick",
    "kahnToeDrop",
    "kahnToeStep",
    "kahnToeTap",
    "kahnTrench",
    "kahnWing",
    "kahnWingChange",
    "kahnZank",
    "kahnZink",
    "keyboardBebung2DotsAbove",
    "keyboardBebung2DotsBelow",
    "keyboardBebung3DotsAbove",
    "keyboardBebung3DotsBelow",
    "keyboardBebung4DotsAbove",
    "keyboardBebung4DotsBelow",
    "keyboardLeftPedalPictogram",
    "keyboardMiddlePedalPictogram",
    "keyboardPedalD",
    "keyboardPedalDot",
    "keyboardPedalE",
    "keyboardPedalHalf",
    "keyboardPedalHalf2",
    "keyboardPedalHalf3",
    "keyboardPedalHeel1",
    "keyboardPedalHeel2",
    "keyboardPedalHeel3",
    "keyboardPedalHeelToToe",
    "keyboardPedalHeelToe",
    "keyboardPedalHookEnd",
    "keyboardPedalHookStart",
    "keyboardPedalHyphen",
    "keyboardPedalP",
    "keyboardPedalParensLeft",
    "keyboardPedalParensRight",
    "keyboardPedalPed",
    "keyboardPedalS",
    "keyboardPedalSost",
    "keyboardPedalToe1",
    "keyboardPedalToe2",
    "keyboardPedalToeToHeel",
    "keyboardPedalUp",
    "keyboardPedalUpNotch",
    "keyboardPedalUpSpecial",
    "keyboardPlayWithLH",
    "keyboardPlayWithLHEnd",
    "keyboardPlayWithRH",
    "keyboardPlayWithRHEnd",
    "keyboardPluckInside",
    "keyboardRightPedalPictogram",
    "kievanAccidentalFlat",
    "kievanAccidentalSharp",
    "kievanAugmentationDot",
    "kievanCClef",
    "kievanEndingSymbol",
    "kievanNote8thStemDown",
    "kievanNote8thStemUp",
    "kievanNoteBeam",
    "kievanNoteHalfStaffLine",
    "kievanNoteHalfStaffSpace",
    "kievanNoteQuarterStemDown",
    "kievanNoteQuarterStemUp",
    "kievanNoteReciting",
    "kievanNoteWhole",
    "kievanNoteWholeFinal",
    "kodalyHandDo",
    "kodalyHandFa",
    "kodalyHandLa",
    "kodalyHandMi",
    "kodalyHandRe",
    "kodalyHandSo",
    "kodalyHandTi",
    "leftRepeatSmall",
    "legerLine",
    "legerLineNarrow",
    "legerLineWide",
    "luteBarlineEndRepeat",
    "luteBarlineFinal",
    "luteBarlineStartRepeat",
    "luteDuration16th",
    "luteDuration32nd",
    "luteDuration8th",
    "luteDurationDoubleWhole",
    "luteDurationHalf",
    "luteDurationQuarter",
    "luteDurationWhole",
    "luteFingeringRHFirst",
    "luteFingeringRHSecond",
    "luteFingeringRHThird",
    "luteFingeringRHThumb",
    "luteFrench10thCourse",
    "luteFrench7thCourse",
    "luteFrench8thCourse",
    "luteFrench9thCourse",
    "luteFrenchAppoggiaturaAbove",
    "luteFrenchAppoggiaturaBelow",
    "luteFrenchFretA",
    "luteFrenchFretB",
    "luteFrenchFretC",
    "luteFrenchFretD",
    "luteFrenchFretE",
    "luteFrenchFretF",
    "luteFrenchFretG",
    "luteFrenchFretH",
    "luteFrenchFretI",
    "luteFrenchFretK",
    "luteFrenchFretL",
    "luteFrenchFretM",
    "luteFrenchFretN",
    "luteFrenchMordentInverted",
    "luteFrenchMordentLower",
    "luteFrenchMordentUpper",
    "luteGermanALower",
    "luteGermanAUpper",
    "luteGermanBLower",
    "luteGermanBUpper",
    "luteGermanCLower",
    "luteGermanCUpper",
    "luteGermanDLower",
    "luteGermanDUpper",
    "luteGermanELower",
    "luteGermanEUpper",
    "luteGermanFLower",
    "luteGermanFUpper",
    "luteGermanGLower",
    "luteGermanGUpper",
    "luteGermanHLower",
    "luteGermanHUpper",
    "luteGermanILower",
    "luteGermanIUpper",
    "luteGermanKLower",
    "luteGermanKUpper",
    "luteGermanLLower",
    "luteGermanLUpper",
    "luteGermanMLower",
    "luteGermanMUpper",
    "luteGermanNLower",
    "luteGermanNUpper",
    "luteGermanOLower",
    "luteGermanPLower",
    "luteGermanQLower",
    "luteGermanRLower",
    "luteGermanSLower",
    "luteGermanTLower",
    "luteGermanVLower",
    "luteGermanXLower",
    "luteGermanYLower",
    "luteGermanZLower",
    "luteItalianClefCSolFaUt",
    "luteItalianClefFFaUt",
    "luteItalianFret0",
    "luteItalianFret1",
    "luteItalianFret2",
    "luteItalianFret3",
    "luteItalianFret4",
    "luteItalianFret5",
    "luteItalianFret6",
    "luteItalianFret7",
    "luteItalianFret8",
    "luteItalianFret9",
    "luteItalianHoldFinger",
    "luteItalianHoldNote",
    "luteItalianReleaseFinger",
    "luteItalianTempoFast",
    "luteItalianTempoNeitherFastNorSlow",
    "luteItalianTempoSlow",
    "luteItalianTempoSomewhatFast",
    "luteItalianTempoVerySlow",
    "luteItalianTimeTriple",
    "luteItalianTremolo",
    "luteItalianVibrato",
    "luteStaff6Lines",
    "luteStaff6LinesNarrow",
    "luteStaff6LinesWide",
    "lyricsElision",
    "lyricsElisionNarrow",
    "lyricsElisionWide",
    "lyricsHyphenBaseline",
    "lyricsHyphenBaselineNonBreaking",
    "lyricsTextRepeat",
    "medRenFlatHardB",
    "medRenFlatSoftB",
    "medRenFlatWithDot",
    "medRenGClefCMN",
    "medRenLiquescenceCMN",
    "medRenLiquescentAscCMN",
    "medRenLiquescentDescCMN",
    "medRenNatural",
    "medRenNaturalWithCross",
    "medRenOriscusCMN",
    "medRenPlicaCMN",
    "medRenPunctumCMN",
    "medRenQuilismaCMN",
    "medRenSharpCroix",
    "medRenStrophicusCMN",
    "mensuralAlterationSign",
    "mensuralBlackBrevis",
    "mensuralBlackBrevisVoid",
    "mensuralBlackDragma",
    "mensuralBlackLonga",
    "mensuralBlackMaxima",
    "mensuralBlackMinima",
    "mensuralBlackMinimaVoid",
    "mensuralBlackSemibrevis",
    "mensuralBlackSemibrevisCaudata",
    "mensuralBlackSemibrevisOblique",
    "mensuralBlackSemibrevisVoid",
    "mensuralBlackSemiminima",
    "mensuralCclef",
    "mensuralCclefPetrucciPosHigh",
    "mensuralCclefPetrucciPosHighest",
    "mensuralCclefPetrucciPosLow",
    "mensuralCclefPetrucciPosLowest",
    "mensuralCclefPetrucciPosMiddle",
    "mensuralColorationEndRound",
    "mensuralColorationEndSquare",
    "mensuralColorationStartRound",
    "mensuralColorationStartSquare",
    "mensuralCombStemDiagonal",
    "mensuralCombStemDown",
    "mensuralCombStemDownFlagExtended",
    "mensuralCombStemDownFlagFlared",
    "mensuralCombStemDownFlagFusa",
    "mensuralCombStemDownFlagLeft",
    "mensuralCombStemDownFlagRight",
    "mensuralCombStemDownFlagSemiminima",
    "mensuralCombStemUp",
    "mensuralCombStemUpFlagExtended",
    "mensuralCombStemUpFlagFlared",
    "mensuralCombStemUpFlagFusa",
    "mensuralCombStemUpFlagLeft",
    "mensuralCombStemUpFlagRight",
    "mensuralCombStemUpFlagSemiminima",
    "mensuralCustosCheckmark",
    "mensuralCustosDown",
    "mensuralCustosTurn",
    "mensuralCustosUp",
    "mensuralFclef",
    "mensuralFclefPetrucci",
    "mensuralGclef",
    "mensuralGclefPetrucci",
    "mensuralModusImperfectumVert",
    "mensuralModusPerfectumVert",
    "mensuralNoteheadLongaBlack",
    "mensuralNoteheadLongaBlackVoid",
    "mensuralNoteheadLongaVoid",
    "mensuralNoteheadLongaWhite",
    "mensuralNoteheadMaximaBlack",
    "mensuralNoteheadMaximaBlackVoid",
    "mensuralNoteheadMaximaVoid",
    "mensuralNoteheadMaximaWhite",
    "mensuralNoteheadMinimaWhite",
    "mensuralNoteheadSemibrevisBlack",
    "mensuralNoteheadSemibrevisBlackVoid",
    "mensuralNoteheadSemibrevisBlackVoidTurned",
    "mensuralNoteheadSemibrevisVoid",
    "mensuralNoteheadSemiminimaWhite",
    "mensuralObliqueAsc2ndBlack",
    "mensuralObliqueAsc2ndBlackVoid",
    "mensuralObliqueAsc2ndVoid",
    "mensuralObliqueAsc2ndWhite",
    "mensuralObliqueAsc3rdBlack",
    "mensuralObliqueAsc3rdBlackVoid",
    "mensuralObliqueAsc3rdVoid",
    "mensuralObliqueAsc3rdWhite",
    "mensuralObliqueAsc4thBlack",
    "mensuralObliqueAsc4thBlackVoid",
    "mensuralObliqueAsc4thVoid",
    "mensuralObliqueAsc4thWhite",
    "mensuralObliqueAsc5thBlack",
    "mensuralObliqueAsc5thBlackVoid",
    "mensuralObliqueAsc5thVoid",
    "mensuralObliqueAsc5thWhite",
    "mensuralObliqueDesc2ndBlack",
    "mensuralObliqueDesc2ndBlackVoid",
    "mensuralObliqueDesc2ndVoid",
    "mensuralObliqueDesc2ndWhite",
    "mensuralObliqueDesc3rdBlack",
    "mensuralObliqueDesc3rdBlackVoid",
    "mensuralObliqueDesc3rdVoid",
    "mensuralObliqueDesc3rdWhite",
    "mensuralObliqueDesc4thBlack",
    "mensuralObliqueDesc4thBlackVoid",
    "mensuralObliqueDesc4thVoid",
    "mensuralObliqueDesc4thWhite",
    "mensuralObliqueDesc5thBlack",
    "mensuralObliqueDesc5thBlackVoid",
    "mensuralObliqueDesc5thVoid",
    "mensuralObliqueDesc5thWhite",
    "mensuralProlation1",
    "mensuralProlation10",
    "mensuralProlation11",
    "mensuralProlation2",
    "mensuralProlation3",
    "mensuralProlation4",
    "mensuralProlation5",
    "mensuralProlation6",
    "mensuralProlation7",
    "mensuralProlation8",
    "mensuralProlation9",
    "mensuralProlationCombiningDot",
    "mensuralProlationCombiningDotVoid",
    "mensuralProlationCombiningStroke",
    "mensuralProlationCombiningThreeDots",
    "mensuralProlationCombiningThreeDotsTri",
    "mensuralProlationCombiningTwoDots",
    "mensuralProportion1",
    "mensuralProportion2",
    "mensuralProportion3",
    "mensuralProportion4",
    "mensuralProportion5",
    "mensuralProportion6",
    "mensuralProportion7",
    "mensuralProportion8",
    "mensuralProportion9",
    "mensuralProportionMajor",
    "mensuralProportionMinor",
    "mensuralProportionProportioDupla1",
    "mensuralProportionProportioDupla2",
    "mensuralProportionProportioQuadrupla",
    "mensuralProportionProportioTripla",
    "mensuralProportionTempusPerfectum",
    "mensuralRestBrevis",
    "mensuralRestFusa",
    "mensuralRestLongaImperfecta",
    "mensuralRestLongaPerfecta",
    "mensuralRestMaxima",
    "mensuralRestMinima",
    "mensuralRestSemibrevis",
    "mensuralRestSemifusa",
    "mensuralRestSemiminima",
    "mensuralSignumDown",
    "mensuralSignumUp",
    "mensuralTempusImperfectumHoriz",
    "mensuralTempusPerfectumHoriz",
    "mensuralWhiteBrevis",
    "mensuralWhiteFusa",
    "mensuralWhiteLonga",
    "mensuralWhiteMaxima",
    "mensuralWhiteMinima",
    "mensuralWhiteSemibrevis",
    "mensuralWhiteSemiminima",
    "metAugmentationDot",
    "metNote1024thDown",
    "metNote1024thUp",
    "metNote128thDown",
    "metNote128thUp",
    "metNote16thDown",
    "metNote16thUp",
    "metNote256thDown",
    "metNote256thUp",
    "metNote32ndDown",
    "metNote32ndUp",
    "metNote512thDown",
    "metNote512thUp",
    "metNote64thDown",
    "metNote64thUp",
    "metNote8thDown",
    "metNote8thUp",
    "metNoteDoubleWhole",
    "metNoteDoubleWholeSquare",
    "metNoteHalfDown",
    "metNoteHalfUp",
    "metNoteQuarterDown",
    "metNoteQuarterUp",
    "metNoteWhole",
    "metricModulationArrowLeft",
    "metricModulationArrowRight",
    "miscDoNotCopy",
    "miscDoNotPhotocopy",
    "miscEyeglasses",
    "note1024thDown",
    "note1024thUp",
    "note128thDown",
    "note128thUp",
    "note16thDown",
    "note16thUp",
    "note256thDown",
    "note256thUp",
    "note32ndDown",
    "note32ndUp",
    "note512thDown",
    "note512thUp",
    "note64thDown",
    "note64thUp",
    "note8thDown",
    "note8thUp",
    "noteABlack",
    "noteAFlatBlack",
    "noteAFlatHalf",
    "noteAFlatWhole",
    "noteAHalf",
    "noteASharpBlack",
    "noteASharpHalf",
    "noteASharpWhole",
    "noteAWhole",
    "noteBBlack",
    "noteBFlatBlack",
    "noteBFlatHalf",
    "noteBFlatWhole",
    "noteBHalf",
    "noteBSharpBlack",
    "noteBSharpHalf",
    "noteBSharpWhole",
    "noteBWhole",
    "noteCBlack",
    "noteCFlatBlack",
    "noteCFlatHalf",
    "noteCFlatWhole",
    "noteCHalf",
    "noteCSharpBlack",
    "noteCSharpHalf",
    "noteCSharpWhole",
    "noteCWhole",
    "noteDBlack",
    "noteDFlatBlack",
    "noteDFlatHalf",
    "noteDFlatWhole",
    "noteDHalf",
    "noteDSharpBlack",
    "noteDSharpHalf",
    "noteDSharpWhole",
    "noteDWhole",
    "noteDiBlack",
    "noteDiHalf",
    "noteDiWhole",
    "noteDoBlack",
    "noteDoHalf",
    "noteDoWhole",
    "noteDoubleWhole",
    "noteDoubleWholeSquare",
    "noteEBlack",
    "noteEFlatBlack",
    "noteEFlatHalf",
    "noteEFlatWhole",
    "noteEHalf",
    "noteESharpBlack",
    "noteESharpHalf",
    "noteESharpWhole",
    "noteEWhole",
    "noteEmptyBlack",
    "noteEmptyHalf",
    "noteEmptyWhole",
    "noteFBlack",
    "noteFFlatBlack",
    "noteFFlatHalf",
    "noteFFlatWhole",
    "noteFHalf",
    "noteFSharpBlack",
    "noteFSharpHalf",
    "noteFSharpWhole",
    "noteFWhole",
    "noteFaBlack",
    "noteFaHalf",
    "noteFaWhole",
    "noteFiBlack",
    "noteFiHalf",
    "noteFiWhole",
    "noteGBlack",
    "noteGFlatBlack",
    "noteGFlatHalf",
    "noteGFlatWhole",
    "noteGHalf",
    "noteGSharpBlack",
    "noteGSharpHalf",
    "noteGSharpWhole",
    "noteGWhole",
    "noteHBlack",
    "noteHHalf",
    "noteHSharpBlack",
    "noteHSharpHalf",
    "noteHSharpWhole",
    "noteHWhole",
    "noteHalfDown",
    "noteHalfUp",
    "noteLaBlack",
    "noteLaHalf",
    "noteLaWhole",
    "noteLeBlack",
    "noteLeHalf",
    "noteLeWhole",
    "noteLiBlack",
    "noteLiHalf",
    "noteLiWhole",
    "noteMeBlack",
    "noteMeHalf",
    "noteMeWhole",
    "noteMiBlack",
    "noteMiHalf",
    "noteMiWhole",
    "noteQuarterDown",
    "noteQuarterUp",
    "noteRaBlack",
    "noteRaHalf",
    "noteRaWhole",
    "noteReBlack",
    "noteReHalf",
    "noteReWhole",
    "noteRiBlack",
    "noteRiHalf",
    "noteRiWhole",
    "noteSeBlack",
    "noteSeHalf",
    "noteSeWhole",
    "noteShapeArrowheadLeftBlack",
    "noteShapeArrowheadLeftDoubleWhole",
    "noteShapeArrowheadLeftWhite",
    "noteShapeDiamondBlack",
    "noteShapeDiamondDoubleWhole",
    "noteShapeDiamondWhite",
    "noteShapeIsoscelesTriangleBlack",
    "noteShapeIsoscelesTriangleDoubleWhole",
    "noteShapeIsoscelesTriangleWhite",
    "noteShapeKeystoneBlack",
    "noteShapeKeystoneDoubleWhole",
    "noteShapeKeystoneWhite",
    "noteShapeMoonBlack",
    "noteShapeMoonDoubleWhole",
    "noteShapeMoonLeftBlack",
    "noteShapeMoonLeftDoubleWhole",
    "noteShapeMoonLeftWhite",
    "noteShapeMoonWhite",
    "noteShapeQuarterMoonBlack",
    "noteShapeQuarterMoonDoubleWhole",
    "noteShapeQuarterMoonWhite",
    "noteShapeRoundBlack",
    "noteShapeRoundDoubleWhole",
    "noteShapeRoundWhite",
    "noteShapeSquareBlack",
    "noteShapeSquareDoubleWhole",
    "noteShapeSquareWhite",
    "noteShapeTriangleLeftBlack",
    "noteShapeTriangleLeftDoubleWhole",
    "noteShapeTriangleLeftWhite",
    "noteShapeTriangleRightBlack",
    "noteShapeTriangleRightDoubleWhole",
    "noteShapeTriangleRightWhite",
    "noteShapeTriangleRoundBlack",
    "noteShapeTriangleRoundDoubleWhole",
    "noteShapeTriangleRoundLeftBlack",
    "noteShapeTriangleRoundLeftDoubleWhole",
    "noteShapeTriangleRoundLeftWhite",
    "noteShapeTriangleRoundWhite",
    "noteShapeTriangleUpBlack",
    "noteShapeTriangleUpDoubleWhole",
    "noteShapeTriangleUpWhite",
    "noteSiBlack",
    "noteSiHalf",
    "noteSiWhole",
    "noteSoBlack",
    "noteSoHalf",
    "noteSoWhole",
    "noteTeBlack",
    "noteTeHalf",
    "noteTeWhole",
    "noteTiBlack",
    "noteTiHalf",
    "noteTiWhole",
    "noteWhole",
    "noteheadBlack",
    "noteheadCircleSlash",
    "noteheadCircleX",
    "noteheadCircleXDoubleWhole",
    "noteheadCircleXHalf",
    "noteheadCircleXWhole",
    "noteheadCircledBlack",
    "noteheadCircledBlackLarge",
    "noteheadCircledDoubleWhole",
    "noteheadCircledDoubleWholeLarge",
    "noteheadCircledHalf",
    "noteheadCircledHalfLarge",
    "noteheadCircledWhole",
    "noteheadCircledWholeLarge",
    "noteheadCircledXLarge",
    "noteheadClusterDoubleWhole2nd",
    "noteheadClusterDoubleWhole3rd",
    "noteheadClusterDoubleWholeBottom",
    "noteheadClusterDoubleWholeMiddle",
    "noteheadClusterDoubleWholeTop",
    "noteheadClusterHalf2nd",
    "noteheadClusterHalf3rd",
    "noteheadClusterHalfBottom",
    "noteheadClusterHalfMiddle",
    "noteheadClusterHalfTop",
    "noteheadClusterQuarter2nd",
    "noteheadClusterQuarter3rd",
    "noteheadClusterQuarterBottom",
    "noteheadClusterQuarterMiddle",
    "noteheadClusterQuarterTop",
    "noteheadClusterRoundBlack",
    "noteheadClusterRoundWhite",
    "noteheadClusterSquareBlack",
    "noteheadClusterSquareWhite",
    "noteheadClusterWhole2nd",
    "noteheadClusterWhole3rd",
    "noteheadClusterWholeBottom",
    "noteheadClusterWholeMiddle",
    "noteheadClusterWholeTop",
    "noteheadCowellEleventhNoteSeriesHalf",
    "noteheadCowellEleventhNoteSeriesWhole",
    "noteheadCowellEleventhSeriesBlack",
    "noteheadCowellFifteenthNoteSeriesBlack",
    "noteheadCowellFifteenthNoteSeriesHalf",
    "noteheadCowellFifteenthNoteSeriesWhole",
    "noteheadCowellFifthNoteSeriesBlack",
    "noteheadCowellFifthNoteSeriesHalf",
    "noteheadCowellFifthNoteSeriesWhole",
    "noteheadCowellNinthNoteSeriesBlack",
    "noteheadCowellNinthNoteSeriesHalf",
    "noteheadCowellNinthNoteSeriesWhole",
    "noteheadCowellSeventhNoteSeriesBlack",
    "noteheadCowellSeventhNoteSeriesHalf",
    "noteheadCowellSeventhNoteSeriesWhole",
    "noteheadCowellThirdNoteSeriesBlack",
    "noteheadCowellThirdNoteSeriesHalf",
    "noteheadCowellThirdNoteSeriesWhole",
    "noteheadCowellThirteenthNoteSeriesBlack",
    "noteheadCowellThirteenthNoteSeriesHalf",
    "noteheadCowellThirteenthNoteSeriesWhole",
    "noteheadDiamondBlack",
    "noteheadDiamondBlackOld",
    "noteheadDiamondBlackWide",
    "noteheadDiamondClusterBlack2nd",
    "noteheadDiamondClusterBlack3rd",
    "noteheadDiamondClusterBlackBottom",
    "noteheadDiamondClusterBlackMiddle",
    "noteheadDiamondClusterBlackTop",
    "noteheadDiamondClusterWhite2nd",
    "noteheadDiamondClusterWhite3rd",
    "noteheadDiamondClusterWhiteBottom",
    "noteheadDiamondClusterWhiteMiddle",
    "noteheadDiamondClusterWhiteTop",
    "noteheadDiamondDoubleWhole",
    "noteheadDiamondDoubleWholeOld",
    "noteheadDiamondHalf",
    "noteheadDiamondHalfFilled",
    "noteheadDiamondHalfOld",
    "noteheadDiamondHalfWide",
    "noteheadDiamondOpen",
    "noteheadDiamondWhite",
    "noteheadDiamondWhiteWide",
    "noteheadDiamondWhole",
    "noteheadDiamondWholeOld",
    "noteheadDoubleWhole",
    "noteheadDoubleWholeSquare",
    "noteheadDoubleWholeWithX",
    "noteheadHalf",
    "noteheadHalfFilled",
    "noteheadHalfWithX",
    "noteheadHeavyX",
    "noteheadHeavyXHat",
    "noteheadLargeArrowDownBlack",
    "noteheadLargeArrowDownDoubleWhole",
    "noteheadLargeArrowDownHalf",
    "noteheadLargeArrowDownWhole",
    "noteheadLargeArrowUpBlack",
    "noteheadLargeArrowUpDoubleWhole",
    "noteheadLargeArrowUpHalf",
    "noteheadLargeArrowUpWhole",
    "noteheadMoonBlack",
    "noteheadMoonWhite",
    "noteheadNancarrowSine",
    "noteheadNull",
    "noteheadParenthesis",
    "noteheadParenthesisLeft",
    "noteheadParenthesisRight",
    "noteheadPlusBlack",
    "noteheadPlusDoubleWhole",
    "noteheadPlusHalf",
    "noteheadPlusWhole",
    "noteheadRectangularClusterBlackBottom",
    "noteheadRectangularClusterBlackMiddle",
    "noteheadRectangularClusterBlackTop",
    "noteheadRectangularClusterWhiteBottom",
    "noteheadRectangularClusterWhiteMiddle",
    "noteheadRectangularClusterWhiteTop",
    "noteheadRoundBlack",
    "noteheadRoundBlackDoubleSlashed",
    "noteheadRoundBlackLarge",
    "noteheadRoundBlackSlashed",
    "noteheadRoundBlackSlashedLarge",
    "noteheadRoundWhite",
    "noteheadRoundWhiteDoubleSlashed",
    "noteheadRoundWhiteLarge",
    "noteheadRoundWhiteSlashed",
    "noteheadRoundWhiteSlashedLarge",
    "noteheadRoundWhiteWithDot",
    "noteheadRoundWhiteWithDotLarge",
    "noteheadSlashDiamondWhite",
    "noteheadSlashHorizontalEnds",
    "noteheadSlashHorizontalEndsMuted",
    "noteheadSlashVerticalEnds",
    "noteheadSlashVerticalEndsMuted",
    "noteheadSlashVerticalEndsSmall",
    "noteheadSlashWhiteDoubleWhole",
    "noteheadSlashWhiteHalf",
    "noteheadSlashWhiteMuted",
    "noteheadSlashWhiteWhole",
    "noteheadSlashX",
    "noteheadSlashedBlack1",
    "noteheadSlashedBlack2",
    "noteheadSlashedDoubleWhole1",
    "noteheadSlashedDoubleWhole2",
    "noteheadSlashedHalf1",
    "noteheadSlashedHalf2",
    "noteheadSlashedWhole1",
    "noteheadSlashedWhole2",
    "noteheadSquareBlack",
    "noteheadSquareBlackLarge",
    "noteheadSquareBlackWhite",
    "noteheadSquareWhite",
    "noteheadTriangleDownBlack",
    "noteheadTriangleDownDoubleWhole",
    "noteheadTriangleDownHalf",
    "noteheadTriangleDownWhite",
    "noteheadTriangleDownWhole",
    "noteheadTriangleLeftBlack",
    "noteheadTriangleLeftWhite",
    "noteheadTriangleRightBlack",
    "noteheadTriangleRightWhite",
    "noteheadTriangleRoundDownBlack",
    "noteheadTriangleRoundDownWhite",
    "noteheadTriangleUpBlack",
    "noteheadTriangleUpDoubleWhole",
    "noteheadTriangleUpHalf",
    "noteheadTriangleUpRightBlack",
    "noteheadTriangleUpRightWhite",
    "noteheadTriangleUpWhite",
    "noteheadTriangleUpWhole",
    "noteheadVoidWithX",
    "noteheadWhole",
    "noteheadWholeFilled",
    "noteheadWholeWithX",
    "noteheadXBlack",
    "noteheadXDoubleWhole",
    "noteheadXHalf",
    "noteheadXOrnate",
    "noteheadXOrnateEllipse",
    "noteheadXWhole",
    "octaveBaselineA",
    "octaveBaselineB",
    "octaveBaselineM",
    "octaveBaselineV",
    "octaveBassa",
    "octaveLoco",
    "octaveParensLeft",
    "octaveParensRight",
    "octaveSuperscriptA",
    "octaveSuperscriptB",
    "octaveSuperscriptM",
    "octaveSuperscriptV",
    "oneHandedRollStevens",
    "organGerman2Fusae",
    "organGerman2Minimae",
    "organGerman2OctaveUp",
    "organGerman2Semifusae",
    "organGerman2Semiminimae",
    "organGerman3Fusae",
    "organGerman3Minimae",
    "organGerman3Semifusae",
    "organGerman3Semiminimae",
    "organGerman4Fusae",
    "organGerman4Minimae",
    "organGerman4Semifusae",
    "organGerman4Semiminimae",
    "organGerman5Fusae",
    "organGerman5Minimae",
    "organGerman5Semifusae",
    "organGerman5Semiminimae",
    "organGerman6Fusae",
    "organGerman6Minimae",
    "organGerman6Semifusae",
    "organGerman6Semiminimae",
    "organGermanALower",
    "organGermanAUpper",
    "organGermanAugmentationDot",
    "organGermanBLower",
    "organGermanBUpper",
    "organGermanBuxheimerBrevis2",
    "organGermanBuxheimerBrevis3",
    "organGermanBuxheimerMinimaRest",
    "organGermanBuxheimerSemibrevis",
    "organGermanBuxheimerSemibrevisRest",
    "organGermanCLower",
    "organGermanCUpper",
    "organGermanCisLower",
    "organGermanCisUpper",
    "organGermanDLower",
    "organGermanDUpper",
    "organGermanDisLower",
    "organGermanDisUpper",
    "organGermanELower",
    "organGermanEUpper",
    "organGermanFLower",
    "organGermanFUpper",
    "organGermanFisLower",
    "organGermanFisUpper",
    "organGermanFusa",
    "organGermanFusaRest",
    "organGermanGLower",
    "organGermanGUpper",
    "organGermanGisLower",
    "organGermanGisUpper",
    "organGermanHLower",
    "organGermanHUpper",
    "organGermanMinima",
    "organGermanMinimaRest",
    "organGermanOctaveDown",
    "organGermanOctaveUp",
    "organGermanSemibrevis",
    "organGermanSemibrevisRest",
    "organGermanSemifusa",
    "organGermanSemifusaRest",
    "organGermanSemiminima",
    "organGermanSemiminimaRest",
    "organGermanTie",
    "ornamentBottomLeftConcaveStroke",
    "ornamentBottomLeftConcaveStrokeLarge",
    "ornamentBottomLeftConvexStroke",
    "ornamentBottomRightConcaveStroke",
    "ornamentBottomRightConvexStroke",
    "ornamentComma",
    "ornamentDoubleObliqueLinesAfterNote",
    "ornamentDoubleObliqueLinesBeforeNote",
    "ornamentDownCurve",
    "ornamentHaydn",
    "ornamentHighLeftConcaveStroke",
    "ornamentHighLeftConvexStroke",
    "ornamentHighRightConcaveStroke",
    "ornamentHighRightConvexStroke",
    "ornamentHookAfterNote",
    "ornamentHookBeforeNote",
    "ornamentLeftFacingHalfCircle",
    "ornamentLeftFacingHook",
    "ornamentLeftPlus",
    "ornamentLeftShakeT",
    "ornamentLeftVerticalStroke",
    "ornamentLeftVerticalStrokeWithCross",
    "ornamentLowLeftConcaveStroke",
    "ornamentLowLeftConvexStroke",
    "ornamentLowRightConcaveStroke",
    "ornamentLowRightConvexStroke",
    "ornamentMiddleVerticalStroke",
    "ornamentMordent",
    "ornamentObliqueLineAfterNote",
    "ornamentObliqueLineBeforeNote",
    "ornamentObliqueLineHorizAfterNote",
    "ornamentObliqueLineHorizBeforeNote",
    "ornamentOriscus",
    "ornamentPinceCouperin",
    "ornamentPortDeVoixV",
    "ornamentPrecompAppoggTrill",
    "ornamentPrecompAppoggTrillSuffix",
    "ornamentPrecompCadence",
    "ornamentPrecompCadenceUpperPrefix",
    "ornamentPrecompCadenceUpperPrefixTurn",
    "ornamentPrecompCadenceWithTurn",
    "ornamentPrecompDescendingSlide",
    "ornamentPrecompDoubleCadenceLowerPrefix",
    "ornamentPrecompDoubleCadenceUpperPrefix",
    "ornamentPrecompDoubleCadenceUpperPrefixTurn",
    "ornamentPrecompInvertedMordentUpperPrefix",
    "ornamentPrecompMordentRelease",
    "ornamentPrecompMordentUpperPrefix",
    "ornamentPrecompPortDeVoixMordent",
    "ornamentPrecompSlide",
    "ornamentPrecompSlideTrillBach",
    "ornamentPrecompSlideTrillDAnglebert",
    "ornamentPrecompSlideTrillMarpurg",
    "ornamentPrecompSlideTrillMuffat",
    "ornamentPrecompSlideTrillSuffixMuffat",
    "ornamentPrecompTrillLowerSuffix",
    "ornamentPrecompTrillSuffixDandrieu",
    "ornamentPrecompTrillWithMordent",
    "ornamentPrecompTurnTrillBach",
    "ornamentPrecompTurnTrillDAnglebert",
    "ornamentQuilisma",
    "ornamentRightFacingHalfCircle",
    "ornamentRightFacingHook",
    "ornamentRightVerticalStroke",
    "ornamentSchleifer",
    "ornamentShake3",
    "ornamentShakeMuffat1",
    "ornamentShortObliqueLineAfterNote",
    "ornamentShortObliqueLineBeforeNote",
    "ornamentShortTrill",
    "ornamentTopLeftConcaveStroke",
    "ornamentTopLeftConvexStroke",
    "ornamentTopRightConcaveStroke",
    "ornamentTopRightConvexStroke",
    "ornamentTremblement",
    "ornamentTremblementCouperin",
    "ornamentTrill",
    "ornamentTurn",
    "ornamentTurnInverted",
    "ornamentTurnSlash",
    "ornamentTurnUp",
    "ornamentTurnUpS",
    "ornamentUpCurve",
    "ornamentVerticalLine",
    "ornamentZigZagLineNoRightEnd",
    "ornamentZigZagLineWithRightEnd",
    "ottava",
    "ottavaAlta",
    "ottavaBassa",
    "ottavaBassaBa",
    "ottavaBassaVb",
    "pendereckiTremolo",
    "pictAgogo",
    "pictAlmglocken",
    "pictAnvil",
    "pictBambooChimes",
    "pictBambooScraper",
    "pictBassDrum",
    "pictBassDrumOnSide",
    "pictBeaterBow",
    "pictBeaterBox",
    "pictBeaterBrassMalletsDown",
    "pictBeaterBrassMalletsLeft",
    "pictBeaterBrassMalletsRight",
    "pictBeaterBrassMalletsUp",
    "pictBeaterCombiningDashedCircle",
    "pictBeaterCombiningParentheses",
    "pictBeaterDoubleBassDrumDown",
    "pictBeaterDoubleBassDrumUp",
    "pictBeaterFinger",
    "pictBeaterFingernails",
    "pictBeaterFist",
    "pictBeaterGuiroScraper",
    "pictBeaterHammer",
    "pictBeaterHammerMetalDown",
    "pictBeaterHammerMetalUp",
    "pictBeaterHammerPlasticDown",
    "pictBeaterHammerPlasticUp",
    "pictBeaterHammerWoodDown",
    "pictBeaterHammerWoodUp",
    "pictBeaterHand",
    "pictBeaterHardBassDrumDown",
    "pictBeaterHardBassDrumUp",
    "pictBeaterHardGlockenspielDown",
    "pictBeaterHardGlockenspielLeft",
    "pictBeaterHardGlockenspielRight",
    "pictBeaterHardGlockenspielUp",
    "pictBeaterHardTimpaniDown",
    "pictBeaterHardTimpaniLeft",
    "pictBeaterHardTimpaniRight",
    "pictBeaterHardTimpaniUp",
    "pictBeaterHardXylophoneDown",
    "pictBeaterHardXylophoneLeft",
    "pictBeaterHardXylophoneRight",
    "pictBeaterHardXylophoneUp",
    "pictBeaterHardYarnDown",
    "pictBeaterHardYarnLeft",
    "pictBeaterHardYarnRight",
    "pictBeaterHardYarnUp",
    "pictBeaterJazzSticksDown",
    "pictBeaterJazzSticksUp",
    "pictBeaterKnittingNeedle",
    "pictBeaterMallet",
    "pictBeaterMalletDown",
    "pictBeaterMediumBassDrumDown",
    "pictBeaterMediumBassDrumUp",
    "pictBeaterMediumTimpaniDown",
    "pictBeaterMediumTimpaniLeft",
    "pictBeaterMediumTimpaniRight",
    "pictBeaterMediumTimpaniUp",
    "pictBeaterMediumXylophoneDown",
    "pictBeaterMediumXylophoneLeft",
    "pictBeaterMediumXylophoneRight",
    "pictBeaterMediumXylophoneUp",
    "pictBeaterMediumYarnDown",
    "pictBeaterMediumYarnLeft",
    "pictBeaterMediumYarnRight",
    "pictBeaterMediumYarnUp",
    "pictBeaterMetalBassDrumDown",
    "pictBeaterMetalBassDrumUp",
    "pictBeaterMetalDown",
    "pictBeaterMetalHammer",
    "pictBeaterMetalLeft",
    "pictBeaterMetalRight",
    "pictBeaterMetalUp",
    "pictBeaterSnareSticksDown",
    "pictBeaterSnareSticksUp",
    "pictBeaterSoftBassDrumDown",
    "pictBeaterSoftBassDrumUp",
    "pictBeaterSoftGlockenspielDown",
    "pictBeaterSoftGlockenspielLeft",
    "pictBeaterSoftGlockenspielRight",
    "pictBeaterSoftGlockenspielUp",
    "pictBeaterSoftTimpaniDown",
    "pictBeaterSoftTimpaniLeft",
    "pictBeaterSoftTimpaniRight",
    "pictBeaterSoftTimpaniUp",
    "pictBeaterSoftXylophone",
    "pictBeaterSoftXylophoneDown",
    "pictBeaterSoftXylophoneLeft",
    "pictBeaterSoftXylophoneRight",
    "pictBeaterSoftXylophoneUp",
    "pictBeaterSoftYarnDown",
    "pictBeaterSoftYarnLeft",
    "pictBeaterSoftYarnRight",
    "pictBeaterSoftYarnUp",
    "pictBeaterSpoonWoodenMallet",
    "pictBeaterSuperballDown",
    "pictBeaterSuperballLeft",
    "pictBeaterSuperballRight",
    "pictBeaterSuperballUp",
    "pictBeaterTriangleDown",
    "pictBeaterTrianglePlain",
    "pictBeaterTriangleUp",
    "pictBeaterWireBrushesDown",
    "pictBeaterWireBrushesUp",
    "pictBeaterWoodTimpaniDown",
    "pictBeaterWoodTimpaniLeft",
    "pictBeaterWoodTimpaniRight",
    "pictBeaterWoodTimpaniUp",
    "pictBeaterWoodXylophoneDown",
    "pictBeaterWoodXylophoneLeft",
    "pictBeaterWoodXylophoneRight",
    "pictBeaterWoodXylophoneUp",
    "pictBell",
    "pictBellOfCymbal",
    "pictBellPlate",
    "pictBellTree",
    "pictBirdWhistle",
    "pictBoardClapper",
    "pictBongos",
    "pictBrakeDrum",
    "pictCabasa",
    "pictCannon",
    "pictCarHorn",
    "pictCastanets",
    "pictCastanetsWithHandle",
    "pictCelesta",
    "pictCencerro",
    "pictCenter1",
    "pictCenter2",
    "pictCenter3",
    "pictChainRattle",
    "pictChimes",
    "pictChineseCymbal",
    "pictChokeCymbal",
    "pictClaves",
    "pictCoins",
    "pictConga",
    "pictCowBell",
    "pictCrashCymbals",
    "pictCrotales",
    "pictCrushStem",
    "pictCuica",
    "pictCymbalTongs",
    "pictDamp1",
    "pictDamp2",
    "pictDamp3",
    "pictDamp4",
    "pictDeadNoteStem",
    "pictDrumStick",
    "pictDuckCall",
    "pictEdgeOfCymbal",
    "pictEmptyTrap",
    "pictFingerCymbals",
    "pictFlexatone",
    "pictFootballRatchet",
    "pictGlassHarmonica",
    "pictGlassHarp",
    "pictGlassPlateChimes",
    "pictGlassTubeChimes",
    "pictGlsp",
    "pictGlspSmithBrindle",
    "pictGobletDrum",
    "pictGong",
    "pictGongWithButton",
    "pictGuiro",
    "pictGumHardDown",
    "pictGumHardLeft",
    "pictGumHardRight",
    "pictGumHardUp",
    "pictGumMediumDown",
    "pictGumMediumLeft",
    "pictGumMediumRight",
    "pictGumMediumUp",
    "pictGumSoftDown",
    "pictGumSoftLeft",
    "pictGumSoftRight",
    "pictGumSoftUp",
    "pictHalfOpen1",
    "pictHalfOpen2",
    "pictHandbell",
    "pictHiHat",
    "pictHiHatOnStand",
    "pictJawHarp",
    "pictJingleBells",
    "pictKlaxonHorn",
    "pictLeftHandCircle",
    "pictLionsRoar",
    "pictLithophone",
    "pictLogDrum",
    "pictLotusFlute",
    "pictMar",
    "pictMarSmithBrindle",
    "pictMaraca",
    "pictMaracas",
    "pictMegaphone",
    "pictMetalPlateChimes",
    "pictMetalTubeChimes",
    "pictMusicalSaw",
    "pictNormalPosition",
    "pictOnRim",
    "pictOpen",
    "pictOpenRimShot",
    "pictPistolShot",
    "pictPoliceWhistle",
    "pictQuijada",
    "pictRainstick",
    "pictRatchet",
    "pictRecoReco",
    "pictRightHandSquare",
    "pictRim1",
    "pictRim2",
    "pictRim3",
    "pictRimShotOnStem",
    "pictSandpaperBlocks",
    "pictScrapeAroundRim",
    "pictScrapeAroundRimClockwise",
    "pictScrapeCenterToEdge",
    "pictScrapeEdgeToCenter",
    "pictShellBells",
    "pictShellChimes",
    "pictSiren",
    "pictSistrum",
    "pictSizzleCymbal",
    "pictSleighBell",
    "pictSlideBrushOnGong",
    "pictSlideWhistle",
    "pictSlitDrum",
    "pictSnareDrum",
    "pictSnareDrumMilitary",
    "pictSnareDrumSnaresOff",
    "pictSteelDrums",
    "pictStickShot",
    "pictSuperball",
    "pictSuspendedCymbal",
    "pictSwishStem",
    "pictTabla",
    "pictTamTam",
    "pictTamTamWithBeater",
    "pictTambourine",
    "pictTempleBlocks",
    "pictTenorDrum",
    "pictThundersheet",
    "pictTimbales",
    "pictTimpani",
    "pictTomTom",
    "pictTomTomChinese",
    "pictTomTomIndoAmerican",
    "pictTomTomJapanese",
    "pictTriangle",
    "pictTubaphone",
    "pictTubularBells",
    "pictTurnLeftStem",
    "pictTurnRightLeftStem",
    "pictTurnRightStem",
    "pictVib",
    "pictVibMotorOff",
    "pictVibSmithBrindle",
    "pictVibraslap",
    "pictVietnameseHat",
    "pictWhip",
    "pictWindChimesGlass",
    "pictWindMachine",
    "pictWindWhistle",
    "pictWoodBlock",
    "pictWoundHardDown",
    "pictWoundHardLeft",
    "pictWoundHardRight",
    "pictWoundHardUp",
    "pictWoundSoftDown",
    "pictWoundSoftLeft",
    "pictWoundSoftRight",
    "pictWoundSoftUp",
    "pictXyl",
    "pictXylBass",
    "pictXylSmithBrindle",
    "pictXylTenor",
    "pictXylTenorTrough",
    "pictXylTrough",
    "pluckedBuzzPizzicato",
    "pluckedDamp",
    "pluckedDampAll",
    "pluckedDampOnStem",
    "pluckedFingernailFlick",
    "pluckedLeftHandPizzicato",
    "pluckedPlectrum",
    "pluckedSnapPizzicatoAbove",
    "pluckedSnapPizzicatoBelow",
    "pluckedWithFingernails",
    "quindicesima",
    "quindicesimaAlta",
    "quindicesimaBassa",
    "quindicesimaBassaMb",
    "repeat1Bar",
    "repeat2Bars",
    "repeat4Bars",
    "repeatBarLowerDot",
    "repeatBarSlash",
    "repeatBarUpperDot",
    "repeatDot",
    "repeatDots",
    "repeatLeft",
    "repeatRight",
    "repeatRightLeft",
    "rest1024th",
    "rest128th",
    "rest16th",
    "rest256th",
    "rest32nd",
    "rest512th",
    "rest64th",
    "rest8th",
    "restDoubleWhole",
    "restDoubleWholeLegerLine",
    "restHBar",
    "restHBarLeft",
    "restHBarMiddle",
    "restHBarRight",
    "restHalf",
    "restHalfLegerLine",
    "restLonga",
    "restMaxima",
    "restQuarter",
    "restQuarterOld",
    "restQuarterZ",
    "restWhole",
    "restWholeLegerLine",
    "reversedBrace",
    "reversedBracketBottom",
    "reversedBracketTop",
    "rightRepeatSmall",
    "scaleDegree1",
    "scaleDegree2",
    "scaleDegree3",
    "scaleDegree4",
    "scaleDegree5",
    "scaleDegree6",
    "scaleDegree7",
    "scaleDegree8",
    "scaleDegree9",
    "schaefferClef",
    "schaefferFClefToGClef",
    "schaefferGClefToFClef",
    "schaefferPreviousClef",
    "segno",
    "segnoSerpent1",
    "segnoSerpent2",
    "semipitchedPercussionClef1",
    "semipitchedPercussionClef2",
    "smnFlat",
    "smnFlatWhite",
    "smnHistoryDoubleFlat",
    "smnHistoryDoubleSharp",
    "smnHistoryFlat",
    "smnHistorySharp",
    "smnNatural",
    "smnSharp",
    "smnSharpDown",
    "smnSharpWhite",
    "smnSharpWhiteDown",
    "splitBarDivider",
    "staff1Line",
    "staff1LineNarrow",
    "staff1LineWide",
    "staff2Lines",
    "staff2LinesNarrow",
    "staff2LinesWide",
    "staff3Lines",
    "staff3LinesNarrow",
    "staff3LinesWide",
    "staff4Lines",
    "staff4LinesNarrow",
    "staff4LinesWide",
    "staff5Lines",
    "staff5LinesNarrow",
    "staff5LinesWide",
    "staff6Lines",
    "staff6LinesNarrow",
    "staff6LinesWide",
    "staffDivideArrowDown",
    "staffDivideArrowUp",
    "staffDivideArrowUpDown",
    "staffPosLower1",
    "staffPosLower2",
    "staffPosLower3",
    "staffPosLower4",
    "staffPosLower5",
    "staffPosLower6",
    "staffPosLower7",
    "staffPosLower8",
    "staffPosRaise1",
    "staffPosRaise2",
    "staffPosRaise3",
    "staffPosRaise4",
    "staffPosRaise5",
    "staffPosRaise6",
    "staffPosRaise7",
    "staffPosRaise8",
    "stem",
    "stemBowOnBridge",
    "stemBowOnTailpiece",
    "stemBuzzRoll",
    "stemDamp",
    "stemHarpStringNoise",
    "stemMultiphonicsBlack",
    "stemMultiphonicsBlackWhite",
    "stemMultiphonicsWhite",
    "stemPendereckiTremolo",
    "stemRimShot",
    "stemSprechgesang",
    "stemSulPonticello",
    "stemSussurando",
    "stemSwished",
    "stemVibratoPulse",
    "stockhausenTremolo",
    "stringsBowBehindBridge",
    "stringsBowBehindBridgeFourStrings",
    "stringsBowBehindBridgeOneString",
    "stringsBowBehindBridgeThreeStrings",
    "stringsBowBehindBridgeTwoStrings",
    "stringsBowOnBridge",
    "stringsBowOnTailpiece",
    "stringsChangeBowDirection",
    "stringsDownBow",
    "stringsDownBowAwayFromBody",
    "stringsDownBowBeyondBridge",
    "stringsDownBowTowardsBody",
    "stringsDownBowTurned",
    "stringsFouette",
    "stringsHalfHarmonic",
    "stringsHarmonic",
    "stringsJeteAbove",
    "stringsJeteBelow",
    "stringsMuteOff",
    "stringsMuteOn",
    "stringsOverpressureDownBow",
    "stringsOverpressureNoDirection",
    "stringsOverpressurePossibileDownBow",
    "stringsOverpressurePossibileUpBow",
    "stringsOverpressureUpBow",
    "stringsScrapeCircularClockwise",
    "stringsScrapeCircularCounterclockwise",
    "stringsScrapeParallelInward",
    "stringsScrapeParallelOutward",
    "stringsThumbPosition",
    "stringsThumbPositionTurned",
    "stringsTripleChopInward",
    "stringsTripleChopOutward",
    "stringsUpBow",
    "stringsUpBowAwayFromBody",
    "stringsUpBowBeyondBridge",
    "stringsUpBowTowardsBody",
    "stringsUpBowTurned",
    "stringsVibratoPulse",
    "swissRudimentsNoteheadBlackDouble",
    "swissRudimentsNoteheadBlackFlam",
    "swissRudimentsNoteheadHalfDouble",
    "swissRudimentsNoteheadHalfFlam",
    "systemDivider",
    "systemDividerExtraLong",
    "systemDividerLong",
    "textAugmentationDot",
    "textBlackNoteFrac16thLongStem",
    "textBlackNoteFrac16thShortStem",
    "textBlackNoteFrac32ndLongStem",
    "textBlackNoteFrac8thLongStem",
    "textBlackNoteFrac8thShortStem",
    "textBlackNoteLongStem",
    "textBlackNoteShortStem",
    "textCont16thBeamLongStem",
    "textCont16thBeamShortStem",
    "textCont32ndBeamLongStem",
    "textCont8thBeamLongStem",
    "textCont8thBeamShortStem",
    "textHeadlessBlackNoteFrac16thLongStem",
    "textHeadlessBlackNoteFrac16thShortStem",
    "textHeadlessBlackNoteFrac32ndLongStem",
    "textHeadlessBlackNoteFrac8thLongStem",
    "textHeadlessBlackNoteFrac8thShortStem",
    "textHeadlessBlackNoteLongStem",
    "textHeadlessBlackNoteShortStem",
    "textTie",
    "textTuplet3LongStem",
    "textTuplet3ShortStem",
    "textTupletBracketEndLongStem",
    "textTupletBracketEndShortStem",
    "textTupletBracketStartLongStem",
    "textTupletBracketStartShortStem",
    "timeSig0",
    "timeSig0Reversed",
    "timeSig0Turned",
    "timeSig1",
    "timeSig1Reversed",
    "timeSig1Turned",
    "timeSig2",
    "timeSig2Reversed",
    "timeSig2Turned",
    "timeSig3",
    "timeSig3Reversed",
    "timeSig3Turned",
    "timeSig4",
    "timeSig4Reversed",
    "timeSig4Turned",
    "timeSig5",
    "timeSig5Reversed",
    "timeSig5Turned",
    "timeSig6",
    "timeSig6Reversed",
    "timeSig6Turned",
    "timeSig7",
    "timeSig7Reversed",
    "timeSig7Turned",
    "timeSig8",
    "timeSig8Reversed",
    "timeSig8Turned",
    "timeSig9",
    "timeSig9Reversed",
    "timeSig9Turned",
    "timeSigBracketLeft",
    "timeSigBracketLeftSmall",
    "timeSigBracketRight",
    "timeSigBracketRightSmall",
    "timeSigCombDenominator",
    "timeSigCombNumerator",
    "timeSigComma",
    "timeSigCommon",
    "timeSigCommonReversed",
    "timeSigCommonTurned",
    "timeSigCut2",
    "timeSigCut3",
    "timeSigCutCommon",
    "timeSigCutCommonReversed",
    "timeSigCutCommonTurned",
    "timeSigEquals",
    "timeSigFractionHalf",
    "timeSigFractionOneThird",
    "timeSigFractionQuarter",
    "timeSigFractionThreeQuarters",
    "timeSigFractionTwoThirds",
    "timeSigFractionalSlash",
    "timeSigMinus",
    "timeSigMultiply",
    "timeSigOpenPenderecki",
    "timeSigParensLeft",
    "timeSigParensLeftSmall",
    "timeSigParensRight",
    "timeSigParensRightSmall",
    "timeSigPlus",
    "timeSigPlusSmall",
    "timeSigSlash",
    "timeSigX",
    "tremolo1",
    "tremolo2",
    "tremolo3",
    "tremolo4",
    "tremolo5",
    "tremoloDivisiDots2",
    "tremoloDivisiDots3",
    "tremoloDivisiDots4",
    "tremoloDivisiDots6",
    "tremoloFingered1",
    "tremoloFingered2",
    "tremoloFingered3",
    "tremoloFingered4",
    "tremoloFingered5",
    "tripleTongueAbove",
    "tripleTongueBelow",
    "tuplet0",
    "tuplet1",
    "tuplet2",
    "tuplet3",
    "tuplet4",
    "tuplet5",
    "tuplet6",
    "tuplet7",
    "tuplet8",
    "tuplet9",
    "tupletColon",
    "unmeasuredTremolo",
    "unmeasuredTremoloSimple",
    "unpitchedPercussionClef1",
    "unpitchedPercussionClef2",
    "ventiduesima",
    "ventiduesimaAlta",
    "ventiduesimaBassa",
    "ventiduesimaBassaMb",
    "vocalFingerClickStockhausen",
    "vocalHalbGesungen",
    "vocalMouthClosed",
    "vocalMouthOpen",
    "vocalMouthPursed",
    "vocalMouthSlightlyOpen",
    "vocalMouthWideOpen",
    "vocalNasalVoice",
    "vocalSprechgesang",
    "vocalTongueClickStockhausen",
    "vocalTongueFingerClickStockhausen",
    "vocalsSussurando",
    "wiggleArpeggiatoDown",
    "wiggleArpeggiatoDownArrow",
    "wiggleArpeggiatoDownSwash",
    "wiggleArpeggiatoUp",
    "wiggleArpeggiatoUpArrow",
    "wiggleArpeggiatoUpSwash",
    "wiggleCircular",
    "wiggleCircularConstant",
    "wiggleCircularConstantFlipped",
    "wiggleCircularConstantFlippedLarge",
    "wiggleCircularConstantLarge",
    "wiggleCircularEnd",
    "wiggleCircularLarge",
    "wiggleCircularLarger",
    "wiggleCircularLargerStill",
    "wiggleCircularLargest",
    "wiggleCircularSmall",
    "wiggleCircularStart",
    "wiggleGlissando",
    "wiggleGlissandoGroup1",
    "wiggleGlissandoGroup2",
    "wiggleGlissandoGroup3",
    "wiggleRandom1",
    "wiggleRandom2",
    "wiggleRandom3",
    "wiggleRandom4",
    "wiggleSawtooth",
    "wiggleSawtoothNarrow",
    "wiggleSawtoothWide",
    "wiggleSquareWave",
    "wiggleSquareWaveNarrow",
    "wiggleSquareWaveWide",
    "wiggleTrill",
    "wiggleTrillFast",
    "wiggleTrillFaster",
    "wiggleTrillFasterStill",
    "wiggleTrillFastest",
    "wiggleTrillSlow",
    "wiggleTrillSlower",
    "wiggleTrillSlowerStill",
    "wiggleTrillSlowest",
    "wiggleVIbratoLargestSlower",
    "wiggleVIbratoMediumSlower",
    "wiggleVibrato",
    "wiggleVibratoLargeFast",
    "wiggleVibratoLargeFaster",
    "wiggleVibratoLargeFasterStill",
    "wiggleVibratoLargeFastest",
    "wiggleVibratoLargeSlow",
    "wiggleVibratoLargeSlower",
    "wiggleVibratoLargeSlowest",
    "wiggleVibratoLargestFast",
    "wiggleVibratoLargestFaster",
    "wiggleVibratoLargestFasterStill",
    "wiggleVibratoLargestFastest",
    "wiggleVibratoLargestSlow",
    "wiggleVibratoLargestSlowest",
    "wiggleVibratoMediumFast",
    "wiggleVibratoMediumFaster",
    "wiggleVibratoMediumFasterStill",
    "wiggleVibratoMediumFastest",
    "wiggleVibratoMediumSlow",
    "wiggleVibratoMediumSlowest",
    "wiggleVibratoSmallFast",
    "wiggleVibratoSmallFaster",
    "wiggleVibratoSmallFasterStill",
    "wiggleVibratoSmallFastest",
    "wiggleVibratoSmallSlow",
    "wiggleVibratoSmallSlower",
    "wiggleVibratoSmallSlowest",
    "wiggleVibratoSmallestFast",
    "wiggleVibratoSmallestFaster",
    "wiggleVibratoSmallestFasterStill",
    "wiggleVibratoSmallestFastest",
    "wiggleVibratoSmallestSlow",
    "wiggleVibratoSmallestSlower",
    "wiggleVibratoSmallestSlowest",
    "wiggleVibratoStart",
    "wiggleVibratoWide",
    "wiggleWavy",
    "wiggleWavyNarrow",
    "wiggleWavyWide",
    "windClosedHole",
    "windFlatEmbouchure",
    "windHalfClosedHole1",
    "windHalfClosedHole2",
    "windHalfClosedHole3",
    "windLessRelaxedEmbouchure",
    "windLessTightEmbouchure",
    "windMouthpiecePop",
    "windMultiphonicsBlackStem",
    "windMultiphonicsBlackWhiteStem",
    "windMultiphonicsWhiteStem",
    "windOpenHole",
    "windReedPositionIn",
    "windReedPositionNormal",
    "windReedPositionOut",
    "windRelaxedEmbouchure",
    "windRimOnly",
    "windSharpEmbouchure",
    "windStrongAirPressure",
    "windThreeQuartersClosedHole",
    "windTightEmbouchure",
    "windTrillKey",
    "windVeryTightEmbouchure",
    "windWeakAirPressure",
    // SMuFL standard symbol names }}}

    //EXTENSIONS
    //SMuFL stylistic alternates which we need to access directly

    "noteheadDoubleWholeAlt",             // double whole with double side bars
    "4stringTabClefSerif",                // TAB clef in script style
    "6stringTabClefSerif",                // TAB clef in script style
    "cClefFrench",
    "cClefFrench20C",
    "fClefFrench",
    "fClef19thCentury",
    "braceSmall",
    "braceLarge",
    "braceLarger",

    "flag1024thDownStraight",
    "flag1024thUpStraight",
    "flag128thDownStraight",
    "flag128thUpStraight",
    "flag16thDownStraight",
    "flag16thUpStraight",
    "flag256thDownStraight",
    "flag256thUpStraight",
    "flag32ndDownStraight",
    "flag32ndUpStraight",
    "flag512thDownStraight",
    "flag512thUpStraight",
    "flag64thDownStraight",
    "flag64thUpStraight",
    "flag8thDownStraight",
    "flag8thUpStraight",

    "timeSig0Large",
    "timeSig0Small",
    "timeSig0Narrow",
    "timeSig1Large",
    "timeSig1Small",
    "timeSig1Narrow",
    "timeSig2Large",
    "timeSig2Small",
    "timeSig2Narrow",
    "timeSig3Large",
    "timeSig3Small",
    "timeSig3Narrow",
    "timeSig4Large",
    "timeSig4Small",
    "timeSig4Narrow",
    "timeSig5Large",
    "timeSig5Small",
    "timeSig5Narrow",
    "timeSig6Large",
    "timeSig6Small",
    "timeSig6Narrow",
    "timeSig7Large",
    "timeSig7Small",
    "timeSig7Narrow",
    "timeSig8Large",
    "timeSig8Small",
    "timeSig8Narrow",
    "timeSig9Large",
    "timeSig9Small",
    "timeSig9Narrow",
    "timeSigBracketLeftSmallLarge",
    "timeSigBracketLeftSmallNarrow",
    "timeSigBracketRightSmallLarge",
    "timeSigBracketRightSmallNarrow",
    "timeSigBracketLeftLarge",
    "timeSigBracketLeftNarrow",
    "timeSigBracketRightLarge",
    "timeSigBracketRightNarrow",
    "timeSigCommonLarge",
    "timeSigCommonNarrow",
    "timeSigCutCommonLarge",
    "timeSigCutCommonNarrow",
    "timeSigPlusLarge",
    "timeSigPlusNarrow",
    "timeSigPlusSmallLarge",
    "timeSigPlusSmallNarrow",
    "timeSigFractionalSlashLarge",
    "timeSigFractionalSlashNarrow",
    "timeSigEqualsLarge",
    "timeSigEqualsNarrow",
    "timeSigMinusLarge",
    "timeSigMinusNarrow",
    "timeSigMultiplyLarge",
    "timeSigMultiplyNarrow",
    "timeSigParensLeftSmallLarge",
    "timeSigParensLeftSmallNarrow",
    "timeSigParensRightSmallLarge",
    "timeSigParensRightSmallNarrow",
    "timeSigParensLeftLarge",
    "timeSigParensLeftNarrow",
    "timeSigParensRightLarge",
    "timeSigParensRightNarrow",
    "timeSigCommaEqualsLarge",
    "timeSigCommaEqualsNarrow",
    "timeSigFractionQuarterLarge",
    "timeSigFractionQuarterNarrow",
    "timeSigFractionHalfLarge",
    "timeSigFractionHalfNarrow",
    "timeSigFractionThreeQuartersLarge",
    "timeSigFractionThreeQuartersNarrow",
    "timeSigFractionOneThirdLarge",
    "timeSigFractionOneThirdNarrow",
    "timeSigFractionTwoThirdsLarge",
    "timeSigFractionTwoThirdsNarrow",
    "timeSigXLarge",
    "timeSigXNarrow",
    "timeSigOpenPendereckiLarge",
    "timeSigOpenPendereckiNarrow",
    "timeSigCut2Large",
    "timeSigCut2Narrow",
    "timeSigCut3Large",
    "timeSigCut3Narrow",

    //MuseScore-local symbols, precomposed symbols to mimic some Emmentaler glyphs

    "ornamentPrallMordent",         // ornamentPrecompTrillWithMordent ?
    "ornamentUpPrall",              // ornamentPrecompSlideTrillDAnglebert ?
    "ornamentUpMordent",            // ornamentPrecompSlideTrillBach ?
    "ornamentPrallDown",            // ornamentPrecompTrillLowerSuffix ?
    //  "ornamentDownPrall",        // -> SymId::ornamentPrecompMordentUpperPrefix },
    "ornamentDownMordent",          // ornamentPrecompTurnTrillBach ?
    "ornamentPrallUp",              // ornamentPrecompTrillSuffixDandrieu ?
    "ornamentLinePrall",            // ornamentPRecompAppoggTrill ?

    //additional symbols
    "accidentalDoubleFlatParens",
    "accidentalFlatParens",
    "accidentalNaturalParens",
    "accidentalSharpParens",
    "accidentalDoubleSharpParens",

    "noteLongaUp",
    "noteLongaDown",
    "noteLongaSquareUp",
    "noteLongaSquareDown",
    "space"
} };

// should be in sync with s_symNames further up and SymID in symid_p.h

const std::array<muse::TranslatableString, size_t(SymId::lastSym) + 1> SymNames::s_symUserNames { {
    muse::TranslatableString("engraving/sym", "No symbol"),

    // DO NOT edit the SMuFL standard symbol user names (see below) manually!
    // They are generated from tools/fonttools/smufl2sym.{bat,sh} and then
    // copied into this file! Edit tools/fonttools/smufl2sym-in-trans.json
    // to make more strings translatable!

    // SMuFL standard symbol user names {{{
    muse::TranslatableString::untranslatable("4-string tab clef"),
    muse::TranslatableString::untranslatable("6-string tab clef"),
    muse::TranslatableString("engraving/sym", "11 large diesis down, 3° down [46 EDO]"),
    muse::TranslatableString("engraving/sym", "11 large diesis up, (11L), (sharp less 11M), 3° up [46 EDO]"),
    muse::TranslatableString("engraving/sym", "11 medium diesis down, 1°[17 31] 2°46 down, 1/4-tone down"),
    muse::TranslatableString("engraving/sym", "11 medium diesis up, (11M), 1°[17 31] 2°46 up, 1/4-tone up"),
    muse::TranslatableString::untranslatable("11:19 large diesis down"),
    muse::TranslatableString::untranslatable("11:19 large diesis up, (11:19L, apotome less 11:19M)"),
    muse::TranslatableString::untranslatable("11:19 medium diesis down"),
    muse::TranslatableString::untranslatable("11:19 medium diesis up, (11:19M, 11M plus 19s)"),
    muse::TranslatableString::untranslatable("11:49 comma down"),
    muse::TranslatableString::untranslatable("11:49 comma up, (11:49C, 11M less 49C)"),
    muse::TranslatableString::untranslatable("143 comma down"),
    muse::TranslatableString::untranslatable("143 comma up, (143C, 13L less 11M)"),
    muse::TranslatableString::untranslatable("17 comma down"),
    muse::TranslatableString::untranslatable("17 comma up, (17C)"),
    muse::TranslatableString::untranslatable("17 kleisma down"),
    muse::TranslatableString::untranslatable("17 kleisma up, (17k)"),
    muse::TranslatableString::untranslatable("19 comma down"),
    muse::TranslatableString::untranslatable("19 comma up, (19C)"),
    muse::TranslatableString::untranslatable("19 schisma down"),
    muse::TranslatableString::untranslatable("19 schisma up, (19s)"),
    muse::TranslatableString::untranslatable("1 mina down, 1/(5⋅7⋅13)-schismina down, 0.42 cents down"),
    muse::TranslatableString::untranslatable("1 mina up, 1/(5⋅7⋅13)-schismina up, 0.42 cents up"),
    muse::TranslatableString::untranslatable("1 tina down, 7²⋅11⋅19/5-schismina down, 0.17 cents down"),
    muse::TranslatableString::untranslatable("1 tina up, 7²⋅11⋅19/5-schismina up, 0.17 cents up"),
    muse::TranslatableString::untranslatable("23 comma down, 2° down [96 EDO], 1/8-tone down"),
    muse::TranslatableString::untranslatable("23 comma up, (23C), 2° up [96 EDO], 1/8-tone up"),
    muse::TranslatableString::untranslatable("23 small diesis down"),
    muse::TranslatableString::untranslatable("23 small diesis up, (23S)"),
    muse::TranslatableString("engraving/sym", "25 small diesis down, 2° down [53 EDO]"),
    muse::TranslatableString("engraving/sym", "25 small diesis up, (25S, ~5:13S, ~37S, 5C plus 5C), 2° up [53 EDO]"),
    muse::TranslatableString::untranslatable("2 minas down, 65/77-schismina down, 0.83 cents down"),
    muse::TranslatableString::untranslatable("2 minas up, 65/77-schismina up, 0.83 cents up"),
    muse::TranslatableString::untranslatable("2 tinas down, 1/(7³⋅17)-schismina down, 0.30 cents down"),
    muse::TranslatableString::untranslatable("2 tinas up, 1/(7³⋅17)-schismina up, 0.30 cents up"),
    muse::TranslatableString("engraving/sym", "35 large diesis down, 2° down [50 EDO], 5/18-tone down"),
    muse::TranslatableString("engraving/sym", "35 large diesis up, (35L, ~13L, ~125L, sharp less 35M), 2°50 up"),
    muse::TranslatableString("engraving/sym", "35 medium diesis down, 1°[50] 2°[27] down, 2/9-tone down"),
    muse::TranslatableString("engraving/sym", "35 medium diesis up, (35M, ~13M, ~125M, 5C plus 7C), 2/9-tone up"),
    muse::TranslatableString::untranslatable("3 tinas down, 1 mina down, 1/(5⋅7⋅13)-schismina down, 0.42 cents down"),
    muse::TranslatableString::untranslatable("3 tinas up, 1 mina up, 1/(5⋅7⋅13)-schismina up, 0.42 cents up"),
    muse::TranslatableString::untranslatable("49 large diesis down"),
    muse::TranslatableString::untranslatable("49 large diesis up, (49L, ~31L, apotome less 49M)"),
    muse::TranslatableString::untranslatable("49 medium diesis down"),
    muse::TranslatableString::untranslatable("49 medium diesis up, (49M, ~31M, 7C plus 7C)"),
    muse::TranslatableString::untranslatable("49 small diesis down"),
    muse::TranslatableString::untranslatable("49 small diesis up, (49S, ~31S)"),
    muse::TranslatableString::untranslatable("4 tinas down, 5²⋅11²/7-schismina down, 0.57 cents down"),
    muse::TranslatableString::untranslatable("4 tinas up, 5²⋅11²/7-schismina up, 0.57 cents up"),
    muse::TranslatableString::untranslatable("55 comma down, 3° down [96 EDO], 3/16-tone down"),
    muse::TranslatableString::untranslatable("55 comma up, (55C, 11M less 5C), 3°up [96 EDO], 3/16-tone up"),
    muse::TranslatableString("engraving/sym", "5 comma down, 1° down [22 27 29 34 41 46 53 96 EDOs], 1/12-tone down"),
    muse::TranslatableString("engraving/sym", "5 comma up, (5C), 1° up [22 27 29 34 41 46 53 96 EDOs], 1/12-tone up"),
    muse::TranslatableString::untranslatable("5 tinas down, 7⁴/25-schismina down, 0.72 cents down"),
    muse::TranslatableString::untranslatable("5 tinas up, 7⁴/25-schismina up, 0.72 cents up"),
    muse::TranslatableString::untranslatable("5:11 small diesis down"),
    muse::TranslatableString::untranslatable("5:11 small diesis up, (5:11S, ~7:13S, ~11:17S, 5:7k plus 7:11C)"),
    muse::TranslatableString::untranslatable("5:13 large diesis down"),
    muse::TranslatableString::untranslatable("5:13 large diesis up, (5:13L, ~37L, apotome less 5:13M)"),
    muse::TranslatableString::untranslatable("5:13 medium diesis down"),
    muse::TranslatableString::untranslatable("5:13 medium diesis up, (5:13M, ~37M, 5C plus 13C)"),
    muse::TranslatableString::untranslatable("5:19 comma down, 1/20-tone down"),
    muse::TranslatableString::untranslatable("5:19 comma up, (5:19C, 5C plus 19s), 1/20-tone up"),
    muse::TranslatableString::untranslatable("5:23 small diesis down, 2° down [60 EDO], 1/5-tone down"),
    muse::TranslatableString::untranslatable("5:23 small diesis up, (5:23S, 5C plus 23C), 2° up [60 EDO], 1/5-tone up"),
    muse::TranslatableString::untranslatable("5:49 medium diesis down"),
    muse::TranslatableString::untranslatable("5:49 medium diesis up, (5:49M, half apotome)"),
    muse::TranslatableString("engraving/sym", "5:7 kleisma down"),
    muse::TranslatableString("engraving/sym", "5:7 kleisma up, (5:7k, ~11:13k, 7C less 5C)"),
    muse::TranslatableString::untranslatable("6 tinas down, 2 minas down, 65/77-schismina down, 0.83 cents down"),
    muse::TranslatableString::untranslatable("6 tinas up, 2 minas up, 65/77-schismina up, 0.83 cents up"),
    muse::TranslatableString("engraving/sym", "7 comma down, 1° down [43 EDO], 2° down [72 EDO], 1/6-tone down"),
    muse::TranslatableString("engraving/sym", "7 comma up, (7C), 1° up [43 EDO], 2° up [72 EDO], 1/6-tone up"),
    muse::TranslatableString::untranslatable("7 tinas down, 7/(5²⋅17)-schismina down, 1.02 cents down"),
    muse::TranslatableString::untranslatable("7 tinas up, 7/(5²⋅17)-schismina up, 1.02 cents up"),
    muse::TranslatableString::untranslatable("7:11 comma down, 1° down [60 EDO], 1/10-tone down"),
    muse::TranslatableString::untranslatable("7:11 comma up, (7:11C, ~13:17S, ~29S, 11L less 7C), 1° up [60 EDO]"),
    muse::TranslatableString::untranslatable("7:11 kleisma down"),
    muse::TranslatableString::untranslatable("7:11 kleisma up, (7:11k, ~29k)"),
    muse::TranslatableString::untranslatable("7:19 comma down"),
    muse::TranslatableString::untranslatable("7:19 comma up, (7:19C, 7C less 19s)"),
    muse::TranslatableString::untranslatable("8 tinas down, 11⋅17/(5²⋅7)-schismina down, 1.14 cents down"),
    muse::TranslatableString::untranslatable("8 tinas up, 11⋅17/(5²⋅7)-schismina up, 1.14 cents up"),
    muse::TranslatableString::untranslatable("9 tinas down, 1/(7²⋅11)-schismina down, 1.26 cents down"),
    muse::TranslatableString::untranslatable("9 tinas up, 1/(7²⋅11)-schismina up, 1.26 cents up"),
    muse::TranslatableString::untranslatable("Acute, 5 schisma up (5s), 2 cents up"),
    muse::TranslatableString::untranslatable("Double flat, (2 apotomes down)[almost all EDOs], whole-tone down"),
    muse::TranslatableString::untranslatable("Double flat 11:49C-up"),
    muse::TranslatableString::untranslatable("Double flat 143C-up"),
    muse::TranslatableString::untranslatable("Double flat 17C-up"),
    muse::TranslatableString::untranslatable("Double flat 17k-up"),
    muse::TranslatableString::untranslatable("Double flat 19C-up"),
    muse::TranslatableString::untranslatable("Double flat 19s-up"),
    muse::TranslatableString::untranslatable("Double flat 23C-up, 14° down [96 EDO], 7/8-tone down"),
    muse::TranslatableString::untranslatable("Double flat 23S-up"),
    muse::TranslatableString::untranslatable("Double flat 25S-up, 8°down [53 EDO]"),
    muse::TranslatableString::untranslatable("Double flat 49S-up"),
    muse::TranslatableString::untranslatable("Double flat 55C-up, 13° down [96 EDO], 13/16-tone down"),
    muse::TranslatableString::untranslatable("Double flat 5C-up, 5°[22 29] 7°[34 41] 9°53 down, 11/12 tone down"),
    muse::TranslatableString::untranslatable("Double flat 5:11S-up"),
    muse::TranslatableString::untranslatable("Double flat 5:19C-up, 19/20-tone down"),
    muse::TranslatableString::untranslatable("Double flat 5:23S-up, 8° down [60 EDO], 4/5-tone down"),
    muse::TranslatableString::untranslatable("Double flat 5:7k-up"),
    muse::TranslatableString::untranslatable("Double flat 7C-up, 5° down [43 EDO], 10° down [72 EDO], 5/6-tone down"),
    muse::TranslatableString::untranslatable("Double flat 7:11C-up, 9° down [60 EDO], 9/10-tone down"),
    muse::TranslatableString::untranslatable("Double flat 7:11k-up"),
    muse::TranslatableString::untranslatable("Double flat 7:19C-up"),
    muse::TranslatableString::untranslatable("Double sharp, (2 apotomes up)[almost all EDOs], whole-tone up"),
    muse::TranslatableString::untranslatable("Double sharp 11:49C-down"),
    muse::TranslatableString::untranslatable("Double sharp 143C-down"),
    muse::TranslatableString::untranslatable("Double sharp 17C-down"),
    muse::TranslatableString::untranslatable("Double sharp 17k-down"),
    muse::TranslatableString::untranslatable("Double sharp 19C-down"),
    muse::TranslatableString::untranslatable("Double sharp 19s-down"),
    muse::TranslatableString::untranslatable("Double sharp 23C-down, 14°up [96 EDO], 7/8-tone up"),
    muse::TranslatableString::untranslatable("Double sharp 23S-down"),
    muse::TranslatableString::untranslatable("Double sharp 25S-down, 8°up [53 EDO]"),
    muse::TranslatableString::untranslatable("Double sharp 49S-down"),
    muse::TranslatableString::untranslatable("Double sharp 55C-down, 13° up [96 EDO], 13/16-tone up"),
    muse::TranslatableString::untranslatable("Double sharp 5C-down, 5°[22 29] 7°[34 41] 9°53 up, 11/12 tone up"),
    muse::TranslatableString::untranslatable("Double sharp 5:11S-down"),
    muse::TranslatableString::untranslatable("Double sharp 5:19C-down, 19/20-tone up"),
    muse::TranslatableString::untranslatable("Double sharp 5:23S-down, 8° up [60 EDO], 4/5-tone up"),
    muse::TranslatableString::untranslatable("Double sharp 5:7k-down"),
    muse::TranslatableString::untranslatable("Double sharp 7C-down, 5°[43] 10°[72] up, 5/6-tone up"),
    muse::TranslatableString::untranslatable("Double sharp 7:11C-down, 9° up [60 EDO], 9/10-tone up"),
    muse::TranslatableString::untranslatable("Double sharp 7:11k-down"),
    muse::TranslatableString::untranslatable("Double sharp 7:19C-down"),
    muse::TranslatableString("engraving/sym", "Flat, (apotome down)[almost all EDOs], 1/2-tone down"),
    muse::TranslatableString::untranslatable("Flat 11L-down, 8° up [46 EDO]"),
    muse::TranslatableString::untranslatable("Flat 11M-down, 3° down [17 31 EDOs], 7° down [46 EDO], 3/4-tone down"),
    muse::TranslatableString::untranslatable("Flat 11:19L-down"),
    muse::TranslatableString::untranslatable("Flat 11:19M-down"),
    muse::TranslatableString::untranslatable("Flat 11:49C-down"),
    muse::TranslatableString::untranslatable("Flat 11:49C-up"),
    muse::TranslatableString::untranslatable("Flat 143C-down"),
    muse::TranslatableString::untranslatable("Flat 143C-up"),
    muse::TranslatableString::untranslatable("Flat 17C-down"),
    muse::TranslatableString::untranslatable("Flat 17C-up"),
    muse::TranslatableString::untranslatable("Flat 17k-down"),
    muse::TranslatableString::untranslatable("Flat 17k-up"),
    muse::TranslatableString::untranslatable("Flat 19C-down"),
    muse::TranslatableString::untranslatable("Flat 19C-up"),
    muse::TranslatableString::untranslatable("Flat 19s-down"),
    muse::TranslatableString::untranslatable("Flat 19s-up"),
    muse::TranslatableString::untranslatable("Flat 23C-down, 10° down [96 EDO], 5/8-tone down"),
    muse::TranslatableString::untranslatable("Flat 23C-up, 6° down [96 EDO], 3/8-tone down"),
    muse::TranslatableString::untranslatable("Flat 23S-down"),
    muse::TranslatableString::untranslatable("Flat 23S-up"),
    muse::TranslatableString::untranslatable("Flat 25S-down, 7° down [53 EDO]"),
    muse::TranslatableString("engraving/sym", "Flat 25S-up, 3° down [53 EDO]"),
    muse::TranslatableString::untranslatable("Flat 35L-down, 5° down [50 EDO]"),
    muse::TranslatableString::untranslatable("Flat 35M-down, 4° down [50 EDO], 6° down [27 EDO], 13/18-tone down"),
    muse::TranslatableString::untranslatable("Flat 49L-down"),
    muse::TranslatableString::untranslatable("Flat 49M-down"),
    muse::TranslatableString::untranslatable("Flat 49S-down"),
    muse::TranslatableString::untranslatable("Flat 49S-up"),
    muse::TranslatableString::untranslatable("Flat 55C-down, 11° down [96 EDO], 11/16-tone down"),
    muse::TranslatableString::untranslatable("Flat 55C-up, 5° down [96 EDO], 5/16-tone down"),
    muse::TranslatableString::untranslatable("Flat 5C-down, 4°[22 29] 5°[27 34 41] 6°[39 46 53] down, 7/12-tone down"),
    muse::TranslatableString("engraving/sym", "Flat 5C-up, 2°[22 29] 3°[27 34 41] 4°[39 46 53] 5°72 7°[96] down, 5/12-tone down"),
    muse::TranslatableString::untranslatable("Flat 5:11S-down"),
    muse::TranslatableString::untranslatable("Flat 5:11S-up"),
    muse::TranslatableString::untranslatable("Flat 5:13L-down"),
    muse::TranslatableString::untranslatable("Flat 5:13M-down"),
    muse::TranslatableString::untranslatable("Flat 5:19C-down, 11/20-tone down"),
    muse::TranslatableString::untranslatable("Flat 5:19C-up, 9/20-tone down"),
    muse::TranslatableString::untranslatable("Flat 5:23S-down, 7° down [60 EDO], 7/10-tone down"),
    muse::TranslatableString::untranslatable("Flat 5:23S-up, 3° down [60 EDO], 3/10-tone down"),
    muse::TranslatableString::untranslatable("Flat 5:49M-down"),
    muse::TranslatableString::untranslatable("Flat 5:7k-down"),
    muse::TranslatableString("engraving/sym", "Flat 5:7k-up"),
    muse::TranslatableString::untranslatable("Flat 7C-down, 4° down [43 EDO], 8° down [72 EDO], 2/3-tone down"),
    muse::TranslatableString("engraving/sym", "Flat 7C-up, 2° down [43 EDO], 4° down [72 EDO], 1/3-tone down"),
    muse::TranslatableString::untranslatable("Flat 7:11C-down, 6° down [60 EDO], 3/5- tone down"),
    muse::TranslatableString::untranslatable("Flat 7:11C-up, 4° down [60 EDO], 2/5-tone down"),
    muse::TranslatableString::untranslatable("Flat 7:11k-down"),
    muse::TranslatableString::untranslatable("Flat 7:11k-up"),
    muse::TranslatableString::untranslatable("Flat 7:19C-down"),
    muse::TranslatableString::untranslatable("Flat 7:19C-up"),
    muse::TranslatableString::untranslatable("Fractional tina down, 77/(5⋅37)-schismina down, 0.08 cents down"),
    muse::TranslatableString::untranslatable("Fractional tina up, 77/(5⋅37)-schismina up, 0.08 cents up"),
    muse::TranslatableString::untranslatable("Grave, 5 schisma down, 2 cents down"),
    muse::TranslatableString::untranslatable("Shaft down, (natural for use with only diacritics down)"),
    muse::TranslatableString::untranslatable("Shaft up, (natural for use with only diacritics up)"),
    muse::TranslatableString("engraving/sym", "Sharp, (apotome up)[almost all EDOs], 1/2-tone up"),
    muse::TranslatableString::untranslatable("Sharp 11L-up, 8° up [46 EDO]"),
    muse::TranslatableString::untranslatable("Sharp 11M-up, 3° up [17 31 EDOs], 7° up [46 EDO], 3/4-tone up"),
    muse::TranslatableString::untranslatable("Sharp 11:19L-up"),
    muse::TranslatableString::untranslatable("Sharp 11:19M-up"),
    muse::TranslatableString::untranslatable("Sharp 11:49C-down"),
    muse::TranslatableString::untranslatable("Sharp 11:49C-up"),
    muse::TranslatableString::untranslatable("Sharp 143C-down"),
    muse::TranslatableString::untranslatable("Sharp 143C-up"),
    muse::TranslatableString::untranslatable("Sharp 17C-down"),
    muse::TranslatableString::untranslatable("Sharp 17C-up"),
    muse::TranslatableString::untranslatable("Sharp 17k-down"),
    muse::TranslatableString::untranslatable("Sharp 17k-up"),
    muse::TranslatableString::untranslatable("Sharp 19C-down"),
    muse::TranslatableString::untranslatable("Sharp 19C-up"),
    muse::TranslatableString::untranslatable("Sharp 19s-down"),
    muse::TranslatableString::untranslatable("Sharp 19s-up"),
    muse::TranslatableString::untranslatable("Sharp 23C-down, 6° up [96 EDO], 3/8-tone up"),
    muse::TranslatableString::untranslatable("Sharp 23C-up, 10° up [96 EDO], 5/8-tone up"),
    muse::TranslatableString::untranslatable("Sharp 23S-down"),
    muse::TranslatableString::untranslatable("Sharp 23S-up"),
    muse::TranslatableString("engraving/sym", "Sharp 25S-down, 3° up [53 EDO]"),
    muse::TranslatableString::untranslatable("Sharp 25S-up, 7° up [53 EDO]"),
    muse::TranslatableString::untranslatable("Sharp 35L-up, 5° up [50 EDO]"),
    muse::TranslatableString::untranslatable("Sharp 35M-up, 4° up [50 EDO], 6° up [27 EDO], 13/18-tone up"),
    muse::TranslatableString::untranslatable("Sharp 49L-up"),
    muse::TranslatableString::untranslatable("Sharp 49M-up"),
    muse::TranslatableString::untranslatable("Sharp 49S-down"),
    muse::TranslatableString::untranslatable("Sharp 49S-up"),
    muse::TranslatableString::untranslatable("Sharp 55C-down, 5° up [96 EDO], 5/16-tone up"),
    muse::TranslatableString::untranslatable("Sharp 55C-up, 11° up [96 EDO], 11/16-tone up"),
    muse::TranslatableString("engraving/sym", "Sharp 5C-down, 2°[22 29] 3°[27 34 41] 4°[39 46 53] 5°[72] 7°[96] up, 5/12-tone up"),
    muse::TranslatableString::untranslatable("Sharp 5C-up, 4°[22 29] 5°[27 34 41] 6°[39 46 53] up, 7/12-tone up"),
    muse::TranslatableString::untranslatable("Sharp 5:11S-down"),
    muse::TranslatableString::untranslatable("Sharp 5:11S-up"),
    muse::TranslatableString::untranslatable("Sharp 5:13L-up"),
    muse::TranslatableString::untranslatable("Sharp 5:13M-up"),
    muse::TranslatableString::untranslatable("Sharp 5:19C-down, 9/20-tone up"),
    muse::TranslatableString::untranslatable("Sharp 5:19C-up, 11/20-tone up"),
    muse::TranslatableString::untranslatable("Sharp 5:23S-down, 3° up [60 EDO], 3/10-tone up"),
    muse::TranslatableString::untranslatable("Sharp 5:23S-up, 7° up [60 EDO], 7/10-tone up"),
    muse::TranslatableString::untranslatable("Sharp 5:49M-up, (one and a half apotomes)"),
    muse::TranslatableString("engraving/sym", "Sharp 5:7k-down"),
    muse::TranslatableString::untranslatable("Sharp 5:7k-up"),
    muse::TranslatableString("engraving/sym", "Sharp 7C-down, 2° up [43 EDO], 4° up [72 EDO], 1/3-tone up"),
    muse::TranslatableString::untranslatable("Sharp 7C-up, 4° up [43 EDO], 8° up [72 EDO], 2/3-tone up"),
    muse::TranslatableString::untranslatable("Sharp 7:11C-down, 4° up [60 EDO], 2/5-tone up"),
    muse::TranslatableString::untranslatable("Sharp 7:11C-up, 6° up [60 EDO], 3/5- tone up"),
    muse::TranslatableString::untranslatable("Sharp 7:11k-down"),
    muse::TranslatableString::untranslatable("Sharp 7:11k-up"),
    muse::TranslatableString::untranslatable("Sharp 7:19C-down"),
    muse::TranslatableString::untranslatable("Sharp 7:19C-up"),
    muse::TranslatableString::untranslatable("Unused"),
    muse::TranslatableString::untranslatable("Unused"),
    muse::TranslatableString::untranslatable("Unused"),
    muse::TranslatableString::untranslatable("Unused"),
    muse::TranslatableString("engraving/sym", "Combining accordion coupler dot"),
    muse::TranslatableString("engraving/sym", "Combining left hand, 2 ranks, empty"),
    muse::TranslatableString("engraving/sym", "Combining left hand, 3 ranks, empty (square)"),
    muse::TranslatableString("engraving/sym", "Combining right hand, 3 ranks, empty"),
    muse::TranslatableString("engraving/sym", "Combining right hand, 4 ranks, empty"),
    muse::TranslatableString("engraving/sym", "Diatonic accordion clef"),
    muse::TranslatableString("engraving/sym", "Left hand, 2 ranks, 16' stop (round)"),
    muse::TranslatableString("engraving/sym", "Left hand, 2 ranks, 8' stop + 16' stop (round)"),
    muse::TranslatableString("engraving/sym", "Left hand, 2 ranks, 8' stop (round)"),
    muse::TranslatableString("engraving/sym", "Left hand, 2 ranks, full master (round)"),
    muse::TranslatableString("engraving/sym", "Left hand, 2 ranks, master + 16' stop (round)"),
    muse::TranslatableString("engraving/sym", "Left hand, 2 ranks, master (round)"),
    muse::TranslatableString("engraving/sym", "Left hand, 3 ranks, 2' stop + 8' stop (square)"),
    muse::TranslatableString("engraving/sym", "Left hand, 3 ranks, 2' stop (square)"),
    muse::TranslatableString("engraving/sym", "Left hand, 3 ranks, 8' stop (square)"),
    muse::TranslatableString("engraving/sym", "Left hand, 3 ranks, double 8' stop (square)"),
    muse::TranslatableString("engraving/sym", "Left hand, 3 ranks, 2' stop + double 8' stop (tutti) (square)"),
    muse::TranslatableString("engraving/sym", "Pull"),
    muse::TranslatableString("engraving/sym", "Push"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 8' stop + upper tremolo 8' stop + 16' stop (accordion)"),
    muse::TranslatableString("engraving/sym",
                             "Right hand, 3 ranks, lower tremolo 8' stop + 8' stop + upper tremolo 8' stop (authentic musette)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 8' stop + 16' stop (bandoneón)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 16' stop (bassoon)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 8' stop (clarinet)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, lower tremolo 8' stop + 8' stop + upper tremolo 8' stop + 16' stop"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 4' stop + lower tremolo 8' stop + 8' stop + upper tremolo 8' stop"),
    muse::TranslatableString("engraving/sym",
                             "Right hand, 3 ranks, 4' stop + lower tremolo 8' stop + 8' stop + upper tremolo 8' stop + 16' stop"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 4' stop + 8' stop + 16' stop (harmonium)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 4' stop + 8' stop + upper tremolo 8' stop (imitation musette)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, lower tremolo 8' stop"),
    muse::TranslatableString("engraving/sym",
                             "Right hand, 3 ranks, 4' stop + lower tremolo 8' stop + upper tremolo 8' stop + 16' stop (master)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 4' stop + 8' stop (oboe)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 4' stop + 16' stop (organ)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 4' stop (piccolo)"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, lower tremolo 8' stop + upper tremolo 8' stop + 16' stop"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 4' stop + lower tremolo 8' stop + upper tremolo 8' stop"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, lower tremolo 8' stop + upper tremolo 8' stop"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, upper tremolo 8' stop"),
    muse::TranslatableString("engraving/sym", "Right hand, 3 ranks, 8' stop + upper tremolo 8' stop (violin)"),
    muse::TranslatableString("engraving/sym", "Right hand, 4 ranks, alto"),
    muse::TranslatableString("engraving/sym", "Right hand, 4 ranks, bass/alto"),
    muse::TranslatableString("engraving/sym", "Right hand, 4 ranks, master"),
    muse::TranslatableString("engraving/sym", "Right hand, 4 ranks, soft bass"),
    muse::TranslatableString("engraving/sym", "Right hand, 4 ranks, soft tenor"),
    muse::TranslatableString("engraving/sym", "Right hand, 4 ranks, soprano"),
    muse::TranslatableString("engraving/sym", "Right hand, 4 ranks, tenor"),
    muse::TranslatableString("engraving/sym", "Ricochet (2 tones)"),
    muse::TranslatableString("engraving/sym", "Ricochet (3 tones)"),
    muse::TranslatableString("engraving/sym", "Ricochet (4 tones)"),
    muse::TranslatableString("engraving/sym", "Ricochet (5 tones)"),
    muse::TranslatableString("engraving/sym", "Ricochet (6 tones)"),
    muse::TranslatableString("engraving/sym", "Combining ricochet for stem (2 tones)"),
    muse::TranslatableString("engraving/sym", "Combining ricochet for stem (3 tones)"),
    muse::TranslatableString("engraving/sym", "Combining ricochet for stem (4 tones)"),
    muse::TranslatableString("engraving/sym", "Combining ricochet for stem (5 tones)"),
    muse::TranslatableString("engraving/sym", "Combining ricochet for stem (6 tones)"),
    muse::TranslatableString("engraving/sym", "1-comma flat"),
    muse::TranslatableString("engraving/sym", "1-comma sharp"),
    muse::TranslatableString("engraving/sym", "2-comma flat"),
    muse::TranslatableString("engraving/sym", "2-comma sharp"),
    muse::TranslatableString("engraving/sym", "3-comma flat"),
    muse::TranslatableString("engraving/sym", "3-comma sharp"),
    muse::TranslatableString("engraving/sym", "4-comma flat"),
    muse::TranslatableString("engraving/sym", "5-comma sharp"),
    muse::TranslatableString("engraving/sym", "Arrow down (lower by one quarter-tone)"),
    muse::TranslatableString("engraving/sym", "Arrow up (raise by one quarter-tone)"),
    muse::TranslatableString("engraving/sym", "Bakiye (flat)"),
    muse::TranslatableString::untranslatable("Bakiye (sharp)"),
    muse::TranslatableString::untranslatable("Accidental bracket, left"),
    muse::TranslatableString::untranslatable("Accidental bracket, right"),
    muse::TranslatableString("engraving/sym", "Büyük mücenneb (flat)"),
    muse::TranslatableString("engraving/sym", "Büyük mücenneb (sharp)"),
    muse::TranslatableString::untranslatable("Combining close curly brace"),
    muse::TranslatableString("engraving/sym", "Combining lower by one 17-limit schisma"),
    muse::TranslatableString("engraving/sym", "Combining lower by one 19-limit schisma"),
    muse::TranslatableString("engraving/sym", "Combining lower by one 23-limit comma"),
    muse::TranslatableString::untranslatable("Combining lower by one 29-limit comma"),
    muse::TranslatableString("engraving/sym", "Combining lower by one 31-limit schisma"),
    muse::TranslatableString::untranslatable("Combining lower by one 37-limit quartertone"),
    muse::TranslatableString::untranslatable("Combining lower by one 41-limit comma"),
    muse::TranslatableString::untranslatable("Combining lower by one 43-limit comma"),
    muse::TranslatableString::untranslatable("Combining lower by one 47-limit quartertone"),
    muse::TranslatableString("engraving/sym", "Combining lower by one 53-limit comma"),
    muse::TranslatableString::untranslatable("Combining open curly brace"),
    muse::TranslatableString("engraving/sym", "Combining raise by one 17-limit schisma"),
    muse::TranslatableString("engraving/sym", "Combining raise by one 19-limit schisma"),
    muse::TranslatableString("engraving/sym", "Combining raise by one 23-limit comma"),
    muse::TranslatableString::untranslatable("Combining raise by one 29-limit comma"),
    muse::TranslatableString("engraving/sym", "Combining raise by one 31-limit schisma"),
    muse::TranslatableString::untranslatable("Combining raise by one 37-limit quartertone"),
    muse::TranslatableString::untranslatable("Combining raise by one 41-limit comma"),
    muse::TranslatableString::untranslatable("Combining raise by one 43-limit comma"),
    muse::TranslatableString::untranslatable("Combining raise by one 47-limit quartertone"),
    muse::TranslatableString("engraving/sym", "Combining raise by one 53-limit comma"),
    muse::TranslatableString::untranslatable("Syntonic/Didymus comma (80:81) down (Bosanquet)"),
    muse::TranslatableString::untranslatable("Syntonic/Didymus comma (80:81) up (Bosanquet)"),
    muse::TranslatableString("engraving/sym", "Double flat"),
    muse::TranslatableString::untranslatable("Arabic double flat"),
    muse::TranslatableString("engraving/sym", "Double flat equal tempered semitone"),
    muse::TranslatableString("engraving/sym", "Double flat lowered by one syntonic comma"),
    muse::TranslatableString("engraving/sym", "Double flat raised by one syntonic comma"),
    muse::TranslatableString::untranslatable("Reversed double flat"),
    muse::TranslatableString("engraving/sym", "Double flat lowered by three syntonic commas"),
    muse::TranslatableString("engraving/sym", "Double flat raised by three syntonic commas"),
    muse::TranslatableString::untranslatable("Turned double flat"),
    muse::TranslatableString("engraving/sym", "Double flat lowered by two syntonic commas"),
    muse::TranslatableString("engraving/sym", "Double flat raised by two syntonic commas"),
    muse::TranslatableString("engraving/sym", "Double sharp"),
    muse::TranslatableString::untranslatable("Arabic double sharp"),
    muse::TranslatableString("engraving/sym", "Double sharp equal tempered semitone"),
    muse::TranslatableString("engraving/sym", "Double sharp lowered by one syntonic comma"),
    muse::TranslatableString("engraving/sym", "Double sharp raised by one syntonic comma"),
    muse::TranslatableString("engraving/sym", "Double sharp lowered by three syntonic commas"),
    muse::TranslatableString("engraving/sym", "Double sharp raised by three syntonic commas"),
    muse::TranslatableString("engraving/sym", "Double sharp lowered by two syntonic commas"),
    muse::TranslatableString("engraving/sym", "Double sharp raised by two syntonic commas"),
    muse::TranslatableString("engraving/sym", "Enharmonically reinterpret accidental almost equal to"),
    muse::TranslatableString("engraving/sym", "Enharmonically reinterpret accidental equals"),
    muse::TranslatableString("engraving/sym", "Enharmonically reinterpret accidental tilde"),
    muse::TranslatableString::untranslatable("Filled reversed flat and flat"),
    muse::TranslatableString::untranslatable("Filled reversed flat and flat with arrow down"),
    muse::TranslatableString::untranslatable("Filled reversed flat and flat with arrow up"),
    muse::TranslatableString::untranslatable("Filled reversed flat with arrow down"),
    muse::TranslatableString::untranslatable("Filled reversed flat with arrow up"),
    muse::TranslatableString("engraving/sym", "Five-quarter-tones flat"),
    muse::TranslatableString("engraving/sym", "Five-quarter-tones sharp"),
    muse::TranslatableString("engraving/sym", "Flat"),
    muse::TranslatableString::untranslatable("Arabic half-tone flat"),
    muse::TranslatableString("engraving/sym", "Flat equal tempered semitone"),
    muse::TranslatableString::untranslatable("Lowered flat (Stockhausen)"),
    muse::TranslatableString("engraving/sym", "Flat lowered by one syntonic comma"),
    muse::TranslatableString("engraving/sym", "Flat raised by one syntonic comma"),
    muse::TranslatableString::untranslatable("Raised flat (Stockhausen)"),
    muse::TranslatableString::untranslatable("Repeated flat, note on line (Stockhausen)"),
    muse::TranslatableString::untranslatable("Repeated flat, note in space (Stockhausen)"),
    muse::TranslatableString("engraving/sym", "Flat lowered by three syntonic commas"),
    muse::TranslatableString("engraving/sym", "Flat raised by three syntonic commas"),
    muse::TranslatableString::untranslatable("Turned flat"),
    muse::TranslatableString("engraving/sym", "Flat lowered by two syntonic commas"),
    muse::TranslatableString("engraving/sym", "Flat raised by two syntonic commas"),
    muse::TranslatableString::untranslatable("Quarter-tone higher (Alois Hába)"),
    muse::TranslatableString::untranslatable("Three quarter-tones lower (Alois Hába)"),
    muse::TranslatableString::untranslatable("Quarter-tone higher (Alois Hába)"),
    muse::TranslatableString::untranslatable("Quarter-tone lower (Alois Hába)"),
    muse::TranslatableString::untranslatable("Quarter-tone lower (Alois Hába)"),
    muse::TranslatableString::untranslatable("Three quarter-tones higher (Alois Hába)"),
    muse::TranslatableString::untranslatable("Half sharp with arrow down"),
    muse::TranslatableString::untranslatable("Half sharp with arrow up"),
    muse::TranslatableString::untranslatable("Thirteen (raise by 65:64)"),
    muse::TranslatableString::untranslatable("Inverted 13 (lower by 65:64)"),
    muse::TranslatableString::untranslatable("Down arrow (lower by 33:32)"),
    muse::TranslatableString::untranslatable("Inverted seven (raise by 36:35)"),
    muse::TranslatableString::untranslatable("Minus (lower by 81:80)"),
    muse::TranslatableString::untranslatable("Plus (raise by 81:80)"),
    muse::TranslatableString::untranslatable("Seven (lower by 36:35)"),
    muse::TranslatableString::untranslatable("Up arrow (raise by 33:32)"),
    muse::TranslatableString::untranslatable("Koma (flat)"),
    muse::TranslatableString::untranslatable("Koma (sharp)"),
    muse::TranslatableString("engraving/sym", "Koron (quarter tone flat)"),
    muse::TranslatableString::untranslatable("Küçük mücenneb (flat)"),
    muse::TranslatableString("engraving/sym", "Küçük mücenneb (sharp)"),
    muse::TranslatableString::untranslatable("Large double sharp"),
    muse::TranslatableString("engraving/sym", "Lower by one septimal comma"),
    muse::TranslatableString("engraving/sym", "Lower by one tridecimal quartertone"),
    muse::TranslatableString("engraving/sym", "Lower by one undecimal quartertone"),
    muse::TranslatableString("engraving/sym", "Lower by two septimal commas"),
    muse::TranslatableString::untranslatable("Lowered (Stockhausen)"),
    muse::TranslatableString::untranslatable("Narrow reversed flat(quarter-tone flat)"),
    muse::TranslatableString::untranslatable("Narrow reversed flat and flat(three-quarter-tones flat)"),
    muse::TranslatableString("engraving/sym", "Natural"),
    muse::TranslatableString::untranslatable("Arabic natural"),
    muse::TranslatableString("engraving/sym", "Natural equal tempered semitone"),
    muse::TranslatableString("engraving/sym", "Natural flat"),
    muse::TranslatableString::untranslatable("Lowered natural (Stockhausen)"),
    muse::TranslatableString("engraving/sym", "Natural lowered by one syntonic comma"),
    muse::TranslatableString("engraving/sym", "Natural raised by one syntonic comma"),
    muse::TranslatableString::untranslatable("Raised natural (Stockhausen)"),
    muse::TranslatableString::untranslatable("Reversed natural"),
    muse::TranslatableString("engraving/sym", "Natural sharp"),
    muse::TranslatableString("engraving/sym", "Natural lowered by three syntonic commas"),
    muse::TranslatableString("engraving/sym", "Natural raised by three syntonic commas"),
    muse::TranslatableString("engraving/sym", "Natural lowered by two syntonic commas"),
    muse::TranslatableString("engraving/sym", "Natural raised by two syntonic commas"),
    muse::TranslatableString::untranslatable("One and a half sharps with arrow down"),
    muse::TranslatableString::untranslatable("One and a half sharps with arrow up"),
    muse::TranslatableString::untranslatable("One-quarter-tone flat (Ferneyhough)"),
    muse::TranslatableString::untranslatable("One-quarter-tone flat (Stockhausen)"),
    muse::TranslatableString::untranslatable("One-quarter-tone sharp (Ferneyhough)"),
    muse::TranslatableString::untranslatable("One-quarter-tone sharp (Stockhausen)"),
    muse::TranslatableString::untranslatable("One-third-tone flat (Ferneyhough)"),
    muse::TranslatableString::untranslatable("One-third-tone sharp (Ferneyhough)"),
    muse::TranslatableString::untranslatable("Accidental parenthesis, left"),
    muse::TranslatableString::untranslatable("Accidental parenthesis, right"),
    muse::TranslatableString("engraving/sym", "Lower by one equal tempered quarter-tone"),
    muse::TranslatableString("engraving/sym", "Raise by one equal tempered quarter tone"),
    muse::TranslatableString::untranslatable("Quarter-tone flat"),
    muse::TranslatableString::untranslatable("Arabic quarter-tone flat"),
    muse::TranslatableString("engraving/sym", "Quarter-tone flat"),
    muse::TranslatableString::untranslatable("Filled reversed flat (quarter-tone flat)"),
    muse::TranslatableString("engraving/sym", "Quarter-tone flat"),
    muse::TranslatableString::untranslatable("Quarter tone flat (Penderecki)"),
    muse::TranslatableString("engraving/sym", "Reversed flat (quarter-tone flat) (Stein)"),
    muse::TranslatableString::untranslatable("Quarter-tone flat (van Blankenburg)"),
    muse::TranslatableString::untranslatable("Quarter-tone sharp"),
    muse::TranslatableString::untranslatable("Arabic quarter-tone sharp"),
    muse::TranslatableString("engraving/sym", "Quarter-tone sharp"),
    muse::TranslatableString::untranslatable("Quarter tone sharp (Bussotti)"),
    muse::TranslatableString("engraving/sym", "Quarter-tone sharp"),
    muse::TranslatableString("engraving/sym", "Half sharp (quarter-tone sharp) (Stein)"),
    muse::TranslatableString::untranslatable("Quarter tone sharp with wiggly tail"),
    muse::TranslatableString("engraving/sym", "Raise by one septimal comma"),
    muse::TranslatableString("engraving/sym", "Raise by one tridecimal quartertone"),
    muse::TranslatableString("engraving/sym", "Raise by one undecimal quartertone"),
    muse::TranslatableString("engraving/sym", "Raise by two septimal commas"),
    muse::TranslatableString::untranslatable("Raised (Stockhausen)"),
    muse::TranslatableString::untranslatable("Reversed flat and flat with arrow down"),
    muse::TranslatableString::untranslatable("Reversed flat and flat with arrow up"),
    muse::TranslatableString::untranslatable("Reversed flat with arrow down"),
    muse::TranslatableString::untranslatable("Reversed flat with arrow up"),
    muse::TranslatableString("engraving/sym", "Sharp"),
    muse::TranslatableString::untranslatable("Arabic half-tone sharp"),
    muse::TranslatableString("engraving/sym", "Sharp equal tempered semitone"),
    muse::TranslatableString::untranslatable("Lowered sharp (Stockhausen)"),
    muse::TranslatableString("engraving/sym", "Sharp lowered by one syntonic comma"),
    muse::TranslatableString("engraving/sym", "Sharp raised by one syntonic comma"),
    muse::TranslatableString::untranslatable("One or three quarter tones sharp"),
    muse::TranslatableString::untranslatable("Raised sharp (Stockhausen)"),
    muse::TranslatableString::untranslatable("Repeated sharp, note on line (Stockhausen)"),
    muse::TranslatableString::untranslatable("Repeated sharp, note in space (Stockhausen)"),
    muse::TranslatableString::untranslatable("Reversed sharp"),
    muse::TranslatableString("engraving/sym", "Sharp sharp"),
    muse::TranslatableString("engraving/sym", "Sharp lowered by three syntonic commas"),
    muse::TranslatableString("engraving/sym", "Sharp raised by three syntonic commas"),
    muse::TranslatableString("engraving/sym", "Sharp lowered by two syntonic commas"),
    muse::TranslatableString("engraving/sym", "Sharp raised by two syntonic commas"),
    muse::TranslatableString::untranslatable("1/12 tone low"),
    muse::TranslatableString::untranslatable("1/12 tone high"),
    muse::TranslatableString::untranslatable("1/4 tone low"),
    muse::TranslatableString::untranslatable("1/4 tone high"),
    muse::TranslatableString::untranslatable("1/6 tone low"),
    muse::TranslatableString::untranslatable("1/6 tone high"),
    muse::TranslatableString("engraving/sym", "Sori (quarter tone sharp)"),
    muse::TranslatableString::untranslatable("Byzantine-style Bakiye flat (Tavener)"),
    muse::TranslatableString::untranslatable("Byzantine-style Büyük mücenneb sharp (Tavener)"),
    muse::TranslatableString::untranslatable("Arabic three-quarter-tones flat"),
    muse::TranslatableString("engraving/sym", "Three-quarter-tones flat"),
    muse::TranslatableString("engraving/sym", "Three-quarter-tones flat"),
    muse::TranslatableString::untranslatable("Three-quarter-tones flat (Couper)"),
    muse::TranslatableString::untranslatable("Three-quarter-tones flat (Grisey)"),
    muse::TranslatableString::untranslatable("Three-quarter-tones flat (Tartini)"),
    muse::TranslatableString("engraving/sym", "Reversed flat and flat (three-quarter-tones flat) (Zimmermann)"),
    muse::TranslatableString::untranslatable("Arabic three-quarter-tones sharp"),
    muse::TranslatableString("engraving/sym", "Three-quarter-tones sharp"),
    muse::TranslatableString("engraving/sym", "Three-quarter-tones sharp"),
    muse::TranslatableString::untranslatable("Three quarter tones sharp (Bussotti)"),
    muse::TranslatableString("engraving/sym", "One and a half sharps (three-quarter-tones sharp) (Stein)"),
    muse::TranslatableString::untranslatable("Three-quarter-tones sharp (Stockhausen)"),
    muse::TranslatableString("engraving/sym", "Triple flat"),
    muse::TranslatableString("engraving/sym", "Triple sharp"),
    muse::TranslatableString::untranslatable("Two-third-tones flat (Ferneyhough)"),
    muse::TranslatableString::untranslatable("Two-third-tones sharp (Ferneyhough)"),
    muse::TranslatableString::untranslatable("Accidental down"),
    muse::TranslatableString::untranslatable("Accidental less"),
    muse::TranslatableString::untranslatable("Accidental more"),
    muse::TranslatableString::untranslatable("Accidental up"),
    muse::TranslatableString::untranslatable("Wilson minus (5 comma down)"),
    muse::TranslatableString::untranslatable("Wilson plus (5 comma up)"),
    muse::TranslatableString("engraving/sym", "5/6 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "5/6 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "11/12 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "11/12 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/12 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/12 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/6 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/6 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/4 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/4 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/3 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/3 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "5/12 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "5/12 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/2 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "1/2 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "7/12 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "7/12 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "2/3 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "2/3 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "3/4 tone flat (Wyschnegradsky)"),
    muse::TranslatableString("engraving/sym", "3/4 tone sharp (Wyschnegradsky)"),
    muse::TranslatableString::untranslatable("One-third-tone sharp (Xenakis)"),
    muse::TranslatableString::untranslatable("Two-third-tones sharp (Xenakis)"),
    muse::TranslatableString::untranslatable("Choralmelodie (Berg)"),
    muse::TranslatableString::untranslatable("End of stimme"),
    muse::TranslatableString::untranslatable("Hauptrhythmus (Berg)"),
    muse::TranslatableString::untranslatable("Hauptstimme"),
    muse::TranslatableString::untranslatable("Inversion 1"),
    muse::TranslatableString::untranslatable("Nebenstimme"),
    muse::TranslatableString::untranslatable("Start of stimme"),
    muse::TranslatableString::untranslatable("Theme"),
    muse::TranslatableString::untranslatable("Theme 1"),
    muse::TranslatableString::untranslatable("Inversion of theme"),
    muse::TranslatableString::untranslatable("Retrograde of theme"),
    muse::TranslatableString::untranslatable("Retrograde inversion of theme"),
    muse::TranslatableString::untranslatable("Arpeggiato"),
    muse::TranslatableString::untranslatable("Arpeggiato down"),
    muse::TranslatableString::untranslatable("Arpeggiato up"),
    muse::TranslatableString::untranslatable("Black arrow down (S)"),
    muse::TranslatableString::untranslatable("Black arrow down-left (SW)"),
    muse::TranslatableString::untranslatable("Black arrow down-right (SE)"),
    muse::TranslatableString::untranslatable("Black arrow left (W)"),
    muse::TranslatableString::untranslatable("Black arrow right (E)"),
    muse::TranslatableString::untranslatable("Black arrow up (N)"),
    muse::TranslatableString::untranslatable("Black arrow up-left (NW)"),
    muse::TranslatableString::untranslatable("Black arrow up-right (NE)"),
    muse::TranslatableString::untranslatable("Open arrow down (S)"),
    muse::TranslatableString::untranslatable("Open arrow down-left (SW)"),
    muse::TranslatableString::untranslatable("Open arrow down-right (SE)"),
    muse::TranslatableString::untranslatable("Open arrow left (W)"),
    muse::TranslatableString::untranslatable("Open arrow right (E)"),
    muse::TranslatableString::untranslatable("Open arrow up (N)"),
    muse::TranslatableString::untranslatable("Open arrow up-left (NW)"),
    muse::TranslatableString::untranslatable("Open arrow up-right (NE)"),
    muse::TranslatableString::untranslatable("White arrow down (S)"),
    muse::TranslatableString::untranslatable("White arrow down-left (SW)"),
    muse::TranslatableString::untranslatable("White arrow down-right (SE)"),
    muse::TranslatableString::untranslatable("White arrow left (W)"),
    muse::TranslatableString::untranslatable("White arrow right (E)"),
    muse::TranslatableString::untranslatable("White arrow up (N)"),
    muse::TranslatableString::untranslatable("White arrow up-left (NW)"),
    muse::TranslatableString::untranslatable("White arrow up-right (NE)"),
    muse::TranslatableString::untranslatable("Black arrowhead down (S)"),
    muse::TranslatableString::untranslatable("Black arrowhead down-left (SW)"),
    muse::TranslatableString::untranslatable("Black arrowhead down-right (SE)"),
    muse::TranslatableString::untranslatable("Black arrowhead left (W)"),
    muse::TranslatableString::untranslatable("Black arrowhead right (E)"),
    muse::TranslatableString::untranslatable("Black arrowhead up (N)"),
    muse::TranslatableString::untranslatable("Black arrowhead up-left (NW)"),
    muse::TranslatableString::untranslatable("Black arrowhead up-right (NE)"),
    muse::TranslatableString::untranslatable("Open arrowhead down (S)"),
    muse::TranslatableString::untranslatable("Open arrowhead down-left (SW)"),
    muse::TranslatableString::untranslatable("Open arrowhead down-right (SE)"),
    muse::TranslatableString::untranslatable("Open arrowhead left (W)"),
    muse::TranslatableString::untranslatable("Open arrowhead right (E)"),
    muse::TranslatableString::untranslatable("Open arrowhead up (N)"),
    muse::TranslatableString::untranslatable("Open arrowhead up-left (NW)"),
    muse::TranslatableString::untranslatable("Open arrowhead up-right (NE)"),
    muse::TranslatableString::untranslatable("White arrowhead down (S)"),
    muse::TranslatableString::untranslatable("White arrowhead down-left (SW)"),
    muse::TranslatableString::untranslatable("White arrowhead down-right (SE)"),
    muse::TranslatableString::untranslatable("White arrowhead left (W)"),
    muse::TranslatableString::untranslatable("White arrowhead right (E)"),
    muse::TranslatableString::untranslatable("White arrowhead up (N)"),
    muse::TranslatableString::untranslatable("White arrowhead up-left (NW)"),
    muse::TranslatableString::untranslatable("White arrowhead up-right (NE)"),
    muse::TranslatableString("engraving/sym", "Accent above"),
    muse::TranslatableString("engraving/sym", "Accent below"),
    muse::TranslatableString("engraving/sym", "Accent-staccato above"),
    muse::TranslatableString("engraving/sym", "Accent-staccato below"),
    muse::TranslatableString("engraving/sym", "Laissez vibrer (l.v.) above"),
    muse::TranslatableString("engraving/sym", "Laissez vibrer (l.v.) below"),
    muse::TranslatableString("engraving/sym", "Marcato above"),
    muse::TranslatableString("engraving/sym", "Marcato below"),
    muse::TranslatableString("engraving/sym", "Marcato-staccato above"),
    muse::TranslatableString("engraving/sym", "Marcato-staccato below"),
    muse::TranslatableString("engraving/sym", "Marcato-tenuto above"),
    muse::TranslatableString("engraving/sym", "Marcato-tenuto below"),
    muse::TranslatableString("engraving/sym", "Soft accent above"),
    muse::TranslatableString("engraving/sym", "Soft accent below"),
    muse::TranslatableString("engraving/sym", "Soft accent-staccato above"),
    muse::TranslatableString("engraving/sym", "Soft accent-staccato below"),
    muse::TranslatableString("engraving/sym", "Soft accent-tenuto above"),
    muse::TranslatableString("engraving/sym", "Soft accent-tenuto below"),
    muse::TranslatableString("engraving/sym", "Soft accent-tenuto-staccato above"),
    muse::TranslatableString("engraving/sym", "Soft accent-tenuto-staccato below"),
    muse::TranslatableString("engraving/sym", "Staccatissimo above"),
    muse::TranslatableString("engraving/sym", "Staccatissimo below"),
    muse::TranslatableString("engraving/sym", "Staccatissimo stroke above"),
    muse::TranslatableString("engraving/sym", "Staccatissimo stroke below"),
    muse::TranslatableString("engraving/sym", "Staccatissimo wedge above"),
    muse::TranslatableString("engraving/sym", "Staccatissimo wedge below"),
    muse::TranslatableString("engraving/sym", "Staccato above"),
    muse::TranslatableString("engraving/sym", "Staccato below"),
    muse::TranslatableString("engraving/sym", "Stress above"),
    muse::TranslatableString("engraving/sym", "Stress below"),
    muse::TranslatableString("engraving/sym", "Tenuto above"),
    muse::TranslatableString("engraving/sym", "Tenuto-accent above"),
    muse::TranslatableString("engraving/sym", "Tenuto-accent below"),
    muse::TranslatableString("engraving/sym", "Tenuto below"),
    muse::TranslatableString("engraving/sym", "Louré (tenuto-staccato) above"),
    muse::TranslatableString("engraving/sym", "Louré (tenuto-staccato) below"),
    muse::TranslatableString("engraving/sym", "Unstress above"),
    muse::TranslatableString("engraving/sym", "Unstress below"),
    muse::TranslatableString::untranslatable("Augmentation dot"),
    muse::TranslatableString("engraving/sym", "Dashed barline"),
    muse::TranslatableString("engraving/sym", "Dotted barline"),
    muse::TranslatableString("engraving/sym", "Double barline"),
    muse::TranslatableString("engraving/sym", "Final barline"),
    muse::TranslatableString("engraving/sym", "Heavy barline"),
    muse::TranslatableString("engraving/sym", "Heavy double barline"),
    muse::TranslatableString("engraving/sym", "Reverse final barline"),
    muse::TranslatableString("engraving/sym", "Short barline"),
    muse::TranslatableString("engraving/sym", "Single barline"),
    muse::TranslatableString("engraving/sym", "Tick barline"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 1 (widest)"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 10"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 11"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 12"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 13"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 14"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 15 (narrowest)"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 2"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 3"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 4"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 5"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 6"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 7"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 8"),
    muse::TranslatableString::untranslatable("Accel./rit. beam 9"),
    muse::TranslatableString::untranslatable("Accel./rit. beam terminating line"),
    muse::TranslatableString::untranslatable("Brace"),
    muse::TranslatableString::untranslatable("Bracket"),
    muse::TranslatableString::untranslatable("Bracket bottom"),
    muse::TranslatableString::untranslatable("Bracket top"),
    muse::TranslatableString::untranslatable("Bend"),
    muse::TranslatableString::untranslatable("Doit, long"),
    muse::TranslatableString::untranslatable("Doit, medium"),
    muse::TranslatableString::untranslatable("Doit, short"),
    muse::TranslatableString::untranslatable("Lip fall, long"),
    muse::TranslatableString::untranslatable("Lip fall, medium"),
    muse::TranslatableString::untranslatable("Lip fall, short"),
    muse::TranslatableString::untranslatable("Rough fall, long"),
    muse::TranslatableString::untranslatable("Rough fall, medium"),
    muse::TranslatableString::untranslatable("Rough fall, short"),
    muse::TranslatableString::untranslatable("Smooth fall, long"),
    muse::TranslatableString::untranslatable("Smooth fall, medium"),
    muse::TranslatableString::untranslatable("Smooth fall, short"),
    muse::TranslatableString::untranslatable("Flip"),
    muse::TranslatableString::untranslatable("Harmon mute, stem in"),
    muse::TranslatableString::untranslatable("Harmon mute, stem extended, left"),
    muse::TranslatableString::untranslatable("Harmon mute, stem extended, right"),
    muse::TranslatableString::untranslatable("Harmon mute, stem out"),
    muse::TranslatableString::untranslatable("Jazz turn"),
    muse::TranslatableString::untranslatable("Lift, long"),
    muse::TranslatableString::untranslatable("Lift, medium"),
    muse::TranslatableString::untranslatable("Lift, short"),
    muse::TranslatableString::untranslatable("Smooth lift, long"),
    muse::TranslatableString::untranslatable("Smooth lift, medium"),
    muse::TranslatableString::untranslatable("Smooth lift, short"),
    muse::TranslatableString("engraving/sym", "Muted (closed)"),
    muse::TranslatableString::untranslatable("Half-muted (half-closed)"),
    muse::TranslatableString("engraving/sym", "Open"),
    muse::TranslatableString::untranslatable("Plop"),
    muse::TranslatableString::untranslatable("Scoop"),
    muse::TranslatableString::untranslatable("Smear"),
    muse::TranslatableString::untranslatable("Valve trill"),
    muse::TranslatableString("engraving/sym", "Breath mark (comma)"),
    muse::TranslatableString("engraving/sym", "Breath mark (Salzedo)"),
    muse::TranslatableString("engraving/sym", "Breath mark (tick-like)"),
    muse::TranslatableString("engraving/sym", "Breath mark (upbow-like)"),
    muse::TranslatableString::untranslatable("Bridge clef"),
    muse::TranslatableString::untranslatable("Buzz roll"),
    muse::TranslatableString("engraving/sym", "C clef"),
    muse::TranslatableString("engraving/sym", "C clef ottava bassa"),
    muse::TranslatableString::untranslatable("C clef, arrow down"),
    muse::TranslatableString::untranslatable("C clef, arrow up"),
    muse::TranslatableString::untranslatable("C clef change"),
    muse::TranslatableString::untranslatable("Combining C clef"),
    muse::TranslatableString::untranslatable("Reversed C clef"),
    muse::TranslatableString::untranslatable("C clef (19th century)"),
    muse::TranslatableString("engraving/sym", "Caesura"),
    muse::TranslatableString("engraving/sym", "Curved caesura"),
    muse::TranslatableString("engraving/sym", "Short caesura"),
    muse::TranslatableString("engraving/sym", "Single stroke caesura"),
    muse::TranslatableString("engraving/sym", "Thick caesura"),
    muse::TranslatableString::untranslatable("Accentus above"),
    muse::TranslatableString::untranslatable("Accentus below"),
    muse::TranslatableString::untranslatable("Punctum auctum, ascending"),
    muse::TranslatableString::untranslatable("Punctum auctum, descending"),
    muse::TranslatableString::untranslatable("Augmentum (mora)"),
    muse::TranslatableString("engraving/sym", "Caesura"),
    muse::TranslatableString::untranslatable("Plainchant C clef"),
    muse::TranslatableString::untranslatable("Circulus above"),
    muse::TranslatableString::untranslatable("Circulus below"),
    muse::TranslatableString::untranslatable("Connecting line, ascending 2nd"),
    muse::TranslatableString::untranslatable("Connecting line, ascending 3rd"),
    muse::TranslatableString::untranslatable("Connecting line, ascending 4th"),
    muse::TranslatableString::untranslatable("Connecting line, ascending 5th"),
    muse::TranslatableString::untranslatable("Connecting line, ascending 6th"),
    muse::TranslatableString::untranslatable("Plainchant custos, stem down, high position"),
    muse::TranslatableString::untranslatable("Plainchant custos, stem down, highest position"),
    muse::TranslatableString::untranslatable("Plainchant custos, stem down, middle position"),
    muse::TranslatableString::untranslatable("Plainchant custos, stem up, low position"),
    muse::TranslatableString::untranslatable("Plainchant custos, stem up, lowest position"),
    muse::TranslatableString::untranslatable("Plainchant custos, stem up, middle position"),
    muse::TranslatableString::untranslatable("Punctum deminutum, lower"),
    muse::TranslatableString::untranslatable("Punctum deminutum, upper"),
    muse::TranslatableString::untranslatable("Divisio finalis"),
    muse::TranslatableString::untranslatable("Divisio maior"),
    muse::TranslatableString::untranslatable("Divisio maxima"),
    muse::TranslatableString::untranslatable("Divisio minima"),
    muse::TranslatableString::untranslatable("Entry line, ascending 2nd"),
    muse::TranslatableString::untranslatable("Entry line, ascending 3rd"),
    muse::TranslatableString::untranslatable("Entry line, ascending 4th"),
    muse::TranslatableString::untranslatable("Entry line, ascending 5th"),
    muse::TranslatableString::untranslatable("Entry line, ascending 6th"),
    muse::TranslatableString::untranslatable("Episema"),
    muse::TranslatableString::untranslatable("Plainchant F clef"),
    muse::TranslatableString::untranslatable("Ictus above"),
    muse::TranslatableString::untranslatable("Ictus below"),
    muse::TranslatableString::untranslatable("Ligated stroke, descending 2nd"),
    muse::TranslatableString::untranslatable("Ligated stroke, descending 3rd"),
    muse::TranslatableString::untranslatable("Ligated stroke, descending 4th"),
    muse::TranslatableString::untranslatable("Ligated stroke, descending 5th"),
    muse::TranslatableString::untranslatable("Oriscus ascending"),
    muse::TranslatableString::untranslatable("Oriscus descending"),
    muse::TranslatableString::untranslatable("Oriscus liquescens"),
    muse::TranslatableString::untranslatable("Podatus, lower"),
    muse::TranslatableString::untranslatable("Podatus, upper"),
    muse::TranslatableString::untranslatable("Punctum"),
    muse::TranslatableString::untranslatable("Punctum cavum"),
    muse::TranslatableString::untranslatable("Punctum deminutum"),
    muse::TranslatableString::untranslatable("Punctum inclinatum"),
    muse::TranslatableString::untranslatable("Punctum inclinatum auctum"),
    muse::TranslatableString::untranslatable("Punctum inclinatum deminutum"),
    muse::TranslatableString::untranslatable("Punctum linea"),
    muse::TranslatableString::untranslatable("Punctum linea cavum"),
    muse::TranslatableString::untranslatable("Punctum virga"),
    muse::TranslatableString::untranslatable("Punctum virga, reversed"),
    muse::TranslatableString::untranslatable("Quilisma"),
    muse::TranslatableString::untranslatable("Semicirculus above"),
    muse::TranslatableString::untranslatable("Semicirculus below"),
    muse::TranslatableString::untranslatable("Plainchant staff"),
    muse::TranslatableString::untranslatable("Plainchant staff (narrow)"),
    muse::TranslatableString::untranslatable("Plainchant staff (wide)"),
    muse::TranslatableString::untranslatable("Strophicus"),
    muse::TranslatableString::untranslatable("Strophicus auctus"),
    muse::TranslatableString::untranslatable("Strophicus liquescens, 2nd"),
    muse::TranslatableString::untranslatable("Strophicus liquescens, 3rd"),
    muse::TranslatableString::untranslatable("Strophicus liquescens, 4th"),
    muse::TranslatableString::untranslatable("Strophicus liquescens, 5th"),
    muse::TranslatableString::untranslatable("Virgula"),
    muse::TranslatableString::untranslatable("15 for clefs"),
    muse::TranslatableString::untranslatable("8 for clefs"),
    muse::TranslatableString::untranslatable("Combining clef change"),
    muse::TranslatableString("engraving/sym", "Coda"),
    muse::TranslatableString("engraving/sym", "Square coda"),
    muse::TranslatableString::untranslatable("Beat 2, compound time"),
    muse::TranslatableString::untranslatable("Beat 2, simple time"),
    muse::TranslatableString::untranslatable("Beat 3, compound time"),
    muse::TranslatableString::untranslatable("Beat 3, simple time"),
    muse::TranslatableString::untranslatable("Beat 4, compound time"),
    muse::TranslatableString::untranslatable("Beat 4, simple time"),
    muse::TranslatableString::untranslatable("Left-hand beat or cue"),
    muse::TranslatableString::untranslatable("Right-hand beat or cue"),
    muse::TranslatableString::untranslatable("Strong beat or cue"),
    muse::TranslatableString::untranslatable("Unconducted/free passages"),
    muse::TranslatableString::untranslatable("Weak beat or cue"),
    muse::TranslatableString::untranslatable("Begin beam"),
    muse::TranslatableString::untranslatable("Begin phrase"),
    muse::TranslatableString::untranslatable("Begin slur"),
    muse::TranslatableString::untranslatable("Begin tie"),
    muse::TranslatableString::untranslatable("End beam"),
    muse::TranslatableString::untranslatable("End phrase"),
    muse::TranslatableString::untranslatable("End slur"),
    muse::TranslatableString::untranslatable("End tie"),
    muse::TranslatableString::untranslatable("Double flat"),
    muse::TranslatableString::untranslatable("Double sharp"),
    muse::TranslatableString::untranslatable("Flat"),
    muse::TranslatableString::untranslatable("Natural"),
    muse::TranslatableString::untranslatable("Sharp"),
    muse::TranslatableString::untranslatable("Triple flat"),
    muse::TranslatableString::untranslatable("Triple sharp"),
    muse::TranslatableString::untranslatable("Slash for altered bass note"),
    muse::TranslatableString::untranslatable("Augmented"),
    muse::TranslatableString::untranslatable("Double-height left bracket"),
    muse::TranslatableString::untranslatable("Double-height right bracket"),
    muse::TranslatableString::untranslatable("Slash for chord symbols arranged diagonally"),
    muse::TranslatableString::untranslatable("Diminished"),
    muse::TranslatableString::untranslatable("Half-diminished"),
    muse::TranslatableString::untranslatable("Major seventh"),
    muse::TranslatableString::untranslatable("Minor"),
    muse::TranslatableString::untranslatable("Double-height left parenthesis"),
    muse::TranslatableString::untranslatable("Triple-height left parenthesis"),
    muse::TranslatableString::untranslatable("Double-height right parenthesis"),
    muse::TranslatableString::untranslatable("Triple-height right parenthesis"),
    muse::TranslatableString::untranslatable("Curlew (Britten)"),
    muse::TranslatableString::untranslatable("Da capo"),
    muse::TranslatableString::untranslatable("Dal segno"),
    muse::TranslatableString::untranslatable("Daseian excellentes 1"),
    muse::TranslatableString::untranslatable("Daseian excellentes 2"),
    muse::TranslatableString::untranslatable("Daseian excellentes 3"),
    muse::TranslatableString::untranslatable("Daseian excellentes 4"),
    muse::TranslatableString::untranslatable("Daseian finales 1"),
    muse::TranslatableString::untranslatable("Daseian finales 2"),
    muse::TranslatableString::untranslatable("Daseian finales 3"),
    muse::TranslatableString::untranslatable("Daseian finales 4"),
    muse::TranslatableString::untranslatable("Daseian graves 1"),
    muse::TranslatableString::untranslatable("Daseian graves 2"),
    muse::TranslatableString::untranslatable("Daseian graves 3"),
    muse::TranslatableString::untranslatable("Daseian graves 4"),
    muse::TranslatableString::untranslatable("Daseian residua 1"),
    muse::TranslatableString::untranslatable("Daseian residua 2"),
    muse::TranslatableString::untranslatable("Daseian superiores 1"),
    muse::TranslatableString::untranslatable("Daseian superiores 2"),
    muse::TranslatableString::untranslatable("Daseian superiores 3"),
    muse::TranslatableString::untranslatable("Daseian superiores 4"),
    muse::TranslatableString::untranslatable("Double lateral roll (Stevens)"),
    muse::TranslatableString::untranslatable("Double-tongue above"),
    muse::TranslatableString::untranslatable("Double-tongue below"),
    muse::TranslatableString::untranslatable("Colon separator for combined dynamics"),
    muse::TranslatableString::untranslatable("Hyphen separator for combined dynamics"),
    muse::TranslatableString::untranslatable("Slash separator for combined dynamics"),
    muse::TranslatableString::untranslatable("Space separator for combined dynamics"),
    muse::TranslatableString::untranslatable("Crescendo"),
    muse::TranslatableString::untranslatable("Diminuendo"),
    muse::TranslatableString::untranslatable("ff"),
    muse::TranslatableString::untranslatable("fff"),
    muse::TranslatableString::untranslatable("ffff"),
    muse::TranslatableString::untranslatable("fffff"),
    muse::TranslatableString::untranslatable("ffffff"),
    muse::TranslatableString("engraving/sym", "Forte"),
    muse::TranslatableString::untranslatable("Forte-piano"),
    muse::TranslatableString::untranslatable("Forzando"),
    muse::TranslatableString::untranslatable("Left bracket (for hairpins)"),
    muse::TranslatableString::untranslatable("Right bracket (for hairpins)"),
    muse::TranslatableString::untranslatable("Left parenthesis (for hairpins)"),
    muse::TranslatableString::untranslatable("Right parenthesis (for hairpins)"),
    muse::TranslatableString::untranslatable("mf"),
    muse::TranslatableString::untranslatable("mp"),
    muse::TranslatableString::untranslatable("Messa di voce"),
    muse::TranslatableString("engraving/sym", "Mezzo"),
    muse::TranslatableString("engraving/sym", "Niente"),
    muse::TranslatableString::untranslatable("Niente (for hairpins)"),
    muse::TranslatableString::untranslatable("pf"),
    muse::TranslatableString::untranslatable("pp"),
    muse::TranslatableString::untranslatable("ppp"),
    muse::TranslatableString::untranslatable("pppp"),
    muse::TranslatableString::untranslatable("ppppp"),
    muse::TranslatableString::untranslatable("pppppp"),
    muse::TranslatableString("engraving/sym", "Piano"),
    muse::TranslatableString("engraving/sym", "Rinforzando"),
    muse::TranslatableString::untranslatable("Rinforzando 1"),
    muse::TranslatableString::untranslatable("Rinforzando 2"),
    muse::TranslatableString("engraving/sym", "Sforzando"),
    muse::TranslatableString::untranslatable("Sforzando 1"),
    muse::TranslatableString::untranslatable("Sforzando-pianissimo"),
    muse::TranslatableString::untranslatable("Sforzando-piano"),
    muse::TranslatableString::untranslatable("Sforzato"),
    muse::TranslatableString::untranslatable("Sforzatissimo"),
    muse::TranslatableString::untranslatable("Sforzato-piano"),
    muse::TranslatableString("engraving/sym", "Z"),
    muse::TranslatableString::untranslatable("Eight channels (7.1 surround)"),
    muse::TranslatableString::untranslatable("Five channels"),
    muse::TranslatableString::untranslatable("Four channels"),
    muse::TranslatableString::untranslatable("One channel (mono)"),
    muse::TranslatableString::untranslatable("Seven channels"),
    muse::TranslatableString::untranslatable("Six channels (5.1 surround)"),
    muse::TranslatableString::untranslatable("Three channels (frontal)"),
    muse::TranslatableString::untranslatable("Three channels (surround)"),
    muse::TranslatableString::untranslatable("Two channels (stereo)"),
    muse::TranslatableString::untranslatable("Audio in"),
    muse::TranslatableString::untranslatable("Mono audio setup"),
    muse::TranslatableString::untranslatable("Audio out"),
    muse::TranslatableString::untranslatable("Stereo audio setup"),
    muse::TranslatableString::untranslatable("Camera"),
    muse::TranslatableString::untranslatable("Data in"),
    muse::TranslatableString::untranslatable("Data out"),
    muse::TranslatableString::untranslatable("Disc"),
    muse::TranslatableString::untranslatable("Download"),
    muse::TranslatableString::untranslatable("Eject"),
    muse::TranslatableString::untranslatable("Fast-forward"),
    muse::TranslatableString::untranslatable("Headphones"),
    muse::TranslatableString::untranslatable("Headset"),
    muse::TranslatableString::untranslatable("Line in"),
    muse::TranslatableString::untranslatable("Line out"),
    muse::TranslatableString::untranslatable("Loop"),
    muse::TranslatableString::untranslatable("Loudspeaker"),
    muse::TranslatableString::untranslatable("MIDI controller 0%"),
    muse::TranslatableString::untranslatable("MIDI controller 100%"),
    muse::TranslatableString::untranslatable("MIDI controller 20%"),
    muse::TranslatableString::untranslatable("MIDI controller 40%"),
    muse::TranslatableString::untranslatable("MIDI controller 60%"),
    muse::TranslatableString::untranslatable("MIDI controller 80%"),
    muse::TranslatableString::untranslatable("MIDI in"),
    muse::TranslatableString::untranslatable("MIDI out"),
    muse::TranslatableString::untranslatable("Microphone"),
    muse::TranslatableString::untranslatable("Mute microphone"),
    muse::TranslatableString::untranslatable("Unmute microphone"),
    muse::TranslatableString::untranslatable("Mixing console"),
    muse::TranslatableString::untranslatable("Monitor"),
    muse::TranslatableString::untranslatable("Mute"),
    muse::TranslatableString::untranslatable("Pause"),
    muse::TranslatableString::untranslatable("Play"),
    muse::TranslatableString::untranslatable("Power on/off"),
    muse::TranslatableString::untranslatable("Projector"),
    muse::TranslatableString::untranslatable("Replay"),
    muse::TranslatableString::untranslatable("Rewind"),
    muse::TranslatableString::untranslatable("Shuffle"),
    muse::TranslatableString::untranslatable("Skip backwards"),
    muse::TranslatableString::untranslatable("Skip forwards"),
    muse::TranslatableString::untranslatable("Stop"),
    muse::TranslatableString::untranslatable("Tape"),
    muse::TranslatableString::untranslatable("USB connection"),
    muse::TranslatableString::untranslatable("Unmute"),
    muse::TranslatableString::untranslatable("Upload"),
    muse::TranslatableString::untranslatable("Video camera"),
    muse::TranslatableString::untranslatable("Video in"),
    muse::TranslatableString::untranslatable("Video out"),
    muse::TranslatableString::untranslatable("Combining volume fader"),
    muse::TranslatableString::untranslatable("Combining volume fader thumb"),
    muse::TranslatableString::untranslatable("Volume level 0%"),
    muse::TranslatableString::untranslatable("Volume level 100%"),
    muse::TranslatableString::untranslatable("Volume level 20%"),
    muse::TranslatableString::untranslatable("Volume level 40%"),
    muse::TranslatableString::untranslatable("Volume level 60%"),
    muse::TranslatableString::untranslatable("Volume level 80%"),
    muse::TranslatableString("engraving/sym", "F clef"),
    muse::TranslatableString::untranslatable("F clef quindicesima alta"),
    muse::TranslatableString::untranslatable("F clef quindicesima bassa"),
    muse::TranslatableString::untranslatable("F clef ottava alta"),
    muse::TranslatableString::untranslatable("F clef ottava bassa"),
    muse::TranslatableString::untranslatable("F clef, arrow down"),
    muse::TranslatableString::untranslatable("F clef, arrow up"),
    muse::TranslatableString::untranslatable("F clef change"),
    muse::TranslatableString::untranslatable("Reversed F clef"),
    muse::TranslatableString::untranslatable("Turned F clef"),
    muse::TranslatableString("engraving/sym", "Fermata above"),
    muse::TranslatableString("engraving/sym", "Fermata below"),
    muse::TranslatableString("engraving/sym", "Long fermata above"),
    muse::TranslatableString("engraving/sym", "Long fermata below"),
    muse::TranslatableString("engraving/sym", "Long fermata (Henze) above"),
    muse::TranslatableString("engraving/sym", "Long fermata (Henze) below"),
    muse::TranslatableString("engraving/sym", "Short fermata above"),
    muse::TranslatableString("engraving/sym", "Short fermata below"),
    muse::TranslatableString("engraving/sym", "Short fermata (Henze) above"),
    muse::TranslatableString("engraving/sym", "Short fermata (Henze) below"),
    muse::TranslatableString("engraving/sym", "Very long fermata above"),
    muse::TranslatableString("engraving/sym", "Very long fermata below"),
    muse::TranslatableString("engraving/sym", "Very short fermata above"),
    muse::TranslatableString("engraving/sym", "Very short fermata below"),
    muse::TranslatableString::untranslatable("Figured bass 0"),
    muse::TranslatableString::untranslatable("Figured bass 1"),
    muse::TranslatableString::untranslatable("Figured bass 2"),
    muse::TranslatableString::untranslatable("Figured bass 2 raised by half-step"),
    muse::TranslatableString::untranslatable("Figured bass 3"),
    muse::TranslatableString::untranslatable("Figured bass 4"),
    muse::TranslatableString::untranslatable("Figured bass 4 raised by half-step"),
    muse::TranslatableString::untranslatable("Figured bass 5"),
    muse::TranslatableString::untranslatable("Figured bass 5 raised by half-step"),
    muse::TranslatableString::untranslatable("Figured bass 5 raised by half-step 2"),
    muse::TranslatableString::untranslatable("Figured bass diminished 5"),
    muse::TranslatableString::untranslatable("Figured bass 6"),
    muse::TranslatableString::untranslatable("Figured bass 6 raised by half-step"),
    muse::TranslatableString::untranslatable("Figured bass 6 raised by half-step 2"),
    muse::TranslatableString::untranslatable("Figured bass 7"),
    muse::TranslatableString::untranslatable("Figured bass 7 diminished"),
    muse::TranslatableString::untranslatable("Figured bass 7 raised by half-step"),
    muse::TranslatableString::untranslatable("Figured bass 7 lowered by a half-step"),
    muse::TranslatableString::untranslatable("Figured bass 8"),
    muse::TranslatableString::untranslatable("Figured bass 9"),
    muse::TranslatableString::untranslatable("Figured bass 9 raised by half-step"),
    muse::TranslatableString::untranslatable("Figured bass ["),
    muse::TranslatableString::untranslatable("Figured bass ]"),
    muse::TranslatableString::untranslatable("Combining lower"),
    muse::TranslatableString::untranslatable("Combining raise"),
    muse::TranslatableString::untranslatable("Figured bass double flat"),
    muse::TranslatableString::untranslatable("Figured bass double sharp"),
    muse::TranslatableString::untranslatable("Figured bass flat"),
    muse::TranslatableString::untranslatable("Figured bass natural"),
    muse::TranslatableString::untranslatable("Figured bass ("),
    muse::TranslatableString::untranslatable("Figured bass )"),
    muse::TranslatableString::untranslatable("Figured bass +"),
    muse::TranslatableString::untranslatable("Figured bass sharp"),
    muse::TranslatableString::untranslatable("Figured bass triple flat"),
    muse::TranslatableString::untranslatable("Figured bass triple sharp"),
    muse::TranslatableString::untranslatable("Fingering 0 (open string)"),
    muse::TranslatableString::untranslatable("Fingering 0 italic (open string)"),
    muse::TranslatableString::untranslatable("Fingering 1 (thumb)"),
    muse::TranslatableString::untranslatable("Fingering 1 italic (thumb)"),
    muse::TranslatableString::untranslatable("Fingering 2 (index finger)"),
    muse::TranslatableString::untranslatable("Fingering 2 italic (index finger)"),
    muse::TranslatableString::untranslatable("Fingering 3 (middle finger)"),
    muse::TranslatableString::untranslatable("Fingering 3 italic (middle finger)"),
    muse::TranslatableString::untranslatable("Fingering 4 (ring finger)"),
    muse::TranslatableString::untranslatable("Fingering 4 italic (ring finger)"),
    muse::TranslatableString::untranslatable("Fingering 5 (little finger)"),
    muse::TranslatableString::untranslatable("Fingering 5 italic (little finger)"),
    muse::TranslatableString::untranslatable("Fingering 6"),
    muse::TranslatableString::untranslatable("Fingering 6 italic"),
    muse::TranslatableString::untranslatable("Fingering 7"),
    muse::TranslatableString::untranslatable("Fingering 7 italic"),
    muse::TranslatableString::untranslatable("Fingering 8"),
    muse::TranslatableString::untranslatable("Fingering 8 italic"),
    muse::TranslatableString::untranslatable("Fingering 9"),
    muse::TranslatableString::untranslatable("Fingering 9 italic"),
    muse::TranslatableString::untranslatable("Fingering a (anular; right-hand ring finger for guitar)"),
    muse::TranslatableString::untranslatable("Fingering c (right-hand little finger for guitar)"),
    muse::TranslatableString::untranslatable("Fingering e (right-hand little finger for guitar)"),
    muse::TranslatableString::untranslatable("Fingering i (indicio; right-hand index finger for guitar)"),
    muse::TranslatableString::untranslatable("Fingering left bracket"),
    muse::TranslatableString::untranslatable("Fingering left bracket italic"),
    muse::TranslatableString::untranslatable("Fingering left parenthesis"),
    muse::TranslatableString::untranslatable("Fingering left parenthesis italic"),
    muse::TranslatableString::untranslatable("Fingering m (medio; right-hand middle finger for guitar)"),
    muse::TranslatableString::untranslatable("Multiple notes played by thumb or single finger"),
    muse::TranslatableString::untranslatable("Fingering o (right-hand little finger for guitar)"),
    muse::TranslatableString::untranslatable("Fingering p (pulgar; right-hand thumb for guitar)"),
    muse::TranslatableString::untranslatable("Fingering q (right-hand little finger for guitar)"),
    muse::TranslatableString::untranslatable("Fingering right bracket"),
    muse::TranslatableString::untranslatable("Fingering right bracket italic"),
    muse::TranslatableString::untranslatable("Fingering right parenthesis"),
    muse::TranslatableString::untranslatable("Fingering right parenthesis italic"),
    muse::TranslatableString::untranslatable("Fingering s (right-hand little finger for guitar)"),
    muse::TranslatableString::untranslatable("Fingering middle dot separator"),
    muse::TranslatableString::untranslatable("Fingering white middle dot separator"),
    muse::TranslatableString::untranslatable("Fingering forward slash separator"),
    muse::TranslatableString::untranslatable("Finger substitution above"),
    muse::TranslatableString::untranslatable("Finger substitution below"),
    muse::TranslatableString::untranslatable("Finger substitution dash"),
    muse::TranslatableString::untranslatable("Fingering t (right-hand thumb for guitar)"),
    muse::TranslatableString::untranslatable("Fingering T (left-hand thumb for guitar)"),
    muse::TranslatableString::untranslatable("Fingering x (right-hand little finger for guitar)"),
    muse::TranslatableString("engraving/sym", "Combining flag 8 (1024th) below"),
    muse::TranslatableString("engraving/sym", "Combining flag 8 (1024th) above"),
    muse::TranslatableString("engraving/sym", "Combining flag 5 (128th) below"),
    muse::TranslatableString("engraving/sym", "Combining flag 5 (128th) above"),
    muse::TranslatableString("engraving/sym", "Combining flag 2 (16th) below"),
    muse::TranslatableString("engraving/sym", "Combining flag 2 (16th) above"),
    muse::TranslatableString("engraving/sym", "Combining flag 6 (256th) below"),
    muse::TranslatableString("engraving/sym", "Combining flag 6 (256th) above"),
    muse::TranslatableString("engraving/sym", "Combining flag 3 (32nd) below"),
    muse::TranslatableString("engraving/sym", "Combining flag 3 (32nd) above"),
    muse::TranslatableString("engraving/sym", "Combining flag 7 (512th) below"),
    muse::TranslatableString("engraving/sym", "Combining flag 7 (512th) above"),
    muse::TranslatableString("engraving/sym", "Combining flag 4 (64th) below"),
    muse::TranslatableString("engraving/sym", "Combining flag 4 (64th) above"),
    muse::TranslatableString("engraving/sym", "Combining flag 1 (8th) below"),
    muse::TranslatableString("engraving/sym", "Combining flag 1 (8th) above"),
    muse::TranslatableString::untranslatable("Internal combining flag below"),
    muse::TranslatableString::untranslatable("Internal combining flag above"),
    muse::TranslatableString::untranslatable("3-string fretboard"),
    muse::TranslatableString::untranslatable("3-string fretboard at nut"),
    muse::TranslatableString::untranslatable("4-string fretboard"),
    muse::TranslatableString::untranslatable("4-string fretboard at nut"),
    muse::TranslatableString::untranslatable("5-string fretboard"),
    muse::TranslatableString::untranslatable("5-string fretboard at nut"),
    muse::TranslatableString::untranslatable("6-string fretboard"),
    muse::TranslatableString::untranslatable("6-string fretboard at nut"),
    muse::TranslatableString::untranslatable("Fingered fret (filled circle)"),
    muse::TranslatableString::untranslatable("Open string (O)"),
    muse::TranslatableString::untranslatable("String not played (X)"),
    muse::TranslatableString::untranslatable("Function theory angle bracket left"),
    muse::TranslatableString::untranslatable("Function theory angle bracket right"),
    muse::TranslatableString::untranslatable("Function theory bracket left"),
    muse::TranslatableString::untranslatable("Function theory bracket right"),
    muse::TranslatableString::untranslatable("Function theory dominant of dominant"),
    muse::TranslatableString::untranslatable("Function theory minor dominant"),
    muse::TranslatableString::untranslatable("Function theory major dominant"),
    muse::TranslatableString::untranslatable("Function theory 8"),
    muse::TranslatableString::untranslatable("Function theory F"),
    muse::TranslatableString::untranslatable("Function theory 5"),
    muse::TranslatableString::untranslatable("Function theory 4"),
    muse::TranslatableString::untranslatable("Function theory g"),
    muse::TranslatableString::untranslatable("Function theory G"),
    muse::TranslatableString::untranslatable("Function theory greater than"),
    muse::TranslatableString::untranslatable("Function theory i"),
    muse::TranslatableString::untranslatable("Function theory I"),
    muse::TranslatableString::untranslatable("Function theory k"),
    muse::TranslatableString::untranslatable("Function theory K"),
    muse::TranslatableString::untranslatable("Function theory l"),
    muse::TranslatableString::untranslatable("Function theory L"),
    muse::TranslatableString::untranslatable("Function theory less than"),
    muse::TranslatableString::untranslatable("Function theory m"),
    muse::TranslatableString::untranslatable("Function theory M"),
    muse::TranslatableString::untranslatable("Function theory minus"),
    muse::TranslatableString::untranslatable("Function theory n"),
    muse::TranslatableString::untranslatable("Function theory N"),
    muse::TranslatableString::untranslatable("Function theory superscript N"),
    muse::TranslatableString::untranslatable("Function theory 9"),
    muse::TranslatableString::untranslatable("Function theory 1"),
    muse::TranslatableString::untranslatable("Function theory p"),
    muse::TranslatableString::untranslatable("Function theory P"),
    muse::TranslatableString::untranslatable("Function theory parenthesis left"),
    muse::TranslatableString::untranslatable("Function theory parenthesis right"),
    muse::TranslatableString::untranslatable("Function theory prefix plus"),
    muse::TranslatableString::untranslatable("Function theory r"),
    muse::TranslatableString::untranslatable("Function theory repetition 1"),
    muse::TranslatableString::untranslatable("Function theory repetition 2"),
    muse::TranslatableString::untranslatable("Function theory prefix ring"),
    muse::TranslatableString::untranslatable("Function theory minor subdominant"),
    muse::TranslatableString::untranslatable("Function theory minor subdominant of subdominant"),
    muse::TranslatableString::untranslatable("Function theory major subdominant of subdominant"),
    muse::TranslatableString::untranslatable("Function theory major subdominant"),
    muse::TranslatableString::untranslatable("Function theory 7"),
    muse::TranslatableString::untranslatable("Function theory 6"),
    muse::TranslatableString::untranslatable("Function theory double dominant seventh"),
    muse::TranslatableString::untranslatable("Function theory minor tonic"),
    muse::TranslatableString::untranslatable("Function theory tonic"),
    muse::TranslatableString::untranslatable("Function theory 3"),
    muse::TranslatableString::untranslatable("Function theory 2"),
    muse::TranslatableString::untranslatable("Function theory v"),
    muse::TranslatableString::untranslatable("Function theory V"),
    muse::TranslatableString::untranslatable("Function theory 0"),
    muse::TranslatableString("engraving/sym", "G clef"),
    muse::TranslatableString::untranslatable("G clef quindicesima alta"),
    muse::TranslatableString::untranslatable("G clef quindicesima bassa"),
    muse::TranslatableString::untranslatable("G clef ottava alta"),
    muse::TranslatableString::untranslatable("G clef ottava bassa"),
    muse::TranslatableString::untranslatable("G clef ottava bassa with C clef"),
    muse::TranslatableString::untranslatable("G clef ottava bassa (old style)"),
    muse::TranslatableString::untranslatable("G clef, optionally ottava bassa"),
    muse::TranslatableString::untranslatable("G clef, arrow down"),
    muse::TranslatableString::untranslatable("G clef, arrow up"),
    muse::TranslatableString::untranslatable("G clef change"),
    muse::TranslatableString::untranslatable("Combining G clef, number above"),
    muse::TranslatableString::untranslatable("Combining G clef, number below"),
    muse::TranslatableString::untranslatable("Reversed G clef"),
    muse::TranslatableString::untranslatable("Turned G clef"),
    muse::TranslatableString::untranslatable("Glissando down"),
    muse::TranslatableString::untranslatable("Glissando up"),
    muse::TranslatableString::untranslatable("Slashed grace note stem down"),
    muse::TranslatableString::untranslatable("Slashed grace note stem up"),
    muse::TranslatableString::untranslatable("Grace note stem down"),
    muse::TranslatableString::untranslatable("Grace note stem up"),
    muse::TranslatableString::untranslatable("Slash for stem down grace note"),
    muse::TranslatableString::untranslatable("Slash for stem up grace note"),
    muse::TranslatableString::untranslatable("Full barré"),
    muse::TranslatableString::untranslatable("Half barré"),
    muse::TranslatableString::untranslatable("Closed wah/volume pedal"),
    muse::TranslatableString("engraving/sym", "Fade in"),
    muse::TranslatableString("engraving/sym", "Fade out"),
    muse::TranslatableString::untranslatable("Golpe (tapping the pick guard)"),
    muse::TranslatableString::untranslatable("Half-open wah/volume pedal"),
    muse::TranslatableString::untranslatable("Left-hand tapping"),
    muse::TranslatableString::untranslatable("Open wah/volume pedal"),
    muse::TranslatableString::untranslatable("Right-hand tapping"),
    muse::TranslatableString::untranslatable("Guitar shake"),
    muse::TranslatableString::untranslatable("String number 0"),
    muse::TranslatableString::untranslatable("String number 1"),
    muse::TranslatableString::untranslatable("String number 10"),
    muse::TranslatableString::untranslatable("String number 11"),
    muse::TranslatableString::untranslatable("String number 12"),
    muse::TranslatableString::untranslatable("String number 13"),
    muse::TranslatableString::untranslatable("String number 2"),
    muse::TranslatableString::untranslatable("String number 3"),
    muse::TranslatableString::untranslatable("String number 4"),
    muse::TranslatableString::untranslatable("String number 5"),
    muse::TranslatableString::untranslatable("String number 6"),
    muse::TranslatableString::untranslatable("String number 7"),
    muse::TranslatableString::untranslatable("String number 8"),
    muse::TranslatableString::untranslatable("String number 9"),
    muse::TranslatableString::untranslatable("Strum direction down"),
    muse::TranslatableString::untranslatable("Strum direction up"),
    muse::TranslatableString::untranslatable("Guitar vibrato bar dip"),
    muse::TranslatableString::untranslatable("Guitar vibrato bar scoop"),
    muse::TranslatableString::untranslatable("Vibrato wiggle segment"),
    muse::TranslatableString("engraving/sym", "Volume swell"),
    muse::TranslatableString::untranslatable("Wide vibrato wiggle segment"),
    muse::TranslatableString::untranslatable("Belltree"),
    muse::TranslatableString::untranslatable("Damp 3"),
    muse::TranslatableString::untranslatable("Echo"),
    muse::TranslatableString::untranslatable("Echo 2"),
    muse::TranslatableString("engraving/sym", "Gyro"),
    muse::TranslatableString("engraving/sym", "Hand martellato"),
    muse::TranslatableString("engraving/sym", "Mallet, bell on table"),
    muse::TranslatableString("engraving/sym", "Mallet, bell suspended"),
    muse::TranslatableString("engraving/sym", "Mallet lift"),
    muse::TranslatableString("engraving/sym", "Martellato"),
    muse::TranslatableString("engraving/sym", "Martellato lift"),
    muse::TranslatableString("engraving/sym", "Muted martellato"),
    muse::TranslatableString("engraving/sym", "Pluck lift"),
    muse::TranslatableString::untranslatable("Swing"),
    muse::TranslatableString::untranslatable("Swing down"),
    muse::TranslatableString::untranslatable("Swing up"),
    muse::TranslatableString::untranslatable("Table pair of handbells"),
    muse::TranslatableString::untranslatable("Table single handbell"),
    muse::TranslatableString::untranslatable("Metal rod pictogram"),
    muse::TranslatableString::untranslatable("Harp pedal centered (natural)"),
    muse::TranslatableString::untranslatable("Harp pedal divider"),
    muse::TranslatableString::untranslatable("Harp pedal lowered (sharp)"),
    muse::TranslatableString::untranslatable("Harp pedal raised (flat)"),
    muse::TranslatableString::untranslatable("Ascending aeolian chords (Salzedo)"),
    muse::TranslatableString::untranslatable("Descending aeolian chords (Salzedo)"),
    muse::TranslatableString::untranslatable("Damp above (Salzedo)"),
    muse::TranslatableString::untranslatable("Damp below (Salzedo)"),
    muse::TranslatableString::untranslatable("Damp with both hands (Salzedo)"),
    muse::TranslatableString::untranslatable("Damp only low strings (Salzedo)"),
    muse::TranslatableString::untranslatable("Fluidic sounds, left hand (Salzedo)"),
    muse::TranslatableString::untranslatable("Fluidic sounds, right hand (Salzedo)"),
    muse::TranslatableString::untranslatable("Isolated sounds (Salzedo)"),
    muse::TranslatableString::untranslatable("Metallic sounds (Salzedo)"),
    muse::TranslatableString::untranslatable("Metallic sounds, one string (Salzedo)"),
    muse::TranslatableString::untranslatable("Muffle totally (Salzedo)"),
    muse::TranslatableString::untranslatable("Oboic flux (Salzedo)"),
    muse::TranslatableString::untranslatable("Play at upper end of strings (Salzedo)"),
    muse::TranslatableString::untranslatable("Slide with suppleness (Salzedo)"),
    muse::TranslatableString::untranslatable("Snare drum effect (Salzedo)"),
    muse::TranslatableString::untranslatable("Tam-tam sounds (Salzedo)"),
    muse::TranslatableString::untranslatable("Thunder effect (Salzedo)"),
    muse::TranslatableString::untranslatable("Timpanic sounds (Salzedo)"),
    muse::TranslatableString::untranslatable("Whistling sounds (Salzedo)"),
    muse::TranslatableString::untranslatable("Combining string noise for stem"),
    muse::TranslatableString::untranslatable("Tuning key pictogram"),
    muse::TranslatableString::untranslatable("Retune strings for glissando"),
    muse::TranslatableString::untranslatable("Use handle of tuning key pictogram"),
    muse::TranslatableString::untranslatable("Use shank of tuning key pictogram"),
    muse::TranslatableString::untranslatable("Indian drum clef"),
    muse::TranslatableString::untranslatable("Back-chug"),
    muse::TranslatableString::untranslatable("Back-flap"),
    muse::TranslatableString::untranslatable("Back-riff"),
    muse::TranslatableString::untranslatable("Back-rip"),
    muse::TranslatableString::untranslatable("Ball-change"),
    muse::TranslatableString::untranslatable("Ball-dig"),
    muse::TranslatableString::untranslatable("Brush-backward"),
    muse::TranslatableString::untranslatable("Brush-forward"),
    muse::TranslatableString::untranslatable("Chug"),
    muse::TranslatableString::untranslatable("Clap"),
    muse::TranslatableString::untranslatable("Double-snap"),
    muse::TranslatableString::untranslatable("Double-wing"),
    muse::TranslatableString::untranslatable("Draw-step"),
    muse::TranslatableString::untranslatable("Draw-tap"),
    muse::TranslatableString::untranslatable("Flam"),
    muse::TranslatableString::untranslatable("Flap"),
    muse::TranslatableString::untranslatable("Flap-step"),
    muse::TranslatableString::untranslatable("Flat"),
    muse::TranslatableString::untranslatable("Flea-hop"),
    muse::TranslatableString::untranslatable("Flea-tap"),
    muse::TranslatableString::untranslatable("Grace-tap"),
    muse::TranslatableString::untranslatable("Grace-tap-change"),
    muse::TranslatableString::untranslatable("Grace-tap-hop"),
    muse::TranslatableString::untranslatable("Grace-tap-stamp"),
    muse::TranslatableString::untranslatable("Heel"),
    muse::TranslatableString::untranslatable("Heel-change"),
    muse::TranslatableString::untranslatable("Heel-click"),
    muse::TranslatableString::untranslatable("Heel-drop"),
    muse::TranslatableString::untranslatable("Heel-step"),
    muse::TranslatableString::untranslatable("Heel-tap"),
    muse::TranslatableString::untranslatable("Hop"),
    muse::TranslatableString::untranslatable("Jump-apart"),
    muse::TranslatableString::untranslatable("Jump-together"),
    muse::TranslatableString::untranslatable("Knee-inward"),
    muse::TranslatableString::untranslatable("Knee-outward"),
    muse::TranslatableString::untranslatable("Leap"),
    muse::TranslatableString::untranslatable("Leap-flat-foot"),
    muse::TranslatableString::untranslatable("Leap-heel-click"),
    muse::TranslatableString::untranslatable("Left-catch"),
    muse::TranslatableString::untranslatable("Left-cross"),
    muse::TranslatableString::untranslatable("Left-foot"),
    muse::TranslatableString::untranslatable("Left-toe-strike"),
    muse::TranslatableString::untranslatable("Left-turn"),
    muse::TranslatableString::untranslatable("Over-the-top"),
    muse::TranslatableString::untranslatable("Over-the-top-tap"),
    muse::TranslatableString::untranslatable("Pull"),
    muse::TranslatableString::untranslatable("Push"),
    muse::TranslatableString::untranslatable("Riff"),
    muse::TranslatableString::untranslatable("Riffle"),
    muse::TranslatableString::untranslatable("Right-catch"),
    muse::TranslatableString::untranslatable("Right-cross"),
    muse::TranslatableString::untranslatable("Right-foot"),
    muse::TranslatableString::untranslatable("Right-toe-strike"),
    muse::TranslatableString::untranslatable("Right-turn"),
    muse::TranslatableString::untranslatable("Rip"),
    muse::TranslatableString::untranslatable("Ripple"),
    muse::TranslatableString::untranslatable("Scrape"),
    muse::TranslatableString::untranslatable("Scuff"),
    muse::TranslatableString::untranslatable("Scuffle"),
    muse::TranslatableString::untranslatable("Shuffle"),
    muse::TranslatableString::untranslatable("Slam"),
    muse::TranslatableString::untranslatable("Slap"),
    muse::TranslatableString::untranslatable("Slide-step"),
    muse::TranslatableString::untranslatable("Slide-tap"),
    muse::TranslatableString::untranslatable("Snap"),
    muse::TranslatableString::untranslatable("Stamp"),
    muse::TranslatableString::untranslatable("Stamp-stamp"),
    muse::TranslatableString::untranslatable("Step"),
    muse::TranslatableString::untranslatable("Step-stamp"),
    muse::TranslatableString::untranslatable("Stomp"),
    muse::TranslatableString::untranslatable("Stomp-brush"),
    muse::TranslatableString::untranslatable("Tap"),
    muse::TranslatableString::untranslatable("Toe"),
    muse::TranslatableString::untranslatable("Toe-click"),
    muse::TranslatableString::untranslatable("Toe-drop"),
    muse::TranslatableString::untranslatable("Toe-step"),
    muse::TranslatableString::untranslatable("Toe-tap"),
    muse::TranslatableString::untranslatable("Trench"),
    muse::TranslatableString::untranslatable("Wing"),
    muse::TranslatableString::untranslatable("Wing-change"),
    muse::TranslatableString::untranslatable("Zank"),
    muse::TranslatableString::untranslatable("Zink"),
    muse::TranslatableString::untranslatable("Clavichord bebung, 2 finger movements (above)"),
    muse::TranslatableString::untranslatable("Clavichord bebung, 2 finger movements (below)"),
    muse::TranslatableString::untranslatable("Clavichord bebung, 3 finger movements (above)"),
    muse::TranslatableString::untranslatable("Clavichord bebung, 3 finger movements (below)"),
    muse::TranslatableString::untranslatable("Clavichord bebung, 4 finger movements (above)"),
    muse::TranslatableString::untranslatable("Clavichord bebung, 4 finger movements (below)"),
    muse::TranslatableString::untranslatable("Left pedal pictogram"),
    muse::TranslatableString::untranslatable("Middle pedal pictogram"),
    muse::TranslatableString::untranslatable("Pedal d"),
    muse::TranslatableString::untranslatable("Pedal dot"),
    muse::TranslatableString::untranslatable("Pedal e"),
    muse::TranslatableString::untranslatable("Half-pedal mark"),
    muse::TranslatableString::untranslatable("Half pedal mark 1"),
    muse::TranslatableString::untranslatable("Half pedal mark 2"),
    muse::TranslatableString::untranslatable("Pedal heel 1"),
    muse::TranslatableString::untranslatable("Pedal heel 2"),
    muse::TranslatableString::untranslatable("Pedal heel 3 (Davis)"),
    muse::TranslatableString::untranslatable("Pedal heel to toe"),
    muse::TranslatableString::untranslatable("Pedal heel or toe"),
    muse::TranslatableString::untranslatable("Pedal hook end"),
    muse::TranslatableString::untranslatable("Pedal hook start"),
    muse::TranslatableString::untranslatable("Pedal hyphen"),
    muse::TranslatableString::untranslatable("Pedal P"),
    muse::TranslatableString::untranslatable("Left parenthesis for pedal marking"),
    muse::TranslatableString::untranslatable("Right parenthesis for pedal marking"),
    muse::TranslatableString::untranslatable("Pedal mark"),
    muse::TranslatableString::untranslatable("Pedal S"),
    muse::TranslatableString::untranslatable("Sostenuto pedal mark"),
    muse::TranslatableString::untranslatable("Pedal toe 1"),
    muse::TranslatableString::untranslatable("Pedal toe 2"),
    muse::TranslatableString::untranslatable("Pedal toe to heel"),
    muse::TranslatableString::untranslatable("Pedal up mark"),
    muse::TranslatableString::untranslatable("Pedal up notch"),
    muse::TranslatableString::untranslatable("Pedal up special"),
    muse::TranslatableString::untranslatable("Play with left hand"),
    muse::TranslatableString::untranslatable("Play with left hand (end)"),
    muse::TranslatableString::untranslatable("Play with right hand"),
    muse::TranslatableString::untranslatable("Play with right hand (end)"),
    muse::TranslatableString::untranslatable("Pluck strings inside piano (Maderna)"),
    muse::TranslatableString::untranslatable("Right pedal pictogram"),
    muse::TranslatableString::untranslatable("Kievan flat"),
    muse::TranslatableString::untranslatable("Kievan sharp"),
    muse::TranslatableString::untranslatable("Kievan augmentation dot"),
    muse::TranslatableString::untranslatable("Kievan C clef (tse-fa-ut)"),
    muse::TranslatableString::untranslatable("Kievan ending symbol"),
    muse::TranslatableString::untranslatable("Kievan eighth note, stem down"),
    muse::TranslatableString::untranslatable("Kievan eighth note, stem up"),
    muse::TranslatableString::untranslatable("Kievan beam"),
    muse::TranslatableString::untranslatable("Kievan half note (on staff line)"),
    muse::TranslatableString::untranslatable("Kievan half note (in staff space)"),
    muse::TranslatableString::untranslatable("Kievan quarter note, stem down"),
    muse::TranslatableString::untranslatable("Kievan quarter note, stem up"),
    muse::TranslatableString::untranslatable("Kievan reciting note"),
    muse::TranslatableString::untranslatable("Kievan whole note"),
    muse::TranslatableString::untranslatable("Kievan final whole note"),
    muse::TranslatableString::untranslatable("Do hand sign"),
    muse::TranslatableString::untranslatable("Fa hand sign"),
    muse::TranslatableString::untranslatable("La hand sign"),
    muse::TranslatableString::untranslatable("Mi hand sign"),
    muse::TranslatableString::untranslatable("Re hand sign"),
    muse::TranslatableString::untranslatable("So hand sign"),
    muse::TranslatableString::untranslatable("Ti hand sign"),
    muse::TranslatableString::untranslatable("Left repeat sign within bar"),
    muse::TranslatableString::untranslatable("Leger line"),
    muse::TranslatableString::untranslatable("Leger line (narrow)"),
    muse::TranslatableString::untranslatable("Leger line (wide)"),
    muse::TranslatableString::untranslatable("Lute tablature end repeat barline"),
    muse::TranslatableString::untranslatable("Lute tablature final barline"),
    muse::TranslatableString::untranslatable("Lute tablature start repeat barline"),
    muse::TranslatableString::untranslatable("16th note (semiquaver) duration sign"),
    muse::TranslatableString::untranslatable("32nd note (demisemiquaver) duration sign"),
    muse::TranslatableString::untranslatable("Eighth note (quaver) duration sign"),
    muse::TranslatableString::untranslatable("Double whole note (breve) duration sign"),
    muse::TranslatableString::untranslatable("Half note (minim) duration sign"),
    muse::TranslatableString::untranslatable("Quarter note (crotchet) duration sign"),
    muse::TranslatableString::untranslatable("Whole note (semibreve) duration sign"),
    muse::TranslatableString("engraving/sym", "Right-hand fingering, first finger"),
    muse::TranslatableString("engraving/sym", "Right-hand fingering, second finger"),
    muse::TranslatableString("engraving/sym", "Right-hand fingering, third finger"),
    muse::TranslatableString("engraving/sym", "Right-hand fingering, thumb"),
    muse::TranslatableString::untranslatable("10th course (diapason)"),
    muse::TranslatableString::untranslatable("Seventh course (diapason)"),
    muse::TranslatableString::untranslatable("Eighth course (diapason)"),
    muse::TranslatableString::untranslatable("Ninth course (diapason)"),
    muse::TranslatableString::untranslatable("Appoggiatura from above"),
    muse::TranslatableString::untranslatable("Appoggiatura from below"),
    muse::TranslatableString::untranslatable("Open string (a)"),
    muse::TranslatableString::untranslatable("First fret (b)"),
    muse::TranslatableString::untranslatable("Second fret (c)"),
    muse::TranslatableString::untranslatable("Third fret (d)"),
    muse::TranslatableString::untranslatable("Fourth fret (e)"),
    muse::TranslatableString::untranslatable("Fifth fret (f)"),
    muse::TranslatableString::untranslatable("Sixth fret (g)"),
    muse::TranslatableString::untranslatable("Seventh fret (h)"),
    muse::TranslatableString::untranslatable("Eighth fret (i)"),
    muse::TranslatableString::untranslatable("Ninth fret (k)"),
    muse::TranslatableString::untranslatable("10th fret (l)"),
    muse::TranslatableString::untranslatable("11th fret (m)"),
    muse::TranslatableString::untranslatable("12th fret (n)"),
    muse::TranslatableString::untranslatable("Inverted mordent"),
    muse::TranslatableString::untranslatable("Mordent with lower auxiliary"),
    muse::TranslatableString::untranslatable("Mordent with upper auxiliary"),
    muse::TranslatableString::untranslatable("5th course, 1st fret (a)"),
    muse::TranslatableString::untranslatable("6th course, 1st fret (A)"),
    muse::TranslatableString::untranslatable("4th course, 1st fret (b)"),
    muse::TranslatableString::untranslatable("6th course, 2nd fret (B)"),
    muse::TranslatableString::untranslatable("3rd course, 1st fret (c)"),
    muse::TranslatableString::untranslatable("6th course, 3rd fret (C)"),
    muse::TranslatableString::untranslatable("2nd course, 1st fret (d)"),
    muse::TranslatableString::untranslatable("6th course, 4th fret (D)"),
    muse::TranslatableString::untranslatable("1st course, 1st fret (e)"),
    muse::TranslatableString::untranslatable("6th course, 5th fret (E)"),
    muse::TranslatableString::untranslatable("5th course, 2nd fret (f)"),
    muse::TranslatableString::untranslatable("6th course, 6th fret (F)"),
    muse::TranslatableString::untranslatable("4th course, 2nd fret (g)"),
    muse::TranslatableString::untranslatable("6th course, 7th fret (G)"),
    muse::TranslatableString::untranslatable("3rd course, 2nd fret (h)"),
    muse::TranslatableString::untranslatable("6th course, 8th fret (H)"),
    muse::TranslatableString::untranslatable("2nd course, 2nd fret (i)"),
    muse::TranslatableString::untranslatable("6th course, 9th fret (I)"),
    muse::TranslatableString::untranslatable("1st course, 2nd fret (k)"),
    muse::TranslatableString::untranslatable("6th course, 10th fret (K)"),
    muse::TranslatableString::untranslatable("5th course, 3rd fret (l)"),
    muse::TranslatableString::untranslatable("6th course, 11th fret (L)"),
    muse::TranslatableString::untranslatable("4th course, 3rd fret (m)"),
    muse::TranslatableString::untranslatable("6th course, 12th fret (M)"),
    muse::TranslatableString::untranslatable("3rd course, 3rd fret (n)"),
    muse::TranslatableString::untranslatable("6th course, 13th fret (N)"),
    muse::TranslatableString::untranslatable("2nd course, 3rd fret (o)"),
    muse::TranslatableString::untranslatable("1st course, 3rd fret (p)"),
    muse::TranslatableString::untranslatable("5th course, 4th fret (q)"),
    muse::TranslatableString::untranslatable("4th course, 4th fret (r)"),
    muse::TranslatableString::untranslatable("3rd course, 4th fret (s)"),
    muse::TranslatableString::untranslatable("2nd course, 4th fret (t)"),
    muse::TranslatableString::untranslatable("1st course, 4th fret (v)"),
    muse::TranslatableString::untranslatable("5th course, 5th fret (x)"),
    muse::TranslatableString::untranslatable("4th course, 5th fret (y)"),
    muse::TranslatableString::untranslatable("3rd course, 5th fret (z)"),
    muse::TranslatableString::untranslatable("C sol fa ut clef"),
    muse::TranslatableString::untranslatable("F fa ut clef"),
    muse::TranslatableString::untranslatable("Open string (0)"),
    muse::TranslatableString::untranslatable("First fret (1)"),
    muse::TranslatableString::untranslatable("Second fret (2)"),
    muse::TranslatableString::untranslatable("Third fret (3)"),
    muse::TranslatableString::untranslatable("Fourth fret (4)"),
    muse::TranslatableString::untranslatable("Fifth fret (5)"),
    muse::TranslatableString::untranslatable("Sixth fret (6)"),
    muse::TranslatableString::untranslatable("Seventh fret (7)"),
    muse::TranslatableString::untranslatable("Eighth fret (8)"),
    muse::TranslatableString::untranslatable("Ninth fret (9)"),
    muse::TranslatableString::untranslatable("Hold finger in place"),
    muse::TranslatableString::untranslatable("Hold note"),
    muse::TranslatableString::untranslatable("Release finger"),
    muse::TranslatableString::untranslatable("Fast tempo indication (de Mudarra)"),
    muse::TranslatableString::untranslatable("Neither fast nor slow tempo indication (de Mudarra)"),
    muse::TranslatableString::untranslatable("Slow tempo indication (de Mudarra)"),
    muse::TranslatableString::untranslatable("Somewhat fast tempo indication (de Narvaez)"),
    muse::TranslatableString::untranslatable("Very slow indication (de Narvaez)"),
    muse::TranslatableString::untranslatable("Triple time indication"),
    muse::TranslatableString::untranslatable("Single-finger tremolo or mordent"),
    muse::TranslatableString::untranslatable("Vibrato (verre cassé)"),
    muse::TranslatableString::untranslatable("Lute tablature staff, 6 courses"),
    muse::TranslatableString::untranslatable("Lute tablature staff, 6 courses (narrow)"),
    muse::TranslatableString::untranslatable("Lute tablature staff, 6 courses (wide)"),
    muse::TranslatableString("engraving/sym", "Elision"),
    muse::TranslatableString("engraving/sym", "Narrow elision"),
    muse::TranslatableString("engraving/sym", "Wide elision"),
    muse::TranslatableString::untranslatable("Baseline hyphen"),
    muse::TranslatableString::untranslatable("Non-breaking baseline hyphen"),
    muse::TranslatableString::untranslatable("Text repeats"),
    muse::TranslatableString::untranslatable("Flat, hard b (mi)"),
    muse::TranslatableString::untranslatable("Flat, soft b (fa)"),
    muse::TranslatableString::untranslatable("Flat with dot"),
    muse::TranslatableString::untranslatable("G clef (Corpus Monodicum)"),
    muse::TranslatableString::untranslatable("Liquescence"),
    muse::TranslatableString::untranslatable("Liquescent ascending (Corpus Monodicum)"),
    muse::TranslatableString::untranslatable("Liquescent descending (Corpus Monodicum)"),
    muse::TranslatableString::untranslatable("Natural"),
    muse::TranslatableString::untranslatable("Natural with interrupted cross"),
    muse::TranslatableString::untranslatable("Oriscus (Corpus Monodicum)"),
    muse::TranslatableString::untranslatable("Plica"),
    muse::TranslatableString::untranslatable("Punctum (Corpus Monodicum)"),
    muse::TranslatableString::untranslatable("Quilisma (Corpus Monodicum)"),
    muse::TranslatableString::untranslatable("Croix"),
    muse::TranslatableString::untranslatable("Strophicus (Corpus Monodicum)"),
    muse::TranslatableString::untranslatable("Alteration sign"),
    muse::TranslatableString::untranslatable("Black mensural brevis"),
    muse::TranslatableString::untranslatable("Black mensural void brevis"),
    muse::TranslatableString::untranslatable("Black mensural dragma"),
    muse::TranslatableString::untranslatable("Black mensural longa"),
    muse::TranslatableString::untranslatable("Black mensural maxima"),
    muse::TranslatableString::untranslatable("Black mensural minima"),
    muse::TranslatableString::untranslatable("Black mensural void minima"),
    muse::TranslatableString::untranslatable("Black mensural semibrevis"),
    muse::TranslatableString::untranslatable("Black mensural semibrevis caudata"),
    muse::TranslatableString::untranslatable("Black mensural oblique semibrevis"),
    muse::TranslatableString::untranslatable("Black mensural void semibrevis"),
    muse::TranslatableString::untranslatable("Black mensural semiminima"),
    muse::TranslatableString::untranslatable("Mensural C clef"),
    muse::TranslatableString::untranslatable("Petrucci C clef, high position"),
    muse::TranslatableString::untranslatable("Petrucci C clef, highest position"),
    muse::TranslatableString::untranslatable("Petrucci C clef, low position"),
    muse::TranslatableString::untranslatable("Petrucci C clef, lowest position"),
    muse::TranslatableString::untranslatable("Petrucci C clef, middle position"),
    muse::TranslatableString::untranslatable("Coloration end, round"),
    muse::TranslatableString::untranslatable("Coloration end, square"),
    muse::TranslatableString::untranslatable("Coloration start, round"),
    muse::TranslatableString::untranslatable("Coloration start, square"),
    muse::TranslatableString::untranslatable("Combining stem diagonal"),
    muse::TranslatableString::untranslatable("Combining stem down"),
    muse::TranslatableString::untranslatable("Combining stem with extended flag down"),
    muse::TranslatableString::untranslatable("Combining stem with flared flag down"),
    muse::TranslatableString::untranslatable("Combining stem with fusa flag down"),
    muse::TranslatableString::untranslatable("Combining stem with flag left down"),
    muse::TranslatableString::untranslatable("Combining stem with flag right down"),
    muse::TranslatableString::untranslatable("Combining stem with semiminima flag down"),
    muse::TranslatableString::untranslatable("Combining stem up"),
    muse::TranslatableString::untranslatable("Combining stem with extended flag up"),
    muse::TranslatableString::untranslatable("Combining stem with flared flag up"),
    muse::TranslatableString::untranslatable("Combining stem with fusa flag up"),
    muse::TranslatableString::untranslatable("Combining stem with flag left up"),
    muse::TranslatableString::untranslatable("Combining stem with flag right up"),
    muse::TranslatableString::untranslatable("Combining stem with semiminima flag up"),
    muse::TranslatableString::untranslatable("Checkmark custos"),
    muse::TranslatableString::untranslatable("Mensural custos down"),
    muse::TranslatableString::untranslatable("Turn-like custos"),
    muse::TranslatableString::untranslatable("Mensural custos up"),
    muse::TranslatableString::untranslatable("Mensural F clef"),
    muse::TranslatableString::untranslatable("Petrucci F clef"),
    muse::TranslatableString::untranslatable("Mensural G clef"),
    muse::TranslatableString::untranslatable("Petrucci G clef"),
    muse::TranslatableString::untranslatable("Modus imperfectum, vertical"),
    muse::TranslatableString::untranslatable("Modus perfectum, vertical"),
    muse::TranslatableString::untranslatable("Longa/brevis notehead, black"),
    muse::TranslatableString::untranslatable("Longa/brevis notehead, black and void"),
    muse::TranslatableString::untranslatable("Longa/brevis notehead, void"),
    muse::TranslatableString::untranslatable("Longa/brevis notehead, white"),
    muse::TranslatableString::untranslatable("Maxima notehead, black"),
    muse::TranslatableString::untranslatable("Maxima notehead, black and void"),
    muse::TranslatableString::untranslatable("Maxima notehead, void"),
    muse::TranslatableString::untranslatable("Maxima notehead, white"),
    muse::TranslatableString::untranslatable("Minima notehead, white"),
    muse::TranslatableString::untranslatable("Semibrevis notehead, black"),
    muse::TranslatableString::untranslatable("Semibrevis notehead, black and void"),
    muse::TranslatableString::untranslatable("Semibrevis notehead, black and void (turned)"),
    muse::TranslatableString::untranslatable("Semibrevis notehead, void"),
    muse::TranslatableString::untranslatable("Semiminima/fusa notehead, white"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 2nd, black"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 2nd, black and void"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 2nd, void"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 2nd, white"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 3rd, black"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 3rd, black and void"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 3rd, void"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 3rd, white"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 4th, black"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 4th, black and void"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 4th, void"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 4th, white"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 5th, black"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 5th, black and void"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 5th, void"),
    muse::TranslatableString::untranslatable("Oblique form, ascending 5th, white"),
    muse::TranslatableString::untranslatable("Oblique form, descending 2nd, black"),
    muse::TranslatableString::untranslatable("Oblique form, descending 2nd, black and void"),
    muse::TranslatableString::untranslatable("Oblique form, descending 2nd, void"),
    muse::TranslatableString::untranslatable("Oblique form, descending 2nd, white"),
    muse::TranslatableString::untranslatable("Oblique form, descending 3rd, black"),
    muse::TranslatableString::untranslatable("Oblique form, descending 3rd, black and void"),
    muse::TranslatableString::untranslatable("Oblique form, descending 3rd, void"),
    muse::TranslatableString::untranslatable("Oblique form, descending 3rd, white"),
    muse::TranslatableString::untranslatable("Oblique form, descending 4th, black"),
    muse::TranslatableString::untranslatable("Oblique form, descending 4th, black and void"),
    muse::TranslatableString::untranslatable("Oblique form, descending 4th, void"),
    muse::TranslatableString::untranslatable("Oblique form, descending 4th, white"),
    muse::TranslatableString::untranslatable("Oblique form, descending 5th, black"),
    muse::TranslatableString::untranslatable("Oblique form, descending 5th, black and void"),
    muse::TranslatableString::untranslatable("Oblique form, descending 5th, void"),
    muse::TranslatableString::untranslatable("Oblique form, descending 5th, white"),
    muse::TranslatableString::untranslatable("Tempus perfectum cum prolatione perfecta (9/8)"),
    muse::TranslatableString::untranslatable("Tempus imperfectum cum prolatione imperfecta diminution 4"),
    muse::TranslatableString::untranslatable("Tempus imperfectum cum prolatione imperfecta diminution 5"),
    muse::TranslatableString::untranslatable("Tempus perfectum cum prolatione imperfecta (3/4)"),
    muse::TranslatableString::untranslatable("Tempus perfectum cum prolatione imperfecta diminution 1 (3/8)"),
    muse::TranslatableString::untranslatable("Tempus perfectum cum prolatione perfecta diminution 2 (9/16)"),
    muse::TranslatableString::untranslatable("Tempus imperfectum cum prolatione perfecta (6/8)"),
    muse::TranslatableString::untranslatable("Tempus imperfectum cum prolatione imperfecta (2/4)"),
    muse::TranslatableString::untranslatable("Tempus imperfectum cum prolatione imperfecta diminution 1 (2/2)"),
    muse::TranslatableString::untranslatable("Tempus imperfectum cum prolatione imperfecta diminution 2 (6/16)"),
    muse::TranslatableString::untranslatable("Tempus imperfectum cum prolatione imperfecta diminution 3 (2/2)"),
    muse::TranslatableString::untranslatable("Combining dot"),
    muse::TranslatableString::untranslatable("Combining void dot"),
    muse::TranslatableString::untranslatable("Combining vertical stroke"),
    muse::TranslatableString::untranslatable("Combining three dots horizontal"),
    muse::TranslatableString::untranslatable("Combining three dots triangular"),
    muse::TranslatableString::untranslatable("Combining two dots"),
    muse::TranslatableString::untranslatable("Mensural proportion 1"),
    muse::TranslatableString::untranslatable("Mensural proportion 2"),
    muse::TranslatableString::untranslatable("Mensural proportion 3"),
    muse::TranslatableString::untranslatable("Mensural proportion 4"),
    muse::TranslatableString::untranslatable("Mensural proportion 5"),
    muse::TranslatableString::untranslatable("Mensural proportion 6"),
    muse::TranslatableString::untranslatable("Mensural proportion 7"),
    muse::TranslatableString::untranslatable("Mensural proportion 8"),
    muse::TranslatableString::untranslatable("Mensural proportion 9"),
    muse::TranslatableString::untranslatable("Mensural proportion major"),
    muse::TranslatableString::untranslatable("Mensural proportion minor"),
    muse::TranslatableString::untranslatable("Proportio dupla 1"),
    muse::TranslatableString::untranslatable("Proportio dupla 2"),
    muse::TranslatableString::untranslatable("Proportio quadrupla"),
    muse::TranslatableString::untranslatable("Proportio tripla"),
    muse::TranslatableString::untranslatable("Tempus perfectum"),
    muse::TranslatableString::untranslatable("Brevis rest"),
    muse::TranslatableString::untranslatable("Fusa rest"),
    muse::TranslatableString::untranslatable("Longa imperfecta rest"),
    muse::TranslatableString::untranslatable("Longa perfecta rest"),
    muse::TranslatableString::untranslatable("Maxima rest"),
    muse::TranslatableString::untranslatable("Minima rest"),
    muse::TranslatableString::untranslatable("Semibrevis rest"),
    muse::TranslatableString::untranslatable("Semifusa rest"),
    muse::TranslatableString::untranslatable("Semiminima rest"),
    muse::TranslatableString::untranslatable("Signum congruentiae down"),
    muse::TranslatableString::untranslatable("Signum congruentiae up"),
    muse::TranslatableString::untranslatable("Tempus imperfectum, horizontal"),
    muse::TranslatableString::untranslatable("Tempus perfectum, horizontal"),
    muse::TranslatableString::untranslatable("White mensural brevis"),
    muse::TranslatableString::untranslatable("White mensural fusa"),
    muse::TranslatableString::untranslatable("White mensural longa"),
    muse::TranslatableString::untranslatable("White mensural maxima"),
    muse::TranslatableString::untranslatable("White mensural minima"),
    muse::TranslatableString::untranslatable("White mensural semibrevis"),
    muse::TranslatableString::untranslatable("White mensural semiminima"),
    muse::TranslatableString("engraving/sym", "Augmentation dot"),
    muse::TranslatableString::untranslatable("1024th note (semihemidemisemihemidemisemiquaver) stem down"),
    muse::TranslatableString("engraving/sym", "1024th note (semihemidemisemihemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("128th note (semihemidemisemiquaver) stem down"),
    muse::TranslatableString("engraving/sym", "128th note (semihemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("16th note (semiquaver) stem down"),
    muse::TranslatableString("engraving/sym", "16th note (semiquaver) stem up"),
    muse::TranslatableString::untranslatable("256th note (demisemihemidemisemiquaver) stem down"),
    muse::TranslatableString("engraving/sym", "256th note (demisemihemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("32nd note (demisemiquaver) stem down"),
    muse::TranslatableString("engraving/sym", "32nd note (demisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("512th note (hemidemisemihemidemisemiquaver) stem down"),
    muse::TranslatableString("engraving/sym", "512th note (hemidemisemihemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("64th note (hemidemisemiquaver) stem down"),
    muse::TranslatableString("engraving/sym", "64th note (hemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("Eighth note (quaver) stem down"),
    muse::TranslatableString("engraving/sym", "Eighth note (quaver) stem up"),
    muse::TranslatableString::untranslatable("Double whole note (breve)"),
    muse::TranslatableString::untranslatable("Double whole note (square)"),
    muse::TranslatableString::untranslatable("Half note (minim) stem down"),
    muse::TranslatableString("engraving/sym", "Half note (minim) stem up"),
    muse::TranslatableString::untranslatable("Quarter note (crotchet) stem down"),
    muse::TranslatableString("engraving/sym", "Quarter note (crotchet) stem up"),
    muse::TranslatableString("engraving/sym", "Whole note (semibreve)"),
    muse::TranslatableString::untranslatable("Left-pointing arrow for metric modulation"),
    muse::TranslatableString::untranslatable("Right-pointing arrow for metric modulation"),
    muse::TranslatableString::untranslatable("Do not copy"),
    muse::TranslatableString::untranslatable("Do not photocopy"),
    muse::TranslatableString::untranslatable("Eyeglasses"),
    muse::TranslatableString::untranslatable("1024th note (semihemidemisemihemidemisemiquaver) stem down"),
    muse::TranslatableString::untranslatable("1024th note (semihemidemisemihemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("128th note (semihemidemisemiquaver) stem down"),
    muse::TranslatableString::untranslatable("128th note (semihemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("16th note (semiquaver) stem down"),
    muse::TranslatableString::untranslatable("16th note (semiquaver) stem up"),
    muse::TranslatableString::untranslatable("256th note (demisemihemidemisemiquaver) stem down"),
    muse::TranslatableString::untranslatable("256th note (demisemihemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("32nd note (demisemiquaver) stem down"),
    muse::TranslatableString::untranslatable("32nd note (demisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("512th note (hemidemisemihemidemisemiquaver) stem down"),
    muse::TranslatableString::untranslatable("512th note (hemidemisemihemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("64th note (hemidemisemiquaver) stem down"),
    muse::TranslatableString::untranslatable("64th note (hemidemisemiquaver) stem up"),
    muse::TranslatableString::untranslatable("Eighth note (quaver) stem down"),
    muse::TranslatableString::untranslatable("Eighth note (quaver) stem up"),
    muse::TranslatableString::untranslatable("A (black note)"),
    muse::TranslatableString::untranslatable("A flat (black note)"),
    muse::TranslatableString::untranslatable("A flat (half note)"),
    muse::TranslatableString::untranslatable("A flat (whole note)"),
    muse::TranslatableString::untranslatable("A (half note)"),
    muse::TranslatableString::untranslatable("A sharp (black note)"),
    muse::TranslatableString::untranslatable("A sharp (half note)"),
    muse::TranslatableString::untranslatable("A sharp (whole note)"),
    muse::TranslatableString::untranslatable("A (whole note)"),
    muse::TranslatableString::untranslatable("B (black note)"),
    muse::TranslatableString::untranslatable("B flat (black note)"),
    muse::TranslatableString::untranslatable("B flat (half note)"),
    muse::TranslatableString::untranslatable("B flat (whole note)"),
    muse::TranslatableString::untranslatable("B (half note)"),
    muse::TranslatableString::untranslatable("B sharp (black note)"),
    muse::TranslatableString::untranslatable("B sharp (half note)"),
    muse::TranslatableString::untranslatable("B sharp (whole note)"),
    muse::TranslatableString::untranslatable("B (whole note)"),
    muse::TranslatableString::untranslatable("C (black note)"),
    muse::TranslatableString::untranslatable("C flat (black note)"),
    muse::TranslatableString::untranslatable("C flat (half note)"),
    muse::TranslatableString::untranslatable("C flat (whole note)"),
    muse::TranslatableString::untranslatable("C (half note)"),
    muse::TranslatableString::untranslatable("C sharp (black note)"),
    muse::TranslatableString::untranslatable("C sharp (half note)"),
    muse::TranslatableString::untranslatable("C sharp (whole note)"),
    muse::TranslatableString::untranslatable("C (whole note)"),
    muse::TranslatableString::untranslatable("D (black note)"),
    muse::TranslatableString::untranslatable("D flat (black note)"),
    muse::TranslatableString::untranslatable("D flat (half note)"),
    muse::TranslatableString::untranslatable("D flat (whole note)"),
    muse::TranslatableString::untranslatable("D (half note)"),
    muse::TranslatableString::untranslatable("D sharp (black note)"),
    muse::TranslatableString::untranslatable("D sharp (half note)"),
    muse::TranslatableString::untranslatable("D sharp (whole note)"),
    muse::TranslatableString::untranslatable("D (whole note)"),
    muse::TranslatableString::untranslatable("Di (black note)"),
    muse::TranslatableString::untranslatable("Di (half note)"),
    muse::TranslatableString::untranslatable("Di (whole note)"),
    muse::TranslatableString::untranslatable("Do (black note)"),
    muse::TranslatableString::untranslatable("Do (half note)"),
    muse::TranslatableString::untranslatable("Do (whole note)"),
    muse::TranslatableString::untranslatable("Double whole note (breve)"),
    muse::TranslatableString::untranslatable("Double whole note (square)"),
    muse::TranslatableString::untranslatable("E (black note)"),
    muse::TranslatableString::untranslatable("E flat (black note)"),
    muse::TranslatableString::untranslatable("E flat (half note)"),
    muse::TranslatableString::untranslatable("E flat (whole note)"),
    muse::TranslatableString::untranslatable("E (half note)"),
    muse::TranslatableString::untranslatable("E sharp (black note)"),
    muse::TranslatableString::untranslatable("E sharp (half note)"),
    muse::TranslatableString::untranslatable("E sharp (whole note)"),
    muse::TranslatableString::untranslatable("E (whole note)"),
    muse::TranslatableString::untranslatable("Empty black note"),
    muse::TranslatableString::untranslatable("Empty half note"),
    muse::TranslatableString::untranslatable("Empty whole note"),
    muse::TranslatableString::untranslatable("F (black note)"),
    muse::TranslatableString::untranslatable("F flat (black note)"),
    muse::TranslatableString::untranslatable("F flat (half note)"),
    muse::TranslatableString::untranslatable("F flat (whole note)"),
    muse::TranslatableString::untranslatable("F (half note)"),
    muse::TranslatableString::untranslatable("F sharp (black note)"),
    muse::TranslatableString::untranslatable("F sharp (half note)"),
    muse::TranslatableString::untranslatable("F sharp (whole note)"),
    muse::TranslatableString::untranslatable("F (whole note)"),
    muse::TranslatableString::untranslatable("Fa (black note)"),
    muse::TranslatableString::untranslatable("Fa (half note)"),
    muse::TranslatableString::untranslatable("Fa (whole note)"),
    muse::TranslatableString::untranslatable("Fi (black note)"),
    muse::TranslatableString::untranslatable("Fi (half note)"),
    muse::TranslatableString::untranslatable("Fi (whole note)"),
    muse::TranslatableString::untranslatable("G (black note)"),
    muse::TranslatableString::untranslatable("G flat (black note)"),
    muse::TranslatableString::untranslatable("G flat (half note)"),
    muse::TranslatableString::untranslatable("G flat (whole note)"),
    muse::TranslatableString::untranslatable("G (half note)"),
    muse::TranslatableString::untranslatable("G sharp (black note)"),
    muse::TranslatableString::untranslatable("G sharp (half note)"),
    muse::TranslatableString::untranslatable("G sharp (whole note)"),
    muse::TranslatableString::untranslatable("G (whole note)"),
    muse::TranslatableString::untranslatable("H (black note)"),
    muse::TranslatableString::untranslatable("H (half note)"),
    muse::TranslatableString::untranslatable("H sharp (black note)"),
    muse::TranslatableString::untranslatable("H sharp (half note)"),
    muse::TranslatableString::untranslatable("H sharp (whole note)"),
    muse::TranslatableString::untranslatable("H (whole note)"),
    muse::TranslatableString::untranslatable("Half note (minim) stem down"),
    muse::TranslatableString::untranslatable("Half note (minim) stem up"),
    muse::TranslatableString::untranslatable("La (black note)"),
    muse::TranslatableString::untranslatable("La (half note)"),
    muse::TranslatableString::untranslatable("La (whole note)"),
    muse::TranslatableString::untranslatable("Le (black note)"),
    muse::TranslatableString::untranslatable("Le (half note)"),
    muse::TranslatableString::untranslatable("Le (whole note)"),
    muse::TranslatableString::untranslatable("Li (black note)"),
    muse::TranslatableString::untranslatable("Li (half note)"),
    muse::TranslatableString::untranslatable("Li (whole note)"),
    muse::TranslatableString::untranslatable("Me (black note)"),
    muse::TranslatableString::untranslatable("Me (half note)"),
    muse::TranslatableString::untranslatable("Me (whole note)"),
    muse::TranslatableString::untranslatable("Mi (black note)"),
    muse::TranslatableString::untranslatable("Mi (half note)"),
    muse::TranslatableString::untranslatable("Mi (whole note)"),
    muse::TranslatableString::untranslatable("Quarter note (crotchet) stem down"),
    muse::TranslatableString::untranslatable("Quarter note (crotchet) stem up"),
    muse::TranslatableString::untranslatable("Ra (black note)"),
    muse::TranslatableString::untranslatable("Ra (half note)"),
    muse::TranslatableString::untranslatable("Ra (whole note)"),
    muse::TranslatableString::untranslatable("Re (black note)"),
    muse::TranslatableString::untranslatable("Re (half note)"),
    muse::TranslatableString::untranslatable("Re (whole note)"),
    muse::TranslatableString::untranslatable("Ri (black note)"),
    muse::TranslatableString::untranslatable("Ri (half note)"),
    muse::TranslatableString::untranslatable("Ri (whole note)"),
    muse::TranslatableString::untranslatable("Se (black note)"),
    muse::TranslatableString::untranslatable("Se (half note)"),
    muse::TranslatableString::untranslatable("Se (whole note)"),
    muse::TranslatableString::untranslatable("Arrowhead left black (Funk 7-shape re)"),
    muse::TranslatableString::untranslatable("Arrowhead left double whole (Funk 7-shape re)"),
    muse::TranslatableString::untranslatable("Arrowhead left white (Funk 7-shape re)"),
    muse::TranslatableString::untranslatable("Diamond black (4-shape mi; 7-shape mi)"),
    muse::TranslatableString::untranslatable("Diamond double whole (4-shape mi; 7-shape mi)"),
    muse::TranslatableString::untranslatable("Diamond white (4-shape mi; 7-shape mi)"),
    muse::TranslatableString::untranslatable("Isosceles triangle black (Walker 7-shape ti)"),
    muse::TranslatableString::untranslatable("Isosceles triangle double whole (Walker 7-shape ti)"),
    muse::TranslatableString::untranslatable("Isosceles triangle white (Walker 7-shape ti)"),
    muse::TranslatableString::untranslatable("Inverted keystone black (Walker 7-shape do)"),
    muse::TranslatableString::untranslatable("Inverted keystone double whole (Walker 7-shape do)"),
    muse::TranslatableString::untranslatable("Inverted keystone white (Walker 7-shape do)"),
    muse::TranslatableString::untranslatable("Moon black (Aikin 7-shape re)"),
    muse::TranslatableString::untranslatable("Moon double whole (Aikin 7-shape re)"),
    muse::TranslatableString::untranslatable("Moon left black (Funk 7-shape do)"),
    muse::TranslatableString::untranslatable("Moon left double whole (Funk 7-shape do)"),
    muse::TranslatableString::untranslatable("Moon left white (Funk 7-shape do)"),
    muse::TranslatableString::untranslatable("Moon white (Aikin 7-shape re)"),
    muse::TranslatableString::untranslatable("Quarter moon black (Walker 7-shape re)"),
    muse::TranslatableString::untranslatable("Quarter moon double whole (Walker 7-shape re)"),
    muse::TranslatableString::untranslatable("Quarter moon white (Walker 7-shape re)"),
    muse::TranslatableString::untranslatable("Round black (4-shape sol; 7-shape so)"),
    muse::TranslatableString::untranslatable("Round double whole (4-shape sol; 7-shape so)"),
    muse::TranslatableString::untranslatable("Round white (4-shape sol; 7-shape so)"),
    muse::TranslatableString::untranslatable("Square black (4-shape la; Aikin 7-shape la)"),
    muse::TranslatableString::untranslatable("Square double whole (4-shape la; Aikin 7-shape la)"),
    muse::TranslatableString::untranslatable("Square white (4-shape la; Aikin 7-shape la)"),
    muse::TranslatableString::untranslatable("Triangle left black (stem up; 4-shape fa; 7-shape fa)"),
    muse::TranslatableString::untranslatable("Triangle left double whole (stem up; 4-shape fa; 7-shape fa)"),
    muse::TranslatableString::untranslatable("Triangle left white (stem up; 4-shape fa; 7-shape fa)"),
    muse::TranslatableString::untranslatable("Triangle right black (stem down; 4-shape fa; 7-shape fa)"),
    muse::TranslatableString::untranslatable("Triangle right double whole (stem down; 4-shape fa; 7-shape fa)"),
    muse::TranslatableString::untranslatable("Triangle right white (stem down; 4-shape fa; 7-shape fa)"),
    muse::TranslatableString::untranslatable("Triangle-round black (Aikin 7-shape ti)"),
    muse::TranslatableString::untranslatable("Triangle-round white (Aikin 7-shape ti)"),
    muse::TranslatableString::untranslatable("Triangle-round left black (Funk 7-shape ti)"),
    muse::TranslatableString::untranslatable("Triangle-round left double whole (Funk 7-shape ti)"),
    muse::TranslatableString::untranslatable("Triangle-round left white (Funk 7-shape ti)"),
    muse::TranslatableString::untranslatable("Triangle-round white (Aikin 7-shape ti)"),
    muse::TranslatableString::untranslatable("Triangle up black (Aikin 7-shape do)"),
    muse::TranslatableString::untranslatable("Triangle up double whole (Aikin 7-shape do)"),
    muse::TranslatableString::untranslatable("Triangle up white (Aikin 7-shape do)"),
    muse::TranslatableString::untranslatable("Si (black note)"),
    muse::TranslatableString::untranslatable("Si (half note)"),
    muse::TranslatableString::untranslatable("Si (whole note)"),
    muse::TranslatableString::untranslatable("So (black note)"),
    muse::TranslatableString::untranslatable("So (half note)"),
    muse::TranslatableString::untranslatable("So (whole note)"),
    muse::TranslatableString::untranslatable("Te (black note)"),
    muse::TranslatableString::untranslatable("Te (half note)"),
    muse::TranslatableString::untranslatable("Te (whole note)"),
    muse::TranslatableString::untranslatable("Ti (black note)"),
    muse::TranslatableString::untranslatable("Ti (half note)"),
    muse::TranslatableString::untranslatable("Ti (whole note)"),
    muse::TranslatableString::untranslatable("Whole note (semibreve)"),
    muse::TranslatableString::untranslatable("Black notehead"),
    muse::TranslatableString::untranslatable("Circle slash notehead"),
    muse::TranslatableString::untranslatable("Circle X notehead"),
    muse::TranslatableString::untranslatable("Circle X double whole"),
    muse::TranslatableString::untranslatable("Circle X half"),
    muse::TranslatableString::untranslatable("Circle X whole"),
    muse::TranslatableString::untranslatable("Circled black notehead"),
    muse::TranslatableString::untranslatable("Black notehead in large circle"),
    muse::TranslatableString::untranslatable("Circled double whole notehead"),
    muse::TranslatableString::untranslatable("Double whole notehead in large circle"),
    muse::TranslatableString::untranslatable("Circled half notehead"),
    muse::TranslatableString::untranslatable("Half notehead in large circle"),
    muse::TranslatableString::untranslatable("Circled whole notehead"),
    muse::TranslatableString::untranslatable("Whole notehead in large circle"),
    muse::TranslatableString::untranslatable("Cross notehead in large circle"),
    muse::TranslatableString::untranslatable("Double whole note cluster, 2nd"),
    muse::TranslatableString::untranslatable("Double whole note cluster, 3rd"),
    muse::TranslatableString::untranslatable("Combining double whole note cluster, bottom"),
    muse::TranslatableString::untranslatable("Combining double whole note cluster, middle"),
    muse::TranslatableString::untranslatable("Combining double whole note cluster, top"),
    muse::TranslatableString::untranslatable("Half note cluster, 2nd"),
    muse::TranslatableString::untranslatable("Half note cluster, 3rd"),
    muse::TranslatableString::untranslatable("Combining half note cluster, bottom"),
    muse::TranslatableString::untranslatable("Combining half note cluster, middle"),
    muse::TranslatableString::untranslatable("Combining half note cluster, top"),
    muse::TranslatableString::untranslatable("Quarter note cluster, 2nd"),
    muse::TranslatableString::untranslatable("Quarter note cluster, 3rd"),
    muse::TranslatableString::untranslatable("Combining quarter note cluster, bottom"),
    muse::TranslatableString::untranslatable("Combining quarter note cluster, middle"),
    muse::TranslatableString::untranslatable("Combining quarter note cluster, top"),
    muse::TranslatableString::untranslatable("Cluster notehead black (round)"),
    muse::TranslatableString::untranslatable("Cluster notehead white (round)"),
    muse::TranslatableString::untranslatable("Cluster notehead black (square)"),
    muse::TranslatableString::untranslatable("Cluster notehead white (square)"),
    muse::TranslatableString::untranslatable("Whole note cluster, 2nd"),
    muse::TranslatableString::untranslatable("Whole note cluster, 3rd"),
    muse::TranslatableString::untranslatable("Combining whole note cluster, bottom"),
    muse::TranslatableString::untranslatable("Combining whole note cluster, middle"),
    muse::TranslatableString::untranslatable("Combining whole note cluster, top"),
    muse::TranslatableString::untranslatable("4/11 note (eleventh note series, Cowell)"),
    muse::TranslatableString::untranslatable("8/11 note (eleventh note series, Cowell)"),
    muse::TranslatableString::untranslatable("2/11 note (eleventh note series, Cowell)"),
    muse::TranslatableString::untranslatable("2/15 note (fifteenth note series, Cowell)"),
    muse::TranslatableString::untranslatable("4/15 note (fifteenth note series, Cowell)"),
    muse::TranslatableString::untranslatable("8/15 note (fifteenth note series, Cowell)"),
    muse::TranslatableString::untranslatable("1/5 note (fifth note series, Cowell)"),
    muse::TranslatableString::untranslatable("2/5 note (fifth note series, Cowell)"),
    muse::TranslatableString::untranslatable("4/5 note (fifth note series, Cowell)"),
    muse::TranslatableString::untranslatable("2/9 note (ninth note series, Cowell)"),
    muse::TranslatableString::untranslatable("4/9 note (ninth note series, Cowell)"),
    muse::TranslatableString::untranslatable("8/9 note (ninth note series, Cowell)"),
    muse::TranslatableString::untranslatable("1/7 note (seventh note series, Cowell)"),
    muse::TranslatableString::untranslatable("2/7 note (seventh note series, Cowell)"),
    muse::TranslatableString::untranslatable("4/7 note (seventh note series, Cowell)"),
    muse::TranslatableString::untranslatable("1/6 note (third note series, Cowell)"),
    muse::TranslatableString::untranslatable("1/3 note (third note series, Cowell)"),
    muse::TranslatableString::untranslatable("2/3 note (third note series, Cowell)"),
    muse::TranslatableString::untranslatable("2/13 note (thirteenth note series, Cowell)"),
    muse::TranslatableString::untranslatable("4/13 note (thirteenth note series, Cowell)"),
    muse::TranslatableString::untranslatable("8/13 note (thirteenth note series, Cowell)"),
    muse::TranslatableString::untranslatable("Diamond black notehead"),
    muse::TranslatableString::untranslatable("Diamond black notehead (old)"),
    muse::TranslatableString::untranslatable("Diamond black notehead (wide)"),
    muse::TranslatableString::untranslatable("Black diamond cluster, 2nd"),
    muse::TranslatableString::untranslatable("Black diamond cluster, 3rd"),
    muse::TranslatableString::untranslatable("Combining black diamond cluster, bottom"),
    muse::TranslatableString::untranslatable("Combining black diamond cluster, middle"),
    muse::TranslatableString::untranslatable("Combining black diamond cluster, top"),
    muse::TranslatableString::untranslatable("White diamond cluster, 2nd"),
    muse::TranslatableString::untranslatable("White diamond cluster, 3rd"),
    muse::TranslatableString::untranslatable("Combining white diamond cluster, bottom"),
    muse::TranslatableString::untranslatable("Combining white diamond cluster, middle"),
    muse::TranslatableString::untranslatable("Combining white diamond cluster, top"),
    muse::TranslatableString::untranslatable("Diamond double whole notehead"),
    muse::TranslatableString::untranslatable("Diamond double whole notehead (old)"),
    muse::TranslatableString::untranslatable("Diamond half notehead"),
    muse::TranslatableString::untranslatable("Half-filled diamond notehead"),
    muse::TranslatableString::untranslatable("Diamond half notehead (old)"),
    muse::TranslatableString::untranslatable("Diamond half notehead (wide)"),
    muse::TranslatableString::untranslatable("Open diamond notehead"),
    muse::TranslatableString::untranslatable("Diamond white notehead"),
    muse::TranslatableString::untranslatable("Diamond white notehead (wide)"),
    muse::TranslatableString::untranslatable("Diamond whole notehead"),
    muse::TranslatableString::untranslatable("Diamond whole notehead (old)"),
    muse::TranslatableString::untranslatable("Double whole (breve) notehead"),
    muse::TranslatableString::untranslatable("Double whole (breve) notehead (square)"),
    muse::TranslatableString::untranslatable("Double whole notehead with X"),
    muse::TranslatableString::untranslatable("Half (minim) notehead"),
    muse::TranslatableString::untranslatable("Filled half (minim) notehead"),
    muse::TranslatableString::untranslatable("Half notehead with X"),
    muse::TranslatableString::untranslatable("Heavy X notehead"),
    muse::TranslatableString::untranslatable("Heavy X with hat notehead"),
    muse::TranslatableString::untranslatable("Large arrow down (lowest pitch) black notehead"),
    muse::TranslatableString::untranslatable("Large arrow down (lowest pitch) double whole notehead"),
    muse::TranslatableString::untranslatable("Large arrow down (lowest pitch) half notehead"),
    muse::TranslatableString::untranslatable("Large arrow down (lowest pitch) whole notehead"),
    muse::TranslatableString::untranslatable("Large arrow up (highest pitch) black notehead"),
    muse::TranslatableString::untranslatable("Large arrow up (highest pitch) double whole notehead"),
    muse::TranslatableString::untranslatable("Large arrow up (highest pitch) half notehead"),
    muse::TranslatableString::untranslatable("Large arrow up (highest pitch) whole notehead"),
    muse::TranslatableString::untranslatable("Moon notehead black"),
    muse::TranslatableString::untranslatable("Moon notehead white"),
    muse::TranslatableString::untranslatable("Sine notehead (Nancarrow)"),
    muse::TranslatableString::untranslatable("Null notehead"),
    muse::TranslatableString::untranslatable("Parenthesis notehead"),
    muse::TranslatableString::untranslatable("Opening parenthesis"),
    muse::TranslatableString::untranslatable("Closing parenthesis"),
    muse::TranslatableString::untranslatable("Plus notehead black"),
    muse::TranslatableString::untranslatable("Plus notehead double whole"),
    muse::TranslatableString::untranslatable("Plus notehead half"),
    muse::TranslatableString::untranslatable("Plus notehead whole"),
    muse::TranslatableString::untranslatable("Combining black rectangular cluster, bottom"),
    muse::TranslatableString::untranslatable("Combining black rectangular cluster, middle"),
    muse::TranslatableString::untranslatable("Combining black rectangular cluster, top"),
    muse::TranslatableString::untranslatable("Combining white rectangular cluster, bottom"),
    muse::TranslatableString::untranslatable("Combining white rectangular cluster, middle"),
    muse::TranslatableString::untranslatable("Combining white rectangular cluster, top"),
    muse::TranslatableString::untranslatable("Round black notehead"),
    muse::TranslatableString::untranslatable("Round black notehead, double slashed"),
    muse::TranslatableString::untranslatable("Large round black notehead"),
    muse::TranslatableString::untranslatable("Round black notehead, slashed"),
    muse::TranslatableString::untranslatable("Large round black notehead, slashed"),
    muse::TranslatableString::untranslatable("Round white notehead"),
    muse::TranslatableString::untranslatable("Round white notehead, double slashed"),
    muse::TranslatableString::untranslatable("Large round white notehead"),
    muse::TranslatableString::untranslatable("Round white notehead, slashed"),
    muse::TranslatableString::untranslatable("Large round white notehead, slashed"),
    muse::TranslatableString::untranslatable("Round white notehead with dot"),
    muse::TranslatableString::untranslatable("Large round white notehead with dot"),
    muse::TranslatableString::untranslatable("Large white diamond"),
    muse::TranslatableString::untranslatable("Slash with horizontal ends"),
    muse::TranslatableString::untranslatable("Muted slash with horizontal ends"),
    muse::TranslatableString::untranslatable("Slash with vertical ends"),
    muse::TranslatableString::untranslatable("Muted slash with vertical ends"),
    muse::TranslatableString::untranslatable("Small slash with vertical ends"),
    muse::TranslatableString::untranslatable("White slash double whole"),
    muse::TranslatableString::untranslatable("White slash half"),
    muse::TranslatableString::untranslatable("Muted white slash"),
    muse::TranslatableString::untranslatable("White slash whole"),
    muse::TranslatableString::untranslatable("Large X notehead"),
    muse::TranslatableString::untranslatable("Slashed black notehead (bottom left to top right)"),
    muse::TranslatableString::untranslatable("Slashed black notehead (top left to bottom right)"),
    muse::TranslatableString::untranslatable("Slashed double whole notehead (bottom left to top right)"),
    muse::TranslatableString::untranslatable("Slashed double whole notehead (top left to bottom right)"),
    muse::TranslatableString::untranslatable("Slashed half notehead (bottom left to top right)"),
    muse::TranslatableString::untranslatable("Slashed half notehead (top left to bottom right)"),
    muse::TranslatableString::untranslatable("Slashed whole notehead (bottom left to top right)"),
    muse::TranslatableString::untranslatable("Slashed whole notehead (top left to bottom right)"),
    muse::TranslatableString::untranslatable("Square notehead black"),
    muse::TranslatableString::untranslatable("Large square black notehead"),
    muse::TranslatableString::untranslatable("Large square white notehead"),
    muse::TranslatableString::untranslatable("Square notehead white"),
    muse::TranslatableString::untranslatable("Triangle notehead down black"),
    muse::TranslatableString::untranslatable("Triangle notehead down double whole"),
    muse::TranslatableString::untranslatable("Triangle notehead down half"),
    muse::TranslatableString::untranslatable("Triangle notehead down white"),
    muse::TranslatableString::untranslatable("Triangle notehead down whole"),
    muse::TranslatableString::untranslatable("Triangle notehead left black"),
    muse::TranslatableString::untranslatable("Triangle notehead left white"),
    muse::TranslatableString::untranslatable("Triangle notehead right black"),
    muse::TranslatableString::untranslatable("Triangle notehead right white"),
    muse::TranslatableString::untranslatable("Triangle-round notehead down black"),
    muse::TranslatableString::untranslatable("Triangle-round notehead down white"),
    muse::TranslatableString::untranslatable("Triangle notehead up black"),
    muse::TranslatableString::untranslatable("Triangle notehead up double whole"),
    muse::TranslatableString::untranslatable("Triangle notehead up half"),
    muse::TranslatableString::untranslatable("Triangle notehead up right black"),
    muse::TranslatableString::untranslatable("Triangle notehead up right white"),
    muse::TranslatableString::untranslatable("Triangle notehead up white"),
    muse::TranslatableString::untranslatable("Triangle notehead up whole"),
    muse::TranslatableString::untranslatable("Void notehead with X"),
    muse::TranslatableString::untranslatable("Whole (semibreve) notehead"),
    muse::TranslatableString::untranslatable("Filled whole (semibreve) notehead"),
    muse::TranslatableString::untranslatable("Whole notehead with X"),
    muse::TranslatableString::untranslatable("X notehead black"),
    muse::TranslatableString::untranslatable("X notehead double whole"),
    muse::TranslatableString::untranslatable("X notehead half"),
    muse::TranslatableString::untranslatable("Ornate X notehead"),
    muse::TranslatableString::untranslatable("Ornate X notehead in ellipse"),
    muse::TranslatableString::untranslatable("X notehead whole"),
    muse::TranslatableString::untranslatable("a (baseline)"),
    muse::TranslatableString::untranslatable("b (baseline)"),
    muse::TranslatableString::untranslatable("m (baseline)"),
    muse::TranslatableString::untranslatable("v (baseline)"),
    muse::TranslatableString::untranslatable("Bassa"),
    muse::TranslatableString::untranslatable("Loco"),
    muse::TranslatableString::untranslatable("Left parenthesis for octave signs"),
    muse::TranslatableString::untranslatable("Right parenthesis for octave signs"),
    muse::TranslatableString::untranslatable("a (superscript)"),
    muse::TranslatableString::untranslatable("b (superscript)"),
    muse::TranslatableString::untranslatable("m (superscript)"),
    muse::TranslatableString::untranslatable("v (superscript)"),
    muse::TranslatableString::untranslatable("One-handed roll (Stevens)"),
    muse::TranslatableString::untranslatable("Two Fusae"),
    muse::TranslatableString::untranslatable("Two Minimae"),
    muse::TranslatableString::untranslatable("Combining double octave line above"),
    muse::TranslatableString::untranslatable("Two Semifusae"),
    muse::TranslatableString::untranslatable("Two Semiminimae"),
    muse::TranslatableString::untranslatable("Three Fusae"),
    muse::TranslatableString::untranslatable("Three Minimae"),
    muse::TranslatableString::untranslatable("Three Semifusae"),
    muse::TranslatableString::untranslatable("Three Semiminimae"),
    muse::TranslatableString::untranslatable("Four Fusae"),
    muse::TranslatableString::untranslatable("Four Minimae"),
    muse::TranslatableString::untranslatable("Four Semifusae"),
    muse::TranslatableString::untranslatable("Four Semiminimae"),
    muse::TranslatableString::untranslatable("Five Fusae"),
    muse::TranslatableString::untranslatable("Five Minimae"),
    muse::TranslatableString::untranslatable("Five Semifusae"),
    muse::TranslatableString::untranslatable("Five Semiminimae"),
    muse::TranslatableString::untranslatable("Six Fusae"),
    muse::TranslatableString::untranslatable("Six Minimae"),
    muse::TranslatableString::untranslatable("Six Semifusae"),
    muse::TranslatableString::untranslatable("Six Semiminimae"),
    muse::TranslatableString::untranslatable("German organ tablature small A"),
    muse::TranslatableString::untranslatable("German organ tablature great A"),
    muse::TranslatableString::untranslatable("Rhythm Dot"),
    muse::TranslatableString::untranslatable("German organ tablature small B"),
    muse::TranslatableString::untranslatable("German organ tablature great B"),
    muse::TranslatableString::untranslatable("Brevis (Binary) Buxheimer Orgelbuch"),
    muse::TranslatableString::untranslatable("Brevis (Ternary) Buxheimer Orgelbuch"),
    muse::TranslatableString::untranslatable("Minima Rest Buxheimer Orgelbuch"),
    muse::TranslatableString::untranslatable("Semibrevis Buxheimer Orgelbuch"),
    muse::TranslatableString::untranslatable("Semibrevis Rest Buxheimer Orgelbuch"),
    muse::TranslatableString::untranslatable("German organ tablature small C"),
    muse::TranslatableString::untranslatable("German organ tablature great C"),
    muse::TranslatableString::untranslatable("German organ tablature small Cis"),
    muse::TranslatableString::untranslatable("German organ tablature great Cis"),
    muse::TranslatableString::untranslatable("German organ tablature small D"),
    muse::TranslatableString::untranslatable("German organ tablature great D"),
    muse::TranslatableString::untranslatable("German organ tablature small Dis"),
    muse::TranslatableString::untranslatable("German organ tablature great Dis"),
    muse::TranslatableString::untranslatable("German organ tablature small E"),
    muse::TranslatableString::untranslatable("German organ tablature great E"),
    muse::TranslatableString::untranslatable("German organ tablature small F"),
    muse::TranslatableString::untranslatable("German organ tablature great F"),
    muse::TranslatableString::untranslatable("German organ tablature small Fis"),
    muse::TranslatableString::untranslatable("German organ tablature great Fis"),
    muse::TranslatableString::untranslatable("Fusa"),
    muse::TranslatableString::untranslatable("Fusa Rest"),
    muse::TranslatableString::untranslatable("German organ tablature small G"),
    muse::TranslatableString::untranslatable("German organ tablature great G"),
    muse::TranslatableString::untranslatable("German organ tablature small Gis"),
    muse::TranslatableString::untranslatable("German organ tablature great Gis"),
    muse::TranslatableString::untranslatable("German organ tablature small H"),
    muse::TranslatableString::untranslatable("German organ tablature great H"),
    muse::TranslatableString::untranslatable("Minima"),
    muse::TranslatableString::untranslatable("Minima Rest"),
    muse::TranslatableString::untranslatable("Combining single octave line below"),
    muse::TranslatableString::untranslatable("Combining single octave line above"),
    muse::TranslatableString::untranslatable("Semibrevis"),
    muse::TranslatableString::untranslatable("Semibrevis Rest"),
    muse::TranslatableString::untranslatable("Semifusa"),
    muse::TranslatableString::untranslatable("Semifusa Rest"),
    muse::TranslatableString::untranslatable("Semiminima"),
    muse::TranslatableString::untranslatable("Semiminima Rest"),
    muse::TranslatableString::untranslatable("Tie"),
    muse::TranslatableString::untranslatable("Ornament bottom left concave stroke"),
    muse::TranslatableString::untranslatable("Ornament bottom left concave stroke, large"),
    muse::TranslatableString::untranslatable("Ornament bottom left convex stroke"),
    muse::TranslatableString::untranslatable("Ornament bottom right concave stroke"),
    muse::TranslatableString::untranslatable("Ornament bottom right convex stroke"),
    muse::TranslatableString::untranslatable("Comma"),
    muse::TranslatableString::untranslatable("Double oblique straight lines NW-SE"),
    muse::TranslatableString::untranslatable("Double oblique straight lines SW-NE"),
    muse::TranslatableString::untranslatable("Curve below"),
    muse::TranslatableString("engraving/sym", "Haydn ornament"),
    muse::TranslatableString::untranslatable("Ornament high left concave stroke"),
    muse::TranslatableString::untranslatable("Ornament high left convex stroke"),
    muse::TranslatableString::untranslatable("Ornament high right concave stroke"),
    muse::TranslatableString::untranslatable("Ornament high right convex stroke"),
    muse::TranslatableString::untranslatable("Hook after note"),
    muse::TranslatableString::untranslatable("Hook before note"),
    muse::TranslatableString::untranslatable("Left-facing half circle"),
    muse::TranslatableString::untranslatable("Left-facing hook"),
    muse::TranslatableString::untranslatable("Ornament left +"),
    muse::TranslatableString::untranslatable("Ornament left shake t"),
    muse::TranslatableString::untranslatable("Ornament left vertical stroke"),
    muse::TranslatableString::untranslatable("Ornament left vertical stroke with cross (+)"),
    muse::TranslatableString::untranslatable("Ornament low left concave stroke"),
    muse::TranslatableString::untranslatable("Ornament low left convex stroke"),
    muse::TranslatableString::untranslatable("Ornament low right concave stroke"),
    muse::TranslatableString::untranslatable("Ornament low right convex stroke"),
    muse::TranslatableString::untranslatable("Ornament middle vertical stroke"),
    muse::TranslatableString("engraving/sym", "Mordent"),
    muse::TranslatableString::untranslatable("Oblique straight line NW-SE"),
    muse::TranslatableString::untranslatable("Oblique straight line SW-NE"),
    muse::TranslatableString::untranslatable("Oblique straight line tilted NW-SE"),
    muse::TranslatableString::untranslatable("Oblique straight line tilted SW-NE"),
    muse::TranslatableString::untranslatable("Oriscus"),
    muse::TranslatableString("engraving/sym", "Pincé (Couperin)"),
    muse::TranslatableString::untranslatable("Port de voix"),
    muse::TranslatableString::untranslatable("Supported appoggiatura trill"),
    muse::TranslatableString::untranslatable("Supported appoggiatura trill with two-note suffix"),
    muse::TranslatableString::untranslatable("Cadence"),
    muse::TranslatableString::untranslatable("Cadence with upper prefix"),
    muse::TranslatableString::untranslatable("Cadence with upper prefix and turn"),
    muse::TranslatableString::untranslatable("Cadence with turn"),
    muse::TranslatableString::untranslatable("Descending slide"),
    muse::TranslatableString::untranslatable("Double cadence with lower prefix"),
    muse::TranslatableString::untranslatable("Double cadence with upper prefix"),
    muse::TranslatableString::untranslatable("Double cadence with upper prefix and turn"),
    muse::TranslatableString::untranslatable("Inverted mordent with upper prefix"),
    muse::TranslatableString::untranslatable("Mordent with release"),
    muse::TranslatableString("engraving/sym", "Mordent with upper prefix"),
    muse::TranslatableString::untranslatable("Pre-beat port de voix followed by multiple mordent (Dandrieu)"),
    muse::TranslatableString("engraving/sym", "Slide"),
    muse::TranslatableString::untranslatable("Slide-trill with two-note suffix (J.S. Bach)"),
    muse::TranslatableString::untranslatable("Slide-trill (D'Anglebert)"),
    muse::TranslatableString::untranslatable("Slide-trill with one-note suffix (Marpurg)"),
    muse::TranslatableString::untranslatable("Slide-trill (Muffat)"),
    muse::TranslatableString::untranslatable("Slide-trill with two-note suffix (Muffat)"),
    muse::TranslatableString::untranslatable("Trill with lower suffix"),
    muse::TranslatableString::untranslatable("Trill with two-note suffix (Dandrieu)"),
    muse::TranslatableString::untranslatable("Trill with mordent"),
    muse::TranslatableString::untranslatable("Turn-trill with two-note suffix (J.S. Bach)"),
    muse::TranslatableString::untranslatable("Turn-trill (D'Anglebert)"),
    muse::TranslatableString::untranslatable("Quilisma"),
    muse::TranslatableString::untranslatable("Right-facing half circle"),
    muse::TranslatableString::untranslatable("Right-facing hook"),
    muse::TranslatableString::untranslatable("Ornament right vertical stroke"),
    muse::TranslatableString::untranslatable("Schleifer (long mordent)"),
    muse::TranslatableString("engraving/sym", "Shake"),
    muse::TranslatableString("engraving/sym", "Shake (Muffat)"),
    muse::TranslatableString::untranslatable("Short oblique straight line NW-SE"),
    muse::TranslatableString::untranslatable("Short oblique straight line SW-NE"),
    muse::TranslatableString("engraving/sym", "Short trill"),
    muse::TranslatableString::untranslatable("Ornament top left concave stroke"),
    muse::TranslatableString::untranslatable("Ornament top left convex stroke"),
    muse::TranslatableString::untranslatable("Ornament top right concave stroke"),
    muse::TranslatableString::untranslatable("Ornament top right convex stroke"),
    muse::TranslatableString("engraving/sym", "Tremblement"),
    muse::TranslatableString("engraving/sym", "Tremblement appuyé (Couperin)"),
    muse::TranslatableString("engraving/sym", "Trill"),
    muse::TranslatableString("engraving/sym", "Turn"),
    muse::TranslatableString("engraving/sym", "Inverted turn"),
    muse::TranslatableString("engraving/sym", "Turn with slash"),
    muse::TranslatableString("engraving/sym", "Turn up"),
    muse::TranslatableString("engraving/sym", "Inverted turn up"),
    muse::TranslatableString::untranslatable("Curve above"),
    muse::TranslatableString::untranslatable("Vertical line"),
    muse::TranslatableString::untranslatable("Ornament zig-zag line without right-hand end"),
    muse::TranslatableString::untranslatable("Ornament zig-zag line with right-hand end"),
    muse::TranslatableString::untranslatable("Ottava"),
    muse::TranslatableString::untranslatable("Ottava alta"),
    muse::TranslatableString::untranslatable("Ottava bassa"),
    muse::TranslatableString::untranslatable("Ottava bassa (ba)"),
    muse::TranslatableString::untranslatable("Ottava bassa (8vb)"),
    muse::TranslatableString::untranslatable("Penderecki unmeasured tremolo"),
    muse::TranslatableString::untranslatable("Agogo"),
    muse::TranslatableString::untranslatable("Almglocken"),
    muse::TranslatableString::untranslatable("Anvil"),
    muse::TranslatableString::untranslatable("Bamboo tube chimes"),
    muse::TranslatableString::untranslatable("Bamboo scraper"),
    muse::TranslatableString::untranslatable("Bass drum"),
    muse::TranslatableString::untranslatable("Bass drum on side"),
    muse::TranslatableString::untranslatable("Bow"),
    muse::TranslatableString::untranslatable("Box for percussion beater"),
    muse::TranslatableString::untranslatable("Brass mallets down"),
    muse::TranslatableString::untranslatable("Brass mallets left"),
    muse::TranslatableString::untranslatable("Brass mallets right"),
    muse::TranslatableString::untranslatable("Brass mallets up"),
    muse::TranslatableString::untranslatable("Combining dashed circle for round beaters (plated)"),
    muse::TranslatableString::untranslatable("Combining parentheses for round beaters (padded)"),
    muse::TranslatableString::untranslatable("Double bass drum stick down"),
    muse::TranslatableString::untranslatable("Double bass drum stick up"),
    muse::TranslatableString::untranslatable("Finger"),
    muse::TranslatableString::untranslatable("Fingernails"),
    muse::TranslatableString::untranslatable("Fist"),
    muse::TranslatableString::untranslatable("Guiro scraper"),
    muse::TranslatableString::untranslatable("Hammer"),
    muse::TranslatableString::untranslatable("Metal hammer, down"),
    muse::TranslatableString::untranslatable("Metal hammer, up"),
    muse::TranslatableString::untranslatable("Plastic hammer, down"),
    muse::TranslatableString::untranslatable("Plastic hammer, up"),
    muse::TranslatableString::untranslatable("Wooden hammer, down"),
    muse::TranslatableString::untranslatable("Wooden hammer, up"),
    muse::TranslatableString::untranslatable("Hand"),
    muse::TranslatableString::untranslatable("Hard bass drum stick down"),
    muse::TranslatableString::untranslatable("Hard bass drum stick up"),
    muse::TranslatableString::untranslatable("Hard glockenspiel stick down"),
    muse::TranslatableString::untranslatable("Hard glockenspiel stick left"),
    muse::TranslatableString::untranslatable("Hard glockenspiel stick right"),
    muse::TranslatableString::untranslatable("Hard glockenspiel stick up"),
    muse::TranslatableString::untranslatable("Hard timpani stick down"),
    muse::TranslatableString::untranslatable("Hard timpani stick left"),
    muse::TranslatableString::untranslatable("Hard timpani stick right"),
    muse::TranslatableString::untranslatable("Hard timpani stick up"),
    muse::TranslatableString::untranslatable("Hard xylophone stick down"),
    muse::TranslatableString::untranslatable("Hard xylophone stick left"),
    muse::TranslatableString::untranslatable("Hard xylophone stick right"),
    muse::TranslatableString::untranslatable("Hard xylophone stick up"),
    muse::TranslatableString::untranslatable("Hard yarn beater down"),
    muse::TranslatableString::untranslatable("Hard yarn beater left"),
    muse::TranslatableString::untranslatable("Hard yarn beater right"),
    muse::TranslatableString::untranslatable("Hard yarn beater up"),
    muse::TranslatableString::untranslatable("Jazz sticks down"),
    muse::TranslatableString::untranslatable("Jazz sticks up"),
    muse::TranslatableString::untranslatable("Knitting needle"),
    muse::TranslatableString::untranslatable("Chime hammer up"),
    muse::TranslatableString::untranslatable("Chime hammer down"),
    muse::TranslatableString::untranslatable("Medium bass drum stick down"),
    muse::TranslatableString::untranslatable("Medium bass drum stick up"),
    muse::TranslatableString::untranslatable("Medium timpani stick down"),
    muse::TranslatableString::untranslatable("Medium timpani stick left"),
    muse::TranslatableString::untranslatable("Medium timpani stick right"),
    muse::TranslatableString::untranslatable("Medium timpani stick up"),
    muse::TranslatableString::untranslatable("Medium xylophone stick down"),
    muse::TranslatableString::untranslatable("Medium xylophone stick left"),
    muse::TranslatableString::untranslatable("Medium xylophone stick right"),
    muse::TranslatableString::untranslatable("Medium xylophone stick up"),
    muse::TranslatableString::untranslatable("Medium yarn beater down"),
    muse::TranslatableString::untranslatable("Medium yarn beater left"),
    muse::TranslatableString::untranslatable("Medium yarn beater right"),
    muse::TranslatableString::untranslatable("Medium yarn beater up"),
    muse::TranslatableString::untranslatable("Metal bass drum stick down"),
    muse::TranslatableString::untranslatable("Metal bass drum stick up"),
    muse::TranslatableString::untranslatable("Metal beater down"),
    muse::TranslatableString::untranslatable("Metal hammer"),
    muse::TranslatableString::untranslatable("Metal beater, left"),
    muse::TranslatableString::untranslatable("Metal beater, right"),
    muse::TranslatableString::untranslatable("Metal beater, up"),
    muse::TranslatableString::untranslatable("Snare sticks down"),
    muse::TranslatableString::untranslatable("Snare sticks up"),
    muse::TranslatableString::untranslatable("Soft bass drum stick down"),
    muse::TranslatableString::untranslatable("Soft bass drum stick up"),
    muse::TranslatableString::untranslatable("Soft glockenspiel stick down"),
    muse::TranslatableString::untranslatable("Soft glockenspiel stick left"),
    muse::TranslatableString::untranslatable("Soft glockenspiel stick right"),
    muse::TranslatableString::untranslatable("Soft glockenspiel stick up"),
    muse::TranslatableString::untranslatable("Soft timpani stick down"),
    muse::TranslatableString::untranslatable("Soft timpani stick left"),
    muse::TranslatableString::untranslatable("Soft timpani stick right"),
    muse::TranslatableString::untranslatable("Soft timpani stick up"),
    muse::TranslatableString::untranslatable("Soft xylophone beaters"),
    muse::TranslatableString::untranslatable("Soft xylophone stick down"),
    muse::TranslatableString::untranslatable("Soft xylophone stick left"),
    muse::TranslatableString::untranslatable("Soft xylophone stick right"),
    muse::TranslatableString::untranslatable("Soft xylophone stick up"),
    muse::TranslatableString::untranslatable("Soft yarn beater down"),
    muse::TranslatableString::untranslatable("Soft yarn beater left"),
    muse::TranslatableString::untranslatable("Soft yarn beater right"),
    muse::TranslatableString::untranslatable("Soft yarn beater up"),
    muse::TranslatableString::untranslatable("Spoon-shaped wooden mallet"),
    muse::TranslatableString::untranslatable("Superball beater down"),
    muse::TranslatableString::untranslatable("Superball beater left"),
    muse::TranslatableString::untranslatable("Superball beater right"),
    muse::TranslatableString::untranslatable("Superball beater up"),
    muse::TranslatableString::untranslatable("Triangle beater down"),
    muse::TranslatableString::untranslatable("Triangle beater plain"),
    muse::TranslatableString::untranslatable("Triangle beater up"),
    muse::TranslatableString::untranslatable("Wire brushes down"),
    muse::TranslatableString::untranslatable("Wire brushes up"),
    muse::TranslatableString::untranslatable("Wood timpani stick down"),
    muse::TranslatableString::untranslatable("Wood timpani stick left"),
    muse::TranslatableString::untranslatable("Wood timpani stick right"),
    muse::TranslatableString::untranslatable("Wood timpani stick up"),
    muse::TranslatableString::untranslatable("Wood xylophone stick down"),
    muse::TranslatableString::untranslatable("Wood xylophone stick left"),
    muse::TranslatableString::untranslatable("Wood xylophone stick right"),
    muse::TranslatableString::untranslatable("Wood xylophone stick up"),
    muse::TranslatableString::untranslatable("Bell"),
    muse::TranslatableString::untranslatable("Bell of cymbal"),
    muse::TranslatableString::untranslatable("Bell plate"),
    muse::TranslatableString::untranslatable("Bell tree"),
    muse::TranslatableString::untranslatable("Bird whistle"),
    muse::TranslatableString::untranslatable("Board clapper"),
    muse::TranslatableString::untranslatable("Bongos"),
    muse::TranslatableString::untranslatable("Brake drum"),
    muse::TranslatableString::untranslatable("Cabasa"),
    muse::TranslatableString::untranslatable("Cannon"),
    muse::TranslatableString::untranslatable("Car horn"),
    muse::TranslatableString::untranslatable("Castanets"),
    muse::TranslatableString::untranslatable("Castanets with handle"),
    muse::TranslatableString::untranslatable("Celesta"),
    muse::TranslatableString::untranslatable("Cencerro"),
    muse::TranslatableString::untranslatable("Center (Weinberg)"),
    muse::TranslatableString::untranslatable("Center (Ghent)"),
    muse::TranslatableString::untranslatable("Center (Caltabiano)"),
    muse::TranslatableString::untranslatable("Chain rattle"),
    muse::TranslatableString::untranslatable("Chimes"),
    muse::TranslatableString::untranslatable("Chinese cymbal"),
    muse::TranslatableString::untranslatable("Choke (Weinberg)"),
    muse::TranslatableString::untranslatable("Claves"),
    muse::TranslatableString::untranslatable("Coins"),
    muse::TranslatableString::untranslatable("Conga"),
    muse::TranslatableString::untranslatable("Cow bell"),
    muse::TranslatableString::untranslatable("Crash cymbals"),
    muse::TranslatableString::untranslatable("Crotales"),
    muse::TranslatableString::untranslatable("Combining crush for stem"),
    muse::TranslatableString::untranslatable("Cuica"),
    muse::TranslatableString::untranslatable("Cymbal tongs"),
    muse::TranslatableString::untranslatable("Damp"),
    muse::TranslatableString::untranslatable("Damp 2"),
    muse::TranslatableString::untranslatable("Damp 3"),
    muse::TranslatableString::untranslatable("Damp 4"),
    muse::TranslatableString::untranslatable("Combining X for stem (dead note)"),
    muse::TranslatableString::untranslatable("Drum stick"),
    muse::TranslatableString::untranslatable("Duck call"),
    muse::TranslatableString::untranslatable("Edge of cymbal"),
    muse::TranslatableString::untranslatable("Empty trapezoid"),
    muse::TranslatableString::untranslatable("Finger cymbals"),
    muse::TranslatableString::untranslatable("Flexatone"),
    muse::TranslatableString::untranslatable("Football rattle"),
    muse::TranslatableString::untranslatable("Glass harmonica"),
    muse::TranslatableString::untranslatable("Glass harp"),
    muse::TranslatableString::untranslatable("Glass plate chimes"),
    muse::TranslatableString::untranslatable("Glass tube chimes"),
    muse::TranslatableString::untranslatable("Glockenspiel"),
    muse::TranslatableString::untranslatable("Glockenspiel (Smith Brindle)"),
    muse::TranslatableString::untranslatable("Goblet drum (djembe, dumbek)"),
    muse::TranslatableString::untranslatable("Gong"),
    muse::TranslatableString::untranslatable("Gong with button (nipple)"),
    muse::TranslatableString::untranslatable("Guiro"),
    muse::TranslatableString::untranslatable("Hard gum beater, down"),
    muse::TranslatableString::untranslatable("Hard gum beater, left"),
    muse::TranslatableString::untranslatable("Hard gum beater, right"),
    muse::TranslatableString::untranslatable("Hard gum beater, up"),
    muse::TranslatableString::untranslatable("Medium gum beater, down"),
    muse::TranslatableString::untranslatable("Medium gum beater, left"),
    muse::TranslatableString::untranslatable("Medium gum beater, right"),
    muse::TranslatableString::untranslatable("Medium gum beater, up"),
    muse::TranslatableString::untranslatable("Soft gum beater, down"),
    muse::TranslatableString::untranslatable("Soft gum beater, left"),
    muse::TranslatableString::untranslatable("Soft gum beater, right"),
    muse::TranslatableString::untranslatable("Soft gum beater, up"),
    muse::TranslatableString::untranslatable("Half-open"),
    muse::TranslatableString("engraving/sym", "Half-open 2 (Weinberg)"),
    muse::TranslatableString::untranslatable("Handbell"),
    muse::TranslatableString::untranslatable("Hi-hat"),
    muse::TranslatableString::untranslatable("Hi-hat cymbals on stand"),
    muse::TranslatableString::untranslatable("Jaw harp"),
    muse::TranslatableString::untranslatable("Jingle bells"),
    muse::TranslatableString::untranslatable("Klaxon horn"),
    muse::TranslatableString::untranslatable("Right hand (Agostini)"),
    muse::TranslatableString::untranslatable("Lion's roar"),
    muse::TranslatableString::untranslatable("Lithophone"),
    muse::TranslatableString::untranslatable("Log drum"),
    muse::TranslatableString::untranslatable("Lotus flute"),
    muse::TranslatableString::untranslatable("Marimba"),
    muse::TranslatableString::untranslatable("Marimba (Smith Brindle)"),
    muse::TranslatableString::untranslatable("Maraca"),
    muse::TranslatableString::untranslatable("Maracas"),
    muse::TranslatableString::untranslatable("Megaphone"),
    muse::TranslatableString::untranslatable("Metal plate chimes"),
    muse::TranslatableString::untranslatable("Metal tube chimes"),
    muse::TranslatableString::untranslatable("Musical saw"),
    muse::TranslatableString::untranslatable("Normal position (Caltabiano)"),
    muse::TranslatableString::untranslatable("On rim"),
    muse::TranslatableString::untranslatable("Open"),
    muse::TranslatableString::untranslatable("Closed / rim shot"),
    muse::TranslatableString::untranslatable("Pistol shot"),
    muse::TranslatableString::untranslatable("Police whistle"),
    muse::TranslatableString::untranslatable("Quijada (jawbone)"),
    muse::TranslatableString::untranslatable("Rainstick"),
    muse::TranslatableString::untranslatable("Ratchet"),
    muse::TranslatableString::untranslatable("Reco-reco"),
    muse::TranslatableString::untranslatable("Left hand (Agostini)"),
    muse::TranslatableString::untranslatable("Rim or edge (Weinberg)"),
    muse::TranslatableString::untranslatable("Rim (Ghent)"),
    muse::TranslatableString::untranslatable("Rim (Caltabiano)"),
    muse::TranslatableString::untranslatable("Rim shot for stem"),
    muse::TranslatableString::untranslatable("Sandpaper blocks"),
    muse::TranslatableString::untranslatable("Scrape around rim (counter-clockwise)"),
    muse::TranslatableString::untranslatable("Scrape around rim (clockwise)"),
    muse::TranslatableString::untranslatable("Scrape from center to edge"),
    muse::TranslatableString::untranslatable("Scrape from edge to center"),
    muse::TranslatableString::untranslatable("Shell bells"),
    muse::TranslatableString::untranslatable("Shell chimes"),
    muse::TranslatableString::untranslatable("Siren"),
    muse::TranslatableString::untranslatable("Sistrum"),
    muse::TranslatableString::untranslatable("Sizzle cymbal"),
    muse::TranslatableString::untranslatable("Sleigh bell"),
    muse::TranslatableString::untranslatable("Slide brush on gong"),
    muse::TranslatableString::untranslatable("Slide whistle"),
    muse::TranslatableString::untranslatable("Slit drum"),
    muse::TranslatableString::untranslatable("Snare drum"),
    muse::TranslatableString::untranslatable("Military snare drum"),
    muse::TranslatableString::untranslatable("Snare drum, snares off"),
    muse::TranslatableString::untranslatable("Steel drums"),
    muse::TranslatableString::untranslatable("Stick shot"),
    muse::TranslatableString::untranslatable("Superball"),
    muse::TranslatableString::untranslatable("Suspended cymbal"),
    muse::TranslatableString::untranslatable("Combining swish for stem"),
    muse::TranslatableString::untranslatable("Indian tabla"),
    muse::TranslatableString::untranslatable("Tam-tam"),
    muse::TranslatableString::untranslatable("Tam-tam with beater (Smith Brindle)"),
    muse::TranslatableString::untranslatable("Tambourine"),
    muse::TranslatableString::untranslatable("Temple blocks"),
    muse::TranslatableString::untranslatable("Tenor drum"),
    muse::TranslatableString::untranslatable("Thundersheet"),
    muse::TranslatableString::untranslatable("Timbales"),
    muse::TranslatableString::untranslatable("Timpani"),
    muse::TranslatableString::untranslatable("Tom-tom"),
    muse::TranslatableString::untranslatable("Chinese tom-tom"),
    muse::TranslatableString::untranslatable("Indo-American tom tom"),
    muse::TranslatableString::untranslatable("Japanese tom-tom"),
    muse::TranslatableString::untranslatable("Triangle"),
    muse::TranslatableString::untranslatable("Tubaphone"),
    muse::TranslatableString::untranslatable("Tubular bells"),
    muse::TranslatableString::untranslatable("Combining turn left for stem"),
    muse::TranslatableString::untranslatable("Combining turn left or right for stem"),
    muse::TranslatableString::untranslatable("Combining turn right for stem"),
    muse::TranslatableString::untranslatable("Vibraphone"),
    muse::TranslatableString::untranslatable("Metallophone (vibraphone motor off)"),
    muse::TranslatableString::untranslatable("Vibraphone (Smith Brindle)"),
    muse::TranslatableString::untranslatable("Vibraslap"),
    muse::TranslatableString::untranslatable("Vietnamese hat cymbal"),
    muse::TranslatableString::untranslatable("Whip"),
    muse::TranslatableString::untranslatable("Wind chimes (glass)"),
    muse::TranslatableString::untranslatable("Wind machine"),
    muse::TranslatableString::untranslatable("Wind whistle (or mouth siren)"),
    muse::TranslatableString::untranslatable("Wood block"),
    muse::TranslatableString::untranslatable("Wound beater, hard core down"),
    muse::TranslatableString::untranslatable("Wound beater, hard core left"),
    muse::TranslatableString::untranslatable("Wound beater, hard core right"),
    muse::TranslatableString::untranslatable("Wound beater, hard core up"),
    muse::TranslatableString::untranslatable("Wound beater, soft core down"),
    muse::TranslatableString::untranslatable("Wound beater, soft core left"),
    muse::TranslatableString::untranslatable("Wound beater, soft core right"),
    muse::TranslatableString::untranslatable("Wound beater, soft core up"),
    muse::TranslatableString::untranslatable("Xylophone"),
    muse::TranslatableString::untranslatable("Bass xylophone"),
    muse::TranslatableString::untranslatable("Xylophone (Smith Brindle)"),
    muse::TranslatableString::untranslatable("Tenor xylophone"),
    muse::TranslatableString::untranslatable("Trough tenor xylophone"),
    muse::TranslatableString::untranslatable("Trough xylophone"),
    muse::TranslatableString::untranslatable("Buzz pizzicato"),
    muse::TranslatableString::untranslatable("Damp"),
    muse::TranslatableString::untranslatable("Damp all"),
    muse::TranslatableString::untranslatable("Damp for stem"),
    muse::TranslatableString::untranslatable("Fingernail flick"),
    muse::TranslatableString::untranslatable("Left-hand pizzicato"),
    muse::TranslatableString::untranslatable("Plectrum"),
    muse::TranslatableString("engraving/sym", "Snap pizzicato above"),
    muse::TranslatableString("engraving/sym", "Snap pizzicato below"),
    muse::TranslatableString::untranslatable("With fingernails"),
    muse::TranslatableString::untranslatable("Quindicesima"),
    muse::TranslatableString::untranslatable("Quindicesima alta"),
    muse::TranslatableString::untranslatable("Quindicesima bassa"),
    muse::TranslatableString::untranslatable("Quindicesima bassa (mb)"),
    muse::TranslatableString("engraving/sym", "Repeat last bar"),
    muse::TranslatableString("engraving/sym", "Repeat last two bars"),
    muse::TranslatableString("engraving/sym", "Repeat last four bars"),
    muse::TranslatableString::untranslatable("Repeat bar lower dot"),
    muse::TranslatableString::untranslatable("Repeat bar slash"),
    muse::TranslatableString::untranslatable("Repeat bar upper dot"),
    muse::TranslatableString::untranslatable("Single repeat dot"),
    muse::TranslatableString::untranslatable("Repeat dots"),
    muse::TranslatableString("engraving/sym", "Left (start) repeat sign"),
    muse::TranslatableString("engraving/sym", "Right (end) repeat sign"),
    muse::TranslatableString("engraving/sym", "Right and left repeat sign"),
    muse::TranslatableString("engraving/sym", "1024th rest"),
    muse::TranslatableString("engraving/sym", "128th (semihemidemisemiquaver) rest"),
    muse::TranslatableString("engraving/sym", "16th (semiquaver) rest"),
    muse::TranslatableString("engraving/sym", "256th rest"),
    muse::TranslatableString("engraving/sym", "32nd (demisemiquaver) rest"),
    muse::TranslatableString("engraving/sym", "512th rest"),
    muse::TranslatableString("engraving/sym", "64th (hemidemisemiquaver) rest"),
    muse::TranslatableString("engraving/sym", "Eighth (quaver) rest"),
    muse::TranslatableString("engraving/sym", "Double whole (breve) rest"),
    muse::TranslatableString("engraving/sym", "Double whole rest on leger lines"),
    muse::TranslatableString::untranslatable("Multiple measure rest"),
    muse::TranslatableString::untranslatable("H-bar, left half"),
    muse::TranslatableString::untranslatable("H-bar, middle"),
    muse::TranslatableString::untranslatable("H-bar, right half"),
    muse::TranslatableString("engraving/sym", "Half (minim) rest"),
    muse::TranslatableString("engraving/sym", "Half rest on leger line"),
    muse::TranslatableString("engraving/sym", "Longa rest"),
    muse::TranslatableString("engraving/sym", "Maxima rest"),
    muse::TranslatableString("engraving/sym", "Quarter (crotchet) rest"),
    muse::TranslatableString::untranslatable("Old-style quarter (crotchet) rest"),
    muse::TranslatableString::untranslatable("Z-style quarter (crotchet) rest"),
    muse::TranslatableString("engraving/sym", "Whole (semibreve) rest"),
    muse::TranslatableString("engraving/sym", "Whole rest on leger line"),
    muse::TranslatableString::untranslatable("Reversed brace"),
    muse::TranslatableString::untranslatable("Reversed bracket bottom"),
    muse::TranslatableString::untranslatable("Reversed bracket top"),
    muse::TranslatableString::untranslatable("Right repeat sign within bar"),
    muse::TranslatableString::untranslatable("Scale degree 1"),
    muse::TranslatableString::untranslatable("Scale degree 2"),
    muse::TranslatableString::untranslatable("Scale degree 3"),
    muse::TranslatableString::untranslatable("Scale degree 4"),
    muse::TranslatableString::untranslatable("Scale degree 5"),
    muse::TranslatableString::untranslatable("Scale degree 6"),
    muse::TranslatableString::untranslatable("Scale degree 7"),
    muse::TranslatableString::untranslatable("Scale degree 8"),
    muse::TranslatableString::untranslatable("Scale degree 9"),
    muse::TranslatableString::untranslatable("Schäffer clef"),
    muse::TranslatableString::untranslatable("Schäffer F clef to G clef change"),
    muse::TranslatableString::untranslatable("Schäffer G clef to F clef change"),
    muse::TranslatableString::untranslatable("Schäffer previous clef"),
    muse::TranslatableString("engraving/sym", "Segno"),
    muse::TranslatableString("engraving/sym", "Segno (serpent)"),
    muse::TranslatableString::untranslatable("Segno (serpent with vertical lines)"),
    muse::TranslatableString::untranslatable("Semi-pitched percussion clef 1"),
    muse::TranslatableString::untranslatable("Semi-pitched percussion clef 2"),
    muse::TranslatableString::untranslatable("Flat"),
    muse::TranslatableString::untranslatable("Flat (white)"),
    muse::TranslatableString::untranslatable("Double flat history sign"),
    muse::TranslatableString::untranslatable("Double sharp history sign"),
    muse::TranslatableString::untranslatable("Flat history sign"),
    muse::TranslatableString::untranslatable("Sharp history sign"),
    muse::TranslatableString::untranslatable("Natural (N)"),
    muse::TranslatableString::untranslatable("Sharp stem up"),
    muse::TranslatableString::untranslatable("Sharp stem down"),
    muse::TranslatableString::untranslatable("Sharp (white) stem up"),
    muse::TranslatableString::untranslatable("Sharp (white) stem down"),
    muse::TranslatableString::untranslatable("Split bar divider (bar spans a system break)"),
    muse::TranslatableString::untranslatable("1-line staff"),
    muse::TranslatableString::untranslatable("1-line staff (narrow)"),
    muse::TranslatableString::untranslatable("1-line staff (wide)"),
    muse::TranslatableString::untranslatable("2-line staff"),
    muse::TranslatableString::untranslatable("2-line staff (narrow)"),
    muse::TranslatableString::untranslatable("2-line staff (wide)"),
    muse::TranslatableString::untranslatable("3-line staff"),
    muse::TranslatableString::untranslatable("3-line staff (narrow)"),
    muse::TranslatableString::untranslatable("3-line staff (wide)"),
    muse::TranslatableString::untranslatable("4-line staff"),
    muse::TranslatableString::untranslatable("4-line staff (narrow)"),
    muse::TranslatableString::untranslatable("4-line staff (wide)"),
    muse::TranslatableString::untranslatable("5-line staff"),
    muse::TranslatableString::untranslatable("5-line staff (narrow)"),
    muse::TranslatableString::untranslatable("5-line staff (wide)"),
    muse::TranslatableString::untranslatable("6-line staff"),
    muse::TranslatableString::untranslatable("6-line staff (narrow)"),
    muse::TranslatableString::untranslatable("6-line staff (wide)"),
    muse::TranslatableString::untranslatable("Staff divide arrow down"),
    muse::TranslatableString::untranslatable("Staff divide arrow up"),
    muse::TranslatableString::untranslatable("Staff divide arrows"),
    muse::TranslatableString::untranslatable("Lower 1 staff position"),
    muse::TranslatableString::untranslatable("Lower 2 staff positions"),
    muse::TranslatableString::untranslatable("Lower 3 staff positions"),
    muse::TranslatableString::untranslatable("Lower 4 staff positions"),
    muse::TranslatableString::untranslatable("Lower 5 staff positions"),
    muse::TranslatableString::untranslatable("Lower 6 staff positions"),
    muse::TranslatableString::untranslatable("Lower 7 staff positions"),
    muse::TranslatableString::untranslatable("Lower 8 staff positions"),
    muse::TranslatableString::untranslatable("Raise 1 staff position"),
    muse::TranslatableString::untranslatable("Raise 2 staff positions"),
    muse::TranslatableString::untranslatable("Raise 3 staff positions"),
    muse::TranslatableString::untranslatable("Raise 4 staff positions"),
    muse::TranslatableString::untranslatable("Raise 5 staff positions"),
    muse::TranslatableString::untranslatable("Raise 6 staff positions"),
    muse::TranslatableString::untranslatable("Raise 7 staff positions"),
    muse::TranslatableString::untranslatable("Raise 8 staff positions"),
    muse::TranslatableString::untranslatable("Combining stem"),
    muse::TranslatableString::untranslatable("Combining bow on bridge stem"),
    muse::TranslatableString::untranslatable("Combining bow on tailpiece stem"),
    muse::TranslatableString::untranslatable("Combining buzz roll stem"),
    muse::TranslatableString::untranslatable("Combining damp stem"),
    muse::TranslatableString::untranslatable("Combining harp string noise stem"),
    muse::TranslatableString::untranslatable("Combining multiphonics (black) stem"),
    muse::TranslatableString::untranslatable("Combining multiphonics (black and white) stem"),
    muse::TranslatableString::untranslatable("Combining multiphonics (white) stem"),
    muse::TranslatableString::untranslatable("Combining Penderecki unmeasured tremolo stem"),
    muse::TranslatableString::untranslatable("Combining rim shot stem"),
    muse::TranslatableString::untranslatable("Combining sprechgesang stem"),
    muse::TranslatableString::untranslatable("Combining sul ponticello (bow behind bridge) stem"),
    muse::TranslatableString::untranslatable("Combining sussurando stem"),
    muse::TranslatableString::untranslatable("Combining swished stem"),
    muse::TranslatableString::untranslatable("Combining vibrato pulse accent (Saunders) stem"),
    muse::TranslatableString::untranslatable("Stockhausen irregular tremolo (\"Morsen\"), like Morse code)"),
    muse::TranslatableString::untranslatable("Bow behind bridge (sul ponticello)"),
    muse::TranslatableString::untranslatable("Bow behind bridge on four strings"),
    muse::TranslatableString::untranslatable("Bow behind bridge on one string"),
    muse::TranslatableString::untranslatable("Bow behind bridge on three strings"),
    muse::TranslatableString::untranslatable("Bow behind bridge on two strings"),
    muse::TranslatableString::untranslatable("Bow on top of bridge"),
    muse::TranslatableString::untranslatable("Bow on tailpiece"),
    muse::TranslatableString::untranslatable("Change bow direction, indeterminate"),
    muse::TranslatableString("engraving/sym", "Down bow"),
    muse::TranslatableString::untranslatable("Down bow, away from body"),
    muse::TranslatableString::untranslatable("Down bow, beyond bridge"),
    muse::TranslatableString::untranslatable("Down bow, towards body"),
    muse::TranslatableString::untranslatable("Turned down bow"),
    muse::TranslatableString::untranslatable("Fouetté"),
    muse::TranslatableString::untranslatable("Half-harmonic"),
    muse::TranslatableString("engraving/sym", "Harmonic"),
    muse::TranslatableString::untranslatable("Jeté (gettato) above"),
    muse::TranslatableString::untranslatable("Jeté (gettato) below"),
    muse::TranslatableString::untranslatable("Mute off"),
    muse::TranslatableString::untranslatable("Mute on"),
    muse::TranslatableString::untranslatable("Overpressure, down bow"),
    muse::TranslatableString::untranslatable("Overpressure, no bow direction"),
    muse::TranslatableString::untranslatable("Overpressure possibile, down bow"),
    muse::TranslatableString::untranslatable("Overpressure possibile, up bow"),
    muse::TranslatableString::untranslatable("Overpressure, up bow"),
    muse::TranslatableString::untranslatable("Scrape, circular clockwise"),
    muse::TranslatableString::untranslatable("Scrape, circular counter-clockwise"),
    muse::TranslatableString::untranslatable("Scrape, parallel inward"),
    muse::TranslatableString::untranslatable("Scrape, parallel outward"),
    muse::TranslatableString("engraving/sym", "Thumb position"),
    muse::TranslatableString::untranslatable("Turned thumb position"),
    muse::TranslatableString::untranslatable("Triple chop, inward"),
    muse::TranslatableString::untranslatable("Triple chop, outward"),
    muse::TranslatableString("engraving/sym", "Up bow"),
    muse::TranslatableString::untranslatable("Up bow, away from body"),
    muse::TranslatableString::untranslatable("Up bow, beyond bridge"),
    muse::TranslatableString::untranslatable("Up bow, towards body"),
    muse::TranslatableString::untranslatable("Turned up bow"),
    muse::TranslatableString::untranslatable("Vibrato pulse accent (Saunders) for stem"),
    muse::TranslatableString::untranslatable("Swiss rudiments doublé black notehead"),
    muse::TranslatableString::untranslatable("Swiss rudiments flam black notehead"),
    muse::TranslatableString::untranslatable("Swiss rudiments doublé half (minim) notehead"),
    muse::TranslatableString::untranslatable("Swiss rudiments flam half (minim) notehead"),
    muse::TranslatableString("engraving/sym", "System divider"),
    muse::TranslatableString("engraving/sym", "Extra long system divider"),
    muse::TranslatableString("engraving/sym", "Long system divider"),
    muse::TranslatableString::untranslatable("Augmentation dot"),
    muse::TranslatableString::untranslatable("Black note, fractional 16th beam, long stem"),
    muse::TranslatableString::untranslatable("Black note, fractional 16th beam, short stem"),
    muse::TranslatableString::untranslatable("Black note, fractional 32nd beam, long stem"),
    muse::TranslatableString::untranslatable("Black note, fractional 8th beam, long stem"),
    muse::TranslatableString::untranslatable("Black note, fractional 8th beam, short stem"),
    muse::TranslatableString::untranslatable("Black note, long stem"),
    muse::TranslatableString::untranslatable("Black note, short stem"),
    muse::TranslatableString::untranslatable("Continuing 16th beam for long stem"),
    muse::TranslatableString::untranslatable("Continuing 16th beam for short stem"),
    muse::TranslatableString::untranslatable("Continuing 32nd beam for long stem"),
    muse::TranslatableString::untranslatable("Continuing 8th beam for long stem"),
    muse::TranslatableString::untranslatable("Continuing 8th beam for short stem"),
    muse::TranslatableString::untranslatable("Headless black note, fractional 16th beam, long stem"),
    muse::TranslatableString::untranslatable("Headless black note, fractional 16th beam, short stem"),
    muse::TranslatableString::untranslatable("Headless black note, fractional 32nd beam, long stem"),
    muse::TranslatableString::untranslatable("Headless black note, fractional 8th beam, long stem"),
    muse::TranslatableString::untranslatable("Headless black note, fractional 8th beam, short stem"),
    muse::TranslatableString::untranslatable("Headless black note, long stem"),
    muse::TranslatableString::untranslatable("Headless black note, short stem"),
    muse::TranslatableString::untranslatable("Tie"),
    muse::TranslatableString::untranslatable("Tuplet number 3 for long stem"),
    muse::TranslatableString::untranslatable("Tuplet number 3 for short stem"),
    muse::TranslatableString::untranslatable("Tuplet bracket end for long stem"),
    muse::TranslatableString::untranslatable("Tuplet bracket end for short stem"),
    muse::TranslatableString::untranslatable("Tuplet bracket start for long stem"),
    muse::TranslatableString::untranslatable("Tuplet bracket start for short stem"),
    muse::TranslatableString::untranslatable("Time signature 0"),
    muse::TranslatableString::untranslatable("Reversed time signature 0"),
    muse::TranslatableString::untranslatable("Turned time signature 0"),
    muse::TranslatableString::untranslatable("Time signature 1"),
    muse::TranslatableString::untranslatable("Reversed time signature 1"),
    muse::TranslatableString::untranslatable("Turned time signature 1"),
    muse::TranslatableString::untranslatable("Time signature 2"),
    muse::TranslatableString::untranslatable("Reversed time signature 2"),
    muse::TranslatableString::untranslatable("Turned time signature 2"),
    muse::TranslatableString::untranslatable("Time signature 3"),
    muse::TranslatableString::untranslatable("Reversed time signature 3"),
    muse::TranslatableString::untranslatable("Turned time signature 3"),
    muse::TranslatableString::untranslatable("Time signature 4"),
    muse::TranslatableString::untranslatable("Reversed time signature 4"),
    muse::TranslatableString::untranslatable("Turned time signature 4"),
    muse::TranslatableString::untranslatable("Time signature 5"),
    muse::TranslatableString::untranslatable("Reversed time signature 5"),
    muse::TranslatableString::untranslatable("Turned time signature 5"),
    muse::TranslatableString::untranslatable("Time signature 6"),
    muse::TranslatableString::untranslatable("Reversed time signature 6"),
    muse::TranslatableString::untranslatable("Turned time signature 6"),
    muse::TranslatableString::untranslatable("Time signature 7"),
    muse::TranslatableString::untranslatable("Reversed time signature 7"),
    muse::TranslatableString::untranslatable("Turned time signature 7"),
    muse::TranslatableString::untranslatable("Time signature 8"),
    muse::TranslatableString::untranslatable("Reversed time signature 8"),
    muse::TranslatableString::untranslatable("Turned time signature 8"),
    muse::TranslatableString::untranslatable("Time signature 9"),
    muse::TranslatableString::untranslatable("Reversed time signature 9"),
    muse::TranslatableString::untranslatable("Turned time signature 9"),
    muse::TranslatableString::untranslatable("Left bracket for whole time signature"),
    muse::TranslatableString::untranslatable("Left bracket for numerator only"),
    muse::TranslatableString::untranslatable("Right bracket for whole time signature"),
    muse::TranslatableString::untranslatable("Right bracket for numerator only"),
    muse::TranslatableString::untranslatable("Control character for denominator digit"),
    muse::TranslatableString::untranslatable("Control character for numerator digit"),
    muse::TranslatableString::untranslatable("Time signature comma"),
    muse::TranslatableString("engraving/sym", "Common time"),
    muse::TranslatableString::untranslatable("Reversed common time"),
    muse::TranslatableString::untranslatable("Turned common time"),
    muse::TranslatableString("engraving/sym", "Cut time (Bach)"),
    muse::TranslatableString("engraving/sym", "Cut triple time (9/8)"),
    muse::TranslatableString("engraving/sym", "Cut time"),
    muse::TranslatableString::untranslatable("Reversed cut time"),
    muse::TranslatableString::untranslatable("Turned cut time"),
    muse::TranslatableString::untranslatable("Time signature equals"),
    muse::TranslatableString::untranslatable("Time signature fraction ½"),
    muse::TranslatableString::untranslatable("Time signature fraction ⅓"),
    muse::TranslatableString::untranslatable("Time signature fraction ¼"),
    muse::TranslatableString::untranslatable("Time signature fraction ¾"),
    muse::TranslatableString::untranslatable("Time signature fraction ⅔"),
    muse::TranslatableString::untranslatable("Time signature fraction slash"),
    muse::TranslatableString::untranslatable("Time signature minus"),
    muse::TranslatableString::untranslatable("Time signature multiply"),
    muse::TranslatableString::untranslatable("Open time signature (Penderecki)"),
    muse::TranslatableString::untranslatable("Left parenthesis for whole time signature"),
    muse::TranslatableString::untranslatable("Left parenthesis for numerator only"),
    muse::TranslatableString::untranslatable("Right parenthesis for whole time signature"),
    muse::TranslatableString::untranslatable("Right parenthesis for numerator only"),
    muse::TranslatableString::untranslatable("Time signature +"),
    muse::TranslatableString::untranslatable("Time signature + (for numerators)"),
    muse::TranslatableString::untranslatable("Time signature slash separator"),
    muse::TranslatableString::untranslatable("Open time signature"),
    muse::TranslatableString::untranslatable("Combining tremolo 1"),
    muse::TranslatableString::untranslatable("Combining tremolo 2"),
    muse::TranslatableString::untranslatable("Combining tremolo 3"),
    muse::TranslatableString::untranslatable("Combining tremolo 4"),
    muse::TranslatableString::untranslatable("Combining tremolo 5"),
    muse::TranslatableString("engraving/sym", "Divide measured tremolo by 2"),
    muse::TranslatableString("engraving/sym", "Divide measured tremolo by 3"),
    muse::TranslatableString("engraving/sym", "Divide measured tremolo by 4"),
    muse::TranslatableString("engraving/sym", "Divide measured tremolo by 6"),
    muse::TranslatableString::untranslatable("Fingered tremolo 1"),
    muse::TranslatableString::untranslatable("Fingered tremolo 2"),
    muse::TranslatableString::untranslatable("Fingered tremolo 3"),
    muse::TranslatableString::untranslatable("Fingered tremolo 4"),
    muse::TranslatableString::untranslatable("Fingered tremolo 5"),
    muse::TranslatableString::untranslatable("Triple-tongue above"),
    muse::TranslatableString::untranslatable("Triple-tongue below"),
    muse::TranslatableString::untranslatable("Tuplet 0"),
    muse::TranslatableString::untranslatable("Tuplet 1"),
    muse::TranslatableString::untranslatable("Tuplet 2"),
    muse::TranslatableString("engraving/sym", "Tuplet 3"),
    muse::TranslatableString::untranslatable("Tuplet 4"),
    muse::TranslatableString::untranslatable("Tuplet 5"),
    muse::TranslatableString::untranslatable("Tuplet 6"),
    muse::TranslatableString::untranslatable("Tuplet 7"),
    muse::TranslatableString::untranslatable("Tuplet 8"),
    muse::TranslatableString::untranslatable("Tuplet 9"),
    muse::TranslatableString::untranslatable("Tuplet colon"),
    muse::TranslatableString::untranslatable("Wieniawski unmeasured tremolo"),
    muse::TranslatableString::untranslatable("Wieniawski unmeasured tremolo (simpler)"),
    muse::TranslatableString::untranslatable("Unpitched percussion clef 1"),
    muse::TranslatableString::untranslatable("Unpitched percussion clef 2"),
    muse::TranslatableString::untranslatable("Ventiduesima"),
    muse::TranslatableString::untranslatable("Ventiduesima alta"),
    muse::TranslatableString::untranslatable("Ventiduesima bassa"),
    muse::TranslatableString::untranslatable("Ventiduesima bassa (mb)"),
    muse::TranslatableString::untranslatable("Finger click (Stockhausen)"),
    muse::TranslatableString::untranslatable("Halb gesungen (semi-sprechgesang)"),
    muse::TranslatableString::untranslatable("Mouth closed"),
    muse::TranslatableString::untranslatable("Mouth open"),
    muse::TranslatableString::untranslatable("Mouth pursed"),
    muse::TranslatableString::untranslatable("Mouth slightly open"),
    muse::TranslatableString::untranslatable("Mouth wide open"),
    muse::TranslatableString::untranslatable("Nasal voice"),
    muse::TranslatableString::untranslatable("Sprechgesang"),
    muse::TranslatableString::untranslatable("Tongue click (Stockhausen)"),
    muse::TranslatableString::untranslatable("Tongue and finger click (Stockhausen)"),
    muse::TranslatableString::untranslatable("Combining sussurando for stem"),
    muse::TranslatableString::untranslatable("Arpeggiato wiggle segment, downwards"),
    muse::TranslatableString::untranslatable("Arpeggiato arrowhead down"),
    muse::TranslatableString::untranslatable("Arpeggiato downward swash"),
    muse::TranslatableString::untranslatable("Arpeggiato wiggle segment, upwards"),
    muse::TranslatableString::untranslatable("Arpeggiato arrowhead up"),
    muse::TranslatableString::untranslatable("Arpeggiato upward swash"),
    muse::TranslatableString::untranslatable("Circular motion segment"),
    muse::TranslatableString::untranslatable("Constant circular motion segment"),
    muse::TranslatableString::untranslatable("Constant circular motion segment (flipped)"),
    muse::TranslatableString::untranslatable("Constant circular motion segment (flipped, large)"),
    muse::TranslatableString::untranslatable("Constant circular motion segment (large)"),
    muse::TranslatableString::untranslatable("Circular motion end"),
    muse::TranslatableString::untranslatable("Circular motion segment, large"),
    muse::TranslatableString::untranslatable("Circular motion segment, larger"),
    muse::TranslatableString::untranslatable("Circular motion segment, larger still"),
    muse::TranslatableString::untranslatable("Circular motion segment, largest"),
    muse::TranslatableString::untranslatable("Circular motion segment, small"),
    muse::TranslatableString::untranslatable("Circular motion start"),
    muse::TranslatableString::untranslatable("Glissando wiggle segment"),
    muse::TranslatableString::untranslatable("Group glissando 1"),
    muse::TranslatableString::untranslatable("Group glissando 2"),
    muse::TranslatableString::untranslatable("Group glissando 3"),
    muse::TranslatableString::untranslatable("Quasi-random squiggle 1"),
    muse::TranslatableString::untranslatable("Quasi-random squiggle 2"),
    muse::TranslatableString::untranslatable("Quasi-random squiggle 3"),
    muse::TranslatableString::untranslatable("Quasi-random squiggle 4"),
    muse::TranslatableString("engraving/sym", "Sawtooth line segment"),
    muse::TranslatableString::untranslatable("Narrow sawtooth line segment"),
    muse::TranslatableString("engraving/sym", "Wide sawtooth line segment"),
    muse::TranslatableString::untranslatable("Square wave line segment"),
    muse::TranslatableString::untranslatable("Narrow square wave line segment"),
    muse::TranslatableString::untranslatable("Wide square wave line segment"),
    muse::TranslatableString::untranslatable("Trill wiggle segment"),
    muse::TranslatableString::untranslatable("Trill wiggle segment, fast"),
    muse::TranslatableString::untranslatable("Trill wiggle segment, faster"),
    muse::TranslatableString::untranslatable("Trill wiggle segment, faster still"),
    muse::TranslatableString::untranslatable("Trill wiggle segment, fastest"),
    muse::TranslatableString::untranslatable("Trill wiggle segment, slow"),
    muse::TranslatableString::untranslatable("Trill wiggle segment, slower"),
    muse::TranslatableString::untranslatable("Trill wiggle segment, slower still"),
    muse::TranslatableString::untranslatable("Trill wiggle segment, slowest"),
    muse::TranslatableString::untranslatable("Vibrato largest, slower"),
    muse::TranslatableString::untranslatable("Vibrato medium, slower"),
    muse::TranslatableString::untranslatable("Vibrato / shake wiggle segment"),
    muse::TranslatableString::untranslatable("Vibrato large, fast"),
    muse::TranslatableString("engraving/sym", "Vibrato large, faster"),
    muse::TranslatableString::untranslatable("Vibrato large, faster still"),
    muse::TranslatableString::untranslatable("Vibrato large, fastest"),
    muse::TranslatableString::untranslatable("Vibrato large, slow"),
    muse::TranslatableString::untranslatable("Vibrato large, slower"),
    muse::TranslatableString("engraving/sym", "Vibrato large, slowest"),
    muse::TranslatableString::untranslatable("Vibrato largest, fast"),
    muse::TranslatableString::untranslatable("Vibrato largest, faster"),
    muse::TranslatableString::untranslatable("Vibrato largest, faster still"),
    muse::TranslatableString::untranslatable("Vibrato largest, fastest"),
    muse::TranslatableString::untranslatable("Vibrato largest, slow"),
    muse::TranslatableString::untranslatable("Vibrato largest, slowest"),
    muse::TranslatableString::untranslatable("Vibrato medium, fast"),
    muse::TranslatableString::untranslatable("Vibrato medium, faster"),
    muse::TranslatableString::untranslatable("Vibrato medium, faster still"),
    muse::TranslatableString::untranslatable("Vibrato medium, fastest"),
    muse::TranslatableString::untranslatable("Vibrato medium, slow"),
    muse::TranslatableString::untranslatable("Vibrato medium, slowest"),
    muse::TranslatableString::untranslatable("Vibrato small, fast"),
    muse::TranslatableString::untranslatable("Vibrato small, faster"),
    muse::TranslatableString::untranslatable("Vibrato small, faster still"),
    muse::TranslatableString::untranslatable("Vibrato small, fastest"),
    muse::TranslatableString::untranslatable("Vibrato small, slow"),
    muse::TranslatableString::untranslatable("Vibrato small, slower"),
    muse::TranslatableString::untranslatable("Vibrato small, slowest"),
    muse::TranslatableString::untranslatable("Vibrato smallest, fast"),
    muse::TranslatableString::untranslatable("Vibrato smallest, faster"),
    muse::TranslatableString::untranslatable("Vibrato smallest, faster still"),
    muse::TranslatableString::untranslatable("Vibrato smallest, fastest"),
    muse::TranslatableString::untranslatable("Vibrato smallest, slow"),
    muse::TranslatableString::untranslatable("Vibrato smallest, slower"),
    muse::TranslatableString::untranslatable("Vibrato smallest, slowest"),
    muse::TranslatableString::untranslatable("Vibrato start"),
    muse::TranslatableString::untranslatable("Wide vibrato / shake wiggle segment"),
    muse::TranslatableString::untranslatable("Wavy line segment"),
    muse::TranslatableString::untranslatable("Narrow wavy line segment"),
    muse::TranslatableString::untranslatable("Wide wavy line segment"),
    muse::TranslatableString::untranslatable("Closed hole"),
    muse::TranslatableString::untranslatable("Flatter embouchure"),
    muse::TranslatableString::untranslatable("Half-closed hole"),
    muse::TranslatableString::untranslatable("Half-closed hole 2"),
    muse::TranslatableString::untranslatable("Half-open hole"),
    muse::TranslatableString::untranslatable("Somewhat relaxed embouchure"),
    muse::TranslatableString::untranslatable("Somewhat tight embouchure"),
    muse::TranslatableString::untranslatable("Mouthpiece or hand pop"),
    muse::TranslatableString::untranslatable("Combining multiphonics (black) for stem"),
    muse::TranslatableString::untranslatable("Combining multiphonics (black and white) for stem"),
    muse::TranslatableString::untranslatable("Combining multiphonics (white) for stem"),
    muse::TranslatableString::untranslatable("Open hole"),
    muse::TranslatableString::untranslatable("Much more reed (push inwards)"),
    muse::TranslatableString::untranslatable("Normal reed position"),
    muse::TranslatableString::untranslatable("Very little reed (pull outwards)"),
    muse::TranslatableString::untranslatable("Relaxed embouchure"),
    muse::TranslatableString::untranslatable("Rim only"),
    muse::TranslatableString::untranslatable("Sharper embouchure"),
    muse::TranslatableString::untranslatable("Very tight embouchure / strong air pressure"),
    muse::TranslatableString::untranslatable("Three-quarters closed hole"),
    muse::TranslatableString::untranslatable("Tight embouchure"),
    muse::TranslatableString::untranslatable("Trill key"),
    muse::TranslatableString::untranslatable("Very tight embouchure"),
    muse::TranslatableString::untranslatable("Very relaxed embouchure / weak air-pressure"),
    // SMuFL standard symbol user names }}}

    //EXTENSIONS
    //SMuFL stylistic alternates which we need to access directly

    muse::TranslatableString::untranslatable("Double whole note (breve), single vertical strokes"),
    muse::TranslatableString::untranslatable("4-string tab clef (serif)"),
    muse::TranslatableString::untranslatable("6-string tab clef (serif)"),
    muse::TranslatableString::untranslatable("C clef (French, 18th century)"),
    muse::TranslatableString::untranslatable("C clef (French, 20th century)"),
    muse::TranslatableString::untranslatable("F clef (French, 18th century)"),
    muse::TranslatableString::untranslatable("F clef (19th century)"),
    muse::TranslatableString::untranslatable("Small brace"),
    muse::TranslatableString::untranslatable("Large brace"),
    muse::TranslatableString::untranslatable("Larger brace"),

    muse::TranslatableString("engraving/sym", "Combining flag 8 (1024th) below straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 8 (1024th) above straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 5 (128th) below straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 5 (128th) above straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 2 (16th) below straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 2 (16th) above straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 6 (256th) below straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 6 (256th) above straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 3 (32nd) below straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 3 (32nd) above straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 7 (512th) below straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 7 (512th) above straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 4 (64th) below straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 4 (64th) above straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 1 (8th) below straight"),
    muse::TranslatableString("engraving/sym", "Combining flag 1 (8th) above straight"),

    muse::TranslatableString::untranslatable("Time signature 0 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 0 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 0 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 1 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 1 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 1 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 2 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 2 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 2 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 3 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 3 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 3 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 4 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 4 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 4 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 5 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 5 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 5 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 6 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 6 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 6 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 7 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 7 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 7 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 8 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 8 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 8 (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature 9 (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature 9 (small staff)"),
    muse::TranslatableString::untranslatable("Time signature 9 (large, narrow)"),
    muse::TranslatableString::untranslatable("Left bracket for numerator only (outside staff)"),
    muse::TranslatableString::untranslatable("Left bracket for numerator only (large, narrow)"),
    muse::TranslatableString::untranslatable("Right bracket for numerator only (outside staff)"),
    muse::TranslatableString::untranslatable("Right bracket for numerator only (large, narrow)"),
    muse::TranslatableString::untranslatable("Left bracket for whole time signature (outside staff)"),
    muse::TranslatableString::untranslatable("Left bracket for whole time signature (narrow, large)"),
    muse::TranslatableString::untranslatable("Right bracket for whole time signature (outside staff)"),
    muse::TranslatableString::untranslatable("Right bracket for whole time signature (large, narrow)"),
    muse::TranslatableString::untranslatable("Common time (outside staff)"),
    muse::TranslatableString::untranslatable("Common time (large, narrow)"),
    muse::TranslatableString::untranslatable("Cut time (outside staff)"),
    muse::TranslatableString::untranslatable("Cut time (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature + (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature + (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature + (for numerators) (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature + (for numerators) (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature fraction slash (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature fraction slash (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature equals (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature equals (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature minus (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature minus (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature multipy (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature multipy (large, narrow)"),
    muse::TranslatableString::untranslatable("Left parenthesis for numerator only (outside staff)"),
    muse::TranslatableString::untranslatable("Left parenthesis for numerator only (large, narrow)"),
    muse::TranslatableString::untranslatable("Right parenthesis for numerator only (outside staff)"),
    muse::TranslatableString::untranslatable("Right parenthesis for numerator only (large, narrow)"),
    muse::TranslatableString::untranslatable("Left parenthesis for whole time signature (outside staff)"),
    muse::TranslatableString::untranslatable("Left parenthesis for whole time signature (large, narrow)"),
    muse::TranslatableString::untranslatable("Right parenthesis for whole time signature (outside staff)"),
    muse::TranslatableString::untranslatable("Right parenthesis for whole time signature (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature comma (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature comma (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature fraction ¼ (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature fraction ¼ (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature fraction ½ (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature fraction ½ (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature fraction ¾ (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature fraction ¾ (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature fraction ⅓ (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature fraction ⅓ (large, narrow)"),
    muse::TranslatableString::untranslatable("Time signature fraction ⅔ (outside staff)"),
    muse::TranslatableString::untranslatable("Time signature fraction ⅔ (large, narrow)"),
    muse::TranslatableString::untranslatable("Open time signature (outside staff)"),
    muse::TranslatableString::untranslatable("Open time signature (large, narrow)"),
    muse::TranslatableString::untranslatable("Open time signature (Penderecki) (outside staff)"),
    muse::TranslatableString::untranslatable("Open time signature (Penderecki) (large, narrow)"),
    muse::TranslatableString::untranslatable("Cut time (Bach) (outside staff)"),
    muse::TranslatableString::untranslatable("Cut time (Bach) (large, narrow)"),
    muse::TranslatableString::untranslatable("Cut triple time (9/8) (outside staff)"),
    muse::TranslatableString::untranslatable("Cut triple time (9/8) (large, narrow)"),

    //MuseScore-local symbols, precomposed symbols to mimic some Emmentaler glyphs

    muse::TranslatableString("engraving/sym", "Prall mordent"),
    muse::TranslatableString("engraving/sym", "Up prall"),
    muse::TranslatableString("engraving/sym", "Up mordent"),
    muse::TranslatableString("engraving/sym", "Prall down"),
    //  muse::TranslatableString("engraving/sym", "Down prall"),
    muse::TranslatableString("engraving/sym", "Down mordent"),
    muse::TranslatableString("engraving/sym", "Prall up"),
    muse::TranslatableString("engraving/sym", "Line prall"),

    //additional symbols

    muse::TranslatableString::untranslatable("Parenthesised double flat accidental"),
    muse::TranslatableString::untranslatable("Parenthesised flat accidental"),
    muse::TranslatableString::untranslatable("Parenthesised natural accidental"),
    muse::TranslatableString::untranslatable("Parenthesised sharp accidental"),
    muse::TranslatableString::untranslatable("Parenthesised double sharp accidental"),

    muse::TranslatableString::untranslatable("noteLongaUp"),
    muse::TranslatableString::untranslatable("noteLongaDown"),
    muse::TranslatableString::untranslatable("noteLongaSquareUp"),
    muse::TranslatableString::untranslatable("noteLongaSquareDown"),
    muse::TranslatableString("engraving/sym", "Space")
} };

//! Conversion table of old symbol names (1.3)
//! The mapping corresponds to fonts/mscore/glyphnames.json and must be in sync with it
//! symNames must be in sync with enum class SymId
const std::map<AsciiStringView, SymId> SymNames::s_oldNameToSymIdHash {
    //{ "ornamentDownPrall",         SymId::ornamentPrecompMordentUpperPrefix },
    { "clef eight",                SymId::clef8 },
    //{ "clef one"                    SymId::},
    //{ "clef five"                   SymId::},
    { "whole rest",                SymId::restWhole },            // rests.0
    { "half rest",                 SymId::restHalf },             // rests.1
    { "outside whole rest",        SymId::restWholeLegerLine },   // rests.0o
    { "outside half rest",         SymId::restHalfLegerLine },    // rests.1o
    { "rest M3",                   SymId::restMaxima },           // rests.M3
    { "breve rest",                SymId::restDoubleWhole },      // rests.M1
    { "longa rest",                SymId::restLonga },            // rests.M2
    { "quart rest",                SymId::restQuarter },          // rests.2
    { "clas quart rest",           SymId::restQuarterOld },       // rests.2classical
    { "eight rest",                SymId::rest8th },              // rests.3
    { "16' rest",                  SymId::rest16th },             // rests.4
    { "32' rest",                  SymId::rest32nd },             // rests.5
    { "64' rest",                  SymId::rest64th },             // rests.6
    { "128' rest",                 SymId::rest128th },            // rests.7
    //{ "256' rest",                 SymId::rest256th },            // rests.8
    //{ "512' rest",                 SymId::rest512th },            // rests.9
    //{ "1024' rest",                SymId::rest1024th },           // rests.10

    { "sharp",                     SymId::accidentalSharp },                        // accidentals.sharp
    { "sharp arrow up",            SymId::accidentalThreeQuarterTonesSharpArrowUp },              // accidentals.sharp.arrowup - typo in 1.3
    { "sharp arrow both",          SymId::accidentalQuarterToneSharpArrowDown },            // accidentals.sharp.arrowdown
    //{ "sharp arrow both",          SymId::noSym },         // accidentals.sharp.arrowboth sharp with both arrows missing in SMuFL
    { "sharp slash",               SymId::accidentalQuarterToneSharpStein },        // accidentals.sharp.slashslash.stem
    { "sharp slash2",              SymId::accidentalBuyukMucennebSharp },           // accidentals.sharp.slashslashslash.stemstem
    { "sharp slash3",              SymId::accidentalKomaSharp },                    // accidentals.sharp.slashslashslash.stem
    { "sharp slash4",              SymId::accidentalThreeQuarterTonesSharpStein },  // accidentals.sharp.slashslash.stemstemstem
    { "natural",                   SymId::accidentalNatural },                            // accidentals.natural
    { "natural arrow up",          SymId::accidentalQuarterToneSharpNaturalArrowUp },     // accidentals.natural.arrowup
    { "natural arrow down",        SymId::accidentalQuarterToneFlatNaturalArrowDown },    // accidentals.natural.arrowdown
    //{ "natural arrow both",        SymId::noSym },         // accidentals.natural.arrowboth natural with both arrows missing in SMuFL
    { "flat",                      SymId::accidentalFlat },                               // accidentals.flat
    { "flat arrow up",             SymId::accidentalQuarterToneFlatArrowUp },             // accidentals.flat.arrowup
    { "flat arrow both",           SymId::accidentalThreeQuarterTonesFlatArrowDown },     // accidentals.flat.arrowdown  - typo in 1.3
    //{ "flat arrow both",           SymId::noSym },         // accidentals.flat.arrowboth flat with both arrows missing in SMuFL
    { "flat slash",                SymId::accidentalBakiyeFlat },                         // accidentals.flat.slash
    { "flat slash2",               SymId::accidentalBuyukMucennebFlat },                  // accidentals.flat.slashslash
    { "mirrored flat2",            SymId::accidentalThreeQuarterTonesFlatZimmermann },    // accidentals.mirroredflat.flat
    { "mirrored flat",             SymId::accidentalQuarterToneFlatStein },     // accidentals.mirroredflat more than one candidate in SMuFL: first occurring chosen
    //{ "mirrored flat slash",       SymId::noSym },                     // accidentals.mirroredflat.backslash - mirrored-slashed flat missing in SMuFL
    { "flat flat",                 SymId::accidentalDoubleFlat },       // accidentals.flatflat
    //{ "flat flat slash",           SymId::noSym },                     // slashes double flat missing in SMuFL
    { "sharp sharp",               SymId::accidentalDoubleSharp },      // accidentals.doublesharp
    { "sori",                      SymId::accidentalSori },             // accidentals.sori
    { "koron",                     SymId::accidentalKoron },            // accidentals.koron
    { "right parenthesis",         SymId::noteheadParenthesisRight },   // accidentals.rightparen SMULF parenth. for noteheads used instead
    { "left parenthesis",          SymId::noteheadParenthesisLeft },    // accidentals.leftparen

    { "arrowheads.open.01",        SymId::arrowheadWhiteRight },  // arrowheads.open.01 similar, not identical in SMuFL
    { "arrowheads.open.0M1",       SymId::arrowheadWhiteLeft },   // arrowheads.open.0M1
    { "arrowheads.open.11",        SymId::arrowheadWhiteUp },     // arrowheads.open.11
    { "arrowheads.open.1M1",       SymId::arrowheadWhiteDown },   // arrowheads.open.1M1
    { "arrowheads.close.01",       SymId::arrowheadBlackRight },  // arrowheads.close.01
    { "arrowheads.close.0M1",      SymId::arrowheadBlackLeft },   // arrowheads.close.0M1
    { "arrowheads.close.11",       SymId::arrowheadBlackUp },     // arrowheads.close.11
    { "arrowheads.close.1M1",      SymId::arrowheadBlackDown },   // arrowheads.close.1M1

    { "dot",                       SymId::augmentationDot },            // dots.dot
    { "longa up",                  SymId::noteLongaUp },                // noteheads.uM2
    { "longa down",                SymId::noteLongaDown },              // noteheads.dM2
    { "brevis head",               SymId::noteheadDoubleWhole },        // noteheads.sM1
    { "brevis double head",        SymId::noSym },                      // noteheads.sM1double, stylistic alternate in SMuFL
    { "whole head",                SymId::noteheadWhole },              // noteheads.s0
    { "half head",                 SymId::noteheadHalf },               // noteheads.s1
    { "quart head",                SymId::noteheadBlack },              // noteheads.s2
    { "whole diamond head",        SymId::noteheadDiamondWhole },       // noteheads.s0diamond
    { "half diamond head",         SymId::noteheadDiamondHalf },        // noteheads.s1diamond
    { "diamond head",              SymId::noteheadDiamondBlack },       // noteheads.s2diamond
    { "whole triangle head",       SymId::noteheadTriangleDownWhole },  // noteheads.s0triangle
    { "down half triangle head",   SymId::noteheadTriangleDownHalf },   // noteheads.d1triangle
    { "up half triangle head",     SymId::noteheadTriangleDownHalf },   // noteheads.u1triangle
    { "up quart triangle head",    SymId::noteheadTriangleDownBlack },  // noteheads.u2triangle
    { "down quart triangle head",  SymId::noteheadTriangleDownBlack },  // noteheads.d2triangle
    { "whole slash head",          SymId::noteheadSlashWhiteWhole },    // noteheads.s0slash
    { "half slash head",           SymId::noteheadSlashWhiteHalf },     // noteheads.s1slash
    { "quart slash head",          SymId::noteheadSlashHorizontalEnds },      // noteheads.s2slash
    { "whole cross head",          SymId::noteheadXWhole },             // noteheads.s0cross
    { "half cross head",           SymId::noteheadXHalf },              // noteheads.s1cross
    { "cross head",                SymId::noteheadXBlack },             // noteheads.s2cross
    { "x circle head",             SymId::noteheadCircleX },            // noteheads.s2xcircle
    //{ "s0do head",                 SymId::noSym },         // noteheads.s0do Whole DO triangle up missing in SMuFL
    { "d1do head",                 SymId::noteShapeTriangleUpWhite },   // noteheads.d1do
    { "u1do head",                 SymId::noteShapeTriangleUpWhite },   // noteheads.u1do
    { "d2do head",                 SymId::noteShapeTriangleUpBlack },   // noteheads.d2do
    { "u2do head",                 SymId::noteShapeTriangleUpBlack },   // noteheads.u2do
    //{ "s0re head",                 SymId::noSym },         // noteheads.s0re Whole moon-shaped RE missing in SMuFL
    { "u1re head",                 SymId::noteShapeMoonWhite },         // noteheads.u1re
    { "d1re head",                 SymId::noteShapeMoonWhite },         // noteheads.d1re
    { "u2re head",                 SymId::noteShapeMoonBlack },         // noteheads.u2re
    { "d2re head",                 SymId::noteShapeMoonBlack },         // noteheads.d2re
    //{ "s0mi head",                 SymId::noSym },         // noteheads.s0mi Whole diamond-shaped MI missing in SMuFL
    { "s1mi head",                 SymId::noteShapeDiamondWhite },      // noteheads.s1mi
    { "s2mi head",                 SymId::noteShapeDiamondBlack },      // noteheads.s2mi
    //{ "u0fa head",                 SymId::noSym },         // noteheads.u0fa Whole triangle-left shaped (up) FA missing in SMuFL
    //{ "d0fa head",                 SymId::noSym },         // noteheads.d0fa Whole triangle-left shaped (dn) FA missing in SMuFL
    { "u1fa head",                 SymId::noteheadTriangleUpRightWhite },     // noteheads.u1fa
    { "d1fa head",                 SymId::noteShapeTriangleRightWhite },      // noteheads.d1fa
    { "u2fa head",                 SymId::noteheadTriangleUpRightBlack },     // noteheads.u2fa
    { "d2fa head",                 SymId::noteheadTriangleLeftBlack },        // noteheads.d2fa
    //{ "s0la head",                 SymId::noSym },         // noteheads.s0la Whole rectangle shaped LA missing in SMuFL
    { "s1la head",                 SymId::noteShapeSquareWhite },             // noteheads.s1la
    { "s2la head",                 SymId::noteShapeSquareBlack },             // noteheads.s2la
    //{ "s0ti head",                 SymId::noSym },         // Whole rounded-triangle shaped TI missing in SMuFL
    { "u1ti head",                 SymId::noteShapeTriangleRoundWhite },      // noteheads.u1ti
    { "d1ti head",                 SymId::noteShapeTriangleRoundWhite },      // noteheads.d1ti
    { "u2ti head",                 SymId::noteShapeTriangleRoundBlack },      // noteheads.u2ti
    { "d2ti head",                 SymId::noteShapeTriangleRoundBlack },      // noteheads.d2ti
    //{ "s0sol head",                SymId::noSym },         // noteheads.s0sol Whole rounded shaped SOL missing in SMuFL
    { "s1sol head",                SymId::noteShapeRoundWhite },        // noteheads.s1sol
    { "s2sol head",                SymId::noteShapeRoundBlack },        // noteheads.s2sol

    { "ufermata",                  SymId::fermataAbove },               // scripts.ufermata
    { "dfermata",                  SymId::fermataBelow },               // scripts.dfermata
    { "snappizzicato",             SymId::pluckedSnapPizzicatoAbove },  // scripts.snappizzicato
    { "ushortfermata",             SymId::fermataShortAbove },          // scripts.ushortfermata
    { "dshortfermata",             SymId::fermataShortBelow },          // scripts.dshortfermata
    { "ulongfermata",              SymId::fermataLongAbove },           // scripts.ulongfermata
    { "dlongfermata",              SymId::fermataLongBelow },           // scripts.dlongfermata
    { "uverylongfermata",          SymId::fermataVeryLongAbove },       // scripts.uverylongfermata
    { "dverylongfermata",          SymId::fermataVeryLongBelow },       // scripts.dverylongfermata
    { "thumb",                     SymId::stringsThumbPosition },       // scripts.thumb
    { "sforza to accent",          SymId::articAccentAbove },           // scripts.sforzato
    //{ "espressivo",                SymId::noSym },         // scripts.espr <> 'espressivo' removed and not present in SMuFL
    { "staccato",                  SymId::articStaccatoAbove },         // scripts.staccato
    { "ustaccatissimo",            SymId::articStaccatissimoAbove },    // scripts.ustaccatissimo
    { "dstaccatissimo",            SymId::articStaccatissimoBelow },    // scripts.dstaccatissimo
    { "tenuto",                    SymId::articTenutoAbove },           // scripts.tenuto
    { "uportato",                  SymId::articTenutoStaccatoAbove },   // scripts.uportato
    { "dportato",                  SymId::articTenutoStaccatoBelow },   // scripts.dportato
    { "umarcato",                  SymId::articMarcatoAbove },          // scripts.umarcato
    { "dmarcato",                  SymId::articMarcatoBelow },          // scripts.dmarcato
    { "fadein",                    SymId::guitarFadeIn },               // scripts.fadein
    { "fadeout",                   SymId::guitarFadeOut },              // scripts.fadeout
    { "volumeswell",               SymId::guitarVolumeSwell },          // scripts.volumeswell
    { "wigglesawtooth",            SymId::wiggleSawtooth },             // scripts.wiggleSawtooth
    { "wigglesawtoothwide",        SymId::wiggleSawtoothWide },         // scripts.wiggleSawtoothWide
    { "wigglevibratolargefaster",  SymId::wiggleVibratoLargeFaster },   // scripts.wiggleVibratoLargeFaster
    { "wigglevibratolargeslowest", SymId::wiggleVibratoLargeSlowest },  // scripts.wiggleVibratoLargeSlowest
    { "ouvert",                    SymId::brassMuteOpen },              // scripts.open
    { "halfopen",                  SymId::brassMuteHalfClosed },        // scripts.halfopen CHECK!!! pre 2.0 only
    { "plus stop",                 SymId::brassMuteClosed },            // scripts.stopped
    { "up bow",                    SymId::stringsUpBow },               // scripts.upbow
    { "down bow",                  SymId::stringsDownBow },             // scripts.downbow
    { "reverse turn",              SymId::ornamentTurnInverted },       // scripts.reverseturn
    { "turn",                      SymId::ornamentTurn },               // scripts.turn
    { "vertical turn",             SymId::ornamentTurnUp },             // scripts.verticalturn
    { "reverse vertical turn",     SymId::ornamentTurnUpS },            // scripts.reverseverticalturn
    { "trill",                     SymId::ornamentTrill },              // scripts.trill
    { "upedal heel",               SymId::keyboardPedalHeel1 },         // scripts.upedalheel
    { "dpedalheel",                SymId::keyboardPedalHeel2 },         // scripts.dpedalheel
    { "upedal toe",                SymId::keyboardPedalToe1 },          // scripts.upedaltoe
    { "dpedal toe",                SymId::keyboardPedalToe2 },          // scripts.dpedaltoe
    { "flageolet",                 SymId::stringsHarmonic },            // scripts.flageolet

    { "segno",                     SymId::segno },                      // scripts.segno
    { "varsegno",                  SymId::segnoSerpent1 },  // not identical to SMuFL, but very similar
    { "coda",                      SymId::coda },                       // scripts.coda
    { "varied coda",               SymId::codaSquare },                 // scripts.varcoda
    { "rcomma",                    SymId::breathMarkSalzedo },          // scripts.rcomma
    //{ "lcomma",                    SymId::noSym },         // reversed breath missing in SMuFL
    //{ "rvarcomma",                 SymId::noSym },         // straight breath missing in SMuFL
    //{ "lvarcomma",                 SymId::noSym },         // reversed straight breath missing in SMuFL
    { "arpeggio",                  SymId::wiggleArpeggiatoUp },         // scripts.arpeggio
    { "trillelement",              SymId::wiggleTrill },                // scripts.trill_element
    { "arpeggio arrow down",       SymId::wiggleArpeggiatoDownArrow },  // scripts.arpeggio.arrow.M1
    { "arpeggio arrow up",         SymId::wiggleArpeggiatoUpArrow },    // scripts.arpeggio.arrow.1
    { "trill element",             SymId::wiggleTrillFastest },         // scripts.trilelement
    { "prall",                     SymId::ornamentShortTrill },         // scripts.prall
    { "mordent",                   SymId::ornamentMordent },            // scripts.mordent
    { "prall prall",               SymId::ornamentTremblement },        // scripts.prallprall

    { "prall mordent",             SymId::ornamentPrallMordent },       // scripts.prallmordent
    { "up prall",                  SymId::ornamentUpPrall },            // scripts.upprall
    { "up mordent",                SymId::ornamentUpMordent },          // scripts.upmordent
    { "prall down",                SymId::ornamentPrallDown },          // scripts.pralldown
    //{ "down prall",                SymId::ornamentDownPrall },        // scripts.downprall
    { "down mordent",              SymId::ornamentDownMordent },        // scripts.downmordent
    { "prall up",                  SymId::ornamentPrallUp },            // scripts.prallup
    { "line prall",                SymId::ornamentLinePrall },          // scripts.lineprall

    { "schleifer",                 SymId::ornamentPrecompSlide },       // scripts.schleifer
    { "caesura straight",          SymId::caesura },                    // scripts.caesura.straight
    { "caesura curved",            SymId::caesuraCurved },              // scripts.caesura.curved

    { "eight flag",                SymId::flag8thUp },                  // flags.u3
    { "sixteenth flag",            SymId::flag16thUp },                 // flags.u4
    { "thirtysecond flag",         SymId::flag32ndUp },                 // flags.u5
    { "sixtyfour flag",            SymId::flag64thUp },                 // flags.u6
    { "128flag",                   SymId::flag128thUp },                // flags.u7
    //{ "256flag",                   SymId::flag256thUp },              // flags.u8
    //{ "512flag",                   SymId::flag512thUp },              // flags.u9
    //{ "1024flag",                  SymId::flag1024thUp },             // flags.u10
    { "deight flag",               SymId::flag8thDown },                // flags.d3
    { "grace dash",                SymId::graceNoteSlashStemUp },       // flags.ugrace
    { "dgrace dash",               SymId::graceNoteSlashStemDown },     // flags.dgrace
    { "dsixteenth flag",           SymId::flag16thDown },               // flags.d4
    { "dthirtysecond flag",        SymId::flag32ndDown },               // flags.d5
    { "dsixtyfourth flag",         SymId::flag16thDown },               // flags.d6
    { "d128flag",                  SymId::flag128thDown },              // flags.d7
    //{ "d256flag",                  SymId::flag256thDown },            // flags.d8
    //{ "d512flag",                  SymId::flag512thDown },            // flags.d9
    //{ "d1024flag",                 SymId::flag1024thDown },           // flags.d10

    { "alto clef",                 SymId::cClef },                      // clefs.C
    { "calto clef",                SymId::cClefChange },                // clefs.C_change
    { "bass clef",                 SymId::fClef },                      // clefs.F
    { "cbass clef",                SymId::fClefChange },                // clefs.F_change
    { "trebleclef",                SymId::gClef },                      // clefs.G
    { "ctrebleclef",               SymId::gClefChange },                // clefs.G_change
    { "percussion clef",           SymId::unpitchedPercussionClef1 },   // clefs.percussion
    { "cpercussion clef",          SymId::unpitchedPercussionClef1 },   // clefs.percussion_change stylistic alternate in SMuFL
    { "tab clef",                  SymId::sixStringTabClef },           // clefs.tab
    { "ctab clef",                 SymId::sixStringTabClef },           // clefs.tab_change  stylistic alternate in SMuFL
    { "four four meter",           SymId::timeSigCommon },              // timesig.C44
    { "allabreve",                 SymId::timeSigCutCommon },           // timesig.C22
    { "pedalasterisk",             SymId::keyboardPedalUp },            // pedal.*
    { "pedaldash",                 SymId::keyboardPedalHyphen },        // pedal.M
    { "pedaldot",                  SymId::keyboardPedalDot },           // pedal..
    { "pedalP",                    SymId::keyboardPedalP },             // pedal.P
    { "pedald",                    SymId::keyboardPedalD },             // pedal.d
    { "pedale",                    SymId::keyboardPedalE },             // pedal.e
    { "pedal ped",                 SymId::keyboardPedalPed },           // pedal.Ped
    { "bracket tips up",           SymId::bracketTop },                 // brackettips.uright
    { "bracket tips down",         SymId::bracketBottom },              // brackettips.dright
    { "bracket tips left up",      SymId::reversedBracketTop },         // brackettips.uleft
    { "bracket tips left down",    SymId::reversedBracketBottom },      // brackettips.dleft
    { "acc dot",                   SymId::accdnCombDot },               // accordion.accDot
    { "acc freebase",              SymId::accdnCombLH2RanksEmpty },     // accordion.accFreebase
    { "acc stdbase",               SymId::accdnCombRH4RanksEmpty },     // accordion.accStdbase
    { "acc bayanbase",             SymId::accdnCombLH3RanksEmptySquare }, // accordion.accBayanbase
    //{ "acc old ee",                 SymId::noSym },                   // accordion.accOldEE   missing in SMuFL
    { "acc discant",               SymId::accdnCombRH3RanksEmpty },     // accordion.accDiscant
    //{ "push",                       SymId::???? },                    // accordion.push
    //{ "pull",                       SymId::???? },                    // accordion.pull
    //{ "space",                      SymId::noSym },                   // space                punctuation missing in SMuFL?
    //{ "plus",                       SymId::noSym },                   // plus                   "
    //{ "comma",                      SymId::noSym },                   // comma                   "
    //{ "hyphen",                     SymId::noSym },                   // hyphen                  "
    //{ "period",                     SymId::noSym },                   // period                  "
    { "zero",                      SymId::timeSig0 },                   // zero
    { "one",                       SymId::timeSig1 },                   // one
    { "two",                       SymId::timeSig2 },                   // two
    { "three",                     SymId::timeSig3 },                   // three
    { "four",                      SymId::timeSig4 },                   // four
    { "five",                      SymId::timeSig5 },                   // five
    { "six",                       SymId::timeSig6 },                   // six
    { "seven",                     SymId::timeSig7 },                   // seven
    { "eight",                     SymId::timeSig8 },                   // eight
    { "nine",                      SymId::timeSig9 },                   // nine
    { "f",                         SymId::dynamicForte },               // f
    { "m",                         SymId::dynamicMezzo },               // m
    { "p",                         SymId::dynamicPiano },               // p
    { "r",                         SymId::dynamicRinforzando },         // r
    { "s",                         SymId::dynamicSforzando },           // s
    { "z",                         SymId::dynamicZ },                   // z
    { "longa up alt",              SymId::noteLongaSquareUp },          // noteheads.uM2alt
    { "longa down alt",            SymId::noteLongaSquareDown },        // noteheads.dM2alt
    { "brevis head alt",           SymId::noteheadDoubleWholeSquare },  // noteheads.sM1alt
    { "time sig C dot",            SymId::mensuralProlation5 },         // timesig.Cdot
    { "time sig O",                SymId::mensuralProlation2 },         // timesig.O
    { "time sig O cut",            SymId::mensuralProlation3 },         // timesig.Ocut
    { "time sig O dot",            SymId::mensuralProlation1 },         // timesig.Odot
    { "clefs.tab2",                SymId::sixStringTabClefSerif },      // clefs.tab2  -- pre 2.0 only
};
