
/* Handle downloads */

#include "gnutella.h"

#include "interface.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <dirent.h>

GSList *sl_downloads = NULL;

guint32 count_downloads  = 0;

gboolean send_pushes = TRUE;

gchar dl_tmp[4096];

void send_push_request(gchar *, guint32, guint16);
void download_move_to_dir(struct download *, gchar *, gchar *);
// void downloads_find_partial_file(struct download *); // EXPERIMENTAL! NOT USED

#define IS_DOWNLOAD_QUEUED(d)  ((d)->status == GTA_DL_QUEUED)

#define IS_DOWNLOAD_STOPPED(d) \
		(  (d)->status == GTA_DL_ABORTED \
		|| (d)->status == GTA_DL_ERROR \
		|| (d)->status == GTA_DL_COMPLETED  )

#define IS_DOWNLOAD_RUNNING(d) \
		(  (d)->status == GTA_DL_HEADERS \
		|| (d)->status == GTA_DL_RECEIVING )

#define IS_DOWNLOAD_ATTEMPTED(d) \
		(  (d)->status == GTA_DL_CONNECTING \
		|| (d)->status == GTA_DL_PUSH_SENT \
		|| (d)->status == GTA_DL_FALLBACK \
		|| (d)->status == GTA_DL_REQ_SENT \
		|| (d)->status == GTA_DL_HEADERS \
		|| (d)->status == GTA_DL_RECEIVING \
	   || (d)->status == GTA_DL_TIMEOUT_WAIT  )

#define IS_DOWNLOAD_IN_PUSH_MODE(d) (d->push)

#define IS_DOWNLOAD_VISIBLE(d) (d->visible)

/* ------------------------------------------------------------------------------------------------ */

/* Return the current number of running downloads */

guint32 count_running_downloads(void)
{
	GSList *l;
	guint32 n = 0, q = 0;

	for (l = sl_downloads; l; l = l->next) {
		if (IS_DOWNLOAD_RUNNING((struct download *) l->data)) n++;
		if (IS_DOWNLOAD_QUEUED((struct download *) l->data)) q++;
		}

	gui_update_c_downloads(n);
	gui_update_c_queued(q);

	return n;
}

// Return the current number of downloads being attempted

guint32 count_attempted_downloads(void)
{
	GSList *l;
	guint32 n = 0;

	for (l = sl_downloads; l; l = l->next)
		if (IS_DOWNLOAD_ATTEMPTED((struct download *) l->data))
			n++;

	return n;
}

// Return the number of downloads attempted from the same host

guint32 count_running_downloads_with_guid(gchar *guid)
{
  GSList *l;
  guint32 n = 0;

	for (l = sl_downloads; l; l = l->next)
		if (IS_DOWNLOAD_ATTEMPTED((struct download *) l->data))
		  if (!memcmp(((struct download *) l->data)->guid, guid, 16))
			 n++;

	return n;
}

// Return number of downloads attempted with the same name

guint32 count_running_downloads_with_name(gchar *name)
{
  GSList *l;
  guint32 n = 0;

  for (l = sl_downloads; l; l = l->next)
    if (IS_DOWNLOAD_ATTEMPTED((struct download *) l->data)
       && strcmp(((struct download *)l->data)->file_name, name) == 0)
      ++n;

  return n;
}

// Return number of downloads attempted with the same size

guint32 count_running_downloads_with_size(guint32 size)
{
  GSList *l;
  guint32 n = 0;

  for (l = sl_downloads; l; l = l->next)
    if (IS_DOWNLOAD_ATTEMPTED((struct download *) l->data)
       && ((struct download *)l->data)->size == size) ++n;

  return n;
}


/* GUI operations --------------------------------------------------------------------------------- */

/* Add a download to the GUI */

void download_gui_add(struct download *d)
{
	gchar *titles[3];
	gint row;

	g_return_if_fail(d);

	if (IS_DOWNLOAD_VISIBLE(d))
	{
		g_warning("download_gui_add() called on already visible download '%s' !\n", d->file_name);
		return;
	}

	titles[0] = d->file_name;
	titles[1] = short_size(d->size);
	titles[2] = "";

	if (IS_DOWNLOAD_QUEUED(d))	/* This is a queued download */
	{
		row = gtk_clist_append(GTK_CLIST(clist_download_queue), titles);
		gtk_clist_set_row_data(GTK_CLIST(clist_download_queue), row, (gpointer) d);
	}
	else								/* This is an active download */
	{
		row = gtk_clist_append(GTK_CLIST(clist_downloads), titles);
		gtk_clist_set_row_data(GTK_CLIST(clist_downloads), row, (gpointer) d);
	}

	d->visible = TRUE;
}

/* Remove a download from the GUI */

void download_gui_remove(struct download *d)
{
	gint row;

	g_return_if_fail(d);

	if (!IS_DOWNLOAD_VISIBLE(d))
	{
		g_warning("download_gui_remove() called on unvisible download '%s' !\n", d->file_name);
		return;
	}

	if (IS_DOWNLOAD_QUEUED(d))
	{
		if (selected_queued_download == d) selected_queued_download = (struct download *) NULL;
		row = gtk_clist_find_row_from_data(GTK_CLIST(clist_download_queue), (gpointer) d);
		if (row != -1) gtk_clist_remove(GTK_CLIST(clist_download_queue), row);
		else g_warning("download_gui_remove(): Queued download '%s' not found in clist !?\n", d->file_name);
	}
	else
	{
		if (selected_active_download == d) selected_active_download = (struct download *) NULL;
		row = gtk_clist_find_row_from_data(GTK_CLIST(clist_downloads), (gpointer) d);
		if (row != -1) gtk_clist_remove(GTK_CLIST(clist_downloads), row);
		else g_warning("download_gui_remove(): Active download '%s' not found in clist !?\n", d->file_name);
	}

	d->visible = FALSE;

	gui_update_download_abort_resume();
	gui_update_download_clear();
}

/* Remove stopped downloads */

void downloads_clear_stopped(gboolean all, gboolean now)
{
	GSList *l = sl_downloads;
	time_t current_time = 0;

	/* If all == TRUE: remove COMPLETED | ERROR | ABORTED, else remove only COMPLETED */
	/* If now == TRUE: remove immediately, else remove only downloads idle since at least 3 seconds */

	if(l && !now)
	  current_time = time(NULL);

	while (l)
	{
		struct download *d = (struct download *) l->data;
		l = l->next;

		if (!IS_DOWNLOAD_STOPPED(d)) continue;

		if (all)
		{
			if (now || (current_time - d->last_update) > 3) download_free(d);
		}
		else if (d->status == GTA_DL_COMPLETED)
		{
			if (now || (current_time - d->last_update) > 3) download_free(d);
		}
	}

	gui_update_download_abort_resume();
	gui_update_download_clear();
}

/* Downloads management --------------------------------------------------------------------------- */

void download_stop(struct download *d, guint32 new_status, gchar *reason)
{
	/* Stop an active download, close its socket and its data file descriptor */

	g_return_if_fail(d);

	if (IS_DOWNLOAD_QUEUED(d))
	{
		g_warning("download_stop() called on queued download '%s'!\n", d->file_name);
		return;
	}

	if (IS_DOWNLOAD_STOPPED(d))
	{
		g_warning("download_stop() called on stopped download '%s'!\n", d->file_name);
		return;
	}

	if (new_status != GTA_DL_ERROR && new_status != GTA_DL_ABORTED && new_status != GTA_DL_COMPLETED && new_status != GTA_DL_TIMEOUT_WAIT)
	{
		g_warning("download_stop(): unexpected new status %d !\n", new_status);
		return;
	}

	if (d->status == new_status)
	{
		g_warning("download_stop(): download '%s' already in state %d\n", d->file_name, new_status);
		return;
	}

	/* Close output file */

	if (d->file_desc != -1) { close(d->file_desc); d->file_desc = -1; }

	if (reason) strncpy(d->remove_msg, reason, 255);
	else d->remove_msg[0] = '\0';
	d->remove_msg[255] = '\0'; // always terminate the string, strncpy doesn't always do it

	/* Close socket */

	if (d->socket)
	{
		d->socket->resource.download = (struct download *) NULL;
		socket_destroy(d->socket);
		d->socket = (struct gnutella_socket *) NULL;
	}

	/* Register the new status, and update the GUI if needed */

	d->status = new_status;

	d->last_update = time((time_t *) NULL);

	if (IS_DOWNLOAD_VISIBLE(d)) gui_update_download(d, TRUE);

	if (IS_DOWNLOAD_VISIBLE(d))
	{
		gui_update_download_abort_resume();
		gui_update_download_clear();
	}

	count_running_downloads();

	if(d->restart_timer_id) {
	  g_warning("download_stop: download %s has a restart_timer_id.\n", d->file_name);
	  g_source_remove(d->restart_timer_id);
	  d->restart_timer_id = 0;
	}
}

void download_kill(struct download *d)
{
	/* Kill a active download: remove it from the GUI, and unlink() the file */

	g_return_if_fail(d);

	if (IS_DOWNLOAD_QUEUED(d))
	{
		g_warning("download_queue(): Download is already queued ?!\n");
		return;
	}

	g_snprintf(dl_tmp, sizeof(dl_tmp), "%s/%s", d->path, d->file_name);
	unlink(dl_tmp);

	download_free(d);
}

void download_queue(struct download *d)
{
	/* Put a download in the queue : */

	/* - it's a new download, but we have reached the max number of running downloads */
	/* - the user requested it with the popup menu "Move back to the queue" */

	g_return_if_fail(d);

	if (IS_DOWNLOAD_QUEUED(d))
	{
		g_warning("download_queue(): Download is already queued ?!\n");
		return;
	}

	if (IS_DOWNLOAD_VISIBLE(d)) download_gui_remove(d);

	if (IS_DOWNLOAD_ATTEMPTED(d)) download_stop(d, GTA_DL_ABORTED, NULL);

	d->status = GTA_DL_QUEUED;

	download_gui_add(d);

	gui_update_download(d, TRUE);

	if(d->restart_timer_id) {
	  g_source_remove(d->restart_timer_id);
	  d->restart_timer_id = 0;
	}
}

/* (Re)start a stopped or queued download */

void download_start(struct download *d)
{
	struct stat st;
	gboolean moveit = FALSE;

	g_return_if_fail(d);

	/* If the output file already exists, we have to send a partial request */
        /* Moved here so multiple downloads of existing files drop out when
          they are smaller than the existing file */

	g_snprintf(dl_tmp, sizeof(dl_tmp), "%s/%s", d->path, d->file_name);

	if (stat(dl_tmp, &st) != -1) d->skip = st.st_size;
	else {
		g_snprintf(dl_tmp, sizeof(dl_tmp), "%s/%s", move_file_path, d->file_name);
		if (stat(dl_tmp, &st) != -1) {
			d->skip = st.st_size;
				/* Is there anything to get at all? */
			if (d->size > d->skip) moveit = TRUE; // move it later if all is ok
			}
		else d->skip = 0;
		}
	d->pos = d->skip; // we start it at d->pos

       /* Is there anything to get at all? */
       if (d->size <= d->pos)
       {
               if (IS_DOWNLOAD_QUEUED(d) && IS_DOWNLOAD_VISIBLE(d)) download_gui_remove(d);
               d->status = GTA_DL_CONNECTING;
               download_stop(d, GTA_DL_COMPLETED, "Nothing more to get");
               return;
       }

       /* If theres already a download for this file, or > max hosts or > max downloads*3 , don't queue up */

       if (count_running_downloads_with_name(d->file_name) != 0
           || count_running_downloads_with_guid(d->guid) >= max_host_downloads
           || count_running_downloads() >= max_downloads)
       {
               // DEBUG: g_warning("Download for %s already running", d->file_name);
               if (!IS_DOWNLOAD_QUEUED(d)) download_queue(d);
               return;
       }

	if (IS_DOWNLOAD_QUEUED(d) && IS_DOWNLOAD_VISIBLE(d)) download_gui_remove(d);
	if(!send_pushes) d->push = FALSE;

	if (!IS_DOWNLOAD_IN_PUSH_MODE(d) && check_valid_host(d->ip, d->port))	/* Direct download */
	{
		d->status = GTA_DL_CONNECTING;
		d->socket = socket_connect(d->ip, d->port, GTA_TYPE_DOWNLOAD);

		// move it to active download list
		if (!IS_DOWNLOAD_VISIBLE(d)) download_gui_add(d);

		if (!d->socket) { // show the problem
			download_stop(d, GTA_DL_ERROR, "Connection failed");
			gui_update_download(d, TRUE);
			return;
			}
		d->socket->resource.download = d;
		d->socket->pos = 0;
	}
	else 													/* We have to send a push request */
	{
		d->status = GTA_DL_PUSH_SENT;
		download_push(d);
		if (!IS_DOWNLOAD_VISIBLE(d)) download_gui_add(d);
	}

	if (moveit) download_move_to_dir(d, move_file_path, d->path); // move done -> downloads dir

	gui_update_download(d, TRUE);

	count_running_downloads();
}

/* pick up new downloads from the queue as needed */

void download_pickup_queued(void) // we don't use this anymore, done in main loop
{
return;
/*
	guint row;
	gtk_clist_freeze(GTK_CLIST(clist_download_queue));
	row=0;
	while (row<GTK_CLIST(clist_download_queue)->rows && count_running_downloads() < max_downloads)
	{
		struct download *d = (struct download *) gtk_clist_get_row_data(GTK_CLIST(clist_download_queue), row);

		if (!IS_DOWNLOAD_QUEUED(d)) g_warning("download_pickup_queued(): Download '%s' is not in queued state ! (state = %d)\n", d->file_name, d->status);

               if (count_running_downloads_with_guid(d->guid) < max_host_downloads
                   && count_running_downloads_with_name(d->file_name) == 0)
			download_start(d);
		else row++;
	}
	gtk_clist_thaw(GTK_CLIST(clist_download_queue));
*/
}


void download_push(struct download *d)
{
	g_return_if_fail(d);

	if(!send_pushes) {
	  d->push = FALSE;
         if (++d->retries <= download_max_retries) download_queue(d);
	  else download_stop(d, GTA_DL_ERROR, "Timeout");
	  return;
	}

       d->push = FALSE; // TRUE;
	d->socket = socket_listen(0, 0, GTA_TYPE_DOWNLOAD);
	if (!d->socket) { download_stop(d, GTA_DL_ERROR, "Internal error"); return; }
	d->socket->resource.download = d;
	d->socket->pos = 0;
	send_push_request(d->guid, d->record_index, d->socket->local_port);
}

/* Direct download failed, let's try it with a push request */

void download_fallback_to_push(struct download *d, gboolean user_request)
{
	g_return_if_fail(d);

	if (IS_DOWNLOAD_QUEUED(d))
	{
		g_warning("download_fallback_to_push() called on a queued download !?!\n");
		return;
	}

	if (IS_DOWNLOAD_STOPPED(d)) return;

	if (!d->socket) g_warning("download_fallback_to_push(): no socket for '%s'\n", d->file_name);
	else
	{
		d->socket->resource.download = NULL;
		socket_destroy(d->socket);
		d->socket = NULL;
	}

	if (d->file_desc != -1) { close(d->file_desc); d->file_desc = -1; }

	if (user_request) d->status = GTA_DL_PUSH_SENT; else d->status = GTA_DL_FALLBACK;

	download_push(d);

	gui_update_download(d, TRUE);
}

/* Downloads creation and destruction ------------------------------------------------------------- */

/* Create a new download */

void download_new(gchar *file, gchar *orig_file, guint32 size, guint32 record_index, guint32 ip, guint16 port, gchar *guid)
{
	struct download *d, *d2;
	gchar *s, *fname;
	GSList *l = (GSList *) NULL;

	if (!file || strlen(file) <= 0) return; // file - name to save it under
	if (!orig_file || strlen(orig_file) <= 0) return;
	fname = g_strdup(file);
	if (!fname) return;
	s = fname;  // correct the file name first, any slash = underline, for DOS or Linux file systems
	while (*s) { if (*s == '/') *s = '_'; if (*s == '\\') *s = '_'; s++; }

	for (l = sl_downloads; l; l = l->next) { // check for same guid & filename on any download
		d2 = (struct download *) l->data;
		if (memcmp(d2->guid, guid, 16) == 0) { // check fast one first
			if (strcmp(d2->file_name, fname) == 0) {
				if (IS_DOWNLOAD_QUEUED(d2)) { // queued? update the info in case it changed
					d2->size = size;
					d2->record_index = record_index;
					d2->ip = ip;
					d2->port = port;
					d2->queue_date = time((time_t *) NULL);
					}
				g_free(fname); // free this string
				return; // don't create a new download record
				}
			}
		}

	d = (struct download *) g_malloc0(sizeof(struct download));	

	d->path = g_strdup(save_file_path);
	d->file_name = fname; // pass this allocated string pointer on
	d->orig_file_name = g_strdup(orig_file); // original filename used to send requests
	d->size = size;
	d->record_index = record_index;
	d->ip = ip;
	d->port = port;
	d->file_desc = -1;
	memcpy(d->guid, guid, 16);
	d->restart_timer_id = 0;
	d->timeout_delay = 0;
	d->queue_date = time((time_t *) NULL);
	d->remove_msg[0] = '\0';

	sl_downloads = g_slist_prepend(sl_downloads, (gpointer) d);

	download_queue(d); // then stick it into the queue, it will start soon
}

/* Free a download. */

void download_free(struct download *d)
{
	g_return_if_fail(d);

	if (IS_DOWNLOAD_VISIBLE(d)) download_gui_remove(d);

	if (IS_DOWNLOAD_ATTEMPTED(d)) download_stop(d, GTA_DL_ABORTED, NULL);

	sl_downloads = g_slist_remove(sl_downloads, (gpointer) d);

	if(d->restart_timer_id)
	  g_source_remove(d->restart_timer_id);

	if (d->path) g_free(d->path);
	if (d->file_name) g_free(d->file_name);
	if (d->orig_file_name) g_free(d->orig_file_name);
	g_free(d);
}

/* ------------------------------------------------------------------------------------------------ */

void download_abort(struct download *d)
{
	g_return_if_fail(d);

	if (IS_DOWNLOAD_QUEUED(d))
	{
		g_warning("download_abort() called on queued download '%s'!\n", d->file_name);
		return;
	}

	if (IS_DOWNLOAD_STOPPED(d)) return;

	download_stop(d, GTA_DL_ABORTED, NULL);
}


void download_resume(struct download *d)
{
	g_return_if_fail(d);

	if (IS_DOWNLOAD_QUEUED(d))
	{
		g_warning("download_resume() called on queued download '%s'!\n", d->file_name);
		return;
	}

	if (IS_DOWNLOAD_ATTEMPTED(d)) return;

	download_start(d);
}

/* IO functions ----------------------------------------------------------------------------------- */

/* Based on patch from Myers W. Carpenter <myers@fil.org> */

void download_move_to_dir(struct download *d, gchar *olddir, gchar *newdir)
{
	/* Move a complete file to move_file_path */

	gchar dl_src[4096];
	gchar dl_dest[4096];
	gint return_tmp, return_tmp2;

       if (!strcmp(olddir, newdir)) return;

       g_snprintf(dl_src, sizeof(dl_src), "%s/%s", olddir, d->file_name);
       g_snprintf(dl_dest, sizeof(dl_dest), "%s/%s", newdir, d->file_name);

	/* First try and link it to the new location */

	return_tmp = rename(dl_src, dl_dest);

	if (return_tmp == -1 && (errno == EXDEV || errno == EPERM))
	{
		/* link failed becase either the two paths aren't on the */
		/* same filesystem or the filesystem doesn't support hard */
	  	/* links, so we have to do a copy. */

		gint tmp_src, tmp_dest;
		gboolean ok = FALSE;

		if ((tmp_src  = open(dl_src, O_RDONLY)) < 0)
		{
			g_warning("Unable to open() file '%s' (%s) !\n", dl_src, g_strerror(errno));
			return;
		}

		if ((tmp_dest = open(dl_dest, O_WRONLY | O_CREAT | O_TRUNC, 0644)) < 0)
		{
			close(tmp_src);
			g_warning("Unable to create file '%s' (%s) !\n", dl_src, g_strerror(errno));
			return;
		}

		for (;;) 
		{
			return_tmp = read(tmp_src, dl_tmp, sizeof(dl_tmp));

			if (!return_tmp) { ok = TRUE; break; }

			if (return_tmp < 0)
			{ 
                               g_warning("download_move_to_dir(): error reading while moving file to save directory (%s)\n", g_strerror(errno));
				break;
			}

			return_tmp2 = write(tmp_dest, dl_tmp, return_tmp);

			if (return_tmp2 < 0)
			{ 
                               g_warning("download_move_to_dir(): error writing while moving file to save directory (%s)\n", g_strerror(errno));
				break;
			}

			if (return_tmp < sizeof(dl_tmp)) { ok = TRUE; break; }
		}
	
		close(tmp_dest); close(tmp_src);
		if (ok) unlink(dl_src);
	}

	return;
}

/* Send a push request */

void send_push_request(gchar *guid, guint32 file_id, guint16 local_port)
{
	struct gnutella_msg_push_request m;

	message_set_muid(&(m.header));

	m.header.function = GTA_MSG_PUSH_REQUEST;
	m.header.ttl = my_ttl;
	m.header.hops = 0;

	WRITE_GUINT32_LE(sizeof(struct gnutella_push_request), m.header.size);

	memcpy(&(m.request.guid), guid, 16);

	WRITE_GUINT32_LE(file_id, m.request.file_id);
	WRITE_GUINT32_BE((force_local_ip)? forced_local_ip : local_ip, m.request.host_ip);
	WRITE_GUINT16_LE(local_port, m.request.host_port);

	message_add(m.header.muid, GTA_MSG_PUSH_REQUEST, NULL);

	sendto_all((guchar *) &m, NULL, sizeof(struct gnutella_msg_push_request), TRUE);
}

/* Send the HTTP request for a download */

gboolean download_send_request(struct download *d)
{
	gchar *temp_name;

	g_return_val_if_fail(d, FALSE);

	if (!d->socket)
	{
		g_warning("download_send_request(): No socket for '%s'\n", d->file_name);
		download_stop(d, GTA_DL_ERROR, "Internal Error");
		return FALSE;
	}

	/* Send the HTTP Request */

	temp_name = encode_url(d->orig_file_name); // URL encode the file name for HTTP call
	if (d->skip) g_snprintf(dl_tmp, sizeof(dl_tmp),
		"GET /get/%i/%s HTTP/1.0\r\nConnection: Keep-Alive\r\nRange: bytes=%u-\r\nUser-Agent: %s\r\n\r\n",
		d->record_index, temp_name, d->skip, name_version);
	else g_snprintf(dl_tmp, sizeof(dl_tmp),
		"GET /get/%i/%s HTTP/1.0\r\nConnection: Keep-Alive\r\nUser-Agent: %s\r\n\r\n",
		d->record_index, temp_name, name_version);

	g_free(temp_name); // encode_url allocates memory for the new string

	if (write(d->socket->file_desc, dl_tmp, strlen(dl_tmp)) < 0)
	{
		download_stop(d, GTA_DL_ERROR, "Write failed");
		return FALSE;
	}

	/* Update the GUI */

	d->status = GTA_DL_REQ_SENT;

	gui_update_download(d, TRUE);

	return TRUE;
}

/* not used anymore, see main loop for queue pickup
gboolean download_queue_w(gpointer dp) {
  struct download *d = (struct download *)dp;
  download_queue(d);
  download_pickup_queued();
  return TRUE;
}
*/
/*
void download_start_restart_timer(struct download *d) {
  d->restart_timer_id = g_timeout_add(60 * 1000, download_queue_w, d);
}
*/

/* Read data on a download socket */

void download_read(gpointer data, gint source, GdkInputCondition cond)
{
	struct gnutella_socket *s = (struct gnutella_socket *) data;
	struct download *d;
	guint32 pos, temppos, offs = 0;
	gint32 r;
	gint http_status=-1;
	gchar http_status_string[4]="   ", *seek;
	gboolean end;

	g_return_if_fail(s);

	d = s->resource.download;
	g_return_if_fail(d);

	if (cond & GDK_INPUT_EXCEPTION) {
		download_stop(d, GTA_DL_ERROR, "Failed (Input Exception)");
		return;
		}

	r = read(s->file_desc, s->buffer + s->pos, MIN(sizeof(s->buffer) - s->pos, d->size - d->pos));

	if (r <= 0) { // no data, may be busy or a socket error so try again, but will time out via d->retries
		download_retry(d); // put it back into the queue with proper delay
		return;
		}
	// if (r < 0 && errno == EAGAIN) return; // I think it's better to just try it again
	// if (r < 0) { download_stop(d, GTA_DL_ERROR, "Failed (Read Error)"); return; }

	d->retries=0; /* successful read means our retry was successful */
	s->pos += r; // s->pos marks the end of data in the buffer

	switch (d->status)
	{
		case GTA_DL_REQ_SENT: case GTA_DL_PUSH_SENT: case GTA_DL_FALLBACK:
		case GTA_DL_RECEIVING: case GTA_DL_HEADERS:
			break;

		default:
			g_warning("download_read(): UNEXPECTED DOWNLOAD STATUS %d\n", d->status);
	}

	if (d->status == GTA_DL_REQ_SENT)
	{
		d->status = GTA_DL_HEADERS; // request was sent, now we wait for the headers
		gui_update_download(d, TRUE);
	}

	if (d->status == GTA_DL_PUSH_SENT || d->status == GTA_DL_FALLBACK) // handle push GIV return
	{

		if (!g_strncasecmp(s->buffer, "GIV ", 4))
			{
				if ((s->buffer[s->pos - 1] == '\n') && (s->buffer[s->pos - 2] == '\n')) { // got it all?
					if (dbg > 4) debug_show_hex("Got GIV push string",s->buffer,s->pos);
					download_send_request(d); // we can now send the request like normal back on the connection
					s->pos = 0; // forget about anything else in the buffer, await next headers
					return;
					}
			}
	}

	if (d->status == GTA_DL_HEADERS)
	{
		end = FALSE;

		do
		{
			seek = (gchar *) memchr(s->buffer, '\n', s->pos);

			if (!seek) return;

			if (*(seek - 1) == '\r') *(seek - 1) = 0; // look for CRLF and zero it, if not header is messed up!
			else {
					download_stop(d, GTA_DL_ERROR, "Malformed HTTP header !");
					return;
					}
			*seek++ = 0;
			if (!*(s->buffer)) end = TRUE;   // We have all the headers because we got a CRLF twice
			else {
					if (dbg > 3) printf("%s\n",s->buffer);

					if (!g_strncasecmp(s->buffer, "HTTP", 4) ){ // ok, we have a HTTP status
						offs=0;
						do {
							offs++; // move past the spaces
							} while (g_strncasecmp(s->buffer+offs, " ", 1) && *(s->buffer+offs));
						offs++;
						strncpy(http_status_string,s->buffer+offs,3); // get the status number
						http_status=atoi(http_status_string);
						if ((http_status > 299) || (http_status < 200)) {
							if (http_status >= 500 && http_status <= 599) { // 503 is typical "busy"
								download_retry(d); // put it back into the queue with proper delay
								}
							else {
								if (dbg > 1) printf("bad http status, aborting\n");
								download_stop(d, GTA_DL_ERROR, s->buffer);
								}
							return;
							}
						d->ok = TRUE;
						}
					else if (!g_strncasecmp(s->buffer, "Content-Length:", 15)) {
							guint32 z = atol(s->buffer + 15);
							if (!z) { download_stop(d, GTA_DL_ERROR, "Bad length !?"); return; }
							else if (z + d->skip != d->size) {
									if (z == d->size) {
										d->skip = 0;
										d->pos = 0;
										if (dbg > 1) printf("File '%s': server seems to have ignored our range request of %u.\n", d->file_name, d->size);
										download_stop(d, GTA_DL_ERROR, "Resume request ignored, STOP");
										return;
										}
									else if (d->size - d->skip > 1000 && z < 1000) {
												download_stop(d, GTA_DL_ERROR, "Length to short, probably busy!? STOP");
												return;
												}
									else {
										if (dbg > 1) printf("File '%s': expected size %u but server says %u, forget this file\n", d->file_name, d->size, z + d->skip);
										download_stop(d, GTA_DL_ERROR, "Length returned incorrect, STOP");
										return;
										// something is wrong, forget this file, get another, we used to do the following
//										if (dbg > 1) printf("File '%s': expected size %u but server says %u, believing it...\n", d->file_name, d->size, z + d->skip);
//										d->size = z + d->skip;
										}
									}
							} else if (!g_strncasecmp(s->buffer, "Content-Range:", 14)) {
										// Store Content-Range
										} else if (!g_strncasecmp(s->buffer, "Server:", 7)) {
													// Store Server if you want
													} else if (!g_strncasecmp(s->buffer, "Content-type:", 13)) {
																// Store Content type if you want
																} else if (!d->ok) {
																	// check a few other cases I have seen that get past our HTTP check
																	// server is still responding so queue it up again
																	if (strstr(s->buffer, "limit reached") || strstr(s->buffer, "Busy")) {
																		download_retry(d); // put it back into the queue with proper delay
																		return;
																		}
											if (dbg > 1) printf("File '%s': FIXME FIXME FIXME unhandled strange header: %s", d->file_name, s->buffer);
																				d->retries = 0; /* FIXME Hack */
																				download_stop(d, GTA_DL_ERROR, s->buffer);
																				return;
																			} else {
										if (dbg > 3) printf("File '%s': unhandled extra header '%s' but successfull!", d->file_name, s->buffer);
																					}
				}
			temppos = (seek - s->buffer); // how many bytes in this line
			pos = s->pos - temppos;
			if ((pos > sizeof(s->buffer)) || (temppos < 0) || (temppos > sizeof(s->buffer))) { // don't depend on values always being correct
				download_retry(d); // something went wrong, try again new
				return;
				}
			if (pos > 0) {
				memmove(s->buffer, seek, pos); // slide all bytes over, erasing last header line
				}
			s->pos = pos; // adjust the end of buffer position

		} while (!end);

		d->start_date = time((time_t *) NULL);
		d->status = GTA_DL_RECEIVING;
		gui_update_download(d, TRUE);
	} // if HEADERS

	if (d->status == GTA_DL_RECEIVING) /* If we have data, write it to the output file */
	{
		if (d->file_desc == -1 && s->pos > 0) /* The output file is not yet open */
		{
			struct stat st;

			g_snprintf(dl_tmp, sizeof(dl_tmp), "%s/%s", d->path, d->file_name);

			if (stat(dl_tmp, &st) != -1) /* File exists, we'll append the data to it */
			{
				d->file_desc = open(dl_tmp, O_WRONLY);
				if (d->file_desc != -1) lseek(d->file_desc, 0, SEEK_END);
			}
			else d->file_desc = open(dl_tmp, O_WRONLY | O_CREAT | O_TRUNC, 0644);

			if (d->file_desc == -1)
			{
				g_warning("Unable to open() file '%s'! (%s)\n", dl_tmp, g_strerror(errno));
				download_stop(d, GTA_DL_ERROR, "Unable to open file !");
				return;
			}
		}

		if (s->pos > 0) if (write(d->file_desc, s->buffer, s->pos) < 0)
		{
			download_retry(d); // something went wrong, try again new
			return;
		}
	
		d->pos += s->pos;
		s->pos = 0;

		if (d->pos >= d->size)
		{
			download_stop(d, GTA_DL_COMPLETED, NULL);
                       download_move_to_dir(d, d->path, move_file_path);
			count_downloads++;
			gui_update_count_downloads();
			return;
		}
	}

	gui_update_download(d, FALSE);
}

void download_retry(struct download *d)
{
	GSList *l = (GSList *) NULL;

	/* download_stop() sets the time, so all we need to do is set the delay */

	if (d->timeout_delay == 0)
		d->timeout_delay = download_retry_timeout_min;
	else {
		d->timeout_delay += 10; // this is reasonable for each retry
		if (d->start_date) {
			/* We forgive a little while the download is working */
			d->timeout_delay -= (time((time_t *)NULL) - d->start_date) / 10;
		}
	}
	if (d->timeout_delay < download_retry_timeout_min) // keep it within set limits
		d->timeout_delay = download_retry_timeout_min;
	if (d->timeout_delay > download_retry_timeout_max)
		d->timeout_delay = download_retry_timeout_max;

	for (l = sl_downloads; l; l = l->next) { // set all with this GUID to same delay so we don't hammer this host
		if (!memcmp(((struct download *) l->data)->guid, d->guid, 16)) {
			((struct download *) l->data)->timeout_delay = d->timeout_delay;
			}
		}

	download_stop(d, GTA_DL_TIMEOUT_WAIT, NULL); // this will actually re-queue it
}


/*

// EXPERIMENTAL! NOT USED Still playing around with this code to not get dups in partial downloads
// From Scan Man

void downloads_find_partial_file(struct download *d)
{
	gchar sizestr[64];
	gint n, prev_file=0;
	struct dirent **namelist;

	g_snprintf(sizestr, sizeof(sizestr), ".napshare-%d", d->size);
	n = scandir(d->path, &namelist, 0, alphasort);
	if (n < 0) g_warning("Can't get directory listing!");
	else {
		while(n--) {
			if(strstr((char *)namelist[n]->d_name, (char *)sizestr)) {
				prev_file = 1;
				g_snprintf(dl_tmp, sizeof(dl_tmp), "%s/%s", d->path, namelist[n]->d_name);
				}
			free(namelist[n]);
			}
		free(namelist);
		}
	if(!prev_file) g_snprintf(dl_tmp, sizeof(dl_tmp), "%s/%s.napshare-%d", d->path, d->file_name, d->size);
}
*/


/* vi: set ts=3: */
