
/* Misc functions */

#include "gnutella.h"

#include <sys/stat.h>

#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>


// table used to encode a url line 1 = encode it
const static gchar urlchr_table[256] =
{
  1,  1,  1,  1,   1,  1,  1,  1,   /* NUL SOH STX ETX  EOT ENQ ACK BEL */
  1,  1,  1,  1,   1,  1,  1,  1,   /* BS  HT  LF  VT   FF  CR  SO  SI  */
  1,  1,  1,  1,   1,  1,  1,  1,   /* DLE DC1 DC2 DC3  DC4 NAK SYN ETB */
  1,  1,  1,  1,   1,  1,  1,  1,   /* CAN EM  SUB ESC  FS  GS  RS  US  */
  1,  1,  1,  1,   1,  1,  1,  1,   /* SP  !   "   #    $   %   &   '   */
  0,  0,  0,  1,   1,  0,  0,  1,   /* (   )   *   +    ,   -   .   /   */
  0,  0,  0,  0,   0,  0,  0,  0,   /* 0   1   2   3    4   5   6   7   */
  0,  0,  1,  1,   1,  1,  1,  1,   /* 8   9   :   ;    <   =   >   ?   */
  1,  0,  0,  0,   0,  0,  0,  0,   /* @   A   B   C    D   E   F   G   */
  0,  0,  0,  0,   0,  0,  0,  0,   /* H   I   J   K    L   M   N   O   */
  0,  0,  0,  0,   0,  0,  0,  0,   /* P   Q   R   S    T   U   V   W   */
  0,  0,  0,  1,   1,  1,  1,  0,   /* X   Y   Z   [    \   ]   ^   _   */
  1,  0,  0,  0,   0,  0,  0,  0,   /* `   a   b   c    d   e   f   g   */
  0,  0,  0,  0,   0,  0,  0,  0,   /* h   i   j   k    l   m   n   o   */
  0,  0,  0,  0,   0,  0,  0,  0,   /* p   q   r   s    t   u   v   w   */
  0,  0,  0,  1,   1,  1,  1,  1,   /* x   y   z   {    |   }   ~   DEL */

  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,
  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,
  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,
  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,

  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,
  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,
  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,
  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1,  1, 1, 1, 1
};


// table used for fast hex conversion
const static gchar hex_table[16] =
{
  '0',  '1',  '2',  '3',   '4',  '5',  '6',  '7',
  '8',  '9',  'A',  'B',   'C',  'D',  'E',  'F'
};



gchar *ip_to_gchar(guint32 ip)
{
	struct in_addr ia;
	ia.s_addr = g_htonl(ip);
	return inet_ntoa(ia);
}


gchar *ip_port_to_gchar(guint32 ip, guint16 port)
{
	static gchar a[128];
	struct in_addr ia;
	ia.s_addr = g_htonl(ip);
	g_snprintf(a, sizeof(a), "%s:%u", inet_ntoa(ia), port);
	return a;
}

#if defined(_WIN32) || !defined(HAVE_INET_ATON)
/* 
 * Copied from icecast
 */
int 
inet_aton(const char *s, struct in_addr *a)
{
    int lsb, b2, b3, msb;
    if (sscanf(s, "%d.%d.%d.%d", &lsb, &b2, &b3, &msb) < 4)
                return -1;

    a->s_addr = lsb + (b2 << 8) + (b3 << 16) + (msb << 24);
                return 0;
}
#endif


guint32 gchar_to_ip(gchar *str)
{
	/* Returns 0 if str is not a valid IP */

	struct in_addr ia;
	gint r;
	r = inet_aton(str, &ia);
	if (r) return g_ntohl(ia.s_addr);
	return 0;
}

guint32 host_to_ip(gchar *host, gboolean errflg)
{
	struct hostent *he = gethostbyname(host);
	if (he) return g_ntohl(*(guint32 *) (he->h_addr_list[0]));
#if defined(HAVE_HERROR)
	else if (errflg) herror("gethostbyname()");
#else
        else if (errflg) g_warning("gethostbyname('%s') failed!\n",host);
#endif
 
	return 0;
}

/* Checks for RFC1918 private addresses; returns TRUE if is a private address. */
gboolean is_private_ip(guint32 ip)
{
    /* 10.0.0.0 -- (10/8 prefix) */
    if ((ip & 0xff000000) == 0xa000000)
    {
        return TRUE;
    }

    /* 172.16.0.0 -- (172.16/12 prefix) */
    if ((ip & 0xfff00000) == 0xac100000)
    {
        return TRUE;
    }

    /* 192.168.0.0 -- (192.168/16 prefix) */
    if ((ip & 0xffff0000) == 0xc0a80000)
    {
        return TRUE;
    }

    return FALSE;
}


/* Check whether path is a directory */
gboolean is_directory(gchar *path)
{
	struct stat st;
	if (stat(path, &st) == -1) return FALSE;
	return S_ISDIR(st.st_mode);
}

/* Returns a number of bytes in a more readable form */

gchar *short_size(guint32 size)
{
	static gchar b[256];

	if (size < 1024) g_snprintf(b, sizeof(b), "%u Bytes", size);
	else if (size < 1048576) g_snprintf(b, sizeof(b), "%.1fKB", (float) size / 1024.0);
	else if (size < 1073741824) g_snprintf(b, sizeof(b), "%.1fMB", (float) size / 1048576.0);
	else g_snprintf(b, sizeof(b), "%.1fGB", (float) size / 1073741824.0);

	return b;
}

/* Returns the ip:port of a node */

gchar *node_ip(struct gnutella_node *n)
{
	return ip_port_to_gchar(n->ip, n->port);
}

/* Dumps a gnutella message (debug) */

void message_dump(struct gnutella_node *n)
{
	gint32 size, ip, index, count, total;
	gint16 port, speed;

	printf("Node %s: ",    node_ip(n));
	printf("Func 0x%.2x ", n->header.function);
	printf("TTL = %d ",    n->header.ttl);
	printf("hops = %d ",   n->header.hops);

	READ_GUINT32_LE(n->header.size, size);

	printf(" data = %u", size);

	if (n->header.function == GTA_MSG_SEARCH)
	{
		READ_GUINT16_LE(n->buffer, speed);
		printf(" Speed = %d Query = '%s'", speed, n->buffer + 2);
	}
	else if (n->header.function == GTA_MSG_INIT_RESPONSE)
	{
		READ_GUINT16_LE(n->buffer, port);
		READ_GUINT32_BE(n->buffer + 2, ip);
		READ_GUINT32_LE(n->buffer + 6, count);
		READ_GUINT32_LE(n->buffer + 10, total);

		printf(" Host = %s Port = %d Count = %d Total = %d", ip_to_gchar(ip), port, count, total);
	}
	else if (n->header.function == GTA_MSG_PUSH_REQUEST)
	{
		READ_GUINT32_BE(n->buffer + 20, ip);
		READ_GUINT32_LE(n->buffer + 16, index);
		READ_GUINT32_LE(n->buffer + 24, port);

		printf(" Index = %d Host = %s Port = %d ", index, ip_to_gchar(ip), port);
	}

	printf("\n");
}



// make any non printable chars '.'
void str_dejunk(char *str)
{

	int c = 0, x = 0;

	c = strlen(str);

	for (;x<c;x++) {
		str[x] = str[x] & 0x0000007F; // hack off bit 7 for ASCII
		if (str[x] < 0x20 || str[x] == 0x7F) str[x] = '.'; // no ctrls or DEL
		}

	return;
}



// encode a URL link. escape special characters
// remember we allocate memory for this so call free() when done
gchar *encode_url(gchar *str)
{

	guint c = 0, x = 0;
	guint16 y, z; // using int16 to keep things within limits
	gchar *outstr, *t;

	c = strlen(str);
	outstr = t = (gchar *) g_malloc((c * 3) + 3); // max string will ever get

	for (;x<c;x++) {
		y = str[x];
		y = y & 0x00FF; // no more than 255
		z = (y >> 4) & 0x000F;
		if (urlchr_table[y]) { // use lookup table for speed, 1 = encode it
			*t++ = '%';
			*t++ = hex_table[z];
			y = y & 0x000F;
			*t++ = hex_table[y];
			}
		else *t++ = y;
		}

	*t = 0;

	return outstr;
}




// Displays hex & ascii lines to the terminal (for debug)
//
// Displays the "title" then the characters in "s", # of bytes to print in "b"

void debug_show_hex(gchar *title, gchar *s, gint b)
{
	int i,x,y,z,end;
	char temp[18];

	printf ("----------------- %s\n",title);

	if ((b < 1) || (s == NULL)) { // check everything, this is for debug
		printf ("ERROR - debug_show_hex, value out of range\n");
		return;
		}

	i = x = end = 0;
	while (1) {
		if (end) {
			printf("   ");
			temp[i] = ' ';
			}
		else {
			z = s[x] & 0x000000FF;
			printf("%.2X ",z);
			z = z & 0x0000007F; // hack off bit 7 for ASCII
      	if (z < 0x20) z = '.'; // no non printables
			if (z == 0x7F) z = '.'; // no DEL
			temp[i] = z ; // save it for later ASCII print
			}
		if (++i >= 16) {
			printf(" ");
			for (y=0; y<16; y++){ //do 16 bytes ASCII
				printf("%c",temp[y]);
				}
    		printf("\n");
			if (end || ((x + 1) >= b)) break;
			i = 0;
			}
		if (++x >= b) end = 1;
		}
	printf ("----------------- Bytes = %d\n",b);
}


/* vi: set ts=3: */

